;+
; NAME:
;	nrbPlot
;
; PURPOSE:
;	This procedure plots a NURBS surface or curve
;
; CATEGORY:
;	NURBS library
;
; CALLING SEQUENCE:
;	nrbPlot,n1,nx,ny
;
; INPUTS:
;    n1: a NURBS structure (as created by nrbStruct() ) or the evaluated 
;        points (see nrbEval() )
;    NX: number of elements along x direction (not need if the n1 is not a structure)
;    NY: number of elements along y direction (not need if the n1 is not a structure)
;
;
; KEYWORD PARAMETERS:
;    itools: Set this keyword to 1 for plotting using iTools (Default=1)
;    style: as required by graphic routine
;    color: as required by graphic routine
;    Other keywords are passed to the iplot or isurface applications 
;
; OUTPUTS:
;    A plot window
;
; PROCEDURE:
;	
; EXAMPLE:
;	
;	
; MODIFICATION HISTORY:
;	written by Davide Bianchi. ESRF. 2006
;
;-	
PRO nrbplot, nrb, nx, ny, style=style,color=color, _extra=e, itools=itools, $
surf=surf

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'NURBS: nrbPlot: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   RETURN
endif


;=== plot default ===;
IF n_elements(itools) EQ 0 THEN itools=1 
IF n_elements(color) EQ 0 THEN color=[0,0,0]
IF n_elements(style) EQ 0 THEN style = 1


dim = size(nrb,/type) ;CASE statment

CASE dim OF

;=== structure nrb ===;
8: BEGIN
    au=min(nrb.uknots)
    bu=max(nrb.uknots)
    upar=nrblinear(au,bu,nx)

    ;=== surface nurbs ===;
	IF  n_elements(nrb.dim) EQ 2 THEN BEGIN

    av=min(nrb.vknots)
    bv=max(nrb.vknots)
    vpar=nrblinear(av,bv,ny)
    p=nrbeval(nrb,upar,vpar)

	xgrid=__nrbgrid2mat(p, v=0)
    ygrid=__nrbgrid2mat(p, v=1)
    zgrid=__nrbgrid2mat(p, v=2)

    CASE itools OF
      0: surface, zgrid,xgrid,ygrid,XTITLE='x axis',YTITLE='y axis',ZTITLE='z axis',_extra=e
      1: isurface, zgrid,xgrid,ygrid,style=style,color=color,XTITLE='x axis',YTITLE='y axis',ZTITLE='z axis',_extra=e
      else: Message,'iTools value not found.'
    ENDCASE

    ENDIF ELSE BEGIN
    ;=== curve nurbs ===;
    p=nrbeval(nrb,upar)



    IF n_elements(where(p[*,2] EQ 0D)) EQ n_elements(p[*,2]) THEN BEGIN
      ;=== 2D xOy curve ===;
      CASE itools OF 
        0: plot,reform(p[*,0]),reform(p[*,1]),XTITLE='x axis',YTITLE='y axis',ZTITLE='z axis',_extra=e;
        1: iplot,reform(p[*,0]),reform(p[*,1]),style=style,color=color,XTITLE='x axis',YTITLE='y axis',ZTITLE='z axis',_extra=e;
        else: Message,'Wrong iTool value'
      ENDCASE
    ENDIF ELSE BEGIN
      ;=== 3D xyz curve ===;
      CASE itools OF 
        0: plot_3dbox,reform(p[*,0]),reform(p[*,1]),reform(p[*,2]),XTITLE='x axis',YTITLE='y axis',ZTITLE='z axis',_extra=e;
        1: iplot,reform(p[*,0]),reform(p[*,1]),reform(p[*,2]),style=style,color=color,XTITLE='x axis',YTITLE='y axis',ZTITLE='z axis',_extra=e;
        else: Message,'Wrong iTool value'
      ENDCASE
    ENDELSE
    ENDELSE
    END


;=== EVALUATED POINTS or GRID POINTS (see nrbstruct.pro ---> nrb.coefs) ===;
ELSE : BEGIN
	  ;=== Surface Plot ===;
      IF size(nrb,/n_dimensions) EQ 3 THEN BEGIN
         xgrid=__nrbgrid2mat(nrb, v=0)
         ygrid=__nrbgrid2mat(nrb, v=1)
         zgrid=__nrbgrid2mat(nrb, v=2)
         ;=== plot default ===;

         CASE itools OF
           0: surface, zgrid,xgrid,ygrid,color=color[0],XTITLE='x axis',YTITLE='y axis',ZTITLE='z axis',_extra=e
           1: isurface, zgrid,xgrid,ygrid,style=style,XTITLE='x axis',YTITLE='y axis',ZTITLE='z axis',_extra=e
           else: Message,'iTools value not found.'
         ENDCASE


        ENDIF ELSE BEGIN



        ;=== 2D xOy curve ===;
        IF n_elements(where(nrb[*,2] eq 0)) EQ n_elements(nrb[*,2]) THEN BEGIN
	   CASE itools OF
	     0: plot,reform(nrb[*,0]),reform(nrb[*,1]),XTITLE='x axis',YTITLE='y axis',ZTITLE='z axis',_extra=e;
	     1: iplot,reform(nrb[*,0]),reform(nrb[*,1]),style=style,color=color,XTITLE='x axis',YTITLE='y axis',ZTITLE='z axis',_extra=e;
             else: Message,'Wrong iTool value'
           ENDCASE
         ENDIF ELSE BEGIN
         ;=== 3D xyz curve ===;
           CASE itools OF
             0:plot_3dbox,reform(nrb[*,0]),reform(nrb[*,1]),reform(nrb[*,2]),XTITLE='x axis',YTITLE='y axis',ZTITLE='z axis',_extra=e;
             1:iplot,reform(nrb[*,0]),reform(nrb[*,1]),reform(nrb[*,2]),style=style,color=color,XTITLE='x axis',YTITLE='y axis',ZTITLE='z axis',_extra=e;
             else: Message,'Wrong iTool value'
           ENDCASE
         ENDELSE
        ENDELSE
       END

ENDCASE

END
