;+
; 
; PROCEDURE MD
;
; Main program for ray-tracing a single mirror
;
;
;    PRO md, mirror, source=source,image=image, $
;    angle=angle,degree=degree, $
;    file=file,lx=lx,ly=ly, verbose=verbose, $
;    shadowfile=shadowfile, interMethod=interMethod, $
;    rotation=rotation, $
;    mirr=nrb, a2=a2  ; ouputs
;
;	ARGUMENTS
;
;
;		MIRROR = case number for choosing the mirror
;		0:spherical (default)
;		1:plane
;		2:elliptical
;		3:cylindrical
;		4:toroidal
;		5:file surface (shadow format)
;
;	KEYWORDS
;
;		SOURCE = source plane distance in [cm] (default = 3000)
;		IMAGE = image plane distance in [cm] (default = 1000)
;		ANGLE = incidence angle (default = 5e-3 rad)
;		DEG = if this keyword is set the angle is transoformed from 
;                     degree into radians
;		FILE = waviness file (for mirror=5)
;		SHADOWFILE = source file (default ./begin.dat)
;               LX = mirror width
;               LY = mirror length
;               VERBOSE = set it for verbose output
;               INTERMETHOD = ray-surface incersection method:
;			0 (default) Bezier Clipping.
;			1 Minimization algorithm (Power).
;			2 Approximates surface with planes. 
;               ROT = if set, rotates the mirror 5 degrees arond z (for tests)
;
;	OUTPUT KEYWORDS
;
;		MIRR = set this keyword tro a named variable to
;			retrieve the nurbs surface
;		A2 = set this keyword tro a named variable to
;                       retrieve the source raus in the mirror reference
;			frame. 
;		
;
;-

PRO md, mirror, source=source,image=image, $
    angle=angle,degree=degree, $
    file=file,lx=lx,ly=ly, verbose=verbose, $
    shadowfile=shadowfile, interMethod=interMethod, $
    rotation=rotation, $
    mirr=nrb, a2=a2  ; ouputs


; common variables for intersection using minimization
COMMON intermin, _n0, _x0, _v0  

funit = 10D  ; from cm to used units

IF n_elements(mirror) EQ 0 THEN mirror=0
IF n_elements(shadowfile) EQ 0 THEN shadowfile='begin.dat'
IF n_elements(interMethod) EQ 0 THEN interMethod=0


;=== source defaults ===;
IF n_elements(source) EQ 0 THEN source=3000d  ; source-mirror
IF n_elements(image) EQ 0 THEN image=1000d  ; mirror-imagege
IF n_elements(angle) EQ 0 THEN angle= 5D-3
IF keyword_set(degree) THEN angle=angle*!dpi/180d
IF n_elements(lx) EQ 0 THEN lx= 5d
IF n_elements(ly) EQ 0 THEN ly= 8d

print,'p,q,theta: ',source,image,angle

;=== source translation to the surface system ===;
a0 = readsh(shadowfile,/nolost) 
a1 = shrot(a0,angle,axis=1,/rad)
a2 = shtranslation(a1,[0D,-source*cos(angle),source*sin(angle)])

ray = GetShCol(a2,[1,2,3,4,5,6,10]) ;geoemttical coordinats of the ray (the 10th column is the good-bad ray flag)
ray[0:2,*]=funit*ray[0:2,*] ;spatial convertion into [mm]
plane=shray2plane(ray[0:2,*],ray[3:5,*]) ;struct array, all rays by 2 plane intersection

IF Keyword_Set(verbose) THEN print,'MD: plane done'

CASE mirror OF
  0:  mirr = nrbmakesphere(p=source,q=image,angle=angle,ll=ly,ww=lx)
  1:  mirr = nrbmakeplane(ll=ly,ww=lx)
  2:  mirr = nrbmakeellipsoid(p=source,q=image,angle=angle,ll=ly,ww=lx)
  3:  mirr = nrbmakecylinder(p=p,q=q,angle=angle,ll=ly,ww=lx)
  4:  mirr = nrbmaketoroid(p=source,angle=angle,q=image,ww=lx,ll=ly)
  5:  mirr = file
  ELSE: message,'not right mirror'
ENDCASE

help,/str,mirr
nrbplot,mirr.coefs ; ,100,100
nrbplot,mirr,20,20,/overplot


IF keyword_Set(rotation) THEN BEGIN
  tmpz = vecrotz(!dpi/180D*5D)
  mirr = nrbtransform(mirr,tmpz)
ENDIF

IF funit EQ 1D THEN funitflag=0 ELSE funitflag=1

patch=nrbsupport(mirr,bzr=bzr,intsurf=nrb,mm=funitflag,sort=1)

_n0 = nrb  ; copy to common block

IF Keyword_Set(verbose) THEN print, 'MD: preprocessing done'

;
; ** CALCULATION OF INTERCEPTION **
; Input rays: a2
; Output rays: a3
;

CASE interMethod OF
  0: BEGIN
  ;
  ; Bezier clipping (interMethod EQ 0)
  ;
  IF Keyword_Set(verbose) THEN print,'MD: Intersection by bezier clipping'
  ; 
  ; userGuess (int_guess.pro) should help in finding the intersection and would 
  ; help in identifying double intersections.
  ; It does not work...
  pnt=intersec_trial(patch,plane,ray,index=index,verbose=verbose,$
     tolerance=1D-5,useGuess=0)
  t0=systime(1)

  ;pnt=intersec_points(patch,par,p_index=index)
  ;
  ; calculates reflection
  ;
  newray=reflection(ray,pnt,verbose=0)
  newray[0:2,*]=newray[0:2,*]/funit
  last_check=where(newray[4,*] LT 0)
  ;IF last_check[0] ne -1 THEN newray[6,last_check]=-1
  t1=systime(1)
  print, 'MD: reflection time:  ',t1-t0
  ;
  a3=a2
  mray=a3.ray
  mray([0,1,2,3,4,5,9],*)=newray[[0,1,2,3,4,5,6],*]
  a3.ray=mray
  END
  1: BEGIN
  ;
  ; Minimization algorithm (interMethod EQ 1)
  ;
  ;===========================================================
  IF Keyword_Set(verbose) THEN print,'MD: Intersection by minimization algorithm'
  ; 
  ; intercept
  ;

  ; get inputs
  x1= GetShCol(a2,[1,2,3])
  v1= GetShCol(a2,[4,5,6])
  flag= GetShCol(a2,10)

  ; initialize outputs
  pnt = Replicate({xyz:dblarr(3),normal:dblarr(3), flag:1},a2.npoint)

  FOR i=0,a2.npoint-1 DO BEGIN
    _x0=x1[*,i]  ; for common block
    _v0=v1[*,i]  ; for common block
  
    IF Keyword_Set(verbose) THEN BEGIN
      print,' '
      print,'MD >>>>>>>>>>>>> ray index: ',i
    ENDIF

    ; numerical calculation using interceptnurbs.pro
    tmp = intermincalc(n1=n1,verbose=verbose)
  
    IF Keyword_Set(verbose) THEN BEGIN
      print,'MD intercept point: ',tmp
      print,'MD normal: ',n1
    ENDIF
    IF N_Elements(tmp) EQ 3 THEN flg=1 ELSE flg=-1
    ;x2[*,i]=tmp
    ;n2[*,i]=n1
    pnt[i].xyz = tmp
    pnt[i].normal = n1
    pnt[i].flag = flag[i] OR flg
  ENDFOR
  
  ;
  ; calculates reflection
  ;
  ray = GetShCol(a2,[1,2,3,4,5,6,10])
  newray=reflection(ray,pnt,verbose=0)
  ;
  a3=a2
  mray=a3.ray
  mray([0,1,2,3,4,5,9],*)=newray[[0,1,2,3,4,5,6],*]
  a3.ray=mray
  END
  2: BEGIN
  ;
  ; Planes (approximated method)
  ;
  ;===========================================================
  ray = GetShCol(a2,[1,2,3,4,5,6,10])
  pnt = int_guess(_n0,ray,nu=10,nv=10,/returnPoint,verbose=verbose,$
        plot=1)

  ;
  ; calculates reflection
  ;
  newray=reflection(ray,pnt,verbose=0)
  ;
  a3=a2
  mray=a3.ray
  mray([0,1,2,3,4,5,9],*)=newray[[0,1,2,3,4,5,6],*]
  a3.ray=mray
  END
  else: Message,'interMethod not found.'
ENDCASE

;
; writes the mirr.01 file
;
putrays,a3,'minimirr.01'

;
; writes the star.01 file
;
a4 = shrot(a3,angle,axis=1,/rad)
a4 = retrace(a4,dist=image)
putrays,a4,'ministar.01'

;
; Display results
;
XSh_PlotXY,'minimirr.01', 2,3
wait,0.5
XSh_PlotXY,'ministar.01', 1,3


END
