FUNCTION slopes,input,x,y,slopesrms=slopesrms

;+
; NAME:
;	SLOPES
; PURPOSE:
;       This function calculates the slope errors of a surface along the mirror 
;       length Y and mirror width X. 
; CATEGORY:
;	SHADOW tools
; CALLING SEQUENCE:
;	out = slopes(surface_str) or out = slopes(z,x,y)
; INPUTS:
;	surface_str: a structure with the surface in the format {x:x, y:y, z:z) 
;         or
;	x: the width array of dimensions (NX)
;	y: the length array of dimensions (NY)
;	z: the surface array of dimensions (NX,NY)
; OUTPUTS:
;       out: an array of dimension (2,NX,NY) with the slopes errors in rad
;            along Y in out[0,*,*] and along Y in out[1,*,*]
; KEYWORDS
;	slopesrms: set to a named variable to receive the a 4-dim array with 
;       [slopeErrorRMS_X_arcsec,slopeErrorRMS_Y_arcsec, 
;        slopeErrorRMS_X_urad,slopeErrorRMS_Y_urad]
;
; MODIFICATION HISTORY:
;       MSR 1994 written
;       08-04-15 srio@esrf.eu makes calculations in double precision.
;       2014-09-11 documented
;-
;

on_error,2

input_type = (size(input)) ((size(input)) (0)+1)
if input_type eq 8 then begin
  print,'Structure input selected'
  z=Double(input.z)
  x=Double(input.x)
  y=Double(input.y)
endif else begin
  z=Double(input)
  x=Double(x)
  y=Double(y)
endelse
nx = n_elements(z(*,0))
ny = n_elements(z(0,*))
;
if not(keyword_set(x)) then x=findgen(nx)
if not(keyword_set(y)) then y=findgen(ny)
;
slope=dblarr(2,nx,ny)

; 
; slopes in x direction
;
for i=0L,nx-2 do begin
  step = x(i+1) - x(i)
  slope(0,i,*) = atan( (z(i+1,*) - z(i,*) )/step )
endfor
slope(0,nx-1,*) = slope(0,nx-2,*)
;
; slopes in y direction
; 
for i=0L,ny-2 do begin
  step = y(i+1) - y(i)
  slope(1,*,i) = atan( (z(*,i+1) - z(*,i) )/step )
endfor
slope(1,*,ny-1) = slope(1,*,ny-2)
;
;slope = slope*180./!pi*3600.  ; from rad to arc sec
slopermsX = stddev(slope(0,*,*))
slopermsY = stddev(slope(1,*,*))
slopermsXsec = slopermsX*180./!pi*3600.
slopermsYsec = slopermsY*180./!pi*3600.
slopesrms = [slopermsXsec,slopermsYsec, slopermsX*1e6,slopermsY*1e6]

print,' Slope error rms in X direction: ',slopermsXsec,' arc sec'
print,'                               : ',slopermsX*1e6,' microrads'
print,' Slope error rms in Y direction: ',slopermsYsec,' arc sec'
print,'                               : ',slopermsY*1e6,' microrads'
;
return,slope
end
