PRO Xop_Spawn,command,Group=group,OnlyGetCommand=onlyGetCommand, $
  PreCommand=preCommand,SetWindowsHome=setWindowsHome, $
  CleanFiles=cleanFiles,NoPath=noPath
 
;+
; NAME:
;       XOP_SPAWN
;
; PURPOSE:
;       Spawns XOP programs.
;       The program prepends the directory defined by the environmental
;       variable XOP_BINARIES to the command string.
;       If the environment variable XOP_BINARIES_DEBUG is set,
;       information windows are presented for debugging purposes.
;
; CATEGORY:
;       XOP
;
; CALLING SEQUENCE:
;       XOP_SPAWN,command
;
; INPUTS:
;       command = a string (or array of strings) with command(s) to run.
;
;
; KEYWORD PARAMETERS:
;       Group:  The widget ID of the group leader of the widget. If this
;               keyword is specified, the death of the group leader results in
;               the death of widgets created by XOP_SPAWN.
;	OnlyGetCommand: set this keyword to a named variable to get the
;		command to be run. When this keyword is used, XOP_SPAWN
;		does not spawn the process.
;	PreCommand: A command to be executed previously to the main 
;		block (command). The directory path is not added to 
;		this preCommand. Useful to define envoronment variables.
;	SetWindowsHome: if set, the XOP_HOME environment variable
;		(i.e., set XOP_HOME=c:\...) is added before execution.
;		This keyword is only used for Windows systems.
;	CleanFiles: a list of files to be deleted. This is the first 
;		process that is performed.
;	noPath: Set this keyword to avoid adding the binary path.
;
; OUTPUTS:
;       None
;
; SIDE EFFECTS:
;
; PROCEDURE:
;       Straightforward.
;
; USE:
;       This procedure is to be used by XOP to spawn a command 
;	or array of commands. It first adds the standard XOP
;	binary path to the application name and then starts it.
;
; EXAMPLES:
;       xop_spawn,'epath < epath.inp'
;
; MODIFICATION HISTORY:
;       Written by:     Manuel Sanchez del Rio (srio@esrf.fr) 98/11/05
;	99-01-12 srio@esrf.fr adds preCommand keyword. Allows command to be an array.
;	00-06-13 srio@esrf.fr adds cleanFiles keyword
;	00-06-16 srio@esrf.fr adds Mac support. By now the applescript
;		xop_run.mac and its xop_run.inp file must be on the current directory.
;	2006-04-14 srio@esrf.fr adds noPath keyword.
;	2011-03-18 dejus@aps.anl.gov, added double quotes for commands on WINDOWS
;	2011-04-05 srio@esrf.eu mv double quotes to directory path only.
;	2013-04-23 srio@esrf.eu added /sh kw for unix spawn (to fix problems in linux)
;	2014-01-13 dejus@aps.anl.gov, added check of pipes for Windows and prepends
;                  the binary path to all commands in the pipe. Minor other edits.
;-
Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(Dialog_Parent=group, $
   /Error,'XOP_SPAWN: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   Return
ENDIF

osFamily = SDep()

;
; Clean files
;
IF Keyword_Set(cleanFiles) THEN Delete_Files,cleanFiles,Group=group

;
; For MacOS, remove the "<" sign if it exists
;
IF osFamily EQ 'MACOS' THEN BEGIN
  FOR i=0L,N_Elements(command)-1 DO BEGIN
    line = command[i]
    itmp = StrPos(line,'<')
    IF itmp[0] NE -1 THEN BEGIN
      line_new = StrSubstitute(line,'<','')
      command[i]=line_new
    ENDIF
  ENDFOR
ENDIF


;
; find the location of the binary directory
;
IF osFamily NE 'MACOS' THEN BEGIN
  bindir = Xop_GetEnv('XOP_BINARIES')
  IF StrCompress(bindir,/Remove_All) EQ '' THEN BEGIN
    itmp = Dialog_Message(/Error,/Cancel,Dialog_Parent=group,$
      ['When running: '+command+':', $
       'Directory with XOP binaries was not found.',$
       'It should be defined by the XOP_BINARIES environmental variable.',$
       'Do you want to define it now for the current XOP session?'] )
    IF itmp EQ 'Cancel' THEN Return
    bindir = Dialog_Pickfile(group=group,Title=$
     'Please select the directory where the XOP binaries reside',/Dir)
    IF bindir EQ '' THEN RETURN
    Xop_SetEnv,'XOP_BINARIES='+bindir
  ENDIF

;  ;
;  ; In windows, replace "Program Files" by "progra~1" to avoid problems...
;  ;
;  IF osFamily EQ 'WINDOWS' THEN BEGIN 
;     ;bindir=StrSubstitute(bindir,'Program Files','progra~1')
;     delim = '\'
;     n = StrParse(bindir,delim,list)
;     bindir2 = ''
;     FOR i=0,n DO BEGIN
;       IF StrEgex(list[i],' ') NE -1 THEN list[i]= $
;         StrMid(StrCompress(list[i],/Rem),0,6)+'~1'
;       bindir2 = bindir2+list[i]+delim
;     ENDFOR
;     bindir=bindir2
;  ENDIF
  ;
  ; adds directory separator
  ;
  IF StrCompress(bindir,/Remove_All) NE '' THEN BEGIN
    bindir=bindir+SDep(/DS)
    IF osFamily EQ 'WINDOWS' THEN BEGIN
	bindir = '"' +bindir +'"' ; add double quotes to command for WINDOWS
    ENDIF
  ENDIF
ENDIF ELSE bindir=''

IF Keyword_Set(nopath) THEN bindir=''

;
; gets first word in command (for displaying purposes).
;
ipos = StrPos(command[0],' ')
IF ipos NE -1 THEN ccc = StrMid(command[0],0,ipos) ELSE ccc = command[0]

;
; add path to command
; 
IF osFamily NE 'MACOS' THEN command=bindir+command

;
; if SetWindowsHome keyword defined, add it.
; 
IF osFamily EQ 'WINDOWS' THEN $
  IF Keyword_Set(setWindowsHome) THEN command = $
  ['set XOP_HOME='+Xop_GetEnv('XOP_HOME'),command]
;
; if preCommand keyword is defined, add it.
; 
IF Keyword_Set(preCommand) THEN command = [preCommand,command]

;
; if debug flag is set, show command
;
idebug=0
IF StrCompress(Xop_GetEnv('XOP_BINARIES_DEBUG'),/Remove_All) NE '' THEN BEGIN
  idebug = 1
  itmp = Dialog_Message(/Info,Dialog_Parent=group,/Cancel,$
    ['Debugging message: Start to run: ','',command])
  IF itmp EQ 'Cancel' THEN Return
ENDIF

;
; Spawn message
;
IF N_Elements(OnlyGetCommand) NE 0 THEN  BEGIN
  onlyGetCommand=command
  IF idebug EQ 1 THEN itmp = Dialog_Message(/Info,Dialog_Parent=group,$
    'Debugging message: Process completed (not run).')
  RETURN
ENDIF 


Print,'############# Start Running '+ccc+' ####################'

CASE osFamily OF
  'WINDOWS': BEGIN
	  ; add bindir to all commands in a pipe
	  s1 = StrSplit(command,'|',/extract)
	  IF N_ELEMENTS(s1) gt 1 THEN command = StrJoin(StrTrim(s1,1),' |'+bindir)
	  IF idebug EQ 1 THEN command=[command,'pause'] ELSE command=[command,'rem pause']
	  ;IF N_Elements(command) EQ 1 THEN BEGIN
	  ;  message,/info,'Executing: '+command
	  ;  spawn,command 
	  ;ENDIF ELSE BEGIN
	  OpenW,unit,'xop_spawn.bat',/Get_Lun
	  FOR i=0L, N_Elements(command)-1 DO PrintF,unit,command[i]
	  Free_Lun,unit
	  message,/info,'Executing: xop_spawn.bat'
	  Spawn,'xop_spawn.bat'
	  Wait,0.2
	  IF idebug NE 1 THEN BEGIN ; delete xop_spawn.bat
	    message,/info,'Deleting: xop_spawn.bat'
	    OpenW,unit,'xop_spawn.bat',/Get_Lun,/Delete
	    Free_Lun,unit
	  ENDIF
	  ;
	  ;ENDELSE
	END
  'MACOS' : BEGIN
	pwd=''
	CD,Current=pwd
	Delete_Files,['xop_run.inp']
	OpenW,unit,'xop_run.inp',/Get_Lun
	FOR i=0L, N_Elements(command)-1 DO PrintF,unit,command[i]
	Free_Lun,unit
	message,/info,'Executing AppleScript xop_run.mac that calls xop_run.inp'
	Spawn,pwd+'xop_run.mac'
	;Wait,0.2
	;IF idebug NE 1 THEN BEGIN ; delete xop_run.inp
	;  message,/info,'Deleting: xop_run.inp'
	;  OpenW,unit,'xop_run.inp',/Get_Lun,/Delete
	;  Free_Lun,unit
	;ENDIF
	;
	END
  else : BEGIN ; UNIX
	FOR i=0L,N_Elements(command)-1 DO BEGIN
	  message,/info,'Executing: '+command[i]
	  spawn,command[i],/sh
	ENDFOR
	END
ENDCASE
Print,'############# End Running '+ccc+' ####################'

IF idebug EQ 1 THEN itmp = Dialog_Message(/Info,Dialog_Parent=group,$
    'Debugging message: Process run completed.')
 
END
