; +++++++++++++++++++++++++++++++++++
; Documentation on XPRINT application
; +++++++++++++++++++++++++++++++++++
; 
; 
; This application permits to print the current graphic or
; to create a graphic (PostScript) file. 
; 
; Two printing options are available:
;   i) Create a PostScript file and optionally send it to 
;      to a printer or to another application. 
;      (Use Print to: PS Printer/Application or
;           Print to: PS File) 
;   ii) Send graphic directly to a system defined printer
;       without creating intermediate files. 
;      (Use Print to: System Printer)
; 
; 
; The choice of the printing mode
; ===============================
; The choice between the options described above depends on your 
; system and on your printer.
; 
; The use of PostScript file is recommended when a PostScript 
; printer is available under Unix. It can also be used under 
; Windows but the output should be send to a PostScript
; visualization program (e.g., GSView) otherwise special software 
; is required for spawning the printing command to a PostScript
; printer from a MS-DOS shell.
; PostScript printing allows better quality graphics and a more
; suitable customization of fonts and aspect ratio.
; 
; The System Printer is recommended when no PostScript printer is
; available or when using special requirements (i.e., printing many 
; copies). It uses the IDL "printer" device. For more details, consult 
; the IDL documentation. 
; 
; 
; About PostScript Printing
; =========================
; The option "Print To: PS Printer/Application" permits to send 
;  a temporary file. This file name is described with the characters
;  %F in the command line. Its real name is xprint_tmp.ps and it is
;  deleted after printing. The file is created in the current directory,
;  therefore, write-permission must be set.
;  Examples (in Unix):
;    lpr -Pa4dc222 %F   /*send the temporary file to the printer a4dc222 */
;    ghostview %F      /*injects the temporary file into the ghostview 
;                        application (must be installed) */
;    cat %F            /*shows the contents of the temporary PS file in 
;                        the Standard output */
;    cat %F | mailx -s "PS file" srio@esrf.fr /* emails the file to the routine
;                      author */
; 
; The option "Print To: PS FIle" permits to create a PostScript file.
; 
; Note: The aspect of the printed graphic may differ from the one showed in
;   the screen. In order to obtain the desired quality, few iterations
;   may be needed (changing fonts, aspect ratios, etc.).
; 
; 
; About Fonts in PostScript-printing
; ==================================
; When printing by using a PS file, three font options are available:
;   i) Use vector fonts (Hersey fonts). It is possible some applications 
;      (e.g. xop/xplot)  to change the vector font. It is also possible 
;      to change the fonts and apply special characters by using the 
;      "embedded formatting command" in the title and label strings (see
;      below).
;   ii) Use Hardware fonts (i.e., PostScript font). There is no interactive
;      way to change it by now.
;   iii) Use TrueType fonts. The "embedded formatting commans" are also
;      available. 
; 
; 
; Embedded Formatting Commands
; ============================
; 
; [text extracted from the IDL documentation. Consult it for more details] 
; When you use vector, TrueType, and some device fonts, text strings can 
; include embedded formatting commands that facilitate subscripting, 
; superscripting, and equation formatting. The method used is similar to 
; that developed by Grandle and Nystrom (1980). Embedded formatting commands 
; are always introduced by the exclamation mark, (!). (The string "!!" is 
; used to produce a literal exclamation mark.)
; 
; 
; Changing fonts within a string:
; -------------------------------
; 
; Command	SelectVector Font	SelectTT Font		SelectPS Font	
; !3	Simplex Roman (default)	Helvetica 		Helvetica	
; !4	Simplex Greek		Helvetica Bold		Helvetica Bold	
; !5	Duplex Roman		Helvetica Italic	Helvetica Narrow	
; !6	Complex Roman		Helvetica Bold Italic	Hevetica Narrow Bold Oblique	
; !7	Complex Greek		Times			Times Roman	
; !8	Complex Italic		Times Italic		Times Bold Italic	
; !9	Math/special characters	Symbol			Symbol	
; !M	Math/special characters Symbol			Symbol
; !10	Special characters	Symbol *		Zapf Dingbats	
; !11(!G)	Gothic English		Courier			Courier	
; !12(!W)	Simplex Script		Courier Italic		Courier Oblique	
; !13	Complex Script 		Courier Bold		Palatino 	
; !14	Gothic Italian		Courier Bold Italic	Palatino Italic	
; !15	Gothic German		Times Bold		Palatino Bold	
; !16	Cyrillic		Times Bold Italic	Palatino Bold Italic	
; !17	Triplex Roman		Helvetica *		Avant Garde Book	
; !18	Triplex Italic		Helvetica *		New Century Schoolbook	
; !19				Helvetica *		New Century Schoolbook Bold	
; !20	Miscellaneous		Helvetica *		Undefined User Font	
; !X	Revert to entry font	Revert to entry font	Revert to the entry font
; 
; * The font assigned to this index may be replaced in a future release of IDL.	
; 
; Positioning Commands
; --------------------
; 
; The positioning and other font-manipulation commands are described below.
; Command Action	
; !A	Shift above the division line.	
; !B 	Shift below the division line.	
; !C	"Carriage return," begins a new line of text. Shift back to the 
; 	starting position and down one line.	
; !D	Shift down to the first level subscript and decrease the character 
; 	size by a factor of 0.62.	
; !E	Shift up to the exponent level and decrease the character size by 
; 	a factor of 0.44.
; !I	Shift down to the index level and decrease the character size by 
; 	a factor of 0.44.
; !L	Shift down to the second level subscript. Decrease the character 
; 	size by a factor of 0.62.	
; !N	Shift back to the normal level and original character size.	
; !R	Restore position. The current position is set from the top of the 
; 	saved positions stack.
; !S	Save position. The current position is saved on the top of the saved 
; 	positions stack.
; !U	Shift to upper subscript level. Decrease the character size by a 
; 	factor of 0.62.
; !X	Return to the entry font.	
; !!	Display the ! symbol.	
; 
;
; Unicode formatting
; ------------------
;
; A new embedded formatting command, !Z, within text strings, allows
; character glyphs to be specified by their Unicode values for display in
; graphic windows. This new capability allows applications to display
; characters that are not directly accessible via the keyboard, and are
; therefore difficult to include in string variables. 
; 
; Unicode is an internationally recognized standard for mapping integer
; values to corresponding character glyphs. For more information on the
; Unicode standard, refer to www.unicode.org.
; The exhaustive list of Unicode symbols is at:
; ftp://ftp.unicode.org/Public/UNIDATA/UnicodeData-Latest.txt
; 
; The new !Z formatting command applies only to TrueType fonts (it has no
; real meaning for Hershey fonts or hardware fonts). 
; 
; !Z is handled very much like other embedded formatting commands. The
; syntax is!Z(u0,u1,...,un)
;
; Examples: !Z(2211) -Sumation-, !Z(221a) -Sqrt-, !Z(221e) -Infinity-,
;           !Z(222b) -Integral-, !Z(00c5) -Angstrom sign-
; 
;
;
; Acknowdledgement
; ================
; Xprint uses the PS_FORM routine (slightly modified and renamed to PS_FORM1) 
; from:
;    FANNING SOFTWARE CONSULTING
;    David Fanning, Ph.D.
;    2642 Bradbury Court
;    Fort Collins, CO 80521 USA
;    Phone: 970-221-0438
;    E-mail: davidf@dfanning.com
;    Coyote's Guide to IDL Programming: http://www.dfanning.com
; 
; 
;
; ++++++++++++++++++++++++++++++++
; Interactive use of XPRINT in IDL
; ++++++++++++++++++++++++++++++++
;
; NAME:
;	XPRINT
;
; PURPOSE:
;	This procedure prints a text or some graphics to the printer or
;		to file
;
; CATEGORY:
;	Widgets, general utility
;
; CALLING SEQUENCE:
;	XPRINT, input, Keyword_parameters
;
; OPTIONAL INPUTS:
;	input: same meaning as buffer (see keywords section)
;	
; KEYWORD PARAMETERS:
;	BUFFER:	In case of printing of graphics, you have to
;			store the draw commands in this vector of strings.
;			In case of text you have to omit this keyword.
;
;	DATA:	The data to be printed. For text it is only a string or
;		a vector of strings. For graphics it is usually a matrix
;		which stores the data used by the commands 'BUFFER'.
;
;	GROUP:	The widget ID of the group leader widget.
;	COLOR:	When set to 1, then set the ps device for color output
;	FILE:	name of a file on disk to be printed. Example
;		Xprint,FILE='a.dat',/NOTYPE
;	NORESIZE: avoids to adjust the character size and window
;		Horizontal?Vertical aspect ratio.
;	PARMS = idl structure containing the parameter data needed by BUFFER
;	MODAL = Set this keyword to start the widget as modal widget. 
;		In this case GROUP must be provided. Default: Modal=0
;	NO_BLOCK = Set this keyword to start the widget without blocking
;		the IDL command prompt. Default: NO_BLOCK=1	
;
; OUTPUTS:
;	Normally: the printed text or graphics
;	If you print to file: the file
;	The routine modifies the variables of the printdefaults common
;	block.
;
; OPTIONAL OUTPUTS:
;
; COMMON BLOCKS:
;	common xprint1,str,old_flag
;	str:	the structure containing the parameters 
;	old_flag: If set to 1 the old Xprint version is uses instead
;	 of Xprint 2.0
;
; SIDE EFFECTS:
;
; RESTRICTIONS:
;	You can use this routine if the 'BUFFER' commands use
;	only one variable whose name has to be data. The program gets
;	the value if you set the DATA keyword. The other values have
;	to be transferred as constants.
;	It is possible to use variables in the buffer, but this variables 
;	must have a name included in the list: i,j,a0,...a10,b0,...b10,
;	c0,...,c10 (this is due to the fact that EXECUTE cannot create
;	variables on the fly. The variable must be pre-initializad).
;	The buffer is executed via the EXECUTE command, so it assumes
;	the own limitations of EXECUTE
;
; PROCEDURE:
;
; EXAMPLE:
;	Create a surface plot:
;       img = dist(100)
;       surface,img
;
;	For printing we use the next:
;
;	img = dist(100)
;	buffer=['surface,data']
;	xprint,BUFFER=buffer,DATA=img
;
; MODIFICATION HISTORY:
; 	Written by:	Szabolcs Kesmarki, July 1993.
;	93-11-18 M. Sanchez del Rio adds default common block variables
;			and adds the COLOR keyword
;	93-11-23 M. Sanchez del Rio adds FILE keyword
;	94-02-22 MSR renames it from printing to xprint, and rewrites
;		it in a shorter procedure
;	95-08-18 MSR makes to read the default printer name from
;		the corresponding environment variable
;	96-01-17 MSR makes to read the default printer name from
;		a call to GetSysPrinter()
;	97-02-13 MSR changes the filename when printing to avoid
;		rewriting the xplot.ps file that the user may want to keep. 
;		First attemp to Windows 95 port. Use "message,/info" for 
;		messaging.
;	97-03-05 bug fixed: setting encapsulated and then non-encapsulated 
;		didn't work. Now device,encapsulated=0 is set when printing.
;	97-10-16 srio@esrf.fr makes the text fields larger and maintain
;		the window position in the screen for successive calls (added 
;		frame0 to common block). Define internal variables for
;		buffer executions (a0,...,c10,i,j). uses sdep(). Adds a
;		1 second delay for printing (problems with Solaris).
;	97-11-18 srio@esrf.fr included changes from R. Dejus: 
;       	96-05-20 RJD, added wait statement before executing rm of 
;		xprint.ps Removed resizing of characters and window size.
;		NOTE: the wait is for 1 sec and to avoid resizing
;		the new keyword NORESIZE must be used.
;	98-06-15 srio@esrf.fr fix a bug when printing encapsulated ps.
;	98-12-21 srio@esrf.fr adds the PARMS keyword, as in Xwindow
;	--------
;	99-01-07 srio@esrf.fr completely rewrites xprint. Now xprint 2.0
;	99-03-03 srio@esrf.fr fix bugs fixed. Allows not deleting 
;		temporary file.
;	02-03-12 srio@esrf.fr removes call to xprint_old
;	03-01-07 srio@esrf.fr removes bug message for idl 5.2
;	2007-11-09 srio@esrf.eu adds modal and no_block keywords. Start
;		xprint by default in modal=0 to allow setting the xprinter.
;		Small updates. Version 2.01
;	2008-09-25 srio@esrf.eu fixes a bug: lpr -P -> lpr if no PRINTER 
;		defined. 
;
;-

;
;====================================================================
;
FUNCTION xprint_version
RETURN,'2.01'
END ; xprint_version

;
;====================================================================
;

pro xprint_set,wids,str

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!error_state.msg
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'XPRINT_SET: error caught: '+!error_state.msg)
   catch, /cancel
   on_error,2
   return
endif

Widget_Control,wids.destination, Set_Value=str.destination
Widget_Control,wids.command, Set_Value=str.command
Widget_Control,wids.deleteFile, Set_Value=str.deleteFile
Widget_Control,wids.file, Set_Value=str.file

Widget_Control,wids.orientation, Set_Value=str.orientation
Widget_Control,wids.color, Set_Value=str.color
Widget_Control,wids.encapsulated, Set_Value=str.encapsulated
Widget_Control,wids.font, Set_Value=str.font
Widget_Control,wids.aspect, Set_Value=str.aspect

end ; xprint_set

;
;====================================================================
;
pro xprint_map,wids,str

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!error_state.msg
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'XPRINT_MAP: error caught: '+!error_state.msg)
   catch, /cancel
   on_error,2
   return
endif

case str.destination of
  0: begin ; PS printer/application
	Widget_Control,wids.ps_base,map=1
	Widget_Control,wids.command_base,sensitive=1
	Widget_Control,wids.file_base,sensitive=0
	;Widget_Control,wids.encapsulated,sensitive=0
	Widget_Control,wids.encapsulated,sensitive=1
	Widget_Control,wids.system_base,sensitive=0
	if str.aspect EQ 1 then $
	    Widget_Control,wids.ps_form,sensitive=1 else $
	    Widget_Control,wids.ps_form,sensitive=0
	if str.font EQ 2 then $
	    Widget_Control,wids.set_tt_font,sensitive=1 else $
	    Widget_Control,wids.set_tt_font,sensitive=0
	end
  1: begin ; PS file
	Widget_Control,wids.ps_base,map=1
	Widget_Control,wids.command_base,sensitive=0
	Widget_Control,wids.file_base,sensitive=1
	Widget_Control,wids.encapsulated,sensitive=1
	Widget_Control,wids.system_base,sensitive=0
	if str.aspect EQ 1 then $
	    Widget_Control,wids.ps_form,sensitive=1 else $
	    Widget_Control,wids.ps_form,sensitive=0
	if str.font EQ 2 then $
	    Widget_Control,wids.set_tt_font,sensitive=1 else $
	    Widget_Control,wids.set_tt_font,sensitive=0
	end
  2: begin ; System printer
	Widget_Control,wids.ps_base,map=0
	Widget_Control,wids.command_base,sensitive=0
	Widget_Control,wids.file_base,sensitive=0
	Widget_Control,wids.system_base,sensitive=1
	end
endcase

end ; xprint_map

;
;====================================================================
;
FUNCTION xprint_default_str

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!error_state.msg
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'XPRINT_DEFAULT_STRUCTURE: error caught: '+!error_state.msg)
   catch, /cancel
   on_error,2
   RETURN,0
endif
;
; command 
;
if not(keyword_set(Command)) then begin
  PrinterName = ''
  Command = ''
  CASE sdep() OF
    'UNIX': BEGIN
      ;PrinterName = GetSysPrinter(/Verbose)
      Printername=getenv('PRINTER')
      IF StrCompress(PrinterName,/Rem) EQ '' THEN $
                     PrinterName = GetSysPrinter(/Verbose)
      IF StrCompress(PrinterName,/Rem) EQ '' THEN BEGIN
        IF !version.os EQ 'linux' THEN command = 'lpr  %F' $
	  ELSE command = 'lp %F'
      ENDIF ELSE BEGIN
        IF !version.os EQ 'linux' THEN command = 'lpr -P'+PrinterName+' %F' $
	  ELSE command = 'lp -d'+PrinterName+' %F'
      ENDELSE
      destination = 0
      END
    'WINDOWS': BEGIN
      Command = 'start %F'
      destination = 2
      END
    'MACOS': BEGIN
      END
    else:
  ENDCASE
endif
;
; others
;
orientation = 0
encapsulated = 0
CASE !p.font OF
 -1: font = 0  ; Hersey
 0: font = 1   ; hardware=PostScript
 1: font = 2   ; TrueType
 else: font = 0
ENDCASE
aspect = 0
ps_form_str = ps_form1(/Initialize)
; 
; TrueType font
;
tt_list='Error loading TT fonts!!'
tt_font_manager,Get_List=tt_list
tt_list=['0',tt_list]
tt_charRatio=1.0
tt_preserve=['0','Yes','No']
; 
; structure
;
xprintstr = { destination:destination, Command:command, $
  deleteFile:0, $
  File:'xprint.ps', orientation:orientation, color:0, $
  encapsulated:encapsulated, font:font, aspect:aspect, $
    ;ps_form_str:ps_form_str, $
    ps_form_xsize:ps_form_str.xsize, $
    ps_form_xoff:ps_form_str.xoff, $
    ps_form_ysize:ps_form_str.ysize, $
    ps_form_yoff:ps_form_str.yoff, $
    ps_form_filename:ps_form_str.filename, $
    ps_form_inches:ps_form_str.inches, $
    ps_form_color:ps_form_str.color, $
    ps_form_bits_per_pixel:ps_form_str.bits_per_pixel, $
    ps_form_encapsulated:ps_form_str.encapsulated, $
    ps_form_landscape:ps_form_str.landscape, $
  tt_preserve:tt_preserve, tt_list:tt_list, tt_charRatio:tt_charRatio }

;
;----------------------- WARNING !! -------------------------------
; If this section is modified, please check also dabax_defaults.pro
; If this section is modified, please check also xop_defaults.pro
;------------------------------------------------------------------
;

;
; Overwrite data with values from defaults files
;
sep = sdep(/ds)
filedir = [Xop_GetEnv('XOP_HOME')+sep+'defaults'] 
filedir = [filedir]
IF Xop_GetEnv('XOP_DEFAULTS_DIR') NE '' THEN $
   filedir = [filedir,Xop_GetEnv('XOP_DEFAULTS_DIR')]

nfiles = n_elements(filedir)
if nfiles EQ 1 then filedir = [filedir]
tmp=0
for i=0,nfiles-1 do begin
  file = filedir[i]+sep+'xprint.xop'
  tmp = checkfile(file)
  if tmp EQ 1 then begin
    str_par = Xop_Input_Load(inputFile=file)
    ;IF N_Tags(out.parameters) NE 0 THEN BEGIN
    tmp = xprintstr
    Copy_Structure,str_par, tmp, /OnlyFirstField
    xprintstr = tmp
    message,/info,'Defaults from file: '+file+'
  endif
endfor

RETURN,xprintstr
END ; xprint_default_str
;
;====================================================================
;
PRO xprint_event,event
common xprint1,str,old_flag

; define user variables 
i=0 & j=0 & k=0 
a0=0 & a1=0 & a2=0 & a3=0 & a4=0 & a5=0 & a6=0 & a7=0 & a8=0 & a9=0 & a10=0
b0=0 & b1=0 & b2=0 & b3=0 & b4=0 & b5=0 & b6=0 & b7=0 & b8=0 & b9=0 & b10=0
c0=0 & c1=0 & c2=0 & c3=0 & c4=0 & c5=0 & c6=0 & c7=0 & c8=0 & c9=0 & c10=0
;
catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!error_state.msg
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'XPRINT_EVENT: error caught: '+!error_state.msg)
   catch, /cancel
   on_error,2
   goto,out
endif
thisDevice = !D.name
thisFont = !p.font
;
;
Widget_Control, event.id, get_UValue=eventUValue
Widget_Control, event.id, get_Value=Value
 
if n_elements(eventuvalue) EQ 0 then eventuvalue = ''
if not(keyword_set(eventuvalue)) then eventuvalue = ''
 
stateid = Widget_Info(event.handler,/Child)
Widget_Control, stateid, get_UValue=state, /No_Copy
 
; help,eventuvalue

Case eventUValue OF
    'ACCEPT': BEGIN
	Widget_Control,state.wids.command, Get_Value=tmp
	state.str.command =  tmp(0)
	Widget_Control,/HourGlass
	if state.str.destination EQ 2 then begin
		commands = "set_plot,'printer'"
	endif else begin
	  commands = ["set_plot,'ps'"]
	  case state.str.aspect of
	   0: commands = [commands, "device,landscape = "+$
		strcompress(state.str.orientation,/Rem)]
	   1: BEGIN
		;tmp0 = state.str.ps_form_str
		;tmp0.landscape = state.str.orientation
		tmp0 = ps_form1(/Initialize)

		tmp0.xsize = state.str.ps_form_xsize
		tmp0.xoff = state.str.ps_form_xoff
		tmp0.ysize = state.str.ps_form_ysize
		tmp0.yoff = state.str.ps_form_yoff
		tmp0.filename = state.str.ps_form_filename
		tmp0.inches = state.str.ps_form_inches
		tmp0.color = state.str.ps_form_color
		tmp0.bits_per_pixel = state.str.ps_form_bits_per_pixel
		tmp0.encapsulated = state.str.ps_form_encapsulated
		tmp0.landscape = state.str.ps_form_landscape
		
		commands = [commands, "device,_extra=tmp0"]
		END
	   else:
	  endcase
	  commands = [commands, $
	    "device,encapsulated = "+strcompress(state.str.encapsulated,/Rem)]
	  if state.str.color then commands = [commands,"device,/color,bits=8"] $
		else commands = [commands,"device,color=0"]
	  file = 'xprint_tmp.ps'
	  if state.str.destination EQ 1 then begin
		Widget_Control,state.wids.file,Get_Value=file
	        file = file(0)
		state.str.file = file
	  endif
	  if state.str.destination EQ 1 then begin
	    if checkfile(file) EQ 1 then begin
	       itmp = Dialog_Message(Dialog_Parent=event.top,$
		  ['File exists: '+file,'Overwrite it?'], /Question)
	       if itmp EQ 'No' then begin
		  itmp = Dialog_Message(/Info,Dialog_Parent=event.top,$
		    'Enter a new file name and "Accept"')
		  goto,out
	       endif
	    endif
	  endif
	  commands = [commands,"device,file='"+file+"'"]
	  CASE state.str.font OF
	    0: commands = [commands,'!p.font=-1'] ; Hersey
	    1: commands = [commands,'!p.font=0']  ; Hardware
	    2: BEGIN				  ; TrueType
		commands = [commands,'!p.font=1']
		; here the TT dependencies
		IF StrCompress(state.str.tt_preserve[0],/Rem) NE '0' THEN BEGIN
		  commands=[commands,'tt_font_manager,set_index='+$
			state.str.tt_list[0] ]
		ENDIF
		IF state.str.tt_charRatio NE 1 THEN BEGIN
		  commands=[commands,'tt_font_manager,CharRatio='+$
			String(state.str.tt_charRatio,Format='(F5.2)')+$
			',oldChar=oldc']
		ENDIF
		END
	  ENDCASE
	  ;if state.str.font EQ 0 then commands = [commands,'!p.font=-1'] else $
          ;  commands = [commands,'!p.font=0']
	  if state.str.aspect EQ 0 then begin
	    ch = [!d.x_ch_size, !d.y_ch_size] / float([!d.x_size, !d.y_size])
	    ratio = float(!d.x_size)/float(!d.y_size)

	    ; set charsize Proportional char size (aspect = same) 
	    ; ONLY with HERSEY fonts
	    IF state.str.font EQ 0 THEN $
	     commands = [commands,$
	     "device, set_character_size=!d.x_size*"+vect2string(ch), $
	     "device,xsize = !d.y_size/1000. * "+strcompress(ratio,/rem)]
	  endif
	endelse
	data = state.data
	parms = state.parms
	commands = [commands,state.buffer,'device,/close']
	IF state.str.font EQ 2 AND state.str.tt_charRatio NE 1 THEN BEGIN
	  commands=[commands,'Device,Set_Character_Size=oldc']
	ENDIF
	oldc=0
	for i=0,n_elements(commands)-1 do begin
	  ; message,/info,'Executing: '+commands[i]
if i EQ 2 then device,_extra=tmp0
	  itmp = execute(commands[i])
	  if itmp NE 1 then begin
	    tmp = Dialog_Message(/Error,/Cancel,Dialog_Parent=event.top, $
		['Error executing: '+commands(i), $
		 ' from a set of commands: ','>> '+commands,' ',$
		 '(line number: '+StrCompress(i+1,/Rem)+')',$
		 ' ',' ',$
		 ' Continue?'])
	    if tmp EQ 'Cancel' then goto,out
	  endif
	endfor
	if state.str.destination EQ 0 then begin
	  command = state.str.command
	  IF strPos(command,'%F') LT 0 THEN command=command+' %F'
	  command = StrSubstitute(command, '%F', 'xprint_tmp.ps')
          
	  message,/info,'Spawning: '+command
	  a=''
	  if sdep() EQ "WINDOWS" then spawn,command,a else spawn,command,a,/sh
	  wait,1.5
	  IF a[0] NE '' THEN itmp = $
	    Dialog_Message(/Info,Title='Spawn output',a,Dialog_Parent=event.top)
	  IF state.str.deleteFile EQ 0 THEN Delete_Files,'xprint_tmp.ps'
	endif
	
	; put data in common block
	str = state.str
	; destroy widgets
	Widget_Control,event.handler,/destroy
	END
    'CANCEL': BEGIN
	Widget_Control,event.handler,/destroy
	END
    'HELP': BEGIN
	XHelp,'xprint',group=event.top,/NoMenuBar
	END
    'DESTINATION': BEGIN
	state.str.destination = event.value
	; when sending to printer, force not-eps
	IF event.value EQ 0 THEN BEGIN 
	  Widget_Control,state.wids.encapsulated, Set_value=0
	  state.str.encapsulated = 0
	ENDIF
	xprint_map,state.wids,state.str
	END
    'ORIENTATION': state.str.orientation = event.value
    'COLOR': state.str.color = event.value
    'ENCAPSULATED': state.str.encapsulated = event.value
    'FONT': BEGIN
	state.str.font = event.value
	xprint_map,state.wids,state.str
	END
    'ASPECT': BEGIN
	state.str.aspect = event.value
	xprint_map,state.wids,state.str
	END
    'DELETEFILE': BEGIN
	state.str.deletefile = event.value
	; xprint_map,state.wids,state.str
	END
;    'FILENAME': BEGIN
;	tmp=''
;	Widget_Control,state.wids.file,Get_Value=tmp
;	state.str.file = tmp[0]
;	END
    'PS_FORM': BEGIN
	cancel=0
	; tmp0 = state.str.ps_form_str
	tmp0=ps_form1(/Initialize)

	tmp0.xsize = state.str.ps_form_xsize
	tmp0.xoff = state.str.ps_form_xoff
	tmp0.ysize = state.str.ps_form_ysize
	tmp0.yoff = state.str.ps_form_yoff
	tmp0.filename= state.str.ps_form_filename
	tmp0.inches= state.str.ps_form_inches
	tmp0.color= state.str.ps_form_color
	tmp0.bits_per_pixel= state.str.ps_form_bits_per_pixel
	tmp0.encapsulated = state.str.ps_form_encapsulated
	tmp0.landscape = state.str.ps_form_landscape
		
	tmp = ps_form1(parent=event.handler,color=state.str.color, $
	  encapsulated=state.str.encapsulated, filename=state.str.file,$
	  landscape=state.str.orientation,cancel=cancel,/Xprint, $
	  Defaults=tmp0 )
	;tmp = ps_form1(parent=event.handler,Defaults=tmp0 )
	if cancel EQ 1 then goto,out
	;state.str.ps_form_str = tmp
	state.str.ps_form_xsize = tmp.xsize
	state.str.ps_form_xoff = tmp.xoff
	state.str.ps_form_ysize = tmp.ysize
	state.str.ps_form_yoff = tmp.yoff
	state.str.ps_form_filename= tmp.filename
	state.str.ps_form_inches = tmp.inches
	state.str.ps_form_color = tmp.color
	state.str.ps_form_bits_per_pixel= tmp.bits_per_pixel
	state.str.ps_form_encapsulated = tmp.encapsulated
	state.str.ps_form_landscape = tmp.landscape

	state.str.file = tmp.filename
	state.str.color = tmp.color
	state.str.orientation = tmp.landscape
	state.str.encapsulated = tmp.encapsulated

	xprint_map,state.wids,state.str
	xprint_set,state.wids,state.str
	END
    'SET_TT_FONT': BEGIN
	tmp = {tt_preserve:state.str.tt_preserve, tt_list:state.str.tt_list,$
	  tt_charRatio:state.str.tt_charRatio}
	action=0
	titles=['Preserve current font','Font name','Character ratio']
	XScrMenu,tmp,Group=event.top,Action=action, Titles=titles,$
	  /NoType,/Interp,Wtitle='Set TT font',Flags=['1','w[0] EQ 1','1']
	IF action EQ 'DONT' THEN GoTo,out
	tmp2 = state.str
	Copy_Structure,tmp,tmp2, Group=event.top, /OnlyFirstField
	state.str = tmp2
	END
    'SETUP': BEGIN
;	IF SDep() EQ 'UNIX' THEN BEGIN
;	  itmp = Dialog_Message(Dialog_Parent=event.top, $
;	    ['Attention: due to a bug in IDL 5.2 (UNIX) the System',$
;	    'Printer Setup window cannot be started from here (modal widget).',$
;	    '','Please start it from XOP/Tools or by typing:',$
;	    ' a0=Dialog_PrinterSetUp() ',$
;	    'in the XOP/Tools/IDL Macro.'])
;	ENDIF ELSE BEGIN
	  itmp = Dialog_PrinterSetUp(Dialog_Parent=event.top,$
		Title='System Printer Setup')
;	ENDELSE
	END
    'JOBS': BEGIN
	itmp = Dialog_PrintJob(Dialog_Parent=event.top,$
		Title='System Printer Jobs')
	END
    'DEFAULT': BEGIN
	itmp = Dialog_Message(Dialog_Parent=event.top, /Question, [$
	  'This option writes the current xprint as personal defaults.',$
	  ' ',$
	  'These defaults are written in a file called xprint.xop ',$
	  'sitting in the directory pointed by the environment variable',$
	  'called XOP_DEFAULTS_DIR, which must be defined. ',$
	  'If XOP_DEFAULTS_DIR is undefined, an error message will be ',$
	  'displayed. ',$
	  ' ',$
	  'This file will be read when initializing the xprint parameters.',$
	  ' ',$
	  'Please confirm you want to use the current setup as default: '])
	IF itmp EQ 'No' THEN GoTo,out
	IF Xop_GetEnv('XOP_DEFAULTS_DIR') EQ '' THEN BEGIN
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	    ['Variable XOP_DEFAULTS_DIR not defined.', $
	     '','Failed to save current setup as default.'])
	  GoTo,out
	ENDIF
        str_par = state.str
	tmp1 = '' & tmp2 = ''
	; update the values not stored in state.str
	Widget_Control,state.wids.command,Get_Value=tmp1
	Widget_Control,state.wids.file,Get_Value=tmp2
	str_par.command = tmp1[0]
	str_par.file = tmp2[0]

        Xop_Input_Save,str_par,Group=event.top, $
          Default='xprint.xop',Comment='; xprint(v'+$
        xprint_version()+') input file on '+SysTime()
	END
;    'COMMAND': BEGIN
;        Widget_Control,state.wids.command, Get_Value=tmp
;        state.str.command =  tmp(0)
;	END
    else: message,/info,'Case not found: '+eventUValue
EndCase


out:
IF Type(stateid) EQ 3 THEN $
  IF Widget_Info(stateid,/Valid_ID) THEN $
  Widget_Control, stateid, Set_UValue=state, /No_Copy
Set_Plot,thisDevice
!P.font=thisFont
end

;
;====================================================================
;
PRO xprint,input, buffer=buffer, Group=group, Command=command, File=file,$
  Data=data, color=color, Parms=parms, Old=old, $
  NoType=noType, $ ; not used. This is only for consistency with old version
  modal=modal,no_block=no_block

common xprint1,str,old_flag


catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!error_state.msg
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	Dialog_Parent=group, $
	'XPRINT: error caught: '+!error_state.msg)
   catch, /cancel
   on_error,2
   return
endif

IF N_Elements(old_flag) EQ 0 THEN old_flag=0
IF N_Elements(no_block) EQ 0 THEN no_block=1
IF N_Elements(modal) EQ 0 THEN modal=0


if keyword_set(buffer) then input=buffer
if n_elements(input) EQ 0 then input=''
if n_elements(data) EQ 0 then data=0
if n_elements(parms) EQ 0 then parms=0
if n_elements(str) GT 0 then begin
  ; message,/info,'Input structure already defined'
  xprintstr = str
endif else begin
  xprintstr = xprint_default_str()
  IF Type(xprintstr) NE 8 THEN BEGIN
    Message,'Error getting default parameters.'
    RETURN
  ENDIF
endelse
if n_elements(color) NE 0 then xprintstr.color = color
if Keyword_Set(File) then xprintstr.file = file

;
; define widgets
;
;if N_elements(group) NE 0 then begin
;  wbase=widget_base(/col,title='xprint '+xprint_version(),Modal=modal,$
;	Group_Leader=group,/Floating,TLB_Frame_attr=9)
;  Widget_Center,wbase,Paren=group
;endif else begin
;  Text = ['Unless you are calling XPRINT from the command line, the',$
;   'GROUP keyword MUST be used for modal operation.',$
;   'Please modify your code if neccesary.']
;  if sdep(/w) then begin
;     itmp = Dialog_message(Dialog_Parent=group,text)
;  endif else begin
;    for i=0,n_elements(text)-1 do message,/info,text(i)
;  endelse
;  wbase=widget_base(/col,title='xprint 2.0')
;endelse

wbase=widget_base(/col,title='xprint '+xprint_version(),Modal=modal,$
	Group_Leader=group,Floating=Keyword_Set(group),TLB_Frame_attr=1)
;Widget_Center,wbase,Paren=group

wtmp = widget_base(wbase) ; to store state

;
; main buttons
;
wtmp0 = widget_base(wbase,/Row) ; Main Buttons
  wtmp = widget_button(wtmp0,Value='Accept',UValue='ACCEPT')
  wtmp = widget_button(wtmp0,Value='Cancel',UValue='CANCEL')
  wtmp = widget_button(wtmp0,Value='Help',UValue='HELP')

;
; destination menu
;
wtmp0 = widget_base(wbase,/Col,/Frame) 

wtmp1 = widget_base(wtmp0,/Col) 
wtmp = Widget_Label(wtmp1,Value='Print To:')
wdestination = cw_bgroup(wtmp1,$
  ['PS Printer/Application','PS File','System Printer'],$
  /Exclusive, Uvalue='DESTINATION',/Return_Index,/Row)

wcommand_base = widget_base(wtmp0,/col) 
  wtmpb =  widget_base(wcommand_base,/Row)
    wtmp = Widget_Label(wtmpb,Value='Print Command (%F means fileName):',/Align_left) ;,Units=2 ;,$
    ; SCR_XSIZE=4)
    ;wcommand = Widget_Text(wtmpb,Value='' ,Uvalue='COMMAND',/Edit, $
    wcommand = Widget_Text(wtmpb,Value='' ,Uvalue='ACCEPT',/Edit, $
	XSIZE=30)
  wtmpb =  widget_base(wcommand_base,/Row)
    wtmp = Widget_Label(wtmpb,Value=$
    'Clean temp file (%F=xprint_tmp.ps) on exit:',$
    /Align_left,Units=2)
    wDeleteFile = cw_bgroup(wtmpb,['Yes','No'],/Exclusive, $
    Uvalue='DELETEFILE',/Return_Index,/Row)

wfile_base = widget_base(wtmp0,/Row) 
wtmp = Widget_Label(wfile_base,Value='  File Name:  ',/Align_left,Units=2,$
  SCR_XSIZE=4)
;wfile = CW_PickFile(wfile_base,Value='' , UValue='FILENAME')
wfile = CW_PickFile(wfile_base,Value='' , UValue='ACCEPT')

wsystem_base = widget_base(wtmp0,/Row) 
  wtmp=widget_label(wsystem_base,Value='System printer: ')
  wtmp = widget_button(wsystem_base,Value='SetUp...',UValue='SETUP')
  wtmp = widget_button(wsystem_base,Value='Jobs...',UValue='JOBS')

wdefault_base = widget_base(wtmp0,/Row) 
  wtmp = widget_button(wdefault_base,Value=$
    'Set current values as default...',UValue='DEFAULT')
;
;
;
wtmp = Widget_Base(wbase)
wps_base = Widget_Base(wtmp,/Col,/Frame)

;
;
;
wtmp0 = widget_base(wps_base,/Row) 
wtmp = Widget_Label(wtmp0,Value='Orientation:',Units=2,XSIZE=5,/Align_left)
worientation = cw_bgroup(wtmp0,['Portrait','Landscape'],/Exclusive, $
  Uvalue='ORIENTATION',/Return_Index,/Row)


wtmp0 = widget_base(wps_base,/Row) 
wtmp = Widget_Label(wtmp0,Value='Print:',Units=2,XSize=5,/Align_left)
wcolor = cw_bgroup(wtmp0,['Greyscale','Color'],/Exclusive, $
  Uvalue='COLOR',/Return_Index,/Row)


wtmp0 = widget_base(wps_base,/Row) 
wtmp = Widget_Label(wtmp0,Value='PS File:',Units=2,XSize=5,/Align_left)
wencapsulated = cw_bgroup(wtmp0,['PS','Encapsulated PS'],/Exclusive,$
  Uvalue='ENCAPSULATED',/Return_Index,/Row)

wtmp0 = widget_base(wps_base,/Row) 
wtmp = Widget_Label(wtmp0,Value='PS Font:',Units=2,XSize=5,/Align_left)
wfont = cw_bgroup(wtmp0,['Vector (Hersey)','PostScript','TrueType'],/Exclusive,$
  Uvalue='FONT',/Return_Index,/Row)
wset_tt_font = widget_base(wtmp0,/Row)  ; to be able to map the next button
wtmp = Widget_button(wset_tt_font,Value='Set...',UVALUE='SET_TT_FONT')

wtmp0 = widget_base(wps_base,/Row) 
wtmp = Widget_Label(wtmp0,Value='Aspect ratio:',Units=2,XSize=5,/Align_left)
waspect = cw_bgroup(wtmp0,['Same as window','Customized'],/Exclusive,$
  Uvalue='ASPECT',/Return_Index,/Row)
wps_form = Widget_button(wtmp0,Value='Set...',UVALUE='PS_FORM')



wids = {destination:wdestination, $
  command_base:wcommand_base, command:wcommand, $
  deleteFile:wDeleteFile, $
  file_base:wfile_base, file:wfile, $
  system_base:wsystem_base, $
  ps_base:wps_base, $
  orientation:worientation, color:wcolor, encapsulated:wencapsulated, $
  font:wfont, aspect:waspect, ps_form:wps_form, set_tt_font:wset_tt_font}

state = {wids:wids, str:xprintstr, buffer:input, data:data, parms:parms}

;
; set defaluts
;
; print,'calling xprint_set'
xprint_set,wids,xprintstr
; print,'back xprint_set'
xprint_map,wids,xprintstr
;
;
;
widget_control,Widget_Info(wbase,/Child),set_uvalue=state,/no_copy

widget_control,wbase,/realize
xmanager,'xprint',wbase,GROUP=group,no_block=no_block
end
