function make_str,text,file=file,header=header,trailer=trailer,Group=group, $
         double=double,long=long
;
; makes an idl structure from data in a text string.
;
; limitations: beware of the maximum string length for a command
; and the max number of tags of the structure:.
;

;+
; NAME:
;
; PURPOSE:
;	This function creates an structure (anonymous) from information in 
;	either a string array or in a file.
;	If a string array is selected, each line contains two or three fields
;	separated by "|". Fists field is the initialization value, second field
;	is the tag name, and third field is the value. When third fields is
;	missing, the value is set equal to the initialization value.
;	When a file is selected, a number of comment lines plus N lines with
;	the same arrangements are used.
;	Lines starting with semicolon (;) are considered as comments.
;
; CATEGORY:
;	General utilities.
;
; CALLING SEQUENCE:
;
;	result = make_str(text)
;
; INPUTS:
;	text: the text (string array) or the file name.
;
; OPTIONAL INPUTS:
;	
; KEYWORD PARAMETERS:
;       FILE: when set, "text" is interpreted as file name.
;	HEADER=n; The number of header lines in the file (only valid when
;		FILE is set).
;	TRAILER=n; The number of trailer lines in the file (only valid when
;		FILE is set).
;	GROUP: The widget ID of the widget caller. 
;	DOUBLE: If this keyword is set, use double instead float
;	LONG: If this keyword is set, use long instead integer
;
; OUTPUTS:
;	The structure
;
; PROCEDURE:
;	Straightforward.
;
; EXAMPLE:
;	text = ['0.7 | a',"'Its me again...' | b"]
;	help,/str,make_str(text)
;	** Structure <34d740>, 2 tags, length=12, refs=1:
;	   A               FLOAT          0.700000
;	   B               STRING    'Its me again...'
;
; MODIFICATION HISTORY:
; 	Written by:	M. Sanchez del Rio (srio@esrf.fr) 97-04-07
;	97/11/04  srio@esrf.fr removes the limitation in 255 characters
;		for the structure definition by using create_struct().
;	98/11/05 srio@esrf.fr adds comment lines.
;	98/12/04 srio@esrf.fr avois compressing a string
;	2005/04/06 dejus@aps.anl.gov added the keyword "trailer"
;	2011/09/12 srio@esrf.eu added "double" and "long" kws to make_str call
;	2014/02/27 srio@esrf.eu changed ()->[] for arrays.
;	
;-

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/W) THEN itmp = Dialog_Message(Dialog_Parent=group, $
   /Error,'MAKE_STR: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   Return,0
endif
if keyword_set(file) then begin
  a=read_textfile(text)
  ; remove header comment lines
  if keyword_set(header) then a=a(header:n_elements(a)-1) 
  ; remove trailer comment lines
  if keyword_set(trailer) then a=a(0:n_elements(a)-1-trailer) 
endif else a=text

;
; remove comments (starting with ";")
;
good = where(strmid(a,0,1) NE ';')
IF good[0] EQ -1 THEN Message,'Bad input'
a=a[good]

n=n_elements(a)
tag_descript = strarr(n)
tagnames = strarr(n)
tagvalues = strarr(n)
tagflags=intarr(n)
for i=0,n-1 do begin
  nn=strparse(a(i),'|',list)
  if (nn EQ 2 OR nn EQ 1) then begin
    tag_descript[i]=list[0]
    IF (keyword_set(long) or keyword_set(double)) then begin
      command = 'myvariable = '+tag_descript[i]
      tmp = execute(command)
      IF ( (type(myvariable) EQ 2) and keyword_set(long) ) THEN $
              tag_descript[i]='long('+list[0]+')'
      IF ( (type(myvariable) EQ 4) and keyword_set(double) )  THEN $
              tag_descript[i]='double('+list[0]+')'
    ENDIF

    tagnames[i]=list[1]
    if nn EQ 2 then begin
      tagvalues[i]=list[2] 
      tagflags[i]=1
    endif else begin
      tagvalues[i]=list[0]
    endelse
  endif
endfor
;
; creates the structure with the initialization values
;
out=0
command="out=create_struct('"+strcompress(tagnames[0],/rem)+"',"+$
  tag_descript[0]+")"
;print,'**'+command+'**'
itmp=execute(command)
if itmp NE 1 then Message,'Error executing: '+command
if n GE 2 then for i=1,n-1 do begin
  command = "out=create_struct(out,'"+strcompress(tagnames[i],/rem)+"',"+$
    tag_descript[i]+")"
  ;print,'**'+command+'**'
  itmp=execute(command)
  if itmp NE 1 then message,'Error executing: '+command
endfor

;
; place the tagvalues (when different from init values) in the structure.
;
index = where(tagflags EQ 1)
if n_elements(index) EQ 1 and index(0) EQ -1 then return,out
for i=0,n_elements(index)-1 do begin
  command='out.'+StrCompress(tagnames[index[i]],/Rem)+'='+tagvalues[index[i]]
  ;print,command
  itmp=execute(command)
  IF itmp NE 1 THEN BEGIN
    Message,'Error executing: '+command
    IF SDep(/w) EQ 1 THEN itmp = Dialog_Message(/Error,$
      ['MAKE_STR: Error executing: ',command, 'Error message: '+!err_string])
  ENDIF
endfor
return,out
end
