
;+
; NAME:
;	CONIC_PARS2PTS
; PURPOSE:
;	this function calculates points along a conic cirve
;       from conic parameters
; CATEGORY:
;	Maths: conics
;
; CALLING SEQUENCE:
;	points=conic_pars2pts(center,axes,phi)
; INPUTS:
;       center: [X,Y] coordinates of the center
;       axes: [a,b] semiaxes
;       phi: rotation angle in rads
; OUTPUTS:
;       points: 2-dim array with [X,Y] points
; KEYWORD PARAMETERS:
;       npoints: number of points
;	tMin: minimum theta in rads (default: 0)
;	tMax: maximum theta in rads (default: 2 pi)
;	
; MODIFICATION HISTORY:
;	by M. Sanchez del Rio. ESRF. Grenoble, Nov 2007
;-	
FUNCTION conic_pars2pts,center,axes1,phi, $
  npoints=npoints,tmin=tmin,tmax=tmax
;
; tmin,tmax the angulat limits in rad
; npoints the number of points in 2*pi (reduced otherwise)
; phi ellipse rotation angle.
;

IF N_Elements(npoints) EQ 0 THEN npoints=100
IF N_Elements(phi) EQ 0 THEN phi=0D
IF N_Elements(tmin) EQ 0 THEN tmin=0D
IF N_Elements(tmax) EQ 0 THEN tmax=!dpi*2

IF N_Elements(axes1) EQ 0 THEN axes=[1D0,1D0] ELSE axes=axes1
IF N_Elements(axes) EQ 1 THEN axes=[axes,axes]


;
; create parameter
;
t = MakeArray1(npoints,0D0,!dpi*2)
t = MakeArray1(npoints,0D0,!dpi*2)

;
; transform center to the frame where the ellipse is unrotated
;
x1 = center[0]*cos(-phi)-center[1]*sin(-phi)
y1 = center[0]*sin(-phi)+center[1]*cos(-phi)

;
; ellipse in parametric coordinates (unrotated system)
;
x2 = X1+axes[0]*cos(t)
y2 = Y1+axes[1]*sin(t)
;
; back to rotated system
;
x = X2*cos(phi)-Y2*sin(phi)
y = X2*sin(phi)+Y2*cos(phi)



; 
; calculate angles and eliminate points outside selected range
;

IF abs(tmin-0D) GT 1D-10 OR abs(tmax-!dpi*2) GT 1D-10 THEN BEGIN
  aa = atan2( (y-center[1]),(x-center[0]) )
  iNeg = where(aa LT 0)
  IF iNeg[0] NE -1 THEN aa[iNeg] = aa[iNeg]+2D0*!dpi

  itmp = where((aa GE tmin) AND (aa LE tmax))
  IF itmp[0] NE -1 THEN BEGIN
     x=x[itmp]
     y=y[itmp]
     aa=aa[itmp]
  ENDIF
ENDIF

;for i=0,n_elements(aa)-1 do print,i,x[i],y[i],aa[i] ; *180/!pi

points=Make_Set(x,y)
return,points

END

