;+
; NAME:
;	XUS_SET
; PURPOSE:
;	Pop-up dialog widget for entering information for progam US.
; CATEGORY:
;	GAMS Index N1: Data handling; input, output
; CALLING SEQUENCE:
;	XUS_SET, data [,ACTION=action, GROUP=group]
; INPUTS:
;	data:	Anonymous structure. Parameters are contained in data.dat1
; KEYWORD PARAMETERS:
;	ACTION:	Indicates whether data changed. ACTION='DO' if data
;		are changed, and ACTION='DONT' if data are unchanged
;		(user canceled the action).
;	GROUP:	Group leader for this widget.  If the group leader dies, 
;		then this widget dies as well.
; OUTPUTS:
;	data:	Anonymous structure.  Result are overwritten in data.dat1
; COMMON BLOCKS:
;	None
; PROCEDURE:
;	The routine uses compound widgets and anonymous structures to pass
;	the data to the pop-up event handler.  The user is not forced to 
;	press the carriage return after each entry.  The input fields are
;	still checked for valid entries and valid ranges.
; EXAMPLE:
;	XUS_SET, data, ACTION=action, GROUP=event.top
;	IF action eq 'DONT' THEN GOTO, LBL900 ; cancel
;	<save input file (us.dat) and execute US>
; MODIFICATION HISTORY:
; 	Written by:	Roger J. Dejus, XFD/APS, April, 1995.
;
;        Date     | Name  | Description
; ----------------+-------+-----------------------------------------------------
; 04-JAN-1996     | RJD   | Modified the number of decimal places for the sigx1
;                 |       | and sigy1 variables to four and the number of
;                 |       | decimals to three for the distance (d variable) when
;                 |       | being displayed. Updated the version number to show
;                 |       | that US v. 1.8 is executed.
; ----------------+-------+-----------------------------------------------------
; 18-JAN-1996     | MSR   | Replaced calls to xtext -> widget_message
; ----------------+-------+-----------------------------------------------------
; 05-FEB-1997     | MSR   | Added Help Button in main window.
; ----------------+-------+-----------------------------------------------------
; 22-SEP-1997     | RJD   | Introduced usage of sdep for system dependencies.
;                 |       | Removed usage of "arch" for setting scroll sizes; 
;                 |       | only one scroll size is used instead. Removed
;                 |       | the method "Non-zero emittance; infinite-N" from the
;                 |       | methods menu which was for test purposes only. 
;                 |       | Changed the labels and headings for "brightness" to
;                 |       | "brilliance".
; ----------------+-------+-----------------------------------------------------
; 03-DEC-1998     | MSR   | XUS_SET under IDL5.1 crashes 
;                 |       | at the second "cancel". The problem was
;                 |       | related to the "handlers" and I did not
;                 |       | understand why. Solved by using pointers
;                 |       | instead. Added catch.
;		  |       | Although obsolete, I keep /MODAL in XMANAGER call,
;		  |       | otherwise IDL does not allow the use of scroll bars.
; ----------------+-------+-----------------------------------------------------
; 28-JAN-2008     | MSR   | Remove /NO_COPY keywords to avoid crashed in 
;                 |       | case of error.
; ----------------+-------+-----------------------------------------------------
; 14-AUG-2008     | RJD   | Replaced all statements related to "HANDLES" with
;                 |       | pointers. Reintroduced the /No_Copy keyword.
;                 |       | Placed the /MODAL and /FLOATING keywords on the
;                 |       | Widget_Base function call and removed /MODAL on the
;                 |       | XManager call. The cleanup routine is currently not used
;                 |       | for the modal widget. It's kept for future reference.
;                 |       | Introduced the max screen size variable and set to UXGA.
;                 |       | Developed and tested with IDL v7.01 on Sun Solaris 10,
;                 |       | and v7.03 on Windows XP.
; ----------------+-------+-----------------------------------------------------
; 03-FEB-2014     | RJD   | Changed format of sigx and sigy to use 4 decimal places.
;                 |       | Changed default size of plot windows and calculation of
;                 |       | of screen size. Removed trailing white space.
;                 |       | Removed /Modal keyword from base widget because it will
;                 |       | block all other widgets including the help widget.
;                 |       | Added group keyword to help event.
;                 |       | Prepared for XOP v2.4 embedded with IDL v8.3.
; ----------------+-------+-----------------------------------------------------
;
;-

FORWARD_FUNCTION sdep, screensize

;
;===============================================================================
;
PRO XUS_SET_EVENT, event

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Catch, /Cancel
   Message,/Info,'Error caught: '+!error_state.msg
   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
     'XUS_SET_EVENT: Error caught: '+!error_state.msg)
   IF N_Elements(info) NE 0 THEN Widget_Control, event.top, Set_UValue=info, /No_Copy
   RETURN
ENDIF

WIDGET_CONTROL, event.top, GET_UVALUE=info
name = TAG_NAMES(event, /STRUCTURE_NAME)

ds = sdep(/ds)		; system dependent directory separator

IF name eq 'WIDGET_BUTTON' THEN BEGIN
  WIDGET_CONTROL, event.id, GET_UVALUE=uval
  CASE uval OF
    'HELP': BEGIN
    hfile = Xop_GetEnv('XOP_HOME')+ds+'doc'+ds+'txt'+ds+'us_par.txt'
    XDISPLAYFILE1, hfile, GROUP=event.top
    END

    'CANCEL': BEGIN
    *(info.ptr_action) = 'DONT'
    WIDGET_CONTROL, event.top, /DESTROY
    END

    'ACCEPT': BEGIN
;  Title
    WIDGET_CONTROL, info.id_title, GET_VALUE=title & title = title(0)
    rtxt = ' is out of range.'

;  Machine parameters
;  Use on_ioerror to trap conversion errors
    ON_IOERROR, ENERGY_ERROR
    WIDGET_CONTROL, info.id_energy, GET_VALUE=energy & energy = energy(0)
    attl = 'Edit Energy' & atxt = 'Energy (GeV): New Value'
    LBL_ENERGY: energy = FLOAT(energy)
    IF (energy le 0.0) or (energy gt 100.0) THEN BEGIN ; GeV
      junk = WIDGET_MESSAGE(/ERROR,$
	['Energy Error',STRING(energy, FORMAT='(f8.2)') +rtxt])
      xedit, energy, TITLE=attl, TEXT=atxt
      GOTO, LBL_ENERGY
    ENDIF
    ON_IOERROR, CUR_ERROR
    WIDGET_CONTROL, info.id_cur, GET_VALUE=cur & cur = cur(0)
    attl = 'Edit Current' & atxt = 'Current (mA): New Value'
    LBL_CUR: cur = FLOAT(cur)
    IF (cur le 0.0) or (cur gt 10000.0) THEN BEGIN ; mA
      junk = WIDGET_MESSAGE(/ERROR,$
	['Current Error',STRING(cur, FORMAT='(f8.2)') +rtxt])
      xedit, cur, TITLE=attl, TEXT=atxt
      GOTO, LBL_CUR
    ENDIF

;  Beam Parameters
    ON_IOERROR, SIGX_ERROR
    WIDGET_CONTROL, info.id_sigx, GET_VALUE=sigx & sigx = sigx(0) 
    attl = 'Edit Sigmax' & atxt = 'Sigx (mm): New Value'
    LBL_SIGX: sigx = FLOAT(sigx)
    IF (sigx le 0.0) or (sigx gt 10.0) THEN BEGIN ; mm
      junk = WIDGET_MESSAGE(/ERROR,$
	['Sigmax Error',STRING(sigx, FORMAT='(f8.4)') +rtxt])
      xedit, sigx, TITLE=attl, TEXT=atxt & GOTO, LBL_SIGX
    ENDIF
    ON_IOERROR, SIGY_ERROR
    WIDGET_CONTROL, info.id_sigy, GET_VALUE=sigy & sigy = sigy(0) 
    attl = 'Edit Sigmay' & atxt = 'Sigy (mm): New Value'
    LBL_SIGY: sigy = FLOAT(sigy)
    IF (sigy le 0.0) or (sigy gt 10.0) THEN BEGIN ; mm
      junk = WIDGET_MESSAGE(/ERROR, $
	['Sigmay Error',STRING(sigy, FORMAT='(f8.4)') +rtxt])
      xedit, sigy, TITLE=attl, TEXT=atxt & GOTO, LBL_SIGY
    ENDIF
    ON_IOERROR, SIGX1_ERROR
    WIDGET_CONTROL, info.id_sigx1, GET_VALUE=sigx1 & sigx1 = sigx1(0) 
    attl = "Edit Sigmax'" & atxt = 'Sigx1 (mrad): New Value'
    LBL_SIGX1: sigx1 = FLOAT(sigx1)
    IF (sigx1 le 0.0) or (sigx1 gt 10.0) THEN BEGIN ; mrad
      junk = WIDGET_MESSAGE(/ERROR, $
	["Sigmax' Error",STRING(sigx1, FORMAT='(f8.4)') +rtxt])
      xedit, sigx1, TITLE=attl, TEXT=atxt, FORMAT='(f10.4)' & GOTO, LBL_SIGX1
    ENDIF
    ON_IOERROR, SIGY1_ERROR
    WIDGET_CONTROL, info.id_sigy1, GET_VALUE=sigy1 & sigy1 = sigy1(0) 
    attl = "Edit Sigmay'" & atxt = 'Sigy1 (mrad): New Value'
    LBL_SIGY1: sigy1 = FLOAT(sigy1)
    IF (sigy1 le 0.0) or (sigy1 gt 10.0) THEN BEGIN ; mrad
      junk = WIDGET_MESSAGE(/ERROR, $
	["Sigmay' Error",STRING(sigy1, FORMAT='(f8.4)') +rtxt])
      xedit, sigy1, TITLE=attl, TEXT=atxt, FORMAT='(f10.4)' & GOTO, LBL_SIGY1
    ENDIF

;  Undulator Parameters
    ON_IOERROR, PERIOD_ERROR
    WIDGET_CONTROL, info.id_period, GET_VALUE=period & period = period(0) 
    attl = 'Edit Period Length' & atxt = 'Period (cm): New Value'
    LBL_PERIOD: period = FLOAT(period)
    IF (period le 0.0) or (period gt 100.0) THEN BEGIN ; cm
      junk = WIDGET_MESSAGE(/ERROR, $
	['Period Length Error',STRING(period, FORMAT='(f8.2)') +rtxt])
      xedit, period, TITLE=attl, TEXT=atxt & GOTO, LBL_PERIOD
    ENDIF
    ON_IOERROR, NP_ERROR
    WIDGET_CONTROL, info.id_np, GET_VALUE=np & np = np(0) 
    attl = 'Edit Number of Periods' & atxt = 'Np: New Value'
    LBL_NP: np = FIX(np)
    IF (np lt 1) or (np gt 1000) THEN BEGIN
      junk = WIDGET_MESSAGE(/ERROR, $
	['Number of Periods Error',STRING(np, FORMAT='(i8)') +rtxt])
      xedit, np, TITLE=attl, TEXT=atxt & GOTO, LBL_NP
    ENDIF
    ON_IOERROR, KX_ERROR
    WIDGET_CONTROL, info.id_kx, GET_VALUE=kx & kx = kx(0) 
    attl = 'Edit Kx' & atxt = 'Kx: New Value'
    LBL_KX: kx = FLOAT(kx)
    IF (kx lt 0.0) or (kx gt 100.0) THEN BEGIN
      junk = WIDGET_MESSAGE(/ERROR, $
	['Kx Error',STRING(kx, FORMAT='(f8.3)') +rtxt])
      xedit, kx, TITLE=attl, TEXT=atxt & GOTO, LBL_KX
    ENDIF
    ON_IOERROR, KY_ERROR
    WIDGET_CONTROL, info.id_ky, GET_VALUE=ky & ky = ky(0) 
    attl = 'Edit Ky' & atxt = 'Ky: New Value'
    LBL_KY: ky = FLOAT(ky)
    IF (ky lt 0.0) or (ky gt 100.0) THEN BEGIN
      junk = WIDGET_MESSAGE(/ERROR, $
	['Ky Error',STRING(ky, FORMAT='(f8.3)') +rtxt])
      xedit, ky, TITLE=attl, TEXT=atxt & GOTO, LBL_KY
    ENDIF

;  Scan Parameters
    ON_IOERROR, EMIN_ERROR
    WIDGET_CONTROL, info.id_emin, GET_VALUE=emin & emin = emin(0) 
    attl = 'Edit Emin' & atxt = 'Emin (eV): New Value'
    LBL_EMIN: emin = FLOAT(emin)
    IF (emin le 0.0) or (emin gt 1.0e5) THEN BEGIN ; eV
      junk = WIDGET_MESSAGE(/ERROR, $
	['Emin Error',STRING(emin, FORMAT='(f10.1)') +rtxt])
      xedit, emin, TITLE=attl, TEXT=atxt & GOTO, LBL_EMIN
    ENDIF
    ON_IOERROR, EMAX_ERROR
    WIDGET_CONTROL, info.id_emax, GET_VALUE=emax & emax = emax(0) 
    attl = 'Edit Emax' & atxt = 'Emax (eV): New Value'
    LBL_EMAX: emax = FLOAT(emax)
    IF (emax le 0.0) or (emax gt 1.0e6) THEN BEGIN ; eV
      junk = WIDGET_MESSAGE(/ERROR, $
	['Emax Error',STRING(emax, FORMAT='(f10.1)') +rtxt])
      xedit, emax, TITLE=attl, TEXT=atxt & GOTO, LBL_EMAX
    ENDIF
    IF (emax le emin) THEN BEGIN
      junk = WIDGET_MESSAGE(/ERROR, $
	['Emax Error',STRING(emax, FORMAT='(f10.1)') +' less than emin.'])
      xedit, emax, TITLE=attl, TEXT=atxt & GOTO, LBL_EMAX
    ENDIF
    ON_IOERROR, N_ERROR
    WIDGET_CONTROL, info.id_n, GET_VALUE=n & n = n(0) 
    attl = 'Edit Number of Energy-Points' & atxt = 'N: New Value'
    LBL_N: n = LONG(n)
    IF (n lt 1) or (n gt 40000) THEN BEGIN
      junk = WIDGET_MESSAGE(/ERROR, $
	['Number of Energy-Points Error',STRING(n, FORMAT='(i8)') +rtxt])
      xedit, n, TITLE=attl, TEXT=atxt & GOTO, LBL_N
    ENDIF

;  Pinhole Parameters
    ON_IOERROR, D_ERROR
    WIDGET_CONTROL, info.id_d, GET_VALUE=d & d = d(0) 
    attl = 'Edit Distance' & atxt = 'D (m): New Value'
    LBL_D: d = FLOAT(d)
    IF (d lt 0.0) or (d gt 1000.0) THEN BEGIN ; m
      junk = WIDGET_MESSAGE(/ERROR, $
	['Distance Error',STRING(d, FORMAT='(f8.3)') +rtxt])
      xedit, d, TITLE=attl, TEXT=atxt, FORMAT='(f10.3)' & GOTO, LBL_D
    ENDIF
    ON_IOERROR, XPC_ERROR
    WIDGET_CONTROL, info.id_xpc, GET_VALUE=xpc & xpc = xpc(0) 
    IF d eq 0.0 THEN BEGIN ; Angular units
      attl = 'Edit X-center of Pinhole' & atxt = 'Xpc (mrad): New Value'
    ENDIF ELSE BEGIN ; Spatial units
      attl = 'Edit X-center of Pinhole' & atxt = 'Xpc (mm): New Value'
    ENDELSE
    LBL_XPC: xpc = FLOAT(xpc)
    IF (xpc lt 0.0) or (xpc gt 100.0) THEN BEGIN ; mm or mrad
      junk = WIDGET_MESSAGE(/ERROR, $
	['X-center Error',STRING(xpc, FORMAT='(f8.3)') +rtxt])
      xedit, xpc, TITLE=attl, TEXT=atxt & GOTO, LBL_XPC
    ENDIF
    ON_IOERROR, YPC_ERROR
    WIDGET_CONTROL, info.id_ypc, GET_VALUE=ypc & ypc = ypc(0) 
    IF d eq 0.0 THEN BEGIN ; Angular units
      attl = 'Edit Y-center of Pinhole' & atxt = 'Ypc (mrad): New Value'
    ENDIF ELSE BEGIN ; Spatial units
      attl = 'Edit Y-center of Pinhole' & atxt = 'Ypc (mm): New Value'
    ENDELSE
    LBL_YPC: ypc = FLOAT(ypc)
    IF (ypc lt 0.0) or (ypc gt 100.0) THEN BEGIN ; mm or mrad
      junk = WIDGET_MESSAGE(/ERROR, $
	 ['Y-center Error',STRING(ypc, FORMAT='(f8.3)') +rtxt])
      xedit, ypc, TITLE=attl, TEXT=atxt & GOTO, LBL_YPC
    ENDIF
    ON_IOERROR, XPS_ERROR
    WIDGET_CONTROL, info.id_xps, GET_VALUE=xps & xps = xps(0) 
    IF d eq 0.0 THEN BEGIN ; Angular units
      attl = 'Edit X-size of Pinhole' & atxt = 'Xps (mrad): New Value'
    ENDIF ELSE BEGIN ; Spatial units
      attl = 'Edit X-size of Pinhole' & atxt = 'Xps (mm): New Value'
    ENDELSE
    LBL_XPS: xps = FLOAT(xps)
    IF (xps lt 0.0) or (xps gt 200.0) THEN BEGIN ; mm or mrad
      junk = WIDGET_MESSAGE(/ERROR, $
	['X-size Error',STRING(xps, FORMAT='(f8.3)') +rtxt])
      xedit, xps, TITLE=attl, TEXT=atxt & GOTO, LBL_XPS
    ENDIF
    ON_IOERROR, YPS_ERROR
    WIDGET_CONTROL, info.id_yps, GET_VALUE=yps & yps = yps(0) 
    IF d eq 0.0 THEN BEGIN ; Angular units
      attl = 'Edit Y-size of Pinhole' & atxt = 'Yps (mrad): New Value'
    ENDIF ELSE BEGIN ; Spatial units
      attl = 'Edit Y-size of Pinhole' & atxt = 'Yps (mm): New Value'
    ENDELSE
    LBL_YPS: yps = FLOAT(yps)
    IF (yps lt 0.0) or (yps gt 200.0) THEN BEGIN ; mm or mrad
      junk = WIDGET_MESSAGE(/ERROR, $
	['Y-size Error',STRING(yps, FORMAT='(f8.3)') +rtxt])
      xedit, yps, TITLE=attl, TEXT=atxt & GOTO, LBL_YPS
    ENDIF
    ON_IOERROR, NXP_ERROR
    WIDGET_CONTROL, info.id_nxp, GET_VALUE=nxp & nxp = nxp(0) 
    attl = 'Edit X-Number of Points' & atxt = 'Nxp: New Value'
    LBL_NXP: nxp = FIX(nxp)
    IF (nxp lt 1) or (nxp gt 50) THEN BEGIN
      junk = WIDGET_MESSAGE(/ERROR, $
	['X-Number of Points Error',STRING(nxp, FORMAT='(i6)') +rtxt])
      xedit, nxp, TITLE=attl, TEXT=atxt & GOTO, LBL_NXP
    ENDIF
    ON_IOERROR, NYP_ERROR
    WIDGET_CONTROL, info.id_nyp, GET_VALUE=nyp & nyp = nyp(0) 
    attl = 'Edit Y-Number of Points' & atxt = 'Nyp: New Value'
    LBL_NYP: nyp = FIX(nyp)
    IF (nyp lt 1) or (nyp gt 50) THEN BEGIN
      junk = WIDGET_MESSAGE(/ERROR, $
	['Y-Number of Points Error',STRING(nyp, FORMAT='(i6)') +rtxt])
      xedit, nyp, TITLE=attl, TEXT=atxt & GOTO, LBL_NYP
    ENDIF

;  Cancel conversion errors
    ON_IOERROR, NULL

;  Mode
    WIDGET_CONTROL, info.id_mode, GET_VALUE=mode & mode = mode +1
;  Method
    WIDGET_CONTROL, info.id_method, GET_VALUE=method & method = method +2
    IF method eq 2 THEN method = 1
;  Harmonic number
    WIDGET_CONTROL, info.id_harm, GET_VALUE=harm
    IF harm eq 0 THEN BEGIN ; All harmonics
      iharm = 0
    ENDIF ELSE IF harm eq 1 THEN BEGIN; Lowest order harmonic
      iharm = -1
    ENDIF ELSE BEGIN ; Given harmonic number
      harm_ids = info.ids.harm_ids
      harm_tx  = info.harm_tx & slen_tx = STRLEN(harm_tx) & sfirst = slen_tx
      WIDGET_CONTROL, harm_ids(2), GET_VALUE=harm_value
      harm_value = STRMID(harm_value, sfirst, STRLEN(harm_value) -sfirst) ; strip text
      iharm = fix(harm_value)
    ENDELSE

;  Calc Parameters
    calc_ids = info.ids.calc_ids
    calc_tx  = info.calc_tx & slen_tx = STRLEN(calc_tx)

    WIDGET_CONTROL, calc_ids(0), GET_VALUE=calc_value & sfirst = slen_tx(0)
    calc_value = STRMID(calc_value, sfirst, STRLEN(calc_value) -sfirst) ; strip text
    nphi = fix(calc_value)
    WIDGET_CONTROL, calc_ids(1), GET_VALUE=calc_value & sfirst = slen_tx(1)
    calc_value = STRMID(calc_value, sfirst, STRLEN(calc_value) -sfirst) ; strip text
    nalpha = fix(calc_value)
    WIDGET_CONTROL, calc_ids(2), GET_VALUE=calc_value & sfirst = slen_tx(2)
    calc_value = STRMID(calc_value, sfirst, STRLEN(calc_value) -sfirst) ; strip text
    calpha2 = float(calc_value)
    WIDGET_CONTROL, calc_ids(3), GET_VALUE=calc_value & sfirst = slen_tx(3)
    calc_value = STRMID(calc_value, sfirst, STRLEN(calc_value) -sfirst) ; strip text
    nomega = fix(calc_value)
    WIDGET_CONTROL, calc_ids(4), GET_VALUE=calc_value & sfirst = slen_tx(4)
    calc_value = STRMID(calc_value, sfirst, STRLEN(calc_value) -sfirst) ; strip text
    comega = float(calc_value)
    WIDGET_CONTROL, calc_ids(5), GET_VALUE=calc_value & sfirst = slen_tx(5)
    calc_value = STRMID(calc_value, sfirst, STRLEN(calc_value) -sfirst) ; strip text
    nsigma = fix(calc_value)

;  Run mode
    WIDGET_CONTROL, info.id_run_mode, GET_VALUE=run_mode
    IF run_mode eq 0 THEN run_mode_name = 'foreground' ELSE run_mode_name = 'background'

    dat1 = {Xus, title:title, energy:energy, cur:cur, text_machine:'', $
	    sigx:sigx, sigy:sigy, sigx1:sigx1, sigy1:sigy1, text_beam:'', $
	    period:period, np:np, kx:kx, ky:ky, text_undulator:'', $
	    emin:emin, emax:emax, n:n, text_energy:'', $
	    d:d, xpc:xpc, ypc:ypc, xps:xps, yps:yps, nxp:nxp, nyp:nyp, text_pinhole:'', $
	    mode:mode, method:method, iharm:iharm, text_mode:'', $
	    nphi:nphi, nalpha:nalpha, calpha2:calpha2, nomega:nomega, $
	    comega:comega, nsigma:nsigma, text_calc:'', $
	    run_mode_name:run_mode_name}
    *(info.ptr_action) = 'DO'
    *(info.ptr_data) = dat1
    WIDGET_CONTROL, event.top, /DESTROY
    RETURN

;  Error Handling
;  Machine Parameters
    ENERGY_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, $
	['Energy Error',!err_string])
    xedit, energy, TITLE=attl, TEXT=atxt & GOTO, LBL_ENERGY
    END
    CUR_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, $
	['Current Error',!err_string])
    xedit, cur, TITLE=attl, TEXT=atxt & GOTO, LBL_CUR
    END

;  Beam Parameters
    SIGX_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Sigmax Error',!err_string])
    xedit, sigx, TITLE=attl, TEXT=atxt & GOTO, LBL_SIGX
    END
    SIGY_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Sigmay Error',!err_string])
    xedit, sigy, TITLE=attl, TEXT=atxt & GOTO, LBL_SIGY
    END
    SIGX1_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR,["Sigmax' Error",!err_string])
    xedit, sigx1, TITLE=attl, TEXT=atxt & GOTO, LBL_SIGX1
    END
    SIGY1_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR,["Sigmay' Error",!err_string])
    xedit, sigy1, TITLE=attl, TEXT=atxt & GOTO, LBL_SIGY1
    END

;  Undulator Parameters
    PERIOD_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR,['Period Length Error',!err_string])
    xedit, period, TITLE=attl, TEXT=atxt & GOTO, LBL_PERIOD
    END
    NP_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Number of Periods Error',!err_string])
    xedit, np, TITLE=attl, TEXT=atxt & GOTO, LBL_NP
    END
    KX_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Kx Error',!err_string])
    xedit, kx, TITLE=attl, TEXT=atxt & GOTO, LBL_KX
    END
    KY_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Ky Error',!err_string])
    xedit, ky, TITLE=attl, TEXT=atxt & GOTO, LBL_KY
    END

;  Scan Parameters
    EMIN_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Emin Error',!err_string])
    xedit, emin, TITLE=attl, TEXT=atxt & GOTO, LBL_EMIN
    END
    EMAX_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Emax Error',!err_string])
    xedit, emax, TITLE=attl, TEXT=atxt & GOTO, LBL_EMAX
    END
    N_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, $
	['Number of Energy-Points Error',!err_string])
    xedit, n, TITLE=attl, TEXT=atxt & GOTO, LBL_N
    END

;  Pinhole Parameters
    D_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Distance Error',!err_string])
    xedit, d, TITLE=attl, TEXT=atxt & GOTO, LBL_D
    END
    XPC_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['X-center Error',!err_string])
    xedit, xpc, TITLE=attl, TEXT=atxt & GOTO, LBL_XPC
    END
    YPC_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Y-center Error',!err_string])
    xedit, ypc, TITLE=attl, TEXT=atxt & GOTO, LBL_YPC
    END
    XPS_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['X-size Error',!err_string])
    xedit, xps, TITLE=attl, TEXT=atxt & GOTO, LBL_XPS
    END
    YPS_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['Y-size Error',!err_string])
    xedit, yps, TITLE=attl, TEXT=atxt & GOTO, LBL_YPS
    END
    NXP_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR, ['X-Number of Points Error',!err_string])
    xedit, nxp, TITLE=attl, TEXT=atxt & GOTO, LBL_NXP
    END
    NYP_ERROR: BEGIN
    junk = WIDGET_MESSAGE(/ERROR,['Y-Number of Points Error',!err_string])
    xedit, nyp, TITLE=attl, TEXT=atxt & GOTO, LBL_NYP
    END

    END ; Accept
  ENDCASE

ENDIF ELSE BEGIN
  WIDGET_CONTROL, event.id, GET_UVALUE=uval
  CASE uval OF
    'MODE': BEGIN
    bg_method  = info.id_method
    method_ids = info.ids.method_ids
    WIDGET_CONTROL, event.id, GET_VALUE=mode_value
    WIDGET_CONTROL, bg_method, GET_VALUE=method_value

    CASE mode_value OF ; de-sensitize buttons not applicable,set default
      0: BEGIN ; Angular/spatial flux density distribution
      WIDGET_CONTROL, method_ids(0), SENSITIVE=1
      WIDGET_CONTROL, method_ids(1), SENSITIVE=1
      WIDGET_CONTROL, method_ids(2), SENSITIVE=0
      IF (method_value eq 2) THEN BEGIN
        WIDGET_CONTROL, bg_method, SET_VALUE=0   ; default
      ENDIF 
      END

      1: BEGIN ; Angular/spatial flux density spectrum
      WIDGET_CONTROL, method_ids(0), SENSITIVE=1
      WIDGET_CONTROL, method_ids(1), SENSITIVE=1
      WIDGET_CONTROL, method_ids(2), SENSITIVE=1
      END
      
      2: BEGIN ; On-axis brilliance spectrum
      WIDGET_CONTROL, method_ids(0), SENSITIVE=1
      WIDGET_CONTROL, method_ids(1), SENSITIVE=1
      WIDGET_CONTROL, method_ids(2), SENSITIVE=1
      END

      3: BEGIN ; Flux spectrum through a pinhole
      WIDGET_CONTROL, method_ids(0), SENSITIVE=1
      WIDGET_CONTROL, method_ids(1), SENSITIVE=1
      WIDGET_CONTROL, method_ids(2), SENSITIVE=1
      END

      4: BEGIN ; Flux spectrum integrated over all angles
      WIDGET_CONTROL, method_ids(0), SENSITIVE=0
      WIDGET_CONTROL, method_ids(1), SENSITIVE=0
      WIDGET_CONTROL, method_ids(2), SENSITIVE=1
      IF (method_value eq 0) or (method_value eq 1) THEN BEGIN
        WIDGET_CONTROL, bg_method, SET_VALUE=2 ; default
      ENDIF 
      END

      5: BEGIN ; Power denisty and integrated power
      WIDGET_CONTROL, method_ids(0), SENSITIVE=1
      WIDGET_CONTROL, method_ids(1), SENSITIVE=1
      WIDGET_CONTROL, method_ids(2), SENSITIVE=0
      IF (method_value eq 2) THEN BEGIN
        WIDGET_CONTROL, bg_method, SET_VALUE=0 ; default
      ENDIF 
      END
    ENDCASE ; mode_value
    END

    'METHOD': BEGIN ; nothing
    END

    'HARM': BEGIN
    WIDGET_CONTROL, event.id, GET_VALUE=harm_value
    IF harm_value eq 3 THEN BEGIN ; edit harmonic number
      harm_ids = info.ids.harm_ids
      harm_tx  = info.harm_tx
      ssz      = info.ssz & xsz = ssz[0] & ysz = ssz[1]
      harm_title = 'Harmonic Number'
      harm_text  = 'New harmonic'
      slen_tx  = STRLEN(harm_tx)
      sfirst   = slen_tx
      WIDGET_CONTROL, harm_ids(2), GET_VALUE=harm_value
      harm_value = STRMID(harm_value, sfirst, STRLEN(harm_value) -sfirst) ; strip text
      harm_value = fix(harm_value)
      xedit, harm_value, TITLE=harm_title, TEXT=harm_text, FORMAT = '(i6)', $
			 XOFFSET=xsz/5.0, YOFFSET=ysz/2.0
      harm_name = harm_tx +STRING(harm_value, FORMAT='(i6)')
      WIDGET_CONTROL, harm_ids(2), SET_VALUE=harm_name ; set new value
      WIDGET_CONTROL, event.id, SET_VALUE=2 ; select default button
    ENDIF
    END

    'CALC_PARAM': BEGIN
    calc_ids = info.ids.calc_ids
    calc_tx  = info.calc_tx
    ssz      = info.ssz & xsz = ssz[0] & ysz = ssz[1]
    calc_ix  = event.value
    calc_title = 'Intrinsic Parameter'
    calc_text = calc_tx(calc_ix) +': New Value'
    slen_tx  = STRLEN(calc_tx)
    sfirst   = slen_tx(calc_ix)
    WIDGET_CONTROL, calc_ids(calc_ix), GET_VALUE=calc_value
    calc_value = STRMID(calc_value, sfirst, STRLEN(calc_value) -sfirst) ; strip text

    CASE calc_ix OF ; index of button
      0: BEGIN ; nphi
      calc_value = fix(calc_value)
      xedit, calc_value, TITLE=calc_title, TEXT=calc_text, FORMAT = '(i10)', $
			 XOFFSET=xsz/10.0, YOFFSET=ysz/2.0
      calc_name = calc_tx(calc_ix) +STRING(calc_value, FORMAT='(i10)')
      END
      1: BEGIN ; nalpha
      calc_value = fix(calc_value)
      xedit, calc_value, TITLE=calc_title, TEXT=calc_text, FORMAT = '(i10)', $
			 XOFFSET=xsz/10.0, YOFFSET=ysz/2.0
      calc_name = calc_tx(calc_ix) +STRING(calc_value, FORMAT='(i10)')
      END
      2: BEGIN ; calpha2
      calc_value = float(calc_value)
      xedit, calc_value, TITLE=calc_title, TEXT=calc_text, FORMAT = '(f10.1)', $
			 XOFFSET=xsz/10.0, YOFFSET=ysz/2.0
      calc_name = calc_tx(calc_ix) +STRING(calc_value, FORMAT='(f10.1)')
      END
      3: BEGIN ; nomega
      calc_value = fix(calc_value)
      xedit, calc_value, TITLE=calc_title, TEXT=calc_text, FORMAT = '(i10)', $
			 XOFFSET=xsz/10.0, YOFFSET=ysz/2.0
      calc_name = calc_tx(calc_ix) +STRING(calc_value, FORMAT='(i10)')
      END
      4: BEGIN ; comega
      calc_value = float(calc_value)
      xedit, calc_value, TITLE=calc_title, TEXT=calc_text, FORMAT = '(f10.1)', $
			 XOFFSET=xsz/10.0, YOFFSET=ysz/2.0
      calc_name = calc_tx(calc_ix) +STRING(calc_value, FORMAT='(f10.1)')
      END
      5: BEGIN ; nsigma
      calc_value = fix(calc_value)
      xedit, calc_value, TITLE=calc_title, TEXT=calc_text, FORMAT = '(i10)', $
			 XOFFSET=xsz/10.0, YOFFSET=ysz/2.0
      calc_name = calc_tx(calc_ix) +STRING(calc_value, FORMAT='(i10)')
      END
    ENDCASE

    WIDGET_CONTROL, calc_ids(calc_ix), SET_VALUE=calc_name ; set new value
    END ; CALC_PARAM

    'RUN_MODE': BEGIN ; nothing
    END

  ENDCASE ; uval
ENDELSE

END ; XUS_SET_EVENT

;
;===============================================================================
;
;PRO XUS_SET_Cleanup, setbase_modal	; not useful for modal widget
;
;;print, '*** IN XUS_SET_Cleanup ***'
;WIDGET_CONTROL, setbase_modal, GET_UVALUE=info, /No_Copy
;IF n_elements(info) eq 0 THEN RETURN
;
;; Free pointers
;ptr_free, info.ptr_action, info.ptr_data
;RETURN
;END
;
;===============================================================================
;
PRO XUS_SET, data, ACTION=action, GROUP=group

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Catch, /Cancel
   Message,/Info,'Error caught: '+!error_state.msg
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'XUS_SET: Error caught: '+!error_state.msg)
   RETURN
ENDIF

ssz = screensize()      ; get screen size

;  Set size of scrollable region
xsz = round(ssz[0]*1.00)
ysz = round(ssz[1]*1.30)
 
;  Definitions and labels
XUS_SET_LBL1='Xus input parameters'

; save
;setbase_modal = WIDGET_BASE(TITLE=XUS_SET_LBL1, COLUMN=1, $
;  GROUP_LEADER=group, /Modal, /Floating)
; need to create new base with scrollbars as scrollbars are not allowed with a modal base widget
;setbase = WIDGET_BASE(setbase_modal, COLUMN=1, $
;  X_SCROLL_SIZE=xsz, Y_SCROLL_SIZE=ysz )

; remove modal keyword but keep variable setbase_modal for simplicity (calls to XTC_SET is with /no_block instead)
;setbase_modal = WIDGET_BASE(TITLE=XUS_SET_LBL1, COLUMN=1, $
;  GROUP_LEADER=group, /Floating, X_SCROLL_SIZE=xsz,Y_SCROLL_SIZE=ysz)	; note now the base widget is no longer modal although the variable name may indicate otherwise
; DO not use scroll bars
setbase_modal = WIDGET_BASE(TITLE=XUS_SET_LBL1, COLUMN=1, $
  GROUP_LEADER=group, /Floating)	; note now the base widget is no longer modal although the variable name may indicate otherwise

setbase = setbase_modal	; need to copy to setbase 

dat1 = data.dat1
;  Control buttons
b40 = WIDGET_BASE(setbase, ROW=1, /FRAME)
bt  = WIDGET_BUTTON(b40, VALUE='Accept', UVALUE='ACCEPT')
bt  = WIDGET_BUTTON(b40, VALUE='Cancel', UVALUE='CANCEL')
bt  = WIDGET_BUTTON(b40, VALUE='Help', UVALUE='HELP')

;  Title information
b0 = WIDGET_BASE(setbase, ROW=1, /FRAME)
bs = WIDGET_BASE(b0, COLUMN=1)
fd_title = CW_FIELD(bs, TITLE='Title', VALUE=dat1.title, /ROW, XSIZE=80)

;  Machine Parameters
b1 = WIDGET_BASE(setbase, ROW=1, /FRAME)
lb = WIDGET_LABEL(b1, VALUE='Machine Parameters  ')
bs = WIDGET_BASE(b1, COLUMN=2)
fd_energy = CW_FIELD(bs, TITLE='Energy (GeV)', /COLUMN, $
	    VALUE=STRING(dat1.energy, FORMAT='(f10.2)'))
fd_cur    = CW_FIELD(bs, TITLE='Current (mA)', /COLUMN, $
	    VALUE=STRING(dat1.cur, FORMAT='(f10.2)'))

;  Beam Parameters
b2 = WIDGET_BASE(setbase, ROW=1, /FRAME)
lb = WIDGET_LABEL(b2, VALUE='Beam Parameters     ')
bs = WIDGET_BASE(b2, COLUMN=4)
fd_sigx  = CW_FIELD(bs, TITLE='Sigx (mm)', /COLUMN, $
	   VALUE=STRING(dat1.sigx, FORMAT='(f10.4)'))
fd_sigy  = CW_FIELD(bs, TITLE='Sigy (mm)', /COLUMN, $
	   VALUE=STRING(dat1.sigy, FORMAT='(f10.4)'))
fd_sigx1 = CW_FIELD(bs, TITLE='Sigx1 (mrad)', /COLUMN, $
	   VALUE=STRING(dat1.sigx1, FORMAT='(f10.4)'))
fd_sigy1 = CW_FIELD(bs, TITLE='Sigy1 (mrad)', /COLUMN, $
	   VALUE=STRING(dat1.sigy1, FORMAT='(f10.4)'))

;  Undulator Parameters
b3 = WIDGET_BASE(setbase, ROW=1, /FRAME)
lb = WIDGET_LABEL(b3, VALUE='Undulator Parameters')
bs = WIDGET_BASE(b3, COLUMN=4)
fd_period = CW_FIELD(bs, TITLE='Period length (cm)', /COLUMN, $
	    VALUE=STRING(dat1.period, FORMAT='(f10.2)'))
fd_np     = CW_FIELD(bs, TITLE='Number of periods', /COLUMN, $
            VALUE=STRING(dat1.np, FORMAT='(i10)'))
fd_kx     = CW_FIELD(bs, TITLE='Deflection parameter Kx', /COLUMN, $
	    VALUE=STRING(dat1.kx, FORMAT='(f10.3)'))
fd_ky     = CW_FIELD(bs, TITLE='Deflection parameter Ky', /COLUMN, $
	    VALUE=STRING(dat1.ky, FORMAT='(f10.3)'))

;  Scan Parameters
b4 = WIDGET_BASE(setbase, ROW=1, /FRAME)
lb = WIDGET_LABEL(b4, VALUE='Scan Parameters     ')
bs = WIDGET_BASE(b4, COLUMN=3)
fd_emin = CW_FIELD(bs, TITLE='Minimum energy (eV)', /COLUMN, $
	  VALUE=STRING(dat1.emin, FORMAT='(f10.1)'))
fd_emax = CW_FIELD(bs, TITLE='Maximum energy (eV)', /COLUMN, $
	  VALUE=STRING(dat1.emax, FORMAT='(f10.1)'))
fd_n    = CW_FIELD(bs, TITLE='Number of energy-points', /COLUMN, $
	  VALUE=STRING(dat1.n, FORMAT='(i10)'))

;  Pinhole Parameters
b5 = WIDGET_BASE(setbase, ROW=1, /FRAME)
lb = WIDGET_LABEL(b5, VALUE='Pinhole Parameters  ')
bs = WIDGET_BASE(b5, ROW=1)
fd_d = CW_FIELD(bs, TITLE='Distance (m)', /COLUMN, $
       VALUE=STRING(dat1.d, FORMAT='(f10.3)'))
bs = WIDGET_BASE(bs, COLUMN=3)
IF dat1.d eq 0.0 THEN BEGIN ; Angular units
  xpc_title = 'X-center of pinhole (mrad)'
  ypc_title = 'Y-center of pinhole (mrad)'
  xps_title = 'X-size of pinhole (mrad)'
  yps_title = 'Y-size of pinhole (mrad)'
ENDIF ELSE BEGIN ; Spatial units
  xpc_title = 'X-center of pinhole (mm)'
  ypc_title = 'Y-center of pinhole (mm)'
  xps_title = 'X-size of pinhole (mm)'
  yps_title = 'Y-size of pinhole (mm)'
ENDELSE

fd_xpc = CW_FIELD(bs, TITLE=xpc_title, /COLUMN, $
	 VALUE=STRING(dat1.xpc, FORMAT='(f10.3)'))
fd_ypc = CW_FIELD(bs, TITLE=ypc_title, /COLUMN, $
	 VALUE=STRING(dat1.ypc, FORMAT='(f10.3)'))
fd_xps = CW_FIELD(bs, TITLE=xps_title, /COLUMN, $ 
	 VALUE=STRING(dat1.xps, FORMAT='(f10.3)'))
fd_yps = CW_FIELD(bs, TITLE=yps_title, /COLUMN, $
	 VALUE=STRING(dat1.yps, FORMAT='(f10.3)'))
fd_nxp = CW_FIELD(bs, TITLE='X number of points', /COLUMN, $
	 VALUE=STRING(dat1.nxp, FORMAT='(i10)'))
fd_nyp = CW_FIELD(bs, TITLE='Y number of points', /COLUMN, $
	 VALUE=STRING(dat1.nyp, FORMAT='(i10)'))

;  Menus
b8 = WIDGET_BASE(setbase, ROW=1, /FRAME)

;  Mode
initial_state = dat1.mode -1
mode_names = ['Angular/spatial flux density distribution', $
              'Angular/spatial flux density spectrum', $
	      'On-axis brilliance spectrum', $
	      'Flux spectrum through a pinhole', $
	      'Flux spectrum integrated over all angles', $
	      'Power density and integrated power']
bg_mode = CW_BGROUP(b8, mode_names, LABEL_TOP='Mode', SET_VALUE=initial_state, $
	            UVALUE='MODE', IDS=mode_ids, /COLUMN, /EXCLUSIVE, /FRAME, /NO_RELEASE)

;  Method
initial_state = dat1.method -2
IF initial_state eq -1 THEN initial_state = 0
method_names = ['Non-zero emittance; finite-N', $
	        'Zero     emittance; finite-N', $
	        'Non-zero emittance; infinite-N/conv.']
bg_method = CW_BGROUP(b8, method_names, LABEL_TOP='Method', SET_VALUE=initial_state, $
	              UVALUE='METHOD', IDS=method_ids, /COLUMN, /EXCLUSIVE, /FRAME, $
		      /NO_RELEASE)

IF dat1.mode eq 1 THEN BEGIN
  WIDGET_CONTROL, method_ids(2), sensitive = 0; de-sensitize options
ENDIF
IF dat1.mode eq 5 THEN BEGIN
  WIDGET_CONTROL, method_ids(0), sensitive = 0; de-sensitize options
  WIDGET_CONTROL, method_ids(1), sensitive = 0; de-sensitize options
ENDIF
IF dat1.mode eq 6 THEN BEGIN
  WIDGET_CONTROL, method_ids(2), sensitive = 0; de-sensitize options
ENDIF

;  Harmonic number
IF dat1.iharm eq 0 THEN BEGIN ; All harmonics
  initial_state = 0
  harm_value = 1
ENDIF ELSE IF dat1.iharm eq -1 THEN BEGIN ; Lowest order harmonic
  initial_state = 1
  harm_value = 1
ENDIF ELSE BEGIN
  initial_state = 2
  harm_value = dat1.iharm
ENDELSE
harm_tx = 'Harmonic #'
harm_names = ['All harmonics', $
              'Lowest order harmonic', $
	       harm_tx +STRING(harm_value, FORMAT='(i6)'), $
	      'Edit harmonic number']
bg_harm = CW_BGROUP(b8, harm_names, LABEL_TOP='Harmonic Number', SET_VALUE=initial_state, $
	            UVALUE='HARM', IDS=harm_ids, /COLUMN, /EXCLUSIVE, /FRAME, $
		    /NO_RELEASE)

;  Scroll down menus
b12 = WIDGET_BASE(setbase, ROW=1, /FRAME)
;  Calculational parameters
b13 = WIDGET_BASE(b12, COLUMN=1)
calc_tx = ['nphi   ', $
           'nalpha ', $
	   'calpha2', $
	   'nomega ', $
	   'comega ', $
	   'nsigma ']
calc_names = [calc_tx(0) +STRING(dat1.nphi,    FORMAT='(i10)'), $
              calc_tx(1) +STRING(dat1.nalpha,  FORMAT='(i10)'), $
	      calc_tx(2) +STRING(dat1.calpha2, FORMAT='(f10.1)'), $
	      calc_tx(3) +STRING(dat1.nomega,  FORMAT='(i10)'), $
	      calc_tx(4) +STRING(dat1.comega,  FORMAT='(f10.1)'), $
	      calc_tx(5) +STRING(dat1.nsigma,  FORMAT='(i10)')]
bg_calc = CW_BGROUP(b13, calc_names, LABEL_TOP='Intrinsic Parameters', $
	            UVALUE='CALC_PARAM', IDS=calc_ids, COLUMN=1, /FRAME, /NO_RELEASE, $ 
		    X_SCROLL_SIZE=xsz/6.0, /SCROLL)

;  Run mode
b15 = WIDGET_BASE(b12, COLUMN=1) 
run_mode_names = ['foreground', $
                  'background']
index = WHERE(run_mode_names eq dat1.run_mode_name)
initial_state = index(0)
bg_run_mode = CW_BGROUP(b15, run_mode_names, LABEL_TOP='Run Mode', $
	      SET_VALUE=initial_state, UVALUE='RUN_MODE', $
	      IDS=run_mode_ids, /COLUMN, /EXCLUSIVE, /FRAME, /NO_RELEASE)

ids = {mode_ids:mode_ids, method_ids:method_ids, harm_ids:harm_ids, calc_ids:calc_ids}
ptr_action = ptr_new('DONT')
ptr_data = ptr_new(/Allocate_Heap)

info = {id_title:fd_title, id_energy:fd_energy, id_cur:fd_cur, $
	id_sigx:fd_sigx, id_sigy:fd_sigy, id_sigx1:fd_sigx1, id_sigy1:fd_sigy1, $
	id_period:fd_period, id_np:fd_np, id_kx:fd_kx, id_ky:fd_ky, $
	id_emin:fd_emin, id_emax:fd_emax, id_n:fd_n, $
	id_d:fd_d, id_xpc:fd_xpc, id_ypc:fd_ypc, $
	id_xps:fd_xps, id_yps:fd_yps, id_nxp:fd_nxp, id_nyp:fd_nyp, $
	id_mode:bg_mode, id_method:bg_method, $ 
	id_harm:bg_harm, id_calc:bg_calc, id_run_mode:bg_run_mode, $
	ids:ids, harm_tx:harm_tx, calc_tx:calc_tx, ssz:ssz, $
	ptr_action:ptr_action, ptr_data:ptr_data}

WIDGET_CONTROL, setbase_modal, /REALIZE

WIDGET_CONTROL, setbase_modal, SET_UVALUE=info, /NO_COPY

XMANAGER, 'XUS_SET', setbase_modal, GROUP_LEADER=group	; not necessary to call cleanup routine as all cleanup follows here

action = *ptr_action
IF action eq 'DO' THEN BEGIN
  dat1 = *ptr_data
  data = {dat1:dat1}
ENDIF

; Free pointers
ptr_free, ptr_action, ptr_data

END ; XUS_SET
