FUNCTION us_read3d_plt, z, x, y, p1, p2, p3, p4, S3=s3, SP=sp, NOPOL=nopol, SKIP=skip, FNAME=fname, GROUP=group

;  Function to read 3d data generated from the code us.f (spectral
;  calculations) or code SPECTRA from T. Tanaka, Japan.
;  Data are read in the first quadrant and if the aperture was centered,
;  symmetry is used to extend the data. The extensions in x and y need not
;  be the same. The error status is the return value. If there is no error,
;  0 is returned. If there is an error, either -1 or the error_status flag
;  from routine rascii is returned.
;
;       Author: Roger J. Dejus (dejus@aps.anl.gov), XFD/APS, March, 1995.
;       MODIFICATION HISTORY:
;
;        Date     | Name  | Description
; ----------------+-------+-----------------------------------------------------
; 11-NOV-1997     | RJD   | Renamed calls: read_ascii -> rascii, xtext ->
;                 |       | widget_message, and renamed the routine itself
;                 |       | read3d_plt -> us_read3d_plt to reflect that the
;                 |       | output must indeed come from the code us.f.
; ----------------+-------+-----------------------------------------------------
; 12-JAN-2001     | MSR   | Bug reported by Roger fixed: Default fname
;                 |       | changed from us.plt to us.out
; ----------------+-------+-----------------------------------------------------
; 13-AUG-2008     | RJD   | Added the CATCH error handler and the GROUP keyword.
; ----------------+-------+-----------------------------------------------------
; 30-SEP-2009     | RJD   | Added to read data from code SPECTRA, which
;                 |       | increments the y-coordinate before the x-coordinate.
;                 |       | Set the kewword "SP" to indicate a SPECTRA output
;                 |       | file.
; ----------------+-------+-----------------------------------------------------
; 12-DEC-2012     | RJD   | Added keyword NOPOL to omit checking for polarization data.
;                 |       | This is useful to be able to read 3-d data from code
;                 |       | SPECTRA, which only stores x,y,z, and not the Stoke's
;                 |       | parameters.
; ----------------+-------+-----------------------------------------------------
; 18-JAN-2013     | RJD   | Added keyword SKIP which is passed to routine RASCII
;                 |       | to be able to skip header lines.
; ----------------+-------+-----------------------------------------------------

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Catch, /Cancel
   Message,/Info,'Error caught: Error reading US 3d plot file'
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'US_READ3D_PLT: Error caught: Error reading US 3d plot file')
   RETURN,error_status
ENDIF

if keyword_set(s3) then s3 = -1.0 else s3 = +1.0 ;switch sign for 3rd & 4th if set
if n_elements(fname) eq 0 then fname = 'us.out'
a = rascii(fname, npoints=nd, skip=skip, error_status=error_status)
if error_status ne 0 then begin
  Message, 'US_READ3D_PLT: Error reading 3d plot file'  ; issue Message to jump to the catch error handler
;  print, 'US_READ3D_PLT: Error reading 3d plot file'
;  return, error_status
endif
ncol = nd(0) & nrow = nd(1)

;  Check if 3d data file
if not keyword_set(nopol) then begin
  if ncol ne 7 then begin  ; (w/  polarization)
    junk = Dialog_message(/ERROR,Dialog_Parent=group,['',' US_READ3D_PLT',' Error reading file...',$
      ' This seems not a proper',' output 3d-file  '])
    return, -1
  endif
endif

;  Number of unique elements in x and y
ix = uniq(a(0,*),sort(a(0,*)))
iy = uniq(a(1,*),sort(a(1,*)))
nx = n_elements(ix)
ny = n_elements(iy)
;xa = reform(a(0,ix)) ; unique x values
;ya = reform(a(1,iy)) ; unique y values

;  Check if consistent gridding of data
if nx*ny ne nrow then begin
  print, 'US_READ3D_PLT: Inconsistent gridding of data'
  return, -1
endif

;  Define new dimensions
mx_sz = 2*nx -1 & mx = nx -1
my_sz = 2*ny -1 & my = ny -1

; Prepare data in first quadrant to be reflected
if not keyword_set(sp) then begin ; output from code US
  x  = transpose(reform(a(0,*), ny, nx))
  y  = transpose(reform(a(1,*), ny, nx))
  z  = transpose(reform(a(2,*), ny, nx))
  p1 = transpose(reform(a(3,*), ny, nx))
  p2 = transpose(reform(a(4,*), ny, nx))
  p3 = transpose(reform(a(5,*), ny, nx))
  p4 = transpose(reform(a(6,*), ny, nx))
endif else begin                ; output from code SPECTRA
  x  =          (reform(a(0,*), ny, nx))
  y  =          (reform(a(1,*), ny, nx))
  z  =          (reform(a(2,*), ny, nx))
  if not keyword_set(nopol) then begin
    p1 =                (reform(a(3,*), ny, nx))
    p2 =                (reform(a(4,*), ny, nx))
    p3 =                (reform(a(5,*), ny, nx))
    p4 =                (reform(a(6,*), ny, nx))
  endif
endelse

;  If aperture not centered cannot make use of symmetry
if x(0,0) ne 0.0 or y(0,0) ne 0.0 then return, 0

; Apply symmetry
; Define new arrays
xn  = fltarr(mx_sz, my_sz)
yn  = fltarr(mx_sz, my_sz)
zn  = fltarr(mx_sz, my_sz)
p1n = fltarr(mx_sz, my_sz)
p2n = fltarr(mx_sz, my_sz)
p3n = fltarr(mx_sz, my_sz)
p4n = fltarr(mx_sz, my_sz)

; First quadrant
xn (mx,my) = +x
yn (mx,my) = +y
zn (mx,my) =  z
if not keyword_set(nopol) then begin
  p1n(mx,my) =  p1
  p2n(mx,my) = +p2
  p3n(mx,my) =  p3
  p4n(mx,my) =  p4
endif

; Second quadrant
xn (0,my) = -rotate(x ,5)
yn (0,my) = +rotate(y ,5)
zn (0,my) =  rotate(z ,5)
if not keyword_set(nopol) then begin
  p1n(0,my) =  rotate(p1,5)
  p2n(0,my) = -rotate(p2,5)
  p3n(0,my) =  rotate(p3,5)
  p4n(0,my) =  rotate(p4,5)
endif

; Third quadrant
xn (0,0) =   -rotate(x ,2)
yn (0,0) =   -rotate(y ,2)
zn (0,0) =    rotate(z ,2)
if not keyword_set(nopol) then begin
  p1n(0,0) =    rotate(p1,2)
  p2n(0,0) =   +rotate(p2,2)
  p3n(0,0) = s3*rotate(p3,2)
  p4n(0,0) =   rotate(p4,2)
endif

; Fourth quadrant
xn (mx,0) =   +rotate(x ,7)
yn (mx,0) =   -rotate(y ,7)
zn (mx,0) =    rotate(z ,7)
if not keyword_set(nopol) then begin
  p1n(mx,0) =    rotate(p1,7)
  p2n(mx,0) =   -rotate(p2,7)
  p3n(mx,0) = s3*rotate(p3,7)
  p4n(mx,0) =    rotate(p4,7)
endif

; Make assignments
x  = xn
y  = yn
z  = zn
if not keyword_set(nopol) then begin
  p1 = p1n
  p2 = p2n
  p3 = p3n
  p4 = p4n
endif

return, 0
END ; us_read3d_plt
