
;+
;
;================================   MARE    =====================================
;
;
; MARE is an widget based graphical interface to calculate: 
;  - Spaghetti plots (lambda versis Psi for multiple crystal reflection)
;  - The Umweganregung peak location plot (the diffracted wavelength 
;       lambda vs. Psi) for a given primary reflection,i.e., an 
;       horizontal cut of the spaghetti plot.  
;  - The Glitches spectrum (the negative intensity for versus the
;        wavelength) or a vertical cut of the spaghetti plot. 
;
;  Psi is the azimutal angle of totation, i.e., the totation around
;    the H vector (main reflection)
;
;
;
; In other words, if a crystal is set with a particular Bragg angle to match 
; a given reflection (inputs: H,K,L) at a given wavelength (input: WaveLength), 
; many other (secondary) reflections are excited when the crystal is rotated 
; around the azimutal angle Psi, without changing the Bragg angle. 
;
; The plot (WaveLength,Psi) of the possible reflections is calculated and
; contain all possible reflection curves up to a maximum reflection 
; (input: H Max,  K Max, L Max). 
;
; Umweg plot: 
; The intersection of these curves with an horizontal line at the wavelength 
; of the primary reflection (input: WaveLength) gives the position of the 
; peaks in the unweg plot. The width of each peak depends on the pendent of 
; the curve at the intersection. For that, the Psi1 and Psi2 intersection 
; angles with a band of width (input: DeltaWaveLength) are calculated. With 
; this width and the intensity of the diffraction line, it is possible to 
; compute a Gaussian that "roughly" describe the peak.
; 
; Glitches plot: 
; The intersection of these curves with a vertical line at a given Psi 
; gives the position of the peaks in the glitches plot. The width of each 
; peak is the difference between the wavelength values for Psi+/-DeltaPsi
; With this width and the intensity of the diffraction line, it is possible to 
; compute a Gaussian that "roughly" describe the peak.
; 
;
; The program computes first all Miller indices hkl associated to the
; different crystal planes which meet the diffraction condition (multiple
; diffraction events) in the two-beam multiple diffraction approximation.
;
; The structure factor, independent of the energy (i.e., no anomalous
; scattering) is calculated for each reflection. 
;
; In the spaghetti plot all reflections, except all that have structure
; factor zero (i.e., forbidden reflections) are plotted. 
;
; The azimuthal angle Psi is calculated relative to an arbitrarily chosen 
; reference direction perpendicular to the scattering vector of the primary 
; reflection.  (See Ref). The considered reflection is displayed in the 
; header of the xop macro that produces the plot.  
;
; The theoretical frame of the program is essentially provided in:
;     B.H. Cole et al, "Simultaneous Diffraction: Indexing Umweganregungen 
;     Peaks in Simple Cases", Acta Cryst. (1962), 15, 138-144
;
; The peak intensity is calculated approximately, as Fh^2*L*P, 
; being L the lorentz factor, and P the polarization factor. The latter
;  values are computed in the function bragg_lorentz(). 
;
; Note that this intensity is very approximated, because: 
;       - The formula is not valid for perfect crystals, only for powders.
;       - The total intensity must be the sum of the primary reflection
;         plus the intensity of the so-called operative reflections plus the
;         contribution of the cooperative ones. 
;       - Anomalous scattering factors are not considered. 
;       - The lorentz and polarization factor are not correctly used
;
; Therefore the resulting plots should be taken with care. 
; For a more detailed and accurate method, the following references may be used: 
;       Rossmanith, Acta Cryst (1992) A48 596-610
;       Rossmanith, Acta Cryst (2006) A62 174-177
;       Rossmanith, J. Appl. Cryst. (2003) 36, 1467-1474  
;
; The program should deal with any crystal geometry, however, only the 
; cubic geometries have been tested. 
;
;
; DESCRIPTION OF THE CONTROLS IN THE MAIN WINDOW:
;
;  File:
;   MARE input parameters: This option allows to save the current
;		parameters to a file for later loading. It also allows
;		to save the current parameters as defaults for being
;		used when the application is initialized. In the last
;		case, the file is named "application".xop (where 
;		"application " is the name of the current XOP
;		application) and is written in the directory pointed
;		by the XOP_DEFAULTS_DIR environment variable (which
;		must be set). The parameter file is ASCII and can be
;		read and edited with care.
;  Quit: to exit from the program
;
; Set_Parameters:
;  Set Parameters: to define the parameters for the calculation.
;		The same result is obtained pressing the "Set Parameters"
;		button in the main BM window. 
;               Please refer to the information under the HELP
;		button for a complete description of the parameters. 
;  Set Defaults: Sets the default parameters.
;
;  Help:   Shows the BM help (this text).
;
;
;  COPYRIGHT:
; 	MARE belongs to XOP package and it is distributed within XOP.
; 	PLEASE REFER TO THE XOP COPYRIGHT NOTICE
; 
;  REFERENCE:
; 	Published calculations made with XOP should refer:
; 
; 	M. Sanchez del Rio and R. J. Dejus 
;         "Status of XOP: an x-ray optics software toolkit"
;         SPIE Proceedings Vol. 5536 (2004) pp.171-174
; 
;         http://dx.doi.org/10.1117/12.560903
; 
;  LAST MODIFICATION: srio@esrf.eu 2012-03-01
; 
; 
;-
; =========================================================================
;
;	MODIFICATION HISTORY:
;       by  Manuel Sanchez del Rio. ESRF. January 2007.
;       This program replaces MAMON. 
;       It fixes some bugs (with the reflection rules, and intensity
;       values). However, neutrons are not yet implemented. 
;
;       20120301 srio@esrf.eu includes fh edge (requested by ferrero@esrf.eu).
;                             Version 1.01.
;
; =========================================================================
;
Function mare_version
return,'1.01'
end
;
;=====================================================================
;
PRO mare_event,event


Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
     'MARE_EVENT: error caught: '+!err_string)
   Catch, /Cancel
   If Type(stateid) EQ 3 THEN $
     If Widget_Info(stateid,/Valid_Id) AND N_Elements(state) NE 0 THEN $
    Widget_Control,stateid,Set_UValue=state ; ,/No_Copy
   On_Error,2
   RETURN
ENDIF

Widget_Control, event.id, get_UValue=eventUValue

if n_elements(eventuvalue) EQ 0 then eventuvalue = ''
if not(keyword_set(eventuvalue)) then eventuvalue = ''

stateid = Widget_Info(event.handler,/Child)
Widget_Control, stateid, get_UValue=state ; , /No_Copy

case eventuvalue of

  'FILEINPUT': BEGIN
    action=''
    Widget_Control,event.id, Get_Value=action
    CASE action OF
      'Load from file...': BEGIN
        if sdep() EQ 'UNIX' then filter='*.xop' else filter=0
        str_par = Xop_Input_Load(Title=$
        'Select MARE input file...',$
        /NoConf,Filter=filter,Group=event.top)
        IF Type(str_par) EQ 8 THEN BEGIN
          tmp = state.str.parameters
          Copy_Structure,str_par, tmp, Group=event.top , /OnlyFirstField
          state.str.parameters = tmp
        ENDIF
      END
      'Save to file...': BEGIN
        str_par = state.str.parameters
        Xop_Input_Save,str_par,File='mare.xop',$
          /Write, Group=event.top, Comment='; xop/mare(v'+$
        mare_version()+') input file on '+SysTime()
      END
      'Save as default': BEGIN
        str_par = state.str.parameters
        Xop_Input_Save,str_par,Group=event.top, $
          Default='mare.xop',Comment='; xop/mare(v'+$
        mare_version()+') input file on '+SysTime()
      END
    ENDCASE
  END
  'QUIT':begin	
	delete_files,['mare.bra','mare.out','mare.inf']
	widget_control,/destroy,event.top
	return
	end
  'HELP': Xhelp,'mare',GROUP=event.top

  'SETDEF': BEGIN
		itmp = Dialog_Message(Dialog_Parent=event.top,$
		/Question,['This option initializes the',$
		'mare parameters to their default values.',$
		'Then you must click Set_parameters to run the program.',$
		'Please confirm.'],title='mare')
		if itmp eq 'No' then goto,out
		state.str.parameters = state.str_defaults
	END
  'SET':begin	
	delete_files,['mare.bra','mare.out','mare.inf']
	str1 = state.str.parameters
        ;helpcmd="xdisplayfileNative,'"+Xop_GetEnv('XOP_HOME')+$
        ;         sdep(/ds)+'doc'+sdep(/ds)+'txt'+sdep(/ds)+"mare_par'"
        helpcmd='xhelp,"mare_par"'
  	XScrMenu,str1,/Interp,/NoType,action=action,Ncol=3, $
		titles=state.str.titles,flags=state.str.flags, $
		help=helpcmd,wtitle='mare input parameters', $
		Group=event.top
  	if  action EQ 'DONT' then goto,out
  	widget_control,/hourglass

	;
	; run mare_calc
	;
        state.str.parameters = str1
        mare_calc,str1,error=error
        IF error EQ 1 THEN BEGIN
           itmp = Dialog_Message('MARE: no good reflections found', $
             DIALOG_PARENT=event.top,/Error)
        ENDIF

	end
  else:
endcase

out:
Widget_Control, stateid, set_UValue=state ; , /No_Copy
end
;
;=======================================================================
;
PRO mare,GROUP=group, InputFile=inputFile, No_Block=no_Block
;
Forward_Function dabax_defaults
;
Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'XOP_IFC_READ: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

IF xregistered('mare') THEN RETURN

IF N_Elements(no_block) EQ 0 THEN no_block=1

wbase = widget_base(/COLUMN,TITLE='mare '+mare_version(),$
	MBAR=wMenuBar)
;
;
; the blocks box
;
Bbox=widget_base(wbase,/Column) ; also to store state
 

;Menu bar
wFile =  WIDGET_BUTTON(wMenuBar, VALUE='File', /MENU)
  wtmp0 = widget_button(wFile,VALUE='mare input parameters', /Menu)
    wtmp = widget_button(wtmp0,VALUE='Load from file...',UValue='FILEINPUT')
    wtmp = widget_button(wtmp0,VALUE='Save to file...',UValue='FILEINPUT')
    wtmp = widget_button(wtmp0,VALUE='Save as default',UValue='FILEINPUT')
  wtmp = widget_button(wFile,VALUE='Quit', UVALUE='QUIT',/SEPARATOR)

wSetParameters = widget_button(wMenuBar,VALUE='Set_Parameters', /MENU)
  wtmp = widget_button(wSetParameters,VALUE='Set Parameters', UVALUE='SET')
  wtmp = widget_button(wSetParameters,VALUE='Set Defaults', UVALUE='SETDEF')

;wResults = widget_button(wMenuBar,VALUE='Show',/MENU)
;  wtmp = widget_button(wResults,VALUE='Plot crystal parameters', UVALUE='SHOW')
;  wtmp = widget_button(wResults,VALUE='Show crystal parameters', UVALUE='SHOW')
;  wtmp = widget_button(wResults,VALUE='Show crystallographic parameters', $
;        UVALUE='SHOW')


wHelpMenu = WIDGET_BUTTON(wMenuBar, VALUE='Help', /HELP)
  wtmp = WIDGET_BUTTON(wHelpMenu, VALUE='mare', UVALUE='HELP')

tmp = widget_button(Bbox,VALUE='Set Parameters',UVALUE='SET')



if sdep() EQ 'WINDOWS' then $
font = 'VERDANA*BOLD*ITALIC*24' else $
font = '-adobe-helvetica-bold-o-normal--18-180-75-75-p-104-iso8859-1'

junk = WIDGET_LABEL( Bbox, FONT=font, VALUE='mare')
junk = WIDGET_LABEL( Bbox, FONT=font, VALUE='Multiple Beam Diffraction.')


str = dabax_defaults('mare',group=wbase)
str_defaults = str.parameters


IF KeyWord_Set(inputFile) THEN BEGIN
  str_par = Xop_Input_Load(InputFile=inputFile)
  IF Type(str_par) EQ 8 THEN BEGIN
    tmp = str.parameters
    Copy_Structure,str_par, tmp, Group=group, /OnlyFirstField
    str.parameters = tmp
  ENDIF
ENDIF
state = { str:str, str_defaults:str_defaults }

widget_control,Widget_Info(wbase,/Child),set_uvalue=state
widget_control,wbase,/REALIZE
xmanager,'mare',wbase,GROUP=group, No_Block=no_Block
;
end
