Function atomic_constants,mdescriptor,h,file=input,return=out,group=group

; 
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       ATOMIC CONSTANTS
;
; PURPOSE:
;	Returns atomic constants from DABAX.
;	
; CATEGORY:
;       X-Ray optics. DABAX data base.
;
; CALLING SEQUENCE:
;	out = atomic_constants(id,file,return=return)
; INPUTS:
;	id: an identifier (or an array of identifiers) to be found in the 
;	scan title (i.e. 'Si') or the scan number (i.e. 14)
; OPTIONAL INPUTS:
;	h (input/output) the dabax handle of the file to be used.
;	  If h is undefined or equal to 0, then the returned value is the
;	  dabax handle. If h is defined, then uses this value to point to
;	  the dabax file. This option is useful when calling 
;	  atomic_constants sequantially: the first call initiallizes h and 
;	  this value is reused (saving time) in the following calls.
;	
; KEYWORDS:
;	File = the DABAX  inout file (if not set, AtomicConstants.dat is taken)
;	Return = the variable to be returned. You can use either the
;		number of the column in the DABAX file, or a text
;		identifier (case insensitive) listed below:
;		RETURN='AtomicRadius'	or RETURN=0
;		RETURN='CovalentRadius'	or RETURN=1
;		RETURN='AtomicMass'	or RETURN=2
;		RETURN='BoilingPoint'	or RETURN=3
;		RETURN='MeltingPoint'	or RETURN=4
;		RETURN='Density'	or RETURN=5
;		RETURN='AtomicVolume'	or RETURN=6
;		RETURN='CoherentScatteringLength'	or RETURN=7
;		RETURN='IncoherentX-section'	or RETURN=8
;		RETURN='Absorption@1.8A'	or RETURN=9
;		RETURN='DebyeTemperature' or RETURN=10
;		RETURN='ThermalConductivity' or RETURN=11
;		RETURN='Z' or  RETURN=1000
;
;       group = The widget ID of the widget that calls the routine.
;               This is passed to widget_message in its dialog_parent
;               keyword to center the window message.
;
;
; OUTPUT:
;	out: the value of the selected parameter
;
; PROCEDURE:
;	Straightforward
;
; EXAMPLES:
;	print,atomic_constants('Si',return='AtomicMass')
;	    28.085500
;	print,atomic_constants(14,return='AtomicMass')
;           28.085500
;	print,atomic_constants([14,27],return='AtomicMass')
;	    28.085500       58.933200
;
; MODIFICATION HISTORY:
;       96-09-23 Written by M. Sanchez del Rio (srio@esrf.fr)
;       96-10-10 MSR adds FILE keyword and h optional input. 
;		Corrects a bug when single character symbols were used.
;	96-10-11 MSR adds the array option.
;	97-01-15 MSR adapts for Windows.
;	97-10-16 MSR adds group keyword.
;	2002-10-08 MSR adds DebyeTemperature and ThermalConductivity from 
;		Kittel's book
;	2007-01-31 srio@esrf.fr fixes a bug in displaying bad DW number
;-
on_error,2

if not(keyword_set(input)) then input='AtomicConstants.dat'
if not(keyword_set(h)) then h = dabax_access(input)


FOR MM = 0,N_ELEMENTS(mdescriptor)-1 DO BEGIN
descriptor = mdescriptor(MM)
; to handle elements with 1 char symbol:
if strlen(descriptor) eq 1 then descriptor = descriptor+' '
if type(descriptor) EQ 7 then $
  igood = dabax_select(h,descriptor,/casematch,group=group) else $
  igood = spec_scan(h,strcompress(descriptor,/rem),/index)
;print,'Using scan: ',spec_name(h,igood,/index)
z = double(strsubstitute(spec_headers(h,igood,'#S',/index),'#S',''))
IF MM EQ 0 THEN zetas=z ELSE zetas=[zetas,z]

data = spec_data(h,igood,/index)
;help,data

if n_elements(out) eq 0 then out = 0
if type(out) eq 7 then begin
  case strupcase(out) of
    'ATOMICRADIUS': 	out = 0
    'COVALENTRADIUS':	out = 1
    'ATOMICMASS':	out = 2
    'BOILINGPOINT':	out = 3
    'MELTINGPOINT':	out = 4
    'DENSITY':		out = 5
    'ATOMICVOLUME':	out = 6
    'COHERENTSCATTERINGLENGTH':	out = 7
    'INCOHERENTX-SECTION':	out = 8
    'ABSORPTION@1.8A':	out = 9
    'Z':		out = 1000
    'DEBYETEMPERATURE':	out = 10
    'THERMALCONDUCTIVITY':out = 11
    else: begin
	message,/info,'Return parameter unknown.'
	return,0
	end
  endcase
endif
IF out EQ 1000 THEN label = 'Z' ELSE label = (spec_labels(h,igood,/index))[out]
if out eq 1000 then outval = fix(z) else outval = data(out)
if MM eq 0 then moutval = outval else moutval = [moutval,outval]
ENDFOR

; 
; Warn in case of bad values
;
iBad = Where(moutval -0.01 LE 1e-5) 
IF iBad[0] NE -1 THEN BEGIN
  z = double(strsubstitute(spec_headers(h,zetas[ibad[0]],'#S',/index),'#S',''))
  itmp = Dialog_Message('Bad value (-0.01) found for '+label+' Z='+StrCompress(Fix(Z),/Rem))
ENDIF

return,moutval
end
