C
C     PROGRAM  crystal_ml
C
C     Calculates perfect crystal diffraction profiles using the 
C     multi lamellar method.
C
C     It uses DABAX library to get structure factors
C     Bragg [reflection] and Laue [transmission] cases
C
C     Ref.: R. Caciuffo,C. Ferrero, O. Francescangeli, S. Melone,
C     Rev.Sci.Instrum., Vol. 61, No. 11, November 1990
c
c     !!! It is recommended to use this code mainly for Bragg geometry
c     Caution is needed while using the code in the Laue case and                
c     the related results should be
c     compared with those of other codes using different approaches
C
C	list of main  variables                             
C       ------------------------
C
C    mu = linear absorption coefficient 
C    sk = x-ray path inside k-th layer
C    r0 = classical electron radius (m)
C    v = unit cell volume
C    thetab = BRAGG ANGLE
C    b = asymmetry factor
C    alpha = angle between crystal surface and bragg planes
C    cgross = polarization factor
C    cklein = dimensionless quantity related to curvature radius radius
C    lambda = incident wavelength [m]
C    a_small = parameter a in eq 10 from Caciuffo et al.
C    radius = crystal curvature radius
C    g = parameter related to absorption
C    t0 = thickness of one crystal layer 
C    t = crystal thickness
C    crystal_ml_ref = reflectivity of the curved crystal
C    ntot = total number of points to be calculated 
C    qsmall = parameter in formula 9
C    z = parameter in formula 12
C    w (in subroutine crystal_ml_ref) = parameter in formula 13
C    v (in subroutine crystal_ml_ref)= parameter in formula 14
C    agross = reduced thickness
C
C
C
C	modification history (M. Sanchez del Rio)
C       --------------------
C
C	reflex < 2.0, old version from C. Ferrero
C	reflex 2.0, Running on both Sun and HP systems. Bragg and Laue
C	  geometries.
C	reflex 2.1, October 1994. Automatic set of the structure
C	  factors. Nicer user input interface. General cosmetics.
C	crystal_ml 1.0 , January 1996. The program has been almost completely
C	  rewritten. 
C	 The code has been cleaned and fixed. No implicit variables. Works for
C	  any number of points and layers. Rename to be used with xop 2.0
C       srio@esrf.fr 2002/06/20 introduces the reflectivity of the first two 
C		layers in the output file. This is useful to estimate the
C		applicability of the method, looking ate the ratio of single 
C		layer reflectivity to the total one and to the overlap 
C		between the two consecutive layers. 
C       srio@esrf.fr 2002/06/21 Fete de la musique. Cleaned the use of DeltaY
C
C
C	Plese send all your comments and bugs to srio@esrf.fr
C
C	(c) 1997,2002 European Synchrotron Radiation Facility
C	    Scientific Software
C	    Contact: M. Sanchez del Rio, srio@esrf.fr
C

	implicit none

c input
        character*80    file_refl
        character*160    top_title
	real*8		phot,y0,ymax,dy0,alpha,radius,t,kstar
	integer*4	i_geometry,irun,ipol,iunits(3)

c calculations
	real*8		ref
	integer*4	idebugger
	character*5 	geo 
	integer*4	ntot,i
	real*8		rk0,thetab,rn,d_spacing,lambda,cgross
        complex*16      f_0,f_h,f_hbar
        complex*16      psi_0,psi_h,psi_hbar
        complex*16      refrac
        complex*16      struct
	real*8		d,v,mu
	real*8		tmp
	real*8		r1layer,r2layer
        real*8 deltay
	integer*4	j,jold,iblank2

	       complex*16 chi0,chih,cchih
	       real*8	chi0i, chi0r, fh,thetain,thetaout
	       complex*16	qsmall
	       real*8		gam0,gamh,ga0,gah
	       real*8		b,g,t1,c1,c2,c3,ta
	       real*8		q1,q2,q3,agross,t0,sk,fabs
	       real*8		cklein,a_small,tfact,y,e
	       integer*4	n

c constants
	real*8	TODEG,TWOPI,PI,R0


C common blocks
      COMMON /GEN   / CGROSS,B,MU,a_small,QSMALL,CHIH,SK,AGROSS,G,N
      common /add/chi0,cchih,deltay,gam0,rk0,t0,geo


      DATA    TODEG   / 57.2957 79513 08232 08767 98155 D0 /
C      DATA    TORAD   /  0.0174 53292 51994 32957 69237 D0 /
      DATA    TWOPI   /  6.2831 85307 17958 64679 25287 D0 /
      DATA    PI      /  3.1415 92653 58979 32384 62643 D0 /
C      DATA    R0      /  2.818 D-15/
      DATA    R0      /  2.817940289458 D-15/

c
c debugger set (0=No, 1=Yes)
c
	IDEBUGGER = 0
c
c default values
c
*case 1: Laue Si bent crystal
	file_refl = 'xcrystal.bra'		!crystal input file
	PHOT = 33144.0		!photon energy [eV]
	I_GEOMETRY = 2		! 1: Bragg, 2: Laue
	Y0 = -25.		! Ymin
	YMAX = 25.		! Ymax
	DY0 = 0.2		! Ystep
	IPOL = 1			! polarization [1=s, 2=p, 3=unpol]
	ALPHA = 63.78		! asymmetry angle
	radius = 12.98		! R [m]
	T = 0.700000		! crystal thickness [mm]
	kstar = 0.282352D0      ! elastic coefficient [Ge:0.377, Si:0.282352]


*case 2: Bragg Si crystal
c	file_refl = 'xcrystal.bra'		!crystal input file
c        PHOT = 8000.0          !photon energy [eV]
c        I_GEOMETRY = 1          ! 1: Bragg, 2: Laue
c        Y0 = -5.                ! Ymin
c        YMAX = 5.               ! Ymax
c        DY0 = 0.05              ! Ystep
c	 IPOL = 1                ! polarization [1=s, 2=p, 3=unpol]
c        ALPHA = 0.0000           ! asymmetry angle
c        radius = 1000.           ! R min [m]
c        T = 1.00000            ! crystal thickness [mm]
c        kstar = 0.282352D0      ! elastic coefficient [Ge:0.377, Si:0.282352]
C

C
C---------------------- starts input section ---------------------
C
      WRITE (*,*) '***********************************************'
      WRITE (*,*) '************** crystal_ml 1.0 *****************'
      WRITE (*,*) '***********************************************'
      write(*,*)  ' '
      WRITE (*,*) 'THIS PROGRAM CALCULATES REFLECTIVITY CURVES FOR'
      WRITE (*,*) '                BENT CRYSTALS'
      write(*,*)  ' '
C      write(*,*)  ' Run crystal_ml with the input parameters from: '
C      write(*,*)  '     [0] default values'
C      write(*,*)  '     [1] input from keyboard'
C      write(*,*)  '     [2] input from file cryst_ml.inp'
C	read(*,*) IRUN
	IRUN = 2

C	if (irun.eq.1) call crystal_inp(file_refl,PHOT,
C     $I_GEOMETRY,Y0,YMAX,DY0,IPOL,ALPHA,radius,T,KSTAR)

	if (irun.eq.2) then
	  write(*,*) 'Reding inputs from file cryst_ml.inp'
	  open (11,file='cryst_ml.inp',status='old')
	  read(11,'(a)') file_refl
	  read(11,'(a)') top_title
	  read(11,*) PHOT
	  read(11,*) I_GEOMETRY
	  read(11,*) Y0
	  read(11,*) YMAX
	  read(11,*) DY0
	  read(11,*) IPOL
	  read(11,*) radius
	  read(11,*) ALPHA
	  read(11,*) T
	  read(11,*) KSTAR
	  close(11)
	  write(*,*) 'Done.'
	endif
C----------------------- end of input section -------------------------

c
c open output files
c
      open (11,file='cryst_ml.inf',status='unknown')
      WRITE (11,*) '***********************************************'
      WRITE (11,*) '************ CRYSTAL_ML 1.0 *******************'
      WRITE (11,*) '***********************************************'
	open (10,file='cryst_ml.dat',status='unknown')
	write(10,'(a)') '#F crystal_ml.dat'
	write(10,'(a)') '#S 1 crystal_ml run'
	write(10,'(a)') '#C results of crystal_ml [multi lamellar] run'
	write(10,'(a)') '#N 6'
	write(10,'(a)') '#UT0 '//top_title(1:iblank2(top_title))
	write(10,'(a)') '#L angle[mrad]  angle[deg]  y
     1  1st layer Ref  2nd layer Ref  Reflectivity'
c 
c immediate definition of other variables
c
	if (t.eq.0) t=1000.0
	if (radius.eq.0) radius=1000000.0
	iunits(1) = 6
	iunits(2) = 10
	iunits(3) = 11

	lambda = 12398.54/phot
        rk0=1.00d0/lambda
	if (alpha.GE.90.0) alpha = 89.99
	ntot = int((ymax-y0)/dy0+1.0d0)    

	if (i_geometry.EQ.1) then
	  geo = 'BRAGG' 
	else if (i_geometry.EQ.2) then
	  geo = 'LAUE'
	else
	  do i=1,3
	    write(iunits(i),*) 
     $           'cryst_ml aborted: wrong crystal geometry '
	    if (iunits(i).ne.6) close(iunits(i))
	  enddo
	  stop
	endif

c
c write info
c
      WRITE (11,*) '**Input section:'
      write(11,*) 'crystal input file: ',file_refl
      write(11,*) 'photon energy [eV]: ',PHOT
      write(11,*) 'geometry [1: Bragg, 2: Laue]',I_GEOMETRY
      write(11,*) 'Y min: ',Y0
      write(11,*) 'Y max: ',YMAX
      write(11,*) 'Y step: ',DY0
      write(11,*) 'Polarization [1=s,2=p,3=unpolarized]: ',IPOL
      write(11,*) 'crystal curvarure radius [m]: ',radius
      write(11,*) 'asymmetry angle [deg]: ',ALPHA
      write(11,*) 'crystal thickness [mm]: ',T
      write(11,*) 'Poisson ratio   : ',kstar

      write(11,*) '**derived:'
      write(11,*) 'photon wavelength [A]: ',LAMBDA
      write(11,*) 'photon wavenumber (1/lambda) [A^-1]: ',rk0
      write(11,*) 'geometry: ',GEO
      write(11,*) 'number of points: ',NTOT
      write(11,*) ' '


	if (idebugger.EQ.1) then 
          write(*,*) 'End of Input Section, Parameters are: '
	  write(*,*) '>> Selected inp file : ',file_refl
	  write (*,*) '>> energy [eV] = ',phot
	  write (*,*) '>> i_geometry = ',i_geometry
          write (*,*) '>> Y0 = ',Y0
          write (*,*) '>> YMAX = ',YMAX
	  write (*,*) '>> DY0 = ',DY0
          write (*,*) '>> IPOL [polarization]  = ',IPOL
	  write(*,*) '>> radius : ',radius
	  write(*,*) '>> ALPHA [asymmetry angle]: ',ALPHA
	  write(*,*) '>> T [thickness [m]: ',T
          write(*,*) '>> End of Input Section Parameters. '
	  write (*,*) '>> lambda [Angstrom] = ',lambda
	  write (*,*) '>> geo = ',geo
          write (*,*) '>> NTOT (# of points)=',NTOT
          write(*,*) ' '
	endif
C---------------------- start calculations ----------------------------

C--------------------- starts struct fact  calculations ---------------------

C Call crystal_fh to read the BRAGG file
	    call crystal_fh(-1,file_refl,phot,thetab,
     $f_h,f_hbar,f_0,psi_h,psi_hbar,psi_0,refrac,
     $rn,d_spacing)


C calculates Bragg angle

	thetab = dasin(lambda*1.0E-8/2.0D0/d_spacing)

      if (ipol.eq.1) then                                             
        cgross = 1.00D0                                             
      else if (ipol.eq.2) then                                      
        cgross = dabs(dcos(2.00d0*thetab))                         
      else if (ipol.eq.3) then                                    
        cgross = (1.00d0+dabs(dcos(2.00d0*thetab)))/2.00D0       
      else                                                     
          do i=1,3
            write(iunits(i),*) 
     $      'cryst_ml aborted: wrong polarization number.'
            if (iunits(i).ne.6) close(iunits(i))
          enddo
          stop
      end if

C
C interlattice distances and cell volume
C
	d = d_spacing*1d-2     ! d in meters
	v = 2.817939 D-13/rn   ! V in cm^3
	v = v*1d-6             ! V in m^3

	     if (idebugger.eq.1) then 
	     write(*,*)'-------- from crystal_fh : -------'
	     write(*,*)'d_spacing [cm]: ',d_spacing
	     write(*,*)'Lambda [A]: ',Lambda
	     write(*,*)'Lambda [cm]: ',Lambda*1.0E-8
	     write(*,*)'Phot [eV]: ',Phot
	     write(*,*)'Theta Bragg [rad]: ',thetab
	     write(*,*)'Theta Bragg [deg]: ',thetab*todeg
	     write(*,*)'-------- derived from crystal_fh : -------'
	     write(*,*)'IPOL : ',IPOL
	     write(*,*)'CGROSS : ',CGROSS
	     write (*,*) 'd [m]= ',d
	     write (*,*) 'v [m^3]= ',v
	     write(*,*)'-------- end from crystal_fh : -------'
	     endif

C Structure factor

	     write(11,*)' '
	     write(11,*)' '
	     write(11,*)'##########################################'
	     write(11,*)'Starts data from Struct Fact calculation.'
	     call crystal_fh(11,file_refl,phot,thetab,
     $f_h,f_hbar,f_0,psi_h,psi_hbar,psi_0,refrac,
     $rn,d_spacing)
	     write(11,*)'Ends data from Structure Factor calculation.'
	     write(11,*)'##########################################'
	     write(11,*)' '

#if lahey
	     mu =  2.0D0 * twopi *(-aimag(refrac)) / (lambda*1.0d-8)
#else
	     mu =  2.0D0 * twopi *(-dimag(refrac)) / (lambda*1.0d-8)
#endif
	     mu = mu * 1.0d2              !m^-1
cfer  struct = sqrt(f_h * f_hbar)
#if lahey
	     struct = sqrt(f_h * f_hbar)
#else
	     struct = cdsqrt(f_h * f_hbar)
#endif
	     lambda = lambda * 1.0d-10    !m

	     if (idebugger.EQ.1) then
	       write(*,*)'-------- str factors : -------'
	       write (*,*) '** RN         : ',RN
	       write (*,*) '** D_SPACING  : ',D_SPACING
	       write (*,*) '** F_H     : ',F_H
	       write (*,*) '** F_HBAR  : ',F_HBAR
	       write (*,*) '** F_0     : ',F_0
	       write (*,*) ' '
	       write (*,*) '** PSI_H   : ',PSI_H
	       write (*,*) '** PSI_HBAR: ',PSI_HBAR
	       write (*,*) '** PSI_0   : ',PSI_0
	       write (*,*) ' '
	       write (*,*) '** REFRAC  : ',REFRAC
	       write(*,*)'-------- derived from str factors : -------'
	       write (*,*) '** STRUCT : ',STRUCT
	       write (*,*) '** MU [m^-1]  : ',MU
	       write (*,*) '** LAMBDA [m]  : ',LAMBDA
	       write(*,*)'-------- end from str factors : -------'
	     endif

C--------------------- end struct fact  calculations ---------------------

C

	     chih = psi_h
	     cchih =psi_hbar
	     chi0 = psi_0
#if lahey
	     chi0i = -aimag(psi_0)
	     chi0r = -real(psi_0,8)
#else
	     chi0i = -dimag(psi_0)
	     chi0r = -dreal(psi_0)
#endif
c       write (*,*) '0*** chi0r (double precision) =',chi0r
c     	fh = abs(struct)
#if lahey
      	fh = abs(struct)
#else
      	fh = cdabs(struct)
#endif

C
C calculate director cosines
C
      alpha = alpha*pi/180.00d0
C      teta = thetab+alpha                                      
      if (geo.eq.'BRAGG') then
C These lines are wrong MSR 97/10/09
C	     thetain = thetab-ALPHA
C	     thetaout = thetab+alpha
	     thetain = thetab+ALPHA
	     thetaout = thetab-alpha
        gam0 = dsin(thetain)
csrio        gamh = dsin(thetaout)
        gamh = -dsin(thetaout)
csrio        B = -gam0/gamh 
        B = gam0/gamh 
      else if (geo.eq.'LAUE') then
c
c --- changed on May, 11th,1997 because Manolo didn't like the previous
c --- notation
c
C         chi = pi/2.0d0-ALPHA
c thetain = thetab-(pi/2.0d0-ALPHA)
c thetaout = thetab+(pi/2.0d0-ALPHA)
C         gam0 = dcos(thetab-chi)
C         gamh = dcos(thetab+chi)
c        gam0 = dcos(thetain)
c        gamh = dcos(thetaout)
c        B = gam0/gamh
c----------------------------------------------------------------------
      thetain = thetab + ALPHA
      thetaout = thetab - ALPHA
      gam0 = dsin(thetain)
      gamh = dsin(thetaout)
      B = - gam0/gamh
C SHOULDN't BE LIKE THIS?
c      gamh = -dsin(thetaout)
c      B = gam0/gamh
c----------------------------------------------------------------------
      endif
	if (idebugger.EQ.1) then 
	  write (*,*) '>> geo: ',geo
C	  write (*,*) '>> teta: ',teta
	  write (*,*) '>> thetab: ',ALPHA
	  write (*,*) '>> ALPHA: ',ALPHA
C	  write (*,*) '>> chi: ',chi
          WRITE (*,*) '>> Asymmetry factor      b = ',B         
          WRITE (*,*) '>> Incident  cosine GAMMA0 = ',GAM0     
          WRITE (*,*) '>> Reflected cosine GAMMAH = ',GAMH    
	endif

      if ((geo.eq.'BRAGG'.and.b.gt.0.0d0).or.(geo.eq.'LAUE'.and.b.
     &lt.0.0d0)) then
          do i=1,3
            write(iunits(i),*) 
     $      'cryst_ml aborted: Asymmetry factor not '
     $      //'consistent with chosen geometry'
            write (iunits(i),*) geo ,'  ',' B = ',B
            if (iunits(i).ne.6) close(iunits(i))
          enddo
          stop
      endif

	     qsmall = b*chih*cchih*cgross*cgross 
cfer  g = (1.00d0-b)*chi0i/(2.00d0*cgross*dabs(b)**0.50d0*dabs(chih))

#if lahey
	     g = (1.00d0-b)*chi0i/(2.00d0*cgross*dabs(b)**0.50d0*
     &sqrt(abs(chih*cchih)))
#else
	     g = (1.00d0-b)*chi0i/(2.00d0*cgross*dabs(b)**0.50d0*
     &dsqrt(cdabs(chih*cchih)))
#endif
      
	     t1 = (1.0d0-1.00d0/b)*chi0r/(2.0d0*dsin(2.0d0*thetab))          

	     c1 = pi*v**2*gam0*(b-1.00d0)                                
cfer  c2 = 1.00d0+b*gamh**2*(1.00d0+kstar)                     
	     c2 = 1.00d0+b*gamh**2*(1.00d0+kstar/(1.0d0-kstar))                     

	     ga0 = dabs(gam0)                                             
	     gah = dabs(gamh)                                            
	     ta = (1.00/mu)*ga0*gah/(ga0+gah)
	     q1 = dabs(gam0*gamh)                                
#if lahey
	     q2 = lambda*sqrt(q1)                              
#else
	     q2 = lambda*dsqrt(q1)                              
#endif

      c3 = r0*r0*lambda**3*fh*fh*b*radius
	     cklein = c1*c2/c3
	if (idebugger.EQ.1) then
	  write(*,*) '**** g',g
	  write(*,*) '**** v',v
	  write(*,*) '**** gam0',gam0
	  write(*,*) '**** pi*gam0(b-1.0d0)',pi*gam0*(b-1.0d0)
	  write(*,*) '**** pi*gam0(b-1.0d0)*v*v,',pi*gam0*(b-1.0d0)*v*v
          write(*,*) '**** KSTAR',KSTAR
          write(*,*) '**** C1',C1
          write(*,*) '**** C2',C2
          write(*,*) '**** C3',C3
          write(*,*) '**** CKLEIN',CKLEIN
	end if
      write (11,*) '>> asymmetry angle [rad] = ',alpha
      write (11,*) '>> asymmetry angle [deg] = ',alpha*todeg
      write (11,*) ' '
      WRITE (11,*) '>> Bragg angle [rad] = ',thetab
      WRITE (11,*) '>> Bragg angle [deg] = ',thetab*todeg
      write (11,*) ' '
      WRITE (11,*) '>> Incident angle [rad] = ',thetain
      WRITE (11,*) '>> Incident angle [deg] = ',thetain*todeg
      write (11,*) ' '
      WRITE (11,*) '>> Output angle [rad] = ',thetaout
      WRITE (11,*) '>> Output angle [deg] = ',thetaout*todeg
      write (11,*) ' '
      WRITE (11,*) '>> Dimensionless curvature = ',CKLEIN
      write (11,*) '>> Curvature radius (m)    = ',radius
      write (11,*) ' '
      write (11,*) 
     $'>> Max. path of diffr. x-rays inside cryst. [mm]: ',ta*1d3
      write (11,*) ' '

      if (geo.eq.'BRAGG') then
	deltay = 2.0D0

Csrio        agross = 2.00d0/dabs(cklein)                                    
        agross = deltay/dabs(cklein)
      else if (geo.eq.'LAUE') then
	deltay = PI/2.0D0
Csrio        agross = pi/(2.00d0*dabs(cklein))                               
        agross = deltay/dabs(cklein)
      endif
	if (idebugger.EQ.1) then 
        write (*,*) '**** Q1 = ',Q1
        write (*,*) '**** Q2 = ',Q2
        write (*,*) '**** AGROSS = ',AGROSS
	endif

cfer  q3 = pi*abs(chih)*cgross
#if lahey
      q3 = pi*sqrt(abs(chih*cchih))*cgross
#else
      q3 = pi*dsqrt(cdabs(chih*cchih))*cgross
#endif
      t0 = q2*agross/q3                                                 

      IF (radius.GE.100.00D0) THEN                                       
	t = t*1d-3          ! from mm to meters
        t0 = t                                                          
        write (*,*) 'Infinite curvature radius. NLAYERS = 1'            
      ELSE                                                              
        write(*,*)'>>> Max. path of diffr. x-rays inside cryst. [mm]: ',
     $	ta*1d3
        write(*,*)'>>> Thickness of a single cryst. slab [mm]: ',
     $	t0*1d3,' mm'        
	T = T*1d-3
        if (t.lt.t0) then
          do i=1,3
            write(iunits(i),*) 
     $      'cryst_ml aborted: crystal thickness < slab thickness'
            if (iunits(i).ne.6) close(iunits(i))
          enddo
          stop
       endif
      ENDIF

      sk = t0/dabs(gamh)
      n = idnint(t/t0) 
      if (idebugger.EQ.1) write (*,*) 'No. of single crystal layers N ='
     $ ,N
      if (idebugger.EQ.1) write (*,*) 'sk : ',sk

      if (geo.eq.'BRAGG') then
        fabs = 1.0d0
      else 
        fabs = dexp(-(t-t0)*mu/gamh) 
      endif
c
c write parameters in "crystal_ml.inf"
c
      write (11,*) ' '
C      write (11,*) '>> Absorption coefficient mu (1/m)= ',MU      
C      write (11,*) ' '
      write (11,*) '>> Polarization factor = ',cgross
      write (11,*) '>> Asymmetry factor      B = ',B
      write (11,*) '>> Incident  cosine GAMMA0 = ',GAM0
      write (11,*) '>> Reflected cosine GAMMAH = ',GAMH
      write (11,*) ' '
c      write (11,*) '>> Number of points in a curve   NTOT = ',NTOT+1
      write (11,*) '>> Thickness of a single cryst. layer T0 (m) = ',t0
      write (11,*) 
     $'>> Thickness of a single cryst. layer T0 (microns) = ',t0*1.0D6
      write (11,*) ' '
      write (11,*) '>> No. of single crystal layers  N = ',N
      write (11,*) ' '
      write (11,*) ' '


C                            
C --- Small a in zachariasen refr.   
C                                   
      a_small = pi*t0/(lambda*gam0)           
C                                               
c      write (*,*) '0*** aux1 = ',chih*cchih
C                                               
#if lahey
      tfact = cgross*sqrt(abs(chih*cchih))/(b*dsin(2.0d0*thetab))*
     &sqrt(dabs(b))
#else
      tfact = cgross*dsqrt(cdabs(chih*cchih))/(b*dsin(2.0d0*thetab))*
     &dsqrt(dabs(b))
#endif
      write (11,*) '>>> 1*y [microrad] = ',tfact*1d6
      write (11,*) '>>> 1*y [deg] = ',tfact*todeg
      write (11,*) '>>> 1*y [arc sec] = ',tfact*todeg*3600.0D0
      write (11,*) ' '
      write (11,*) '>>> theta_b shift [microrad] = ',t1*1e6
      write (11,*) '>>> theta_b shift [deg] = ',t1*todeg
      write (11,*) '>>> theta_b shift [arc sec] = ',t1*todeg*3600.0D0
      write (11,*) ' '
      write (*,*) '>>> No. of single crystal layers for calculation: ',N
      WRITE (*,*) '>>> Dimensionless curvature = ',CKLEIN
      write(*,*) ' '
      write(*,*) '>>> Working... Please wait. <<<'


c
c loop over the number of points 
c
	y = y0                                                         
        write(*,*) 'done (%) :   0'
	jold = 0
	do 1313 i = 1,ntot

          tmp=float(i)/float(ntot)*100.0
          j = int(tmp/10.0)
          if (j.ne.jold) then
            write(*,*) 'done (%) : ',int(tmp)
            jold = j
          endif

Csrio::	  e = crystal_ml_ref(y)*fabs
	  CALL crystal_ml_ref(y,ref,r1layer,r2layer)
	  e = ref*fabs
	  write (10,'(6(g15.8,1x))') 1.0D3*(-t1-(tfact*y)),
     1          (-t1-(tfact*y))*180.0/PI,y,r1layer,r2layer,e
	  y = y + dy0
1313	continue
	close(10)


	close(11)
	END                                                     

C                                                                       
C********************************************************************C  
C     REAL    FUNCTION    crystal_ml_ref
C
C Calculate the reflectivity given by Eq. 6 of Caciuffo et al.
C Inputs: 
C         X the Y value of Zachariasen
C Outputs: 
C         ml_ref the reflectivity of the crystal
C         r1layer the reflectivity of the first layer
C         r2layer the reflectivity of the second layer
C
C********************************************************************C  
C                                                                       
Csrio::	double precision FUNCTION crystal_ml_ref (X,crystal_ml_ref)
	SUBROUTINE crystal_ml_ref (X,ml_ref,r1layer,r2layer)
	implicit none

C	common block variable declaration 
      REAL*8 CGROSS,B,MU,SK,AGROSS,G,GAM0,RK0,T0,ml_ref
      integer*4	N
      COMPLEX*16 CHIH,QSMALL
      COMPLEX*16 CHI0,CCHIH
      character*5 geo

C the rest of variables
C      double precision crystal_ml_ref
      COMPLEX*16 Z,C3                                   
      real*8 ztmp,q_big,a_small,x,y,zr,zi,zabsq,w,v
      real*8 d1,d2,d3,d4,ufr,uex,fact,trod
      real*8 prodt
      real*8 tsmall,rsmall,tsmall_old,rc1
      complex*16 u_small
      real*8 deltay
      real*8 r1layer,r2layer
      integer*4	i


C      DATA    PI      /  3.1415 92653 58979 32384 62643 D0 /

C common blocks
      COMMON /GEN   / CGROSS,B,MU,a_small,QSMALL,CHIH,SK,AGROSS,G,N
      COMMON /ADD/ CHI0,CCHIH,deltay,GAM0,RK0,T0,geo
C                                                                       
C start code
C
      Y = X                                                             
      ml_ref = 0.00D0                                                    
      PRODT = 1.00D0                                                    
      tsmall_old = 1.00D0
      r1layer = 0.0D0
      r2layer = 0.0D0

C starts loop
      DO 20 I=1,N                                                       
C                                                                       
C Eq 12 from Caciuffo et al.
C
cfer    ztmp = CGROSS*ABS(CHIH)*SQRT(ABS(B))
#if lahey
        ztmp = CGROSS*sqrt(abs(CHIH*cchih))*SQRT(ABS(B))
#else
        ztmp = CGROSS*dsqrt(cdabs(CHIH*cchih))*DSQRT(DABS(B))
#endif
        ZR = Y*ztmp                                                     
        ZI = G*ztmp                                                     
c       ZABSQ = ABS(Z)**2                                             
#if lahey
        Z = CMPLX(ZR,ZI,8)
        ZABSQ = ABS(Z)**2                                             
#else
        Z = DCMPLX(ZR,ZI)
        ZABSQ = cdABS(Z)**2                                             
#endif
C
C Eq 13 from Caciuffo et al.
C
cfer    q_big = ABS(QSMALL+Z*Z)
#if lahey
        q_big = ABS(QSMALL+Z*Z)
        u_small = SQRT(QSMALL+Z*Z)
        W = aIMAG(u_small)
        V = REAL(u_small,8)                                                   
        C3 = CONJG(Z)*u_small
#else
        q_big = cdABS(QSMALL+Z*Z)
        u_small = cdSQRT(QSMALL+Z*Z)
        W = dIMAG(u_small)
        V = dREAL(u_small)                                                   
        C3 = DCONJG(Z)*u_small
#endif
C
C
C

C Zachariasen variables
C D: delta

C
C RC1 is the numerator in Eq. 8 excluding q_big
C
        RC1 = DEXP(2.0*AGROSS*(1.0+B)*G/(1.0-B))                        

C
C D1,...,D4 are the addends  of the denominator of Eq. 7
C
C UFR is the numerator in Eq. 7 excluding the polarization (CGROSS^2)
C  factor
C UEX is the numerator in Eq. 8 excluding q_big
C
	  D1 = 0.0D0
	  D2 = 0.0D0
	  D3 = 0.0D0
	  D4 = 0.0D0
	  UFR = 0.0D0
	  UEX = 0.0D0
	  TSMALL = 0.0D0
	  RSMALL = 0.0D0
          IF (DABS(2.0*a_small*W).GT.175.0D0) THEN 
  	    if (n.eq.1) write (*,*) 'Approximating sin(av) by its average'
            if (geo.eq.'BRAGG') then
              D1 = q_big+ZABSQ                                                 
              D2 = 0.0D0                                                    
              D4 = 0.0D0                                                    
#if lahey
              D3 = 2.0D0*REAL(-C3,8)*SIGN(1.0D0,a_small*W) 
              UFR = ABS(B)*ABS(CHIH*CCHIH)                               
              UEX = DEXP(DLOG(RC1)-2.0D0*DABS(a_small*W))
#else
              D3 = 2.0D0*drEAL(-C3)*DSIGN(1.0D0,a_small*W) 
              UFR = dABS(B)*cdABS(CHIH*CCHIH)                               
              UEX = DEXP(DLOG(RC1)-2.0D0*DABS(a_small*W))
#endif
            else if (geo.eq.'LAUE') then
              if (n.eq.1) write (*,*) '!!!  over =',2.0*a_small*w 
            endif
          ELSE                                                            
#if lahey
            D1 = q_big+(q_big+ZABSQ)*DABS(SINH(a_small*W))**2
            D2 = (q_big-ZABSQ)*DSIN(a_small*V)**2
            D3 = REAL(-C3,8)*SINH(2.0D0*a_small*W)
#else
            D1 = q_big+(q_big+ZABSQ)*DABS(DSINH(a_small*W))**2
            D2 = (q_big-ZABSQ)*DSIN(a_small*V)**2
            D3 = dREAL(-C3)*DSINH(2.0D0*a_small*W)
#endif
#if lahey
            D4 = AIMAG(C3)*dSIN(2.0D0*a_small*V)
            UFR = (DSIN(a_small*V)**2+DABS(SINH(a_small*W))**2)
     $*DABS(B)*ABS(CHIH)**2
#else
            D4 = dIMAG(C3)*dSIN(2.0D0*a_small*V)
            UFR = (DSIN(a_small*V)**2+DABS(DSINH(a_small*W))**2)
     $*DABS(B)*CDABS(CHIH)**2
#endif
            UEX = RC1
          END IF

c 
c reflectivity (rsmall) and transmission (tsmall)
c
        if (geo.eq.'BRAGG') then
          RSMALL = CGROSS*CGROSS*UFR/(D1-D2+D3+D4)                     
          TSMALL = q_big*UEX/(D1-D2+D3+D4)                                 

        else if (geo.eq.'LAUE') then
          RSMALL = CGROSS*CGROSS*UFR*dexp(-mu*sk)/q_big
          TSMALL = (d1-d2-d3-d4)*uex/q_big                  

        endif


        IF (I.EQ.1) THEN                                                
          FACT = 0.00D0                                                 
        ELSE                                                            
          FACT = 1.00D0                                                 
        END IF                                                          

C
C Calculates the Nth term of the Eq. 6 and accumulates the result
C
C        RHOCH = FACT*MU*SK                                                      
C
C tsmall_old is tsmall corresponding to the layer I-1
C TROD is the term inside the Product sign in Eq.6
C

C	write(*,*) '::::::::: N: ',N
C	write(*,*) '::::::::: FACT: ',FACT
C	write(*,*) '::::::::: MU: ',MU
C	write(*,*) '::::::::: SK: ',SK
C	write(*,*) '::::::::: EXP: ',DEXP(-1.0D0*FACT*MU*SK)
C	write(*,*) '::::::::: '
C	write(*,*) '::::::::: RSMALL,TSMALL: ',RSMALL,TSMALL
C	write(*,*) '::::::::: RSMALL,TSMALL: ',RSMALL,TSMALL
C	write(*,*) '::::::::: '
        TROD = TSMALL_old*DEXP(-1.0D0*FACT*MU*SK)
C        TROD = TSMALL_old
C PRODT is all the parenthesis in Eq. 6 correspondint at the single I
        PRODT = PRODT*TROD                                            
C accumulate terms to the total sum
        ml_ref = ml_ref+(RSMALL*PRODT)
C	write(*,*) '//// I,ml_ref,TSMALL,RSMALL: ',
C     $I,ml_ref,TSMALL,RSMALL
	if (i.EQ.1) then
	  r1layer = RSMALL
	endif
	if (i.EQ.2) then
	  r2layer = RSMALL
	endif

C redefine TSMALL_OLD to be used in the next iteration
        TSMALL_old = TSMALL

Csrio::        IF (I-N) 10,20,20                                               
Csrio::   10  if (geo.eq.'BRAGG') then 
Csrio::        Y = Y+DELTAY(1)                                                 
Csrio::       else if (geo.eq.'LAUE') then
Csrio::        Y = Y+DELTAY(2)                                                 
Csrio::       endif
Csrio::	write(22,*) X,I,RSMALL,TSMALL,PRODT,ml_ref
        Y = Y+DELTAY
   20   CONTINUE                                                        

      RETURN                                                            
      E N D                                                             

