;+
; =================================  DELIA ===============================
;
;		
;   DELIA (Dispersive Exafs caLIbration and Analysis)  is a widget based 
;   graphical interface to perform the interpolation, calibration, 
;   extraction and Fourier transform of Dispersive Exafs spectra.
;
;   For the extraction and Fourier transform, it calls XAID_MU2CHI and 
;   XAID_FF, respectively, to perform the operations in a single spectrum. 
;   Once the parameters have been set from the analysis of that spectrum, 
;   the operations are replicated over the totallity of spectra. 
;
;		
; ============= Non interactive use of DELIA  ============================
;
;	NAME:
;		DELIA
;	PURPOSE:
;		 Dispersive Exafs caLIbration and Analysis
; 	CATEGORY:
;		Widgets. XAFS data analysis.
;	CALLING SEQUENCE:
;		DELIA 
;	KEYWORD PARAMETERS:
;		GROUP = The widget ID of the widget that calls Xplot.
;		When this ID is specified, a death of the caller results 
;		in a death of DELIA.
;	OUTPUTS:
;		Open the widget utility.
;	COMMON BLOCKS:
;		None
;	SIDE EFFECTS:
;		If not active, starts Xmanager
;	RESTRICTIONS:
;		Unknown.
;	PROCEDURE:
;		Four steps: 1) interpolation, 2) calibration, 3) extraction 
;		and 4) Fourier transform. 
;	MODIFICATION HISTORY:
;		Written by  Manuel Sanchez del Rio. ESRF. 2 October 1998
;		98-10-13 srio@esrf.fr adds e0 alignment, checks for
;			duplicated points, improves xaid_calib and makes
;			other improvements. version 1.0Beta2
;		98-10-13 srio@esrf.fr adds "sort" kw to gete0, displays
;			shifted values in a separate window. 
;			Version 1.0Beta3
;		98-10-26 srio@esrf.fr problems with interpolation/align
;			fixed. Added Back Fourier Transform. Cosmetics.
;			Added tools menus. Version 1.0Beta4
;		98-11-02 srio@esrf.fr changes the alignment method:
;			now it uses GetEdge. Version 1.0Beta5.
;		98-11-20 srio@esrf.fr Adds FIle/load file with Chi/Mu
;			Version 1.0Beta6.
;		99-10-03 srio@esrf.fr adds binary file RAW data input.
;			Version 1.0Beta7.
;		00-07-20 srio@esrf.fr last touches for xop 2.0
;			Version 1.0
;		06-03-03 srio@esrf.fr Improtant improvements. v 2.0Beta1
;		06-07-18 srio@esrf.fr Updates for CCD files. v 2.0Beta2
;
;-
;
;========================================================================
;
FUNCTION Delia_Make_Text_Array,a,group=Group,NLimit=nlimit, Header=header 
Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/w) THEN itmp = Dialog_Message(/Error,Dialog_Parent=group,$
	'DELIA_MAKE_TEXT_ARRAY: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN,'<error converting numerical array to text>'
ENDIF

IF Keyword_Set(nlimit) EQ 0 THEN nlimit = 300
IF Keyword_Set(header) EQ 0 THEN header = ''

npoints = (Size(a))[1:2]
ntxt = (npoints[1] < 2*nlimit)
txt = strarr(1+ ntxt )
txt[0]=header
IF npoints[1] LE 2*nlimit THEN BEGIN
  FOR i=1L,ntxt do txt[i]=string(a[*,i-1],format='('+$
    strcompress(npoints[0],/rem)+'(G11.5,X))')
ENDIF ELSE BEGIN
  FOR i=1L,nlimit-2 do txt[i]=string(a[*,i-1],format='('+$
    strcompress(npoints[0],/rem)+'(G11.5,X))')
  txt[nlimit-1]='...'
  txt[nlimit]='...<skipped>...'
  j=nlimit
  FOR i=npoints[1]-nlimit+1,npoints[1] DO BEGIN
    j = j+1
    txt[j]=string(a[*,i-1],format='('+$
    strcompress(npoints[0],/rem)+'(G11.5,X))')
  ENDFOR
ENDELSE
IF Keyword_Set(header) EQ 0 THEN txt = txt[1:N_Elements(txt)-1]
RETURN,txt
END ; Delia_Make_Text_Array

;
;========================================================================
;
FUNCTION Delia_Read_Binary,file, Group =group

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/w) THEN itmp = Dialog_Message(/Error,$
	'DELIA_READ_BINARY: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN,0
ENDIF

IF N_Params() EQ 0 THEN BEGIN
  pwd = 0
  CD,Current=pwd
  file = Dialog_PickFile(/Read,Dialog_Parent=group,Path=pwd)
  IF file EQ '' THEN RETURN,0
ENDIF

; file = 'test_200.out'
OpenR,unit,file,/Get_lun,/Swap_If_Little_Endian
fsize = (FStat(unit)).size
npoints=fsize/3/4

a=FltArr(3,npoints)
ReadU,unit,a
Free_Lun,unit
RETURN,a
END ; Delia_Read_Binary


;
;========================================================================
;

Function delia_reform_arrays,x,block,index,pp1,pp2,pp3,pp4,text=txt,$
 xplot=xplot, exodus=exodus, xrange=xrange, $
 _Extra=extra,No_block=no_block,NLimit=nlimit,coltitles=coltitles

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'DELIA_REFORM_ARRAYS: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   return,0
endif

;+
;
; Function delia_reform_arrays,x,block,index,pp1,pp2,pp3,pp4,text=txt,$
;  xplot=xplot, exodus=exodus, xrange=xrange, $
;
; x : array of npts
; block = array of (nn,npts)
; index = array of nn [index]
; pp1 ... pp4  = array of npts
; 
; reform arrays
;
;message,/info,'... Rearranging arrays ...'
;
;-

IF Keyword_Set(xrange) THEN BEGIN
  igood = where(x GE xrange[0] AND x LE xrange[1])
  IF igood[0] NE -1 THEN BEGIN
    x=x[igood]
    block_old=block
    block=FltArr(N_Elements(block_old[*,0]),N_Elements(x))
    FOR i=0,N_Elements(block_old[*,0])-1 DO $
		block[i,*]=Reform(block_old[i,igood])
    ;IF N_Elements(index) GT 0 THEN index=index[igood]
    IF N_Elements(pp1) GT 0 THEN pp1=pp1[igood]
    IF N_Elements(pp2) GT 0 THEN pp2=pp2[igood]
    IF N_Elements(pp3) GT 0 THEN pp3=pp3[igood]
    IF N_Elements(pp4) GT 0 THEN pp4=pp4[igood]
  ENDIF
ENDIF

npts = n_elements(x)
nn=n_elements(block[*,0])
npars = n_params()
if npars LT 2 then message,'Usage: out=delia_reform_arrays(x,block)'
if n_elements(index) EQ 0 then begin
  index=findgen(nn)
  npars = npars+1
endif
;if n_elements(pp2) EQ 0 then pp2=fltarr(npts)

out2 = fltarr(npars>3,nn*npts)
for i=0L,nn-1 do begin
  out2[0,i*npts:(i*npts)+npts-1]=reform(x)
  out2[1,i*npts:(i*npts)+npts-1]=index[i]
  if n_elements(pp1) GT 0 then out2[2,i*npts:(i*npts)+npts-1]=reform(pp1)
  if n_elements(pp2) GT 0 then out2[3,i*npts:(i*npts)+npts-1]=reform(pp2)
  if n_elements(pp3) GT 0 then out2[4,i*npts:(i*npts)+npts-1]=reform(pp3)
  if n_elements(pp4) GT 0 then out2[5,i*npts:(i*npts)+npts-1]=reform(pp4)
  out2[npars-1,i*npts:(i*npts)+npts-1]=block[i,*]
endfor
txt = Delia_Make_Text_Array(out2,NLimit=nlimit)
message,/info,'... Done ...'
; 
; display
;
;widget_control,state.wids.mu.text,set_value=txt
if keyword_set(xplot) then begin
  xplot,out2,xcol=1,no_block=no_block,p=p,_extra=extra,coltitles=coltitles
  xplot_mesh,p,flag=1,col=2,interactive=0
  xplot_refresh,p
endif
if keyword_set(exodus) then begin
   out3=out2[ [0,3,1],*]
   exodus,/xafs,/only,parent=p,/Quit_Button
   exodus_loadfile,p,isel=10,data=out3
endif


return,out2
end ; delia_reform_arrays
;
;========================================================================
;
PRO delia_mappanels, wids
;
; interpolation
Widget_Control,wids.mu.interp,Get_Value=interp

CASE interp OF
  '0': Widget_Control,wids.mu.base_interp2,Map=0
  '1': Widget_Control,wids.mu.base_interp2,Map=1
  else:
ENDCASE

end ; delia_mappanels


;
;========================================================================
;
PRO delia_load_raw_seq,wid,columnIndices=columnIndices

	Widget_Control,wid,Get_UValue=state
	Widget_Control,state.wids.raw.type,Get_Value=itype
	Widget_Control,state.wids.raw.file,Get_Value=file

	IF file EQ '' THEN RETURN
	if checkfile(file) NE 1 then begin
	  itmp = Dialog_Message(Dialog_Parent=wId,/Error,$
	     'File not found: '+file)
	  RETURN
	endif
	widget_control,/hourglass
	widget_control,state.wids.raw.text,set_value='<none>' 
	widget_control,state.wids.mu.text,set_value='<none>'
	widget_control,state.wids.labelMain,set_value=$
	 'Loading file... Please wait.'
	npoints=0
	CASE itype OF 
	'0': BEGIN ; ascii
	   	a=rascii(file,npoints=npoints)
	   END
	'1': BEGIN ; binary
		a=Delia_Read_Binary(file,Group=wId)
		npoints = (Size(a))[1:2]
	   END
	else: 
	ENDCASE

        IF Keyword_Set(columnIndices) THEN BEGIN
	  col1=columnIndices[0]+1
          col2=columnIndices[1]+1
          col3=columnIndices[2]+1
        ENDIF ELSE BEGIN
	  ;
	  ; guess columns
	  ;
	  col1=1  & col2=1 & col3=1
	  FOR icol=0,2 DO BEGIN
	  IF (a[icol,0] EQ 0) AND (a[icol,1] EQ 1) AND (a[icol,2] EQ 2) THEN BEGIN
		  col1=icol+1
	  ENDIF ELSE BEGIN
	    IF (a[icol,0] EQ  a[icol,1]) AND (a[icol,0] EQ  a[icol,2]) THEN $
		  col3=icol+1 ELSE $
		  col2=icol+1
	  ENDELSE
	  ENDFOR
        ENDELSE
	widget_control,state.wids.raw.col1,set_value=col1
	widget_control,state.wids.raw.col2,set_value=col2
	widget_control,state.wids.raw.col3,set_value=col3
	tmp = Reform(a[col3-1,*])
	nspectra=N_Elements( tmp[UNIQ(tmp, SORT(tmp))] )

	; 
	; update info
	;
	widget_control,state.wids.labelMain,set_value=$
	 'File contains '+strcompress(npoints[1],/rem)+' lines '+$
	 'and '+strcompress(npoints[0],/rem)+' columns. Converting to text...'
	widget_control,/hourglass
	header=''
	for i=1L,npoints[0] do header=header+'   Column '+$
	  strcompress(i,/rem)+' '

	txt = Delia_Make_Text_Array(a,Header=header,Group=wId,NLimit=300)
	widget_control,state.wids.raw.text,set_value=txt
	; 
	; store data
	;
	for i=0,n_elements(state.ptr)-1 do $
	  if Ptr_Valid(state.ptr[i]) then Ptr_Free,state.ptr[i]
help,a
	state.ptr[0]=Ptr_New(a)
	state.raw_type='Sequential'

	widget_control,state.wids.labelMain,set_value=$
	 strcompress(nspectra,/Rem)+' spectra, '+$
	 strcompress(npoints[1],/rem)+' lines, '+$
	 strcompress(npoints[0],/rem)+' columns in '+file


	Widget_Control,wid,Set_UValue=state
END ; delia_load_raw_seq 


;
;========================================================================
;
PRO delia_load_raw_ccd,wid


	Widget_Control,wid,Get_UValue=state

	Widget_Control,state.wids.raw.ccdfile,Get_Value=ccdfile
	IF ccdfile EQ '' THEN RETURN
	if checkfile(ccdfile) NE 1 then begin
	  itmp = Dialog_Message(Dialog_Parent=wid,/Error,$
	     'File not found: '+ccdfile)
	  RETURN
	endif

	Widget_Control,state.wids.raw.ccdtype,Get_Value=itype

	widget_control,state.wids.labelMain,set_value=$
	 'Loading file... Please wait.'
	Widget_Control,/HourGlass
	npoints=0
	CASE itype OF 
	'0': BEGIN ; ascii
	     a=rascii(ccdfile,npoints=npoints,Buffer=2048)
	     as = Size(a)
	     nx = as[1]
	     ny = as[2]
             nspectra=nx-1
	     npoints = ny
	     pixels = a[0,*]
	   END
	'1': BEGIN ; EDF
	     a0=read_edf(ccdfile)
	     as = Size(a0)
	     nx = as[1]
	     ny = as[2]
             nspectra=nx
	     npoints = ny
	     pixels = FIndGen(ny)
             a = FltArr(nx+1,ny)
             a[1:nx,*]=a0
             a[0,*]=pixels
	   END
	else: 
	ENDCASE

	widget_control,state.wids.labelMain,set_value= $
	StrCompress(nspectra,/Rem)+' spectra of '+StrCompress(npoints,/Rem)+$
	   ' points read from file: '+ccdfile
	widget_control,/hourglass
	;
	; store data
	;
	for i=0,n_elements(state.ptr)-1 do $
	  if Ptr_Valid(state.ptr[i]) then Ptr_Free,state.ptr[i]
	state.ptr[0]=Ptr_New(a)
	state.raw_type='CCD'
	; 
	; display
	; 
	Widget_Control,state.wids.raw.draw,Get_Value=drawIndex
	WSet, drawIndex
	TvScl,Transpose( ConGrid(a[1:nx-1,*],500,500) )

        Widget_Control,wid,Set_UValue=state

END ; delia_load_raw_ccd
;
;========================================================================
;
PRO delia_event,event

Forward_Function xaid_mu2chi_getstr

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'DELIA: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   goto,out
endif

;
;

Widget_Control, event.id, get_UValue=eventUValue


CASE eventUValue OF
  'TAB': Value=''
  'TAB_RAW': Value='TAB_RAW'
  'SWITCH':Value=''
  else: Widget_Control, event.id, get_Value=Value
ENDCASE
 

if n_elements(eventuvalue) EQ 0 then eventuvalue = ''
if not(keyword_set(eventuvalue)) then eventuvalue = ''
 
stateid = Widget_Info(event.handler,/Child)


;
; eventUValue that do not need state.
;
Case eventUValue OF
  'LOAD_RAW_SEQ_EXAMPLE': BEGIN
        Widget_Control, stateid, get_UValue=state 
	Widget_Control,state.wids.raw.type,Get_Value=itype

        ds = sdep(/ds)
        ds=sDep(/ds)
        path = GetEnv('XOP_HOME')+ds+'extensions'+ds+'xaid'+$
          ds+'Examples'+ds
	CASE itype OF 
	'0': BEGIN ; ascii
                defaultFile = path+'f29'
	   END
	'1': BEGIN ; binary
                defaultFile = path+'f29.bin'
	   END
	else: 
	ENDCASE
	Widget_Control,state.wids.raw.file,Set_Value=defaultFile
	delia_load_raw_seq,stateID,columnIndices=[0,1,5]
	RETURN

	END

  'LOAD_RAW_SEQ': BEGIN
	delia_load_raw_seq,stateID
	RETURN
	END

  'LOAD_RAW_CCD': BEGIN
	delia_load_raw_ccd,stateID
	RETURN
	END

  'LOAD_RAW_CCD_EXAMPLE': BEGIN

	Widget_Control,stateId,Get_UValue=state
	Widget_Control,state.wids.raw.ccdtype,Get_Value=itype

        ds = sdep(/ds)
        ds=sDep(/ds)
        path = GetEnv('XOP_HOME')+ds+'extensions'+ds+'xaid'+$
          ds+'Examples'+ds
	CASE itype OF 
	'0': BEGIN ; ascii
                defaultFile = path+'Rh_23_log_0'
	   END
	'1': BEGIN ; binary
                defaultFile = path+'Rh_23_log_0.edf'
	   END
	else: 
	ENDCASE

	IF defaultfile EQ '' THEN RETURN
	if checkfile(defaultfile) NE 1 then begin
	  itmp = Dialog_Message(Dialog_Parent=stateId,/Error,$
	     'File not found: '+defaultfile)
	  RETURN
	endif
	Widget_Control,state.wids.raw.ccdfile,Set_Value=defaultFile
	Widget_Control,stateId,Set_UValue=state,/Hourglass
	delia_load_raw_ccd,stateID
	RETURN
	END
  ELSE:
ENDCASE


Widget_Control, stateid, get_UValue=state ; , /No_Copy

;delia_mappanels,state.wids
 
Case eventUValue OF
  ;
  ; Menu Bar
  ;
  'QUIT': BEGIN
	for i=0,n_elements(state.ptr)-1 do $
	  if Ptr_Valid(state.ptr[i]) then Ptr_Free,state.ptr[i]
	Widget_Control,event.handler,/destroy
	END
  ;
  ; RAW PANEL
  ;
  'RAW_VIEW#': BEGIN   
	; only for sequential files
	IF state.raw_type EQ 'CCD' THEN GoTo,out

	if Ptr_Valid(state.ptr[0]) then a=*(state.ptr[0]) else begin
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	  'Data not found. Please load a data file.')
	  goto,out
	endelse
	; 
	; Get indices
	;
	indexcol=0
	widget_control,state.wids.raw.col3,get_value=indexcol

	indexCol = (indexCol-1) > 0
	allIndices = Reform(a[indexCol,*])

	;
	; get selected_index
	;
	Widget_Control,state.wids.raw.view,Get_Value=selected_index
	Widget_Control,event.id,Get_Value=val
	reset=0
	m1=Min(allIndices,Max=m2)
  
	CASE StrCompress(val,/Rem) OF
	    '|<': selected_index = m1
	    '<':  selected_index=selected_index-1 
	    '>':  selected_index=selected_index+1
	    '>|': selected_index = m2
	    'Reset':  reset=1
	    else: 
	ENDCASE
	Widget_Control,state.wids.raw.view,Set_Value=selected_index
	IF selected_index LT m1 THEN BEGIN
	   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	    'Wrong index value. Reset to: '+StrCompress(m1))
	   Widget_Control,state.wids.raw.view,Set_Value=m1
	   goto,out
	ENDIF
	IF selected_index GT m2 THEN BEGIN
	   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	    'Wrong index value. Reset to: '+StrCompress(m2))
	   Widget_Control,state.wids.raw.view,Set_Value=m2
	   goto,out
	ENDIF
	; 
	; Get data
	;

	xcol=0 & ycol=0 
	widget_control,state.wids.raw.col1,get_value=xcol
	widget_control,state.wids.raw.col2,get_value=ycol
	xcol = (xcol-1) > 0
	ycol = (ycol-1) > 0
	;indexCol = (indexCol-1) > 0

	indices = where( a[indexCol,*] EQ selected_index)

	if indices[0] EQ -1 then begin
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	  'Wrong index value.')
	  goto,out
	endif else begin
	  arrx = Reform(a[xcol,indices] )
	  arri = Reform(a[indexcol,indices] )
	  arry = Reform(a[ycol,indices] )
	  arr = Make_Set(arrx,arri,arry)
	endelse
        ;
        ; Plot
        ;
	if widget_info(state.wids.xplot,/Valid_Id) then begin
	    if reset EQ 0 then begin
	        xplot_controls_action,state.wids.xplot,clr_lines=255
		xplot_savecurrent,state.wids.xplot 
	    endif else begin
		xplot_plotmgr,state.wids.xplot,/delete_all
	    endelse
	    xplot_loadfile,state.wids.xplot, arr
	    xplot_controls_action,state.wids.xplot, clr_lines=3
	endif else begin
	    xplot,p=p,arr,group=event.top,No_Block=state.no_block
	    xplot_controls_action,p, clr_lines=3
	    state.wids.xplot=p
	endelse

	END

  'RAW_CURSOR': BEGIN
       if Ptr_Valid(state.ptr[0]) then a=*(state.ptr[0]) else GoTo,out
       pixels = Reform(a[0,*])
       ; remove first column
       as = Size(a)
       nx = as[1]
       ny = as[2]
       ax = Reform(a[0,*])
       a = a[1:nx-1,*]
       nx=nx-1

       widget_control,state.wids.raw.draw,get_value=windownum
       wset,windownum
       coords_d = [event.x,event.y]
       coordPixel = Long( Float(coords_d[0])/499*ny ) 
       coordX = ax[0] + Float(coords_d[0])/499 * ax[ny-1]
       coordSpectrum = Long( Float(coords_d[1])/499*nx ) 

       widget_control,state.wids.raw.cursor,SET_VALUE= $
	'Pixel:'+strcompress(coordPixel,/REM)+ $
	' X: '+strcompress(coordX,/REM)+ $
	' Spectrum:'+strcompress(coordSpectrum,/REM)+ $
	' Value:'+strcompress(a[coordSpectrum<(nx-1),coordPixel<(ny-1)],/REM)

       IF event.press EQ 1 THEN BEGIN

	;IF event.clicks EQ 1 THEN reset=0 ELSE reset=1
	reset=0

	arr = Make_Set(pixels,Reform(a[coordSpectrum<(nx-1),*]) )
	IF event.clicks EQ 1 THEN BEGIN ; xplot
	  if widget_info(state.wids.xplotCCD,/Valid_Id) then begin
	      if reset EQ 0 then begin
	          xplot_controls_action,state.wids.xplotCCD,clr_lines=255
		  xplot_savecurrent,state.wids.xplotCCD
	      endif else begin
		  xplot_plotmgr,state.wids.xplotCCD,/delete_all
	      endelse
	      xplot_loadfile,state.wids.xplotCCD, arr
	      xplot_controls_action,state.wids.xplotCCD, clr_lines=3
	  endif else begin
	      xplot,p=p,arr,group=event.top,No_Block=state.no_block
	      xplot_controls_action,p, clr_lines=3
	      state.wids.xplotCCD=p
	  endelse
	ENDIF ELSE IF event.clicks EQ 2 THEN BEGIN ; Exodus
          ;
          ; with exodus
          ;
	  if widget_info(state.wids.exodusCCD,/Valid_Id) then begin
              exodus_loadfile,state.wids.exodusCCD,isel=0,data=arr,title='Spectrum:'+strcompress(coordSpectrum,/REM)
	  endif else begin
              exodus,/xafs,/only,parent=p
	      state.wids.exodusCCD=p
              exodus_loadfile,p,isel=0,data=arr,title='Spectrum:'+strcompress(coordSpectrum,/REM)
	  endelse
	ENDIF
       ENDIF


	END
  'RAW_DISPLAY': BEGIN
	if Ptr_Valid(state.ptr[0]) then a=*(state.ptr[0]) else begin
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	  'Data not found. Please load a data file.')
	  goto,out
	endelse
	Widget_Control,event.id,Get_Value=val
	case val of
	  'iContour': BEGIN
	  	sa = size(a)
		icontour,a[1:sa[1]-1,*]
		END
	  'iSurface': BEGIN
	  	sa = size(a)
		isurface,a[1:sa[1]-1,*]
		END
	  'iImage': BEGIN
	  	sa = size(a)
		iImage,a[1:sa[1]-1,*]
		END
	  'Xsurface1': BEGIN
	  	sa = size(a)
		Xsurface1,a[1:sa[1]-1,*]
		END
	  'Export EDF': BEGIN
back2:
                file = Dialog_Pickfile(Dialog_Parent=event.top,$
                   file='delia.edf',Filter='*.edf', $
                   title='Export to EDF file')
                IF file EQ '' THEN RETURN
		IF CheckFile(file) THEN BEGIN
		 itmp = Dialog_Message(/Info,/Cancel,Dialog_Parent=event.top, $
		   ['File exists: '+file,'Overwrite it?'])
		  IF itmp EQ 'Cancel' THEN GoTo,out
		  IF itmp EQ 'No' THEN GoTo,back2
		ENDIF
	  	sa = size(a)
		a2=a[1:sa[1]-1,*]
		a2=a2-min(a2)
		;a2=Transpose(a2)
                write_edf,a2,file=file,Title= $
                  'Data exported to EDF by XOP/XAID/DELIA'
		END
	  'Xplot2d': BEGIN
	  	sa = size(a)
		a2=a[1:sa[1]-1,*]
		a2=Transpose(a2-min(a2))
		Xplot2D,a2
		END
	  'Refresh': BEGIN
		; 
		; display
		; 
		Widget_Control,state.wids.raw.draw,Get_Value=drawIndex
		WSet, drawIndex
		as = Size(a)
		nx = as[1]
		ny = as[2]
		TvScl,Transpose( ConGrid(a[1:nx-1,*],500,500) )
		END
	 else: print,'Case not found: '+val
	endcase
	END


  'RAW_ROTATE': BEGIN
	if Ptr_Valid(state.ptr[0]) then a1=*(state.ptr[0]) else begin
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	  'Data not found. Please load a data file.')
	  goto,out
	endelse
	Widget_Control,event.id,Get_Value=val
	Widget_Control,event.id,Set_Value=0

	case Fix(val) of
         1:a=rotate(a1,1) 
         2:a=rotate(a1,3) 
         3:a=rotate(a1,2) 
         4:a=rotate(a1,5) 
         5:a=rotate(a1,7) 
	 else: print,'Case not found: '+val
	endcase
        *(state.ptr[0])=a
	; 
	; display
	; 
	Widget_Control,state.wids.raw.draw,Get_Value=drawIndex
	WSet, drawIndex
	as = Size(a)
	nx = as[1]
	ny = as[2]
	TvScl,Transpose( ConGrid(a[1:nx-1,*],500,500) )
	END


  'RAW_SEQUENTIAL_EXODUS': BEGIN
	if Ptr_Valid(state.ptr[0]) then a=*(state.ptr[0]) else begin
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	  'Data not found. Please load a data file.')
	  goto,out
	endelse
	Widget_Control,state.wids.raw.col1,Get_Value=col1
	Widget_Control,state.wids.raw.col2,Get_Value=col2
	Widget_Control,state.wids.raw.col3,Get_Value=col3
	exodus,/xafs,/only,parent=p
	exodus_loadfile,p,isel=10,data=a[ [col1,col2,col3]-1,* ]
	END
  'RAW_SEQUENTIAL_MESH': BEGIN
	if Ptr_Valid(state.ptr[0]) then a=*(state.ptr[0]) else begin
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	  'Data not found. Please load a data file.')
	  goto,out
	endelse
	  widget_control,state.wids.raw.col1,get_value=col1
	  widget_control,state.wids.raw.col2,get_value=col2
	  widget_control,state.wids.raw.col3,get_value=col3
	  ncol = N_Elements(a[*,0])
	  coltitles='col'+StrCompress(SIndGen(ncol)+1)
	  coltitles[col1-1]='X'
	  coltitles[col2-1]='Y'
	  coltitles[col3-1]='indices'
	  xplot,a,xcol=col1,ycol=col2,/no_block,p=p,wtitle=$
	  'Raw data (3D)',coltitles= coltitles
	  xplot_mesh,p,flag=1,col=col3,interactive=0
	  xplot_refresh,p
	END

  'RAW_SEQUENTIAL_BIN': BEGIN
	if Ptr_Valid(state.ptr[0]) then a=*(state.ptr[0]) else begin
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	  'Data not found. Please load a data file.')
	  goto,out
	endelse
back3:
        file = Dialog_Pickfile(Dialog_Parent=event.top,$
                   file='delia.bin',Filter='*.bin', $
                   title='Export to EDF file')
        IF file EQ '' THEN RETURN
	IF CheckFile(file) THEN BEGIN
		 itmp = Dialog_Message(/Info,/Cancel,Dialog_Parent=event.top, $
		   ['File exists: '+file,'Overwrite it?'])
		  IF itmp EQ 'Cancel' THEN GoTo,out
		  IF itmp EQ 'No' THEN GoTo,back3
	ENDIF
	widget_control,state.wids.raw.col1,get_value=col1
	widget_control,state.wids.raw.col2,get_value=col2
	widget_control,state.wids.raw.col3,get_value=col3
	ncol = N_Elements(a[*,0])
        a2=a[[col1,col2,col3]-1,*]
	;coltitles='col'+StrCompress(SIndGen(ncol)+1)
	;coltitles[col1-1]='X'
	;coltitles[col2-1]='Y'
	;coltitles[col3-1]='indices'
	;xplot,a,xcol=col1,ycol=col2,/no_block,p=p,wtitle=$
	;'Raw data (3D)',coltitles= coltitles
	;xplot_mesh,p,flag=1,col=col3,interactive=0
	;xplot_refresh,p


	 OpenW,unit,file,/Get_Lun,/Swap_If_Little_Endian

	 WriteU,unit,a2
	 ;nn = N_Elements(a2[0,*])
	 ;FOR i=0L,nn-1 DO BEGIN
	 ;    set = (state.ompp)->value(iList[i])
	 ;    set1=FltArr(3,N_Elements(set[0,*]))
	 ;    set1[1:2,*]=set
	 ;    ii = Float(i)
	 ;    set1[0,*]=ii
	 ;    WriteU,unit,a1
	 ;ENDFOR
	 Free_Lun,unit
         Message,/Info,'Data exported to DELIA BIN file: '+file

	END

  ;
  ; MU PANEL
  ;
  'SWITCH': BEGIN
	switchindex=event.tab

	if switchindex EQ 1 then begin
	  if Ptr_Valid(state.ptr[1]) then begin
            ;delia_mappanels,state.wids
	    widget_control,state.wids.labelMain,Set_Value= $
	     'Use 2D or 3D buttons to visualize data and chi(k) panel to process.'
	    GoTo,out
	  endif
	  if not(Ptr_Valid(state.ptr[0])) then begin
	    itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	    'Data not found. Please load a data file.')
	    ;delia_mappanels,state.wids
	    goto,out
	  endif
	  a=*(state.ptr[0])
	  xcol=0 & ycol=0 & indexcol=0
	  widget_control,state.wids.raw.col1,get_value=xcol
	  widget_control,state.wids.raw.col2,get_value=ycol
	  widget_control,state.wids.raw.col3,get_value=indexcol
	  xcol = (xcol-1) > 0
	  ycol = (ycol-1) > 0
	  indexCol = (indexCol-1) > 0
	  ;
	  ; get the index values
	  ;
          tmp = reform(a[indexCol,*])
	  arrIndex = tmp[Uniq( tmp, Sort(tmp) )]
	  n_arrIndex = n_elements(arrIndex)
	  ;
	  ; get the single extrema of spectra
	  ; 
	  widget_control,state.wids.labelMain,Set_Value= 'Computing extrema...'
	  for i=0L,n_arrIndex-1 do begin
	    tmp = a[ *, where(a[indexCol,*] EQ arrIndex[i]) ]
	    if i EQ 0 then begin
	       x2=0
	       x1 = min(tmp[xcol,*],max=x2) 
	    endif else begin
	      tmpmax=0
	      tmpmin = min(tmp[xcol,*],max=tmpmax)
	      x1 = [x1,tmpmin]
	      x2 = [x2,tmpmax]
	    endelse
	  endfor
	  x1max=0
	  x1min=min(x1,max=x1max)
	  x2max=0
	  x2min=min(x2,max=x2max)
	  ; 
	  ; update widgets
	  ;
	  widget_control,state.wids.mu.labelInt1,Set_Value='from (minima) '+$
	     vect2string( [x1min,x1max] )
	  widget_control,state.wids.mu.labelInt2,Set_Value='to (maxima) '+$
	     vect2string( [x2min,x2max] )
	  widget_control,state.wids.mu.intMin,Set_Value=max([x1min,x1max])
	  widget_control,state.wids.mu.intMax,Set_Value=min([x2min,x2max])
	  widget_control,state.wids.labelMain,Set_Value= $
		'Please check/change parameters and press "RUN" button'
	endif


	if switchindex EQ 2 then begin ; chi panel
	  if Ptr_Valid(state.ptr[2]) then begin
            ;delia_mappanels,state.wids
	    widget_control,state.wids.labelMain,Set_Value= $
	     'Use 2D or 3D buttons to visualize data and Fourier panel to process.'
	    GoTo,out
	  endif
	  widget_control,state.wids.labelMain,Set_Value=$
	   'Extraction on a single spectrum and then replication'
	  if not(Ptr_Valid(state.ptr[1]) ) then begin
	    itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	    'Mu Data not found. Please process data uning "mu" panel.')
;	    widget_control,state.wids.panels,set_value=0
	    ;delia_mappanels,state.wids
	    goto,out
	  endif
	endif

	if switchindex EQ 3 then begin ; ft panel
	  widget_control,state.wids.labelMain,Set_Value=$
	   'FT on a single spectrum and then replication'
	  if not(Ptr_Valid(state.ptr[2]) ) then begin
	    itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	    'Chi Data not found. Please process data uning "chi" panel.')
	    goto,out
	  endif
	  Widget_Control,state.wids.chi.kkmin,Get_Value=kkmin
	  Widget_Control,state.wids.chi.kkmax,Get_Value=kkmax
	  Widget_Control,state.wids.ft.kmin,Set_Value=kkmin
	  Widget_Control,state.wids.ft.kmax,Set_Value=kkmax
	  Widget_Control,state.wids.ft.k2min,Set_Value=kkmin
	  Widget_Control,state.wids.ft.k2max,Set_Value=kkmax
	endif


	END
   'MU_GO': BEGIN
	if Ptr_Valid(state.ptr[0]) then a=*(state.ptr[0]) else begin
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	  'Data not found. Please load a data file.')
	  goto,out
	endelse

	widget_control,state.wids.mu.text,set_value='<none>'

	; 
	; Rearrange data 
	;
	IF state.raw_type EQ 'Sequential' THEN BEGIN
	  ;
	  ; get inputs from interface
	  ;
	  xcol=0 & yxol=0 & indexcol=0
	  widget_control,state.wids.raw.col1,get_value=xcol
	  widget_control,state.wids.raw.col2,get_value=ycol
	  widget_control,state.wids.raw.col3,get_value=indexcol
  
	  ;
	  ; column indices
	  ;
	  xcol = (xcol-1) > 0
	  ycol = (ycol-1) > 0
	  indexCol = (indexCol-1) > 0
	  ;
	  ; get the good columns (tmpx,tmpy and tmpindex)
	  ;
	  tmpindex = a[indexcol,*]
	  tmpx = a[xcol,*]
	  tmpy = a[ycol,*]
  
	  ;
	  ; make an array with the indices
	  ;
          tmp = reform(a[indexCol,*])
	  arrIndex = tmp[Uniq( tmp, Sort(tmp) )]
	  n_arrIndex = n_elements(arrIndex)

	ENDIF ELSE BEGIN ; CCD
	  sa = size(a)
	  nspectra=sa[1]-1
	  npoints=sa[2]

	  arrx = Reform(a[0,*])
	  tmpx = FltArr(npoints,nspectra)
	  FOR i=0L,nspectra-1 DO tmpx[*,i]=arrx
	  tmpx = Reform(tmpx,nspectra*npoints)
          
	  tmpy = FltArr(npoints,nspectra)
	  FOR i=0L,nspectra-1 DO tmpy[*,i]=a[i+1,*]
	  tmpy = Reform(tmpy,nspectra*npoints)
	
	  tmpIndex = FltArr(npoints,nspectra)
	  FOR i=0L,nspectra-1 DO tmpIndex[*,i]=i
	  tmpIndex = Reform(tmpIndex,nspectra*npoints)
	  ;
	  ; make an array with the indices
	  ;
          tmp = reform(tmpIndex)
	  arrIndex = tmp[Uniq( tmp, Sort(tmp) )]
	  n_arrIndex = n_elements(arrIndex)
	  ; arrays for further use
	  tmp3x = arrx
	  out = a[1:nspectra,*] 
	ENDELSE
	; 
	; interpolation
	; 
	Widget_Control,state.wids.mu.interp,Get_value=iinterpolate
	  iinterpolate=Fix(iinterpolate)
	IF iinterpolate EQ 0 THEN BEGIN ; do not interpolate
	  IF state.raw_type EQ 'Sequential' THEN BEGIN
	    ; check if arrays are similar (first and last)
	    igood1=where( tmpindex EQ arrIndex[0])
	    igood2=where( tmpindex EQ arrIndex[n_arrIndex-1])
	    itmp = Total(abs(tmpx[igood1]-tmpx[igood2]))
	    IF itmp GT 1e-4 THEN BEGIN
	       itmp = Dialog_Message(/Question,Dialog_Parent=event.top, $
		  ['Warning: It seems that the spectra abscissas are',$
		   'different from one spectrum to another. ','',$
		   'You may need interpolation','Abort operation?'])
	       IF itmp EQ 'Yes' THEN GoTo,out
	    ENDIF
	    out = fltarr(n_elements(arrIndex),N_Elements(igood1) )
	    FOR i=0,n_arrIndex-1 do begin
	      igood=where(tmpindex EQ arrIndex[i])
	      tmp3x=tmpx[igood] 
	      out[i,*]=Reform(tmpy[igood])
	    ENDFOR
	    ;
	  ENDIF
	ENDIF ELSE BEGIN ; interpolate
	  ; 
	  ; interpolate 
	  ;
	  IF state.raw_type EQ 'Sequential' THEN BEGIN
	    xmin=0 & xmax=0 & npts=0
	    widget_control,state.wids.mu.intMin,get_value=xmin
	    widget_control,state.wids.mu.intMax,get_value=xmax
	    widget_control,state.wids.mu.intN,get_value=npts
	    out = fltarr(n_elements(arrIndex),npts)
  
	    tmp3x = xmin + ( findgen(npts)/float(npts-1) )*(xmax-xmin)
	    widget_control,state.wids.labelMain,Set_Value= $
		'Calculating interpolation...'
	    widget_control,/hourglass

	    for i=0,n_arrIndex-1 do begin
	      widget_control,state.wids.labelMain,Set_Value= $
		  'Calculating interpolation spectrum '+$
		  strcompress(i+1)+' of '+strcompress(n_arrIndex)+'...'
	      good=where(tmpindex EQ arrIndex[i])
	      tmp2x=tmpx[good] ;;;; - e0shifted[i]
	      tmp2y=tmpy[good]
	      ;
	      ; use only non-duplicated points
	      ;
	      good2 = Uniq(tmp2x, Sort(tmp2x))
	      tmp2x = tmp2x[good2] 
	      tmp2y = tmp2y[good2]
	      tmp3y = interpol(tmp2y,tmp2x,tmp3x)
	      out[i,*] = reform(tmp3y)
	    endfor
	  ENDIF ELSE BEGIN ; 'CDD'
	    itmp = Dialog_Message('Skipped interpolation (Not needed with CCD data)', $
		Dialog_Parent=event.top)
	  ENDELSE
	ENDELSE ; interpolation

	; 
	; calibrate the X array
	; 

	widget_control,state.wids.labelMain,Set_Value= $
		'Calculating calibration...'
	a=0 & b=0 & c=0 & x0=0
	widget_control,state.wids.mu.cal_a,get_value=a
	widget_control,state.wids.mu.cal_b,get_value=B
	widget_control,state.wids.mu.cal_c,get_value=c
	widget_control,state.wids.mu.cal_x0,get_value=x0
	if (a EQ 0 and b EQ 0 and c EQ 0) then begin
	  tmp3x_cal = tmp3x
	endif else begin
	  tmp = tmp3x-x0 ; tmp3x[0]
	  tmp3x_cal = tmp*tmp*c + tmp*b + a
	endelse

	widget_control,state.wids.labelMain,Set_Value=$
		'Preparing plot...'
	; 
	; fill text
	;
	txt = 1
	tmp = delia_reform_arrays(tmp3x_cal,out,arrIndex,tmp3x,$
	  xplot=0,group=event.top,text=txt,wtitle='Mu 3D plot',$
	  No_Block=state.no_block)
	txt = ['Xcalibrated      Index  Xoriginal         Mu', $
	  temporary(txt)]
	widget_control,state.wids.mu.text,set_value=txt
	widget_control,state.wids.labelMain,Set_Value='Done'
	;
	; save data in Ptr
	;
	str = {x:tmp3x, energy:tmp3x_cal, index:arrIndex, mu:out}
        for i=1,n_elements(state.ptr)-1 do $
          if Ptr_Valid(state.ptr[i]) then Ptr_Free,state.ptr[i]
	state.ptr[1]=Ptr_New(str)
	END

  'MU_LOAD': BEGIN
	CD,Current=pwd
	itmp = Dialog_Message(/Info,/Cancel,Dialog_Parent=event.top,$
	  ['This option loads a multicolumn ASCII file as ',$
	   'writen by XPLOT(2D), with: ',$
	   'Column 1: the E values',$
	   'Column 2: Mu for the 1st spectrum ',$
	   'Column 3: Mu for the 2nd spectrum ',$
	   '...','  ',$
	   '(Tip: if you want to load a sequential file with an index',$
	   ' column, load it in the Raw panel and switch to Mu panel.',$
	   ' Do not apply interpolation and use calibration parameters:',$
	   ' a=0, b=1, c=0, x0=0 )' ])
	IF itmp EQ 'Cancel' THEN GoTo,out
	file = Dialog_PickFile(Group=event.top, /Read, Dialog_Parent=event.top)
	IF file EQ '' THEN GoTo,out
	data = RAscii(file)
	dimData=Size(data)
	npoints=dimData[2]
	nspectra=dimData[1]-1

	e = Reform(data[0,*])
	mu = data[1:nspectra,*]
	idx = FindGen(nspectra)

	; 
	; fill text
	;
	txt = 1
	tmp = delia_reform_arrays(e,mu,idx,e,$
	  xplot=0,group=event.top,text=txt,wtitle='Mu 3D plot',$
	  No_Block=state.no_block)
	txt = ['     E           Index        E          Mu', $
	  temporary(txt)]
	widget_control,state.wids.mu.text,set_value=txt
	widget_control,state.wids.labelMain,Set_Value='Done'
	;
	; save data in Ptr
	;
	str = {x:e, energy:e, index:idx, mu:mu}
        for i=1,n_elements(state.ptr)-1 do $
          if Ptr_Valid(state.ptr[i]) then Ptr_Free,state.ptr[i]
	state.ptr[1]=Ptr_New(str)
	END




  'MU_GET': BEGIN
	deliaid = state.wids
	xplotid = state.wids.xplot
	xaid_calib,deliaid=deliaid,xplotid=xplotid,group=event.top,$
	  No_Block=state.no_block
	END
  'MU_PLOT': BEGIN
	if Ptr_Valid(state.ptr[1]) then a=*(state.ptr[1]) else begin
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	  [' Data containing Mu not found.',$
	   ' Please load a data file and/or ',$
	   ' RUN interpolate & calibrate.'])
	  goto,out
	endelse
	val=''
	widget_control,event.id,get_value=val,/HourGlass
	case val of
	  '2 D': BEGIN
		nnx = 1+n_elements(a.mu[*,0])
		nny = n_elements(a.mu[0,*])
		out2 = fltarr(nnx,nny)
		out2[0,*]=a.energy
		out2[1:nnx-1,0:nny-1]=a.mu
		xplot,out2,No_Block=state.no_block,$
		  coltitles=['Energy [eV]','Index: '+$
		  string(a.index)], group=event.top, Wtitle='Mu 2D plot'
		END
	  '3 D': BEGIN
		tmp = delia_reform_arrays(a.energy,a.mu,a.index,a.x,$
		  /xplot,group=event.top,wtitle='Mu 3D plot',No_Block=$
		  state.no_block,coltitles=['E','Index','Xorig','Mu'])
		END
	  'EXODUS...': BEGIN
		tmp = delia_reform_arrays(a.energy,a.mu,a.index,a.x,$
		  /exodus,group=event.top,No_Block=$
		  state.no_block)
		END
	  'iSurface': BEGIN
		iSurface,a.mu,a.index,a.energy,Xtitle='index',YTitle='E'
		END
	  'iContour': BEGIN
		iContour,a.mu,a.index,a.energy,Xtitle='index',YTitle='E'
		END
	  'iImage': BEGIN
		iImage,a.mu,a.index,a.energy,Xtitle='index',YTitle='E'
		END
	  'Xplot2d': BEGIN
                tmp = Transpose(a.mu-min(a.mu))
		Xplot2d,tmp ; ,Xtitle='index',YTitle='E',a.index,a.energy,Xtitle='index',YTitle='E'
		END
	  'Xsurface1': BEGIN
		xsurface1,a.mu,a.index,a.energy,Xtitle='index',YTitle='E'
		END
	  'XANES analysis with FuFiFa...': BEGIN
		fufifa,Parent=p
		nspectra=n_elements(a.mu[*,0])
		Titles= 'Index: '+string(a.index)
		FOR i=0L,nspectra-1 DO BEGIN
		  data = Make_Set(a.energy,Reform( (a.mu)[i,*] ))
                  itmp = i
		  fufifa_loadfile,p,isel=2,data=data,title=titles[i],$
			append=itmp
		ENDFOR
		END
	  'Compute Eo shifts': BEGIN
		itmp = DIalog_Message(/Info,/Cancel,Dialog_Parent=event.top, $
		   'This option computes Eo (Eo=Max(Deriv)) vs spetrum index')
		IF itmp EQ 'Cancel' THEN Return
		nspectra=n_elements(a.mu[*,0])
		out = FltArr(2,nspectra)
		out[0,*] = a.index
		FOR i=0L,nspectra-1 DO BEGIN
		  data = Make_Set(a.energy,Reform( (a.mu)[i,*] ))
		  e0 = gete0(data)
		  out[1,i]=e0
		ENDFOR
		xplot,out,coltitles=['Index','Eo'],/no_block, wTitle='Eo shifts'
		END
	  'Export calibrated data...': BEGIN
		CASE state.raw_type OF
		  'Sequential': BEGIN
			Widget_Control,state.wids.raw.file,Get_Value=file0
			IF file0 EQ '' THEN file0='delia'
			END
		  'CCD': BEGIN
			Widget_Control,state.wids.raw.ccdfile,Get_Value=file0
			IF file0 EQ '' THEN file0='delia'
			END
		ENDCASE
back1:
		file = Dialog_Pickfile(/Write,Dialog_Parent=event.top, $
			File=file0+'.mu')
		IF file EQ '' THEN GoTo,out
		IF CheckFile(file) THEN BEGIN
		  itmp = Dialog_Message(/Info,/Cancel,Dialog_Parent=event.top, $
		   ['File exists: '+file,'Overwrite it?'])
		  IF itmp EQ 'Cancel' THEN GoTo,out
		  IF itmp EQ 'No' THEN GoTo,back1
		ENDIF
		out=a.mu
		energy=a.energy
		out1=FltArr(N_Elements(out[*,0])+1,N_Elements(out[0,*]))
		out1[1:N_Elements(out[*,0]),*]=out
		out1[0,*]=energy
		Openw,unit,file,/Get_Lun
		ncol = N_Elements(out1[*,0])
		Widget_Control,/HourGlass
		PrintF,unit,'#F '+file
		PrintF,unit,'#D '+SysTime()
		PrintF,unit,'#C Written by XOP/DELIA'
		PrintF,unit,'#S 1 Calibrated data'
		PrintF,unit,'#U raw data file: '+file0
		PrintF,unit,'#U calibration coeffs: '
		PrintF,unit,'#U Xnew[i] = a + b*(X[i]-X[0]) + c*(X[i]-X[0])^2'
		Widget_Control,state.wids.mu.cal_a,Get_Value=cal_a
		Widget_Control,state.wids.mu.cal_b,Get_Value=cal_b
		Widget_Control,state.wids.mu.cal_c,Get_Value=cal_c
		Widget_Control,state.wids.mu.cal_x0,Get_Value=cal_x0
		PrintF,unit,'#U     a =  '+StrCompress(cal_a,/Rem) 
		PrintF,unit,'#U     b =  '+StrCompress(cal_b,/Rem) 
		PrintF,unit,'#U     c =  '+StrCompress(cal_c,/Rem) 
		PrintF,unit,'#U     x0 =  '+StrCompress(cal_x0,/Rem) 
		PrintF,unit,'#N '+StrCompress(N_Elements(out1[*,0]),/Rem)

		labels = StrCompress(sIndGen(N_Elements(out1[*,0])-1),/Rem)
		labels = 's'+labels
		labels =  ['Energy[eV]',labels]
		tmp = '#L '
		FOR i=0L,N_Elements(out1[*,0])-1 DO tmp = tmp+'  '+labels[i]
		PrintF,unit,tmp

		FOR i=0L,N_Elements(out1[0,*])-1 DO $
			PrintF,unit,out1[*,i],$
			Format='('+StrCompress(ncol,/Rem)+'G12.6)'
		Free_Lun,unit
		END
	  else:
	endcase
	END
  'CHI_MU2CHI': BEGIN
	str = *(state.ptr[1])
	allIndices=str.index


	indexcol=0
	widget_control,state.wids.chi.mu2chi,get_value=indexcol,/hourglass
	widget_control,event.id,get_value=val

	m1=Min(allIndices,Max=m2)

	CASE StrCompress(val,/Rem) OF
	  '|<': indexcol = m1
	  '<':  indexcol=indexcol-1 
	  '>':  indexcol=indexcol+1
	  '>|': indexcol = m2
	  'Reset':  reset=1
	  else: 
	ENDCASE
	Widget_Control,state.wids.chi.mu2chi,Set_Value=indexcol
	IF indexcol LT m1 THEN BEGIN
	   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	    'Wrong index value. Reset to: '+StrCompress(m1))
	   Widget_Control,state.wids.chi.mu2chi,Set_Value=m1
	   goto,out
	ENDIF
	IF indexcol GT m2 THEN BEGIN
	   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	    'Wrong index value. Reset to: '+StrCompress(m2))
	   Widget_Control,state.wids.chi.mu2chi,Set_Value=m2
	   goto,out
	ENDIF



	myindex = where(str.index EQ indexcol)
	if myindex[0] EQ -1 then begin
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	    'Spectrum with selected index('+strcompress(indexcol,/rem)+$
	    ') not found')
	  goto,out
	endif
	set = fltarr(2,n_elements(str.energy))
	set[0,*] = reform(str.energy)
	set[1,*] = reform(str.mu[myindex,*])

;  	Widget_Control, stateid, set_UValue=state

	p=state.wids.chi.parentMu2Chi
	if Widget_Info(p,/Valid_ID) then begin
	  xaid_mu2chi_loadfile,p,set
	  str=xaid_mu2chi_getstr(p)
	  str.file= $
	    '<var from DELIA. Index: '+StrCompress(myindex,/Rem)+' >'
	  xaid_mu2chi_putstr,p,str
	  xaid_mu2chi_plot,p
	endif else begin
	  ;
	  ; get from DELIA panel
	  ;
	  Widget_Control,state.wids.chi.preMin,Get_Value=preMin
	  Widget_Control,state.wids.chi.preMax,Get_Value=preMax
	  Widget_Control,state.wids.chi.e0,Get_Value=e0
	  Widget_Control,state.wids.chi.jump,Get_Value=jump
	  Widget_Control,state.wids.chi.npol,Get_Value=npol
	  Widget_Control,state.wids.chi.degree,Get_Value=degree
	  Widget_Control,state.wids.chi.knots,Get_Value=knots
	  Widget_Control,state.wids.chi.extraction,Get_Value=extraction
	  Widget_Control,state.wids.chi.kkmin,Get_Value=kkmin
	  Widget_Control,state.wids.chi.kkmax,Get_Value=kkmax
	  ; 
	  ; open xaid_mu2chi
	  ;
	  xaid_mu2chi,set,group=event.top, deliaid=state.wids, $
	    Parent=parent
	  str=xaid_mu2chi_getstr(parent)
	  ; 
	  ; copy values from delia panel
	  ;
	  str.file= $
	    '<var from DELIA. Index: '+StrCompress(myindex,/Rem)+' >'
	  str.preMin=preMin
	  str.preMax=preMax
	  str.e0=e0
	  str.jump=jump
	  str.npol=npol
	  str.degree=degree
	  str.knots=knots
	  str.extraction=extraction
	  str.kkmin=kkmin
	  str.kkmax=kkmax
	  ; 
	  xaid_mu2chi_putstr,parent,str
	  xaid_mu2chi_plot,parent
	  ; store mu2chi info
	  state.wids.chi.parentMu2Chi=parent
	  Widget_Control,event.top,Set_UValue=state
	endelse
	END


  'CHI_GO': BEGIN
	str = *(state.ptr[1])
	a_energy = str.energy
	a_index  = str.index
	a_mu     = str.mu

	Widget_Control,event.id,Get_Value=val
	CASE val OF
	  '** Calculate extraction OVER ALL spectra **': BEGIN
		i_good = LIndGen(N_Elements(a_index))
		a_good=a_index
	        out2=FltArr(N_Elements(a_good),N_Elements(a_energy))
	        out3=FltArr(N_Elements(a_good),N_Elements(a_energy))
		END
	  'Recalculate some spectra...': BEGIN
                if NOT(Ptr_Valid(state.ptr[2]))  then begin
		   itmp = Dialog_Message(/Error, Dialog_Parent=event.top, $
			'Please calculate ALL SPECTRA before.')
		   GoTo,out
		endif
		str = *(state.ptr[2])
		out2=str.spline
		out3=str.chi
		tmp = {min:Min(a_index), max:Max(a_index)}
		XScrMenu,tmp,/NoType,/Interp,Action=action, $
		  WTitle='Select limits for the index values...',$
		  Titles=['Min index','Max index'], $
		  Dialog_Parent=event.top
		IF action EQ 'DONT' THEN GoTo,out
		i_good = Where((a_index GE tmp.min) AND (a_index LE tmp.max))
		IF i_good[0] EQ -1 THEN BEGIN
		  itmp = Dialog_Message(/Error,Dialog_parent=event.top, $
			'No indices found in such an interval: '+$
			vect2string([tmp.min,tmp.max]) )
		  GoTo,out
		ENDIF
		a_good = a_index[i_good]
		END
	  else:
	ENDCASE
	n_good = N_Elements(i_good)
	t0 = sysTime(1)
	FOR i=0,n_good-1 DO BEGIN
	  ii=i_good[i]
	  mu = Reform(a_mu[ii,*]) 
	  set = Make_Set(a_energy,mu)
	  ; 
	  Widget_Control,state.wids.chi.preMin,Get_Value=preMin
	  Widget_Control,state.wids.chi.preMax,Get_Value=preMax
	  Widget_Control,state.wids.chi.e0,Get_Value=e0
	  Widget_Control,state.wids.chi.jump,Get_Value=jump
	  Widget_Control,state.wids.chi.npol,Get_Value=npol
	  Widget_Control,state.wids.chi.degree,Get_Value=degree
	  Widget_Control,state.wids.chi.knots,Get_Value=knots
	  Widget_Control,state.wids.chi.extraction,Get_Value=extraction
	  strinp = xaid_defaults('xaid_mu2chi')
	  strinp.preMin=preMin
	  strinp.preMax=preMax
	  strinp.e0=e0
	  strinp.jump=jump
	  strinp.npol=npol
	  strinp.degree=degree
	  strinp.knots=knots
          tmp = (strinp.extraction)
	  tmp[0]=StrCompress(extraction,/Rem)
          strinp.extraction = tmp
	  xaid_mu2chi_calc,set=set,str=strinp,out=out,coltitles=coltitles
	  kk = Reform(out[2,*])
	  out2[ii,*] = Reform(out[7,*])
	  out3[ii,*] = Reform(out[8,*])
	  widget_control,state.wids.labelMain,Set_Value='Done '+$
		String(100.*Float(i+1)/n_good,Format='(I4)')+$
		'% ('+StrCompress(i+1)+' out of '+StrCompress(n_good)+')'
	ENDFOR
	t1 = sysTime(1)-t0
        widget_control,state.wids.labelMain,Set_Value='Done ('+$
	  StrCompress(n_good)+' spectra in '+$
	  String(t1,Format='(I6)')+' seconds, or '+$
	  String(t1/n_good,Format='(F10.6)')+' s/spectrum)'

	;
	; save data in Ptr
	;
	str = {k:kk, index:a_index, chi:out3, spline:out2}
	for i=2,n_elements(state.ptr)-1 do $
          if Ptr_Valid(state.ptr[i]) then Ptr_Free,state.ptr[i]
	state.ptr[2]=Ptr_New(str)

	;
	; plot
	;
	;widget_control,state.wids.labelMain,Set_Value=$
	;   'Preparing plots...'
	txt=1
	widget_control,state.wids.chi.kkmin,get_value=kmin
	widget_control,state.wids.chi.kkmax,get_value=kmax

	tmp = delia_reform_arrays(kk,out3,a_index,/xplot,text=txt,$
	  wtitle='Chi 3D plot',No_Block=state.no_block,xrange=[kmin,kmax])
	widget_control,state.wids.chi.text,Set_Value=txt
	;widget_control,state.wids.labelMain,Set_Value='Done'

	END


  'CHI_PLOT': BEGIN
	if Ptr_Valid(state.ptr[2]) then a=*(state.ptr[2]) else begin
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	  ' Data containing Chi not found.')
	  goto,out
	endelse
	val=''
	widget_control,event.id,get_value=val
	widget_control,state.wids.chi.kkmin,Get_Value=kkmin
	widget_control,state.wids.chi.kkmax,Get_Value=kkmax
	case val of
	  '2 D': BEGIN
		nnx = 1+n_elements(a.chi[*,0])
		nny = n_elements(a.chi[0,*])
		out2 = fltarr(nnx,nny)
		out2[0,*]=a.k
		out2[1:nnx-1,0:nny-1]=a.chi
		IF kkmin NE kkmax THEN BEGIN
		  igood = where( (out2[0,*] GE kkmin) AND $
			(out2[0,*] LE kkmax) )
		  out2=out2[*,igood]
		ENDIF
		xplot,out2,/no_block,coltitles=['k [A]','Chi for: '+$
		  string(a.index)], group=event.top, Wtitle='Chi 2D plot'
		END
	  '2 D (splines)': BEGIN
		nnx = 1+n_elements(a.spline[*,0])
		nny = n_elements(a.spline[0,*])
		out2 = fltarr(nnx,nny)
		out2[0,*]=a.k
		out2[1:nnx-1,0:nny-1]=a.spline
		IF kkmin NE kkmax THEN BEGIN
		  igood = where( (out2[0,*] GE kkmin) AND $
			(out2[0,*] LE kkmax) )
		  out2=out2[*,igood]
		ENDIF
		xplot,out2,/no_block,coltitles=['k [A]','Spl for: '+$
		  string(a.index)], group=event.top, Wtitle='Splines 2D plot'
		END
	  '3 D': BEGIN
		tmp = delia_reform_arrays(a.k,a.chi,a.index,$
		  /xplot,group=event.top,wtitle='Chi 3D plot',$
		  No_Block=state.no_block,xrange=[kkmin,kkmax])
		END
	  '3 D (splines)': BEGIN
		tmp = delia_reform_arrays(a.k,a.spline,a.index,$
		  /xplot,group=event.top,wtitle='Splines 3D plot',$
		  No_Block=state.no_block,xrange=[kkmin,kkmax])
		END
	  'Xsurface1': BEGIN
		igood = Where(a.k GE kkmin and a.k LE kkmax)
		k = a.k
		achi = a.chi
		IF igood[0] NE -1 THEN BEGIN
		  achi=achi[*,igood] 
		  k=k[igood] 
		ENDIF
		Xsurface1,achi,a.index,k,Xtitle='index',YTitle='k'
		END
	  'iSurface': BEGIN
		igood = Where(a.k GE kkmin and a.k LE kkmax)
		k = a.k
		achi = a.chi
		IF igood[0] NE -1 THEN BEGIN
		  achi=achi[*,igood] 
		  k=k[igood] 
		ENDIF
		iSurface,achi,a.index,k,Xtitle='index',YTitle='k'
		END
	  'iContour': BEGIN
		igood = Where(a.k GE kkmin and a.k LE kkmax)
		k = a.k
		achi = a.chi
		IF igood[0] NE -1 THEN BEGIN
		  achi=achi[*,igood] 
		  k=k[igood] 
		ENDIF
		iContour,achi,a.index,k,Xtitle='index',YTitle='k'
		END
	  'iImage': BEGIN
		igood = Where(a.k GE kkmin and a.k LE kkmax)
		k = a.k
		achi = a.chi
		IF igood[0] NE -1 THEN BEGIN
		  achi=achi[*,igood] 
		  k=k[igood] 
		ENDIF
		iImage,achi,a.index,k,Xtitle='index',YTitle='k'
		END
	  'Export chi data...': BEGIN
		igood = Where(a.k GE kkmin and a.k LE kkmax)
		k = a.k
		achi = a.chi
		IF igood[0] NE -1 THEN BEGIN
		  achi=achi[*,igood] 
		  k=k[igood] 
		ENDIF

		CASE state.raw_type OF
		  'Sequential': BEGIN
			Widget_Control,state.wids.raw.file,Get_Value=file0
			IF file0 EQ '' THEN file0='delia'
			END
		  'CCD': BEGIN
			Widget_Control,state.wids.raw.ccdfile,Get_Value=file0
			IF file0 EQ '' THEN file0='delia'
			END
		ENDCASE
backChi:
		file = Dialog_Pickfile(/Write,Dialog_Parent=event.top, $
			File=file0+'.chi')
		IF file EQ '' THEN GoTo,out
		IF CheckFile(file) THEN BEGIN
		  itmp = Dialog_Message(/Info,/Cancel,Dialog_Parent=event.top, $
		   ['File exists: '+file,'Overwrite it?'])
		  IF itmp EQ 'Cancel' THEN GoTo,out
		  IF itmp EQ 'No' THEN GoTo,backChi
		ENDIF
		out1=FltArr(N_Elements(achi[*,0])+1,N_Elements(achi[0,*]))
		out1[1:N_Elements(achi[*,0]),*]=achi
		out1[0,*]=k
		Openw,unit,file,/Get_Lun
		ncol = N_Elements(out1[*,0])
		Widget_Control,/HourGlass
		PrintF,unit,'#F '+file
		PrintF,unit,'#D '+SysTime()
		PrintF,unit,'#C Written by XOP/DELIA'
		PrintF,unit,'#S 1 Mu data'
		PrintF,unit,'#U raw data file: '+file0

		PrintF,unit,'#U calibration coeffs: '
		PrintF,unit,'#U Xnew[i] = a + b*(X[i]-X[0]) + c*(X[i]-X[0])^2'
		Widget_Control,state.wids.mu.cal_a,Get_Value=cal_a
		Widget_Control,state.wids.mu.cal_b,Get_Value=cal_b
		Widget_Control,state.wids.mu.cal_c,Get_Value=cal_c
		Widget_Control,state.wids.mu.cal_x0,Get_Value=cal_x0
		PrintF,unit,'#U     a =  '+StrCompress(cal_a,/Rem) 
		PrintF,unit,'#U     b =  '+StrCompress(cal_b,/Rem) 
		PrintF,unit,'#U     c =  '+StrCompress(cal_c,/Rem) 
		PrintF,unit,'#U     x0 =  '+StrCompress(cal_x0,/Rem) 

		PrintF,unit,'#U EXAFS extraction parameters: '
		Widget_Control,state.wids.chi.preMin,Get_Value=preMin
		Widget_Control,state.wids.chi.preMax,Get_Value=preMax
		Widget_Control,state.wids.chi.e0,Get_Value=e0
		Widget_Control,state.wids.chi.jump,Get_Value=jump
		Widget_Control,state.wids.chi.degree,Get_Value=degree
		Widget_Control,state.wids.chi.knots,Get_Value=knots
		Widget_Control,state.wids.chi.extraction,Get_Value=extraction
		PrintF,unit,'#U     preMin =  '+StrCompress(preMin,/Rem) 
		PrintF,unit,'#U     preMax =  '+StrCompress(preMax,/Rem) 
		PrintF,unit,'#U     e0 =  '+StrCompress(e0,/Rem) 
		PrintF,unit,'#U     jump =  '+StrCompress(jump,/Rem) 
		PrintF,unit,'#U     degree =  '+degree
		PrintF,unit,'#U     knots =  '+knots
		PrintF,unit,'#U     extraction(0=experimental,1=Constant,2=Lengeler-Einsenberg) =  '+extraction

		PrintF,unit,'#N '+StrCompress(N_Elements(out1[*,0]),/Rem)

		labels = StrCompress(Fix(a.index),/Rem)
		labels = 's'+labels
		labels =  ['Chi[A**-1]',labels]
		tmp = '#L '
		FOR i=0L,N_Elements(out1[*,0])-1 DO tmp = tmp+'  '+labels[i]
		PrintF,unit,tmp

		FOR i=0L,N_Elements(out1[0,*])-1 DO $
			PrintF,unit,out1[*,i],$
			Format='('+StrCompress(ncol,/Rem)+'G12.6)'
		Free_Lun,unit

		END
	endcase
	END
  'FT_CHI2FT': BEGIN
	indexcol=0
	widget_control,state.wids.ft.chi2ft,get_value=indexcol
	str = *(state.ptr[2])
	allIndices=str.index

	indexcol=0
	widget_control,state.wids.ft.chi2ft,get_value=indexcol,/hourglass
	widget_control,event.id,get_value=val

	m1=Min(allIndices,Max=m2)

	CASE StrCompress(val,/Rem) OF
	  '|<': indexcol = m1
	  '<':  indexcol=indexcol-1 
	  '>':  indexcol=indexcol+1
	  '>|': indexcol = m2
	  'Reset':  reset=1
	  else: 
	ENDCASE
	Widget_Control,state.wids.ft.chi2ft,Set_Value=indexcol
	IF indexcol LT m1 THEN BEGIN
	   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	    'Wrong index value. Reset to: '+StrCompress(m1))
	   Widget_Control,state.wids.ft.chi2ft,Set_Value=m1
	   goto,out
	ENDIF
	IF indexcol GT m2 THEN BEGIN
	   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	    'Wrong index value. Reset to: '+StrCompress(m2))
	   Widget_Control,state.wids.ft.chi2ft,Set_Value=m2
	   goto,out
	ENDIF

	myindex = where(str.index EQ indexcol)
	if myindex[0] EQ -1 then begin
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	    'Spectrum with selected index('+strcompress(indexcol,/rem)+$
	    ') not found')
	  goto,out
	endif
	set = fltarr(2,n_elements(str.k))
	set[0,*] = reform(str.k)
	set[1,*] = reform(str.chi[myindex,*])

	p=state.wids.ft.parentFF
	if Widget_Info(p,/Valid_ID) then begin
	  xaid_ff_loadfile,p,set
	endif else begin
	  ;
	  xaid_ff,set,group=event.top, deliaid=state.wids, $
	    Parent=parent
	  ; store info
	  state.wids.ft.parentFF=parent
	  Widget_Control,event.top,Set_UValue=state
	endelse
	END

  'FT_GO': BEGIN
	str = *(state.ptr[2])
	; 
	; preedge
	;
	widget_control,state.wids.labelMain,Set_Value=$
	   'calculating Fourier & Back F transforms...'
	kMin=0 & kMin=0 & k2Min=0 & k2Min=0
	widget_control,state.wids.ft.kMin,get_value=kMin
	widget_control,state.wids.ft.kMax,get_value=kMax
	widget_control,state.wids.ft.k2Min,get_value=k2Min
	widget_control,state.wids.ft.k2Max,get_value=k2Max
	rMin=0 & rMin=0 & r2Min=0 & r2Min=0
	widget_control,state.wids.ft.rMin,get_value=rMin
	widget_control,state.wids.ft.rMax,get_value=rMax
	widget_control,state.wids.ft.r2Min,get_value=r2Min
	widget_control,state.wids.ft.r2Max,get_value=r2Max
	filter=0 & kpower=0 & rshell=0
	widget_control,state.wids.ft.filter,get_value=filter
	widget_control,state.wids.ft.kpower,get_value=kpower
	widget_control,state.wids.ft.rshell,get_value=rshell
	filter=fix(filter)+1
	kpower=fix(kpower)
	app_value = 0
	widget_control,state.wids.ft.app_value,get_value=app_value
	set1 = fltarr(2,n_elements(str.k))
	set1[0,*] = reform(str.k)
	nspectra = n_elements(str.chi[*,0])
	npts=long(n_elements(str.k))

	widget_control,/hourglass
	;out = fltarr(nspectra,npoints)
	for i=0L,nspectra-1 do begin
	  widget_control,state.wids.labelMain,Set_Value=$
	   'calculating FT & BFT for spectrum: '+strcompress(i+1)+' of '+$
	   strcompress(nspectra)
	  ;
	  ; ft calculation
	  ;
	  ft=0
	  set1[1,*] = str.chi[i,*]
	  ftrset,set1,ft,method=1,xrange=[kMin,kMax],rrange=[rMin,rMax], $
	    kpower=kpower,window=filter,windpar=app_value,$ ; /plotfourier,$
	    npoint=4096,kstep=0.04
	  if i EQ 0 then begin
	    npts_out = n_elements(ft[0,*])
	    ; 5 columns: r index ft_real, ft_imag ft_mod
	    out = fltarr(5,npts_out*nspectra)
	  endif
	  out[0,i*npts_out:(i*npts_out)+npts_out-1] = reform(ft[0,*])
	  out[1,i*npts_out:(i*npts_out)+npts_out-1] = str.index[i]
	  out[2,i*npts_out:(i*npts_out)+npts_out-1] = reform(ft[2,*])
	  out[3,i*npts_out:(i*npts_out)+npts_out-1] = reform(ft[3,*])
	  out[4,i*npts_out:(i*npts_out)+npts_out-1] = reform(ft[1,*])
	  ;
	  ; bf calculation
	  ;
	  bftrset,ft,setbf,rrange=[r2Min,r2Max], krange=[k2Min,k2Max], $
	  	window=filter,wrange=[kMin,kMax],method=1,$
	  	rshell=rshell,windpar=app_value
	  if i EQ 0 then begin
	    npts_out2 = n_elements(setbf[0,*])
	    ; 5 columns: k index bft_real, bft_modulus bft_phase
	    out2 = fltarr(5,npts_out2*nspectra)
	  endif
	  out2[0,i*npts_out2:(i*npts_out2)+npts_out2-1] = reform(setbf[0,*])
	  out2[1,i*npts_out2:(i*npts_out2)+npts_out2-1] = str.index[i]
	  out2[2,i*npts_out2:(i*npts_out2)+npts_out2-1] = reform(setbf[2,*])
	  out2[3,i*npts_out2:(i*npts_out2)+npts_out2-1] = reform(setbf[3,*])
	  out2[4,i*npts_out2:(i*npts_out2)+npts_out2-1] = reform(setbf[1,*])
	endfor

	widget_control,state.wids.labelMain,Set_Value='Done'
	windowlist = 1  & window_ftr,names=windowlist

	;
	; save data in Ptr
	;
        for i=3,n_elements(state.ptr)-1 do $
          if Ptr_Valid(state.ptr[i]) then Ptr_Free,state.ptr[i]
	state.ptr[3]=Ptr_New({npoints:npts_out,nspectra:nspectra,data:out,$
	  kmin:kmin,kmax:kmax,kpower:kpower,filter:filter})
	state.ptr[4]=Ptr_New({npoints:npts_out2,nspectra:nspectra,data:out2,$
	  k2min:k2min,k2max:k2max,r2min:r2min,r2max:r2max,rshell:rshell})
	END



  'FT_PLOT': BEGIN
	val=''
	widget_control,event.id,get_value=val
	case val of
	  'Fourier (2D)': BEGIN
		plotwhat='fourier'
		plotdim=2
		END
	  'Fourier (3D)': BEGIN
		plotwhat='fourier'
		plotdim=3
		END
	  'Back F  (2D)': BEGIN
		plotwhat='back'
		plotdim=2
		END
	  'Back F  (3D)': BEGIN
		plotwhat='back'
		plotdim=3
		END
	  'Xsurface1': BEGIN
		if Ptr_Valid(state.ptr[3]) then a=*(state.ptr[3]) else begin
		  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		  ' Data containing Fourier Transform not found.')
		  goto,out
		endelse
		  mydata = fltarr(a.nspectra+1,a.npoints)
		  rr = Reform(a.data[0,0:a.npoints-1])
		  mydataindex = reform(a.data[1,*])
		  mydataindex=mydataindex[uniq(mydataindex)]
		  icol=4
		  mydatay = reform(a.data[icol,*])
		  tmp=reform(mydatay,a.npoints,a.nspectra)
		  Xsurface1,tmp,rr,mydataindex,Xtitle='R',ytitle='Index'
		Return
		END
	  'iSurface': BEGIN
		if Ptr_Valid(state.ptr[3]) then a=*(state.ptr[3]) else begin
		  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		  ' Data containing Fourier Transform not found.')
		  goto,out
		endelse
		  mydata = fltarr(a.nspectra+1,a.npoints)
		  rr = Reform(a.data[0,0:a.npoints-1])
		  mydataindex = reform(a.data[1,*])
		  mydataindex=mydataindex[uniq(mydataindex)]
		  icol=4
		  mydatay = reform(a.data[icol,*])
		  tmp=reform(mydatay,a.npoints,a.nspectra)
		  iSurface,tmp,rr,mydataindex,Xtitle='R',ytitle='Index'
		Return
		END
	  'iContour': BEGIN
		if Ptr_Valid(state.ptr[3]) then a=*(state.ptr[3]) else begin
		  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		  ' Data containing Fourier Transform not found.')
		  goto,out
		endelse
		  mydata = fltarr(a.nspectra+1,a.npoints)
		  rr = Reform(a.data[0,0:a.npoints-1])
		  mydataindex = reform(a.data[1,*])
		  mydataindex=mydataindex[uniq(mydataindex)]
		  icol=4
		  mydatay = reform(a.data[icol,*])
		  tmp=reform(mydatay,a.npoints,a.nspectra)
		  iContour,tmp,rr,mydataindex,Xtitle='R',ytitle='Index'
		Return
		END
	  'iImage': BEGIN
		if Ptr_Valid(state.ptr[3]) then a=*(state.ptr[3]) else begin
		  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		  ' Data containing Fourier Transform not found.')
		  goto,out
		endelse
		  mydata = fltarr(a.nspectra+1,a.npoints)
		  rr = Reform(a.data[0,0:a.npoints-1])
		  mydataindex = reform(a.data[1,*])
		  mydataindex=mydataindex[uniq(mydataindex)]
		  icol=4
		  mydatay = reform(a.data[icol,*])
		  tmp=reform(mydatay,a.npoints,a.nspectra)
		  iImage,tmp,rr,mydataindex,Xtitle='R',ytitle='Index'
		Return
		END
	  'Export |FT| data...': BEGIN
		if Ptr_Valid(state.ptr[3]) then a=*(state.ptr[3]) else begin
		  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		  ' Data containing Fourier Transform not found.')
		  goto,out
		endelse
		mydata = fltarr(a.nspectra+1,a.npoints)
		rr = Reform(a.data[0,0:a.npoints-1])
		mydataindex = reform(a.data[1,*])
		mydataindex=mydataindex[uniq(mydataindex)]
		icol=4
		mydatay = reform(a.data[icol,*])
		out=reform(mydatay,a.npoints,a.nspectra)
		out=transpose(out)

		CASE state.raw_type OF
		  'Sequential': BEGIN
			Widget_Control,state.wids.raw.file,Get_Value=file0
			IF file0 EQ '' THEN file0='delia'
			END
		  'CCD': BEGIN
			Widget_Control,state.wids.raw.ccdfile,Get_Value=file0
			IF file0 EQ '' THEN file0='delia'
			END
		ENDCASE
backFT:
		file = Dialog_Pickfile(/Write,Dialog_Parent=event.top, $
			File=file0+'.ft')
		IF file EQ '' THEN GoTo,out
		IF CheckFile(file) THEN BEGIN
		  itmp = Dialog_Message(/Info,/Cancel,Dialog_Parent=event.top, $
		   ['File exists: '+file,'Overwrite it?'])
		  IF itmp EQ 'Cancel' THEN GoTo,out
		  IF itmp EQ 'No' THEN GoTo,backFT
		ENDIF
		out1=FltArr(N_Elements(out[*,0])+1,N_Elements(out[0,*]))
		out1[1:N_Elements(out[*,0]),*]=out
		out1[0,*]=rr
		Openw,unit,file,/Get_Lun
		ncol = N_Elements(out1[*,0])
		Widget_Control,/HourGlass
		PrintF,unit,'#F '+file
		PrintF,unit,'#D '+SysTime()
		PrintF,unit,'#C Written by XOP/DELIA'
		PrintF,unit,'#S 1 Mu data'
		PrintF,unit,'#U raw data file: '+file0

		PrintF,unit,'#U calibration coeffs: '
		PrintF,unit,'#U Xnew[i] = a + b*(X[i]-X[0]) + c*(X[i]-X[0])^2'
		Widget_Control,state.wids.mu.cal_a,Get_Value=cal_a
		Widget_Control,state.wids.mu.cal_b,Get_Value=cal_b
		Widget_Control,state.wids.mu.cal_c,Get_Value=cal_c
		Widget_Control,state.wids.mu.cal_x0,Get_Value=cal_x0
		PrintF,unit,'#U     a =  '+StrCompress(cal_a,/Rem) 
		PrintF,unit,'#U     b =  '+StrCompress(cal_b,/Rem) 
		PrintF,unit,'#U     c =  '+StrCompress(cal_c,/Rem) 
		PrintF,unit,'#U     x0 =  '+StrCompress(cal_x0,/Rem) 

		PrintF,unit,'#U EXAFS extraction parameters: '
		Widget_Control,state.wids.chi.preMin,Get_Value=preMin
		Widget_Control,state.wids.chi.preMax,Get_Value=preMax
		Widget_Control,state.wids.chi.e0,Get_Value=e0
		Widget_Control,state.wids.chi.jump,Get_Value=jump
		Widget_Control,state.wids.chi.degree,Get_Value=degree
		Widget_Control,state.wids.chi.knots,Get_Value=knots
		Widget_Control,state.wids.chi.extraction,Get_Value=extraction
		PrintF,unit,'#U     preMin =  '+StrCompress(preMin,/Rem) 
		PrintF,unit,'#U     preMax =  '+StrCompress(preMax,/Rem) 
		PrintF,unit,'#U     e0 =  '+StrCompress(e0,/Rem) 
		PrintF,unit,'#U     jump =  '+StrCompress(jump,/Rem) 
		PrintF,unit,'#U     degree =  '+degree
		PrintF,unit,'#U     knots =  '+knots
		PrintF,unit,'#U     extraction(0=experimental,1=Constant,2=Lengeler-Einsenberg) =  '+extraction

		PrintF,unit,'#U FT parameters: '
		Widget_Control,state.wids.ft.kmin,Get_Value=kmin
		Widget_Control,state.wids.ft.kmax,Get_Value=kmax
		Widget_Control,state.wids.ft.rmin,Get_Value=rmin
		Widget_Control,state.wids.ft.rmax,Get_Value=rmax
		Widget_Control,state.wids.ft.filter,Get_Value=filter
		Widget_Control,state.wids.ft.app_value,Get_Value=app_value
		Widget_Control,state.wids.ft.kPower,Get_Value=kPower
		PrintF,unit,'#U     kMin =  '+StrCompress(kMin,/Rem) 
		PrintF,unit,'#U     kMax =  '+StrCompress(kMax,/Rem) 
		PrintF,unit,'#U     rMin =  '+StrCompress(rMin,/Rem) 
		PrintF,unit,'#U     rMax =  '+StrCompress(rMax,/Rem) 
		PrintF,unit,'#U     filter (0=Gaussian,1=Hanning,2=None,3=Parzel,4=Wetch,'
                PrintF,unit,'#U            5=Hamming,6=Tukey,7=Papul,8=Kaiser'
                PrintF,unit,'#U     filter =  '+StrCompress(filter,/Rem) 
		PrintF,unit,'#U     app_value =  '+StrCompress(app_value,/Rem) 
		PrintF,unit,'#U     kPower =  '+StrCompress(kPower,/Rem) 


		PrintF,unit,'#N '+StrCompress(N_Elements(out1[*,0]),/Rem)

		labels = StrCompress(Fix(myDataIndex),/Rem)
		labels = 's'+labels
		labels =  ['FT[A]',labels]
		tmp = '#L '
		FOR i=0L,N_Elements(out1[*,0])-1 DO tmp = tmp+'  '+labels[i]
		PrintF,unit,tmp

		FOR i=0L,N_Elements(out1[0,*])-1 DO $
			PrintF,unit,out1[*,i],$
			Format='('+StrCompress(ncol,/Rem)+'G12.6)'
		Free_Lun,unit
		Return
		END
	endcase

	case plotwhat of
	  'fourier': BEGIN
		if Ptr_Valid(state.ptr[3]) then a=*(state.ptr[3]) else begin
		  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		  ' Data containing Fourier Transform not found.')
		  goto,out
		endelse
		windowlist = 1  & window_ftr,names=windowlist
		if plotdim EQ 2 then begin
		  isel = wmenu2(['Calcel','Plot Real part of FT',$
		    'Plot Imaginary part of FT','Plot FT Modulus'], $
			Dialog_Parent=event.top)
		  case isel of
			0: goto,out
			1: icol=2
			2: icol=3
			3: icol=4
		  endcase
		  mydata = fltarr(a.nspectra+1,a.npoints)
		  mydata[0,*] = a.data[0,0:a.npoints-1]
		  mydataindex = reform(a.data[1,*])
		  mydataindex=mydataindex[uniq(mydataindex)]
		  coltitles=['R [A]',strcompress(mydataindex,/rem)]
		  mydatay = reform(a.data[icol,*])
		  tmp=reform(mydatay,a.npoints,a.nspectra)
		  mydata[1:a.nspectra,*]=transpose(tmp)
		  xplot,mydata,group=event.top,/no_block, coltitles=coltitles,$
		  wtitle='Fourier Transform (2D)',$
		  title='FT k='+vect2string([a.kMin,a.kMax])+'; Weight=k^'+$
                  strcompress(a.kpower,/rem)+'; Window= '+windowlist[a.filter-1]
		endif else begin
		  xplot,a.data,xcol=1,ycol=5,/no_block,p=p,wtitle=$
		  'Fourier Transform (3D)',$
		  coltitles= ['R [A]','Index','Real(FT)',$
		  'Imag(FT)','Mod(FT)'],$
		  title='FT k='+vect2string([a.kMin,a.kMax])+'; Weight=k^'+$
		  strcompress(a.kpower,/rem)+'; Window= '+windowlist[a.filter-1]
		  xplot_mesh,p,flag=1,col=2,interactive=0
                  xplot_refresh,p
		endelse
		END
	  'back': BEGIN
		if Ptr_Valid(state.ptr[4]) then a=*(state.ptr[4]) else begin
		  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		  ' Data containing Back Fourier Transform not found.')
		  goto,out
		endelse

		if plotdim EQ 2 then begin
		  isel = wmenu2(['Calcel','Plot Modulus of BFT',$
		    'Plot Phase of BFT','Plot real part of BFT'], $
			Dialog_Parent=event.top)
		  case isel of
			0: goto,out
			1: icol=2
			2: icol=3
			3: icol=4
		  endcase
		  mydata = fltarr(a.nspectra+1,a.npoints)
		  mydata[0,*] = a.data[0,0:a.npoints-1]
		  mydataindex = reform(a.data[1,*])
		  mydataindex=mydataindex[uniq(mydataindex)]
		  coltitles=['K [A^-1]',strcompress(mydataindex,/rem)]
		  mydatay = reform(a.data[icol,*])
		  tmp=reform(mydatay,a.npoints,a.nspectra)
		  mydata[1:a.nspectra,*]=transpose(tmp)
		  xplot,mydata,group=event.top,/no_block, coltitles=coltitles,$
		  wtitle='Back Fourier Transform (2D)',$
		  title='BFT k='+vect2string([a.k2Min,a.k2Max])+$
		  ' r='+vect2string([a.r2Min,a.r2Max])+$
		  '; R shell= '+strcompress(a.rshell,/rem)

		endif else begin
		  xplot,a.data,xcol=1,ycol=5,/no_block,p=p,wtitle=$
		  'Back Fourier Transform (3D)',$
		  coltitles=['K [A^-1]','Index',$
		  'Modulus(BFT)','Phase(BFT)','real(BFT)'],$
		   title='BFT k='+vect2string([a.k2Min,a.k2Max])+$
		   ' r='+vect2string([a.r2Min,a.r2Max])+$
		  '; R shell= '+strcompress(a.rshell,/rem)
		  xplot_mesh,p,flag=1,col=2,interactive=0
                  xplot_refresh,p
		endelse


		END
	endcase
	END

  'MAPPANELS': delia_mappanels,state.wids
  'ABOUT': XAID_Help,Group=event.top
  'HELP': BEGIN
	widget_control,event.id,get_value=tmp
	xhelp,tmp,GROUP=event.top
	END
  'DABAX_EXAFS': BEGIN
	Widget_Control,/Hourglass
	file = 'XAFS_DATA_EXAFS_CHI.dat'
	h=dabax_access(file(0))
	xplot,spec=h,wtitle=file(0),xtitle='-1',ytitle='-1',title='#S'
	END
  'DABAX_XANES': BEGIN
	Widget_Control,/Hourglass
	file = 'XAFS_DATA_XANES.dat'
	h=dabax_access(file(0))
	xplot,spec=h,wtitle=file(0),xtitle='-1',ytitle='-1',title='#S'
	END
  'XOP_WD': BEGIN
	Xop_WD,Group=event.top
	END
  'TAB_RAW': BEGIN
	IF event.tab EQ 0 THEN BEGIN
		state.raw_type='Sequential'
		Widget_Control,state.wids.raw.draw,Get_Value=w
		WSet,w
		tvscl,fltarr(500,500)
	ENDIF ELSE IF event.tab EQ 1 THEN BEGIN
		state.raw_type='CCD'
		Widget_Control,state.wids.raw.text,Set_Value='<none>'
	ENDIF
	END
  'TAB': 
  'TAB_RAW': BEGIN
	CASE event.tab OF
	  0: state.raw_type='CCD'
	  1: state.raw_type='Sequential'
	  else:
	ENDCASE
	END 
  '':
  else: BEGIN
	tmp=''
	Widget_Control,event.id,Get_UValue=tmp,/Hourglass
	command = tmp[0]+ $  ;',Group='+StrCompress(event.Id,/Rem)+'L'+$
	 ',No_Block='+StrCompress(state.no_block)
	Message,/info,'Executing: '+command
	itmp = Execute(command)
	END
EndCase


out:
if Widget_Info(stateid,/Valid_ID) then $
  Widget_Control, stateid, set_UValue=state ; , /No_Copy
end ; delia_event
;
;====================================================================
;

PRO delia, group=group, No_Block=no_Block

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'DELIA: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   return
endif
IF N_Elements(no_block) EQ 0 THEN no_block=1

;
; color setting
;
Device,Get_Decomposed=usingDecomposed
IF usingDecomposed EQ 1 THEN BEGIN
  itmp = Dialog_Message(/Question, $
     ['DELIA does not work properly with decomposed colors',$
     'Turn decomposed colors off?'])
  IF itmp EQ 'Yes' THEN Device,Decomposed=0  
ENDIF
Tek_Color
;
; define widgets
;
wbase=widget_base(/col,title='DELIA 2.0Beta2',MBar=wMenuBar)

wtmp = widget_base(wbase) ; to store state

wFileMenu = Widget_Button(wmenuBar,Value='File',/Menu)
;  wtmp = Widget_Button(wFileMenu,Value='Load file with ASCII RAW data...',$
;	UValue='RAW_LOAD_ASCII')
;  wtmp = Widget_Button(wFileMenu,Value='Load file with Mu(E)...',UValue='LOADFILE')
;  wtmp = Widget_Button(wFileMenu,Value='Load file with Chi(k)...',UValue='LOADFILE')
  wtmp0 = widget_button(wFileMenu,Value='Export CCD file',/Menu)
    wtmp = widget_button(wtmp0,Value='Export calibrated data...', UValue='MU_PLOT')
    wtmp = widget_button(wtmp0,Value='Export chi data...', UValue='CHI_PLOT')
    wtmp = widget_button(wtmp0,Value='Export |FT| data...', UValue='FT_PLOT')
  wtmp = Widget_Button(wFileMenu,Value='Quit',UValue='QUIT',/Separator)
wExafsToolsMenu = Widget_Button(wmenuBar,Value='EXAFS_Tools',/Menu)
  wtmp = Widget_Button(wExafsToolsMenu,Value='DELIA',$
	UValue='DELIA')
  wtmp = Widget_Button(wExafsToolsMenu,Value='Main window (XAID)',$
	UValue='XAID')
  wtmp = Widget_Button(wExafsToolsMenu,Value='Calibration (XAID_CALIB)',$
	UValue='XAID_CALIB')
;  wtmp = Widget_Button(wExafsToolsMenu,Value='Calibration (XAID_CALIB)',$
;	UValue='XAID_CALIB')
;  wtmp = Widget_Button(wExafsToolsMenu,Value='Extraction (XAID_MU2CHI)',$
;	UValue='XAID_MU2CHI')
;  wtmp = Widget_Button(wExafsToolsMenu,Value='Fourier Filtering (XAID_FF)',$
;	UValue='XAID_FF')
;wDabaxMenu = Widget_Button(wmenuBar,Value='DABAX_Tools',/Menu)
;  wtmp = Widget_Button(wDabaxMenu,Value='f1 and f2 values',$
;	UValue='XF1F2')
;  wtmp = Widget_Button(wDabaxMenu,Value='McKale phases',$
;	UValue='XMCKALE')
;  wtmp = Widget_Button(wDabaxMenu,Value='EXAFS spectra database',$
;	/Separator,UValue='DABAX_EXAFS')
;  wtmp = Widget_Button(wDabaxMenu,Value='XANES spectra database',$
;	UValue='DABAX_XANES')
wToolsMenu = Widget_Button(wmenuBar,Value='Tools',/Menu)
  wtmp = Widget_Button(wToolsMenu,Value='Plotting tool...',$
	UValue='XPLOT')
  wtmp = Widget_Button(wToolsMenu,Value='Display file...',$
        UValue='XOP_FILE_DISPLAY')
  wtmp = Widget_Button(wToolsMenu,Value='Change working directory...',$
	UValue='XOP_WD')
  wtmp = Widget_Button(wToolsMenu,Value='Change color table',$
	UValue='XLOADCT;')
  wtmp = Widget_Button(wToolsMenu,Value='Default color table',$
	UValue='XOP_LOAD_IMAGE,/Set_Default;')

wHelpMenu = Widget_Button(wmenuBar,Value='Help',/Help)
  wtmp = Widget_Button(wHelpMenu,Value='About XAID',UValue='ABOUT')
  wtmp = Widget_Button(wHelpMenu,Value='delia',UValue='HELP')
;
; TAB sub panels
;
wpanels = Widget_Tab(wbase,UVal='SWITCH')
;wtmp0 = widget_base(wbase)
  wbase_raw = widget_base(wpanels,title='Raw',/Col,/Frame)
  wbase_mu = widget_base(wpanels,title='mu(E)',/Col,/Frame)
  wbase_chi = widget_base(wpanels,title='chi(k)',/Col,/Frame)
  wbase_ft = widget_base(wpanels,title='Fourier',/Col,/Frame)



wlabelMain = widget_text(wbase,Value=$
;  '**********************************************************************')
  '*****Welcome to DELIA (Dispersive Exafs caLIbration and Analysis)******')
  ;/Dynamic_Resize)
;
; Raw data panel
;
wtmp0 = widget_base(wbase_raw,/Col,/Frame)
  wtmp = Widget_Tab(wtmp0,UVal='TAB_RAW')
  wrawCcd = widget_base(wtmp,title='CCD File',/Col,/Frame)
  wrawSeq = widget_base(wtmp,title='Sequential File',/Col,/Frame)
  ; 
  ; Sequential file
  ;
  wTmpBase1 = Widget_Base(wrawSeq,/Row)
    wtmp = Widget_Label(wTmpBase1,Value= $
        '** Loading and Visualizing RAW data **')
    wTmp = widget_button(wTmpBase1,Value='Load Example',$
     UValue='LOAD_RAW_SEQ_EXAMPLE')
  ;wTmpBase1 = Widget_Base(wrawSeq,/Row)
    wRawSeqType = cw_droplist(wTmpBase1,Value=['0','Ascii','Binary'],$
	Title='Type of file: ',UValue='')
  ;wtmp = widget_label(wrawSeq,Value='Raw data file: ',/Align_Left)
  wRawSeqFile = cw_pickfile(wrawSeq,Title=' ',Value='<none>',$
	UValue='LOAD_RAW_SEQ',XSize=60)


  wRawSeqText = widget_text(wrawSeq,Value='<none>',XSIZE=20,YSIZE=20,/Scroll)
  wtmp0 = widget_base(wrawSeq,/Row)
    wcol1 = cw_field(wtmp0,Title='Col. with X: ',Value='1',UValue='COL',$
        /Integer,XSize=2)
    wcol2 = cw_field(wtmp0,Title='Col. with Mu: ',Value='2',UValue='COL',$
        /Integer,XSize=2)
    wcol3 = cw_field(wtmp0,Title='Col. with Index: ',Value='3',UValue='COL',$
        /Integer,XSize=2)
  wtmp0 = Widget_Base(wrawSeq,/Row)
    wtmp = Widget_Label(wtmp0,Value='Plot:')
    wtmp = Widget_Button(wtmp0,Value='EXODUS...',UValue='RAW_SEQUENTIAL_EXODUS')
    wtmp = Widget_Button(wtmp0,Value='3 D Mesh',UValue='RAW_SEQUENTIAL_MESH')
    wtmp = Widget_Button(wtmp0,Value='Export Delia Bin',UValue='RAW_SEQUENTIAL_BIN')

  wtmp0 = widget_base(wrawSeq,/Row)
    wview = cw_field(wtmp0,Title='Plot spectrum with index: ',Value='0',UValue=$
	'RAW_VIEW#',/Float,XSize=8,/Return_event)
    wtmp = widget_button(wtmp0,Value='Go',UValue='RAW_VIEW#')
    wtmp = widget_button(wtmp0,Value='|<',UValue='RAW_VIEW#')
    wtmp = widget_button(wtmp0,Value='<',UValue='RAW_VIEW#')
    wtmp = widget_button(wtmp0,Value='>',UValue='RAW_VIEW#')
    wtmp = widget_button(wtmp0,Value='>|',UValue='RAW_VIEW#')
    wtmp = widget_button(wtmp0,Value='Reset',UValue='RAW_VIEW#')
  ; 
  ; CCD file
  ;

  wTmpBase1 = Widget_Base(wRawCcd,/Row)
    wtmp = Widget_Label(wTmpBase1,Value= $
        '** Loading and Visualizing RAW data **')
    wTmp = widget_button(wTmpBase1,Value='Load Example',$
     UValue='LOAD_RAW_CCD_EXAMPLE')
  ;wTmpBase1 = Widget_Base(wrawCcd,/Row)
    wRawCcdType = cw_droplist(wTmpBase1,Value=['1','Ascii','EDF'],$
	Title='Type of file: ',UValue='')
  ;wtmp = widget_label(wrawCcd,Value='CCD data file: ',/Align_Left)
  ;wtmp = widget_label(wRawCcd,Value='CCD data file: ',/Align_Left)
  wRawCcdFile = cw_pickfile(wRawCcd,Title=' ',Value='<none>',$
	UValue='LOAD_RAW_CCD',XSize=60)


  wtmp=Widget_Label(wRawCcd,Value='Map of spectra. H:pixels, V:spectra') 
  wtmp=Widget_Label(wRawCcd,Value='single-click for Xplot view. Double-click for Exodus')
  wRawDraw=Widget_Draw(wRawCcd,XSize=500,YSize=500, $
	/MOTION_EVENTS,UVALUE='RAW_CURSOR',RETAIN=2, Button_Events=1)
  tmp = Widget_Base(wRawCcd,/Row)
    wtmp=Widget_Button(tmp,Value='Refresh',UValue='RAW_DISPLAY')
    wRawCursor=Widget_Label(tmp,Value='**********************************************************')
  tmp = Widget_Base(wRawCcd,/Row)
    wRotate= cw_droplist(tmp,Value=['0','Select:',$
        'Rotate 90 Clockwise', $
        'Rotate 90 CounterClockwise', $
        'Rotate 180', $
        'Mirror up/down',$
        'Mirror left/right'],$
         Title='Modify image: ',UValue='RAW_ROTATE')
  tmp = Widget_Base(wRawCcd,/Row)
    wtmp=Widget_Button(tmp,Value='Xsurface1',UValue='RAW_DISPLAY')
    wtmp=Widget_Button(tmp,Value='iSurface',UValue='RAW_DISPLAY')
    wtmp=Widget_Button(tmp,Value='iContour',UValue='RAW_DISPLAY')
    wtmp=Widget_Button(tmp,Value='iImage',UValue='RAW_DISPLAY')
    wtmp=Widget_Button(tmp,Value='Xplot2d',UValue='RAW_DISPLAY')
    wtmp=Widget_Button(tmp,Value='Export EDF',UValue='RAW_DISPLAY')
    ;wtmp=Widget_Button(tmp,Value='Xplot2D',UValue='RAW_DISPLAY')
	
;
; Mu data panel
;
wbase_interp = widget_base(wbase_mu,/Col,/Frame)

  wtmp0 = widget_base(wbase_interp,/Row)
    wtmp = widget_label(wtmp0,Value=' Interpolation: ',/Align_Left)
    wInterp = cw_droplist(wtmp0,value=['0','No','Yes'],UValue='MAPPANELS')

  wBase_interp2 = widget_base(wbase_interp,/Col)

    wtmp0 = widget_base(wbase_interp2,/Col)
      wtmp = widget_label(wtmp0,/Align_Left,Value=$
	'X extrema in the intervals : ')
      wlabelInt1 = widget_label(wtmp0,/Align_Left,Value=$
	'  from (minima) [0,0] ',/Dynamic_Resize)
      wlabelInt2 = widget_label(wtmp0,/Align_Left,Value=$
	'  to (maxima) [0,0]',/Dynamic_Resize)
      wtmp1 = widget_base(wtmp0,/Row)
            wIntMin = cw_field(wtmp1,Title='Min X:  ',$
	        Value='0',/Float,XSize=8)
            wIntMax = cw_field(wtmp1,Title='Max X:  ',$
        	Value='0',/Float,XSize=8)
            wIntN = cw_field(wtmp1,Title='Number of points:  ',$
	        Value='512',/Integer,XSize=4)

wbase_cal = widget_base(wbase_mu,/Col,/Frame)
  wtmp = widget_label(wbase_cal,/Align_Left,Value=$
     'Calibration:  Xnew[i] = a + b*(X[i]-X[0]) + c*(X[i]-X[0])^2 ')
  wtmp0 = widget_base(wbase_cal,/Row)
    wcal_a = cw_field(wtmp0,Title='a: ',Value='0',/Float,XSize=8)
    wcal_b = cw_field(wtmp0,Title='b: ',Value='1',/Float,XSize=8)
    wcal_c = cw_field(wtmp0,Title='c: ',Value='0',/Float,XSize=8)
    wcal_x0 = cw_field(wtmp0,Title='x0: ',Value='0',/Float,XSize=8)
    wtmp = widget_button(wtmp0,Value='Get...',UValue='MU_GET')
 

wbase_res = widget_base(wbase_mu,/Col,/Frame)
  wtmp0 = widget_base(wbase_res,/Row)
    wtmp = widget_button(wtmp0,Value='** RUN interpolate & calibrate **',$
      UValue='MU_GO')
    wtmp = widget_button(wtmp0,Value='Load file with Mu(E)...', $
      UValue='MU_LOAD')
  wtmp = widget_label(wbase_res,Value='Result: ',/Align_Left)
  wTextMu = widget_text(wbase_res,Value='<none>',XSIZE=60,YSIZE=10,/Scroll)
  wtmp0 = widget_base(wbase_res,/Row)
    wtmp = widget_label(wtmp0,/Align_Left,Value=$
	'Plot results: ')
    wtmp = widget_button(wtmp0,Value='2 D',UValue='MU_PLOT')
    wtmp = widget_button(wtmp0,Value='3 D',UValue='MU_PLOT')
    wtmp = widget_button(wtmp0,Value='EXODUS...',UValue='MU_PLOT')
    wtmp = widget_button(wtmp0,Value='Xsurface1',UValue='MU_PLOT')
    wtmp = widget_button(wtmp0,Value='iSurface',UValue='MU_PLOT')
    wtmp = widget_button(wtmp0,Value='iContour',UValue='MU_PLOT')
    wtmp = widget_button(wtmp0,Value='iImage',UValue='MU_PLOT')
    wtmp = widget_button(wtmp0,Value='Xplot2d',UValue='MU_PLOT')
  wtmp0 = widget_base(wbase_res,/Row)
    wtmp = widget_button(wtmp0,Value='Compute Eo shifts', $
	UValue='MU_PLOT')
    wtmp = widget_button(wtmp0,Value='XANES analysis with FuFiFa...', $
	UValue='MU_PLOT')
    wtmp = widget_button(wtmp0,Value='Export calibrated data...', $
	UValue='MU_PLOT')

;
; Chi data panel
;
strMu2Chi = xaid_defaults('xaid_mu2chi')

wtmp0 = widget_base(wbase_chi,/Row,/Frame)
  wMu2Chi = cw_field(wtmp0,$
   Title='Prepare extraction using spectrum with index: ',$
   Value='0',/Float,XSize=12,UValue='CHI_MU2CHI',/Return_Events)
  wtmp = widget_button(wtmp0,Value='Go',UValue='CHI_MU2CHI')
  wtmp = widget_button(wtmp0,Value='|<',UValue='CHI_MU2CHI')
  wtmp = widget_button(wtmp0,Value='<',UValue='CHI_MU2CHI')
  wtmp = widget_button(wtmp0,Value='>',UValue='CHI_MU2CHI')
  wtmp = widget_button(wtmp0,Value='>|',UValue='CHI_MU2CHI')
wtmp0 = widget_base(wbase_chi,/Col,/Frame)
  wtmp = widget_label(wtmp0,/Align_Left,$
    Value='Parameters to calculate EXAFS signal')
  wtmp = widget_label(wtmp0,/Align_Left,$
    Value=' ')
  wtmp = widget_label(wtmp0,/Align_Left,Value=$
      'PreEdge Linear Fit (Does not apply if Min=Max): ')

  wtmp1 = widget_base(wtmp0,/Row)
      wPreMin = cw_field(wtmp1,Title=$
      ' E Min:',Value=strMu2Chi.preMin,/Float,XSize=10)
      wPreMax = cw_field(wtmp1,Title='E Max:',Value=strMu2Chi.preMax, $
	/Float,XSize=10)
  wtmp1 = widget_base(wtmp0,/Row)
    wE0 = cw_field(wtmp1,Title='Eo:',Value=strMu2Chi.e0,/Float,XSize=10,$
	UValue='')
    wJump = cw_field(wtmp1,Title='Jump: ',Value=strMu2Chi.jump,/Float,XSize=10)

  wtmp1 = widget_base(wtmp0,/Row)
    wNPol = cw_field(wtmp1,Title='PostEdge: N polynomials: ', $
	Value=strmu2chi.npol, /Int, XSize=10,/Return_Events)
    wtmp = widget_label(wtmp1,/Align_Left,Value='Degrees:')
    wDegree = widget_text(wtmp1,Value=strmu2chi.degree,XSize=15,/Edit)
  wtmp1 = widget_base(wtmp0,/Row)
    wtmp = widget_label(wtmp1,/Align_Left,Value='PostEdge: Knots (in k):')
    wknots = widget_text(wtmp1,Value=strmu2chi.knots,XSize=50,/Edit)
  wExtraction = CW_DROPLIST(wtmp0,Title='Extraction type: ', $
        VALUE=strMu2Chi.extraction,UVALUE='Plot')


wtmp0 = widget_base(wbase_chi,/Col,/Frame)
  wtmp1 = widget_base(wtmp0,/Row)
    wKKMin = cw_field(wtmp1,Title='Plot chi in the interval: k min: ',$
  	Value=strMu2Chi.kkmin,/Float,XSize=10)
    wKKMax = cw_field(wtmp1,Title='k max: ',Value=strMu2Chi.kkmax,/Float,XSize=10)

  wtmp1 = widget_base(wtmp0,/Row)
    wtmp = widget_button(wtmp1,$
      Value='** Calculate extraction OVER ALL spectra **',UValue='CHI_GO')
    wtmp = widget_button(wtmp1,$
      Value='Recalculate some spectra...',UValue='CHI_GO')
  wTextChi = widget_text(wtmp0,Value='<none>',XSIZE=60,YSIZE=10,/Scroll)
  wtmp1 = widget_base(wtmp0,/Row)
    wtmp = widget_label(wtmp1,/Align_Left,Value='Plot results: ')
    wtmp = widget_button(wtmp1,Value='2 D',UValue='CHI_PLOT')
    wtmp = widget_button(wtmp1,Value='2 D (splines)',UValue='CHI_PLOT')
    wtmp = widget_button(wtmp1,Value='3 D',UValue='CHI_PLOT')
    wtmp = widget_button(wtmp1,Value='3 D (splines)',UValue='CHI_PLOT')
  wtmp1 = widget_base(wtmp0,/Row)
    wtmp = widget_button(wtmp1,Value='Xsurface1',UValue='CHI_PLOT')
    wtmp = widget_button(wtmp1,Value='iSurface',UValue='CHI_PLOT')
    wtmp = widget_button(wtmp1,Value='iContour',UValue='CHI_PLOT')
    wtmp = widget_button(wtmp1,Value='iImage',UValue='CHI_PLOT')
    wtmp = widget_button(wtmp1,Value='Export chi data...',UValue='CHI_PLOT')
;
; FT data panel
;

wtmp0 = widget_base(wbase_ft,/Row,/Frame)
  wChi2FT = cw_field(wtmp0,Title=$
   'Prepare FT & BFT using spectrum with index: ',Value='0',$
   /Int,XSize=4,UValue='FT_CHI2FT',/Return_Events)
  wtmp = widget_button(wtmp0,Value='Go',UValue='FT_CHI2FT')
  wtmp = widget_button(wtmp0,Value='|<',UValue='FT_CHI2FT')
  wtmp = widget_button(wtmp0,Value='<',UValue='FT_CHI2FT')
  wtmp = widget_button(wtmp0,Value='>',UValue='FT_CHI2FT')
  wtmp = widget_button(wtmp0,Value='>|',UValue='FT_CHI2FT')

wtmp0 = widget_base(wbase_ft,/Col,/Frame)
  wtmp = widget_label(wtmp0,/Align_Left,Value=$
    'Parameters to calculate Fourier tramsform')

  wtmp1 = widget_base(wtmp0,/Row)
    wKmin = cw_field(wtmp1,Title=$
      'Interval in K (Full interval if Min=Max):  K Min:',Value='0',/Float,$
      XSize=4)
    wKmax = cw_field(wtmp1,Title='K Max:',Value='0',/Float,XSize=4)
  wtmp1 = widget_base(wtmp0,/Row)
    wRmin = cw_field(wtmp1,  Title='Interval in R:   Rmin',vALUE='0.0',$
      /Float,XSize=4)
    wRmax = cw_field(wtmp1,Title='Rmax:',Value='8.0',/Float,XSize=4)

  wtmp1 = widget_base(wtmp0,/Row)
    windowlist = 1 & window_ftr,names=windowlist
    wFilter = cw_droplist(wtmp1, $
	VALUE=['0',windowlist], $
	TITLE='Window: ',UValue='FT_FILTERDROP' )
    wapp_base = widget_base(wtmp1,/Row) ; to allow mapping
    wapp_label = widget_label(wapp_base,Value='Appodization width:')
    wapp_value = cw_field(wapp_base,Title=' ',Value='0.2',/Float,XSize=4)

  wKpower = cw_droplist(wtmp0, VALUE=['0','0','1','2','3','4'], $
	TITLE='Weight data with K Power: ')


;;;;;;;;;;;
wtmp0 = widget_base(wbase_ft,/Col,/Frame)
  wtmp = widget_label(wtmp0,/Align_Left,Value=$
    'Parameters to calculate BACK Fourier tramsform')
  wtmp1 = widget_base(wtmp0,/Row)
    wR2min = cw_field(wtmp1,Title=$
      'R interval: R Min:',Value='0',/Float,XSize=4)
    wR2max = cw_field(wtmp1,Title='R Max:',Value='8',/Float,XSize=4)
  wtmp1 = widget_base(wtmp0,/Row)
    wK2min = cw_field(wtmp1,Title=$
      'K interval:  K Min:',$
	Value='0',/Float,XSize=4)
    wK2max = cw_field(wtmp1,Title='K Max:',Value='0',/Float,XSize=4)
  wRShell = cw_field(wtmp0,Title='Shell radius for the back phase:',$
	Value='0',/Float,XSize=4)
;;;;;;;;;;;

   
wtmp = widget_button(wbase_ft,Value= $
	'**CALCULATE FT  & BFT OVER ALL SPECTRA**',UValue='FT_GO')
wtmp0 = widget_base(wbase_ft,/Row)
  wtmp = widget_label(wtmp0,Value='Plot results: ',/Align_Left)
  wtmp = widget_button(wtmp0,Value='Fourier (2D)',UValue='FT_PLOT')
  wtmp = widget_button(wtmp0,Value='Fourier (3D)',UValue='FT_PLOT')
  wtmp = widget_button(wtmp0,Value='Back F  (2D)',UValue='FT_PLOT')
  wtmp = widget_button(wtmp0,Value='Back F  (3D)',UValue='FT_PLOT')
wtmp0 = widget_base(wbase_ft,/Row)
  wtmp = widget_button(wtmp0,Value='Xsurface1',UValue='FT_PLOT')
  wtmp = widget_button(wtmp0,Value='iSurface',UValue='FT_PLOT')
  wtmp = widget_button(wtmp0,Value='iContour',UValue='FT_PLOT')
  wtmp = widget_button(wtmp0,Value='iImage',UValue='FT_PLOT')
  wtmp = widget_button(wtmp0,Value='Export |FT| data...',UValue='FT_PLOT')
;
;
;

; pointer array: 
;   index 0 : raw data
;   index 1 : mu (structure)
;   index 2 : chi (structure)
;   index 3 : fourier transform (matrix)
;   index 4 : back fourier transform (matrix)
ptr = PtrArr(5)


widsRaw = {file:wRawSeqFile, type:wRawSeqType, text:wRawSeqText,$
  view:wview, col1:wcol1, col2:wcol2, col3:wcol3, $
  ccdFile:wRawCcdFile, ccdtype:wRawCcdType, draw:wRawDraw, cursor:wRawCursor }
widsMu = { $    
  ;  interpolation
  interp:wInterp, base_interp2:wBase_Interp2, $
;  align:wAlign, eShifts:wEShifts, $ 
  labelInt1:wLabelInt1, labelInt2:wLabelInt2, intMin:wIntMin, $
  intmax:wIntMax, intN:wIntN, $
  ; calibration
  cal_a:wcal_a, cal_b:wcal_b, cal_c:wcal_c, cal_x0:wcal_x0, $
  ; results window
  text:wTextMu }
widsChi = {mu2chi:wMu2Chi, preMin:wPreMin, preMax:wPreMax, e0:wE0, $
  jump:wJump, nPol:wNPol, degree:wdegree, knots:wknots, $
  extraction:wExtraction, kkMin:wKKMin, kkmax:wKKMax, $
  text:wTextChi, parentMu2Chi:0L }
widsFT = {chi2ft:wChi2FT, kMin:wkMin, kMax:wkMax, rMin:wrMin, rMax:wrMax, $
  filter:wFilter, app_base:wApp_base, app_label:wapp_label, $
  app_value:wapp_value, kPower:wkPower, $
  k2Min:wK2Min,k2Max:wK2Max, rShell:wRShell, r2Min:wR2Min, r2Max:wR2Max,$
  parentFF:0L }

wids = {labelMain:wlabelMain,$
  panel_ids:[wbase_raw,wbase_mu,wbase_chi,wbase_ft], $
  xplot:0L,xplotCCD:0L,exodusCCD:0L,  $
  raw:widsRaw, mu:widsMu, chi:widsChi, ft:widsFT }

state = {wids:wids, ptr:ptr, no_block:no_block, raw_type:'Sequential'}

delia_mappanels,wids
;
;
widget_control,Widget_Info(wbase,/Child),set_uvalue=state ; ,/no_copy

widget_control,wbase,/realize

xmanager,'delia',wbase,GROUP=group,No_Block=no_block
end
