PRO waviness_calc,str,out,slopes=slopes,nowrite=nowrit, $
    valslopes=valslopes,dataslopes=dataslopes
;+
; NAME: 	WAVINESS_CALC
;
; PURPOSE:      This program generates a random error surface
;
;               (XSH_WAVINESS implements a Graphical Interface)
;
;               The main method for slope errors (called hereafter waviness) is 
;               described in: 
;                  M. Sanchez del Rio and A. Marcelli,
;                  NIM-A319 (1992) 170-177
;                  http://dx.doi.org/10.1016/0168-9002(92)90550-N
; 
;               An alternative method (called hereafter harmonic_sum) consists
;               in a simple sum of harmonics:
;                 z = \sum_{n=1}^{N} {b_n cos( 2 \pi l/L + \psi_n) }
;
;                 (see e.g., Eq 11 in Shi et al, J Synchrotron Rad. 21 (2014)
;                  http://dx.doi.org/10.1107/S160057751400650X )
;
;               The switching of one or another method is done via the
;               sign of "Estimated Slope Error RMS [arcsec]" in the 
;               input interface, or input_str.slp in the input structure,
;               Set >0 for waviness, <0 for harmonic_sum
;
; CATEGORY:     SHADOW tools
;	
; CALLING SEQUENCE:
;	waviness_calc,input_str,out
; INPUTS:
;	input_str: a structure with the inputs, as created by waviness_read():
;                  or defined in the interface xsh_waviness:
;                IDL> help,/str,waviness_read(/default)
;                ** Structure <11d0958>, 11 tags...
;                   FILE            STRING    'waviness.dat'
;                   NPOINTX         LONG               100
;                   NPOINTY         LONG               100
;                   WIDTH           FLOAT           20.1000
;                   XLENGTH         FLOAT           113.100
;                   N               LONG                60
;                   SLP             FLOAT          0.900000
;                   ISEED           LONG          97865651
;                   C               DOUBLE    Array[5001]
;                   Y               DOUBLE    Array[5001]
;                   G               DOUBLE    Array[5001]
;
;                For the default method (waviness):
;
;                   FILE: name of the output file
;                   NPOINTX number of points along mirror width
;                   NPOINTY number of points along mirror width
;                   XLENGTH is the mirror length (apologies for the X)
;                   WIDTH is the mirror width
;                   N is n_max in Eq.8 (see reference)
;                   SLP estimation of the target slope error RMS in arcsec
;                       for the default method (waviness). Must be >0
;                       For the other method (see later) it must be negative,
;                   ISEED seed to initializa random generator
;                   C an array of coeff C_n in Eq. 9. 
;                     Only the first N+1 indices are read.
;                   Y an array of coeff y_n^0 in Eq. 8 normalized.
;                     This is the initial constant shift. 
;                     One means the profile can be shifted to any position 
;                     along the mirror length. Zero means no initial shift.
;                   G an array of coeff g_n (random shift) in Eq. 8 normalized.
;                       
;
;                For the alternative method (harmonic_sum): 
;
;                   To use this method, SLP must be set negative. 
;                   In this method the profile is:
;                   z(y) = \sum_{n=1}^{N}
;                    { b_0 n^{g_n} cos( 2 \pi (0.5+(y/L) + r_n y_g ) }
;                   or in other words, a sum of N harmonics with a power
;                   law of exponent g_n.
;
;                   y is the coordinate along the mirror, with zero in the 
;                     mirror center.
;                   b_0 is a constant equal to abs(SLP) [labeled
;                      "Estimated Slope Error RMS [arcsec]" in the interface.
;                   g_n is the exponent of n stored in G for each harmonic
;                   L is the mirror length XLENGTH
;                   r_n is a random value in [0,1) for each harmonic
;                   y_g is a normalized interval to apply the random shift
;                     stored in Y. 0 means no random shift, 1 means shift in 
;                     the full mirror length.
;                   Note that in this method C[0],Y[0] and G[0] are not used
;                     as summation index starts with one.
; 
; OUTPUTS:
;       a file with the generated surface in SHADOW/PRESURFACE format
;
; OPTIONAL OUTPUTS:
;	out: a structure with the surface {x:x, y:y, z:z}
;
; KEYWORDS
;       nowrite: if set does not write file for SHADOW. Default: nowrite=0, 
;                thus a file with the name defined in the input structure
;                is written with the surface in the SHADOW-PRESURFACE format.
;	slopes = a flag to calculate the slop error values via slopes(out).
;                If set, the following keyword received the calculates data:
;       valslopes = an 4-dim array with slope errors RMS in 
;                   [X_arcsec,Y_arcsec,X_urad,Y_urad]
;       dataslopes = an (2,NX,NY) array with slope errors in rad along 
;                    X (in dataslopes[0,*,*]) and Y (in dataslopes[1,*,*])
;
; MODIFICATION HISTORY:
;			June 1991	first version
;			September 1991	update 
;			October 1991	moved to Sun
;			September 1992	distribution version
;			February 2006	translated to IDL (srio@esrf.fr)
;			2014-09-09 srio@esrf.eu dimensionated to 5000pts.
;                                added new method (SLP<0). Added doc.
;
;-
on_error,2
 
IF N_Elements(str) EQ 0 THEN str=waviness_read(/Default)
IF N_Elements(nowrite) EQ 0 THEN nowrite=0

method=0
if (str.slp LT 0) then begin
    method=1
    print,'WAVINESS_CALC: Using default method: waviness'
endif else begin
    print,'WAVINESS_CALC: Using method: harmonic sum'
endelse

t0=systime(1)

x=DblArr(str.npointx)
y=DblArr(str.npointy)
coe=DblArr(str.n+1)
shi=DblArr(str.n+1)
ranc=DblArr(str.n+1)
shin=DblArr(str.n+1)
zz=DblArr(str.npointx,str.npointy)
xx=DblArr(str.npointx)
yy=DblArr(str.npointy)

;c
;c  reads input
;c
	yfact = 1.D0
	slp = str.slp/3600D0*!dpi/180.D0   ; pass to rads

	xfact = str.xlength/!dpi
	i_plot = 0
	xntot = 0.0D0
	iseed1=str.iseed
	iseed2=str.iseed+11L
        tmp = r_myrand(initseed=[iseed1,iseed2],/OnlyInit)

if (method EQ 0) then begin
        FOR i=0L,str.n DO BEGIN
         a1=(str.c)[i]
         a2=(str.y)[i]
         a3=(str.g)[i]
	 xntot = xntot + a1

         ; eq 9 in ref.
	 amp = slp*str.xlength/(0.5D0+double(i))/sqrt(2.D0)/!dpi
	 coe(i) = a1*amp

	 shin(i) = a2
	 ranc(i) = a3 

        ENDFOR

	FOR i=0L,str.n DO BEGIN
	  coe(i) = coe(i)/sqrt(xntot)       ; for a single harmonic
        ENDFOR

        ;c
        ;c	begin calculations
        ;c
	stp = !dpi/(str.npointy-1)
	stpx = str.width/(str.npointx-1)
	stpy = str.xlength/(str.npointy-1)
	k=0D0
label10:
	xx(k)=-(str.width/2)+stpx*k
	FOR j=0L,str.n DO BEGIN
	  tmp = r_myrand(1.0D0)
	  shi(j)=shin(j)*!dpi+tmp*!dpi*ranc(j)
        ENDFOR

	y[*]=0.0D0

	FOR j=0L,str.n DO BEGIN
	    x= -!dpi/2 + DIndGen(str.npointy)*stp
	    nn=double(j)
	    ynew= coe[j]*( (-1D0)^nn*cos((2D0*nn+1D0)*(x+shi[j])))
	    y=ynew+y
        ENDFOR

	
	x=x*xfact
	yy = x
	y=y*yfact
	zz(k,*)=y
	k=k+1
	IF (k LT Double(str.npointx)) THEN GOTO,label10
endif else begin

	xx = makearray1(str.npointx,-str.width/2D0,str.width/2.0D0)
	yy = makearray1(str.npointy,-str.xlength/2D0,str.xlength/2.0D0)
	for k=0L,str.npointx-1 do begin
            zz1 = yy*0.0D0
	    FOR j=1L,str.n DO BEGIN
                arg = 2.0D0 * !dpi * j * (0.5d0 + yy/str.xlength)
                ran1 = 2.0D0 * !dpi * r_myrand(1.0D0) *str.y[j]
                ;zz1 = zz1 + 2.1D-10*str.c[j]*double(j)^(str.g[j])*cos(arg+ran1)
                zz1 = zz1 + str.c[j]*double(j)^(str.g[j])*cos(arg+ran1)
            ENDFOR
            zz[k,*]=zz1*abs(str.slp) ; global multiplicative factor str.slp
        endfor
endelse

	; 
	; define structure with result 
	;
	out={z:zz,x:xx,y:yy}

;c
;c	writes file for shadow
;c
	IF not(nowrite) THEN BEGIN
          openw,unit,str.file,/get_lun
	  printf,unit, str.npointx, str.npointy
	  printf,unit, yy
	  FOR i=0L,str.npointx-1 DO BEGIN
	    printf,unit,xx(i),zz(i,*)
          ENDFOR
	  Free_Lun,unit
	  Print,'File written to disk: ',str.file
	  print,'   Don`t forget to run PRESURFACE before SHADOW'
	  print,' '
	ENDIF

;c
;c  calculate the slope error rms of the whole surface
;c
	IF Keyword_Set(slopes) THEN BEGIN
	  dataslopes = slopes(out,slopesrms=valslopes)
	ENDIF

END ; waviness_calc
