function READ_PAR, filename
;+
; NAME:
;	READ_PAR
;
; PURPOSE:
;
; 	Read in (ASCII) files ".geasc" and ".gen" from the Laue suite CCP4.
;	Returns the parameters of the experiment as read in these files.
;
; CATEGORY:
;	File managing
;
; CALLING SEQUENCE:
;	Par = READ_PAR(filename) ;
;
; INPUTS:
;	filename : the path name of the file to process (without extension)
;	: BOTH the .geasc and the .gen files must be present. This is because
;	some parameters not present or possibly wrong in the .geasc are found
;	in the .gen file !
;
; OUTPUTS:
;	Par : the structure array containing the parameters
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None.
;
; PROCEDURE:
;	Parameters -->
;   CRYSTAL	FLTARR(6) : alpha,beta,gamma,a*,b*,c* in reciprocal space
;   SG_NO		  : Space group # 
;   SYSTEM_NO:		  : System #
;   CENTERING_NO:	  : Centering #
;   PHIS        FLTARR(3) : the crystal setting angles [degrees]
;   SPINDLE	FLOAT  	  : the spindle value
;   CDD         FLOAT     : crystal to detector distance [mm]
;   RASTER      FLOAT     : pixel size [um]
;   RASTX       INT       : number of pixel in X dimension
;   RASTY       INT       : number of pixel in Y dimension
;   CENX        FLOAT     : X position of beam center [pixels] from lower left of image
;   CENY        FLOAT     : Y position of beam center [pixels] from lower left of image
;   RADIUS      FLOAT     : maximum radius from beam center [mm] used for prdicted spots
;   LMIN        FLOAT     : minimum wavelength [A]
;   LMAX        FLOAT     : maximum wavelength [A]
;   DMIN        FLOAT     : resolution limit [A] used for predicted spots
;   YSCALE      FLOAT     : Y scale for pixel non squareness
;   TWIST       FLOAT     : correction for twist [0.01 degrees]
;   TILT        FLOAT     : correction for tilt [0.01 degrees]
;   BULGE       FLOAT     : correction for bulge [0.01 degrees]
;   R_REJ       FLOAT     : minimum radius from beam center [mm] used for prdicted spots
;   GAIN        FLOAT     : gain of the detector at 1.5418 A
;   BG_GAIN     FLOAT     : gain of the detector at the mean background wavelength
;   BEAMSIZE    FLOAT     : beamsize [mm]
;   SAMPLE_DEPTH    FLOAT : sample depth [mm]
;   DARK_CURRENT    FLOAT : dark current of the detector [counts]
;   BOXSIZE     STRUCTURE : size of box for data integration [pixel]
;   NOISE_PERCENTAG INT   : initial percentage of the pixels taken into account for 
;			    background calculation.
;   SATVALUE	LONG	  : saturation value
;   OVLP_DIST   FLOAT     : distance [mm] between spot centers, below which spots are considered overlapped
;   OVLP_MIN_DIST   FLOAT     : distance [mm] between spot centers, below which spots are considered impossible
;				to deconvolute
;      Remark --> The last 8 parameters are entered later with the progrm WRITEPAR_LAUE.PRO
;		  They are used for IDL data integration. 
;
;
; MODIFICATION HISTORY:
;	D.Bourgeois, June 1995.
;-

;on_error,2                              ;Return to caller if an error occurs



parameters = {	crystal : {crys, angle : FLTARR(3), axis : FLTARR(3)}, $
		sg_no: 0, $
		system_no: 0, $
		centering_no: 0, $
	  	phis : {phi, x : 0.0d, y : 0.0d, z : 0.0d}, $
		spindle : 0.0d, $
		cdd : 0.0d, $
		raster : 0.0d, $
		rastx : 0, $
		rasty : 0, $
		cenx : 0.0d, $
		ceny : 0.0d, $
		radius : 0.0d, $
		lmin : 0.0d, $
		lmax : 0.0d, $
		dmin : 0.0d, $
		yscale : 0.0d, $
		twist : 0.0d, $
		tilt : 0.0d, $
		bulge : 0.0d, $
		r_rej : 0.0d, $
		gain : 0.0d, $
		bg_gain : 0.0d, $
		beamsize : 0.0d, $
		sample_depth : 0.0d, $
		dark_current : 0.0d, $
		noise_percentage : 0, $
		satvalue : 0L, $
		ovlp_dist : 0.0d, $
		ovlp_min_dist : 0.0d, $
		boxsize : {x:0,y:0} }


;initialize parameters
dummy=0.0
axis = FLTARR(3)
angle = FLTARR(3)
phis = FLTARR(3)
lmin=0.0 & lmax=0.0 
xc_offset=0.0 & yc_offset=0.0 & centerx=0.0 & centery=0.0
cdd=0.0 & radius=0.0 & dmin=0.0
twist=0.0 & tilt=0.0 & bulge=0.0
yscale=0.0
r_rej=0.0
rastx=0 & rasty=0
raster=0.0

;open the 'gen' file 
OPENR,1,filename+'.gen'
PRINT,'Reading .gen file ...'

READF,1,FORMAT='(10(/),F12.5)',r_rej
READF,1,FORMAT='(/)'
READF,1,FORMAT='(/)'
READF,1,rastx,rasty,dummy,dummy,dummy,raster

CLOSE,1

OPENR,1,filename+'.geasc'
PRINT,'Reading .geasc file ...'

;Read the header :
centerx = FLTARR(6)
centery = FLTARR(6)
omega = FLTARR(6)
cell = FLTARR(6)
phi = FLTARR(3)
l = FLTARR(3)
dmin = 0.0
radius = 0.0
spindle = 0.0
xc_shift = FLTARR(6)
yc_shift = FLTARR(6)
wc_shift = FLTARR(6)
delta = 0.0
twist = FLTARR(6)
tilt = FLTARR(6)
bulge = FLTARR(6)
cdd = 0.0
yscale = FLTARR(6)
refl_number = 0L

READF,1,FORMAT='(3(/),TR5,6(F12.4))',centerx
READF,1,FORMAT='(TR5,6(F12.4))',centery
READF,1,FORMAT='(TR5,6(F12.4))',omega
READF,1,FORMAT='(TR5,6(F12.6))',cell
READF,1,FORMAT='(TR5,2(I4))',system_no, centering_no
READF,1,FORMAT='(TR5,F12.4)',dummy ;this is the raster, but it is wrong !!
READF,1,FORMAT='(/,TR5,3(F10.3))',phi
READF,1,FORMAT='(TR5,3(F10.4))',l
READF,1,FORMAT='(TR5,F10.4)',dmin
READF,1,FORMAT='(TR5,F10.4)',radius
READF,1,FORMAT='(TR5,F10.4)',spindle
READF,1,FORMAT='(TR5,6(F12.5))',xc_shift
READF,1,FORMAT='(TR5,6(F12.5))',yc_shift
READF,1,FORMAT='(TR5,6(F12.5))',wc_shift
READF,1,FORMAT='(TR5,F12.4)',delta
READF,1,FORMAT='(TR5,6(F12.5))',twist
READF,1,FORMAT='(TR5,6(F12.5))',tilt
READF,1,FORMAT='(TR5,6(F12.5))',bulge
READF,1,FORMAT='(TR5,F12.5)',cdd
READF,1,FORMAT='(TR5,6(F12.5))',yscale

parameters.crystal.axis=cell(0:2)
parameters.crystal.angle=cell(3:5)
parameters.system_no= system_no
parameters.centering_no= centering_no
parameters.phis.x=phi(0)
parameters.phis.y=phi(1)
parameters.phis.z=phi(2)
parameters.spindle=spindle
parameters.lmin=l(1)
parameters.lmax=l(2)
parameters.dmin=dmin
parameters.cdd=cdd
parameters.radius=radius
parameters.cenx = 1000*(centerx(0) + xc_shift(0))/raster
parameters.ceny = 1000*(centery(0) + yc_shift(0))/raster*yscale(0)
parameters.twist=twist(0)
parameters.tilt=tilt(0)
parameters.bulge=bulge(0)
parameters.yscale=yscale(0)
parameters.r_rej=r_rej ;this parameter not present in the geasc file
parameters.rastx=rastx ;this parameter not present in the geasc file
parameters.rasty=rasty ;this parameter not present in the geasc file
parameters.raster=raster ;this parameter wrong in the geasc file


READF,1,FORMAT='(16(/),TR17,A5)',refl_number
PRINT,FORMAT='("Number of reflections predicted by Lauegen :",I0)',refl_number

;close the file
CLOSE,1
RETURN,parameters
END

