function CALC_ROTATION, par, rot_axis, spindle, mat
;+
; NAME:
;	CALC_ROTATION
;
; PURPOSE:
;	Given the rotation angles in PAR, the determined rotation axis 
;	and the rotation around it CALC_ROTATION determines the new 
;	rotation matrix.
;	From this the new angles in PAR are determined and PAR is 
;	returned with the new angles (with the spindle tag set to
;	zero).
;	Use FIND_ROTAXIS to determine the rotation axis from two
;	determined orientations.
;
; CATEGORY:
;	Laue data processing, Crystallography
;
; CALLING SEQUENCE:
; 	new_par= CALC_ROTATION(par, rot_axis, spindle [, mat])
;
; INPUTS:
;	PAR: As from READ_GEASC. Note that the SPINDLE tag
;		of the structure is added to PHIZ before the 
;		rotation around ROT_AXIS is done. 
;	ROT_AXIS: As output from FIND_ROTAXIS.
;		(The rotation axis in the form of its cosine values
;		along the three axes.)
;	SPINDLE: The positive rotation around this axis.
;
; KEYWORDED PARAMETERS:
;	MAT: Optional output. The rotation matrix as determined by the new
;		PhiX, PhiY and PhiZ. Note this is not the matrix M:
;		  x= M # [h,k,l]
;		To obtain M:
;		  M= CALC_ROTMAT(new_par)
;
; OUTPUTS:
;	Returns PAR with PhiX, PhiY and PhiZ corresponding to the
;	new orientation and SPINDLE set to zero.
;
; COMMON BLOCKS:
;	Common with CALC_ROTATION2:
;	COMMON calc_rot_par, cry, crz, crmat01
;
; SIDE EFFECTS:
;
; RESTRICTIONS:
;
; PROCEDURE:
;	Rotation matrix from Giacovazzo, p71.
;	
; MODIFICATION HISTORY:
;	Written by Thomas Ursby, September 1995
;
;-

; Common variables with CALC_ROTATION2 in the rare case of numerical
; solution being demanded for PhiX:
COMMON calc_rot_par, cry, crz, crmat01

; Reset par.spindle to zero:
npar= par
npar.phis.z= npar.phis.z + npar.spindle
npar.spindle= 0.0

; The rotation in PAR:
; (Note that we are interested in the rotation matrix defined by PhiX,
;  PhiY and PhiZ and not the matrix relating hkl to the rotated position
;  of the hkl-vector, see CALC_ROTMAT.)
dummy= CALC_ROTMAT(npar,mat1)

; The rotation around ROT_AXIS:
rot_axis=double(rot_axis)
l1= rot_axis(0) & l2= rot_axis(1) & l3= rot_axis(2)
sc= sin(double(spindle)*!dpi/180.0)
cc= cos(double(spindle)*!dpi/180.0)
; Giacovazzo p71:
m1= [[1,0,0],[0,1,0],[0,0,1]] 
m2= [[0,-l3,l2],[l3,0,-l1],[-l2,l1,0]] 
m3= [[l1^2,l1*l2,l1*l3],[l1*l2,l2^2,l2*l3],[l1*l3,l2*l3,l3^2]] 
mat2= TRANSPOSE(cc*m1 + sc*m2 + (1-cc)*m3)
; Me:
;p= (2*(l2 ge 0) - 1)*sqrt(1-l3^2)
;m1=[[1,0,0],[0,l3,-p],[0,p,l3]]
;p= sqrt(1-l1^2)
;m2=[[l1,0,p],[0,1,0],[-p,0,l1]]
;m3=[[1,0,0],[0,cc,-sc],[0,sc,cc]]
;m1= TRANSPOSE(m1) & m2= TRANSPOSE(m2) & m3= TRANSPOSE(m3)
;mat2= INVERT(m1,/DOUBLE) # INVERT(m2,/DOUBLE) # m3 # m2 # m1

; The new rotation matrix:
mat= mat2 # mat1

; Determine the new angles PhiX, PhiY and PhiZ (Spindle is now zero):
; Algorithm taken from the Mathematica routine LSS_ROT:
phiy= -asin(mat(2,0))
phiz=  (2*(mat(1,0) ge 0.0)-1) * acos(mat(0,0)/cos(phiy)) 
IF (abs(mat(2,1)) lt 0.000001) THEN BEGIN
  ;Numerical solution...
  cry= phiy & crz= phiz & crmat01= mat(0,1)
  x= [0.0]
  phix= (NEWTON(x,'CALC_ROTATION2'))(0)
  WHILE (phix lt -1.0/2*!dpi) DO phix= phix+2*!dpi
  WHILE (phix gt  3.0/2*!dpi) DO phix= phix-2*!dpi
ENDIF ELSE BEGIN
  phix= asin(mat(2,1)/cos(phiy))
  temp= TOTAL( CROSSP([0,0,1], mat(*,0)) * (-mat(*,1)))
  phix= ((temp lt 0.0)+1)/2 * phix + $
	((temp ge 0.0)+1)/2 * (!dpi-phix)
ENDELSE

npar.phis.x= 180.0/!dpi*phix
npar.phis.y= 180.0/!dpi*phiy
npar.phis.z= 180.0/!dpi*phiz

RETURN, npar

END
