function LOC_REFINE2,struc,image,par,image2, show=show
;+
; NAME:
;	LOC_REFINE2
;
; PURPOSE:
;	Matches predictions with observations in order to detect saturated
;	spots and to get a rough idea of peak maximum and mean background.
;
; CATEGORY:
;	Data processing
;
; CALLING SEQUENCE:
;	matched = LOC_REFINE2(struc,image,par,image2,/SHOW)
;
; INPUTS:
;	struc : the reflection list (a structure produced with the
;	        program REFLIST to read the content of the CCP4 file
;		"*.geasc". 
;	image : the original data image (usually part of the full data)
;
;	par : the parameters (from reflist.pro)
;	show : keyword set if the surfaces are to be displayed
;
; OUTPUTS:
;	matched : the reflection list matching the real data
;	image2 : optionnal output ; contains an image of the background.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None
; PROCEDURE:
;	See above
;
; MODIFICATION HISTORY:
;	D.Bourgeois, June 1997.
;-

;PARAMETERS TO CHANGE SEE THE SIGN " <---- HERE"

bctime=0
tottime=0

;a spot is saturated above
satvalue = par.satvalue
PRINT,'Saturation limit :',satvalue

;position of the beam center on the original Laue image [pixels] 
centerX = par.cenx-1
centerY = par.ceny-1

;pixel size [mm]
; scale_y is taken from LAUEGEN
  scaleY = par.yscale
;rebinning factor (if a compressed version of the image is used)
  reb = 1; <---- HERE
;original pixel size [micrometers]
  pix_size = par.raster
pix_sizeX = 0.001*reb*pix_size
pix_sizeY = 0.001*reb*pix_size/scaleY

;this is the size of the box to look in to search for a spot arount the
;click point
smallbox_size  = 1 ; <---- HERE
;PRINT,'Box size to look around for peaks [pixels] :',smallbox_size

;box for background
bigbox_size  = {box, x:FIX(par.boxsize.x/2.), y:FIX(par.boxsize.y/2.)}
maxpos = COMPLEX(bigbox_size.x,bigbox_size.y)

;size of "image"
imageX = (size(image))(1)
imageY = (size(image))(2)

;position of the studied area relative to the center of the beam [mm]
minX = pix_sizeX*(-1*centerX+bigbox_size.x+2)
minY = pix_sizeY*(-1*centerY+bigbox_size.y+2)
maxX = pix_sizeX*(-1*centerX+imageX-1-bigbox_size.x-2)
maxY = pix_sizeY*(-1*centerY+imageY-1-bigbox_size.y-2)

IF KEYWORD_SET(show) THEN BEGIN
;create a window containing the simulated data and a small for 3D surface
;representation
;save the window numbers
main_window = !D.WINDOW
WIN,512,512,/FREE
surf_window = !D.WINDOW
ENDIF

;create a reduced structure array which contains only the spots of interest
;  The criteria of selection is that the position of the spot be inside
;  the box defined by minX, maxX, minY and maxY
spots = struc(WHERE((struc(*).x GE minX) AND $
		    (struc(*).x LE maxX) AND $
		    (struc(*).y GE minY) AND $
		    (struc(*).y LE maxY)))

PRINT,'Number of input spots : ',(SIZE(struc))(1)
PRINT,'Number of spots selected : ',(SIZE(spots))(1)

;sort the reflections in "spots" according to resolution sphere
PRINT,'Sorting the spots according to resolution'
spots = spots(REVERSE(SORT(spots.res)))

;define the potentially matched spots
matched = spots
j = 0L

;define the number of saturated spots
number_sat = 0

;define a waiting list ; we allow for 75% of waiting spots
number_wait = LONG((SIZE(spots))(1)*0.75)
k = -1
waitspots = REPLICATE(spots(0),number_wait)
waitpar = REPLICATE({maxval0:0.,maxval1:0.,bc_coeff:FLTARR(3),maxpos : FLTARR(2), bc_mean : 0.},number_wait)

;make a backup of the original image
image2 = image

;Go through all reflections of the structure spots
PRINT,'Matching spots ...'
cr = STRING("15b)

;tottime=SYSTIME(1)
FOR i=LONG(0),(SIZE(spots))(1)-1 DO BEGIN

;m0=-31
;m1=24
;m2=26
;IF ((spots(i).m(0) EQ m0) AND (spots(i).m(1) EQ m1) AND (spots(i).m(2) EQ m2)) THEN show=1 ELSE show=0

IF i/100 EQ i/100.0 THEN PRINT,FORMAT="($,I5,' ie : ',F3.0,'%',A)",i,100.*i/FLOAT((SIZE(spots))(1)),cr
;IF i/100 EQ i/100.0 THEN PRINT,'Time for bc : ',bctime,'  Total time : ',SYSTIME(1)-tottime

 ;get the position in pixels
 ;We have to use FIX, not ROUND : here is the justification : each pixel contains the integrated signal
 ;within that pixel. For exemple if the predicted position is 2.8 then it means that pixel 2 contains
 ;the integrated signal between 2 and 3, which has a higher value that the integrated signal between 3 and 4.
 ;A perfectly centered profile should have a predicted position of 2.5

  xpos = FIX(spots(i).x/pix_sizeX+centerX)
  ypos = FIX(spots(i).y/pix_sizeY+centerY)

; the value of image at this position is :
orig_value = image(xpos,ypos)

;scan through a box around this position
smallbox = image(xpos-smallbox_size:xpos+smallbox_size,ypos-smallbox_size:ypos+smallbox_size)

;get the big box around the pixel
bigbox = image(xpos-FIX(par.boxsize.x/2.):xpos+FIX(par.boxsize.x/2.),ypos-FIX(par.boxsize.y/2.):ypos+FIX(par.boxsize.y/2.))

;bctime0=SYSTIME(1)

;let's start by defining the background
bc_mean =  BACK2D_QUICK(bigbox,par.noise_percentage,par.dark_current)
;bctime=bctime+SYSTIME(1)-bctime0

;maxpos_new will now point to the new maximum inside bigbox
maxpos_bbox = WHERESUB(WHERE(bigbox EQ MAX(smallbox)),bigbox)

IF (SIZE(maxpos_bbox))(1) GT 1 THEN BEGIN
;there are several possibilities for maxpos
;we take the one closest to the old maximum
closest=WHERE((FLOAT(maxpos-maxpos_bbox)^2+IMAGINARY(maxpos-maxpos_bbox)^2) EQ MIN(FLOAT(maxpos-maxpos_bbox)^2+IMAGINARY(maxpos-maxpos_bbox)^2))
maxpos_bbox=maxpos_bbox(closest(0))
ENDIF ELSE maxpos_bbox=maxpos_bbox(0)

;Is there a higher intensity around ?
searchbox=bigbox(FLOAT(maxpos_bbox)-1:FLOAT(maxpos_bbox)+1,IMAGINARY(maxpos_bbox)-1:IMAGINARY(maxpos_bbox)+1)

;where is the maximum of searchbox inside bigbox
maxpos_search_bbox = WHERESUB(WHERE(bigbox EQ MAX(searchbox)),bigbox)
IF (SIZE(maxpos_search_bbox))(1) GT 1 THEN BEGIN
;there are several possibilities for maxpos_search_bbox
;we take the one closest to the old maximum
closest=WHERE((FLOAT(maxpos-maxpos_search_bbox)^2+IMAGINARY(maxpos-maxpos_search_bbox)^2) EQ MIN(FLOAT(maxpos-maxpos_search_bbox)^2+IMAGINARY(maxpos-maxpos_search_bbox)^2))
maxpos_search_bbox=maxpos_search_bbox(closest(0))
ENDIF ELSE maxpos_search_bbox=maxpos_search_bbox(0)

maxpos_search_bbox=[FLOAT(maxpos_search_bbox),IMAGINARY(maxpos_search_bbox)]

;the position of maxpos_search_bbox in image2 is maxpos_search_bbox(0)+xpos-bigbox_size.x,
;and same for y
maxpos_s_im2 = [maxpos_search_bbox(0)+xpos-bigbox_size.x,maxpos_search_bbox(1)+ypos-bigbox_size.y]
 
;check where is the maximum 
 IF MAX(searchbox) GT MAX(smallbox) THEN BEGIN
   k = k + 1
   waitspots(k) =  spots(i)
   waitpar(k).bc_coeff = [0,0,bc_mean]
   waitpar(k).maxval0 = orig_value  ;original value not background subtracted
   waitpar(k).maxval1 = MAX(searchbox)  ;not background subtracted
   waitpar(k).bc_mean = bc_mean
   waitpar(k).maxpos = maxpos_s_im2

  ENDIF ELSE BEGIN ; it's OK put the spot on the matching list
 
 
 ;take out the found peak on image2
 image2(xpos-FIX(par.boxsize.x/2.):xpos+FIX(par.boxsize.x/2.),ypos-FIX(par.boxsize.y/2.):ypos+FIX(par.boxsize.y/2.)) = bc_mean
 max_value=MAX(smallbox) 
 matched(j)=spots(i)
 matched(j).bc_coeff =  [0,0,bc_mean]
 matched(j).intp(0) = max_value
 IF max_value GE satvalue THEN BEGIN
	;PRINT, 'Saturated spot at :',i
	matched(j).close = 1
	number_sat = number_sat + 1
 ENDIF
 j = j + 1

  ENDELSE ; OK put the spot on the matching list

IF KEYWORD_SET(show) THEN BEGIN
PRINT,''
PRINT,'Spot #',j - 1
PRINT,'Max(bigbox) (not background subtracted) :',MAX(bigbox)
PRINT,'Max(smallbox) (not background subtracted) :',MAX(smallbox)
IF MAX(searchbox) GT MAX(smallbox) THEN PRINT,'--> Waiting list #',k
SURFACE,bigbox
ok=''
READ,'bigbox OK ?',ok
SURFACE,smallbox
ok=''
READ,'small box OK ?',ok
SURFACE,searchbox
ok=''
READ,'searchbox OK ?',ok
ENDIF ;show

ENDFOR ; go through each spot


;How many spots on the waiting list ?
;PRINT,'Number of ambiguous spots :',k+1
;we examine which of these waiting spots must be selected
recovered = 0
FOR i=0L,k DO BEGIN ;go through each ambiguous spot

 matched(j) = waitspots(i)
 matched(j).bc_coeff=waitpar(i).bc_coeff

 ;is the spot not ambiguous any more ?
 IF image2(FIX((waitpar(i).maxpos)(0)),FIX((waitpar(i).maxpos)(1))) EQ  waitpar(i).maxval1 THEN BEGIN ; spot not ambiguous
 
 recovered = recovered + 1
  
 image2(FIX((waitpar(i).maxpos)(0))-bigbox_size.x:FIX((waitpar(i).maxpos)(0))+bigbox_size.x, FIX((waitpar(i).maxpos)(1))-bigbox_size.y:FIX((waitpar(i).maxpos)(1))+bigbox_size.y) = waitpar(i).bc_mean

 matched(j).intp(0) = waitpar(i).maxval1
 IF waitpar(i).maxval1 GE satvalue THEN BEGIN
  PRINT, 'Saturated spot at :',j
  matched(j).close = 1
  number_sat = number_sat + 1
 ENDIF ; saturated

 ENDIF ELSE BEGIN; spot is ambiguous

  matched(j).intp(0) = waitpar(i).maxval0
  IF waitpar(i).maxval0 GE satvalue THEN BEGIN
   PRINT, 'Saturated spot at :',j
   matched(j).close = 1
   number_sat = number_sat + 1
  ENDIF ; saturated

 ENDELSE ; spot is ambiguous

j = j + 1
ENDFOR ; go through all ambiguous spots

;PRINT,'Number of recovered spots :',recovered
PRINT,'Number of saturated spots :',number_sat
IF KEYWORD_SET(show) THEN WSET,main_window

RETURN,matched(0:j-1)
END

