FUNCTION INTEGRATE_PF3,spot_list,ref_profile,image,image_b,reb,par,ref_distance,ref_ang_width,chi_square_test,peak_variance,success_flag,bc_coeff=bc_coeff,show_weak=show_weak,show_ovlp=show_ovlp,show_m=show_m,miller=miller,sym_rad=sym_rad,box_scale_integ=box_scale_integ,second_interp=second_interp,study_win=study_win,main_win=main_win,box2_scale_integ=box2_scale_integ,get_box=get_box
;+
; NAME:
;	INTEGRATE_PF3
;
; PURPOSE:
;	Profile fitting by standard Least Square method, with overlap
;	deconvolution. Optimization of fitting cutoff level.
;
; CATEGORY:
;	Laue processing.
;
; CALLING SEQUENCE:
;	integrated = INTEGRATE_PF3(spot_list,ref_profile,image,image_b,par, $
;			ref_distance,ref_ang_width,chi_square_test,$
;			peak_variance,success_flag,bc_coeff=bc_coeff,/show_weak,/show_ovlp,$
;			show_m,miller=miller,sym_rad=sym_rad,box_scale_integ=box_scale_integ, $
;			second_interp=second_interp, $
;			study_win=study_win,main_win=main_win )
;
; INPUTS:
;	spot_list : a structure containing the spot information
;	ref_profile : list of structure containing the learned profiles, 
;	and associated parameters
;	image : the original image
;	image_b : the byte image with spot regions marked
;	reb : the rebinning factor
;	par : the parameters of the Laue experiment
;	ref_distance : the maximum radial distance to look for a reference peak
;	   	from the position of the currently integrated peak [mm]
;	ref_ang_width : the maximum tengential distance to look for a 
;		reference peak from the position of the currently integrated 
;		peak [degrees]
;	show_weak : keyword set if the procedure is to be displayed for weak 
;			(non overlapped) peaks.
;	show_ovlp : keyword set if the procedure is to be displayed for 
;			overlapped peaks.
;	show_m : keyword set if one reflection specified by miller is to be
;		 checked
;	miller : INTARR[3] containing the Miller indices of the reflection 
;		to be studied if the keyword /show_m is set
;	sym_rad : set this keyword if the reference peak search is to be extented
;		to symmetrical sector if necessary
;	second_interp : set this keyword if you want to allow a second interpolation 
;		of reference profile in case of bad chi square tests.
;	study_win : window id of the window displaying surface plots, contour ... in debug modes
;	main_win : window id of the main window displaying the image
;
; OUTPUTS:
;	integrated : a FLTARR containing the integrated intensities
;	chi_square_test : [optionnal] gives an estimate of the goodness of fit
;	peak_variance : [optionnal] gives the expected variance of peaks
;	bc_coeff : [optionnal] gives the background parameters : [0,n_back,bc_mean]
;	success_flag : [optionnal] returns 0 if integration successful
;		1, if problem with background, 2 if problem at integration
;	box_scale_integ : similar to the returned array "integrated" but contains scaled, box
;		integrated values
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None.
;
; PROCEDURE:
;	Its an unbiased profile fitting procedure. 
;	FIRST --> the spots are recognized as being overlapped or not.
;	SECOND --> CASE OF OVERLAPPED SPOTS
;	The spots involved in the overlap are obtained with the routine 
;	GET_OVLP_REG.PRO.
;	Each of these spots are background subtracted with the routine
;	BACK2D_POLY4.PRO.
;	Then a learned profile is obtained for each spot by calculating
;	a weighted average of the neighbouring reference profiles.
;	A LEAST_SQUARE correlation matrix is computed with the help
;	of the routine GET_CORREL2.PRO.
;	This matrix is inverted by the Cholewsky method. (NR_CHOLDC,NR_CHOLSL)
;	A chi-square test is evaluated for the integration.
;	THIRD --> CASE OF NON OVERLAPPED SPOTS
;	Each spot is background subtracted with the routine
;	BACK2D_POLY4.PRO.
;	Then a learned profile is obtained by calculating
;	a weighted average of the neighbouring reference profiles.
;	The integration is done by unbiased LEAST_SQUARE fit (J minimizes
;	the criterium Sum(J*Pi - Qi)^2, with Pi being the intensities,
;	of the learned profile, and Qi being the measured intensities)
;	over a limited region determined by the routine GET_FITLEVEL.PRO.
;	A chi-square test is evaluated for the integration.
;	The variances are calculated with the routine GET_PFVAR.PRO
;	If the /show_weak or /show_ovlp keywords are used, various information
;	 are printed,
;	and the profiles (learned, original and difference) are plotted.
;
; MODIFICATION HISTORY:
;	Dominique Bourgeois, June 97.
;-

;ON_ERROR,2                              ;Return to caller if an error occurs

;*************
;GET GENERAL PARAMETERS
;*************

;get the dimension and size of the spot_list
number_spots=(SIZE(spot_list))(1)

;get the dimension and size of the reference profile
sizearea_x = par.boxsize.x
sizearea_y = par.boxsize.y

;get pixel sizes and position of the image center
centerX = par.cenx - 1
centerY = par.ceny - 1
scaleY = par.yscale
pix_size = par.raster
pix_sizeX = 0.001*pix_size
pix_sizeY = 0.001*pix_size/scaleY


;create the array for the integrated intensities
integrated = FLTARR(number_spots)
box_scale_integ = FLTARR(number_spots)
box2_scale_integ = FLTARR(number_spots)
chi_square_test = FLTARR(number_spots)
peak_variance = FLTARR(number_spots)
bc_coeff=FLTARR(3,number_spots,/NOZERO)
success_flag = FLTARR(number_spots)
ovlp_err = 0

IF KEYWORD_SET(show_weak) OR KEYWORD_SET(show_ovlp) OR KEYWORD_SET(show_m) THEN BEGIN
 ;current window
 IF NOT N_ELEMENTS(main_win) THEN main_win=!D.WINDOW
 IF NOT N_ELEMENTS(study_win) THEN BEGIN
   WIN,512,512
   study_win=!D.WINDOW
 ENDIF
ENDIF

;if a particular reflexion is to be studied, is it overlapped or not

IF KEYWORD_SET(show_m) THEN BEGIN

w = WHERE((spot_list.m(0) EQ miller(0)) AND (spot_list.m(1) EQ miller(1)) AND (spot_list.m(2) EQ miller(2)),count)
IF count EQ 0 THEN BEGIN
 PRINT,'Spot not found !'
 RETURN,0
ENDIF

IF count EQ 1 THEN BEGIN
 PRINT,'Spot found !'
 ;is it overlapped ?
 first_ref = w(0)
 IF spot_list(w).ovlp GT 0 THEN BEGIN
 PRINT,'Overlapped spot !'
 old_first_ref = w
 ;check all reflections connected to spot_list(w)
 FOR loc_i=0,spot_list(w).ovlp-1 DO BEGIN
  first_ref = WHERE(spot_list.id EQ spot_list(w).connect(loc_i),count)
  IF count EQ 1 THEN first_ref = first_ref < old_first_ref 
 ENDFOR
 miller = spot_list(first_ref).m
 ;give a flag
 show_m_flag = 1
ENDIF ELSE show_m_flag = 2

ENDIF ; (count = 1)
last_ref=first_ref

ENDIF ELSE BEGIN ;(show_m)
 first_ref=0
 last_ref=number_spots-1
ENDELSE

;parameters for chi_square calculation
chi_mult=1/5.
;chi_mult=1.

;*******************
;START INTEGRATING
;*******************

;arrays for statistics
stat = {non_ovlp:0L,ovlp2:0L,ovlp3:0L,ovlp4:0L,ovlp5_9:0L,ovlp10_49:0L,ovlp_more50:0L,bad_bg:0L,bad_ii:0L,bad_other:0L}
cr = STRING("15b)
stat_format = "($,'w:',I4,' o2:',I4, ' o3:',I4,' o4:',I4,' o5-9:',I4,' o10-49:',I4,' o>50:',I4,' pb_bg: ',I4,' pb_ii: ',I4,' pb : ',I4,' n:',I5,' ie: ',F4.0,'%',A)"


;atan for ref_profile
at_ref = ATAN(ref_profile.y,ref_profile.x)*180/!Pi
sq_ref = SQRT(ref_profile.x^2 + ref_profile.y^2)

;run the procedure for each spot
first_ref=(first_ref>0L)
last_ref=(last_ref<(number_spots-1))
FOR i=first_ref(0),last_ref(0) DO BEGIN ;(for_loop1)
;FOR i=2000,3000 DO BEGIN ;(for_loop1)
;print,'i======',i,' miller : ',spot_list(i).m

 IF KEYWORD_SET(show_m) THEN BEGIN
  IF ((spot_list(i).m(0) EQ miller(0)) AND (spot_list(i).m(1) EQ miller(1)) AND (spot_list(i).m(2) EQ miller(2))) THEN BEGIN
   IF show_m_flag EQ 1 THEN show_ovlp = 1 ELSE show_weak =1
  ENDIF
 ENDIF

;is it an overlapped spot ?
escape :
IF spot_list(i).ovlp GT 0 THEN BEGIN ;(if_loop1) yes it is overlapped

; ***********************
;START INTEGRATING OVERLAPPED SPOTS
; ***********************

;***
;PRINT INFO
;***
IF KEYWORD_SET(show_ovlp) THEN BEGIN
 PRINT,''
 PRINT,'Peak number :',i
 PRINT,'Miller indices :',spot_list(i).m
 PRINT,'Overlapping situation'
ENDIF

;***
;get the spots involved in the overlap.
;***
; ovlp_spots is a structure similar to spot_list, but containing the 
; spots involved in the overlap.
ovlp_spots = GET_OVLP_REG(spot_list,i,index_list)
num_ovlp = (SIZE(ovlp_spots))(1)

IF num_ovlp EQ 1 THEN BEGIN
 ;PRINT,'No overlapped spot found, treating as non overlapped ...'

 ;still treat the spot as non overlapped but put the flag to 3 via ovlp_err
 spot_list(i).ovlp = 0
 ovlp_err = 3
 GOTO, escape 
ENDIF ELSE BEGIN ; (else_loop1) escape if no overlap

;define spots caracteristiques
;caution : only some arrays can be initialized with random numbers
cara_spots = REPLICATE({xpos : 0, ypos : 0, area : FLTARR(sizearea_x,sizearea_y,/NOZERO),indices : PTR_NEW(),adapted_profile : FLTARR(sizearea_x,sizearea_y), background : FLTARR(sizearea_x,sizearea_y,/NOZERO), bc_coeff : FLTARR(3,/NOZERO), flag : 0, count : 0, fit_level : 0.0, peak_area : PTR_NEW(), wcom : INTARR(sizearea_x*sizearea_y), ncom : 0}, num_ovlp)

;***
;START STUDY FOR EACH OVERLAPPED SPOT
;***
re_ovlp_flag_integ = 0
loc_re_ovlp_flag_integ = 0
re_integ_ovlp :

FOR k=0,num_ovlp-1 DO BEGIN ;(for_loop2)

 ;get the center [raster]
 IF re_ovlp_flag_integ THEN GOTO,re_integ2_ovlp

 cara_spots(k).xpos = FIX(ovlp_spots(k).x/pix_sizeX+centerX)
 cara_spots(k).ypos = FIX(ovlp_spots(k).y/pix_sizeY+centerY)

 ;get the corresponding area in image
 area = image(cara_spots(k).xpos - FIX(sizearea_x/2.) : $
    cara_spots(k).xpos + FIX(sizearea_x/2.) , cara_spots(k).ypos - $
    FIX(sizearea_y/2.):cara_spots(k).ypos + FIX(sizearea_y/2.))

 ;and multiply by the allowed region
 area_b = area*(image_b(cara_spots(k).xpos - FIX(sizearea_x/2.):cara_spots(k).xpos + FIX(sizearea_x/2.) , $
		cara_spots(k).ypos - FIX(sizearea_y/2.):cara_spots(k).ypos + FIX(sizearea_y/2.)) EQ 0)

 ;***
 ;subtract background 
 ;***
 ;cara_spots(k).background =  BACK2D_POLY4(area,par.noise_percentage,par.bg_gain,par.dark_current,flag,coeff,bc_mean,n_back,chi_square,max_slope=par.bg_max_slope) 
 cara_spots(k).background =  BACK2D_POLY4(area_b,par.noise_percentage,par.bg_gain,par.dark_current,flag,coeff,bc_mean,n_back,chi_square,max_slope=par.bg_max_slope)

 cara_spots(k).flag = flag
 cara_spots(k).bc_coeff = [0,n_back,bc_mean]
 bc_coeff(*,index_list)=cara_spots.bc_coeff
 cara_spots(k).area = area - cara_spots(k).background
 ;the value of flag can be used to possibly remove the stuff with bad background

 ;****
 ;get a "learned profile" by a weighted average of the reference profiles
 ;****

 IF KEYWORD_SET(show_ovlp) THEN BEGIN
  WSET,study_win
  PRINT,'----> OVERLAPPED SPOT :'
  cara_spots(k).adapted_profile=GET_ADAPTED_PROFILE(ovlp_spots(k).x,ovlp_spots(k).y,ovlp_spots(k).x/pix_sizeX+centerX,ovlp_spots(k).y/pix_sizeY+centerY,ref_profile,ref_distance,ref_ang_width,at=at_ref,sq=sq_ref,sym_rad=sym_rad,/SHOW,area=cara_spots(k).area)
 ENDIF ELSE $
  cara_spots(k).adapted_profile=GET_ADAPTED_PROFILE(ovlp_spots(k).x,ovlp_spots(k).y,ovlp_spots(k).x/pix_sizeX+centerX,ovlp_spots(k).y/pix_sizeY+centerY,ref_profile,ref_distance,ref_ang_width,at=at_ref,sq=sq_ref,sym_rad=sym_rad)

 re_integ2_ovlp:

 ;optimization of fit_level by variance minimization
 IF KEYWORD_SET(show_ovlp) THEN $
  cara_spots(k).fit_level = 1*GET_FITLEVEL(cara_spots(k).adapted_profile,ovlp_spots(k).intp(0),cara_spots(k).bc_coeff(1),cara_spots(k).bc_coeff(2),par.dark_current,bg_gain=par.bg_gain,gain=par.gain*1.5418/ovlp_spots(k).l,/show) $
 ELSE cara_spots(k).fit_level = 1*GET_FITLEVEL(cara_spots(k).adapted_profile,ovlp_spots(k).intp(0),cara_spots(k).bc_coeff(1),cara_spots(k).bc_coeff(2),par.dark_current,bg_gain=par.gain,gain=par.gain*1.5418/ovlp_spots(k).l)

 cara_spots(k).peak_area = PTR_NEW(WHERE(cara_spots(k).adapted_profile GE MAX(cara_spots(k).adapted_profile)*cara_spots(k).fit_level/100.,ct))

 cara_spots(k).count = ct

 ;pixel indices in image of peak_area
 x_y_coord = WHERESUB(*cara_spots(k).peak_area,cara_spots(k).area)

 cara_spots(k).indices = PTR_NEW(LONG(cara_spots(k).xpos - FIX(sizearea_x/2.) + FLOAT(x_y_coord) + par.rastx*(cara_spots(k).ypos - FIX(sizearea_y/2.) + IMAGINARY(x_y_coord))))

 ;***
 ;PRINT INFO
 ;***
 IF KEYWORD_SET(show_ovlp) THEN BEGIN
  center_area = WHERESUB(WHERE(cara_spots(k).area EQ MAX(cara_spots(k).area)), cara_spots(k).area)
  center_adapted = WHERESUB(WHERE(cara_spots(k).adapted_profile EQ MAX(cara_spots(k).adapted_profile)), cara_spots(k).adapted_profile)
  shift_area_adapted = center_area(0) - center_adapted(0)
  PRINT,'Shift between peaks [raster], [x,y] for peak ',k,' :',shift_area_adapted(0)
 ENDIF

ENDFOR ; (for_loop2) run over k

;***
;ARE THERE SPOTS TOO CLOSE TO BE DECONVOLUTED 
;***
w = WHERE(ovlp_spots.close EQ 2, num_too_close)
IF num_too_close GT 0 THEN BEGIN
 ;separate between spots too close to be deconvoluted and spots OK !
 cara_too_close = cara_spots(w)
 index_too_close = index_list(w)
 ovlp_too_close = ovlp_spots(w)
 integrated(index_too_close) = 0
 box_scale_integ(index_too_close) = 0
 peak_variance(index_too_close) = 1
 w = WHERE(ovlp_spots.close NE 2,num_ovlp)
 IF num_ovlp GT 0 THEN BEGIN
  cara_spots = cara_spots(w)
  ovlp_spots = ovlp_spots(w)
  index_list = index_list(w)
 ENDIF ELSE GOTO,escape2
 IF KEYWORD_SET(show_ovlp) THEN BEGIN
  PRINT,'Number of spots to deconvolute : ',num_ovlp
  PRINT,'Number of spots too close to deconvolute : ',num_too_close
 ENDIF ; (show)
 FOR k=0,num_ovlp-1 DO BEGIN
  FOR k2=0,num_too_close-1 DO BEGIN
   IF MIN(ABS(ovlp_spots(k).connect - ovlp_too_close(k2).id)) EQ 0 THEN BEGIN
      	w = WHERE_CORREL(cara_spots(k),cara_too_close(k2),nu_pix)
	IF nu_pix NE 0 THEN BEGIN
         (*cara_spots(k).peak_area)(w) = -1
         (*cara_spots(k).indices)(w) = -1
        ENDIF
   ENDIF ; (there is correlation) 
  ENDFOR ; (over k2)

 w=WHERE(*cara_spots(k).peak_area GE 0,nu_pix)
 IF nu_pix GE 1 THEN BEGIN
  cara_spots(k).peak_area=PTR_NEW((*cara_spots(k).peak_area)(w))
  cara_spots(k).indices=PTR_NEW((*cara_spots(k).indices)(w))
  cara_spots(k).count = nu_pix
 ENDIF ELSE cara_spots(k).count = 0

 ENDFOR ;(over k)

 ;Take out spots entirely eaten
 w = WHERE(cara_spots.count EQ 0,num_eaten)
 IF num_eaten GT 0 THEN BEGIN
  ;set to 0 the integrated intensity for the eaten spots
  index_eaten = index_list(w)
  integrated(index_eaten) = 0
  box_scale_integ(index_eaten) = 0
  peak_variance(index_eaten) = 1
  ;reduce the set of ovlp spots to deconvolute
  w = WHERE(cara_spots.count GT 0,num_ovlp)
  IF num_ovlp GT 0 THEN BEGIN
   cara_spots = cara_spots(w)
   ovlp_spots = ovlp_spots(w)
   index_list = index_list(w)
  ENDIF ELSE GOTO,escape2
 ENDIF ; spots eaten
ENDIF ; spots too close to be deconvoluted
 
;******
;START CALCULATING INTEGRATED INTENSITIES (UNBIASED FITTING PROCEDURE ; WITH FIT_LEVEL CUTOFF)
;******

;calculate the b [num_ovlp] vector 
b_vect = FLTARR(num_ovlp)
FOR k=0,num_ovlp-1 DO b_vect(k) = TOTAL(cara_spots(k).area(*cara_spots(k).peak_area)*cara_spots(k).adapted_profile(*cara_spots(k).peak_area))

;***
;calculate the correlation matrix (dimensions = [num_ovlp,num_ovlp])
;***
corr_matrix=FLTARR(num_ovlp,num_ovlp)

;this matrix is symmetric, we only need to calculate the upper region

;in order to calculate later a chi_square test we need to know the number
;of pixels in the overlapped region, which are common to 2 peaks (this
;number is related to the degree of freedom). We ignore
;those common to 3 peaks, which may lead to a small error at the end
number_common_indices = 0

FOR row=0,num_ovlp-1 DO BEGIN ;(for_loop3)
 FOR column=0,row DO BEGIN ;(for_loop4)
  ;calculate corr_matrix(row,column)
  IF row EQ column THEN corr_matrix(row,column)= TOTAL(cara_spots(row).adapted_profile(*cara_spots(row).peak_area)^2) ELSE BEGIN
   ;cross term --> look for a correlation
   IF MIN(ABS(ovlp_spots(row).connect - ovlp_spots(column).id)) EQ 0 THEN BEGIN
    ;there is correlation : we look for the common pixels
    ;print,'Connecting id ',ovlp_spots(row).id,' with id ',ovlp_spots(column).id
    corr_matrix(row,column) = GET_CORREL2(cara_spots(row),cara_spots(column),nu_pix,wcom1,wcom2)

    IF nu_pix GT 0 THEN BEGIN
     cara_spots(row).wcom(cara_spots(row).ncom:cara_spots(row).ncom+nu_pix-1)=wcom1
     cara_spots(row).ncom=cara_spots(row).ncom+nu_pix
     cara_spots(column).wcom(cara_spots(column).ncom:cara_spots(column).ncom+nu_pix-1)=wcom2
     cara_spots(column).ncom=cara_spots(column).ncom+nu_pix
    ENDIF
    ;number_common_indices = number_common_indices + nu_pix
   ENDIF 
  ENDELSE ;(cross term)
 ENDFOR ;(for_loop4)
ENDFOR ;(for_loop3)

;******
;MATRIX INVERSION (Cholevsky method) and calculation of integrated intensities.
;******

;print info
IF KEYWORD_SET(show_ovlp) THEN print,'corr_matrix :',corr_matrix

;produce variance matrix
orig_corr_matrix = corr_matrix

;check if more than one spot is integrated
IF (SIZE(index_list))(1) GT 1 THEN BEGIN

CATCH, Error_status	
;error_status=0
IF Error_status NE 0 THEN BEGIN	;This statement begins the error handler.
  CATCH,/CANCEL
  PRINT, 'Error message:', !ERR_STRING
  ;Handles the error
  ;if the error is due to a MESSAGE command then quit the program
  IF Error_status EQ -5 THEN RETURN,0
  ;if not fix the CHOLDC problem
  corr_matrix(*)=0
  FOR loc_index=0, num_ovlp-1 DO corr_matrix(loc_index,loc_index)=1  
  cara_spots.flag = 5
  ;PRINT,'Miller indices : ',spot_list(index_list).m
ENDIF


NR_CHOLDC,corr_matrix,p
integrated(index_list) = NR_CHOLSL(corr_matrix,p,b_vect)
ENDIF ELSE integrated(index_list(0)) = b_vect(0)/corr_matrix(0) 

;get the scaled box integrated values
FOR loc_index=0,(SIZE(index_list))(1)-1 DO BEGIN
 corr_area=*cara_spots(loc_index).peak_area
 loc_w=WHERE(corr_area NE 0,loc_ct)
 IF loc_ct LE cara_spots(loc_index).ncom THEN box_scale_integ(index_list(loc_index)) = integrated(index_list(loc_index)) ELSE BEGIN
  IF cara_spots(loc_index).ncom GT 0 THEN corr_area(cara_spots(loc_index).wcom(0:cara_spots(loc_index).ncom-1))=0
  corr_area=corr_area(WHERE(corr_area NE 0))
  box_scale_integ(index_list(loc_index)) = TOTAL(cara_spots(loc_index).area(corr_area))/FLOAT(TOTAL(cara_spots(loc_index).adapted_profile(corr_area)))
 ENDELSE
ENDFOR

;****
;PRINT AND PLOT INFO
;****
IF KEYWORD_SET(SHOW_OVLP) THEN BEGIN ;(if_loop2)
 FOR loc_index=0,(SIZE(index_list))(1)-1 DO $
  PRINT,'Miller indices for spot :',loc_index,':',spot_list(index_list(loc_index)).m
 IF num_too_close GT 0 THEN BEGIN 
  FOR loc_index=0,(SIZE(index_too_close))(1)-1 DO $
   PRINT,'Miller indices for spot (too close to integrate):',loc_index,':',spot_list(index_too_close(loc_index)).m
 ENDIF

 WSET,main_win
 dummy=PIXLAUE2(spot_list(index_list),reb,image,par,draw=8,color=2)
 WSET,study_win

 ;first get an image containing the whole of the overlapping region.
 ;get the xmax,xmin,ymax,ymin
 xmin = par.rastx
 ymin = par.rasty
 xmax = 0
 ymax = 0

 FOR k=0,num_ovlp-1 DO BEGIN
  xy_coord = WHERESUB(*cara_spots(k).indices,image)
  x_coord = FLOAT(xy_coord)
  y_coord = IMAGINARY(xy_coord)
  xmin = xmin < MIN(x_coord)
  ymin = ymin < MIN(y_coord)
  xmax = xmax > MAX(x_coord)
  ymax = ymax > MAX(y_coord)
 ENDFOR

 IF num_too_close GT 0 THEN BEGIN
  FOR k=0,num_too_close-1 DO BEGIN
   xy_coord = WHERESUB(*cara_too_close(k).indices,image)
   x_coord = FLOAT(xy_coord)
   y_coord = IMAGINARY(xy_coord)
   xmin = xmin < MIN(x_coord)
   ymin = ymin < MIN(y_coord)
   xmax = xmax > MAX(x_coord)
   ymax = ymax > MAX(y_coord)
  ENDFOR
 ENDIF

 ;best is to get a square image
 big_size = (ymax - ymin) > (xmax - xmin)
 loc_centerX = (xmin + xmax)/2.
 loc_centerY = (ymin + ymax)/2.
 xmax = ROUND(loc_centerX + big_size/2.+1)
 xmin = ROUND(loc_centerX - big_size/2.-1)
 ymax = ROUND(loc_centerY + big_size/2.+1)
 ymin = ROUND(loc_centerY - big_size/2.-1)

 ;indice of the left bottom pixel
 min_indice = ymin*par.rastx + xmin

 ;this is the original (with background subtracted) image
 ovlp_im=image(xmin:xmax,ymin:ymax) - BACK2D_POLY4(image(xmin:xmax,ymin:ymax),par.noise_percentage,par.bg_gain,par.dark_current,max_slope=par.bg_max_slope)

 ;we now want the fitted profile image : for that we need to play with the 
 ;field "indices", in order to shift them from "image" to "ovlp_im"
 ;the formula is given by :

 FOR k=0,num_ovlp-1 DO BEGIN
  cara_spots(k).indices = PTR_NEW(*cara_spots(k).indices-min_indice+(par.rastx-(xmax-xmin+1))*(ymin-*cara_spots(k).indices/par.rastx))
 ENDFOR
 ;we can now build the image of learned profiles
 profile_im=FLTARR(xmax-xmin+1,ymax-ymin+1)

 FOR k=0,num_ovlp-1 DO BEGIN
  profile_im(*cara_spots(k).indices)=profile_im(*cara_spots(k).indices)+integrated(index_list(k))*cara_spots(k).adapted_profile(*cara_spots(k).peak_area) 

  ;we can now display
  PRINT,'Position of maximum [x,y] for spot ',k,' (profile) : ',WHERESUB(WHERE(cara_spots(k).adapted_profile EQ MAX(cara_spots(k).adapted_profile)),cara_spots(k).adapted_profile)
 ENDFOR
 PRINT,'Fitted profile'
 ;CONTOUR,profile_im,LEVELS=[(INDGEN(10)+1)*(0.01>MAX(profile_im))/10.],C_COLORS=!D.N_COLORS - 150
 CONTOUR,profile_im,LEVELS=[(INDGEN(10)+1)*(0.01>MAX(profile_im))/10.],C_COLORS=!D.N_COLORS - 1
 SURFACE,profile_im
 OK=''
 READ,'OK ? [Y/N]',OK

 FOR k=0,num_ovlp-1 DO PRINT,'Position of maximum [x,y] for spot ',k,' (original) : ',WHERESUB(WHERE(cara_spots(k).area EQ MAX(cara_spots(k).area)),cara_spots(k).area)
 PRINT,'Original profile'
; CONTOUR,ovlp_im,LEVELS=[(INDGEN(10)+1)*MAX(ovlp_im)/10.],/NOERASE,C_COLORS=!D.N_COLORS - 60
 CONTOUR,ovlp_im,LEVELS=[(INDGEN(10)+1)*MAX(ovlp_im)/10.],/NOERASE,C_COLORS=!D.N_COLORS - 2
 SURFACE,ovlp_im
 READ,'OK ? [Y/N]',OK

 PRINT,'Difference profile'
 SURFACE,profile_im - ovlp_im
 READ,'OK ? [Y/N]',OK

 PRINT,'Background calculations'
 FOR k=0,num_ovlp-1 DO BEGIN
  PRINT,'Spot # :',k
  PRINT,'Backgroud for reference profile :'
  SURFACE,cara_spots(k).adapted_profile < MAX(cara_spots(k).adapted_profile)/2.,ax=0
  READ,'OK ? [Y/N]',OK

  PRINT,'Backgroud for spot :'
  dummy = cara_spots(k).area
  SURFACE,dummy < MAX(dummy)/2.,ax=0
  READ,'OK ? [Y/N]',OK
 ENDFOR

 FOR k=0,num_ovlp-1 DO BEGIN
  PRINT,'Profile fitted Integrated intensity for spot ',k,' :',integrated(index_list(k))
  PRINT,'Scales box Integrated intensity for spot ',k,' :',box_scale_integ(index_list(k))
  PRINT,'Percentage of used profile [%] :',100.*cara_spots(k).count/(SIZE(cara_spots(k).adapted_profile))(4)
 ENDFOR

ENDIF ; (if_loop2) show


;***********
;estimate the goodness of fit (chi_square_test)
;***********
;the variance^2 has to be multiplied by par.gain (the variance is sigma = gain*
;SQRT(number of Xrays) = SQRT(gain)*SQRT(number of counts)
chi_square = 0
;the gain is proportionnal to energy and equal to par.gain at 1.5418 A
;the chi_square value is multiplied by chi_mult because experimentally, the
;chi_square test tends to undestimate the fit quality (and it has been checked
;that the esimated variances are not far from corresponding to truth)

;estimate the number of common indices
FOR k=0,num_ovlp-1 DO BEGIN
 unique_indices=N_ELEMENTS(UNIQ(cara_spots(k).wcom,SORT(cara_spots(k).wcom)))-1
 all_indices=cara_spots(k).ncom
 number_common_indices = number_common_indices + unique_indices - all_indices/2. + all_indices/3.
ENDFOR

;the chi_square must be calculated independantly for each spot if there is
;no correlation between them. If yes, a common chi_square value is given.

IF number_common_indices GT 0 THEN BEGIN ; real overlap
 ;in this case there is a real overlapping situation

 FOR k=0,num_ovlp-1 DO chi_square = chi_square + chi_mult*TOTAL((integrated(index_list(k))*cara_spots(k).adapted_profile(*cara_spots(k).peak_area) - cara_spots(k).area(*cara_spots(k).peak_area))^2/(par.gain*1.5418/ovlp_spots(k).l)/(1>(cara_spots(k).area(*cara_spots(k).peak_area) + (1+1/cara_spots(k).bc_coeff(1))*(cara_spots(k).bc_coeff(2) - par.dark_current))))

 ;we must count the number of pixels involved into the overlapping situation
 ;we withdraw num_ovlp at the end to account for the "num_ovlp" correlations
 ;count is the degree of freedom = number_pixels_involved - num_ovlp
 count = TOTAL(cara_spots.count) - number_common_indices - num_ovlp

 proba = 1 - CHI_SQR1(chi_square,count>1)

 IF KEYWORD_SET(second_interp) AND proba LT 0.95 AND re_ovlp_flag_integ EQ 0 THEN BEGIN ; reintegrate if not perfect
  loc_max_loc_proba=1

  FOR k=0,num_ovlp-1 DO BEGIN
   corr_area=*cara_spots(k).peak_area
   IF cara_spots(k).ncom GT 0 THEN corr_area(cara_spots(k).wcom(0:cara_spots(k).ncom-1))=0
   corr_area=corr_area(WHERE(corr_area NE 0,loc_count))

   loc_chi_square = chi_mult*TOTAL((integrated(index_list(k))*cara_spots(k).adapted_profile(corr_area) - cara_spots(k).area(corr_area))^2/(par.gain*1.5418/ovlp_spots(k).l)/(1>(cara_spots(k).area(corr_area) + (1+1/cara_spots(k).bc_coeff(1))*(cara_spots(k).bc_coeff(2) - par.dark_current))))

   loc_proba = 1 - CHI_SQR1(loc_chi_square,loc_count)
   loc_max_loc_proba = loc_max_loc_proba < loc_proba

   IF KEYWORD_SET(show_ovlp) THEN BEGIN
    PRINT,'Chi square test for spot ',k,' is : ',loc_proba
    IF loc_proba LT 0.95 THEN PRINT,'Will try to interpolate ...' ELSE PRINT,'Not interpolating ...'
   ENDIF
   IF loc_proba LT 0.95 THEN BEGIN
    IF KEYWORD_SET(show_ovlp) THEN cara_spots(k).adapted_profile = GET_BEST_OVLP_PROFILE(cara_spots(k),w_max_corr,/SHOW) ELSE $
				  cara_spots(k).adapted_profile = GET_BEST_OVLP_PROFILE(cara_spots(k),w_max_corr)
    ;normalize the integrated intensity of the reference profile to 1.
    cara_spots(k).adapted_profile = cara_spots(k).adapted_profile/TOTAL(cara_spots(k).adapted_profile)
   ENDIF
  ENDFOR


  IF loc_max_loc_proba LT 0.95 THEN BEGIN
   ;redoing the calculation can only be done once
   re_ovlp_flag_integ = 1
   ;reset the peak areas
   cara_spots.peak_area=PTR_NEW()
   ;reset the indices
   cara_spots.indices=PTR_NEW()
   ;reset the common indices
   cara_spots.wcom(*)=0
   cara_spots.ncom(*)=0
   GOTO,re_integ_ovlp
  ENDIF
 ENDIF

 ;the calculation should be rejected if chi_square_test is too low,
 ; ie proba < 0.001
 IF proba LE 0.001 THEN BEGIN
  FOR k=0,num_ovlp-1 DO BEGIN
   IF cara_spots(k).flag NE 1 THEN cara_spots(k).flag = 2 > cara_spots(k).flag
  ENDFOR
 ENDIF
 ;IF proba LE 0.001 THEN stop
 ;IF re_ovlp_flag_integ THEN print,'New proba : ',proba

 ;IF proba LE 0.001 THEN PRINT,'Still bad !'

 chi_square_test(index_list) = proba
 success_flag(index_list) = cara_spots.flag

 IF KEYWORD_SET(show_ovlp) THEN BEGIN
  PRINT,'Chi square test :', proba
  PRINT,'Common chi square value :',chi_square
  PRINT,'Value for which chi_square test = 0.001 :',CHI_SQR1(0.001,count)

  FOR k=0,num_ovlp-1 DO BEGIN
   PRINT,'Flag [0 : OK ; 1 : bad background ; 2 : bad ii] for spot :',k,' is : ', cara_spots(k).flag
  ENDFOR
 ENDIF

 ;note the chi_square test is performed over the whole of the region.

ENDIF ELSE BEGIN ; end real overlap

 ; in this case the each spot is considered independantly.

 FOR k=0,num_ovlp-1 DO BEGIN ; for_l2
  chi_square = chi_mult*TOTAL((integrated(index_list(k))*cara_spots(k).adapted_profile(*cara_spots(k).peak_area) - cara_spots(k).area(*cara_spots(k).peak_area))^2/(par.gain*1.5418/ovlp_spots(k).l)/(1>(cara_spots(k).area(*cara_spots(k).peak_area) + (1+1/cara_spots(k).bc_coeff(1))*(cara_spots(k).bc_coeff(2) - par.dark_current))))

  count = cara_spots(k).count - 1
  proba = 1 - CHI_SQR1(chi_square,count)

  ;the calculation should be rejected if chi_square_test is too low,
  ; ie proba < 0.001 and background was ok.
  IF proba LE 0.001 AND cara_spots(k).flag NE 1 THEN cara_spots(k).flag =  2 > cara_spots(k).flag

  chi_square_test(index_list(k)) = proba
  success_flag(index_list(k)) = cara_spots(k).flag

  IF KEYWORD_SET(show_ovlp) THEN BEGIN
   PRINT,'Chi square test for peak ',k,':', proba
   PRINT,'Chi square value :',chi_square
   PRINT,'Value for which chi_square test = 0.001 :',CHI_SQR1(0.001,count)
   PRINT,'Flag [0 : OK ; 1 : bad background ; 2 : bad ii] :',cara_spots(k).flag
  ENDIF

  IF KEYWORD_SET(second_interp) AND proba LT 0.95 AND re_ovlp_flag_integ EQ 0 THEN BEGIN ;reintegrate if not perfect
   IF KEYWORD_SET(show_ovlp) THEN BEGIN
    PRINT, 'Will try to interpolate spot # : ',k
   ENDIF 
   ;redoing the calculation can only be done once
   loc_re_ovlp_flag_integ = 1
   ;test the correlation 

   IF KEYWORD_SET(show_ovlp) THEN cara_spots(k).adapted_profile = GET_BEST_OVLP_PROFILE(cara_spots(k),w_max_corr,/SHOW) ELSE $
				cara_spots(k).adapted_profile = GET_BEST_OVLP_PROFILE(cara_spots(k),w_max_corr)
   ;normalize the integrated intensity of the reference profile to 1.
   cara_spots(k).adapted_profile = cara_spots(k).adapted_profile/TOTAL(cara_spots(k).adapted_profile)
  ENDIF

 ENDFOR ; for_l2

 IF loc_re_ovlp_flag_integ EQ 1 AND re_ovlp_flag_integ EQ 0 THEN BEGIN
  ;allow only one reintegration
  re_ovlp_flag_integ=1
  ;reset the peak areas
  cara_spots.peak_area=PTR_NEW()
  ;reset the indices
  cara_spots.indices=PTR_NEW()
  ;reset the common indices
  cara_spots.wcom(*)=0
  cara_spots.ncom(*)=0
  GOTO,re_integ_ovlp
 ENDIF

ENDELSE ; consider each spot independantly


;***********
;estimate the variance
;***********
;the variance calculation considers only the region where the peaks seat.

;check if more than one spot is involved
IF (SIZE(index_list))(1) GT 1 THEN BEGIN
 ;symetrize the variance matrix

 FOR loc_i = 0,num_ovlp-1 DO FOR loc_j = loc_i+1,num_ovlp-1 DO orig_corr_matrix(loc_i,loc_j) = orig_corr_matrix(loc_j,loc_i)

 ;get the variance of b_vect
 covar_matrix = FLTARR(num_ovlp,num_ovlp)

 FOR row=0,num_ovlp-1 DO BEGIN ;(for_loop3)
  FOR column=0,row DO BEGIN ;(for_loop4)
   ;calculate corr_matrix(row,column)
   IF row EQ column THEN  covar_matrix(row,column) = GET_OVLPV(cara_spots(row),ovlp_spots(row).l,par) ElSE BEGIN
    ;cross term --> look for a correlation
    IF MIN(ABS(ovlp_spots(row).connect - ovlp_spots(column).id)) EQ 0 THEN BEGIN
     ;there is correlation : we look for the common pixels
     ;print,'Connecting id ',ovlp_spots(row).id,' with id ',ovlp_spots(column).id
     covar_matrix(row,column) = GET_OVLPCOV(cara_spots(row),cara_spots(column),ovlp_spots(row).l,ovlp_spots(column).l,par)
    ENDIF 
   ENDELSE ;(cross term)
  ENDFOR ;(for_loop4)
 ENDFOR ;(for_loop3)

 ;symetrize the variance matrix
 FOR loc_i = 0,num_ovlp-1 DO FOR loc_j = loc_i+1,num_ovlp-1 DO covar_matrix(loc_i,loc_j) = covar_matrix(loc_j,loc_i)

 inv_orig_corr_matrix = INVERT(orig_corr_matrix)

 ovlp_var_matrix = inv_orig_corr_matrix # covar_matrix # TRANSPOSE(inv_orig_corr_matrix)

 FOR k=0,num_ovlp-1 DO peak_variance(index_list(k)) = SQRT(ovlp_var_matrix(k,k)>0)

ENDIF ELSE peak_variance(index_list(0)) = SQRT(GET_1OVLPVAR(cara_spots(0).adapted_profile,cara_spots(0).area+cara_spots(0).background,*cara_spots(0).peak_area,cara_spots(0).bc_coeff(1),cara_spots(0).bc_coeff(2),par.gain*1.5418/ovlp_spots(0).l,par.gain,par.dark_current))
; case of only one spot

;***
;PRINT INFO
;***
IF KEYWORD_SET(show_ovlp) THEN BEGIN
 FOR k=0,num_ovlp-1 DO PRINT,'Peak variance (peak ',k,') :',peak_variance(index_list(k))
ENDIF

;IF num_ovlp GE 10 THEN PRINT,FORMAT = '("o",i2,$)',num_ovlp ELSE PRINT,FORMAT = '("o",i1,$)',num_ovlp
;IF num_ovlp GE 10 THEN BEGIN
 ;IF re_ovlp_flag_integ EQ 0 THEN PRINT,FORMAT = '("o",i2,$)',num_ovlp 
 ;IF re_ovlp_flag_integ EQ 1 THEN PRINT,FORMAT = '("p",i2,$)',num_ovlp 
;ENDIF ELSE BEGIN
 ;IF re_ovlp_flag_integ EQ 0 THEN PRINT,FORMAT = '("o",i1,$)',num_ovlp
 ;IF re_ovlp_flag_integ EQ 1 THEN PRINT,FORMAT = '("p",i1,$)',num_ovlp
;ENDELSE

IF num_ovlp EQ 2 THEN stat.ovlp2=stat.ovlp2+num_ovlp
IF num_ovlp EQ 3 THEN stat.ovlp3=stat.ovlp3+num_ovlp
IF num_ovlp EQ 4 THEN stat.ovlp4=stat.ovlp4+num_ovlp
IF num_ovlp GE 5 AND num_ovlp LT 10 THEN stat.ovlp5_9=stat.ovlp5_9+num_ovlp
IF num_ovlp GE 10 AND num_ovlp LT 50 THEN stat.ovlp10_49=stat.ovlp10_49+num_ovlp
IF num_ovlp GE 50 THEN stat.ovlp_more50=stat.ovlp_more50+num_ovlp
w_bad_bg = WHERE(cara_spots.flag EQ 1,ct_bad_bg)
stat.bad_bg = stat.bad_bg + ct_bad_bg
w_bad_ii = WHERE(cara_spots.flag EQ 2,ct_bad_ii)
stat.bad_ii = stat.bad_ii + ct_bad_ii
w_bad_other = WHERE(cara_spots.flag GT 2,ct_bad_other)
stat.bad_other = stat.bad_other + ct_bad_other


ENDELSE ;  (else_loop1) for the case of already treated spots


; ***********************
; END OF OVERLAP CASE
; ***********************

ENDIF ELSE BEGIN ;(if_loop1, else_loop2)
IF spot_list(i).ovlp EQ 0 THEN BEGIN ;if the spot not already treated for
					;overlap problems
;***
;PRINT INFO
;***

IF KEYWORD_SET(show_weak) THEN BEGIN
 WSET,main_win
 dummy=PIXLAUE2(spot_list(i),reb,image,par,draw=8,color=2)
 WSET,study_win

 PRINT,''
 PRINT,'Peak number :',i
 PRINT,'Miller indices :',spot_list(i).m
ENDIF



;***
;handles catch procedure
;***
CATCH, Error_status	
;error_status=0
IF Error_status NE 0 THEN BEGIN	;This statement begins the error handler.
  CATCH,/CANCEL
  PRINT, 'Error message:', !ERR_STRING
  ;Handles the error
  ;Basically go to the next spot and put a flag of 6 to the spot
  integrated(i) = -9999. 
  box_scale_integ(i) = -9999.
  success_flag(i) = 6
  GOTO,escape_non_ovlp
  ;PRINT,'Miller indices : ',spot_list(i).m
ENDIF



;***
;get the corresponding area in image
;***
xpos_fl=spot_list(i).x/pix_sizeX+centerX
ypos_fl=spot_list(i).y/pix_sizeY+centerY
xpos = FIX(xpos_fl)
ypos = FIX(ypos_fl)

area = image(xpos - FIX(sizearea_x/2.):xpos + FIX(sizearea_x/2.) , $
		ypos - FIX(sizearea_y/2.):ypos + FIX(sizearea_y/2.))


area_b = area*(image_b(xpos - FIX(sizearea_x/2.):xpos + FIX(sizearea_x/2.) , $
		ypos - FIX(sizearea_y/2.):ypos + FIX(sizearea_y/2.)) EQ 0)


;surface,area_b
;ok=''
;read,'ok?',ok

re_integ_flag = 0

;***
;subtract background
;***

;if i eq 3403 then stop

;background =  BACK2D_POLY4(area,par.noise_percentage,par.bg_gain,par.dark_current,flag,coeff,bc_mean,n_back,chi_square,max_slope=par.bg_max_slope)
background =  BACK2D_POLY4(area_b,par.noise_percentage,par.bg_gain,par.dark_current,flag,coeff,bc_mean,n_back,chi_square,max_slope=par.bg_max_slope)

;the value of flag can be used to possibly remove the stuff with bad background

area = area - background

;surface,area

;************
;GET THE REFERENCE LEARNED PROFILE by a weighted average of the reference profiles
;************

IF KEYWORD_SET(show_weak) THEN PRINT,'---> NON OVERLAPPED SPOT :'

IF KEYWORD_SET(show_weak) THEN $
 adapted_profile=GET_ADAPTED_PROFILE(spot_list(i).x,spot_list(i).y,xpos_fl,ypos_fl,ref_profile,ref_distance,ref_ang_width,at=at_ref,sq=sq_ref,sym_rad=sym_rad,/SHOW,area=area) ELSE $
 adapted_profile=GET_ADAPTED_PROFILE(spot_list(i).x,spot_list(i).y,xpos_fl,ypos_fl,ref_profile,ref_distance,ref_ang_width,at=at_ref,sq=sq_ref,sym_rad=sym_rad)

re_integ:
;*******
;CALCULATION OF INTEGRATED INTENSITY (UNBIASED PROFILE FITTING)
;*******

;optimization of fit_level by variance minimization
IF KEYWORD_SET(show_weak) THEN $
 fit_level = 1*GET_FITLEVEL(adapted_profile,spot_list(i).intp(0),n_back,bc_mean,par.dark_current,bg_gain=par.bg_gain,gain=par.gain*1.5418/spot_list(i).l,/show) $
ELSE  fit_level = 1*GET_FITLEVEL(adapted_profile,spot_list(i).intp(0),n_back,bc_mean,par.dark_current,bg_gain=par.bg_gain,gain=par.gain*1.5418/spot_list(i).l)

;print,'fit_level,intensity,bc,nback,sob : ',fit_level,spot_list(i).intp(0)-bc_mean,bc_mean,n_back,(spot_list(i).intp(0)-bc_mean)/bc_mean

peak_area = WHERE(adapted_profile GE MAX(adapted_profile)*fit_level/100.,count)

integrated(i) = TOTAL(adapted_profile(peak_area)*area(peak_area))/TOTAL(adapted_profile(peak_area)^2)
bc_coeff(*,i)=[0,n_back,bc_mean]

box_scale_integ(i) = TOTAL(area(peak_area))/TOTAL(adapted_profile(peak_area))
;box_scale_integ(i) = TOTAL(area(WHERE(adapted_profile NE 0)))

;weighted integrated intensity CHECK FORMULA WITH GAIN
;integrated(i) = TOTAL(adapted_profile(peak_area)*area(peak_area)/(gain*(1>(area(peak_area) + (1+1/n_back)*(bc_mean - par.dark_current)))))/TOTAL(adapted_profile(peak_area)^2/(gain*(1>(area(peak_area) + (1+1/n_back)*(bc_mean - par.dark_current)))))


;***
;PRINT INFO
;***

IF KEYWORD_SET(SHOW_WEAK) THEN BEGIN
 center_area = WHERESUB(WHERE(area EQ MAX(area)), area)
 center_adapted = WHERESUB(WHERE(adapted_profile EQ MAX(adapted_profile)), adapted_profile)
 shift_area_adapted = center_area(0) - center_adapted(0)
 PRINT,'Shift between peaks :',shift_area_adapted
 PRINT,'Fitted profile'
 PRINT,'Percentage of used profile [%] :',100.*count/(SIZE(adapted_profile))(4)
 temp = adapted_profile
 temp(*) = 0
 temp(peak_area) = adapted_profile(peak_area)
; CONTOUR,temp,LEVELS=[(INDGEN(10)+1)*MAX(temp)/10.],C_COLORS=!D.N_COLORS - 150
 CONTOUR,temp,LEVELS=[(INDGEN(10)+1)*MAX(temp)/10.],C_COLORS=!D.N_COLORS - 1
 OK=''
 READ,'OK ? [Y/N]',OK

 PRINT,'Original profile'
 PRINT,'Position of maximum [x,y] :',WHERESUB(WHERE(area EQ MAX(area)),area)
 PRINT,'Maximum :',MAX(area)
 PRINT,'Maximum in kept region :',MAX(area(peak_area))
; CONTOUR,area,LEVELS=[(INDGEN(10)+1)*MAX(area)/10.],/NOERASE,C_COLORS=!D.N_COLORS - 60
 CONTOUR,area,LEVELS=[(INDGEN(10)+1)*MAX(area)/10.],/NOERASE,C_COLORS=!D.N_COLORS - 3
 READ,'OK ? [Y/N]',OK

 PRINT,'Kept region from Original profile'
 temp(*) = 0
 temp(peak_area) = area(peak_area)
; CONTOUR,temp,LEVELS=[(INDGEN(10)+1)*MAX(temp)/10.],/NOERASE,C_COLORS=!D.N_COLORS - 190
 CONTOUR,temp,LEVELS=[(INDGEN(10)+1)*MAX(temp)/10.],/NOERASE,C_COLORS=!D.N_COLORS - 4
 READ,'OK ? [Y/N]',OK

 PRINT,'Difference profile'
 PRINT,'Profile fitted integrated intensity :',integrated(i)
 PRINT,'Scaled box integrated intensity :',box_scale_integ(i)
 SURFACE,integrated(i)*adapted_profile - area
 READ,'OK ? [Y/N]',OK

 PRINT,'Background level for ref profile'
 dum=FLTARR(2*par.boxsize.x,2*par.boxsize.y)
 dum(FIX(par.boxsize.x/2.):FIX(par.boxsize.x/2.)+par.boxsize.x-1,FIX(par.boxsize.y/2.):FIX(par.boxsize.y/2.)+par.boxsize.y-1)=adapted_profile
 SURFACE,dum<Max(dum)/2.,ax=0
 READ,'OK ? [Y/N]',OK

 PRINT,'Background level for area'
 dum=FLTARR(2*par.boxsize.x,2*par.boxsize.y)
 dum(FIX(par.boxsize.x/2.):FIX(par.boxsize.x/2.)+par.boxsize.x-1,FIX(par.boxsize.y/2.):FIX(par.boxsize.y/2.)+par.boxsize.y-1)=area
 SURFACE,dum<Max(dum)/2.,ax=0
 READ,'OK ? [Y/N]',OK

ENDIF

;***
;get an estimate of the goodness of fit (chi_square_test)
;***

;the gain is (see below)
gain = par.gain*1.5418/spot_list(i).l


;the chi_square value is multiplied by chi_mult because experimentally, the
;chi_square test tends to undestimate the fit quality (and it has been checked
;that the esimated variances are not far from corresponding to truth)
chi_square = chi_mult*TOTAL((integrated(i)*adapted_profile(peak_area) - area(peak_area))^2/gain/(1>(area(peak_area) + (1+1/n_back)*(bc_mean - par.dark_current))))

proba = 1 - CHI_SQR1(chi_square,count - 1)
; gammaq(degrees of freedom/2,chi_square/2) = 1 - CHI_SQR1(chi_square,degrees of freedom) (see Numerical recipes in C. p.525)

;Do real box integration if low chi_square test
;IF proba LT 0 THEN BEGIN
;Do real box integration if required
IF KEYWORD_SET(get_box) THEN BEGIN

;print,'i========',i
;if i eq 5220 then stop
 ;where is the maximum : if it is too far, rely on the profile fit.
 tmp_center = (WHERESUB(WHERE(area EQ MAX(area(peak_area))),area))(0)
; IF SQRT(ABS(FLOAT(tmp_center)-par.boxsize.x/2)^2 + ABS(IMAGINARY(tmp_center)-par.boxsize.y/2)^2) GE 3 THEN BEGIN
  ;PRINT,'Large shift : ',ABS(FLOAT(tmp_center)-par.boxsize.x/2)>ABS(IMAGINARY(tmp_center)-par.boxsize.y/2)
;  box2_scale_integ(i)=integrated(i)
; ENDIF ELSE BEGIN ;if the shift is not too big then do box integration.
  IF MAX(par.overall_region) NE 0 THEN area(par.overall_region)=0
;  area(*par.regions(spot_list(i).sector-1))=0
  area(*par.regions(spot_list(i).sector))=0

  ;get the optimum cutoff level [% of MAX(area)]
  cutoff = GET_BOXLEVEL(area,bc_coeff(1,i),bc_coeff(2,i),par.dark_current)
 
  ;get the best contour for integration
  region = GET_BOXREG(area,cutoff,par.boxsize)

  ;the integrated intensity is then
  ;IF spot_list(i).res GT 4.5 OR integrated(i) LT 0 THEN BEGIN
  ; PRINT,'PF integrated intensity : ',integrated(i)
  ; PRINT,'Scaled box integrated intensity : ',box_scale_integ(i)
  box2_scale_integ(i) = TOTAL(area(region))
;if (spot_list(i).m(0) EQ 4) AND (spot_list(i).m(1) EQ 20) AND (spot_list(i).m(2) EQ 11) then stop
  ;PRINT,'Real box integrated intensity : ',box_scale_integ(i)
  ; PRINT,'Resolution : ',spot_list(i).res
  ; tmp = image(xpos - FIX(sizearea_x/2.):xpos + FIX(sizearea_x/2.) , $
  ;		ypos - FIX(sizearea_y/2.):ypos + FIX(sizearea_y/2.))
  ; SURFACE,tmp
  ; ok=''
  ; READ,'Ok ?',ok
  ;ENDIF
 ;ENDELSE
ENDIF
;end real box integration

;Do not allow boxscaled integration if there was a strong shift (this can only be done for non-ovlp spots.
IF proba LT 1 THEN BEGIN
;print,'i========',i
;if i eq 5220 then stop
 ;where is the maximum : if it is too far, rely on the profile fit.
 tmp_center = (WHERESUB(WHERE(area EQ MAX(area(peak_area))),area))(0)
 ;get an idea of the FWHM of the adapted profile
 w_fwhm=WHERE(adapted_profile GE MAX(adapted_profile)/2.,ct_fwhm)
 fwhm_radius=(ROUND(SQRT(ct_fwhm/!pi))/2)>2
 IF SQRT(ABS(FLOAT(tmp_center)-par.boxsize.x/2)^2 + ABS(IMAGINARY(tmp_center)-par.boxsize.y/2)^2) GE fwhm_radius THEN BEGIN
  ;PRINT,'Large shift : ',ABS(FLOAT(tmp_center)-par.boxsize.x/2)>ABS(IMAGINARY(tmp_center)-par.boxsize.y/2)
;PRINT,'PF, BOX II : ',integrated(i),box_scale_integ(i)
  IF MAX(area(peak_area))/2.5 GT area(par.boxsize.x/2,par.boxsize.y/2) THEN box_scale_integ(i)=integrated(i)
;  IF MAX(area(peak_area))/2.5 GT area(par.boxsize.x/2,par.boxsize.y/2) THEN PRINT,'CHANGED BACK ! TO PF VALUE'
;PRINT,'Radius for FWHM [raster] : ',fwhm_radius
;PRINT,'Max of area : ',MAX(area(peak_area))
;PRINT,'Value at center :',area(par.boxsize.x/2,par.boxsize.y/2)
;ok=''
;READ,'ok?',ok
 ENDIF
ENDIF
;end do not allow boxscaled integration

IF KEYWORD_SET(second_interp) AND proba LT 0.95 AND re_integ_flag EQ 0 AND flag EQ 0 THEN BEGIN ; reintegrate if not perfect
 flag=0
 ;redoing the calculation can only be done once
 re_integ_flag = 1

 ;get a better adapted profile
 IF KEYWORD_SET(show_weak) THEN BEGIN
  PRINT,'Chi square test is : ', proba
  PRINT,'Will try to interpolate !'
  adapted_profile=GET_BEST_PROFILE3(adapted_profile,area,peak_area,w_max_corr,/SHOW)
 ENDIF ELSE adapted_profile=GET_BEST_PROFILE3(adapted_profile,area,peak_area,w_max_corr)
  adapted_profile = adapted_profile/TOTAL(adapted_profile)
 GOTO,re_integ
ENDIF

;the calculation should be rejected if chi_square_test is too low and background was correct (ne 1)
;ie proba < 0.001
IF proba LE 0.001 AND flag NE 1 THEN flag = 2

chi_square_test(i) = proba
success_flag(i) = flag > ovlp_err
;reset ovlp_err
ovlp_err = 0

;***
;get an estimate of the variance
;***
;the gain is supposed to be proportionnal to X-ray energy
;par.gain is the gain at 1.5418 A

peak_variance(i) = SQRT(GET_PFVAR(adapted_profile,area+background,fit_level,n_back,bc_mean,par.gain*1.5418/spot_list(i).l,par.bg_gain,par.dark_current))


;***
;PRINT INFO
;***
IF KEYWORD_SET(show_weak) THEN BEGIN
 PRINT,'Chi square test :', proba
 PRINT,'Chi square value :',chi_square
 PRINT,'Value for which chi_square test = 0.001 :',CHI_SQR1(0.001,count-1)
 PRINT,'Ratio [>1 good, < 1 bad] :',CHI_SQR1(0.001,count-1)/chi_square

 PRINT,'Flag [0 : OK ; 1 : bad background ; 2 : bad ii] :',flag
 PRINT,'Peak variance :',peak_variance(i)
ENDIF

;IF flag EQ 0 AND re_integ_flag EQ 0 THEN PRINT,FORMAT = '($, "*")'
;IF flag EQ 0 AND re_integ_flag EQ 1 THEN PRINT,FORMAT = '($, "+")'
;IF flag EQ 1 THEN PRINT,FORMAT = '($, "b")'
;IF flag EQ 2 THEN PRINT,FORMAT = '($, "i")'
;IF flag GT 2 THEN PRINT,FORMAT = '($, "s")'

stat.non_ovlp=stat.non_ovlp+1
IF flag EQ 1 THEN stat.bad_bg = stat.bad_bg + 1
IF flag EQ 2 THEN stat.bad_ii = stat.bad_ii + 1
IF flag GT 2 THEN stat.bad_other = stat.bad_other + 1

ENDIF ; for the case of already treated spots
escape_non_ovlp:
ENDELSE ; case for a non overlapped spot

IF KEYWORD_SET(show_m) THEN BEGIN
show_ovlp = 0
show_weak = 0
ENDIF

escape2:

;print statistics
IF (i MOD 100) EQ 0 THEN PRINT,FORMAT=stat_format,stat,i+1,100.*(i+1)/FLOAT(number_spots),cr

ENDFOR  ;end of the main loop going on for each spot.

PRINT,''
PRINT,'  --> Number of spots integrated :',number_spots
PRINT,''

RETURN,integrated

END

