function GET_REF_PROF7, image,red,ref,par,max_number_peaks,min_sob,ref_distance,ref_ang_width,min_ref_peaks,show=show
;+
; NAME:
;	GET_REF_PROF7
;
; PURPOSE:
;	Select reference peaks for profile fitted IDL integration software.
;	A maximum of number_peaks peaks are selected, which all show a 
;	minimum peak_value/background equal to min_sob. These peaks are
;	carefully checked for FWHM, dead pixels and spikes or spurious
;	peaks
;
; CATEGORY:
;	Laue processing.
;
; CALLING SEQUENCE:
;	Reference_profiles = GET_REF_PROF7(image,ref,par,number_peaks,
;		min_sob,ref_distance,ref_ang_width,/show)
;
; INPUTS:
;	image : the original image
;	red : the rebinned version of image
;	ref : the reflexion list, corrected for positions, produced by the 
;			MATCHLAUE4.PRO routine.
;	par : the parameters of the experiment
;	max_number_peaks : the maximum number of peaks to select
;	min_sob : the minimum peak_value to background ratio to select a peak
;	ref_distance : the maximum radial distance to look for a reference peak
;	   	from the position of the currently integrated peak [mm]
;	ref_ang_width : the maximum tengential distance to look for a 
;		reference peak from the position of the currently integrated 
;		peak [degrees]
;	show : keyword set if the reference profile is to be visualized
;
; OUTPUTS:
;	reference_profile : a structure containing the reference profiles and
;			their parameters.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None.
;
; PROCEDURE:
;	Straightforward.	
;
; MODIFICATION HISTORY:
;	D.Bourgeois, February 1995.
;-


;start the procedure if some peaks were selected and sort the peak according
;to peak maximum.

IF (SIZE(ref))(1) GT 1 THEN sort_ref = (ref(REVERSE(SORT(ref.intp(0)))))(0:((max_number_peaks-1)<((SIZE(ref))(1)-1))) ELSE MESSAGE,'Error : no spot selected !'


;initialize some parameters.
iteration=0  ; iteration number
number_spots=0 ; number of satisfactory spots (having passed the filters)
old_number_spots=0 ; number of satisfactory spots in previous cycle
number_ini_spots=0 ; number of spots satisfying the I/SIGI criterium (before filters)
min_sob2=1.05*min_sob
min_ref_peaks2=min_ref_peaks
constraints=1. ; constraints coefficient for filters

PRINT,'Minimum number of reference peaks set to : ',min_ref_peaks 

REPEAT BEGIN
	min_sob2=min_sob2/1.05

	IF (min_sob2 le 0.1) THEN MESSAGE, 'Can not find reference peaks!' 

	;we select for the peaks of sufficient signal to background ratio
	possible_ini_index = WHERE(sort_ref.intp(0)/sort_ref.bc_coeff(2) GT min_sob2)

	IF (SIZE(possible_ini_index))(0) GT 0 THEN poss_ini_ref = sort_ref(possible_ini_index) $
	ELSE MESSAGE,'Error, no spot selected ... ' 

	;the number of potential spots is :
	number_ini_spots = (SIZE(possible_ini_index))(1)
;	PRINT,'Number of peaks with sufficient I/Sig(I) :',number_ini_spots

ENDREP UNTIL  (number_ini_spots GE 1.25*min_ref_peaks)

	PRINT,'I/SIGI limit set to : ', min_sob2


REPEAT BEGIN

;	PRINT,'***Number of selectable spots :',number_spots
;	PRINT,'***Old number of selectable spots :',old_number_spots
;	PRINT,'***Number of spots with good I/SIGI :',number_ini_spots
;	PRINT,'***Number of iteration :',iteration

IF ((number_ini_spots GE 2*number_spots) AND (iteration GE 1)) THEN BEGIN
	;only release constraints on the filters
	constraints=0.8*constraints
	PRINT,'Reducing constraints ...'
	min_ref_peaks2=FIX(min_ref_peaks2/1.2)
	PRINT,'Minimum number of reference peaks reduced to : ',min_ref_peaks2 
ENDIF 

IF ((number_ini_spots LT 2*number_spots) AND (iteration GE 1)) THEN BEGIN

	;only release constraints on the sob
	min_sob2=min_sob2/1.1

	IF (min_sob2 le 0.1) THEN MESSAGE, 'Can not find reference peaks!' 
	PRINT,'Reducing I/SIGI limit to : ', min_sob2

	;we select for the peaks of sufficient signal to background ratio
	possible_ini_index = WHERE(sort_ref.intp(0)/sort_ref.bc_coeff(2) GT min_sob2)

	IF (SIZE(possible_ini_index))(0) GT 0 THEN poss_ini_ref = sort_ref(possible_ini_index) $
	ELSE MESSAGE,'Error, no spot selected ... ' 

	;the new number of potential spots is :
	number_ini_spots = (SIZE(possible_ini_index))(1)
	PRINT,'Number of peaks with sufficient I/Sig(I) :',number_ini_spots
ENDIF


IF ((old_number_spots GE (SIZE(spots))(1)-1) AND (iteration GE 1)) THEN BEGIN
;in this case the number of spots having passed the filters has not increased
	PRINT,'Too few good spots ...'
	PRINT,'Minimum number of reference peaks reduced to : ',(SIZE(spots))(1)
	GOTO,set_ok 
ENDIF

;save old number of spots 
old_number_spots = (SIZE(spots))(1)

;*****
;get all spots of interest  (as structures) :
;*****

ini_spots = GET_SPOTS(poss_ini_ref,image,red,par)

IF KEYWORD_SET(show) THEN BEGIN ;show
main_window = !D.WINDOW
WIN,512,512
sim_window = !D.WINDOW
ENDIF ;show

;*****
;now apply all filters
;*****
filters :

;selection of good reference spots : all spots are checked for
;FWHM and then compared with the global mean profile and checked 
;for deviation :

;******
;remove any bad FWHM
;******

sigma = STDEV(ini_spots.fwhm,mean)
possible_index = WHERE((ini_spots.fwhm - mean)^2 LE (1/constraints)*5.0*sigma^2,count)

IF count EQ 0 THEN MESSAGE,'Not a single good reference spot [FWHM] !'
spots = ini_spots(possible_index)
poss_ref = poss_ini_ref(possible_index)

PRINT,'Number of reference peaks rejected from bad FWHM :',number_ini_spots-count

number_spots=(SIZE(spots))(1)


;******
;remove spots with bad background 
;******

IF MAX(spots.flag) NE 0 THEN BEGIN
possible_index = WHERE(spots.flag EQ 0,count)

IF count EQ 0 THEN MESSAGE,'Not a single good reference spot [BACKGROUND] !'
PRINT,'Number of spots rejected over bad background :', (SIZE(spots))(1) - (SIZE(possible_index))(1)

spots = spots(possible_index)
poss_ref = poss_ref(possible_index)

;the number of potential spots is now:

number_spots = (SIZE(spots))(1)

ENDIF ELSE PRINT,'Number of spots rejected over bad background :',0


;******
;remove spots which are shifted
;******

FOR i=0,number_spots-1 DO spots(i).intp(1)=MAX(spots(i).sp)
bs=par.boxsize
bsx2 = FIX(par.boxsize.x/2.)
bsy2 = FIX(par.boxsize.y/2.)

;possible_index = WHERE( spots.sp(FIX((bs.y+1)*bs.x/2.-bs.x-1)) EQ LONG(spots.intp(1)) OR $
;spots.sp(FIX((bs.y+1)*bs.x/2.-bs.x)) EQ LONG(spots.intp(1)) OR $
;spots.sp(FIX((bs.y+1)*bs.x/2.-1)) EQ LONG(spots.intp(1)) OR $
;spots.sp(FIX((bs.y+1)*bs.x/2.)) EQ LONG(spots.intp(1)),count)

possible_index = WHERE( $
spots.sp(FIX(bs.x*bsy2+bsx2)) EQ LONG(spots.intp(1)) OR $
spots.sp(FIX(bs.x*bsy2+bsx2-1)) EQ LONG(spots.intp(1)) OR $
spots.sp(FIX(bs.x*bsy2+bsx2+1)) EQ LONG(spots.intp(1)) OR $
spots.sp(FIX(bs.x*(bsy2-1)+bsx2)) EQ LONG(spots.intp(1)) OR $
spots.sp(FIX(bs.x*(bsy2-1)+bsx2-1)) EQ LONG(spots.intp(1)) OR $
spots.sp(FIX(bs.x*(bsy2-1)+bsx2+1)) EQ LONG(spots.intp(1)) OR $
spots.sp(FIX(bs.x*(bsy2+1)+bsx2)) EQ LONG(spots.intp(1)) OR $
spots.sp(FIX(bs.x*(bsy2+1)+bsx2-1)) EQ LONG(spots.intp(1)) OR $
spots.sp(FIX(bs.x*(bsy2+1)+bsx2+1)) EQ LONG(spots.intp(1)) $
,count)

IF count EQ 0 THEN MESSAGE,'Not a single good reference spot [SHIFT] !'
PRINT,'Number of spots rejected because too large shift :', (SIZE(spots))(1) - (SIZE(possible_index))(1)

spots = spots(possible_index)
poss_ref = poss_ref(possible_index)

;the number of potential spots is now:

number_spots = (SIZE(spots))(1)



;******
;remove any spurious peak or spike
;******
number_spike = 0
number_dead = 0

good_spots = spots
j = 0

FOR i=0,number_spots-1 DO BEGIN ;go through all spots
peak = spots(i).sp - spots(i).bc

;get the cutoff level
cutoff = (1/constraints)*GET_BOXLEVEL(peak,spots(i).bc_coeff(1),spots(i).bc_coeff(2),par.dark_current)

;is there a spike or a dead pixel ?
spike = GET_SPIKE(peak,cutoff,par.boxsize,show=show)
number_spike = number_spike + spike
dead_pix = GET_DEADPIX(peak,cutoff,par.boxsize,show=show)
number_dead = number_dead + dead_pix

IF (spike EQ 0) AND (dead_pix EQ 0) THEN BEGIN
good_spots(j) = spots(i)
j = j + 1
ENDIF


IF KEYWORD_SET(show) THEN BEGIN

IF i ge 0 THEN BEGIN
PRINT,'Spot number :',i
PRINT,'Miller indices :',spots(i).m

PRINT,'Original peak'
SURFACE,peak
OK=''
PRINT,'OK ? [Y/N]'
READ,OK

ENDIF

ENDIF
;****

ENDFOR ; going through all spots

IF j GT 0 THEN spots = good_spots(0:j-1) ELSE MESSAGE,'Not a single good reference spot !'

PRINT,'Number of spots rejected for spikes or too uneven background :',number_spike
PRINT,'Number of spots rejected for dead pixels :',number_dead
PRINT,'Number of selected spots :',(SIZE(spots))(1)

number_spots = (SIZE(spots))(1)
iteration=iteration+1

ENDREP UNTIL ((SIZE(spots))(1) ge min_ref_peaks2)

set_ok:
;****
IF KEYWORD_SET(SHOW) THEN BEGIN ;begin show
FOR i=0,number_spots-1 DO BEGIN
 dum = spots(i).sp - spots(i).bc
PRINT,'PosMax :',WHERESUB(WHERE(dum EQ MAX(dum)),dum)
ENDFOR

FOR i=0,(SIZE(spots))(1)-1 DO BEGIN
;SURFACE,spots(i).sp
WSET,main_window
dummy = PIXLAUE2(spots(i),red,image,par,draw=8,color=3)
;OK=''
;READ,'OK ? [Y/N]',OK
WSET,sim_window
ENDFOR
ENDIF ; show

RETURN,spots
END


