function SCALE, arr1, arr2, w=w, type=type, k=k, b=b, ssq=ssq, $
		convk=convk, convb=convb
;+
; NAME:
;	SCALE
;
; PURPOSE:
;	Calculate a scale factor between arr1 and arr2.
;
; CATEGORY:
;	Math, Crystallography
;
; CALLING SEQUENCE:
; 	scale = SCALE(arr1,arr2 [, w=w, type=type, k=k, b=b, ssq=ssq, $
;		convk=convk, convb=convb])
;
; INPUTS:
;	ARR1,ARR2: The arrays containing the data to be scaled.
;
; KEYWORDED PARAMETERS:
;	W: Weighting. Array of the same dimension as ARR1 and ARR2.
;		ARR1 and ARR1 are multplied by Sqrt(W) during the scaling
;		(<=> Weight=W). 
;	TYPE: Type of scale factor. String equal to 
;		'sum' => K= Total(arr2)/Total(arr1)
;		         The function returns K.
;		'lsq' => K= Least square fitted scale factor.
;		         The function returns K.
;		'blsq'=> Least square fit of K and B. 
;		         The function returns K*exp(-B*s^2) (Array).
;		'kb'  => Use the supplied K and B to do a K,B-scaling.
;		Default is 'lsq' unless the keyword SSQ is set then default
;		is 'blsq'.
;	K: The global scale factor.
;	B: The global B-factor. Normally if B-factor scaling is requsted
;		then K*exp(-B*s^2) is returned as an array and K and B
;		can be read by the keywords K and B.
;	SSQ: An array of the same dimensions as ARR1 and ARR2 containing
;		the square s-value of the reflection (s^2=(d*/2)^2=(1/2d)^2). 
;		If given then the deafult TYPE changes.
;	CONVK,CONVB: For TYPE='blsq'. If the shifts in k and b are smaller
;		then these valuse the refinement is considered as converged.
;		Default 0.01.
;
; OUTPUTS:
;	Returns the scale factor(s). ( arr1= SCALE(arr1,arr2)*arr1 )
;	Either a real number or an array of real numbers of the same
;	dimensions as ARR1 and ARR2. (See type).
;	K: The global scale factor.
;	B: The global B-factor. Normally if B-factor scaling is requsted
;		then K*exp(-B*s^2) is returned as an array and K and B
;		can be read by the keywords K and B.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None
;
; RESTRICTIONS:
;	None
;
; PROCEDURE:
;	Straightforward
;
; MODIFICATION HISTORY:
;	Thomas Ursby, February 1995
;-

;ON_ERROR, 2

IF NOT KEYWORD_SET(w) THEN w=1.0
IF NOT KEYWORD_SET(type) THEN $
  IF KEYWORD_SET(ssq) THEN type='blsq' $
    ELSE type='lsq'
IF NOT KEYWORD_SET(convk) THEN convk=0.01
IF NOT KEYWORD_SET(convb) THEN convb=0.01
i1=sqrt(w)*arr1 & i2=sqrt(w)*arr2

CASE (type) OF
  'sum' : BEGIN
	    scale= TOTAL(i2)/TOTAL(i1)
	    k=scale
          END
  'lsq' : BEGIN
	    scale= TOTAL(i1*i2)/TOTAL(i1*i1)
	    k=scale
          END
  'blsq': BEGIN
      deltak=2*convk & deltab=2*convb & cycles=0
      k= TOTAL(i1*i2)/TOTAL(i1*i1) & b=0.0
      WHILE ((abs(deltak) gt convk)or(abs(deltab) gt convb)) DO BEGIN
        y= i2 - k*i1*exp(-b*ssq)
	a1= i1*exp(-b*ssq)
        a2= -k*i1*ssq*exp(-b*ssq)
	a= [[a1],[a2]]
;	x= invert(transpose(a) # a, status, /DOUBLE) $
	x= invert(transpose(a) # a, status)  $
		# transpose(a) # y
	IF (status eq 1) THEN MESSAGE, 'Matrix inversion failed.'
	IF (status eq 2) THEN PRINT, 'Matrix inversion badly conditioned.'
	deltak= x(0) & deltab= x(1)
	k=k+deltak & b=b+deltab
        cycles=cycles+1
      ENDWHILE
      scale= k*exp(-b*ssq)
    END
  'kb'  : BEGIN
      scale= k*exp(-b*ssq)
    END
  ELSE  : PRINT, 'Type ',type,' not allowed.'
ENDCASE

RETURN, scale

END