function READ_SIGMAA, filename, res, lnsiga, plot=plot 
;+
; NAME:
;	READ_SIGMAA
;
; PURPOSE:
;	Read log file from Sigmaa and extracts the RMS error coordinates.
;	If the PLOT keyword is set the fit will be plotted.
;
; CATEGORY:
;	Crystallography.
;
; CALLING SEQUENCE:
; 	READ_SIGMAA, filename [,/plot]
;
; INPUTS:
;	FILENAME: Name of the Sigmaa log file.
;
; KEYWORDED PARAMETERS:
;	PLOT: If set then the Sigmaa plot is displayed, with a least
;		line fitted on top of it.
;		The Sigmaa plot is : 
;		Ln(SigA) = 1/2*Ln(SigP/SigN) - Pi^3*(Delta_r)^2*(Sin(Theta)/lambda)^2
;		Ie : If Ln(SigA) is plotted as function of (Sin(Theta)/lambda)^2, a
;		straight line is obtained with slope = -Pi^3(Delta_r)^2, where 
;		Delta_r is the RMS coordinate error.
;
;		
; OUTPUTS:
;	Delta_r : The RMS coordinate error.
;	LnSigA : [Optional] the sigmaa values in bins of (Sin(Theta)/lambda)
;	Res : [Optional] the resolution bins (Sin(Theta)/lambda).
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None
;
; RESTRICTIONS:
;
; PROCEDURE:
;	Straightforward
;
; MODIFICATION HISTORY:
;	Dominique Bourgeois, March 1996
;-


dummy=''
OPENR, file, filename, /GET_LUN

; Find and read the number of lambda ranges:
searchstr = ' DATA for SIGMAA PLOT'
start= 0
length = STRLEN(searchstr)
WHILE (NOT EOF(file)) AND $
	(NOT (STRMID(dummy,start,length) EQ searchstr)) DO BEGIN
  READF,file,dummy
ENDWHILE
IF EOF(file) THEN GOTO, file_error

READF,file,dummy
searchstr = ' ln(SigmaA) vs sthol**2 should give straight line with'
start= 0
length = STRLEN(searchstr)
format = '(2F10.6)'
res=FLTARR(100) & lnsiga=FLTARR(100)
stholsq = 0. & lnsigmaa = 0.
i=0
REPEAT BEGIN
  READF,file,dummy
IF (STRMID(dummy,start,length) NE searchstr) THEN BEGIN
  READS,dummy,FORMAT=format,stholsq,lnsigmaa
  res(i) = SQRT(stholsq) & lnsiga(i) = lnsigmaa
  i = i + 1
ENDIF
ENDREP UNTIL ((EOF(file)) OR (STRMID(dummy,start,length) EQ searchstr))

res = res(0:i-1) & lnsiga = lnsiga(0:i-1)

;do the fit
x = res^2 & y = lnsiga
w = 0 & nfit = 0

REPEAT BEGIN
poly = POLY_FIT(x,y,1,yfit,yband,sigma)
w = WHERE(ABS(yfit-y) LE 1.5*sigma,ct)

yold=y

IF ((SIZE(y))(1)-ct) GT 0 THEN BEGIN 
x=x(w) & y=y(w)
ENDIF
nfit = nfit + 1
ENDREP UNTIL (SIZE(w))(1) EQ (SIZE(yold))(1) 

slope = poly(1) & intercept = poly(0)
delta_r = SQRT(-slope/26.3119)

PRINT,'RMS coordinate error [A]: ', delta_r
PRINT,'Number of cycles to fit : ', nfit

FREE_LUN, file
; Plot the last curve as a solid line:
IF KEYWORD_SET(plot) THEN BEGIN
    PLOT, res^2, lnsiga, PSYM=1,TITLE='SIGMAA PLOT',$
		XTITLE='(Sin(theta)/Lambda)^2',YTITLE='Ln(Sigmaa)',CHARSIZE=1.5
    OPLOT, x,y, PSYM=2
    OPLOT, x,yfit, LINESTYLE=0
PRINT,'Crosses are rejected points'
PRINT,'Stars are accepted points'
ENDIF

RETURN, delta_r

file_error: 
  FREE_LUN, file
  MESSAGE, 'Error reading file.'

END

