pro READ_LN, filename, bins, lambda, pol, ranges, polorder, lmin, lmax, $
		plot=plot, oplot=oplot, last_plot=last_plot, $
		x=x, y=y, $
		rfactor=rfactor, scale=scale, k=k, b=b
;+
; NAME:
;	READ_LN
;
; PURPOSE:
;	Read log file from Lauenorm and extract:
;		*The normalisation bin scale factors
;		*The fitted polynomial
;	All iterations are read and the last one is returned.
;	If the PLOT keyword is set all the fits will be plotted
;	together with the bin scale factors (inverted!) and
;	the last fit in the file will be highlighted.
;
; CATEGORY:
;	Input, Laue processing
;
; CALLING SEQUENCE:
; 	READ_LN, filename [, bins, lambda, pol, ranges, polorder, lmin, lmax, $
;			/plot, /oplot, /last_plot$
;			x=x, y=y, $
;			/rfactor, /scale, k=k, b=b ]
;
; INPUTS:
;	FILENAME: Name of the Lauenorm log file.
;
; KEYWORDED PARAMETERS:
;	PLOT: If set then the bin factors and the fit are plotted
;		(in the current graphics window). NOTE: What is
;		plotted are the inverted scale factors and polynomial(s).
;		The plots are normalised to maximum 1.0
;	OPLOT: Same as PLOT but the curves are overplotted on the current
;		graph.
;	LAST_PLOT: Only in combination with PLOT or OPLOT. If set then
;		only the last curve is plotted.
;	RFACTOR: If set then the file is searched for all R-factors
;		(lines starting with  RFACT) and the rfactors are
;		printed.
;	SCALE: If set then the scale factors and the B-factors of the 
;		different film packs will be output to K and B. 
;
; OUTPUTS:
;	Optional:
;	BINS: The bin scale factors. 
;	LAMBDA: The wavelength of the bin scale factors.
;	POL: The polynomial coefficients. DBLARR(ranges,max(polorder)+1)
;	RANGES: The number of lambda ranges.
;	POLORDER: The order(s) of the polynom(s). INTARR(ranges)
;	LMIN,LMAX: The minimum and maximum lambda in the ranges.
;		FLTARR(ranges)
;	X,Y: Two vectors with x,y-coordinates for the last plot.
;		Supply as arguments for PLOT if you want to redo the plot.
;		They are DBLARR(number_of_ranges,plot_points). 
;		(plot_points=100)
;	K,B: The scale and B-factors of the different film packs (see SCALE).
;		k,b=FLTARR(film_packs) where (the number of) film_packs is
;		read from the file. 
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None
;
; RESTRICTIONS:
;	The parsing of the logfile heavily depends on the output format
;	of Lauenorm: That it was correctly interpreted and that it hasn't
;	changed (or is different on another machine). Further the routine
;	will not work for the case a polynomial was input to the program
;	and applied (and not refined in the program); use READ_LN_SCALE 
;	in this case. The routine will probably also have problems in some
;	other unusual(?) configurations. 
;
; PROCEDURE:
;	Straightforward
;
; MODIFICATION HISTORY:
;	Thomas Ursby, February 1995
;	Modified to read several ranges and improved. July 1995.
;	Modified to read the scale factors. November 1995.
;-

ON_ERROR, 2
plot_points=100
ok='' & dummy='' & ranges=0 & dummyint=0
IF (KEYWORD_SET(plot) or KEYWORD_SET(oplot)) THEN first_plot=1 $
ELSE first_plot=0

OPENR, file, filename, /GET_LUN

; Find and read the number of film packs:
searchstr = ' NUMBER OF FILM PACKS ='
start= 0
length = STRLEN(searchstr)
WHILE (NOT EOF(file)) AND $
	(NOT (STRMID(dummy,start,length) EQ searchstr)) DO BEGIN
  READF,file,dummy
ENDWHILE
IF EOF(file) THEN GOTO, file_error
READS, STRMID(dummy,length,50), film_packs

; Find and read the number of lambda ranges:
searchstr = ' NUMBER OF LAMBDA RANGES ='
start= 0
length = STRLEN(searchstr)
WHILE (NOT EOF(file)) AND $
	(NOT (STRMID(dummy,start,length) EQ searchstr)) DO BEGIN
  READF,file,dummy
ENDWHILE
IF EOF(file) THEN GOTO, file_error
READS, STRMID(dummy,length,50), ranges

; Find and read the number of iterations (of lambda and film 
; pack scaling):
searchstr = ' NUMBER OF ITERATIONS OF SCALING PROCEDURE    ='
start= 0
length = STRLEN(searchstr)
WHILE (NOT EOF(file)) AND $
	(NOT (STRMID(dummy,start,length) EQ searchstr)) DO BEGIN
  READF,file,dummy
ENDWHILE
IF EOF(file) THEN GOTO, file_error
READS, STRMID(dummy,length,50), iterations

; Find and read the number of cycles of lambda bin scaling in each
; iteration:
searchstr = ' NUMBER OF CYCLES OF REFINEMENT PER ITERATION ='
start= 0
length = STRLEN(searchstr)
WHILE (NOT EOF(file)) AND $
	(NOT (STRMID(dummy,start,length) EQ searchstr)) DO BEGIN
  READF,file,dummy
ENDWHILE
IF EOF(file) THEN GOTO, file_error
READS, STRMID(dummy,length,50), lbin_cycles

; Find and read the number of cycles of film pack scaling in each
; iteration:
searchstr = ' NUMBER OF CYCLES OF REFINEMENT PER ITERATION ='
start= 0
length = STRLEN(searchstr)
WHILE (NOT EOF(file)) AND $
	(NOT (STRMID(dummy,start,length) EQ searchstr)) DO BEGIN
  READF,file,dummy
ENDWHILE
IF EOF(file) THEN GOTO, file_error
READS, STRMID(dummy,length,50), film_cycles

nb_bins= INTARR(ranges) & polorder= INTARR(ranges)
lmin= FLTARR(ranges) & lmax= FLTARR(ranges)
FOR i=0,ranges-1 DO BEGIN
  ; Find and read the lambda limits:
  searchstr = ' MINIMUM LAMBDA ='
  start= 0 & lmin0=0.0 & lmax0=0.0
  length = STRLEN(searchstr)
  WHILE (NOT EOF(file)) AND $
	(NOT (STRMID(dummy,start,length) EQ searchstr)) DO BEGIN
    READF, file, dummy
  ENDWHILE
  IF EOF(file) THEN GOTO, file_error
  dummy=STRMID(dummy,length,STRLEN(dummy))
  READS,dummy,lmin0
  dummy=STRMID(dummy,STRPOS(dummy,'LAMBDA =')+8,STRLEN(dummy))
  READS,dummy,lmax0
  lmin(i)=lmin0 & lmax(i)=lmax0

  ; Find and read the number of bins:
  searchstr = ' BINS PARAMETER ='
  start= 0
  length = STRLEN(searchstr)
  WHILE (NOT EOF(file)) AND $
	(NOT (STRMID(dummy,start,length) EQ searchstr)) DO BEGIN
    READF, file, dummy
  ENDWHILE
  IF EOF(file) THEN GOTO, file_error
  READS, STRMID(dummy,length,50), dummyint
  nb_bins(i) = dummyint

  ; Find and read the order of polynomial fit:
  searchstr = ' ORDER FOR CURVE FITTING ='
  start= 0
  length = STRLEN(searchstr)
  WHILE (NOT EOF(file)) AND $
	(NOT (STRMID(dummy,start,length) EQ searchstr)) DO BEGIN
    READF,file,dummy
  ENDWHILE
  IF EOF(file) THEN GOTO, file_error
  READS, STRMID(dummy,length,50), dummyint
  polorder(i)= dummyint < (nb_bins(i)-1)
ENDFOR

bins = DBLARR(TOTAL(nb_bins))
lambda = DBLARR(TOTAL(nb_bins))
x= DBLARR(ranges,plot_points)
y= DBLARR(ranges,plot_points)
pol= DBLARR(ranges,max(polorder)+1)

WHILE NOT EOF(file) DO BEGIN
  FOR i=0,ranges-1 DO BEGIN
    ; Find the polynomial coefficients:
    searchstr = ' POLYNOMIAL COEFFICIENTS'
    start= 0
    length = STRLEN(searchstr)
    WHILE (NOT EOF(file)) AND $
	(NOT (STRMID(dummy,start,length) EQ searchstr)) DO BEGIN
      READF, file, dummy
    ENDWHILE

    ; Just to save time and avoiding replotting the last plot:
    IF EOF(file) THEN GOTO, file_close

    ; Read the polynomial coefficients:
    sdummy=''
    searchstr = ' MEAN SQUARE DEVIATIONS'
    start  = 0
    length = STRLEN(searchstr)
    IF NOT EOF(file) THEN READF, file, dummy
    WHILE (NOT EOF(file)) AND $
	(NOT (STRMID(dummy,start,length) EQ searchstr)) DO BEGIN
      sdummy=sdummy+' '+dummy 
      READF, file, dummy
    ENDWHILE
    IF EOF(file) THEN GOTO, file_error
    dummyarr= pol(i,0:polorder(i))
    READS, sdummy, dummyarr
    pol(i,0:polorder(i))= dummyarr
  ENDFOR

  ; Find and read the bin scale factors:
  binstr = { nb:0, l:0.0, nb_ref:0L, sc:0.0, psc:0.0 }
  searchstr = '1C U R V E   F I T T I N G   A N A L Y S I S'
  start  = 0
  length = STRLEN(searchstr)
  WHILE (NOT EOF(file)) AND $
	(NOT (STRMID(dummy,start,length) EQ searchstr)) DO BEGIN
    READF, file, dummy
  ENDWHILE
  IF EOF(file) THEN GOTO, file_error
  FOR i=0,ranges-1 DO BEGIN
    searchstr = STRING(TOTAL(nb_bins(0:i))-nb_bins(i)+1, FORMAT='(I5)')
    start  = 0
    length = STRLEN(searchstr)
    WHILE (NOT EOF(file)) AND $
	(NOT (STRMID(dummy,start,length) EQ searchstr)) DO BEGIN
      READF, file, dummy
    ENDWHILE
    IF EOF(file) THEN GOTO, file_error
    FOR j=TOTAL(nb_bins(0:i))-nb_bins(i),TOTAL(nb_bins(0:i))-1 DO BEGIN
      READS, dummy, binstr
      bins(j)= binstr.sc
      lambda(j)= binstr.l
      READF, file, dummy
      READF, file, dummy
    ENDFOR
  ENDFOR

  ; Do the plotting
  binrange= DBLARR(ranges)
  IF (first_plot) THEN BEGIN
    FOR i=0,ranges-1 DO BEGIN
      binrange(i)= (lambda(TOTAL(nb_bins(0:i))-1) - $
	lambda(TOTAL(nb_bins(0:i))-nb_bins(i))) / float(nb_bins(i)-1)
      lmin= lambda(TOTAL(nb_bins(0:i))-nb_bins(i)) - binrange(i)/2
      lmax= lambda(TOTAL(nb_bins(0:i))-1) + binrange(i)/2
      x(i,*)= (lmax-lmin)*findgen(plot_points)/(plot_points-1) + lmin
      y(i,*)= 1.0/POLY(x(i,*),pol(0,0:polorder(0)))
    ENDFOR
    lmin0= MIN(x,MAX=lmax0) 
    ymax= MAX(y)
    IF KEYWORD_SET(plot) THEN $
	PLOT, x(0,*), y(0,*)/ymax, $
		linestyle=5, XRANGE=[lmin0, lmax0], YRANGE=[0.0,1.1]
    first_plot=0
  ENDIF
  IF ((KEYWORD_SET(plot) or $
	KEYWORD_SET(oplot)) and not KEYWORD_SET(last_plot)) THEN BEGIN
    FOR i=0,ranges-1 DO $
      y(i,*)= 1.0/POLY(x(i,*),pol(i,0:polorder(i)))
    ymax= MAX(y)
    FOR i=0,ranges-1 DO $
      OPLOT, x(i,*), y(i,*)/ymax, linestyle=5
    OPLOT, lambda, 1.0/(bins*ymax), psym=1
    READ, ok
  ENDIF
ENDWHILE

file_close: FREE_LUN, file

; Plot the last curve as a solid line:
IF (KEYWORD_SET(plot) or KEYWORD_SET(oplot)) THEN BEGIN
  IF KEYWORD_SET(last_plot) THEN BEGIN 
    ; Calculate y (actually only needed if more than one curve in the file).
    FOR i=0,ranges-1 DO $
      y(i,*)= 1.0/POLY(x(i,*),pol(i,0:polorder(i)))
    ymax= MAX(y)
  ENDIF
  IF (KEYWORD_SET(last_plot) and KEYWORD_SET(plot)) THEN BEGIN 
    PLOT, x(0,*), y(0,*)/ymax, $
		linestyle=0, XRANGE=[lmin0, lmax0], YRANGE=[0.0,1.1]
    FOR i=1,ranges-1 DO $
      OPLOT, x(i,*), y(i,*)/ymax, linestyle=0
    OPLOT, lambda, 1.0/(bins*ymax), psym=2
  ENDIF ELSE BEGIN
    FOR i=0,ranges-1 DO $
      OPLOT, x(i,*), y(i,*)/ymax, linestyle=0
    OPLOT, lambda, 1.0/(bins*ymax), psym=2
  ENDELSE
  y=y/ymax
ENDIF

; Read and print R-factors:
IF KEYWORD_SET(rfactor) THEN BEGIN
  OPENR, file, filename, /GET_LUN
  searchstr = ' RFACT'
  start= 0
  length = STRLEN(searchstr)
  REPEAT BEGIN
    READF,file,dummy
    WHILE (NOT EOF(file)) AND $
	(NOT (STRMID(dummy,start,length) EQ searchstr)) DO BEGIN
      READF,file,dummy
    ENDWHILE
    IF NOT EOF(file) THEN PRINT, STRMID(dummy,0,15)
    IF (STRMID(dummy,start,length+1) EQ ' RFACT3') THEN PRINT, ''
  ENDREP UNTIL EOF(file)
  FREE_LUN, file
ENDIF

IF KEYWORD_SET(scale) THEN BEGIN
  OPENR, file, filename, /GET_LUN
  searchstr = ' ITERATION NUMBER'
  start= 0 & found=0 & no_iter=0 & no_cycle=0
  length = STRLEN(searchstr)
  ; If number of lambda bin cycles are greater or equal to the number
  ; of film pack cycles then we should look for the second occurence
  ; of the string below:
  IF (lbin_cycles ge film_cycles) THEN found=-1
  ; Look for the string before the last table of scale factors:
  REPEAT BEGIN
    READF,file,dummy
    IF (STRMID(dummy,start,length) EQ searchstr) THEN BEGIN
      dummy=STRMID(dummy,length,STRLEN(dummy))
      READS,dummy,no_iter
      dummy=STRMID(dummy,STRPOS(dummy,'CYCLE')+5,STRLEN(dummy))
      READS,dummy,no_cycle
      IF ((no_iter eq iterations) and (no_cycle eq film_cycles)) THEN $
	found=found+1
    ENDIF
  ENDREP UNTIL (EOF(file) or (found eq 1))
  IF (found eq 1) THEN BEGIN
    ; If the string was found then read the scale factors:
    READF,file,dummy & READF,file,dummy
    rows=FIX((film_packs+1)/2)
    k=FLTARR(film_packs) & b=FLTARR(film_packs) & table=FLTARR(16,rows)
    READF, file, table
    k(0:rows-1)=table(1,*) & k(film_packs-rows:film_packs-1)=table(9,*)
    b(0:rows-1)=table(2,*) & b(film_packs-rows:film_packs-1)=table(10,*)
  ENDIF ELSE PRINT,'Unable to read scale factors.'
  FREE_LUN, file
ENDIF

RETURN

file_error: 
  FREE_LUN, file
  MESSAGE, 'Error reading file.'

END
