function INTEG_MAP_PEAK, map, selected_peaks, par, radius=radius, sigma_cut=sigma_cut, map_index=map_index, mask_index=mask_index, average=average, only_above=only_above
;+
; NAME:
;	INTEG_MAP_PEAK
;
; PURPOSE:
;	Integrate electron density around part of a PDB model in a density
;	map. 
;
; CATEGORY:
;	Crystallography (and...)
;
; CALLING SEQUENCE:
; 	integrated_density= INTEG_MAP_PEAK(map, selected_peaks, par, 
;	[radius=radius, sigma_cut=sigma_cut], [map_index=map_index], 
;	[mask_index=mask_index], average=average, /only_above])
;
; INPUTS:
;	MAP: The density map (a CCP4 type density map, read by READ_MAP.PRO)
;	SELECTED_PEAKS: Part of a PDB model: part of a structure of the type 
;		returned by READ_PDB2.PRO. These 'peaks' are in fact dummy
;		atoms placed at the position of peak density by the program 
;		mapman
;	RADIUS: The search procedure (search3D) will be carried out in a box
;		with size=2*radius. 
;	PAR: The parameters returned by READ_MAP (in the keyword PAR).
;	MASK_INDEX: supplied as input to restrict integration of grid points 
;		to a mask.
;
; KEYWORDED PARAMETERS:
;	SIGMA_CUT: Sigma cut for the SEARCH3D routine. All grid points in the map
;		within the sigma level at "coodinates" and the "sigma_cut"*par.sigma 
;		level will be considered. Default=3
;	ONLY_ABOVE: If set, only the part of the density which is ABOVE the threhold
;		"sigma_cut"*par.sigma will be considered. So that a peak "just" above 
;		the threshold will give almost a zero intensity, as if the peak was 
;		not there.		
;
; OUTPUTS:
;	Returns the integrated density.
;	AVERAGE: Contains the averaged density within the sphere.
;	MAP_INDEX: supplies the coordinates of the map that were counted in the integration. 
;		Can also be supplied as input to disallow integration of grid points already
;		used in a previous integration.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None
;
; RESTRICTIONS:
;	SEEMS TO RUN ON SGI ONLY
;	MAP_INTEG2 can't handle  par.skew not equal 0 (see READ_MAP
;	    and the definition of the CCP4 map format), i.e.
;	    MAP_INTEG is relying on that the axis are as the PDB-standard
;	    (It is using PDBSCALE to get the transformation matrix
;	     between orthogonal and fractional coordinates. If these
;	     fractional coordinates doesn't correspond to the axis
;	     of the maps then everyhing is wrong.)
;	If the map is not containing the full sphere around the integration
;	     point, then it will integrate only those points available
;	     but not give any error message. (I think...)
;
; PROCEDURE:
;	Straightforward
;
; MODIFICATION HISTORY:
;	D. Bourgeoi, Feb. 2002



IF KEYWORD_SET(sigma_cut) EQ 0 THEN sigma_cut=3
IF KEYWORD_SET(radius) EQ 0 THEN radius=10
s_map = SIZE(map)
IF KEYWORD_SET(map_index) EQ 0 THEN map_index = BYTARR(s_map(1),s_map(2),s_map(3)) 
w0=WHERE(map_index EQ 1,ct0_grid_points) 
int_density=0

IF KEYWORD_SET(only_above) THEN only_above=1 ELSE only_above=0 

IF (SIZE(selected_peaks))(0) NE 0 THEN BEGIN
 n_peaks=(SIZE(selected_peaks))(1)
 xyz=FLTARR(n_peaks,3)
 FOR i=0,n_peaks-1 DO BEGIN

;PRINT,''
;PRINT,'Peak residue: ',selected_peaks(i).rn

  w1=WHERE(map_index EQ 1,ct1_grid_points) 

;print,'coordinates: ',[selected_peaks(i).x,selected_peaks(i).y,selected_peaks(i).z]

 IF KEYWORD_SET(mask_index) THEN $
	int_density = int_density+MAP_INTEG2(map, [selected_peaks(i).x,selected_peaks(i).y,selected_peaks(i).z] , radius, par, sigma_cut=sigma_cut, map_index=map_index, mask_index = mask_index, average=average, only_above=only_above, /force_integration) $
	ELSE int_density = int_density+MAP_INTEG2(map, [selected_peaks(i).x,selected_peaks(i).y,selected_peaks(i).z] , radius, par, sigma_cut=sigma_cut, map_index=map_index, average=average, only_above=only_above)


 ; PRINT,'Integrated intensity: ',int_density
  w2=WHERE(map_index EQ 1,ct2_grid_points)
  new_grid_points=ct2_grid_points-ct1_grid_points 
  PRINT,'Number of new grid points: ',new_grid_points

 ENDFOR
 IF (ct2_grid_points-ct0_grid_points) GT 0 THEN average=int_density/FLOAT(ct2_grid_points-ct0_grid_points) ELSE average = 0
 PRINT,'Integrated density: ',int_density
ENDIF 
RETURN,int_density

END


