function CHEBYCHEV, coeff, data,lmin=lmin,lmax=lmax,lnorm=lnorm
;+
; NAME:
;	CHEBYCHEV
;
; PURPOSE:
;	Calculates Chebytchev polynomia.
;
; CATEGORY:
;	Signal, image processing.
;
; CALLING SEQUENCE:
;	y = CHEBYCHEV(coeff,data,[lmin=lmin],[lmax=lmax],[lnorm=lnorm]) 
;
; INPUTS:
;	coeff :	The chebytchev coefficients
;	data : The data to which the tchebytchev polynomia are applied. 
;	lmin : the minimum value of data (minimum wavelength) to consider
;	lmax : the maximum value of data (maximum wavelength) to consider
;	lnorm : the nromalization value of data (normalization wavelength) to consider (will give a value of 1)
;
; OUTPUTS:
;	Returns an array (1D) containing chebytchev(coeff, data)
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None.
;
; PROCEDURE:
;	Straightforward.
;
; MODIFICATION HISTORY:
;	D.Bourgeois, October 98.
;-

IF N_ELEMENTS(coeff) EQ 0 OR N_ELEMENTS(data) EQ 0 THEN MESSAGE,'Need to give parameters'

IF N_ELEMENTS(lmin) EQ 0 THEN min_data = MIN(data) ELSE min_data=lmin
IF N_ELEMENTS(lmax) EQ 0 THEN max_data = MAX(data) ELSE max_data=lmax

n_coeff = (SIZE(coeff))(1)
IF (SIZE(data))(0) EQ 0 THEN n_data=1 ELSE n_data = (SIZE(data))(1)
result = FLTARR(n_data)
data2=(2*data - max_data - min_data)/(max_data - min_data)

FOR i=0,n_data-1 DO BEGIN
 result(i) = coeff(0)/2.
 FOR j=1,n_coeff-1 DO BEGIN
  result(i) = result(i) + coeff(j)*COS(j*ACOS(data2(i)))
 ENDFOR
ENDFOR

IF N_ELEMENTS(lnorm) EQ 0 THEN rnorm = result(FIX(n_data/2)) ELSE BEGIN
 rnorm = coeff(0)/2.
 datanorm = (2*lnorm - max_data - min_data)/(max_data - min_data)
 FOR j=1,n_coeff-1 DO BEGIN
  rnorm = rnorm + coeff(j)*COS(j*ACOS(datanorm))
 ENDFOR
ENDELSE

RETURN,result/rnorm
END
