FUNCTION GET_ADAPTED_PROFILE,x,y,xr_fl,yr_fl,ref_profile,ref_distance,ref_ang_width,at=at,sq=sq,sym_rad=sym_rad,show=show,area=area
;+
; NAME:
;	GET_ADAPTED_PROFILE
;
; PURPOSE:
;	Builds a reference profile by weighted averaging, from a list of available profiles.
;
; CATEGORY:
;	Data processing.
;
; CALLING SEQUENCE:
;	adapted_profile = GET_ADAPTED_PROFILE( $
;				x,y,$
;				xr_fl,yr_fl,$
;				ref_profile,$
;				ref_distance,ref_ang_width,$
;				at=at,sq=sq,$
;				sym_rad=sym_rad,$
;				area=area, /SHOW)
;
;
; INPUTS:
;	x,y : the coordinates of the spot to be profile fitted [mm]
;	xr_fl,yr_fl : the coordinates of the spot to be profile fitted [raster units]
;	ref_profile : list of structure containing the learned profiles, 
;	and associated parameters
;	ref_distance : the maximum radial distance to look for a reference peak
;	   	from the position of the currently integrated peak [mm]
;	ref_ang_width : the maximum tengential distance to look for a 
;		reference peak from the position of the currently integrated 
;		peak [degrees]
;	at : the angular position of the reference profiles
;	sq : the radial position of the reference profiles
;	sym_rad : set this keyword if the reference peak search is to be extented
;		to symmetrical sector if necessary
;	area : use this with the SHOW keyword to see the effect of interpolation
;	show: set this keyword to see what's happening.
;
; OUTPUTS:
;	
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None.
;
; PROCEDURE:
;	The reference profile is built from a set of selected profiles which fullfill the
;	following conditions :
;	--> if there are at least two profiles in the sector around the spot to profile fitted, then
;		keep only these two
;	--> if there are less than two, and the SYM_RAD keyword is set, then look
;		over the full radial, twice tangential sector AND look also in the
;		symetrical sector.
;	--> if there less than two profiles and SYM_RAD keyword is not set, then look
;		over the full radial, twice tangential sector.
;	--> if still nothing, exit with an error message.
;
; MODIFICATION HISTORY:
;	Dominique Bourgeois, August 97.
;	Julien Bourgeoi, August 97
;-

;********************
;select a first set of possible reference profiles :
;********************
radius_i = SQRT(x^2+y^2)
angle_i =  ATAN(y,x)*180/!Pi
r_max = radius_i + ref_distance/2.
r_min = (radius_i - ref_distance/2.)>0
phi_max = angle_i + ref_ang_width/2.
IF phi_max GT 180 THEN phi_max = phi_max - 360
phi_min = angle_i - ref_ang_width/2.
IF phi_min LT -180 THEN phi_min = phi_min + 360

IF N_ELEMENTS(at) EQ 0 THEN at = ATAN(ref_profile.y,ref_profile.x)*180/!Pi
IF N_ELEMENTS(sq) EQ 0 THEN sq = SQRT(ref_profile.x^2 + ref_profile.y^2)

IF phi_min LE phi_max THEN $
 ref_index = WHERE( at LE phi_max AND $
		at GE phi_min AND $
	 	sq LE r_max AND $
		sq GT r_min,n_ref) $
ELSE $
 ref_index = WHERE( at LE phi_max OR $
		at GE phi_min AND $
		sq LE r_max AND $
		sq GT r_min,n_ref)


;********************
;handle cases where insufficient number of reference peaks.
;********************


;---> give minimum number of reference peaks
min_nref=3
;---> give maximum number of reference peaks
max_nref=20

IF n_ref LE min_nref THEN BEGIN

 step = 1.2 ; this is the relative increase

 loc_max_rad = (MAX(sq)-radius_i) > ABS(MIN(sq)-radius_i)

 maxsteps = ROUND((ALOG(loc_max_rad)-ALOG(ref_distance/2.))/ALOG(step)) ; this is the maximum number of steps allowed

;print,'Maximum # of steps : ',maxsteps
 nstep = 1 ; we start at step 1.

 REPEAT BEGIN ; iteratively increase the sector where reference spots are searched for.

  IF NOT KEYWORD_SET(sym_rad) THEN BEGIN
  ;********************
  ;do not include symmetrical sector
  ;********************

   ;increase the radial width 
   r_max = radius_i + (step^(nstep))*ref_distance/2.
   r_min = (radius_i - (step^(nstep))*ref_distance/2.)>0
   ;increase the angular width 
   phi_max = angle_i + (step^(nstep))*ref_ang_width/2.
   ;print,'step # ', nstep
   ;print,'phi_max = ',phi_max
   IF phi_max GT 180 THEN phi_max = phi_max - 360
   phi_min = angle_i - (step^(nstep))*ref_ang_width/2.
   ;print,'phi_min = ',phi_min
   IF phi_min LT -180 THEN phi_min = phi_min + 360

   IF phi_min LE phi_max THEN $
    ref_index = WHERE( at LE phi_max AND $
		at GE phi_min AND $
	 	sq LE r_max AND $
		sq GT r_min,n_ref) $
   ELSE $
    ref_index = WHERE( at LE phi_max OR $
		at GE phi_min AND $
		sq LE r_max AND $
		sq GT r_min,n_ref)

  ENDIF ELSE BEGIN

 ;********************
 ;include symmetrical sector
 ;********************

   ;increase the radial width 
   r_max = radius_i + (step^(nstep))*ref_distance/2.
   r_min = (radius_i - (step^(nstep))*ref_distance/2.)>0

   ;take twice the angular width 
   IF angle_i GE 0 THEN angle_i2 = angle_i - 180 ELSE angle_i2 = angle_i + 180 

   ;increase the angular width 
   phi_max = angle_i + (step^(nstep))*ref_ang_width/2.
   ;print,'step # ', nstep
   ;print,'phi_max = ',phi_max
   IF phi_max GT 180 THEN phi_max = phi_max - 360

   phi_max2 = angle_i2 + (step^(nstep))*ref_ang_width/2.
   ;print,'step # ', nstep
   ;print,'phi_max2 = ',phi_max2
   IF phi_max2 GT 180 THEN phi_max2 = phi_max2 - 360

   phi_min = angle_i - (step^(nstep))*ref_ang_width/2.
   ;print,'phi_min = ',phi_min
   IF phi_min LT -180 THEN phi_min = phi_min + 360

   phi_min2 = angle_i2 - (step^(nstep))*ref_ang_width/2.
   ;print,'phi_min2 = ',phi_min2

   IF ((phi_min LE phi_max) AND (phi_min2 LE phi_max2)) THEN $
    ref_index = WHERE((at LE phi_max AND $
		   at GE phi_min) OR $
		  (at LE phi_max2 AND $
                   at GE phi_min2) AND $
	 	   sq LE r_max AND $
		   sq GT r_min,n_ref)

   IF ((phi_min LE phi_max) AND (phi_min2 GT phi_max2)) THEN $
    ref_index = WHERE((at LE phi_max AND $
		   at GE phi_min) OR $
		  (at LE phi_max2 OR $
                   at GE phi_min2) AND $
	 	   sq LE r_max AND $
		   sq GT r_min,n_ref)

   IF ((phi_min GT phi_max) AND (phi_min2 LE phi_max2)) THEN $
    ref_index = WHERE((at LE phi_max OR $
		   at GE phi_min) OR $
		  (at LE phi_max2 AND $
                   at GE phi_min2) AND $
	 	   sq LE r_max AND $
		   sq GT r_min,n_ref)

   IF ((phi_min GT phi_max) AND (phi_min2 GT phi_max2)) THEN $
    ref_index = WHERE((at LE phi_max OR $
		   at GE phi_min) OR $
		  (at LE phi_max2 OR $
                   at GE phi_min2) AND $
	 	   sq LE r_max AND $
		   sq GT r_min,n_ref)


 ENDELSE
;********************

nstep = nstep+1
;ENDREP UNTIL (n_ref GE min_nref OR nstep GE maxsteps)
ENDREP UNTIL ((n_ref GE min_nref) OR (nstep GE maxsteps AND n_ref GE 1) OR (nstep GE 100))

;print,'Number of steps : ', nstep-1
;print,'Number of ref_spots : ',n_ref

  IF KEYWORD_SET(sym_rad) THEN BEGIN
   IF n_ref GT 0 THEN BEGIN
    IF KEYWORD_SET(show) THEN PRINT,'No close reference peaks ! Ref peaks taken from sector enlarged by : ',step^(nstep)
   ENDIF ELSE MESSAGE,'Choose larger area for reference peak search !'
  ENDIF ELSE BEGIN
   IF n_ref GT 0 THEN BEGIN
    IF KEYWORD_SET(show_weak) THEN PRINT,'No close reference peaks ! Ref peaks taken from sector enlarged by (including symmetrical sector) : ',step^(nstep)
   ENDIF ELSE MESSAGE,'Choose larger area for reference peak search !'
  ENDELSE

ENDIF

possible_ref = ref_profile(ref_index)
dist_to_ref_peaks = (possible_ref.x-x)^2 + (possible_ref.y-y)^2

;if too many reference peaks, then  take the "max_nref" best ones
IF n_ref GT max_nref THEN BEGIN
 ;IF n_ref GT 15 THEN PRINT,'Nref was : ',n_ref
 isigi_ref = possible_ref.intp(0)/(possible_ref.bc_coeff(2)>0.1)
 possible_ref = possible_ref((SORT(dist_to_ref_peaks/isigi_ref))(0:max_nref-1))
 n_ref=LONG(max_nref) ;must be set to LONG for the C code !
ENDIF

IF MIN(dist_to_ref_peaks) EQ 0 THEN BEGIN
 ;We are in this case in the presence of one of the reference spots !
 ;If spot list does not contain any of the reference spots, this should never
 ;happen ; However integration is carried on !
 ;PRINT,'Trying to auto-fit a spot ?'
ENDIF ;ELSE BEGIN
 adapted_profile =(possible_ref(0).sp)*1/(SQRT((possible_ref(0).x-x)^2 +(possible_ref(0).y-y)^2)>0.1)
 IF n_ref GT 1 THEN BEGIN
  s = SIZE(adapted_profile)
  xsize = s(1)
  ysize = s(2)
  dum = CALL_EXTERNAL(STRTRIM(GETENV('PROW_C')+'get_adapted_profile.so'),'profile', n_ref, adapted_profile, possible_ref(*).sp, possible_ref(*).x,possible_ref(*).y, x, y, xsize, ysize)
 ENDIF
;ENDELSE

;***
;INTERPOLATE THE PROFILE
;***

IF KEYWORD_SET(show) AND N_ELEMENTS(area) NE 0 THEN BEGIN
 adapted_profile2 = adapted_profile/TOTAL(adapted_profile)

 ;calculate center of gravity and compare with predicted position
 ;before interpolation
 row=TOTAL(adapted_profile,2)>0.1
 column=TOTAL(adapted_profile,1)>0.1
 ;compute center of gravity
 indx=INDGEN((SIZE(adapted_profile))(1))
 indy=INDGEN((SIZE(adapted_profile))(2))
 cgx=TOTAL(indx*row)/TOTAL(row) + 0.5
 cgy=TOTAL(indy*column)/TOTAL(column) + 0.5
 bsx=(SIZE(adapted_profile))(1) & bsy=(SIZE(adapted_profile))(2)
 PRINT,'Expected theoretical spot position in area [x,y] : ',FIX(bsx/2.)+xr_fl-FIX(xr_fl),FIX(bsy/2.)+yr_fl-FIX(yr_fl)
 PRINT,'Center of gravity of reference profile BEFORE interpolation [x,y] : ',cgx,cgy
 PRINT,'Deviation from expected (theoretical) position [x,y] : ',FIX(bsx/2.)+xr_fl-FIX(xr_fl)-cgx,FIX(bsy/2.)+yr_fl-FIX(yr_fl)-cgy
 PRINT,'Will interpolate by shifting reference spot by [x,y] : ', (xr_fl MOD 1) - 0.5, (yr_fl MOD 1) - 0.5

 PRINT,'REF_PROFILE + SPOT (Before interpolation) : '
 temp = adapted_profile2
 CONTOUR,temp,LEVELS=[(INDGEN(10)+1)*MAX(temp)/10.],C_COLORS=!D.N_COLORS - 1
 CONTOUR,area,LEVELS=[(INDGEN(10)+1)*MAX(area)/10.],/NOERASE,C_COLORS=!D.N_COLORS - 2

 OK=''
 READ,'OK ? [Y/N]',OK
ENDIF

 adapted_profile = GET_INTERP_PROFILE(adapted_profile,xr_fl,yr_fl)

 ;***
 ;normalize the integrated intensity of the reference profile to 1. 
 ;***
 adapted_profile = adapted_profile/TOTAL(adapted_profile)

IF KEYWORD_SET(show) THEN BEGIN
 IF N_ELEMENTS(area) NE 0 THEN BEGIN
 
  ;calculate center of gravity and compare with predicted position
  ;after interpolation
  row=TOTAL(adapted_profile,2)>0.1
  column=TOTAL(adapted_profile,1)>0.1
  ;compute center of gravity
  indx=INDGEN((SIZE(adapted_profile))(1))
  indy=INDGEN((SIZE(adapted_profile))(2))
  cgx=TOTAL(indx*row)/TOTAL(row) + 0.5
  cgy=TOTAL(indy*column)/TOTAL(column) + 0.5
  bsx=(SIZE(adapted_profile))(1) & bsy=(SIZE(adapted_profile))(2)
  PRINT,'Center of gravity of reference profile AFTER interpolation [x,y] : ',cgx,cgy
  PRINT,'Deviation from expected (theoretical) position [x,y] : ',FIX(bsx/2.)+xr_fl-FIX(xr_fl)-cgx,FIX(bsy/2.)+yr_fl-FIX(yr_fl)-cgy
 
  PRINT,'REF_PROFILE + SPOT (After interpolation) : '
  temp = adapted_profile
  CONTOUR,temp,LEVELS=[(INDGEN(10)+1)*MAX(temp)/10.],C_COLORS=!D.N_COLORS - 1
  CONTOUR,area,LEVELS=[(INDGEN(10)+1)*MAX(area)/10.],/NOERASE,C_COLORS=!D.N_COLORS - 2

  OK=''
  READ,'OK ? [Y/N]',OK
 ENDIF
 ;***
 ;PRINT INFO
 ;***
 PRINT,'Number of reference peaks : ',(SIZE(possible_ref))(1)
 PRINT,'Spot position [X,Y], [mm]:',x,y
 PRINT,'X values [mm]:',possible_ref.x
 PRINT,'Y values [mm]:',possible_ref.y
 PRINT,'Position of maximum [x,y] :',WHERESUB(WHERE(adapted_profile EQ MAX(adapted_profile)),adapted_profile)

ENDIF

RETURN,adapted_profile

END





