function GET_FITLEVEL,prof,peak_max,n_back,back,dark_current,$
				bg_gain=bg_gain,gain=gain,show=show
;+
; NAME:
;	GET_FITLEVEL
;
; PURPOSE:
;
;	For profile fitting, returns the optimal cutoff for peak integration.
;
; CATEGORY:
;	Data processing
;
; CALLING SEQUENCE:
;	cutoff = GET_FITLEVEL(prof,peak_max,nback,back,dark_current,$
;					bg_gain=bg_gain,gain=gain,/show)
;
; INPUTS:
;	prof : the reference profile for the spot to integrate
;	peak_max : the (non background subtracted) peak value of the spot to integrate
;	nback : the number of pixels used to calculate the background
;	back : the constant level background value (including any offset or dark_current).
;	dark_current : value of offset or dark current in the detector 
;	bg_gain : this is the detector gain for the background
;	gain : this is the detector gain for spot
;	show : set this keyword to show information
;
; OUTPUTS:
;	cutoff : the optimum fit_level [% MAX(prof)]	
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None
;
; PROCEDURE:
;	The variance is calculated for a serie of cutoff values
;	(from 0% to 20%), the cutoff corresponding to the 
;	smallest variance is retained, with a minimum of 0.1%
;
; MODIFICATION HISTORY:
;	D.Bourgeois, April 1997.
;-

IF (NOT KEYWORD_SET(bg_gain)) OR (NOT KEYWORD_SET(gain)) THEN BEGIN
 bg_gain=1 & gain=1
ENDIF

min_sigma = 10.^10
peak = gain*prof/MAX(prof)*(peak_max - back) + bg_gain*(back - dark_current)

;** evaluation of sigma^2
cutoff=0

FOR i=0,20 DO BEGIN
 w=WHERE(prof GE MAX(prof)*(i/100.))

 ;signal
 sig_signal = TOTAL(prof(w)^2*peak(w))/(TOTAL(FLOAT(prof(w))^2))^2

 ;background
 sig_back = bg_gain/FLOAT(n_back)*(back - dark_current)*TOTAL(prof(w))^2/(TOTAL(FLOAT(prof(w))^2))^2

 IF (sig_signal+sig_back) LE min_sigma THEN BEGIN
  cutoff = i
  min_sigma = sig_signal+sig_back
 ENDIF

ENDFOR

IF KEYWORD_SET(show) THEN BEGIN
PRINT,'Minimum sigma^2 [counts], Optimal cutoff [%]:',min_sigma,0.1>cutoff
ENDIF

RETURN,0.1>cutoff
END








