function GET_BOXLEVEL,profile,n_back,back,dark_current,show=show
;+
; NAME
;	GET_BOXLEVEL
;
; PURPOSE:
;
;	For box integration, finds the cutoff level that minimizes Sig(I)/I
;
; CATEGORY:
;	Data processing
;
; CALLING SEQUENCE:
;	cutoff = GET_BOXLEVEL(profile,nback,back,dark_current,/show)
;
; INPUTS:
;	profile : the background subtracted profile for the spot to integrate
;	nback : the number of pixels used to calculate the background
;	back : the constant level background value (including any offset or dark_current).
;	dark_current : value of offset or dark current in the detector 
;	show : set this keyword to show information
;
; OUTPUTS:
;	cutoff : the optimum fit_level [% MAX(profile)]	
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None
;
; PROCEDURE:
;	Sig(I)/I is calculated for a serie of cutoff values
;	(from 0% to 10%, steps of 0.4%), the cutoff corresponding to the 
;	smallest Sig(I)/I is retained, with a minimum of 0.01%
;	See Clive Wilkinson and Hanna W. Khamis, J. Appl. Cryst (1988), 21
;	, p471-478. These guys take an optimum ellipsoid around the peak.
;	Here a more precise contour is taken.
;
; MODIFICATION HISTORY:
;	D.Bourgeois, February 1995.
;-

min_snr = 10.^10
cutoff = 15.

;** evaluation of sig(I)/I
FOR i=0,25 DO BEGIN
w=WHERE(profile GE MAX(profile)*(i/250.),count)

;(sig(I)/I)^2 is given by
II = TOTAL(profile(w))>1.e-5
snr = (II + count*(back - dark_current)*(1 + count/FLOAT(n_back)))/II^2

IF snr LE min_snr THEN BEGIN
cutoff = i/2.5
min_snr = snr
ENDIF

ENDFOR

IF KEYWORD_SET(show) THEN BEGIN
PRINT,'Minimum Sig(I)/I, Optimal cutoff [%]:',SQRT(min_snr),.01>cutoff
ENDIF

RETURN,0.01>cutoff
END








