function GET_1OVLPVAR,ref_prof,prof,peak_area,n_back,back,gain,bg_gain,dark_current,show=show
;+
; NAME:
;	GET_1OVLPVAR
;
; PURPOSE:
;
;	For profile fitting, returns the variance (sigma^2) of an integrated peak
;	according to Poisson statistics. Case of an overlapped peak which is close to a
;	bunch of spots too close together to be deconvoluted. The peak is thus integrated
;	as non integrated, but the profile is truncated to get rid of the region corresponding
;	to the presence of these other spots.
;
; CATEGORY:
;	Data processing
;
; CALLING SEQUENCE:
;	variance = GET_1OVLPVAR(ref_prof,prof,peak_area,n_back,back,gain,bg_gain,dark_current,/show)
;
; INPUTS:
;	ref_prof : the reference profile for the spot to integrate
;	prof : the non background subtracted peak to integrate
;	peak_area : the area where the peak can be integrated (usually truncated) 
;	n_back : the number of pixels used to calculate the background
;	back : the constant level background value (including offset or dark current).
;	gain : the gain of the detector at the peak energy
;	bg_gain : the mean gain of the detector for background
;	dark_current : the dark_current (offset) of the detector
;	show : set this keyword to show information
;
; OUTPUTS:
;	variance : the variance of the integrated intensity.
;
; COMMON BLOCKS:
;	None.
;
; SIDE EFFECTS:
;	None.
;
; RESTRICTIONS:
;	None
;
; PROCEDURE:
;	The formula is adapted from A.G.W Leslie.
;	Molecular Data Processing, Chapter 4
;	However the variance at the (hopefully optimal) fit_level
;	cutoff is given.
;
; MODIFICATION HISTORY:
;	D.Bourgeois, January 1995.
;-


w = peak_area(WHERE(peak_area NE 0))


variance = (gain*TOTAL(ref_prof(w)^2*(prof(w) - back)) + bg_gain*(back-dark_current)*TOTAL(ref_prof(w)^2) + bg_gain*(1/FLOAT(n_back))*(back-dark_current)*TOTAL(ref_prof(w))^2)/(TOTAL(FLOAT(ref_prof(w))^2))^2

IF KEYWORD_SET(show) THEN BEGIN
PRINT,'Variance (sigma^2) [counts] :',variance
ENDIF

RETURN,variance
END








