Function f1f2_calc_mix,input,descriptor,energy,F=FDES,density=density,$
   theta=theta,rough=rough,NAME=NAME,group=group, verbose=verbose

; 
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       F1F2_CALC_MIX
;
; PURPOSE:
;	calculate the elastic Photon-Atom anonalous f1 and f2  coefficients
;	as a function of energy for a given mixture. 
;	It also gives the refractive index 
;	components delta and beta (n=1-delta - i beta), the absorption
;	photoelectric coefficient and the reflectivities (s,p and unpolarized).
;	It users data from DABAX data-base.
;	
; CATEGORY:
;       X-Ray optics. DABAX data base.
;
; CALLING SEQUENCE:
;	f1 = f1f2_calc_mix(input,descriptor[,energy])
; INPUTS:
;	input: a dabax input file for f1f2 data as accepted by dabax_access().
;	descriptor: a string  with either the descriptor or the name of the 
;	mixture. When using the descriptor, you must supply the density (g/cc)
;	The descriptor sintax is the following:
;	string with the chemical formula of the compounds with their weight 
;	fraction between parenthesis.
;               Descriptor examples:
;               SiO2(0.807)B2O3(0.129)Na2O(0.038)Al2O3(0.022)K2O(0.004)
;               H2O(0.99)Cl2(0.01)
;               H(0.11189834)O(0.88810166)
;	The name is one of the names defines in the DABAX file Compounds.dat
;	When using the name input option, the keyword NAME must be set.
;		
; OPTIONAL INPUTS:
;	energy: if undefined, it uses the standard energy grid,
;		which is the grid in the data file for the tabulated
;		cases, and  dindgen(100)/99.*3.0 for the parametrized
;		data. When energy is defined, this value is used as a grid
;		for the results (it interpolates in the case of 
;		tabulated data)
;	
; KEYWORDS:
;	F: selects the output:
;		F=0 (default) returns a 2-col array with f1 and f2
;		F=1  returns f1
;		F=2  returns f2
;  		F=3  returns delta  [n = 1 -delta -i beta]
;  		F=4  returns betaf  [n = 1 -delta -i beta]
;  		F=5  returns Photoelectric linear absorption coefficient
;  		F=6  returns Photoelectric mass absorption coefficient
;  		F=7  returns Photoelectric Cross Section
;  		F=8  returns s-polarized reflectivity
;  		F=9  returns p-polarized reflectivity
;  		F=10  returns unpolarized reflectivity
;  		F=11  returns delta/beta
;	THETA= the incident [grazing] angle in mrad (for reflectivity
;		calculations). It is set to 3.0 (default) when F>=8 and
;		THETA is undefined. It can be an array. In such a case, the
;		returned value os a matrix: out(theta,energy)
;	ROUGH= the rms roughness in Angstroms (for reflectivity
;		calculations. A Debye-Waller model is used.)
;	NAME: Set this keyword to a valid mixture name in the file 
;		Compounds.dat. If not set, it will interpret
;		"descriptor" as a mixture formula.
;	DENSITY= the density value to be used for the calculations. If
;		not defined, and NAME is not set, then it uses 1.0 except
;		in the case when there is a single element in the
;		compund. In such a case it calls atomic_constants to get
;		the value.
;       GROUP = The widget ID of the widget that calls the routine.
;               This is passed to widget_message in its dialog_parent
;               keyword to center the window message.
;	VERBOSE = When set to 1, prints some information.
;
; OUTPUT:
;	out: an array with the values of the selected returm paramaeter.
;
; PROCEDURE:
;	i) calculate the scattering factors f1_i and f2_i for each elements by
;	   calling f1f2_calc
;	ii) Calculate the atomic composition f_i of each element in the 
;	   mixture by calling parse_mixture 
;	iii) The mean scattering factors for the mixture ate the 
;	   averaged scattering factors from the elements (f_i):
;		f1 = Sum{ f_i * f1_i }  (similarly for f2)
;	iv) Returns the value in the correspondent units:
;	
;	The MW (molecular weight) is MW = Sum { f_i * Ai), being Ai
;		the atomic weight of the i-th element available in 
;		the molecule (mixture) formula.
;
;	IMPORTANT NOTICE: 
;
;	   A mixture can be defined, either 
;		i) as a function of the weight fractions 
;		ii) as a molecular mixture. 
;
;	   Therefore, water can be either defined as "H2O" (molecule) 
;		or "H(0.11)O(0.89)" (weight). 	
;	
;		FOR f1f2_calc_mix CALCULATIONS, THE MIXTURE SHOULD BE
;		DEFINED IN ITS MOLECULAR FORM (like "H2O") AND NOT IN ITS 
;		WEIGHT FRACTION (like "H(0.11)O(0.89)"). 
;
;		The reason is that the averaging coefficient f_i is
;		is the atomic fraction and NOT the weight fraction. 
;		
;		When defining the mixture using weight fractions,
;		XOP calculates the atomic fraction from the weight fractions
;		(done in parse_mixture) but the result is inaccurate when
;		the molecular formula is unknown, because the molecular
;		weight is unknown.
;
;		PLEASE NOTE THAT THE PRE-DEFINED COMPOUNDS (IN DABAX
;		FILE Compounds.dat CONTAIN THE MIXTURE FORMULA, AND IN
;		SOME CASES (THE COMPOUNDS OBTAINED FROM NIST DATABASE)
;		THE MOLECULAR FORMULA IS NOT PRESENT (ONLY WEIGHT FORMULA)
;		THEREFORE THE SCATTERING FACTORS AND RELATED VALUES MAY BE 
;		WRONG.
;
;
; REFERENCES: 
;	
;	http://xdb.lbl.gov/Section1/Sec_1-7.pdf
;	   
;
;
; EXAMPLES:
;	EXAMPLE1: s-reflectivity vs. energy using descriptor
;	delvar,energy ; cleans energy variable
;	f = f1f2_calc_mix('f1f2_Henke.dat','Si3N4',energy,f=8,dens=3.44,the=3.0)
;	plot,energy,f,/xlog
;
;	EXAMPLE2: s-reflectivity vs. energy using name
;	delvar,energy ; cleans energy variable
;	f = f1f2_calc_mix('f1f2_Henke.dat','nitride',energy,f=8,/NAME,the=3.0)
;	plot,energy,f,/xlog
;
;	EXAMPLE3: unpolarized refl vs. theta and energy
;	; create 20 points in the energy range [1000.,21000.] (eV)
;	energy = (findgen(20)+1)/20.0*10000.0+1000.0
;	; create 20 points in the theta range [1.0,6.0] (mrad)
;	theta = (findgen(20)+1)/20.0*5.0+1.0
;	out = f1f2_calc_mix('f1f2_BrennanCowan.dat','Zerodur',energy,f=10,$
;		theta=theta,/NAME)
;	;plot
;	surface,out,theta,energy,az=285
;
; MODIFICATION HISTORY:
;       96-11-25 Written by M. Sanchez del Rio (srio@esrf.fr)
;       97-01-09 srio@esrf.fr adds density help and call to
;		atomic_constants to get density when there is a single element.
;       97-01-15 srio@esrf.fr adapts for Windows95.
;       97-10-16 srio@esrf.fr adds GROUP keyword.
;       01-02-28 srio@esrf.fr corrects a bug:
;		The mirror reflectivity calculations for COMPOUNDS (not for 
;		elements) is wrong.  It should be calculated from the global 
;		f' obtained from the average (in formula weight) of the compound
;		elements. XOP calculates fist the reflectivities of the
;		elements and then makes the average in reflectivities, which 
;		is wrong, because the average must be done in f's not in 
;		reflectivities. Not fully tested now!
;	2003-07-07 Added verbose keyword. 
;	2007-06-25 srio@esrf.eu updated doc concerning the mixture definition
;		(after fixing parse_mixture).
;-
on_error,2

IF N_Elements(verbose) EQ 0 THEN verbose=1
molwt=0.0
if keyword_set(name) then begin
  cfile = 'Compounds.dat'
  h = dabax_access(cfile,group=group)
  index = dabax_select(h,descriptor,/casematch,group=group)
  tmp = spec_data(h,index,/index)
  if not(keyword_set(density)) then begin
    rho1 = spec_headers(h,index,['#URHO'],/index)
    density = float(strmid(rho1,5,strlen(rho1)-1))
  endif
  ;not used  z = fix(tmp(0,*))
  ;not used  f = tmp(1,*)
  ;not used  s = atomic_symbols(z)
  ;
  ; get the molecular weight of the mixture
  ;
  ; this is necessary for calculating the molecular weight. However
  ; the other parameters returned are redundant with what is obtained
  ; before. It is then recommended to include the molecular weight in 
  ; the Compounds.dat file to avoid using parse_mixture here.
  ; To be done in the future. srio@esrf.fr 01/02/28
  ; 
  ; Note 2003-07-07:
  ; As the molecular weight (or the atomic weights) are not available
  ; in the file, this step is necessary
  ;
  for1 = spec_headers(h,index,['#UFOR'],/index)
  formula1 = strmid(for1,5,strlen(for1)-1)
  IF verbose THEN Message,/Info,'Using mixture with formula: '+formula1
  n = parse_mixture(formula1,s,f,z,fw,molwt=molwt)
endif else begin
  n = parse_mixture(descriptor,s,f,z,fw,molwt=molwt)
endelse
IF verbose THEN Message,/Info,'Using molecular weight: '+StrCompress(molwt)


if n_elements(s) LT 1 then begin
  message,/info,'Error in mixture descriptor/name (Returning 0): '+mixture
  return,0
endif


if keyword_set(fdes) then if fdes ge 3 then $
  if not(keyword_set(name)) and n_elements(density) eq 0 then begin
    IF n_elements(s) EQ 1 THEN BEGIN
      density = atomic_constants(s(0),return='Density') 
    ENDIF ELSE BEGIN
      density = 1.0
      IF SDep(/W) EQ 1 THEN $
      itmp = Widget_Message(/Info,'Density undefined. Set to one.',$
	Dialog_Parent=group)
    ENDELSE
    Message,/info,'Density value undefined, set to '+strcompress(density)
  endif


if n_elements(density) GT 0 then $
message,/info,'Density ('+descriptor+'): '+strcompress(density)

;
; calculate global f1 and f2
;
for i=0,n_elements(s)-1 do begin
  fi = f1f2_calc(input,s(i),energy,F=0)
  if i eq 0 then begin
     f1 = Reform(fi[0,*])*f(i) 
     f2 = Reform(fi[1,*])*f(i) 
  endif else begin
     f1 = f1 + Reform(fi[0,*])*f(i)
     f2 = f2 + Reform(fi[1,*])*f(i)
  endelse
endfor

; returns result( if user requested fdes)

IF NOT(Keyword_Set(fdes)) THEN fdes=0
CASE Fix(fdes) OF
  0:begin
    fout = make_set(f1,f2)
    return,fout
    end
  1:return,f1
  2:return,f2
  else:
ENDCASE

atwt = molwt
Message,/Info,'Molecular weight is: '+StrCompress(atwt)
avogadro = physical_constants('avogadro')
tocm = physical_constants('hc')*1e-8
re = physical_constants('re')

molecules_per_cc = density*avogadro/atwt
wavelength = tocm/energy   ; in cm
k = molecules_per_cc*re*wavelength*wavelength/2.0/!pi

;
; calculation of refraction index 
;

delta = k*f1
betaf = k*f2
mu = 4.0*!pi*betaf/wavelength

case fdes of
  3:return,delta
  4:return,betaf
  5:return,mu
  6:return,mu/density
  7:return,mu/molecules_per_cc*1.0e+24
  11:return,delta/betaf
  else:
endcase

;
; calculation of reflectivity (piece of code adapted from shadow/abrefc)
;

if n_elements(theta) eq 0 then theta1 = 3.0e-3 else theta1 = theta*1e-3
if n_elements(rough) eq 0 then rough1 = 0.0 else rough1=rough
rough1 = rough1*1.0d-8 ; in cm
; epsi = 1 - alpha - i gamma
alpha = 2.0D0*k*f1
gamma = 2.0D0*k*f2
	for i=0,n_elements(theta1)-1 do begin

     	RHO	=   (SIN(theta1(i)))^2 - ALPHA
     	RHO	=   RHO + SQRT ((SIN(theta1(i))^2 - ALPHA)^2 + GAMMA^2)
     	RHO	=   SQRT(RHO/2)
;** Computes now the reflectivity for s-pol
     	RS1	=   4*(RHO^2)*(SIN(theta1(i))-RHO)^2 + GAMMA^2
     	RS2	=   4*(RHO^2)*(SIN(theta1(i))+RHO)^2 + GAMMA^2
     	RS	=   RS1/RS2
;** Computes now the polarization ratio
     	RATIO1	=   4*RHO^2*(RHO*SIN(theta1(i))-COS(theta1(i))^2)^2 + $
		    GAMMA^2*SIN(theta1(i))^2
     	RATIO2	=   4*RHO^2*(RHO*SIN(theta1(i))+COS(theta1(i))^2)^2 + $
		    GAMMA^2*SIN(theta1(i))^2
     	RATIO	=   RATIO1/RATIO2
;** The reflectivity for p light will be
     	RP	=   RS*RATIO
;** For an unpolarized beam we have
     	RUNP	=   0.5*(RS+RP)
        DebyeWaller = exp( -(4.0*!pi*sin(theta1(i))*rough1/wavelength)^2)
        case fdes of
          8:out = rs*DebyeWaller
          9:out = rp*DebyeWaller
          10:out = runp*DebyeWaller
          else:
        endcase
	if n_elements(theta1) eq 1 then out1 = out else begin
          if i eq 0 then begin
            out1 = fltarr(n_elements(theta1),n_elements(out))
            out1(0,*) = reform(out)
          endif else out1(i,*) = reform(out)
        endelse
        endfor

return,out1
END
