;+
;
; NAME:
;	XTABLE
;
; PURPOSE:
;	Implementation of a table. Functions to add/delete row/columns
;	can be called. The cells of the table must have only one data type.
;	You can specify the type of data with a keyword (see below for details).
;
; CATEGORY:
;	Widget
;
; CALLING SEQUENCE:
;	XTABLE, data [,keywords]
;
; INPUTS:
;	data   = 2-dimensional array of data (float, integer, string)
;
; KEYWORD PARAMETERS:
;	XTitle = An array used for labeling the columns
;	YTitle = An array used for labeling the rows
;	Type   = either 'String', 'Integer', 'Float' or 'Double' to
;		 determine the data type of the input array.
;	XSize  = An integer set to the number of visible columns.
;	YSize  = An integer set to the number of visible rows.
;	CSel   = A two-dimensional array ( = [x1,x2]) that indicates
;		 which cells appear selected
;	RSel   = A two-dimensional array ( = [y1,y2]) that indicates
;		 which rows appear selected
;	Group  = Process ID of the father process if present
;	Wtitle = String set to the title of the window
;	Action = Set to 'OK' if table was edited and saved or 'Cancel'
;		 if no changes were done or discarded.
;       buttons = Set this keyword to create two buttons 'Accept' and
;		 'Cancel'
;	noMenuBar = Set this keyword to avoid creating the menu bar
;	infoText = Set this keyword to a string or string array with
;		  an info message to be displayed before the table.
;
; OUTPUTS:
;		Opens a widget utility. The 2-dimensional array can be
;		saved after being edited by this widget.
;
; SIDE EFFECTS:
;		If not active, starts Xmanager
;
; MODIFICATION HISTORY:
; 	Written by:	Andre Wilms, 20.10.1997
;
;	Modified by:	Andre Wilms, 10.11.1997
;	98/08/28 srio@esrf.fr adds catch statements.
;	2007/12/04 srio@esrf.eu adds buttons,noMenuBar,infoText keywords.
;
;-


PRO table_events, event


infoID = Widget_Info (event.top, /Child)

catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!err_string
  itmp=Dialog_Message(/Error,Dialog_Parent=event.top,$
    'TABLE_EVENT: error caught: '+!err_string)
  catch, /cancel
  if n_elements(info) NE 0 then Widget_Control, infoID, Set_UValue=info, /no_copy
  return
endif

Widget_Control, infoID, get_UValue=info, /no_copy
handler = info.ptr

CASE event.type OF
4 : 	BEGIN
		(*handler).left   = event.sel_left
		(*handler).top    = event.sel_top
		(*handler).right  = event.sel_right
		(*handler).bottom = event.sel_bottom
		
		IF Widget_Info(info.labelID,/Valid_Id) THEN Widget_Control, info.labelID, Sensitive = 0

		IF (*handler).left NE -1L THEN BEGIN
			IF (*handler).left EQ (*handler).right XOR (*handler).top EQ (*handler).bottom THEN $
		               IF Widget_Info(info.labelID,/Valid_Id) THEN $
				Widget_Control, info.labelID, Sensitive = 1
		ENDIF
	END
ELSE : 
ENDCASE

IF Widget_Info (event.top, /valid_ID) THEN BEGIN
	infoID = Widget_Info (event.top, /Child)
	Widget_Control, infoID, set_UValue=info, /no_copy
ENDIF
END



PRO xtable_events, event
infoID = Widget_Info (event.top, /Child)


catch, error_status
if error_status ne 0 then begin
  message,/info,'error caught: '+!err_string
  itmp=Dialog_Message(/Error,Dialog_Parent=event.top,$
    'XTABLE_EVENT: error caught: '+!err_string)
  catch, /cancel
  if n_elements(info) NE 0 then Widget_Control, infoID, Set_UValue=info, /no_copy
  return
endif

Widget_Control, infoID, get_UValue=info, /no_copy
Widget_Control, event.id,  get_UValue=value
handler = info.ptr
dataptr = info.dataptr
xptr    = info.xptr
yptr    = info.yptr

CASE value OF
'QUIT' : BEGIN
		(*handler).action = 'OK'
		Widget_Control, info.tableID, get_value=temp	
		*dataptr = temp
		Widget_Control, event.top, /Destroy
		return
	END

'ABORT' : BEGIN	
		(*handler).action = 'Cancel'
		Widget_Control, event.top, /Destroy
		return
	END		

'APPC' : BEGIN
		data = 'New Column'
		xedit, Text='Name for the new column', data, action=action
		IF action EQ 'OK' THEN BEGIN
			IF Widget_Info(info.delcID,/Valid_Id) THEN $
			  Widget_Control, info.delcID, Sensitive = 1
			*xptr = [*xptr, data]
			*xptr = strtrim(*xptr, 2)
			Widget_Control, info.tableID, column_widths = 78
			Widget_Control, info.tableID, row_heights = 24
			Widget_Control, info.tableID, insert_columns = 1
			Widget_Control, info.tableID, Column_Labels = *xptr
		ENDIF
	END

'APPR' : BEGIN
		data='New Row
		xedit, Text='Name for the new row', data, action=action
		IF action EQ 'OK' THEN BEGIN
			IF Widget_Info(info.delrID,/Valid_Id) THEN $
			  Widget_Control, info.delrID, Sensitive = 1
			*yptr = [*yptr, data]
			*yptr = strtrim(*yptr, 2)
			Widget_Control, info.tableID, column_widths = 78
			Widget_Control, info.tableID, row_heights = 24
			Widget_Control, info.tableID, insert_rows = 1
			Widget_Control, info.tableID, Row_Labels = *yptr
		ENDIF
	END

'INSR' : BEGIN
	END

'DELR' : BEGIN
		Widget_Control, info.tableID, get_value=temp	
		old = size(temp)
		IF (*handler).left NE -1L THEN BEGIN
		IF NOT ((*handler).top EQ 0 AND (*handler).bottom EQ old[2]-1) THEN BEGIN
			Widget_Control, info.tableID, /Delete_Rows
			Widget_Control, info.tableID, get_value=temp	
			result = size(temp)
			if result[0] EQ 1 THEN result[2] = 1

			IF old[2] NE result[2] THEN BEGIN
				IF (*handler).top NE 0 AND (*handler).bottom NE old[2]-1 THEN $
					*yptr = [ (*yptr)[0:(*handler).top - 1] , (*yptr)[(*handler).bottom + 1 : old[2] - 1] ]
				IF (*handler).top EQ 0 AND (*handler).bottom NE old[2]-1 THEN $
					*yptr = [(*yptr)[(*handler).bottom + 1 : old[2] - 1] ]
				IF (*handler).top NE 0 AND (*handler).bottom EQ old[2]-1 THEN $
					*yptr = [(*yptr)[0:(*handler).top - 1] ]

				IF result[0] EQ 1 THEN $
					IF Widget_Info(info.delrID,/Valid_Id) THEN $
					Widget_Control, info.delrID, Sensitive = 0

				IF (*handler).ysize GT result[2] THEN (*handler).ysize = result[2]
				(*handler).width  = (*handler).xsize * 78 + 102
				(*handler).height = (*handler).ysize * 24 + 62

				Widget_Control, info.tableID, scr_xsize=(*handler).width
				Widget_Control, info.tableID, scr_ysize=(*handler).height
			ENDIF
		ENDIF
		ENDIF
	END

'DELC' : BEGIN
		Widget_Control, info.tableID, get_value=temp	
		old = size(temp)
		IF (*handler).left NE -1L THEN BEGIN
		IF NOT ((*handler).left EQ 0 AND (*handler).right EQ old[1]-1) THEN BEGIN
			Widget_Control, info.tableID, /Delete_Columns
			Widget_Control, info.tableID, get_value=temp	
			result = size(temp)

			IF old[1] NE result[1] THEN BEGIN
				IF (*handler).left NE 0 AND (*handler).right NE old[1]-1 THEN $
					*xptr = [ (*xptr)[0:(*handler).left - 1] , (*xptr)[(*handler).right + 1 : old[1] - 1] ]
				IF (*handler).left EQ 0 AND (*handler).right NE old[1]-1 THEN $
					*xptr = [(*xptr)[(*handler).right + 1 : old[1] - 1] ]
				IF (*handler).left NE 0 AND (*handler).right EQ old[1]-1 THEN $
					*xptr = [(*xptr)[0:(*handler).left - 1] ]

				IF result[1] EQ 1 THEN  $
					IF Widget_Info(info.delcID,/Valid_Id) THEN $
					Widget_Control, info.delcID, Sensitive = 0
				IF (*handler).xsize GT result[1] THEN (*handler).xsize = result[1]
				(*handler).width  = (*handler).xsize * 78 + 102
				(*handler).height = (*handler).ysize * 24 + 62
				Widget_Control, info.tableID, scr_xsize=(*handler).width
				Widget_Control, info.tableID, scr_ysize=(*handler).height
			ENDIF
		ENDIF
		ENDIF
	END

'LABEL' : BEGIN
		col = 0
		IF (*handler).left EQ (*handler).right THEN col = 1

		data = ''
		IF col EQ 0 THEN $
			xedit, Text='Name for the new row', data, action=action $
		ELSE $	
			xedit, Text='Name for the new column', data, action=action

		IF action EQ 'OK' THEN BEGIN
			Widget_Control, info.tableID, get_value=temp	
			result = size(temp)
			IF col EQ 1 THEN BEGIN
				IF (*handler).left EQ 0 THEN $
					*xptr = [ data, (*xptr)[1:result[1]-1] ]
				IF (*handler).left EQ result[1]-1 THEN $
					*xptr = [ (*xptr)[0:result[1]-2], data ]
				IF (*handler).left NE 0 AND (*handler).left NE result[1]-1 THEN $
					*xptr = [ (*xptr)[0:(*handler).left-1], data, (*xptr)[(*handler).left+1:result[1]-1] ]
				Widget_Control, info.tableID, Column_Labels = *xptr
			ENDIF		
			IF col EQ 0 THEN BEGIN
				IF (*handler).top EQ 0 THEN $
					*yptr = [ data, (*xptr)[1:result[2]-1] ]
				IF (*handler).top EQ result[2]-1 THEN $
					*yptr = [ (*yptr)[0:result[2]-2], data ]
				IF (*handler).top NE 0 AND (*handler).top NE result[2]-1 THEN $
					*yptr = [ (*yptr)[0:(*handler).top-1], data, (*yptr)[(*handler).top+1:result[2]-1] ]
				Widget_Control, info.tableID, Row_Labels = *yptr
			ENDIF		
		ENDIF
	END

'VISC' : BEGIN
		Widget_Control, info.tableID, get_value=temp	
		result = size(temp)

		data = 0
		xedit, Text='Number of visual columns', data, action=action

		IF action EQ 'OK' THEN BEGIN
			IF data LT 1 THEN data = 1
			IF data GT (*handler).xsize THEN (*handler).xsize = data
			IF data GT (*handler).maxx  THEN data = (*handler).maxx
			IF data GT result[1] THEN data = result[1]
			(*handler).xsize = data
				(*handler).width  = data * 78 + 102
				Widget_Control, info.tableID, scr_xsize=(*handler).width
		END
	END

'VISR' : BEGIN
		Widget_Control, info.tableID, get_value=temp	
		result = size(temp)

		data = 0
		xedit, Text='Number of visual rows', data, action=action

		IF action EQ 'OK' THEN BEGIN
			IF data LT 1 THEN data = 1
			IF data GT (*handler).ysize THEN (*handler).ysize = data
			IF data GT (*handler).maxy  THEN data = (*handler).maxy
			IF data GT result[2] THEN data = result[2]
			(*handler).ysize = data
				(*handler).height = data * 24 + 62
				Widget_Control, info.tableID, scr_ysize=(*handler).height
		END
	END
ENDCASE

IF Widget_Info (event.top, /valid_ID) THEN BEGIN
	infoID = Widget_Info (event.top, /Child)
	Widget_Control, infoID, set_UValue=info, /no_copy
ENDIF
END




PRO xtable, data, Type = type, $
		XTitle = xtitle, YTitle = ytitle, $
		XSize = xsize, YSize = ysize, $
		CSel = csel, RSel = rsel, Group = group, $
		Wtitle = wtitle, Action = action, $
		buttons=buttons,noMenuBar=noMenuBar,infoText=infoText

ERROR = 0
result = size(data)
x_labels=string(indgen(result[1]))
y_labels=string(indgen(result[2]))
x = 0
y = 0
left   = -1L
right  = -1L
top    = -1L
bottom = -1L
selection = 0L

IF result[0] NE 2 THEN ERROR = 1

IF keyword_set(CSel) THEN BEGIN
	selsize = size (csel)
	IF selsize[0] EQ 1 THEN BEGIN
		; commented by srio
		;IF NOT keyword_set(RSel) THEN BEGIN
			IF csel[0] LE csel[1] THEN BEGIN
				IF csel[0] GE 0 THEN BEGIN
					IF csel[1] LT result[1] THEN BEGIN
						left = csel[0]
						right = csel[1]
						top = 0
						bottom = result[2] - 1
						selection = 1
					ENDIF
				ENDIF
			ENDIF
		;ENDIF
	ENDIF
ENDIF

IF keyword_set(RSel) THEN BEGIN
	selsize = size (rsel)
	IF selsize[0] EQ 1 THEN BEGIN
		; commented by srio
		;IF NOT keyword_set(CSel) THEN BEGIN
			IF rsel[0] LE rsel[1] THEN BEGIN
				IF rsel[0] GE 0 THEN BEGIN
					IF rsel[1] LT result[2] THEN BEGIN
;added by srio
if not(keyword_set(CSel)) then begin
						left = 0
						right = result[1] -1
endif
						top = rsel[0]
						bottom = rsel[1]
						selection = 1
					ENDIF
				ENDIF
			ENDIF
		;ENDIF
	ENDIF
ENDIF

IF keyword_set(XTitle) THEN BEGIN
	colsize = size(xtitle)
	if colsize[1] NE result[1] THEN ERROR = 1
	if colsize[0] NE 1 THEN ERROR = 1
	x_labels = xtitle
	if ERROR EQ 1 THEN message,/info, 'XTitle: wrong dimension!'
ENDIF	

IF keyword_set(YTitle) THEN BEGIN
	rowsize = size(ytitle)
	if rowsize[1] NE result[2] THEN ERROR = 1
	if rowsize[0] NE 1 THEN ERROR = 1
	y_labels = ytitle
	if ERROR EQ 1 THEN message,/info, 'YTitle: wrong dimension!'
ENDIF	

data = string(data)
data = strtrim(data, 2)
data = float(data)

IF keyword_set(Type) THEN BEGIN
	type = strupcase(type)
	CASE type OF
		'INTEGER' : data = fix(data)
		'FLOAT'   : data = float(data)
		'DOUBLE'   : data = double(data)
		'STRING'  : data = string(data)
		ELSE	  : ERROR = 1
	ENDCASE
ENDIF

IF ERROR EQ 0 THEN BEGIN

x_labels = strtrim (x_labels, 2)
y_labels = strtrim (y_labels, 2)

device, get_screen = screen

maxx = result[1]
WHILE (maxx * 78 + 102) GT screen[0] DO maxx = maxx - 1
WHILE (maxx * 78 + 102) LT screen[0] DO maxx = maxx + 1
maxx = maxx - 2

maxy = result[2]
WHILE (maxy * 24 + 62) GT screen[1] DO maxy = maxy - 1
WHILE (maxy * 24 + 62) LT screen[1] DO maxy = maxy + 1
maxy = maxy - 4

IF keyword_set(xsize) THEN BEGIN
	IF xsize GT result[1] THEN xsize = result[1]
	IF xsize GT maxx THEN xsize = maxx
	x = xsize
ENDIF

IF keyword_set(ysize) THEN BEGIN
	IF ysize GT result[2] THEN ysize = result[2]
	IF ysize GT maxy THEN ysize = maxy
	y = ysize
ENDIF

IF NOT (keyword_set(Wtitle)) THEN wtitle='XTable 1.0'

IF NOT (keyword_set(Action)) THEN action='OK'

IF Not(Keyword_Set(noMenuBar)) THEN $
  tlb   = Widget_Base (column=1, MBar=MenuBar, title=wtitle, $
			TLB_Frame_Attr=1)  ELSE $
  tlb   = Widget_Base (column=1, title=wtitle, TLB_Frame_Attr=1)
infos = Widget_Base (tlb)

IF Keyword_Set(buttons) THEN BEGIN
  wTmpB=Widget_Base(tlb,/Row)
  wtmp = WIDGET_BUTTON(wtmpB,VALUE='Accept',UValue='QUIT')
  wtmp = WIDGET_BUTTON(wtmpB,VALUE='Cancel',UValue='ABORT')
ENDIF

IF Keyword_Set(infoText) THEN BEGIN
  wTmpB=Widget_Base(tlb,/Col)
  FOR i=0L,N_Elements(infoText)-1 DO $
    wtmp = WIDGET_LABEL(wtmpB,VALUE=infoText[i],UValue='')
ENDIF

table = Widget_Table (tlb, xsize=result[1], ysize=result[2], $
			value=data, /Editable, /All_Events, /Scroll, $ 
			column_labels=x_labels, row_labels=y_labels, $
			Event_Pro='table_events')

IF Not(Keyword_Set(noMenuBar)) THEN BEGIN
FileMenu = WIDGET_BUTTON(MenuBar, 	VALUE='File', 	/MENU)
  babort = WIDGET_BUTTON(FileMenu,	VALUE='Exit and discard changes',	UValue='ABORT')
  bquit  = WIDGET_BUTTON(FileMenu, 	VALUE='Exit and save changes', 		UValue='QUIT')
EditMenu = WIDGET_BUTTON(MenuBar, 	VALUE='Edit',	/MENU)
  bappc  = WIDGET_BUTTON(EditMenu, 	VALUE='Append column', 		UValue='APPC')
  bappr  = WIDGET_BUTTON(EditMenu, 	VALUE='Append row', 		UValue='APPR')
  bdelr  = WIDGET_BUTTON(EditMenu, 	VALUE='Delete selected rows', 	UValue='DELR', /Separator)
  bdelc  = WIDGET_BUTTON(EditMenu, 	VALUE='Delete selected columns',UValue='DELC')
  blabel = WIDGET_BUTTON(EditMenu, 	VALUE='Rename Label', 		UValue='LABEL', /Separator)
OptMenu  = WIDGET_BUTTON(MenuBar, 	VALUE='Options',/MENU)
  bvisc  = WIDGET_BUTTON(OptMenu, 	VALUE='Number of visible columns', UValue='VISC')
  bvisr  = WIDGET_BUTTON(OptMenu, 	VALUE='Number of visible rows',    UValue='VISR')
ENDIF ELSE BEGIN
  blabel = 0L
  bdelr = 0L
  bdelc = 0L
ENDELSE

width = result[1]
IF keyword_set(xsize) THEN width = xsize * 78 + 102 $
ELSE BEGIN
	IF result[1] GT maxx THEN width = maxx
	width = width * 78 + 102
ENDELSE

height = result[2]
IF keyword_set(ysize) THEN height = ysize * 24 + 62 $
ELSE BEGIN
	IF result[2] GT maxy THEN height = maxy
	height = height * 24 + 62
ENDELSE

IF x EQ 0 THEN x = result[1]
IF y EQ 0 THEN y = result[2]

handler = {width:width, $
	   height:height, $
	   maxx:maxx, $
	   maxy:maxy, $
	   xsize:x, $
	   ysize:y, $
	   left:left, $
	   top:top, $
	   right:right, $
	   bottom:bottom, $
	   action:action}

ptr 	= Ptr_New (handler)
dataptr = Ptr_New (data)
xptr 	= Ptr_New (x_labels)
yptr 	= Ptr_New (y_labels)

info = {tableID:table, $
	delrID:bdelr, $
	delcID:bdelc, $
	labelID:blabel, $
	ptr:ptr, $
	xptr:xptr, $
	yptr:yptr, $
	dataptr:dataptr}
	
Widget_Control, table, column_widths=78
Widget_Control, table, row_heights=24
Widget_Control, info.tableID, scr_xsize=(*ptr).width 
Widget_Control, info.tableID, scr_ysize=(*ptr).height

IF Widget_Info(blabel,/Valid_Id) THEN Widget_Control, blabel, Sensitive = 0
Widget_Control, tlb, /Realize
Widget_Control, Widget_Info (tlb, /Child), set_UValue=info, /no_copy

IF selection EQ 1 THEN $
	Widget_Control, table, Set_Table_Select=[left,top,right,bottom]

XManager, 'xtable', tlb, Event='xtable_events',Group=group,/Modal
data = *dataptr
action = (*ptr).action
Ptr_Free, ptr
Ptr_Free, xptr
Ptr_Free, yptr
Ptr_Free, dataptr
ENDIF

END
