
FUNCTION mconvbackground,indata,delta=delta,kernel=kernel,niter=niter, $
  Return=ireturn

;+
; NAME:
;	MCONVBACKGROUND
;
; PURPOSE:
;	This function computes the background by an iterative 
;	procedure. In each iteration, the signal is convoluted with
;       a kernel. For each point the signal is replaced by the 
;       convolution value, only in the case that the latter is larger.
;
; CATEGORY:
;	Mathematics
;
; CALLING SEQUENCE:
;
;	result = MCONVBACKGROUND(inData)
;
; INPUTS:
;	inData: a one-dimensional array (y) with the data
;
; KEYWORD PARAMETERS:
;
;	NITER: The number of iterations (Default=5000)
;	KERNEL: Set this keyword to the convolution kernel. If this
;		keyword is set, the DELTA keyword is irrelevant. 
;		Default: kernel=[1,1,1]
;	DELTA: If KERNEL keyword is undefined, DELTA is an odd integer 
;	       meaning the width of the convolution kernel, which 
;	       is the identity vector (Default: DELTA=3, thus KERNEL=[1,1,1])
;	RETURN:  0: Returns background (Default)
;		 1: Returns InData-Background 
;
; OUTPUTS:
;       a one-dimensional array with the calculated background. 
;
; PROCEDURE:
;	The idea is given by Armando Sole. See, for instance: 
;	V. A. Sole, et al, Spectrochimica Acta B 62 (2007) 63-68:
;	pag 64:
;	Background can be estimated thru an iterative procedure 
;	in which the content of each channel is compared against the
;	average of the content of its neighbourgs at a distace of i 
;	channels. If the content is above the average, it is replaced 
;	by the average.
;
;	For the calculation, it uses the CONVOL function if IDL. The edge
;	values (in a width of delta/2) are not modified.
;
; EXAMPLE:
;
;		yb =  mconvbackground(y)
;
; MODIFICATION HISTORY:
; 	Written by:	M. Sanchez del Rio srio@esrf.fr, 2007-03-19
;
;
;-

on_error,2

;
; prepare inputs
;
IF N_Elements(iReturn) EQ 0 THEN iReturn=0
IF N_Elements(nIter) EQ 0 THEN nIter=5000L
IF N_Elements(delta) EQ 0 THEN delta=3L

IF Not(Keyword_Set(kernel)) THEN BEGIN
  kernel = Replicate(1D,delta)
ENDIF ELSE BEGIN
  delta = N_Elements(kernel)
ENDELSE

y = Double(Reform(inData))

n=N_Elements(y)
normFac = Total(kernel)
deltaHalf=delta/2


; 
; main loop
;
y1 = y

FOR i=0L,niter-1 DO BEGIN
  tmp = convol(y1,kernel)/normFac
  y1[deltaHalf:n-deltaHalf-1] = $
  y1[deltaHalf:n-deltaHalf-1]<tmp[deltaHalf:n-deltaHalf-1]
ENDFOR

CASE ireturn OF
 0:
 1: y1=inData-y1
 else:
ENDCASE
;
; return value
;
RETURN,y1 


END 

