;
;+
;
;==============================  Xpower   =====================================
;
; Xpower is a widget based graphical interface to calculate the transmission 
; in terms of flux or spectral power energy as a function pf photon
; energy for a set of up to five attenuators and mirrors. 
;
; It uses the DABAX files to get reflectivities and transmissions.
;
; DESCRIPTION OF THE CONTROLS IN THE MAIN WINDOW:
;
;  File:
;    Xpower input parameters: This option allows to save the current
;		parameters to a file for later loading. It also allows
;		to save the current parameters as defaults for being
;		used when the application is initialized. In the last
;		case, the file is named "application".xop (where 
;		"application " is the name of the current XOP
;		application) and is written in the directory pointed
;		by the XOP_DEFAULTS_DIR environment variable (which
;		must be set). The parameter file is ASCII and can be
;		read and edited with care.
;  Quit: to exit from the program
;
; Set_Parameters:
;  Set Parameters: to define the parameters for the calculation.
;		The same result is obtained pressing the "Set Parameters"
;		button in the main XPower window. 
;               Please refer to the information under the HELP
;		button for a complete description of the parameters. After 
;		pressing the ACCEPT button, XPower starts running.
;
;		If you whish to calculate values for a single energy point 
;		instead of an array of energy points [default], set the energy 
;		minimum equal to the energy maximum. In this case, the 
;		results will be presented in a common text window for all the 
;		three options in the "Show" menu.
;  Set Defaults: Sets the default parameters.
;
;  Show: Produces a graphic of the results (Local properties, cumulated
;	properties) and shows text parameters. In the case that a single
;	energy point is calculated (by setting the minimum energy equal
;	to the maximum energy), the three menu entries will display the
;	same text parameters.
;      
;       Two values of delta/beta paremeter (in local properties) are given:
;         i) delta/betaPCE where betaPCE is the "total" beta, i.e., calculated
;            from the total cross section (photoelectric+compton+rayleigh)
;         i) delta/betaPC where betaPC soes not consider coherent scattering,
;            i.e., from photoelectric + compton scattering.
;
;  Help:   Shows the xpower help (this text).
;
;
; COPYRIGHT:
;	xpower  belongs to XOP package and it is distributed within XOP.
;	PLEASE REFER TO THE XOP COPYRIGHT NOTICE BEFORE USING IT.
;
; CREDITS:
;	Published calculations made with XOP should refer:
;
;	  M. Sanchez del Rio and R. J. Dejus "XOP: Recent Developments"
;	  SPIE proceedings vol. 3448, pp.340-345, 1998.
;
; LAST MODIFICATION: msr/msr/99-03-16
;
;-
;
; -----------------------------------------------------------------------
;	Author: M. Sanchez del Rio (srio@esrf.fr) , ESRF, Jan 13, 1997
;		Based on the applications using the same philosophy but
;		different databases [xtransmit, xfilter, xtrans]
;	Modification history:
;	97/01/16 srio@esrf.fr adapts for Windows95.
;	97/03/05 srio@esrf.fr fix a bug in save/restore (noticed by F. Legrand)
;	97/03/24 srio@esrf.fr adapts parameter's window look to idl5.0b5
;	97/10/01 srio@esrf.fr renames read_ascii by rascii
;	97/10/27 srio@esrf.fr updates doc.
;	98/08/18 srio@esrf.fr makes a single point calculation. Adds "?"
;		option for formulas.
;	98/12/14 srio@esrf.fr adapts for xop2.0. Version 1.1.
;	99/03/16 srio@esrf.fr updates doc. Some text-bug fixed.
;	07/07/03 srio@esrf.eu adds dose information. v. 1.1
;	08/04/07 srio@esrf.eu adds Mu and Delta/Beta in local output
;	28/03/12 srio@esrf.eu documents that Beta (in Delta/Beta) is calculated
;                             from the total mu, i.e., includes Compton and 
;                             Rayleigh
;
;-
;
;========================================================================
;
FUNCTION Xpower_version
return,'1.12'
end
;
;========================================================================
;
PRO xpower_helppar,group=group,get_compounds=get_compounds
; the help screen in the Set_parameters window

if not(keyword_set(get_compounds)) then message,/info,'Creating help text...'
filecomp = 'Compounds.dat'
file = dabax_pathf(filecomp)
h = dabax_access(file)
n = spec_access(h)
mat_list = strarr(n)

for i=0,n-1 do mat_list(i)=spec_name(h,i+1,/index)

if keyword_set(get_compounds) then begin
  get_compounds=mat_list
  return
endif

sep = sdep(/ds)
txtfile = Xop_getenv('XOP_HOME')+sep+'doc'+sep+'txt'+sep+'xpower_par.txt'
text = read_textfile(txtfile)
text = [text,'       '+mat_list]
message,/info,'Done.'
;xdisplayfile1,text = mat_list,group = group, title='Material list [from DABAX: Compounds.dat]'
OpenW,unit,'xpower_tmp.txt',/Get_Lun
For i=0L,N_Elements(text)-1 DO PrintF,unit,text[i]
Free_Lun,unit
XDisplayFile1,'xpower_tmp.txt'
if sdep() eq "UNIX" then begin
  Wait,0.3 & Spawn,'rm xpower_tmp.txt',/sh
endif else begin
  Wait,0.3 & Spawn,'del xpower_tmp.txt'
endelse
end
;
;========================================================================
;

pro xpower_inp,str,nelem,substance,thick,angle,energies,source=source,flags,$
  density,roughness,group=group
;gives manipulable arrays from the main parameters structure

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
     'XPOWER_INP: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF

nelem = fix(str.nelements(0)) + 1
substance = strarr(nelem)
thick = fltarr(nelem)
angle = fltarr(nelem)
flags = intarr(nelem)
density = strarr(nelem)
roughness = fltarr(nelem)

for i=0,nelem-1 do begin
  command = 'substance(i) = str.el'+strcompress(i+1,/rem)+'_for'
  ;message,/info,'Executing: '+command
  itmp = execute(command)

  command = 'thick(i) = str.el'+strcompress(i+1,/rem)+'_thi'
  ;message,/info,'Executing: '+command
  itmp = execute(command)

  command = 'angle(i) = str.el'+strcompress(i+1,/rem)+'_ang'
  ;message,/info,'Executing: '+command
  itmp = execute(command)

  command = 'flags(i) = str.el'+strcompress(i+1,/rem)+'_flag(0)'
  ;message,/info,'Executing: '+command
  itmp = execute(command)

  command = 'density(i) = str.el'+strcompress(i+1,/rem)+'_den'
  ;message,/info,'Executing: '+command
  itmp = execute(command)

  command = 'roughness(i) = str.el'+strcompress(i+1,/rem)+'_rou'
  ;message,/info,'Executing: '+command
  itmp = execute(command)

endfor

;
; get energy
;
isource = strcompress(str.source(0),/rem)
if isource EQ '0' then begin
  ; 495 pts in [1,100] keV
  energies = findgen(496)/float(495)
  energies = energies*(100000.0-1000.0) + 1000.0
  ;source = 1
  source = energies*0.0+1.0
  return
endif
if isource EQ '1' then begin
  energies = findgen(str.ener_n)/float(str.ener_n-1)
  energies = energies*(str.ener_max-str.ener_min) + str.ener_min
  source = energies*0.0+1.0
endif else begin
  case isource of
    '0': return
    '2': file = str.source_file
    '3': file = 'SRCOMPE'
    '4': file = 'SRCOMPW'
    else:
  endcase
  if checkfile(file) NE 1 then begin
    itmp = widget_message(dialog_parent=group,$
	/ERROR,['File with source not found: '+ file,$
        'Using standard grid'])
    return
  endif else begin
    tmp = rascii(file)
    ncol = n_elements(tmp[*,0])
    if ncol LT 2 then begin
      itmp = widget_message(dialog_parent=group,$
	/ERROR,['Invalid file with source: '+ file,$
        'Using standard grid'])
      return
    endif 
    energies = tmp(0,*)
    source = tmp(1,*)
  endelse
endelse
end
;
;========================================================================
;

Function xpower_par,state,group=group
; presents the Show Parameters Window

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
     'XPOWER_PAR: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN,'Error'
ENDIF

if state.str.parameters.ener_min EQ state.str.parameters.ener_max then $
  isame = 1 else isame = 0
widget_control,/hourglass
handle_value,state.results.local,tmp_local
handle_value,state.results.cumul,tmp
handle_value,state.results.dens,dens
if n_elements(tmp) LE 2 then begin
  itmp = widget_message(dialog_parent=group,$
	/error,['No calculated data.',$
    'Please use the "Set Parameters" button before.'])
  return,'Error'
endif
source  = 0
xpower_inp,state.str.parameters,nelem, $
  substance,thick,angle,energies,flags,density,roughness,$
  source=source,group=group
npoints = n_elements(source)

tt = strarr(12)
tt = ['       '+$
  '*************************** Xpower Results ******************',$
  ' ','  Calculations using DABAX files: '+$
  state.str.parameters.Mu(1+fix(state.str.parameters.Mu(0)))+' and '+$
  state.str.parameters.f1f2(1+fix(state.str.parameters.f1f2(0)))]
if isame EQ 0 then  begin
  tt = [tt,' ' ,'  Source energy (start,end,points): '+$
    strcompress(energies(0),/rem)+',  '+$
    strcompress(energies(npoints-1),/rem)+',  '+$
    strcompress(npoints,/rem)]
endif else begin
  tt = [tt,' ' ,'  Photon energy: '+$
    strcompress(energies(0),/rem)]
endelse
tt=[tt,'  Number of optical elements: '+strcompress(nelem,/rem)]
if isame EQ 0 then begin
  I0 = int_tabulated(energies,source)
  tt = [tt,'  Incoming power [source integral]: '+strcompress(I0,/rem)]
  tt = [tt,'  Normalized Incoming power: '+strcompress(I0/I0,/rem)]
  I1 = I0
endif else begin
  tt = [tt,'  Source value: '+strcompress(source[0],/rem)]
endelse


j=0
for i=0,nelem-1 do begin
  tt = [tt,'  ']
  if flags(i) Eq 0 then begin
    tt=[tt,'      *****   oe '+strcompress(i+1,/rem)+'  [Filter] *************']
    tt=[tt,'      Material: '+substance(i)]
    tt=[tt,'      Density [g/cm^3]: '+StrCompress(dens[i],/Rem)]
    tt=[tt,'      thickness [mm] : '+strcompress(thick(i),/rem)]
  endif else begin
    tt=[tt,'      *****   oe '+strcompress(i+1,/rem)+'  [Mirror] *************']
    tt=[tt,'      Material: '+substance(i)]
    tt=[tt,'      Density [g/cm^3]: '+StrCompress(dens[i],/Rem)]
    tt=[tt,'      grazing angle [mrad]: '+strcompress(angle(i),/rem)]
    tt=[tt,'      roughness [A]: '+strcompress(roughness(i),/rem)]
  endelse
  if isame EQ 0 then begin
    I2 = int_tabulated(energies,tmp(2+i,*))
    tt =[tt, '      Outcoming power: '+strcompress(I2,/rem)]
    tt =[tt, '      Absorbed power: '+strcompress(I1-I2,/rem)]
    tt =[tt, '      Normalized Outcoming power: '+strcompress(I2/I0,/rem)]
    if flags(i) Eq 0 then $
       tt =[tt, '      Absorbed dose Gy.(mm^2 beam cross section)/s: '+strcompress((I1-I2)/(dens[i]*thick[i]*1e-6),/rem)]
    I1 = I2
  endif else begin
    j=j+1
    tt =[tt, '      Local transmission: '+strcompress(tmp_local[j,0],/rem)]
    j=j+1
    tt =[tt, '      Local absorption: '+strcompress(tmp_local[j,0],/rem)]
    tt =[tt, '      Cumulated transmission: '+strcompress(tmp[2+i,0],/rem)]
  endelse
endfor
return,tt
;xdisplayfile1,text=tt,=event.top,title='Xpower results'
end
;
;========================================================================
;
PRO xpower_event,event,GROUP=group
;

;Catch, error_status
;IF error_status NE 0 THEN BEGIN
;   Message,/Info,'error caught: '+!err_string
;   itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
;     'XPOWER_EVENT: error caught: '+!err_string)
;   Catch, /Cancel
;   If Type(stateid) EQ 3 THEN $
;     If Widget_Info(stateid,/Valid_Id) AND N_Elements(state) NE 0 THEN $
;     Widget_Control,stateid,Set_UValue=state
;   On_Error,2
;   RETURN
;ENDIF

WIDGET_CONTROL, event.id, GET_UVALUE=eventuval
stateid = Widget_Info(event.handler,/Child)
widget_control, stateid, get_uvalue = state

CASE eventuval OF
        'QUIT': BEGIN
                  WIDGET_CONTROL,event.top,/DESTROY
		END

        'FILEINPUT': BEGIN
          action=''
          Widget_Control,event.id, Get_Value=action
          CASE action OF
            'Load from file...': BEGIN
              ;if sdep() EQ 'UNIX' then filter='*.xop' else filter=0
              str_par = Xop_Input_Load(Title=$
              'Select xPower input file...',$
              /NoConf,Filter=filter,Group=event.top)
              IF Type(str_par) EQ 8 THEN BEGIN
                tmp = state.str.parameters
                Copy_Structure,str_par, tmp, Group=event.top , /OnlyFirstField
                state.str.parameters = tmp
              ENDIF
	      widget_control,stateid,set_uvalue=state
            END
            'Save to file...': BEGIN
              str_par = state.str.parameters
              Xop_Input_Save,str_par,File='xpower.xop',$
                /Write, Group=event.top, Comment='; xop/xpower(v'+$
              xpower_version()+') input file on '+SysTime()
            END
            'Save as default': BEGIN
              str_par = state.str.parameters
              Xop_Input_Save,str_par,Group=event.top, $
                Default='xpower.xop',Comment='; xop/xpower(v'+$
              xpower_version()+') input file on '+SysTime()
            END
          ENDCASE
        END
	'PARAMS': BEGIN
		tt = xpower_par(state,group=event.top)
		if tt(0) NE 'Error' then $
		xdisplayfile1,text=tt,group=event.top,title='Xpower results'
		END
        'SETDEF': BEGIN
                itmp = widget_message(dialog_parent=event.top,$
		/Question,['This option initializes the',$
                'xpower parameters to their default values.',$
                'Then you must click Set_parameters to run the program.',$
                'Please confirm:'],title='xpower')
                if itmp eq 'No' then return
                state.str.parameters = state.str_defaults
		widget_control,stateid,set_uvalue=state
                END
	'SETPAR': BEGIN
;		helpcmd =  $
;"xdisplayfileNative,'"+Xop_Getenv('XOP_HOME')+$
;		  SDep(/ds)+'doc'+SDep(/ds)+'xpower_par.txt'+"' & "+$
;                  "xpower_helppar"
		str = state.str.parameters
		device, GET_SCREEN = w_size
		w_size = (w_size(0) * [7.5,5]) /9
		XscrMenu,str,/NOTYPE,/INTERP,Nrow=7, $
		TITLES=state.str.titles, FLAGS=state.str.flags, $
		WTITLE='Xpower input data',ACTION=action,$
;HELP='xpower_helppar,group=event.top,/NoMenuBar', $
		HELP='xpower_helppar', $
		/scroll,w_size=w_size
		if action eq 'DONT' then goto,out
		;
		; checks for one-energy-point calculation
		;
		if str.ener_n LE 1 then begin
		  itmp = Dialog_Message(/Question,$
		    Dialog_Parent=event.top, $
		    ['Error: Number of energy points ( '+$
		    strcompress(str.ener_n,/rem)+' ) not allowed.',' ',$
		    'If you whish to calculate a single energy point, then',$
		    'use the same energy values for the minimum and maximum',$
		    'and use a number of points greater that 1 (e.g. 2).',$
		    ' ','Is this what you want (I will do it for you)?'])
		  if itmp EQ 'Yes' then begin
		    str.ener_n = 2
		    str.ener_max = str.ener_min
		  endif
		endif
		;
		; checks for existence of customized file source 
		;
		if strcompress(str.source[0],/rem) EQ '2' then begin
		  iok = checkfile(str.source_file)
		  if iok NE 1 then begin
		    itmp = Dialog_Message(/Question,$
		      Dialog_Parent=event.top, $
		      ['Error: File with source ( '+str.source_file+$
			' ) not found','',$
		      'Load a new one?'])
		    if itmp EQ 'No' then begin
		      itmp = Dialog_Message(/Info, Dialog_Parent=event.top, $
		       'Action aborted. Set Parameters again.')
			goto,out 
		    endif
		    file = dialog_pickfile(dialog_parent=event.top,/read,$
			/must_exist,group=event.top)
		    if file EQ '' then goto,out
		    str.source_file=file
		  endif
		endif
		;
		; checks for '?' in formulas
		;
		formulas = ([str.el1_for,str.el2_for,str.el3_for,str.el4_for,$
		  str.el5_for])[0:fix(str.nelements[0])]
		f_unknown = where(strcompress(formulas,/rem) EQ '?')
		if f_unknown[0] NE -1 then begin
		  tmp=1
		  xpower_helppar,get_compound=tmp
		  for i=0,n_elements(f_unknown)-1 do begin
		    j = f_unknown[i]
		    title=['select compound for oe '+strcompress(j+1,/rem),$
			'(compound formulas and densities are stored in the'+$
			' DABAX file: ',$
			Xop_GetEnv('DABAX_PATH')+SDep(/ds)+'Compounds.dat)']
		    compound = xlist1(tmp,/wait,title=title,group=event.top)
		    if compound EQ '' then begin
		      itmp = Dialog_Message(/Info, Dialog_Parent=event.top, $
		      'Not a valid compound. Action aborted. '+$
		      'Set Parameters again.')
		       goto,out 
		    endif
		    case j of
		      0: str.el1_for = compound
		      1: str.el2_for = compound
		      2: str.el3_for = compound
		      3: str.el4_for = compound
		      4: str.el5_for = compound
		      else:
		    endcase
		  endfor
		endif
		;	
		;	
		state.str.parameters = str
		widget_control,stateid,set_uvalue=state
		;
		; performs calculations
		;
		widget_control,/hourglass
		message,/Info,'Performing calculations. Please wait...'
		source  = 0
		xpower_inp,state.str.parameters,nelem, $
		  substance,thick,angle,energies,flags,density,roughness,$
		  source=source,group=event.top
		if n_elements(energies) EQ 0 then begin ; default source
		  ; 495 pts in [1,100] keV
		  energies = findgen(496)/float(495)
		  energies = energies*(100000.0-1000.0) + 1000.0
		  source = 1.0+energies*0.0
		  state.str.parameters.source[0]='0'
		  widget_control,stateid,set_uvalue=state
		endif
		;local = fltarr(1+2*nelem,n_elements(energies))
		local = fltarr(1+6*nelem,n_elements(energies))
		local(0,*) = energies
		cumul = fltarr(2+nelem,n_elements(energies))
		cumul(0,*) = energies
		cumul(1,*) = source
		MuDataSet = $
		  state.str.parameters.Mu(1+fix(state.str.parameters.Mu(0)))
		fDataSet = $
		  state.str.parameters.f1f2(1+fix(state.str.parameters.f1f2(0)))
		icount = 0
		for i=0,nelem-1 do begin
;		  print,' '
;		  print,'***** oe: ',i+1
;		  print,'***** formula: ',substance(i)
;		  print,'***** density: ',density(i)
		  ; gets density
		  dens=0.0
		  if strcompress(density(i),/rem) eq '?' then $
		    ;"temporary" used to undefine dens
		    junk = temporary(dens) else $
		    dens=float(density(i))
		  itmp = where( state.str.mat_list EQ substance(i))
		  if itmp(0) eq -1 then name = 0 else name = 1

                  
		  tmpA = cross_calc_mix(MuDataSet, substance(i),$
			energies,fcol_tit,density=dens,unit=3,NAME=name,$
			PARTIAL='TotalCrossSection[barn/atom]')

		  tmpCoh = cross_calc_mix(MuDataSet, substance(i),$
			energies,fcol_tit,density=dens,unit=3,NAME=name,$
			PARTIAL='RayleighCoherent[barn/atom]')

      IF N_Elements(tmpA) NE N_Elements(energies) THEN BEGIN
        itmp = Dialog_Message(['Ambiguity in calling cross_calc '+$
          'with label: '+'TotalCrossSection[barn/atom]',$
          'Using first column'])
        tmpA  = Reform(tmpA[0,*])
      ENDIF

		  tmpB = f1f2_calc_mix(fDataSet, substance(i),$
			energies,density=dens,F=10,NAME=name,$
			theta=angle(i), roug=roughness(i) )
		  tmpDelta = f1f2_calc_mix(fDataSet, substance(i),$
			energies,density=dens,F=3,NAME=name)
		  tmpBetaPCE = (tmpA/energies)*physical_constants('hc')*1D-8/4/!dpi
		  tmpBetaPC = ((tmpA-tmpCoh)/energies)*physical_constants('hc')*1D-8/4/!dpi

		  if flags(i) eq 0 then begin ;   filter
;		    print,'***** thick[mm]: ',thick(i)
		    tmp1 = exp(-1.0E-1*tmpA*thick(i))
                    ; transmission
		    icount = icount+1
		    local(icount,*) = tmp1
		    icount = icount+1
                    ; absorttion
		    local(icount,*) = 1.0-tmp1
		    cumul(2+i,*) = cumul(1+i,*)*tmp1

		  endif else begin           ;   mirror
;		    print,'***** angle: ',angle(i)
;		    print,'***** roughness: ',roughness(i)
		    icount = icount+1
		    local(icount,*) = tmpB
		    icount = icount+1
		    local(icount,*) = 1.0-tmpB
		    cumul(2+i,*) = cumul(1+i,*)*tmpB
		  endelse
                  ; mu
		  icount = icount+1
		  local(icount,*) = tmpA
; 
                  ; delta
		  icount = icount+1
                  local(icount,*) =tmpDelta
;
;                  ; beta
;      icount = icount+1
;      local(icount,*) =tmpBeta
;
		              ; delta/betaT
		  icount = icount+1
		  local(icount,*) =tmpDelta/tmpBetaPCE
		  icount = icount+1
		  local(icount,*) =tmpDelta/tmpBetaPC
		  IF i EQ 0 THEN densall = dens ELSE densall=[densall,dens]
		endfor
		; stote data
		handle_value,state.results.local,local,/Set
		handle_value,state.results.cumul,cumul,/Set
		handle_value,state.results.dens,densall,/Set
		message,/Info,'Done.'
		END
	'LOCAL':	BEGIN
		nelem = fix(state.str.parameters.nelements(0)) + 1
		coltitles = ['Photon Energy [eV]',replicate('',6*nelem)]
		icount = 0
		for i=1,nelem do begin
		  icount = icount + 1
		  flag=0
		  command = 'flag = fix(state.str.parameters.el'+$
			strcompress(i,/rem)+'_flag(0))'
		  itmp = execute(command)
		  if flag eq 0 then $
		     coltitles(icount) = '[oe '+strcompress(i,/rem)+'] Trans'$
		     else $
		     coltitles(icount) = '[oe '+strcompress(i,/rem)+'] Refl'
		  icount = icount + 1
		  coltitles(icount) = '[oe '+strcompress(i,/rem)+'] Absor'
		  icount = icount + 1 
		  coltitles(icount) = '[oe '+strcompress(i,/rem)+'] Mu[cm^-1]'
;		  icount = icount + 1
;		  coltitles(icount) = '[oe '+strcompress(i,/rem)+'] delta'
;      icount = icount + 1
;      coltitles(icount) = '[oe '+strcompress(i,/rem)+'] beta'
                  icount = icount + 1
                  coltitles(icount)='[oe '+strcompress(i,/rem)+'] delta'
                  icount = icount + 1
                  coltitles(icount)='[oe '+strcompress(i,/rem)+'] delta/betaPCE'
                  icount = icount + 1
                  coltitles(icount)='[oe '+strcompress(i,/rem)+'] delta/betaPC'
		endfor
		handle_value,state.results.local,tmp
		if n_elements(tmp) LE 2 then begin
		  itmp = widget_message(dialog_parent=event.top,$
			/error,['No calculated data.',$
			'Please use the "Set Parameters" button before.'])
		  return
		endif
		if state.str.parameters.ener_min NE $
		  state.str.parameters.ener_max then begin
                  ncol = N_Elements(tmp[*,0])
		  Xplot,tmp,GROUP=event.top, xtitle='-1',ytitle='-1', $
		   title= 'Local properties of optical elements', $
		   coltitles=coltitles,wtitle='Xpower results',$
		   no_block=state.no_block,Ycol=ncol-3
		endif else begin
		  tt = xpower_par(state,group=event.top)
		  if tt(0) NE 'Error' then $
		  xdisplayfile1,text=tt,group=event.top,title='Xpower results'
		endelse
		END
	'CUMUL':	BEGIN
		nelem = fix(state.str.parameters.nelements(0)) + 1
		coltitles = ['Photon Energy [eV]','Source',replicate('',nelem)]
		for i=1,nelem do $
		  coltitles(i+1) = 'Intens after oe #'+strcompress(i,/rem)
		handle_value,state.results.cumul,tmp
		if n_elements(tmp) LE 2 then begin
		  itmp = widget_message(dialog_parent=event.top,$
			/error,['No calculated data.',$
			'Please use the "Set Parameters" button before.'])
		  return
		endif
		if state.str.parameters.ener_min NE $
		  state.str.parameters.ener_max then begin
		  Xplot,tmp,GROUP=event.top, xtitle='-1',ytitle='-1', $
		   title= 'Cumulative transmission after optical elements', $
		   coltitles=coltitles,wtitle='Xpower results',$
		   no_block=state.no_block
		endif else begin
		  tt = xpower_par(state,group=event.top)
		  if tt(0) NE 'Error' then $
		  xdisplayfile1,text=tt,group=event.top,title='Xpower results'
		endelse
		END
        'HELP':  BEGIN
                Xhelp,'xpower',group=event.top
                END
	else:
endcase
;
out:
end
;
;========================================================================
;
PRO xpower,GROUP=group, InputFile=inputFile, No_Block=no_Block
;
Forward_Function xop_defaults

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   itmp = Dialog_Message(/Error,Dialog_Parent=group, $
     'XPOWER: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN
ENDIF
;if xregistered('xpower') then return

;
; create widgets
;
IF N_Elements(no_block) EQ 0 THEN no_block=1
base=WIDGET_BASE(/COLUMN,TITLE='Xpower '+xpower_version(),MBAR=wMenuBar)


wButtons = widget_base(base,/Column) ; also to store state

wFile = widget_button(wMenuBar,VALUE='File',/MENU)

  wtmp0 = widget_button(wFile,VALUE='Xpower input parameters', /Menu)
    wtmp = widget_button(wtmp0,VALUE='Load from file...',UValue='FILEINPUT')
    wtmp = widget_button(wtmp0,VALUE='Save to file...',UValue='FILEINPUT')
    wtmp = widget_button(wtmp0,VALUE='Save as default',UValue='FILEINPUT')
  wtmp = widget_button(wFile,VALUE='Quit', UVALUE='QUIT',/SEPARATOR)

wSetParameters = widget_button(wMenuBar,VALUE='Set_Parameters', /MENU)
  wtmp = widget_button(wSetParameters,VALUE='Set Parameters', UVALUE='SETPAR')
  wtmp = widget_button(wSetParameters,VALUE='Set Defaults', UVALUE='SETDEF')

wResults = widget_button(wMenuBar,VALUE='Show',/MENU)
  wtmp = widget_button(wResults,VALUE='Local Absorption & Transmission', $
    UVALUE='LOCAL')
  wtmp = widget_button(wResults,VALUE='Cumulative transmission',$
    UVALUE='CUMUL')
  wtmp = widget_button(wResults,VALUE='Parameters', UVALUE='PARAMS')

wHelp = widget_button(wMenuBar,VALUE='Help', /Help)
  wtmp = widget_button(wHelp,VALUE='xpower', UVALUE='HELP')

wtmp = widget_button(BASE,VALUE='Set Parameters', UVALUE='SETPAR')

if sdep() EQ 'WINDOWS' then begin
  ;font1 = 'VERDANA*BOLD*ITALIC*36'
  font2 = 'VERDANA*BOLD*ITALIC*24'
endif else begin
  ;font1='-b&h-lucida-bold-i-normal-sans-24-240-75-75-p-151-iso8859-1'
  font2 = '-adobe-helvetica-bold-o-normal--18-180-75-75-p-104-iso8859-1'
endelse
font1=font2

wtmp = WIDGET_LABEL( BASE, FONT=font1, VALUE=' Xpower')
wtmp = WIDGET_LABEL( BASE, FONT=font2, $
  VALUE=' Attenuation & reflectivity in media.')
wtmp = WIDGET_LABEL( BASE, FONT=font2, $
  VALUE=' Effect on source spectrum')


; stryucture to store the results
results = {local:handle_create(), cumul:handle_create(), dens:handle_create() }

str = xop_defaults('xpower')
str_defaults = str.parameters

IF KeyWord_Set(inputFile) THEN BEGIN
  str_par = Xop_Input_Load(InputFile=inputFile)
  IF Type(str_par) EQ 8 THEN BEGIN
    tmp = str.parameters
    Copy_Structure,str_par, tmp, Group=group, /OnlyFirstField
    str.parameters = tmp
  ENDIF
ENDIF

state = { str:str, str_defaults:str_defaults, results:results, $
          no_block:no_block }
widget_control,Widget_Info(base,/Child),set_uvalue=state
Widget_control,/Realize,base
Xmanager,'xpower',base,Group_leader=group, No_Block=no_Block
;
end

