function dabax_select,h,descriptor,casematch=casematch,group=group, $
  return=flag,new_descriptor=new_descriptor,full_match=full_match
; 
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       DABAX_SELECT
;
; PURPOSE:
;	Returns the DABAX file index matching the descriptor.
;	
; CATEGORY:
;       X-Ray optics. DABAX data base.
;
; CALLING SEQUENCE:
;	out = dabax_select(h,descriptor,casematch=casematch)
; INPUTS:
;	h: the DABAX (or SPEC) handle pointing to the DABAX file.
;	descriptor: an identifier to be find in the scan title (i.e. 'Si').
; KEYWORDS:
;	casematch = when this keyword is set, the search is case sensitive.
;	Full_match: If set, the "descriptor" word must match perfectly the 
;		scan entry in the DABAX file. If not set (default), the 
;		string "descriptor" is searched in the scan titles, and 
;		multiples matches can be found. 
;	group = The widget ID of the widget that calls the routine.
;		This is passed to widget_message in its dialog_parent
;		keyword to center the window message.
;	return = a flag for indicating the returning value. 
;		0: a single index is returned (if multiple scans 
;			match the descriptor, a pop-up window is 
;			started asking the user to select a single one).
;		1: returns ALL matches
;	new_descriptor = set to a named variable to indicate the 
;		selected descriptor (in the case of multiple matches). 
;		Useful for avoiding selecting many times in multiple calls. 
;
; OUTPUT:
;	out: the scan index which title fits the searched descriptor.
;
; PROCEDURE:
;	Manipulation of strings. Use of specaccess library.
;
; EXAMPLES:
;	h = dabax_access('AtomicConstants.dat')
;	print,dabax_select(h,'Si')
;
; MODIFICATION HISTORY:
;       96-10-10 Written by M. Sanchez del Rio (srio@esrf.fr)
;       97-01-14 srio@esrf.fr port to Windows.
;       97-10-16 srio@esrf.fr adds group kw. Uses sdep()
;       00-07-12 srio@esrf.fr adds catch error trapping
;       03-04-22 srio@esrf.fr adds New_Descriptor and Full_Select keywords. 
;-

Catch,error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp=Dialog_Message(/Error, $
    'DABAX_SELECT: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN,0
ENDIF

new_descriptor = descriptor

if keyword_set(casematch) then begin
  des = descriptor
  titles =  h.scan(*).name
endif else begin
  des = strupcase(descriptor)
  titles =  strupcase(h.scan(*).name)
endelse

IF Keyword_Set(full_match) THEN BEGIN
  good = Where(titles EQ des)
ENDIF ELSE BEGIN
  titles = titles+' '  ; to allow search for 'O ' and do not take 'Os'
  a=strpos(titles,des)
  good = where(a NE -1)
ENDELSE

if strcompress(good(0),/rem) EQ '-1' then begin
  txt = 'Descriptor ('+descriptor+') not found in dabax file: '+$
	h.specfile
  if sdep(/w) then $
	itmp = widget_message(dialog_parent=group,/error, txt) else $
	message,/info,txt
  return,0
endif

IF Keyword_Set(flag) THEN RETURN,long(good) + 1

if n_elements(good) GT 1 then begin
  ;if sdep(/w) then begin
    list = strarr(n_elements(good))
    for i=0,n_elements(good)-1 do list[i] = spec_name(h,good(i)+1,/index)
    i = wmenu2(list,Dialog_Parent=group)
    new_descriptor = list[i]
    igood = good(i) + 1
  ;endif else begin
  ;  print,des+" found in scans: ',good
  ;  print,'Please select a single scan: '
  ;    for i=0,n_elements(good)-1 do print,i+1,' '+spec_name(h,good(i)+1,/index)
  ;  read,'<?> ',i
  ;  igood = good(i-1) + 1
  ;endelse
endif else begin
  igood = long(good(0)) + 1
endelse
return,igood
end
