Function atomic_weights,mdescriptor,h,file=input,return=flag,group=group, $
  full_match=full_match

;+
; NAME:
;       ATOMIC_WEIGHTS
;
; PURPOSE:
;	Returns atomic weights from DABAX.
;	
; CATEGORY:
;       X-Ray optics. DABAX data base.
;
; CALLING SEQUENCE:
;	out = atomic_constants(id,file,return=return)
; INPUTS:
;	id: an identifier (or an array of identifiers) to be found in the 
;	scan title (i.e. 'Si')
;
;	If descriptor is of the form "natural.symbol" (e.g., natural.Ge), 
;	  the AveragedAtomicMass is returned. This is equivalent to a call 
;	  with descriptor=symbol with keywors RETURN=2.
; OPTIONAL INPUTS:
;	h (input/output) the dabax handle of the file to be used.
;	  If h is undefined or equal to 0, then the returned value is the
;	  dabax handle. If h is defined, then uses this value to point to
;	  the dabax file. This option is useful when calling 
;	  atomic_constants sequantially: the first call initiallizes h and 
;	  this value is reused (saving time) in the following calls.
;	
; KEYWORDS:
;	Full_Match: If set, the "descriptor" word must match perfectly the 
;		scan entry in the DABAX file. If not set (default), the 
;		string "descriptor" is searched in the scan titles, and when
;		multiples matches are found, the user is asked to select one. 
;		(this keyword is inactivated if descriptor is of the type:
;		 natutal.Symbol)
;	File = the DABAX  inout file (if not set, AtomicConstants.dat is taken)
;	Return = a flag indicating the variable to be returned. 
;		0 for AtomicMass
;		1 for Abundance
;		2 AveragedAtomicMass
;		   (not ask for isotope is multiple matches are found, 
;		    as value should be the same).
;
;       group = The widget ID of the widget that calls the routine.
;               This is passed to widget_message in its dialog_parent
;               keyword to center the window message.
;
;
; OUTPUT:
;	out: the value of the selected parameter
;
; PROCEDURE:
;	It uses the values from the DABAX file AtomicWeights.dat
;
; EXAMPLES:
;	Atomic Mass of the isotope 70Ge
;	IDL> print,atomic_weights('70Ge',ret=0)
;	       69.924300
;	Atomic Mass of the natural Ge
;	IDL> print,atomic_weights('Ge',ret=2)
;	       72.641000
;	IDL> print,atomic_weights('natural.Ge') 
;	       72.641000
;
;
; MODIFICATION HISTORY:
;       2003-04-02 Written by M. Sanchez del Rio (srio@esrf.fr)
;-
on_error,2

if not(keyword_set(input)) then input='AtomicWeights.dat'
if not(keyword_set(h)) then h = dabax_access(input)
IF N_Elements(flag) EQ 0 THEN flag=0


FOR MM = 0,N_ELEMENTS(mdescriptor)-1 DO BEGIN
  descriptor = mdescriptor(MM)

  ; looks for descriptor of the type natural.Ge
  tmp=strPos(descriptor,'.')
  IF tmp[0] NE -1 THEN BEGIN
     descriptor=StrMid(descriptor,tmp+1)
     flag=2
     full_match=0
  ENDIF

  ; to handle elements with 1 char symbol:
  IF Not(Keyword_Set(full_match)) THEN $
    if strlen(descriptor) eq 1 then descriptor = descriptor+' '

  CASE flag OF
    2: BEGIN
       igood = dabax_select(h,descriptor,/casematch,group=group,Return=1, $
		full_match=full_match)
       igood = igood[0]
       data = spec_data(h,igood,/index)
       val = data[2]
       END
    else: BEGIN
       igood = dabax_select(h,descriptor,/casematch,group=group, $
		full_match=full_match) 
       data = spec_data(h,igood,/index)
       val = data[flag]
       END
  ENDCASE

  ;print,'Using scan: ',spec_name(h,igood,/index)
  z = double(strsubstitute(spec_headers(h,igood,'#S',/index),'#S',''))

if n_elements(out) eq 0 then out = val else out=[out,val]
ENDFOR

return,out
end
