;
;
;
FUNCTION peakfinder,y, $
  z=z,$   ; input, number of smooth iterations
  m=m,$   ; input, window of smooth, w=2*m+1
  threshold=threshold, $ 
  analyzethreshold=analyzethreshold, $
  algorithm=algorithm, $
  plot=iplot, Group=group, NPeaks=nPeaks

;+
; NAME:
;	PEAKFINDER
;
; PURPOSE:
;	This function finds the peaks of a 1-D data. 
;
;       It returns a multicolumn array with the abscissa values of 
;	the found peaks
;
;	If no peaks are found, it returns [-1]
;
; CATEGORY:
;	Maths.
;
; CALLING SEQUENCE:
;	result = PeakFinder(y[,x])
;
; INPUTS:
;	y:	  An array with input (ordinates) data.
;
; KEYWORD PARAMETERS:
;
;	INPUTS: 
;	Group:	The widget id of the caller. This is used to center
;	 	the Dialog_Message window(s).
;	m: the number of neighbours in the smooth (smooth window=2*m+1) 
;	z: the number of smooth iterations
;	threshold: The theshold for finding points (Default=0)
;	analyzethreshold: Define this keyword to get an XPlot 
;		window with the Number of Peaks versus Threshold. 
;	algorithm: 0 uses a method based on first derivative (default=0)
;                  1 uses a method based on second derivative. 
;	nPeaks: (output) Number of peaks.
;
; OUTPUTS:
;	An array with the indices of the peaks. 
;
; PROCEDURE:
;       Two methods are proposed:
;       1) Based on first derivative:  
;          i) calculate first derivarive 
;          ii) smooth it (with parameters m,z) 
;          iii) search for points with zero derivative (peak candidates)
;          iv) calculate the maximum amplitude (delta_max) of the derivative 
;              data when going through a zero value
;          v) comparison of each amplitude (delta) of the derivative data when 
;             trought a zero value with delta_max: 
;              if (delta/delta_max > theshold) :=> Peak 
;        
;       2) Based on second derivative:  
;          i) calculate second derivarive S of y 
;          ii) smooth it (with parameters m,z) 
;          iii) Take the negative part of S and calculate the absolute value |S|
;          iv) Calculate the intervals where |S| is clipped by zeroes.
;          v) Take a peak candidate in each interval (the point with highest y
;       value)
;          vi) if |S|>t*F :=> Peak, with: 
;               F is the standard deviation of F
;               t is a normalised threshold, i.e., it gives the maximum number 
;               of peaks for t=0 and no peaks for t=1. 
;        
;       The Number of Peaks is not linear with the threshold. For selecting
;       adequated 
;       theshold values the possibility of analysing the threshold (plot of
;       Number of 
;       peaks versus threshold value) is also available.  
;        
;       References: 
;         M.A. Mariscotti, Nucl. Instrum. Meth. 50 (1967) 309-320 
;         Documentation of FullProf code, Juan Rodriguez-Carvajal.
;       
;
; MODIFICATION HISTORY:
; 	Written by:	M. Sanchez del Rio, srio@esrf.fr
;	24 Sept 2007  (Completely rewritten. Old version with the same name)
;-

Catch, error_status
IF error_status NE 0 THEN BEGIN
   Message,/Info,'error caught: '+!err_string
   IF SDep(/w) THEN itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
     'PEAKFINDER: error caught: '+!err_string)
   Catch, /Cancel
   On_Error,2
   RETURN,0
ENDIF


IF N_Elements(algorithm) EQ 0 THEN algorithm=0 ; 0=First Der, 1=Second Der 

npts = N_Elements(y)
IF N_Elements(iplot) EQ 0 THEN iplot=1
IF N_Elements(threshold) EQ 0 THEN threshold = 0.0
IF N_Elements(m) EQ 0 THEN m = 1
IF N_Elements(z) EQ 0 THEN z = 5


w=2*m+1 ; always odd

print,'PEAKFINDER: using m=',m
print,'PEAKFINDER: using w=',w
print,'PEAKFINDER: using z=',z
print,'PEAKFINDER: using algorithm=',algorithm



;
; Calculate the first and second derivative with reduced variance
;
kernelFD = smoothkernel(m,z,seed=[-1,1,0],/Norm) ; for first derivative
IF N_Elements(kernelFD) GT N_Elements(y) THEN $
   Message,'Too few data points for defined window.'
kernelS = smoothkernel(m,z,/Norm) ; for second first derivative
;

CASE algorithm OF

0:BEGIN  ; Look for zeros of  f'. 
  FD = Convol(Double(y),kernelFD,1D0)  ; first derivative
  ;
  ; search of points with zero derivative
  ;
  fdminus= shift(fd,+1)
  fdplus= shift(fd,-1)
  i_peaks = Where (fdminus GT 0 and fd LT 0) 
  IF i_peaks[0] NE -1 THEN BEGIN
    idiffless = where(y[i_peaks]-y[i_peaks-1] LT 0)
    IF idiffless[0] NE -1 THEN i_peaks[idiffless]=i_peaks[idiffless]-1
    idiffplus = where(y[i_peaks+1]-y[i_peaks] GT 0)
    IF idiffplus[0] NE -1 THEN i_peaks[idiffplus]=i_peaks[idiffplus]+1
  ENDIF ELSE BEGIN
    npeaks=0
    i_peaks=[-1]
  ENDELSE
  ;
  ; for each peak, compute delta
  ;
  npeaks = N_Elements(i_peaks)
  delta = FltArr(npeaks)
  FOR i=0L,npeaks-2 DO BEGIN
    tmp = fd[i_peaks[i]:i_peaks[i+1]]
    delta[i] = max(abs(tmp))
  ENDFOR
  delta = delta/max(delta)
  END
1:BEGIN  ; Mariscotti. Look for intervals where f''<0. Then take Ymax for each interval
  S = Convol(Double(y),kernelS)        ; second derivative
  F = Sqrt(Convol(Double(y),kernelS^2)) ; Str Dev. See Equation [10] in Mariscotti's
  ;
  ; compute zones (intervals) where peaks sit (i.e., where S is negative)
  ;
  ss = abs(s<0)
  ssminus= shift(ss,+1)
  ssplus= shift(ss,-1)
  edgesLeft =  where (ssplus-ss NE 0 AND ssplus-ss EQ ssplus)
  edgesRight = where (ssplus-ss NE 0 AND -(ssplus-ss) EQ ss)
  ;
  ; compute all peaks (i.e., for threshold=0)
  ;
  IF edgesLeft[0] NE -1 THEN BEGIN
    npeaks = N_Elements(edgesLeft)
    i_peaks = LonArr(npeaks)
    FOR i=0L,npeaks-1 DO BEGIN
      IF edgesLeft[i] GE edgesRight[i] THEN Message,'Bad calculations of edges'
      tmps = y[edgesLeft[i]:edgesRight[i]]
      tmpx = LIndGen(edgesRight[i]-edgesLeft[i]+1)
      tmpmax = Max(tmps,tmpi)
      i_peaks[i]=edgesLeft[i]+tmpi
    ENDFOR
   ENDIF ELSE BEGIN
    npeaks=0
    i_peaks=[-1]
   ENDELSE
  END

  ELSE: 
ENDCASE

;
; compute maximum threshold 
;

thMax = 1.0 ; upper limit
thAbs = threshold
IF algorithm EQ 1 THEN BEGIN
  nnn=0
  itmp = 0
  WHILE nnn NE 1 DO BEGIN
    itmp = itmp+1
    igood = Where(ss[i_peaks] GE thMax*f[i_peaks],nnn)
    IF nnn EQ 0 THEN thMax=thMax/2 
    IF nnn GT 1 THEN thMax=thMax*1.5
    ; stop after 10 iterations
    IF itmp GE 10 THEN BEGIN
      Message,/Info,'Failed to compute maximum theshold.'
      nnn=1
    ENDIF
  ENDWHILE
  thAbs = threshold*thMax  ; absolute threshold
ENDIF
print,'PEAKFINDER: Maximum non-normalized threshold: ',thMax
print,'PEAKFINDER: Normalized threshold: ',threshold
print,'PEAKFINDER: Absolute threshold: ',thAbs

;
; analyze threshold
;
IF Keyword_Set(analyzeThreshold) THEN BEGIN
  thi = makearray1(100,1e-5,thMax,/log) 
  nnp = thi*0D0
  FOR i=0L,N_Elements(thi)-1 DO BEGIN
    IF algorithm EQ 1 THEN $
      igood = Where(ss[i_peaks] GE thi[i]*f[i_peaks],nnn) ELSE $
      igood = Where(delta GT thi[i],nnn)
    nnp[i]=nnn
  ENDFOR
  thi=[0,thi]
  nnp=[npeaks,nnp]
  xplot,Make_Set(thi/thMax,thi,nnp),/Xlog,/No_Block, xrange=[1e-5,thMax],$
    colTitles=['Normalised Threshold','Absolute threshold','Number of Peaks'],$
    xtitle='-1',ytitle='-1',wtitle='Threshold analysis'
ENDIF

;
; remove peaks using threshold
;
IF algorithm EQ 1 THEN BEGIN
   igood = Where(ss[i_peaks] GE thAbs*f[i_peaks])
ENDIF ELSE BEGIN
   igood = Where(delta GT thAbs,nnn)
ENDELSE

IF igood[0] EQ -1 THEN BEGIN
   itmp = Dialog_Message(['No peaks left with this threshold value: '+StrCompress(threshold),$
      'Keeping all peaks'])
ENDIF ELSE BEGIN
   print,'PEAKFINDER: Using absolute threshold: ',thAbs
   print,'PEAKFINDER: Number of peaks found: ',N_Elements(i_peaks)
   print,'PEAKFINDER: Number of peaks after applying threshold: ',N_Elements(igood)
   i_peaks = i_peaks[igood]
   npeaks = N_Elements(i_peaks)
ENDELSE


IF i_peaks[0] EQ -1 THEN BEGIN
  Message,/Info,'No peaks found'
  IF SDep(/w) THEN itmp = Dialog_Message('PEAKFINDER: No peaks found.',$
	Dialog_Parent=group)
  npeaks=0L
  RETURN,[-1]
ENDIF

RETURN,i_peaks

END
