FUNCTION electron_e2w,V,direction

;+
; NAME:
;	ELECTRON_E2W
;
; PURPOSE:
;	Conversion Potential(Volts) <-> Wavelength(meters) for
;       relativistic electrons. 
;
; CATEGORY:
;	Physics.
;
; CALLING SEQUENCE:
;
;	result = electron(input,direction)
;
; INPUTS:
;	input: the value to be converted
;       direction: 
;         1 (default): input is electron energy (volts for potential, 
;           eV for e- energy), result is wavelength in meters
;         -1 : input is wavelength in meters, result is potential (volts)
;
; PROCEDURE:
;
;       See formula in wikipedia (search for electron diffraction)
;
;       Other calculations: 
;
;       For non-relativistic electrons use: 
;       wavelength [A] = physical_constants('HCE')/sqrt( energy[eV] )
;
;       For neutrons use: 
;       wavelength [A] = physical_constants('HCN')/sqrt( energy[eV] )
;
;       For photons use: 
;       wavelength [A] = physical_constants('HC')/ energy[eV]
;
; EXAMPLE:
;		print,electron_e2w(200e3)
;               2.5079339e-12 ; i.e, 200 kV = 2.5 pm
;		print,electron_e2w(2.5079339e-12,-1)
;               199999.99
;
; MODIFICATION HISTORY:
; 	Written by:	M. Sanchez del Rio  2008/02/06
;-
IF N_Elements(direction) EQ 0 THEN direction=1

h = physical_constants('H')
me = physical_constants('ME')
ec = physical_constants('EC')
c = physical_constants('C')

CASE direction OF
  1: BEGIN
     den1 = 2D0*me*ec*V
     den2 = ec*V/(2*me*c*c)+1D0
     out = h/sqrt(den1*den2)
     END
  -1: BEGIN
     lambda=V
     AA = (lambda*ec/c)^2
     BB = 2*lambda^2*me*ec
     CC = -h^2
     DISC = BB^2-4*AA*CC
     IF disc LT 0 THEN BEGIN
       itmp = Dialog_Message(/Error,[ $
         'ELECTRON_E2W: Error: Negative discriminant', ' Returning zero'])
       out = 0 
     ENDIF ELSE BEGIN
       out = (-BB+sqrt(DISC))/2/AA
     ENDELSE
     END
  else: Message,'Not valid direction: '+StrCompress(direction)
ENDCASE
RETURN,out

END

