FUNCTION xrf_radrate,element,line,File=file
;______________________________________________________________________________
;+
; NAME:
;       XRF_RadRate()
;
; PURPOSE:
;       This function returns the radiation rate vualue (or matrix) in a 
;	fluorescence line (or array of lines) of an element (or array of 
;	elements).
;
; CATEGORY:
;       DABAX - X-ray fluorescence
;
; CALLING SEQUENCE:
;       Result = XRF_RADRATE(element,line)
;
; INPUTS:
;       Element - The element descriptor (Aymbol or atomic number). 
;		It may be an array.
;
;       Line   - The descriptor of the line (iupac notation).
;		It may be an array (do not mix notations).
;		Nota:
;		The line must be defined in the DABAX file. Additionally, 
;		single lines not present in the default file 
;		(RadiativeRates_KrauseScofield.dat) are defined, but
;		these lines must be called individually (i.e., line
;		cannot be an array).:
;	        	'KL2'= 'KL23'-'KL3'
;	        	'KM2'= 'KM23'-'KM3'
;	        	'Ka'= 'Ka12'
;	        	'Ka2'= 'Ka12'-'Ka1'
;	        	'Kb'= 'Kb31'
;	        	'Kb3'= 'Kb31'-'Kb1'
;			(this equivalence is not done if N_Elements(line)>1)
;
;
; KEYWORDS:
;
;       FILE: The DABAX file name with the tabulation. If undefined (default)
;		it uses RadiativeRates_KrauseScofield.dat
;
; OUTPUT: 
;       A matrix out[i,j]    0<=i<=N_Elements(element)-1, 0<=j<=N_Elements(line)-1
;
; 
; EXAMPLES:
;	IDL> print,xrf_radrate('Fe','Ka1')
;	      0.58110000
;	IDL> print,xrf_radrate([40,50,60],'KL3')
;	      0.55130000
;	      0.53290000
;	      0.51630000
;	IDL> print,xrf_radrate([40,50,60],'KL2')
;	      0.28810000
;	      0.28480000
;	      0.28340000
;
; AUTHOR:
;	M. Sanchez del Rio, srio@esrf.fr, 2003/09/15 
;	2003-10-17 srio@esrf.fr removes dash in iupac notation
;
;______________________________________________________________________________
;-
;
on_error,2
; 
; Call this function recurrently for 
; some lines not defined in the DABAX data files
; 

IF N_Elements(line) EQ 1 THEN BEGIN
  CASE line OF
	'Ka': Return,xrf_radrate(element,'KL23')
	'Ka12': Return,xrf_radrate(element,'KL23')

	'Ka2': Return,xrf_radrate(element,'KL23')-xrf_radrate(element,'KL3')
	'KL2': Return,xrf_radrate(element,'KL23')-xrf_radrate(element,'KL3')

	'Kb'  : Return,xrf_radrate(element,'KM23')
	'Kb31': Return,xrf_radrate(element,'KM23')
	'Kb13': Return,xrf_radrate(element,'KM23')

	'Kb3': Return,xrf_radrate(element,'KM23')-xrf_radrate(element,'KM3')
	'KM2': Return,xrf_radrate(element,'KM23')-xrf_radrate(element,'KM3')

	'Lg4': Return,xrf_radrate(element,'L1O23')
	'Lb2,15': Return,xrf_radrate(element,'L3N45')
	else:
  ENDCASE
ENDIF
;
;
;
IF N_Elements(file) EQ 0 THEN file='RadiativeRates_KrauseScofield.dat'
IF type(element) EQ 7 THEN z = Atomic_Constants(element,Ret='Z') ELSE z=element
IF Strmatch(line[0],'[KL][Iabgl]*') EQ 1 THEN $
  line1 = xrf_iupac2siegbahn(line) ELSE $
  line1 = line

; 

h = Dabax_Access(file)
out = DblArr(N_elements(line1),N_Elements(element))

FOR i=0,N_Elements(element)-1 DO BEGIN
  ; 
  ; Fluorescence line energy 
  ;
  zi=z[i]
  uiupac = spec_labels(h,zi)
  data = Reform(spec_data(h,zi))

  FOR j=0,N_Elements(line1)-1 DO BEGIN
    linej=line1[j]

    igood = where(uiupac EQ linej)
  
    IF igood[0] EQ -1 THEN radrate=0.0D0 ELSE radrate=data[igood[0]]
    IF radrate LE 0 THEN Message,/Info,'Radiation rate fluorescence line undefined: '+$
	String(radrate[0])
  
    out[j,i]=radrate
  ENDFOR

ENDFOR

IF N_Elements(out) EQ 1 THEN out=Reform(out)

RETURN,out
END
