;+
; ============================== XPLOT2D =======================================
;
; XPLOT2D
;
; A software tool for analysing scientific images.
;
; Functionality:
;    - Load and visualize several images
;    - Several file types: EDF, MAR, BINARY, XRD, etc.
;    - Automatically handles gzip MAR compressed files
;    - Loads multiple files
;    - When an image is load, it can be added to the image stack or
;      substitute the first image in the stack.
;    - Visualize images in frames.
;    - Apply several color tables and adjust their limits to data.
;      button: "c f" color full range
;      button: "c s" color stretched range
;    - Define ROIS (rectangle, circle, cake, polygon)
;    - Mask data to ROI or complement of ROI (1-ROI)
;    - Applies mask from another image
;    - Calculations:
;        -azimuthal integration 
;        -histograms
;        -horizontal or vertical integration, etc.
;        -horizontal, vertical and transversal profiles
;    - Simulations: Overplot rings from external x-ray diffraction profiles,
;      defined in with either "twotheta" or "d-spacing" files,
;      containing columns with :
;         twotheta [deg], intensity [,color index]  or
;         d-spacing [A], intensity [,color indexa] , respectively.
;    - Makes a generic operation with one image.
;    - Define/edit image titles
;    -Full calibration using circles (detector perpendicular to beam) or
;     ellimses (detector oblique to the beam)
;    -Allows non interactive use
;    -One level undo
;
; To be done:
;   - Zoom
;   - Mask algebra
;   - Printing
;   - Documentation
;
; Authors: M. Sanchez del Rio, F. Matulic and V. Favre-Nicolin
;
; Version History:
;   Version 1.0 Beta 1  (March 15 2004)
;   Version 1.0 Beta 2  (June 11 2004)
;   Version 1.0 Beta 3  (April 20 2005, UAM-I, Mexico)
;   Version 1.00  (December 19 2007, srio@esrf.eu)
;   Version 1.03  April 2008. Added cartesian to polar
;   Version 1.06  May 2010 srio added possiblity to fix pars in calib
;   Version 1.07  May 2010 srio upgrades rikagu-axis and "+" button.
;                               adds external reader and integral to _azi.spec
;   Version 1.08  June 2010 overplotted diffractogram. Sorted preferences.
;
; Modifications history:
;   - V. Favre-Nicolin, 01/2002 (skeleton)
;   - M. Sanchez del Rio, 07/2002 (painted skeleton)
;   - F. Matulic 24.09.02 completed development of the main prototype
;   - 04-03-15 First beta version.
;   - 04-05-11 2nd beta version.
;   - 04-04-20 3rd beta version: Fast azimuth integral,...
;   - 2007-12-19 First released version 1.00
;   - 2008-02-06 srio@esrf.eu adds support for multichannel images 
;         (loaded into different images).
;   - 2008-02-25 srio@esrf.eu adds peak width in simul. Check for file
;         existence when overplotting rings. v 1.02
;   - 2008-04-30 srio@esrf.eu Added cartesian to polar
;   - 2008-06-06 srio@esrf.eu Dump azi integral to differnet file format. 
;         Added negative stretch values. Added DABAX and rruff database.
;   - 2008-06-26 srio@esrf.eu Added findpeaks facility. Set calculations->
;         corrections (dark and flat fields, warping and rebinning). 
;         Upgraded generic operation to access all images.  v 1.05
;   - 2010-05-05 srio@esrf.eu added possibility to calib with fixed pars
;   - 2010-05-25 srio@esrf.eu upgrades rikagu-axis and "+" button.
;
;===================== Non interactive use of xplot2d ==========================
;
; EXAMPLE OF NON-INTERACTIVE USE:
;
;     print,'>> opening xplot2d...'
;     xplot2d,parent=p
;     
;     print,'>> setting imageformat...'
;     xplot2d_imageformat,p,interactive=0,getStr=str
;     str.filter='*.stl'
;     tmp = str.format
;     tmp[0]='3'
;     str.format=tmp
;     xplot2d_imageformat,p,interactive=0,putStr=str
;     
;     print,'>> loading image...'
;     xplot2d_loadimage,p,file='al2o3_002.stl'
;     
;     print,'>> loading inputs...'
;     xplot2d_loadinputfile,p,file='Al2O3.xop'
;     
;     print,'>> azimuthal integration...'
;     xplot2d_azimuth_integral,p
;     
;     print,'>> others: '
;     xplot2d_annulus,p
;     xplot2d_calibrate_onering,p
;     xplot2d_calibrate_multiring,p
;     xplot2d_displayct,p
;     xplot2d_calc_corrections_sequence,p   ; this applies iteratively:
;             xplo2d_calc_substract,p
;             xplo2d_calc_divide,p
;             xplo2d_calc_warping,p
;             xplo2d_calc_rebin,p
;     
;     print,'>> saving image...'
;     xplot2d_saveimage,p,format='EDF',file='tmp.edf'
;     
;     print,'>> quitting application...
;     xplot2d_quit,p
;
;
;  COPYRIGHT:
; 	XPLOT2D belongs to XOP package and it is distributed within XOP.
; 	PLEASE REFER TO THE XOP COPYRIGHT NOTICE
; 
;  REFERENCE:
; 	Published calculations made with XOP should refer:
; 
; 	M. Sanchez del Rio and R. J. Dejus 
;         "Status of XOP: an x-ray optics software toolkit"
;         SPIE Proceedings Vol. 5536 (2004) pp.171-174
; 
;         http://dx.doi.org/10.1117/12.560903
; 
;  LAST MODIFICATION: srio@esrf.eu 2010-05-26
; 
; 
;-
;
;===============================================================================
;
FUNCTION XPLOT2D_VERSION
RETURN,'1.08'
END


;
;===============================================================================
;
PRO XPLOT2D_AXIS_PROPERTIES_EVENT, event
widget_control,event.top,get_uvalue=info2
widget_control,info2.wXplot2d,get_uvalue=info
widget_control,event.id,get_uvalue=widget
if (widget eq 'UPDATE') or (widget eq 'OK') then begin

  widget_control,info2.x_originID,get_value=x_origin
  info.str.axis_properties.x_origin=x_origin
  widget_control,info2.y_originID,get_value=y_origin
  info.str.axis_properties.y_origin=y_origin
  widget_control,info2.x_pixel_unit_ratioID,get_value=x_pixel_unit_ratio
  info.str.axis_properties.x_pixel_unit_ratio=x_pixel_unit_ratio
  widget_control,info2.y_pixel_unit_ratioID,get_value=y_pixel_unit_ratio
  info.str.axis_properties.y_pixel_unit_ratio=y_pixel_unit_ratio

  pData=info.data->Value([info.image_num],/pointer)
  img = *pData
  datasize=size(img)

  widget_control,info2.wXplot2d,set_uvalue=info

endif
if (widget eq 'OK') then begin
 widget_control, event.top, /destroy

endif

if (widget eq 'Cancel') then begin
 info.str.axis_properties=info2.previousParams ; restore previous values
 widget_control,info2.wXplot2d,set_uvalue=info
 widget_control, event.top, /destroy

endif
xplot2d_REFRESH,info2.wXplot2d
END

;
;===============================================================================
;
PRO XPLOT2D_AXIS_PROPERTIES,wXplot2d

widget_control,wXplot2d,get_uvalue=info

previousParams=info.str.axis_properties ; save previous circle parameters in case user cancels modifications

axis_info_request_dialog = Widget_Base(Column=1,Title='Axis Properties', group_leader=wXplot2d, $
    /Modal, /Floating, /Base_Align_Center)
axis_info_base = Widget_Base(axis_info_request_dialog, Column=1, Frame=1)

x_originID = CW_Field(axis_info_base, Title='x origin value:', Value=info.str.axis_properties.x_origin, /floating, /return_events, uvalue='UPDATE')
x_pixel_unit_ratioID = CW_Field(axis_info_base, Title='x pixel/unit ratio:', Value=info.str.axis_properties.x_pixel_unit_ratio, /floating, /return_events, uvalue='UPDATE')
y_originID = CW_Field(axis_info_base, Title='y origin value:', Value=info.str.axis_properties.y_origin, /floating, /return_events, uvalue='UPDATE')
y_pixel_unit_ratioID = CW_Field(axis_info_base, Title='y pixel/unit ratio:', Value=info.str.axis_properties.y_pixel_unit_ratio, /floating, /return_events, uvalue='UPDATE')

but_base = Widget_Base(axis_info_request_dialog,Row=1)
cancel = Widget_Button(but_base, Value='Cancel',uvalue='Cancel')
accept = Widget_Button(but_base, Value='OK',uvalue='OK')
centertlb, axis_info_request_dialog
    Widget_Control, axis_info_request_dialog, /Realize
    info2 = {wXplot2d:wXplot2d,x_originID:x_originID,y_originID:y_originID,$
             x_pixel_unit_ratioID:x_pixel_unit_ratioID,y_pixel_unit_ratioID:y_pixel_unit_ratioID,previousParams:previousParams}
    Widget_Control,axis_info_request_dialog,set_uvalue=info2
    xmanager, 'XPLOT2D_AXIS_PROPERTIES', axis_info_request_dialog

END

;
;===============================================================================
;
; CALCULATION OF LINE POINTS COORDINATES
;
FUNCTION XPLOT2D_GET_LINE_POINTS,x0,y0,x1,y1,linewidth
   ; given 2 extreme points (long integer coordinates,all positive), this function returns an array
   ; with the subscript of all points of the segment.
   ; linewidth contains the number of pixel in one line (nb of columns)
   ; if xdim > ydim then there is 1 point per column ; else one point per line
   ; coordinates of points may be retrieved by : x= subscript MOD linewidth
   ;						 y= long(subscript/linewidth)
   x0=long(x0) & y0=long(y0) & x1=long(x1) & y1=long(y1)
   xmin=min([x0,x1])
   ymin=min([y0,y1])
   xmax=max([x0,x1])
   ymax=max([y0,y1])
   xdim=xmax-xmin +1
   ydim=ymax-ymin +1
   
   case 1 of
   	xdim : begin	; vertical line
;   		print,'vertical line'
   		line=make_array(ydim,/long,value=x0+ymin*linewidth)	; initial subscript of points
   		line=lindgen(ydim)*linewidth+line
  	end	; endcase for xdim eq 1
   	ydim : begin	; horizontal line
;   		print,'horizontal line'
   		line=make_array(xdim,/long,value=x0+ymin*linewidth)	; initial subscript of points
   		line=line+lindgen(xdim)
   	end	; endcase for ydim eq 1
   	xdim ge ydim : begin
;   		print,'xdim ge ydim'
   	        a=float(y1-y0)/float(x1-x0)	; y=ax + b	, |a|<1
   		if x1 lt x0 then b=y1 else b=y0
   	        line=lindgen(xdim)
   	        line=round(line*a+b)
   	        line=line*linewidth+make_array(xdim,/long,value=xmin)+lindgen(xdim)
   	end	; endcase for xdim > ydim
   	else : begin	; ydim > xdim > 1
;   		print,'ydim gt xdim'
   	        a=float(x1-x0)/float(y1-y0)	; x=ay + b	|a|<1
   		if y1 lt y0 then b=x1 else b=x0
   	        line=lindgen(ydim)
   	        line=round(line*a+b)
   	        line=line+( ymin+lindgen(ydim) )*linewidth
   	end	; endcase for ydim > xdim > 1
   ENDCASE	
   return,line
END


;
;===============================================================================
;

FUNCTION xplot2d_calibrate_fom,myGuess,derivative,$
   theta=theta1,points=points1, weights=weights1  ; initialization values


COMMON myfom2vars, points, npoints, theta, weights

IF N_Elements(points1) NE 0 THEN BEGIN
  points=points1
  npoints=N_Elements(points[0,*])
ENDIF
IF N_Elements(theta1) NE 0 THEN BEGIN
  theta=theta1
ENDIF
IF N_Elements(weights1) NE 0 THEN BEGIN
  weights=weights1
ENDIF 

IF N_Elements(points) EQ 0 OR N_Elements(theta) EQ 0 $
  THEN  Message,'Function not initialized!' 

; exit (for initialization)
IF N_Elements(myGuess) EQ 0 THEN BEGIN
  RETURN,0
ENDIF

;
; calculate FoM
;
;
; needes: points, npoints, (theta): constants
;         VsG,V0G:  5 parameters to optimize
;         

; angles VsG-V0G and VsG-points (in the ellipse)

; first index: vector index 0,1,2
; second index: point index

v0G=myGuess[0:1]

CASE N_Elements(myguess) OF
  3: vsG=[v0G,myGuess[2]]
  5: vsG=myGuess[2:4]
  6: BEGIN
     vsG=myGuess[2:4]
     theta=myGuess[5]
     useTheta=1
     END
  else: Message,'Number of parameters not valid.'
ENDCASE

Vs0 = [vsG[0]-v0G[0], vsG[1]-v0G[1], vsG[2] ]# Replicate(1,npoints)
VsG_x = -Reform(points[0,*])+vsG[0] 
VsG_y = -Reform(points[1,*])+vsG[1] 
VsG_z = Replicate(vsG[2],npoints)
VsP = Make_Set(VsG_x,VsG_y,VsG_z)

cosTheta = total(vs0*vsP,1)/ $
           Sqrt(total(vs0*vs0,1)) / $
           Sqrt(total(vsP*vsP,1))

tmp = acos(cosTheta)

FoM = Total( weights*(tmp-theta)^2 ) / Total(weights)

RETURN,fom*1d20

END   ; xplot2d_calibrate_fom


;
;===============================================================================
;
FUNCTION XPLOT2D_DRAW_CAKE,Xcent,Ycent,rho1,rho2,phi1,phi2
; function which returns the points of a cake outline (in device coordinates)

; need to create an empty array here but since this is impossible, 
; array is initialized with a bogus value
  x_roi=[0] 
  y_roi=[0] 
; determines the fineness of the cake outline
; reducing the step amount increases the number of computed values for the roi
  step_grain=100.0 

; check whether this is a cake and not a circle
  if (rho1 ne 0.0) or ((phi1 ne 0.0) and (phi2 ne 360.0)) then begin 
    ;draw lower arc of the cake (whose radius is rho1)
    if phi1 gt phi2 then begin 
    ;cake arcs cross polar origin axis => split the point creation 
    ; process in 2 loops
      ;first loop goes from phi1 to 360�
      for phi=float(phi1),2*!pi, (2*!pi-phi1)/step_grain do begin
        x_roi=[x_roi,Xcent+rho1*cos(phi)]
        y_roi=[y_roi,Ycent+rho1*sin(phi)]
      endfor
      ;second loop goes from phi1 to 360�
      for phi=0.0,float(phi2), phi1/step_grain do begin
        x_roi=[x_roi,Xcent+rho1*cos(phi)]
        y_roi=[y_roi,Ycent+rho1*sin(phi)]
      endfor

    endif else begin ; normal loop range from phi1 to phi2
      for phi=float(phi1),float(phi2), (phi2-phi1)/step_grain do begin
        x_roi=[x_roi,Xcent+rho1*cos(phi)]
        y_roi=[y_roi,Ycent+rho1*sin(phi)]
      endfor
    endelse
  endif

  ;draw upper arc (whose radius is rho2) starting from phi2
  if phi1 gt phi2 then begin ;same dichotomy as above

    for phi=float(phi2),0.0, -(2*!pi-phi1)/step_grain do begin
      x_roi=[x_roi,Xcent+rho2*cos(phi)]
      y_roi=[y_roi,Ycent+rho2*sin(phi)]
    endfor
    for phi=2*!pi,float(phi1), -phi1/step_grain do begin
      x_roi=[x_roi,Xcent+rho2*cos(phi)]
      y_roi=[y_roi,Ycent+rho2*sin(phi)]
    endfor

  endif else begin

    for phi=float(phi2),float(phi1), -(phi2-phi1)/step_grain do begin
      x_roi=[x_roi,Xcent+rho2*cos(phi)]
      y_roi=[y_roi,Ycent+rho2*sin(phi)]
    endfor

  endelse

  x_roi=x_roi[1:*] ; get rid of the first bogus value
  y_roi=y_roi[1:*]

  points = make_array(2,n_elements(x_roi),/float);create the array for return
  points[0,*] = x_roi
  points[1,*] = y_roi

  return, points

END ; xplot2d_draw_cake

;
;===============================================================================
;
FUNCTION xplot2d_prof2image,pfile,Plot=fPlot, Dummy_Pixels=dummyu_pixels, $
	size=nsize, center=ncenter, Dist=dist, pixelSize=pixelSize1, scale=scale,  $
	e0 = e0, e1 = e1

;
; defaults
;
IF N_Elements(dist) EQ 0 THEN dist = 11.0 ; cm
IF N_Elements(pixelSize1) EQ 0 THEN pixelSize = 78.94e-4 ELSE $
	pixelSize=pixelSize1*1e-4
IF N_Elements(pfile) EQ 0 THEN pfile='calcite_champ.UXD'
IF N_Elements(dummy_pixels) EQ 0 THEN dummy_pixels = 2
IF N_Elements(fPlot) EQ 0 THEN fPlot = 1
IF N_Elements(scale) EQ 0 THEN scale = 1.
IF N_Elements(e0) EQ 0 THEN e0 = 8047.8
IF N_Elements(e1) EQ 0 THEN e1 = 10000.0

IF N_Elements(nsize) EQ 0 THEN BEGIN ; image dimensions
  nx = 2048L
  ny = 2048L
ENDIF ELSE BEGIN
  nx = Long(nsize[0])
  ny = Long(nsize[1])
ENDELSE

IF N_Elements(ncenter) EQ 0 THEN BEGIN ; image center
  cx = 0.
  cy = 0.
ENDIF ELSE BEGIN
  cx = Float(ncenter[0])
  cy = Float(ncenter[1])
ENDELSE


;
; read profile
;
a=rascii(pfile,npoints=np)

; redefine first two points for a flat interpolation
x=a[0,*]
a[1,0]=min(a[1,*])
a[1,1]=a[1,0]

; 
; change energy from Cu Ka to new one
;
a[0,*]=a[0,*]/2.0
a[0,*]=sin(a[0,*]*!pi/180.0)
a[0,*]=a[0,*]*e0/e1
a[0,*]=asin(a[0,*])
a[0,*]=2.0*180.0/!pi*a[0,*]


xplot,make_set(reform(x),reform(a[0,*]),reform(a[1,*])),/no, coltitles=$
	['pixel','twoTheta','intensity']


twoThetaDet = atan(nx*pixelsize/2/dist)
print,'twoThetaDet [deg] : ',twoThetaDet*180/!pi

im=fltarr(nx,ny)
imR=fltarr(nx,ny)
imA=fltarr(nx,ny)


FOR j=0,ny-1 DO BEGIN
  FOR i=0,nx-1 DO BEGIN
    d = sqrt((i-cx)^2+(j-cy)^2) > dummy_pixels ; to avoid infinites 
    imR[i,j]=d*pixelsize ; in cm
    imA[i,j]=atan(d*pixelsize/dist)*180./!pi ; in deg
  ENDFOR
ENDFOR
xlong = reform(imA,nx*ny)
ylong = interpol(a[1,*],a[0,*],xlong)
im = reform(ylong,nx,ny)/(2*!pi*imR)

IF fPlot EQ 1 THEN BEGIN
  window,/free,xsize=nx*scale,ysize=ny*scale
  im2 = congrid(im,nx*scale,ny*scale)
  tvscl,im2
  profiles,im2
ENDIF
RETURN,im
END ; xplot2d_prof2image
;
;===============================================================================
;
FUNCTION xplot2d_simul_dspacing,file, $
        size=nsize, center=v0, sample=vs, $
        energy=energy,outfile=outfile, $
        iexact=iexact, group=group, error=ierror, width=width
;
; defaults
;
ierror=0

IF N_Elements(energy) EQ 0 THEN energy = 21000.0 ; eV
IF N_Elements(file) EQ 0 THEN file = '?'
IF N_Elements(iexact) EQ 0 THEN iexact = 0

IF N_Elements(nsize) EQ 0 THEN BEGIN ; image dimensions
  nx = 3000L
  ny = 3000L
ENDIF ELSE BEGIN
  nx = Long(nsize[0])
  ny = Long(nsize[1])
ENDELSE

IF N_Elements(v0) EQ 0 THEN BEGIN ; image center
  v0=[2095.,1373.]
ENDIF ELSE BEGIN
  v0 = Float(v0)
ENDELSE

IF N_Elements(vS) EQ 0 THEN BEGIN ; image center
  vS=[2095.0,1373.,38.5/100e-4]
ENDIF ELSE BEGIN
  vS = Float(vS)
ENDELSE

wavelength = Float(Physical_Constants('hc')/energy)

;
; lines 
;
IF StrCompress(file,/Rem) EQ '?' OR checkFile(file) EQ 0 THEN BEGIN
      file = Dialog_Pickfile(Title='Get dspacing list',Dialog_Parent=group)
      IF file EQ '' THEN BEGIN
        ierror=1
        RETURN,0
      ENDIF
ENDIF

ref = rascii(file)

thetas = Reform(2E0*asin(wavelength/2/ref[0,*]))
;print,thetas*180/!pi


t0 = systime(1)
xx = (MakeArray1(nx,0.,nx-1)+0.5)#Replicate(1,ny)
yy = Replicate(1,nx)#(MakeArray1(ny,0.,ny-1)+0.5)

rho2 = (xx-vS[0])^2 + (yy-vS[1])^2 + vS[2]^2
tau2 = (xx-v0[0])^2 + (yy-v0[1])^2 

;
; apply cosine theorem to get the angle subtended from each pixel-vS
; with the beam
;
; norm v0S = v0-vS 
vv2 = (v0[0]-vS[0])^2 + (v0[1]-vS[1])^2 + vS[2]^2
cosTheta = (tau2 - rho2 - vv2)/(-2D0*Sqrt(rho2)*Sqrt(vv2))
Theta = acos(cosTheta)

IF N_Elements(width) EQ 0 THEN width=0.02
sigma = width*!dpi/180 
Intens = FltArr(nx,ny)
print,'<><> Time Before loop: ',systime(1)-t0

IF iexact THEN BEGIN
  ;
  ; Slow method
  ;
  FOR i=0,N_Elements(thetas)-1 DO BEGIN
    tmp = theta-thetas[i]
    Intens = Intens + exp( -(theta-thetas[i])^2/(2*sigma^2))
    ;igood = Where (tmp LE (2.35*sigma)*4)
    ;Intens[igood] = Intens[igood] + exp( -(tmp[igood])^2/(2*sigma^2))
  ENDFOR
  
ENDIF ELSE BEGIN
  ;
  ; quick method: use histogram on theta, and apply operations 
  ; only for thetas close to the matched ones
  ;
  thetaInterval = [min(thetas)-(2.35*sigma)*4,max(thetas)+(2.35*sigma)*4]
  htheta = histogram(theta,min=thetaInterval[0],max=thetaInterval[1],BINSIZE=2.35*sigma,$
     Location=hx,Reverse_Indices=ri)
  FOR i=0,N_Elements(thetas)-1 DO BEGIN
    ;tmp = theta-thetas[i]
    igood = Where ( abs(hx-thetas[i]) LE (2.35*sigma)*4)
    IF igood[0] NE -1 THEN BEGIN
      FOR j0=0,N_Elements(igood)-1 DO BEGIN
        j = igood[j0]
        IF RI[j] NE RI[j+1] THEN  BEGIN
            igood2 = RI[ RI[j] : RI[j+1]-1 ]
            Intens[igood2] = Intens[igood2] + exp( -(theta[igood2]-thetas[i])^2/(2*sigma^2))
        ENDIF
      ENDFOR
    ENDIF
  ENDFOR
  ENDELSE
  
t1=systime(1)

IF Keyword_Set(outfile) THEN write_edf,Intens,File=outfile


;
; compute coeff of the conic
;
print,'Using v0: '+Vect2string(v0)
print,'Using vs: '+Vect2string(vs)
xs = vs[0]
ys = vs[1]
zs = vs[2]
x0 = v0[0]
y0 = v0[1]
FOR i=0,N_Elements(thetas)-1 DO BEGIN
   theta = thetas[i]
   coeff = [$
       (-(x0*xs) + xs^2 - y0*ys + ys^2 + zs^2)^2 -  $
         ((x0 - xs)^2 + (y0 - ys)^2 + zs^2)* $
          (xs^2 + ys^2 + zs^2)*Cos(theta)^2, $
        -2*(y0 - ys)*((x0 - xs)*xs + (y0 - ys)*ys - zs^2) +  $
         2*ys*((x0 - xs)^2 + (y0 - ys)^2 + zs^2)* $
          Cos(theta)^2,-2*(x0 - xs)* $
          ((x0 - xs)*xs + (y0 - ys)*ys - zs^2) +  $
         2*xs*((x0 - xs)^2 + (y0 - ys)^2 + zs^2)* $
          Cos(theta)^2,2*(x0 - xs)*(y0 - ys), $
        (y0 - ys)^2 - ((x0 - xs)^2 + (y0 - ys)^2 + zs^2)* $
          Cos(theta)^2,(x0 - xs)^2 -  $
         ((x0 - xs)^2 + (y0 - ys)^2 + zs^2)*Cos(theta)^2 ]
  print,'Using theta: ',theta,' rad ',theta*180/!pi,' deg'
  print,'   Coeff: '+Vect2String(coeff)
ENDFOR

Intens=Intens*60000/max(Intens)
RETURN,Intens

END ; xplot2d_simul_dspacing
;
;===============================================================================
;
PRO XPLOT2D_SCROLL_PALETTE_EVENT,event

widget_control,event.top,get_uval=info

widget_control,event.id,get_uval=uvalid
CASE uvalid[0] OF
'Dismiss': BEGIN
  Widget_Control,/Destroy,event.top
  RETURN
  END
'Reset': BEGIN
   Widget_Control,event.id,Get_Value=val
   widget_control,info.wids.Xplot2D,get_uval=infoXplot2d
   ;infoXplot2D.str.display.scroll_step=val[0]
   scroll_vec=[0,0] 
   infoXplot2D.str.display.image_position=[0,0] 
   widget_control,info.wids.Xplot2D,set_uval=infoXplot2d
   Widget_Control,info.wids.x,Set_Value=(infoXplot2D.str.display.image_position)[0]
   Widget_Control,info.wids.y,Set_Value=(infoXplot2D.str.display.image_position)[1]
   END
'SCROLL': BEGIN
   widget_control,info.wids.Xplot2D,get_uval=infoXplot2d
   widget_control,event.id,get_val=val
   step=infoXplot2D.str.display.scroll_step
   CASE val OF
   '-': scroll_vec=[0,0]
   '<': scroll_vec=[-step,0] 
   '>': scroll_vec=[step,0] 
   '^': scroll_vec=[0,step] 
   'v': scroll_vec=[0,-step] 
   '/': BEGIN
        CASE uvalid[1] OF 
          'UP': scroll_vec=[step,step] 
          'DOWN': scroll_vec=[-step,-step] 
        ENDCASE
        END
   '\': BEGIN
        CASE uvalid[1] OF 
          'UP': scroll_vec=[-step,step] 
          'DOWN': scroll_vec=[step,-step] 
        ENDCASE
        END
   else:
   ENDCASE
   infoXplot2D.str.display.image_position= infoXplot2D.str.display.image_position+scroll_vec
   Widget_Control,info.wids.x,Set_Value=(infoXplot2D.str.display.image_position)[0]
   Widget_Control,info.wids.y,Set_Value=(infoXplot2D.str.display.image_position)[1]
   widget_control,info.wids.Xplot2D,set_uval=infoXplot2d
  END
'SCROLL_STEP': BEGIN
   Widget_Control,event.id,Get_Value=val
   widget_control,info.wids.Xplot2D,get_uval=infoXplot2d
   infoXplot2D.str.display.scroll_step=val[0]
   ;infoXplot2D.str.display.scroll_vec=[0,0] 
   widget_control,info.wids.Xplot2D,set_uval=infoXplot2d
   END
'SCROLL_ORIGIN_X': BEGIN
   Widget_Control,event.id,Get_Value=val
   widget_control,info.wids.Xplot2D,get_uval=infoXplot2d
   tmp = infoXplot2D.str.display.image_position
   tmp[0]=val[0]
   infoXplot2D.str.display.image_position=tmp
   widget_control,info.wids.Xplot2D,set_uval=infoXplot2d
   END
'SCROLL_ORIGIN_Y': BEGIN
   Widget_Control,event.id,Get_Value=val
   widget_control,info.wids.Xplot2D,get_uval=infoXplot2d
   tmp = infoXplot2D.str.display.image_position
   tmp[1]=val[0]
   infoXplot2D.str.display.image_position=tmp
   widget_control,info.wids.Xplot2D,set_uval=infoXplot2d
   END
else: Print,'Case item not found: ',uvalid
ENDCASE
xplot2d_refresh,info.wids.Xplot2d
END ; xplot2d_scroll_palette_event
;
;===============================================================================
;
PRO XPLOT2D_SCROLL_PALETTE,wXplot2d,wTLB,no_block=no_block


widget_control,wXplot2d,get_uvalue=infoXplot2d
dy = infoXplot2d.str.display
wTLB=Widget_base(TLB_FRAME_ATTR=1,Title='Scroll palette',/Column)

;scrollPalette items
wtmpB=Widget_Base(wTLB,/Row)
  wtmp=Widget_Button(wtmpB,Value='Dismiss',UVALUE='Dismiss')
  wtmp=Widget_Button(wtmpB,Value='Reset',UVALUE='Reset')

wtmp=Widget_Label(wTLB,value='Use these buttons to move the view window:')

wtmpB=Widget_Base(wTLB,/Row)
  wtmp=Widget_Button(wtmpB,Value='\',UValue=['SCROLL','UP'])
  wtmp=Widget_Button(wtmpB,Value='^',UValue='SCROLL')
  wtmp=Widget_Button(wtmpB,Value='/',UValue=['SCROLL','UP'])
wtmpB=Widget_Base(wTLB,/Row)
  wtmp=Widget_Button(wtmpB,Value='<',UValue='SCROLL')
  wtmp=Widget_Button(wtmpB,Value='-',UValue='SCROLL')
  wtmp=Widget_Button(wtmpB,Value='>',UValue='SCROLL')
wtmpB=Widget_Base(wTLB,/Row)
  wtmp=Widget_Button(wtmpB,Value='/',UValue=['SCROLL','DOWN'])
  wtmp=Widget_Button(wtmpB,Value='v',UValue='SCROLL')
  wtmp=Widget_Button(wtmpB,Value='\',UValue=['SCROLL','DOWN'])

  wtmp=Widget_Label(wTLB,value='Hit <return> to apply changes')
  wtmp=cw_field(wTLB,value=dy.scroll_step,title='Scroll step [pixels]: ', $
       /integer,/return_events,xsize=4,UValue='SCROLL_STEP')

wtmpB=Widget_Base(wTLB,/Row)
  wX=cw_field(wtmpB,value=dy.image_position[0],title='Image origin [pixels] X:', $
       /integer,/return_events,xsize=4,UValue='SCROLL_ORIGIN_X')
  wY=cw_field(wtmpB,value=dy.image_position[1],title='Y:', $
       /integer,/return_events,xsize=4,UValue='SCROLL_ORIGIN_Y')

centertlb, wTLB

infoXplot2d.wids.Scroll=wTMP
Widget_Control,wXplot2d,Set_uvalue=infoXplot2d
wids={ Xplot2d:wXplot2d, x:wX, y:wY }
info={ wids:wids}
widget_control, wTLB, set_uval=info, /REALIZE
xmanager, 'xplot2d_scroll_PALETTE', wTLB,no_block=no_block,Group=wXplot2d
END ; xplot2d_scroll_palette


;
;=========================================================================
;
function xplot2d_text,text, group=group
;
; returns a text array with particular content (for doc)
;

IF N_Elements(text) EQ 0 THEN text=''
file=GetEnv('XOP_HOME')+SDep(/ds)+'doc'+SDep(/ds)+'txt'+SDep(/ds)+'xplot2d_taggeddoc.txt'
IF checkFile(file) NE 1 THEN BEGIN
  itmp = Dialog_Message(Dialog_parent=group, 'File not found: '+file,/Error)
  RETURN,''
ENDIF
a = tags2struct(file=file)
tnames = tag_names(a)
igood = Where(tnames EQ StrUpCase(text))
IF igood[0] NE -1 THEN RETURN,a.(igood[0]) ELSE RETURN,''

END ; xplot2d_text
;
;===============================================================================
;===============================================================================
;===============================================================================
; NOW THE PROCEDURES AND FUNCTION WITH FIRST ARGUMENT THE PARENT ID
;===============================================================================
;===============================================================================
;===============================================================================
;



;
;===============================================================================
;

;
;+ 
;
; PRO xplot2d_annulus,wid, roi_text=roi_text, $   ; inputs
;     color=color, $                              ; inputs
;     img=img, $                                  ; input/output
;     center=center,axes=axes, $                  ; output
;     extractRoi=extractRoi, histogram=histo , $  ; flag to analyze filling points  
;     roi_indices=roi, weights=weights, x=x, y=y  ; outputawhen extractRoi=1
;
;-
PRO xplot2d_annulus,wid, roi_text=roi_text, $   ; inputs
    color=color, $                              ; inputs
    img=img, $                                  ; input/output
    center=center,axes=axes, $                  ; output
    extractRoi=extractRoi, histogram=histo , $  ; flag to analyze filling points  
    roi_indices=roi, weights=weights, x=x, y=y  ; outputawhen extractRoi=1


Widget_control,wId,get_uValue=info

widget_control,info.wids.Graph,get_value=graphwin
wset,graphwin
p_old = !p
x_old = !x
y_old = !y
!p=info.psysvar.p
!x=info.psysvar.x
!y=info.psysvar.y

IF N_Elements(img) EQ 0 THEN BEGIN
  pData=info.data->Value([info.image_num],/pointer)
  img = *pData
ENDIF

datasize=Long(size(img))

width = info.str.roi.annulus
IF Not(Keyword_Set(roi_text)) THEN roi_text = info.str.roi.last
IF N_Elements(color) EQ 0 THEN color=5
IF STrCompress(roi_text,/Rem) EQ '' THEN BEGIN
  roi=[-1]
  RETURN 
ENDIF

nn = StrParse(roi_text[0],',',list)
CASE StrCompress(list[0],/Rem) OF 
 'conic': BEGIN
    Xcent=Double(List[1])
    Ycent=Double(List[2])
    axis1=Double(List[3])
    IF nn GE 4 THEN axis2=Double(List[4]) ELSE axis2=axis1
    IF nn GE 5 THEN phi=Double(List[5])*!dpi/180 ELSE phi=0D0
    points = conic_pars2pts([Xcent,Ycent],[axis1,axis2],phi)
    points1 = conic_pars2pts([Xcent,Ycent],[axis1,axis2]-width/2,phi)
    points2 = conic_pars2pts([Xcent,Ycent],[axis1,axis2]+width/2,phi)
    center=[Xcent,Ycent]
    axes = [axis1,axis2]
    END
  else: BEGIN
     Print, 'Annulus not implemented for ROI type: '+list[0]
     roi=[-1]
     RETURN
    END
ENDCASE

Plots, points[0,*], points[1,*], color=color, linestyle=1, /data
Plots, points1[0,*], points1[1,*], color=color, /data
Plots, points2[0,*], points2[1,*], color=color, /data

IF Not(Keyword_Set(extractRoi)) THEN RETURN

;
; analysis of the points inside the annulus
;

; calculate points within the ring
roi1=polyfillv( points1[0,*],points1[1,*],datasize[1],datasize[2])
roi2=polyfillv( points2[0,*],points2[1,*],datasize[1],datasize[2])
roi = IntArr(max([roi1,roi2]))
roi[roi1]=1
roi[roi2]=roi[roi2]+1
roi=where(roi EQ 1)

;print,'roi before mask: '

; apply mask
IF roi[0] NE -1 THEN BEGIN
  itmp = IntArr(datasize[1]*datasize[2])-1
  itmp[roi]=1
  ibad = where(img LT 0)
  IF ibad[0] NE -1 THEN itmp[ibad]=-1
  roi=where(itmp GT 0)
ENDIF

;print,'roi after mask: '

IF roi[0] EQ -1 THEN BEGIN
  print,'No good points found.'
  x=[0]
  y=[0]
  weights=[0]
  RETURN
ENDIF

;print,'roi after no points: '

; define weight as intensity value
weights = img[roi]

; create coordinate arrays
x = (findgen(datasize[1])) # Replicate(1.0,datasize[2])
y = Replicate(1.0,datasize[1]) # (findgen(datasize[2]))
x=x[roi]
y=y[roi]

;
; apply histograms
;
IF Keyword_Set(histo) THEN BEGIN
  ; Calculate Angular histogram
  xcol  = atan2( y-center[1],x-center[0], /deg ) 
  binsize=360.0/info.str.pref.annulus_nsectors
  h = histogram(xcol,min=-180,max=180,binsize=binsize,reverse_indices=rev,locations=hx)
  iGoodBins = where(h GT 0, nn)
  xnew = FltArr(nn)
  ynew = FltArr(nn)
  wnew = FltArr(nn)
  FOR ii=0L,nn-1 DO BEGIN
    i=iGoodBins[ii]
    IF REV[i] NE REV[i+1] THEN BEGIN
      iBin = Rev[Rev[I] : Rev[i+1]-1]
      itmpT = Total(weights[iBin])
      IF itmpT GT 0 THEN BEGIN
        xnew[ii] = Total( x[iBin]*weights[iBin] )/iTmpT
        ynew[ii] = Total( y[iBin]*weights[iBin] )/iTmpT
        wnew[ii] = itmpT
      ENDIF
    ENDIF
  ENDFOR
  ; redefine arrays
  x=xnew
  y=ynew
  weights=wnew
ENDIF


!p=p_old
!x=x_old
!y=y_old
END ; xplot2d_annulus


;
;===============================================================================
;

;+ 
; PRO XPLOT2D_APPLY_IMAGE_TO_MASK,wXplot2D,maskarray=mask,refresh=refresh
;-
PRO XPLOT2D_APPLY_IMAGE_TO_MASK,wXplot2D,maskarray=mask,refresh=refresh

on_error,2

widget_control,wXplot2d,get_uvalue=info


pData=info.data->Value([info.image_num],/pointer)
img = *pData
datasize=size(img)

IF N_Elements(refresh) EQ 0 THEN refresh=1

IF Keyword_Set(mask) THEN BEGIN ; mask from imagearray

ENDIF ELSE BEGIN ; mask from image number
   num=info.image_mask
   xedit,num,Text='Image # (i.e., index+1) from where mask is read: ',$
      Title='Set mask', Action=action, Dialog_Parent=wXplot2D
   IF action EQ 'CANCEL' THEN Return
   info.image_mask=num
   pMask=info.data->Value([num-1],/pointer)
   mask = *pMask
ENDELSE
masksize=size(mask)

IF total(masksize) NE total(datasize) THEN BEGIN
  blah=Dialog_message( $
  ['Image and mask have incompatible dimensions'],/Error,$
  Dialog_Parent=wXplot2d)
  Return
ENDIF

mask_pixels = Where(mask LT 0)

if (mask_pixels)[0] gt -1 then begin
  ; perform calc only if region is not empty
  img_old=img
  img[mask_pixels]=mask[mask_pixels]
  *pData=img
  ;write back image data to info structure
  info.data->set,[info.image_num],value=img,aux=img_old
  widget_control,wXplot2d,set_uvalue=info
  IF refresh THEN XPLOT2D_REFRESH,wXplot2d
endif else begin
  blah=dialog_message(['No masked pixels found!'],/ERROR,$
  Dialog_Parent=wXplot2d)
endelse

END ; xplot2d_apply_image_to_mask

;
;===============================================================================
;
;+
;
; PRO XPLOT2D_AZIMUTH_INTEGRAL,wXplot2d,method=method, window=nwindow, $
;    diffractogram=diffractogram, $ ; output: [2theta,Intensity]
;    id_xplot=id_xplot              ; output: id of the xplot window
;
;-

PRO XPLOT2D_AZIMUTH_INTEGRAL,wXplot2d,method=method, window=nwindow, $
    diffractogram=diffractogram, $  ; output: [2theta,Intensity]
    out=tmp, $  ; output: all columns
    id_xplot=p              ; output: id of the xplot window

IF N_Elements(method) EQ 0 THEN method=0
IF N_Elements(nwindow) EQ 0 THEN nwindow=0
widget_control,wXplot2d,get_uvalue=info

;
; get image
;
pData=info.data->Value([info.image_num],/pointer)
img = *pData
datasize=size(img)

Widget_Control,/Hourglass

det = info.str.detector
dist0 = norm( [det.x0-det.xS,det.y0-det.yS,det.zS]) ; in pixel units
dist = dist0*det.pixelsize*1d-4 ; cm

;centre_of_integration = [det.x0,det.y0]
pixelsize=det.pixelsize
e1=det.energy

print,' '
print,'*** Azimuth integration ***'
print,'Direct beam center [pixels]: ',Vect2String([det.x0,det.y0])
print,'Sample position [pixels]: ',Vect2String([det.xS,det.yS,det.zS])
print,'Dist sample-detector [cm]: ',dist
print,'PixelSize [microns]: ',pixelsize
print,'Photon energy [eV]: ',e1

t0 = systime(1)
IF Fix( (det.useEllipses)[0])  EQ 1 THEN BEGIN
  print,'XPLOT2D_AZIMUTH_INTEGRAL: Using ellipses (axial integration)'
  azimuth_integral_axial,img, ttheta,intensity, v0=[det.x0,det.y0], $
    vS=[det.xS,det.yS,det.zS], incr=info.str.pref.step_azimuth_axial, $
    force=Fix((info.str.pref.force_azimuth_axial)[0])
  twotheta = ttheta*180/!pi ; deg
  xx = dist0*tan(ttheta)
  tmpTxt = ', Sample='+Vect2String([det.xS,det.yS,det.zS])
ENDIF ELSE BEGIN
  ; if circles are forced, the XS,YS are ignored
  print,'XPLOT2D_AZIMUTH_INTEGRAL: Using circles (standard integration)'
  dist = det.zS*det.pixelsize*1d-4 ; cm
  ;;;
  azimuth_integral,img, xx,intensity, center=[det.x0,det.y0], $
    exact=0 ,incr=info.str.pref.step_azimuth
  twotheta = 180./!pi*atan(xx*pixelsize*1e-4/dist)  ; deg
  tmpTxt=''
ENDELSE
;

;print,'***                     ***'

dspacing = physical_constants('hc')/2.0/e1/sin(twotheta/2.*!pi/180.)
qq = (!dpi*4)/(dspacing*2)
; IF CHANGING THE NUMBER OF COLUMNS, WATCH OUT IN "HERE_11"
tmp = make_set(twotheta,dspacing,qq,xx,xx*pixelsize,intensity)




coltitles = ['TwoTheta[deg]','dspacing[A]','Q=4pi sin(Theta)/Lambda [A^-1]', $
             'r[pixels]','r[cm]','intensity']

title1=info.data->Value(info.image_num,/title)

title2=title1
isep= strpos(title2,sdep(/ds),/Reverse_Search)
IF isep EQ -1 THEN BEGIN
    filepath='.'+sdep(/ds)
ENDIF ELSE BEGIN
    filepath=strmid(title2,0,isep+1)
    title2=strmid(title2,isep+1)

ENDELSE
title3='DirectBeam='+Vect2String([det.x0,det.y0])+tmpTxt+ $
   ', E='+StrCompress(e1,/Rem)+', D[cm]='+String(dist,Format='(F5.2)')+$
   ', Pixel: '+String(pixelsize,Format='(F5.1)')
title4='Beam='+Vect2String([det.x0,det.y0])+tmpTxt+ $
   ', D[cm]='+String(dist,Format='(F5.2)')+$
   ', E[eV]='+StrCompress(Long(e1))

;
; Prepare reference rings to overplot
;
IF info.ringsFlag EQ 1 THEN BEGIN
  ;
  ; retrieve and prepate reference lines
  ;

  ;linestyle=Fix((info.str.pref.ring_linestyle)[0])
  ;thick=Fix((info.str.pref.ring_thick)[0])

  nn= info.objRings->Info(/N_Elements)
  FOR i=0,nn-1 DO BEGIN
    tmpData = info.objRings->Value([i])
    tmpTitle = info.objRings->Value([i],/title)
    IF i EQ 0 THEN BEGIN
     allTitle=tmpTitle
     allPeak=Reform(tmpData[0,*])
     allInt=Reform(tmpData[1,*])
     allColor=Reform(tmpData[2,*])
     clr=tmpData[2,0]
    ENDIF ELSE BEGIN
     allTitle=[allTitle,tmpTitle]
     allPeak=[allPeak,Reform(tmpData[0,*])]
     allInt=[allInt,Reform(tmpData[1,*])]
     allColor=[allColor,Reform(tmpData[2,*])]
     clr=[clr,tmpData[2,0]]
    ENDELSE
  ENDFOR


  a = Make_Set(allPeak,allInt,allColor)

  ncol = N_Elements(a[*,0])

  ; get 2theta from dspacing
  lambda=physical_constants('hc')/e1
  a[0,*]=2.*asin(lambda/a[0,*]/2) ; in rads
  ; change 2theta to pixels
  peaks = reform(a[0,*]) ; twotheta in rads
  peaks2 = dist*tan(peaks)/(pixelsize*1e-4)
  ; normalize to the max of profile
  intens=Reform(a[1,*])
  intens=intens*(max(intensity)/max(intens))
ENDIF

; 
; open xplot window (if not ready), or prepare it (delete save sets)
;
IF nwindow EQ 0 THEN wtitle='Azimuthal integration ** Autorefreshed ** %C'ELSE $
                       wtitle='Azimuthal integration %C'

IF nwindow EQ 1 OR Widget_Info(info.wids.XplotAzimuth,/Valid_Id) EQ 0 THEN BEGIN
  xplot,Replicate(0,N_Elements(coltitles)),parent=p,/No_Block,$
      Window_Size=[600,400], coltitles=coltitles,wtitle=wtitle
  IF nwindow EQ 0 THEN BEGIN
    info.wids.XplotAzimuth=p
    widget_control,wXplot2d,set_uvalue=info
  ENDIF
ENDIF ELSE BEGIN
  p = info.wids.XplotAzimuth
  xplot_plotmgr,p,/Delete_All
ENDELSE

;
; load reference peaks
;
IF info.ringsFlag EQ 1 THEN BEGIN
  FOR i=0,N_Elements(clr)-1 DO BEGIN
    igood = where(allColor EQ clr[i])
    peaks = Reform(a[0,igood])
    intens1 = Reform(intens[igood])
    ; get 2theta from dspacing
    lambda=physical_constants('hc')/e1
    peaksInD=lambda/2/sin(peaks/2)

    ; HERE_11
    tmp0=make_set([0,Reform(a[0,igood]),0]*180/!pi, $
        [0,peaksInD,0], $
        [0,intens1,0],[0,intens1,0],[0,intens1,0],[0,intens1,0] ) ; ,$
    xplot_loadfile,p,tmp0,/NoRefresh, wtitle=wtitle
    xplot_controls_action,p,clr_lines=clr[i],PSYMBOL='11', $
        PSYMSIGN='1',/NoRefresh
    xplot_savecurrent,p
  ENDFOR
ENDIF

; load data
xplot_loadfile,p,tmp,/NoRefresh
xplot_settitles,p,alltitles=[title4+' '+title2,'-1','-1'],/NoRefresh
xplot_controls_action,p,clr_lines=3,PSYMBOL='0',PSYMSIGN='0'

; write integrated profile to a file
CASE Fix((info.str.imageformat.saveAzimuthal)[0]) OF
  0:
  1:  BEGIN
        itmp=spec_save(hh,tmp,'xplot2d_azi.spec',labels=coltitles, $
       Name=title2+' '+title4,/No_Confirm,comments=$
       ['azimuthal integration done with xop/xplot2d',title1,title3])

       IF itmp NE 0 THEN  BEGIN
         txt='XPLOT2D_AZIMUTH_INTEGRAL: Integral saved to: xplot2d_azi.spec (scan '+StrCompress(itmp,/Rem)+')'
         print,txt
         XPLOT2D_SET_COMMENT,wXplot2d,txt
       ENDIF
 
      END
  2:  BEGIN
      file=info.data->Value([info.image_num],/title)
      IF checkFile(file) NE 1 THEN BEGIN
         Print,'XPLOT2D_AZIMUTH_INTEGRAL: File not existing: '+file
         file = 'xplot2d'
      ENDIF
      isep= strpos(file,sdep(/ds),/Reverse_Search)
      IF isep EQ -1 THEN BEGIN
         filepath='.'+sdep(/ds)
      ENDIF ELSE BEGIN
        filepath=strmid(file,0,isep+1)
        file=strmid(file,isep+1)
      ENDELSE
      isep= strpos(file,'.',/Reverse_Search)
      IF isep EQ -1 THEN BEGIN
      ENDIF ELSE BEGIN
        file=strmid(file,0,isep)
      ENDELSE
      file = file+'_azi.spec'

      itmp=spec_save(hh,tmp,file,labels=coltitles, /OverWrite,  $
       Name=title2+' '+title4,/No_Confirm,comments=$
       ['azimuthal integration done with xop/xplot2d',title1,title3])
       IF itmp NE 0 THEN BEGIN
         txt='XPLOT2D_AZIMUTH_INTEGRAL: Integral saved to: '+file
         print,txt
         XPLOT2D_SET_COMMENT,wXplot2d,txt
       ENDIF
      END
  3:  BEGIN
       x = Reform(tmp[0,*]) 
       y = Reform(tmp[4,*])
       xx = FindGen(N_Elements(x))
       ccc = Poly_Fit(xx,x,2)
       itmp=spec_savemca(hh,y,'xplot2d_azi.mca',  $
       Name=title2+' '+title4,/No_Confirm,comments=$
       ['azimuthal integration done with xop/xplot2d',title1,title3], $
       Calib=ccc,Format='F12.5')
       IF itmp NE 0 THEN BEGIN
        txt='XPLOT2D_AZIMUTH_INTEGRAL: Integral saved to: xplot2d_azi.mca  (scan '+StrCompress(itmp,/Rem)+')'
        print,txt
        XPLOT2D_SET_COMMENT,wXplot2d,txt
      ENDIF
      END
  4:  BEGIN
       x = Reform(tmp[0,*]) 
       y = Reform(tmp[4,*])
       xx = FindGen(N_Elements(x))
       ccc = Poly_Fit(xx,x,2)


      file=info.data->Value([info.image_num],/title)
      IF checkFile(file) NE 1 THEN BEGIN
         Print,'XPLOT2D_AZIMUTH_INTEGRAL: File not existing: '+file
         file = 'xplot2d'
      ENDIF
      isep= strpos(file,sdep(/ds),/Reverse_Search)
      IF isep EQ -1 THEN BEGIN
         filepath='.'+sdep(/ds)
      ENDIF ELSE BEGIN
        filepath=strmid(file,0,isep+1)
        file=strmid(file,isep+1)
      ENDELSE
      isep= strpos(file,'.',/Reverse_Search)
      IF isep EQ -1 THEN BEGIN
      ENDIF ELSE BEGIN
        file=strmid(file,0,isep)
      ENDELSE
      file = file+'_azi.mca'
      itmp=spec_savemca(hh,y,file, $
        Name=title2+' '+title4,/Over, comments= $
       ['azimuthal integration done with xop/xplot2d',title1,title3], $
       Calib=ccc,Format='F12.5',/no_conf)
       IF itmp NE 0 THEN BEGIN
         txt='XPLOT2D_AZIMUTH_INTEGRAL: Integral saved to: '+file
         print,txt
         XPLOT2D_SET_COMMENT,wXplot2d,txt
       ENDIF
      END
ENDCASE

;XPLOT2D_SET_COMMENT,wXplot2d,title2
diffractogram=Make_Set(twoTheta,intensity)
END ; xplot2d_azimuth_integral



;
;===============================================================================
;

PRO XPLOT2D_CREATE_GRID_IMAGE,wXplot2d,interactive=interactive
;+
;
; PRO XPLOT2D_CREATE_GRID_IMAGE,wXplot2d
;
;-

Widget_Control,/HourGlass
widget_control,wXplot2d,get_uvalue=info

str = info.str.create_grid_image

IF Keyword_Set(interactive) THEN BEGIN
  XScrMenu,str,Action=act,Titles= info.str.create_grid_image_titles, $
     Flags=info.str.create_grid_image_flags, $
     Help=xplot2d_text('xplot2d_create_grid_image'), $
     /NoType,/Interp, WTitle='xplot2d_create_grid_image',dialog_Parent=wXplot2d
  IF act EQ 'DONT' THEN RETURN
ENDIF

itype = Fix((str.itype)[0])
s1=str.s1
s2=str.s2
thick = str.thick
period = str.period
amplitude = str.amplitude

CASE iType OF

      1: BEGIN
        ;
        ; mesh
        ;
        a=fltarr(s1,s2)
        x = (findgen(s1)) # Replicate(1.0,s2)
        y = Replicate(1.0,s1) # (findgen(s2))
  
        igood1 = where(x mod period LE thick/2 )
        igood2 = where(x mod period GE (period-thick/2) )
        IF ((igood1[0] NE -1) AND (igood2[0] NE -1)) THEN a[[igood1,igood2]]=1
        ; remove first (incomplete) lines
        ibad =  where(x LE period/2)
        IF ibad[0] NE -1 THEN a[ibad]=0
        ibad =  where(y LE period/2)
        IF ibad[0] NE -1 THEN a[ibad]=0

        igood1 = where(y mod period LE thick/2)
        igood2 = where(y mod period GE (period-thick/2))
        IF ((igood1[0] NE -1) AND (igood2[0] NE -1)) THEN a[[igood1,igood2]]=a[[igood1,igood2]]+2
        a[where(a EQ 3)] = amplitude
      END

      0: BEGIN
  
        ;
        ; gauss
        ;
        ;s1=2048L
        ;s2=s1
        ;thick = 50.0
        ;period = 200.0
        ;amplitude = 1000.0

        GpeaksX=Long(s1/period)
        GpeaksY=Long(s2/period)
        GcenterX=(Findgen(GpeaksX)+1)*period
        GcenterY=(Findgen(GpeaksY)+1)*period
        ;
        ; make Gaussians
        Gsigma=thick/2.35
        ; 2D gaussian
        xx = findgen(s1)
        yy = findgen(s2)
        
        tmp1 = xx*0
        FOR i=0,GpeaksX-1 DO BEGIN
         tmp1 = tmp1 + amplitude*exp(-((xx-GcenterX[i])/2.)^2/Gsigma^2) 
        ENDFOR
        tmp2 = yy*0
        FOR j=0,GpeaksY-1 DO BEGIN
         tmp2 = tmp2 + exp(-((yy-GcenterY[j])/2.)^2/Gsigma^2)
        ENDFOR
        a = tmp1##tmp2 
        END
        ELSE: RETURN
ENDCASE

info.str.create_grid_image=str
Widget_Control,wXplot2d,Set_UValue=info


IF Fix((str.negative)[0]) EQ 1 THEN a=max(a)-a

CASE Fix((str.res)[0]) OF
        0: xplot2d,a,Wtitle='Mesh/grid'
        1: XPLOT2D_ADD_IMAGE,wXplot2d,a,substitute=0
        2: XPLOT2D_ADD_IMAGE,wXplot2d,a,substitute=1
        else:
ENDCASE

END ; xplot2d_create_grid_image


;
;===============================================================================
;

PRO XPLOT2D_CREATE_TEST_IMAGE,wXplot2d 
;+
;
; PRO XPLOT2D_CREATE_TEST_IMAGE,wXplot2d
;
;-

IF N_Elements(wXplot2d) EQ 0 THEN wXplot2d=0L
;widget_control,wXplot2d,get_uvalue=info


str = {what:['0','All','Only one'],n:0L,res:['0','Add image(s) to this window',$
     'In a new Xplot2D window'],size:2048L}
XScrMenu,str,Action=act,Titles= ['How many test images','Which one (0=8)',$
     'Result to','Size in pixels'],$ 
     Flags=['1','w(0) EQ 1','1','1'], $
     /NoType,/Interp, WTitle='xplot2d_create_test_image',dialog_Parent=wXplot2d
IF act EQ 'DONT' THEN RETURN

Widget_Control,/HourGlass

IF Fix((str.what)[0]) EQ 0 then idx = indgen(9) ELSE  idx = str.n

FOR i=0,N_Elements(idx)-1 DO BEGIN
  ii = idx[i]
  tmp = image_illusions(ii,size=str.size,title=title)

  CASE Fix((str.res)[0]) OF
        0: XPLOT2D_ADD_IMAGE,wXplot2d,tmp,substitute=0,title=title
        1: BEGIN
             IF i EQ 0 THEN BEGIN
               xplot2d,Wtitle='Optical Illusions',PARENT=p
               XPLOT2D_ADD_IMAGE,p,tmp,substitute=1,title=title
             ENDIF ELSE BEGIN
               XPLOT2D_ADD_IMAGE,p,tmp,substitute=0,title=title
             ENDELSE
           END
        else:
  ENDCASE
ENDFOR

END ; xplot2d_create_test_image

;
;===============================================================================
;

;+
;
; PRO xplot2d_calibrate_onering,Xplot2DId,$
;    interactive=interactive, $
;    annulus=annulus,     $     ;  0=use ellipse line, 1=use annular region
;    GuessFlag=GuessFlag, $     ;  0=use ellipse line, 1=use current detector parameters
;    dspacing=dspacing          ;  d-spacing for the selected ring
;
;-
PRO xplot2d_calibrate_onering,Xplot2DId,$
   interactive=interactive, $
   annulus=annulus,     $     ;  0=use ellipse line, 1=use annular region
   GuessFlag=GuessFlag, $     ;  0=use ellipse line, 1=use current detector parameters
   dspacing=dspacing          ;  d-spacing for the selected ring



Widget_Control,Xplot2DId,Get_UValue=infoXplot2D,/Hourglass

IF Widget_Info(infoXplot2D.wids.detector,/Valid_Id) THEN BEGIN
  Widget_Control,infoXplot2D.wids.detector,Get_UValue=info
  dialog_parent=infoXplot2D.wids.detector
ENDIF ELSE BEGIN
  dialog_parent=Xplot2DId
ENDELSE

IF N_Elements(interactive) EQ 0 THEN interactive=1 
IF N_Elements(annulus) EQ 0 THEN annulus=1 

;              infoid = event.handler
;              Widget_control, infoid,  Get_uValue=info 
Widget_Control,Xplot2DId,Get_UValue=infoXplot2D,/Hourglass


xplot2d_annulus,Xplot2DId,roi_indices=igood,img=img,center=center,axes=axes, $
  /extractRoi

IF igood[0] EQ -1 THEN BEGIN
  itmp = Dialog_Message(/Info,['Please use the Xplot2D->ROI/Mask menu',$
     'to select an ellipse or a circle'], $
     Dialog_Parent=dialog_parent)
   RETURN
ENDIF


IF interactive THEN BEGIN
  itmp = Dialog_Message(/Question,['Use currently defined annular region? ', $
      '',$
      '(To change the circle/ellipse or the annulus width,',$
      'Cancel here, and change the circle/ellipse or',$
      'the annulus width by using', $
      'Xplot2D->ROI/Mask->Edit Parameters menu)',$
      '','','Yes=annular region','No=ellipse line'],$
      Dialog_Parent=dialog_parent,/Cancel)
  CASE itmp OF
     'Cancel': return
     'No': annulus=0
     'Yes': annulus=1
  ENDCASE
ENDIF

IF annulus THEN BEGIN
   ; annulus region
   s=size(img)
   x = (findgen(s[1])+0.5) # Replicate(1.0,s[2])
   y = Replicate(1.0,s[1]) # (findgen(s[2])+0.5)
   x=x[igood]
   y=y[igood]
   weights = img[igood]
   points=Make_Set(x,y)
   npoints=N_Elements(x)

   IF npoints GT 1000 THEN BEGIN
      IF interactive THEN BEGIN
        itmp = Dialog_Message(/Info, $
           ['Too many points in ROI for direct minimization: '+StrCompress(npoints),$
           ' ',' ','Start rebinning...'],/Cancel,Dialog_Parent=dialog_parent ) 
        IF itmp EQ 'Cancel' THEN RETURN
      ENDIF
       
      ; Calculate Angular histogram
      xcol  = atan2( y-center[1],x-center[0], /deg ) 
         h = histogram(xcol,min=-180,binsize=4,reverse_indices=rev,locations=hx)
         iGoodBins = where(h GT 0, nn)
         xnew = FltArr(nn)
         ynew = FltArr(nn)
         wnew = FltArr(nn)
      FOR ii=0L,nn-1 DO BEGIN
         i=iGoodBins[ii]
         IF REV[i] NE REV[i+1] THEN BEGIN
           iBin = Rev[Rev[I] : Rev[i+1]-1]
           itmpT = Total(weights[iBin])
           xnew[ii] = Total( x[iBin]*weights[iBin] )/iTmpT
           ynew[ii] = Total( y[iBin]*weights[iBin] )/iTmpT
           wnew[ii] = itmpT
;print,ii,i,hx[i],h[i],xnew[ii],ynew[ii],wnew[ii]
         ENDIF
      ENDFOR
      itmp = where(finite(xnew) EQ 1)
      xnew=xnew[itmp] & ynew=ynew[itmp] & wnew=wnew[itmp]
      itmp = where(finite(ynew) EQ 1)
      xnew=xnew[itmp] & ynew=ynew[itmp] & wnew=wnew[itmp]
      itmp = where(finite(wnew) EQ 1)
      xnew=xnew[itmp] & ynew=ynew[itmp] & wnew=wnew[itmp]
      x=xnew
      y=ynew
      weights=wnew
      points=Make_Set(x,y)
    ENDIF
ENDIF ELSE BEGIN
  ; line ROI
  xplot2d_roi_plot,Xplot2DId,/noEnvelope,points=points,center=center,axes=axes
  x=Reform(points[0,*])
  y=Reform(points[1,*])
  weights = Replicate(1.0,N_Elements(points[0,*]))
ENDELSE

;
; get detector parameters
;
det =  infoXplot2D.str.detector
IF Fix( (det.useEllipses)[0] ) EQ 1 THEN BEGIN
  dist = norm([det.x0-det.xS,det.y0-det.yS,det.zS])
ENDIF ELSE BEGIN
  dist = det.zS
ENDELSE
detector_dist = dist*det.pixelsize*1d-4 ; cm
lambda=physical_constants('hc')/det.energy



;
; calculate dspacing guess
;
IF Not(Keyword_Set(dspacing)) THEN BEGIN
  rr = sqrt( (points[1,*]-center[1])^2 + (points[0,*]-center[0])^2 )
  rr = mean(rr)
  ang1 = atan2(rr,dist)
  dspacingGuess = lambda/(sin(mean(ang1)/2)*2)
  
  print,'XPLOT2D_CALIBRATE_ONERING: dspacingGuess: ',dspacingGuess
  
  IF infoXplot2d.ringsFlag EQ 1 THEN BEGIN
    pA=infoXplot2d.objRings->Value([0],/pointer)
    a = *pA
    dspacing=Reform(a[0,*])
    tmp = min(abs(dspacing-dspacingGuess),igood)
    dspacing = dspacing[igood]
  ENDIF ELSE BEGIN
    dspacing=1.0000
  ENDELSE
ENDIF



IF interactive THEN BEGIN
  xedit,dspacing,Text='Enter the dspacing [A] for this reflection',$
    Action=action,Dialog_parent=dialog_parent
  IF action EQ 'CANCEL' THEN RETURN
ENDIF

print,'XPLOT2D_CALIBRATE_ONERING: Using dspacing [A]: ',dspacing

thetaB = asin(lambda/2/dspacing)
dist=0.5*(axes[0]+axes[1])/tan(2*thetaB)

IF Fix( (det.useEllipses)[0] ) EQ 1 THEN BEGIN
  myguess0=[center[0],center[1],center[0],center[1],dist]
  myguess1=[det.x0,det.y0,det.xs,det.ys,det.zs]
  tmpIndices=[2,3,4]
ENDIF ELSE BEGIN
  myguess0=[center[0],center[1],dist]
  myguess1=[det.x0,det.y0,det.zs]
  tmpIndices=[2]
ENDELSE

;
; start optimization
;
IF interactive THEN BEGIN
  itmp = Dialog_Message(/Question,Dialog_Parent=dialog_parent, $
     ['** Starting minimization ** ',$
     '','2Theta [deg] (Fixed)='+StrCompress(2*thetaB*180/!pi,/Rem), $
     'dSpacing [A] (Fixed)='+StrCompress(dspacing), $
     'Using ellipses? (1=Y, 0=N)='+StrCompress( (det.useEllipses)[0] ), $
     '','', $
     'Choice 1: Guess from ellipse parameters: ',$
     'DirectBeamAt='+Vect2String(myGuess0[0:1]),$
     'SampleAt='+Vect2String(myGuess0[tmpIndices]) ,$
     '','', $
     'Choice 2: Guess from current parameters: ',$
     'DirectBeamAt='+Vect2String(myguess1[0:1]),$
     'SampleAt='+Vect2String(myguess1[tmpIndices]) ,$
     '','', $
     'Do you want to use Choice 1? ','',$
     'Yes=Use Choice 1',$
     'No=Use Choice 2',$
     'Cancel=Forget optimization'], $
     /Cancel)

  CASE itmp OF
    'Cancel': RETURN
    'Yes': myGuess=myGuess0
    'No': myGuess=myGuess1
  ENDCASE
ENDIF ELSE BEGIN
  IF Keyword_Set(guessFlag) THEN myGuess=myGuess1 ELSE myGuess=myGuess0
ENDELSE
              
; Initialize xplot2d_calibrate_fom
tmp =  xplot2d_calibrate_fom(points=points,theta=2*thetaB,weights=weights)
              
Widget_Control,/Hourglass
;
; powell
;
;nn = n_elements(myGuess)
;direc = dblarr(nn,nn)
;for i=0,nn-1 do begin 
;  for j=0,nn-1 do begin
;    if i eq j then direc[i,j]=1
;  endfor
;endfor
;myGuessBefore=myGuess
              
;powell,myGuess,direc,1d-12,ff,'xplot2d_calibrate_fom',/Double,ItMax=5000,iter=iter
              

myGuess = TNMin('xplot2d_calibrate_fom',myGuess,AutoDerivative=1,MaxIter=5000, $
   niter=iter,nprint=10,status=status,ParInfo=parinfo)
print,'XPLOT2D_CALIBRATEONERING: TNMin Termination status code: ',status

              
print,'XPLOT2D_CALIBRATE_ONERING: iter: ',iter
print,'XPLOT2D_CALIBRATE_ONERING: xplot2d_calibrate_fom (before): ',xplot2d_calibrate_fom(myGuessBefore)
print,'XPLOT2D_CALIBRATE_ONERING: xplot2d_calibrate_fom (after): ',xplot2d_calibrate_fom(myGuess)
print,'XPLOT2D_CALIBRATE_ONERING: myGuess (before): ',myGuess0
print,'XPLOT2D_CALIBRATE_ONERING: myGuess (after): ',myGuess
              
; plot results
v0=myGuess[0:1]
IF Fix( (det.useEllipses)[0] ) EQ 1 THEN vS=myGuess[2:4] ELSE vS=[myguess[0:1],myguess[2]]
coeffG = conic_detector2coeffs(v0,vS,2*thetaB)
centerG = conic_coeffs2pars(coeffG,phi=phiG,axes=axesG)
pointsG = conic_pars2pts(centerG,axesG,phiG)

;plots,pointsG,color=5
              
IF interactive THEN BEGIN
  itmp = Dialog_Message(/Question,Dialog_Parent=dialog_parent, $
     ['** Results minimization: ',$
     'DirectBeamAt='+Vect2String(v0),$
     'SampleAt='+Vect2String(vS),$
     ' ','Do you want to keep these parameters? '])
  IF itmp EQ 'No' THEN RETURN
ENDIF



IF Widget_Info(infoXplot2D.wids.detector,/Valid_Id) THEN BEGIN ; detector window open
  Widget_Control,info.widsArr[2],Set_Value=v0[0]
  Widget_Control,info.widsArr[3],Set_Value=v0[1]
  Widget_Control,info.widsArr[4],Set_Value=vS[0]
  Widget_Control,info.widsArr[5],Set_Value=vS[1]
  Widget_Control,info.widsArr[6],Set_Value=vS[2]
  Xplot2d_SetDetector_Interactive_UpdateFromPanel,infoXplot2D.wids.detector
ENDIF ELSE BEGIN  ; detector window closed
  det.x0=v0[0]
  det.y0=v0[1]
  det.xS=vS[0]
  det.yS=vS[1]
  det.zS=vS[2]
  infoXplot2D.str.detector=det
 Widget_Control,Xplot2DId,Set_UValue=infoXplot2D
ENDELSE
  

END ; xplot2d_calibrate_onering

;
;===============================================================================
;

;+
;
; PRO xplot2d_calibrate_multiring,Xplot2DId,interactive=interactive,  $
;   action=action,v0=v0,vS=vS ; outputs
;
;-
PRO xplot2d_calibrate_multiring,Xplot2DId,interactive=interactive,  $
  action=action,v0=v0,vS=vS ; outputs

Widget_Control,Xplot2DId,Get_UValue=infoXplot2D,/Hourglass
IF Widget_Info(infoXplot2D.wids.detector,/Valid_Id) THEN BEGIN
  Widget_Control,infoXplot2D.wids.detector,Get_UValue=info
  dialog_parent=infoXplot2D.wids.detector
ENDIF ELSE BEGIN
  dialog_parent=Xplot2DId
ENDELSE

IF N_Elements(interactive) EQ 0 THEN interactive=1 
action='OK'


IF infoXplot2D.ringsFlag NE 1 THEN BEGIN
   itmp = Dialog_Message(/Error,['Multi ring optimization needs to overplot ',$
     'reference rings.','', $
     'Please use the Xplot2D->Simulations->Overplot rings...',$
     'to define reference rings'], $
     Dialog_Parent=dialog_parent)
     action='Cancel'
   RETURN
ENDIF

Widget_Control,/Hourglass
xplot2d_overplotrings,Xplot2DId,x0=x0,y0=y0,axis1=axis1,axis2=axis2,phi=phi,$
        twoTheta=twoTheta,dSpacing=dSpacing

iFlag=Fix(x0*0+1)

tmp = Make_Set(iFlag,dSpacing,twoTheta*180/!pi,axis1,axis2,x0,y0,phi)

IF interactive THEN BEGIN
  xtable,tmp,xtitle=['iFlag','dSpacing[A]','twoTheta[deg]','axis1','axis2','x0','y0','phi'],$
     wtitle='Rings for optimization',Action=action,/noMenu,$
     /Buttons,infoText=$
      ['These are the rings to be used for the optimization.',$
       'You may edit the first column (iFlag) to indicate that the corresponding',$
       'ring should (iFlag=1) or should not (iFlag=-1) be used in the optimization.',$
       'The other columns are info. Any editing will not be considered']
  
  IF action EQ 'Cancel' THEN RETURN
ENDIF
  
iFlag=Reform(tmp[0,*])
;center=[x0[0],y0[0]]
;phi = Reform(phi[0])

Widget_Control,/Hourglass
FOR i=0L,N_Elements(iFlag)-1 DO BEGIN
   print,'Analysing ring '+StrCompress(i+1)+' at d-spacing: '+StrCompress(dSpacing[i])+$
         ' twoTheta[rads]: '+StrCompress(twoTheta[i],/Rem)
   IF iFlag[i] GT 0 THEN BEGIN
       axes=[axis1[i],axis2[i]]
       xplot2d_annulus,Xplot2DId, img=img, color=((i+2) mod 32), $
          roi_text='conic,'+StrCompress(x0[i],/Rem)+','+StrCompress(y0[i],/Rem)+','+$
               StrCompress(axis1[i],/Rem)+','+StrCompress(axis2[i],/Rem)+','+$
               StrCompress(phi[i]*180/!pi,/Rem), /extractRoi, /Histo, $
               roi_indices=igood, x=x,y=y,weights=weights
       IF igood[0] NE -1 THEN BEGIN
         IF N_Elements(xtot) EQ 0 THEN BEGIN
            xtot=x
            ytot=y
            wtot=weights
            theta=Replicate(twoTheta[i],N_Elements(x))
         ENDIF ELSE BEGIN
            xtot=[xtot,x]
            ytot=[ytot,y]
            wtot=[wtot,weights]
            theta=[theta,Replicate(twoTheta[i],N_Elements(x))]
         ENDELSE
print,'    total weight for this ring,  total weight up to this ring: ',total(weights),total(wtot)
       ENDIF
   ENDIF
ENDFOR

IF N_Elements(x) EQ 0 THEN BEGIN
    itmp = Dialog_message(/Error,'Nothing to minimize',Dialog_Parent=dialog_parent)
    RETURN
ENDIF

;FOR i=0L,N_Elements(theta)-1 DO print,xtot[i],ytot[i],wtot[i],theta[i]
;
det =  infoXplot2D.str.detector
;
lambda=physical_constants('hc')/det.energy
IF Fix( (det.useEllipses)[0] ) EQ 1 THEN BEGIN
  myguess=[det.x0,det.y0,det.xs,det.ys,det.zs]
  tmpIndices=[2,3,4]
ENDIF ELSE BEGIN
  myguess=[det.x0,det.y0,det.zs]
  tmpIndices=[2]
ENDELSE
;
;
;
; start optimization
;
IF interactive THEN BEGIN
  ;itmp = Dialog_Message(/Info,Dialog_Parent=dialog_parent, $
  ;   ['** Starting minimization ** ',$
  ;    'dSpacing [A] (Fixed)='+vect2string(dspacing), $
  ;    'Using ellipses? (1=Y, 0=N)='+StrCompress( (det.useEllipses)[0] ), $
  ;    '','', $
  ;    'DirectBeamAt='+Vect2String(myguess[0:1]),$
  ;    'SampleAt='+Vect2String(myguess[tmpIndices]) ],/Cancel)
  ; 
  ;IF itmp EQ 'Cancel' THEN RETURN


  parinfo = replicate({value:0.D, fixed:0, limited:[0,0], $
                       limits:[0.D,0]}, N_Elements(myGuess))
  parinfo(*).value = myGuess
  
  tmp = Make_Set(parinfo.value,$
                 parinfo.fixed,  $
                 Reform((parinfo.limited)[0,*]), $
                 Reform((parinfo.limited)[1,*]), $
                 Reform((parinfo.limits)[0,*]), $
                 Reform((parinfo.limits)[1,*]) )

  xtable,tmp,xtitle=['value','fixedFlag',$
     'limitedLeftFlag','limitedRightFlag',$
     'limitLeft','limitRight' ],$
      ytitle=['BeamXcenter','BeamYcenter','SampleX','SampleY','SampleZ'],$
     wtitle='Parameters for optimization',Action=action,/noMenu,$
     /Buttons,infoText=$
      ['These are the parameters to be used for the optimization.','',$
      ;'dSpacing [A] (Fixed)='+vect2string(dspacing), $
      ;'Using ellipses? (1=Y, 0=N)='+StrCompress( (det.useEllipses)[0] ), $
      ;'','', $
      ;'You may edit these ones: ',$
      'Value: the initial guess ',$
      'fixedFlax: set to 1 to fix the corresponding parameter',$
      'limitedLeftFlag: set to 1 to bound the parameter on the left',$
      'limitedRightFlag: set to 1 to bound the parameter on the right',$
      'limitLeft: value for left boundary',$
      'limitLeft: value for right boundary']
  IF action EQ 'Cancel' THEN RETURN
  parinfo(*).value = Reform(tmp[0,*])
  parinfo(*).fixed = Reform(tmp[1,*])
  parinfo(*).limited = tmp[[2,3],*]
  parinfo(*).limits = tmp[[4,5],*]
ENDIF
              
; Initialize xplot2d_calibrate_fom
tmp =  xplot2d_calibrate_fom(points=Make_Set(xtot,ytot),theta=theta,weights=wtot)
;              
; calculate FoM
; needs: points, weights, theta: constants
;         VsG,V0G:  3 *circle) or 5 (ellipse)  parameters to optimize
;
 
myGuessBefore=myGuess
Widget_Control,/Hourglass
; 
; Powell method
;
;nn = n_elements(myGuess)
;direc = dblarr(nn,nn)
;for i=0,nn-1 do begin 
;for j=0,nn-1 do begin
;  if i eq j then direc[i,j]=1
;endfor
;endfor
;              
;powell,myGuess,direc,1d-12,ff,'xplot2d_calibrate_fom',/Double,ItMax=5000,iter=iter

;
; Truncated-Newton(TNMIN) method 
; 
myGuessBefore = parinfo.value
myGuess = TNMin('xplot2d_calibrate_fom',AutoDerivative=1,MaxIter=5000, $
   niter=iter,nprint=10,status=status,ParInfo=parinfo)
print,'XPLOT2D_CALIBRATE_MULTIRING: TNMin Termination status code: ',status

;              
;              
print,'XPLOT2D_CALIBRATE_MULTIRING: niter: ',iter
print,'XPLOT2D_CALIBRATE_MULTIRING: xplot2d_calibrate_fom (before): ',xplot2d_calibrate_fom(myGuessBefore)
print,'XPLOT2D_CALIBRATE_MULTIRING: xplot2d_calibrate_fom (after): ',xplot2d_calibrate_fom(myGuess)
print,'XPLOT2D_CALIBRATE_MULTIRING: myGuess (before): ',myGuessBefore
print,'XPLOT2D_CALIBRATE_MULTIRING: myGuess (after): ',myGuess

;              ; plot results
v0=myGuess[0:1]
IF Fix( (det.useEllipses)[0] ) EQ 1 THEN vS=myGuess[2:4] ELSE vS=[myguess[0:1],myguess[2]]

IF interactive THEN BEGIN
itmp = Dialog_Message(/Question,Dialog_Parent=dialog_parent, $
  ['** Results minimization: ',$
   'DirectBeamAt='+Vect2String(v0),$
   'SampleAt='+Vect2String(vS),$
   ' ','Do you want to keep these parameters? '])

  IF itmp EQ 'No' THEN BEGIN
    action='Cancel'
    RETURN
  ENDIF
ENDIF
;
; update values
;
IF Widget_Info(infoXplot2D.wids.detector,/Valid_Id) THEN BEGIN ; detector window open
  Widget_Control,info.widsArr[2],Set_Value=v0[0]
  Widget_Control,info.widsArr[3],Set_Value=v0[1]
  Widget_Control,info.widsArr[4],Set_Value=vS[0]
  Widget_Control,info.widsArr[5],Set_Value=vS[1]
  Widget_Control,info.widsArr[6],Set_Value=vS[2]
  ;action = 'update'

  Xplot2d_SetDetector_Interactive_UpdateFromPanel,infoXplot2D.wids.detector
ENDIF ELSE BEGIN  ; detector window closed
  det.x0=v0[0]
  det.y0=v0[1]
  det.xS=vS[0]
  det.yS=vS[1]
  det.zS=vS[2]
  infoXplot2D.str.detector=det
 Widget_Control,Xplot2DId,Set_UValue=infoXplot2D
ENDELSE
;

END  ;  xplot2d_calibrate_multiring
;
;===============================================================================
;

;+
;
; PRO xplot2d_calibrate_scan,Xplot2DId
;
;-
PRO xplot2d_calibrate_scan,Xplot2DId

Widget_Control,Xplot2DId,Get_UValue=infoXplot2D,/Hourglass
IF Widget_Info(infoXplot2D.wids.detector,/Valid_Id) THEN BEGIN
  Widget_Control,infoXplot2D.wids.detector,Get_UValue=info
  dialog_parent=infoXplot2D.wids.detector
ENDIF ELSE BEGIN
  dialog_parent=Xplot2DId
ENDELSE

simul_scan = infoXplot2d.simul_scan
print,'Current values ---------------------------------'
print,'Beam center: ',vect2String( $
             [infoXplot2d.str.detector.x0,infoXplot2d.str.detector.y0])
print,'Sample position: ',vect2String( $
             [infoXplot2d.str.detector.xS,infoXplot2d.str.detector.yS, $
              infoXplot2d.str.detector.zS])
print,'energy: ',infoXplot2d.str.detector.energy
print,'------------------------------------------------'

XScrMenu,simul_scan,Action=act, $
         Titles=['stop at each step','variable','min','max','number of points'],  $
         /NoType,/Interp, Flags=flags, WTitle='Scan parameter', $
         dialog_Parent=dialog_Parent,FieldLen=40

IF act EQ 'DONT' THEN RETURN
infoXplot2d.simul_scan = simul_scan
Widget_Control,Xplot2dId,Set_UValue=infoXplot2d
CASE Fix((simul_scan.var)[0]) OF
           0: var_old = infoXplot2d.str.detector.energy
           1: var_old = infoXplot2d.str.detector.x0
           2: var_old = infoXplot2d.str.detector.y0
           3: var_old = infoXplot2d.str.detector.xS
           4: var_old = infoXplot2d.str.detector.yS
           5: var_old = infoXplot2d.str.detector.zS
ENDCASE

istop = Fix( (simul_scan.stop)[0])

FOR i=0,simul_scan.npoints-1 DO BEGIN
  var = simul_scan.min+(simul_scan.max-simul_scan.min)/(simul_scan.npoints-1)*i
  print,'Using variable: ',var

  IF Widget_Info(infoXplot2D.wids.detector,/Valid_Id) THEN BEGIN
     indx = Fix((simul_scan.var)[0])
     ii = [0,2,3,4]
     indx2 = ii[indx]
     Widget_Control,info.widsArr[indx2],Set_Value=var
     Xplot2d_SetDetector_Interactive_UpdateFromPanel,infoXplot2D.wids.detector
  ENDIF ELSE BEGIN
     CASE Fix((simul_scan.var)[0]) OF
       0: infoXplot2d.str.detector.energy = var
       1: infoXplot2d.str.detector.x0 = var
       2: infoXplot2d.str.detector.y0 = var
       3: infoXplot2d.str.detector.xS = var
       4: infoXplot2d.str.detector.yS = var
       5: infoXplot2d.str.detector.zS = var
     ENDCASE
     Widget_Control,xplot2dId,Set_UValue=infoXplot2d
     xplot2d_refresh,xplot2dId
  ENDELSE

  IF istop THEN BEGIN
     itmp = Dialog_Message(/Question,'Using variable:'+String(var)+'    Continue?', $
            Dialog_Parent=dialog_parent)
     IF itmp EQ 'No' THEN GoTo,salimos
  ENDIF
ENDFOR

salimos:

itmp = Dialog_Message(/Question,'Keep last value of the scan? ('+String(var)+')',$
   Dialog_Parent=dialog_parent)
IF itmp EQ 'No' THEN BEGIN
IF Widget_Info(infoXplot2D.wids.detector,/Valid_Id) THEN BEGIN
   Widget_Control,info.widsArr[indx2],Set_Value=var_old
   Xplot2d_SetDetector_Interactive_UpdateFromPanel,infoXplot2D.wids.detector
ENDIF ELSE BEGIN
   CASE Fix((simul_scan.var)[0]) OF
     0: infoXplot2d.str.detector.energy = var_old
     1: infoXplot2d.str.detector.x0 = var_old
     2: infoXplot2d.str.detector.y0 = var_old
     3: infoXplot2d.str.detector.xS = var_old
     4: infoXplot2d.str.detector.yS = var_old
     5: infoXplot2d.str.detector.zS = var_old
   ENDCASE
   Widget_Control,xplot2dId,Set_UValue=infoXplot2d
   xplot2d_refresh,xplot2dId
ENDELSE
ENDIF

END  ; xplot2d_calibrate_scan

;
;===============================================================================
;
;+
;
; PRO XPLOT2D_DISPLAYCT,wid,MINRANGE=minrange,MAXRANGE=maxrange
;
;-
PRO XPLOT2D_DISPLAYCT,wid,MINRANGE=minrange,MAXRANGE=maxrange
    Widget_Control,wid,Get_UValue=info

    IF N_Elements(minrange) EQ 0 THEN BEGIN
      minrange = info.min
    ENDIF
    IF N_Elements(maxrange) EQ 0 THEN BEGIN
      maxrange = info.max
    ENDIF
    IF (minrange EQ maxrange) THEN BEGIN
       pData=info.data->Value([info.image_num],/pointer)
       img = *pData
       good_indices = where(img GE 0)
       minrange = min(img[good_indices],max=maxrange)
    ENDIF


    ;
    ; DISPLAY COLOR BAR
    ;
    device, window_state=window_state
    IF window_state(info.wids.colorbar) EQ 0 THEN BEGIN ; window not existing - Create it!
       window, /free, xsize=110, ysize=250
       info.wids.colorbar = !d.window
       Widget_Control,wId, Set_UValue=info
    ENDIF
    wset, info.wids.colorbar 
    erase

    if info.str.display.ScaleType EQ 'log' then begin
      ;using fanning's widget for the colourbar here
      fcolorbar, bottom=32, color=1, ncolors=ncolors, position=[0.6, 0.05, 0.8, 0.95], /vertical, MINRANGE=10^minrange, MAXRANGE=10^maxrange, format='(g0.2)'  , /ylog
print,'Log color table in: ',32,ncolors,minrange,maxrange
    endif else begin
      fcolorbar, bottom=32, color=1, ncolors=ncolors, position=[0.6, 0.05, 0.8, 0.95], /vertical, MINRANGE=minrange, MAXRANGE=maxrange, format='(g0.2)'
print,'Lin color table in: ',32,ncolors,minrange,maxrange
    endelse
END ; xplot2d_displayct


;
;===============================================================================
;

;+
;
; PRO xplot2d_imageformat, wId, interactive=interactive, putStr=str, $
;   getStr=tmp, action=action ; output
;
;-
;   MODIFICATION HISTORY:
;     2007-12-05 srio@esrf.eu Modified for xplot2d 1.0. Documented.
;     2008-04-17 srio@esrf.eu Added action kw.

PRO xplot2d_imageformat, wId, interactive=interactive, putStr=str, $
  getStr=tmp, action=action ; output
on_error,2
Widget_Control,wId,Get_UValue=info

action = 'OK'

IF N_Elements(interactive) EQ 0 THEN interactive=1

tmp = info.str.imageFormat

IF interactive THEN BEGIN
  XScrMenu,tmp,Action=act,Titles= info.str.imageFormat_titles, $
     /NoType,/Interp, WTitle='File format',dialog_Parent=wId, $
     Flags=info.str.imageFormat_flags
  IF act EQ 'DONT' THEN BEGIN
     action = 'Cancel'
     RETURN
  ENDIF
ENDIF 

IF Keyword_Set(str) THEN tmp=str 

info.str.imageFormat = tmp
Widget_Control,info.wids.LoadImage,Set_Value='Load Image '+$
    tmp.format[1+Fix(tmp.format[0])]+'...'
Widget_Control,wId, Set_UValue=info ; register the changes

END ; xplot2d_imageformat


;
;=========================================================================
;

;+
;
; PRO xplot2d_loadimage,wXplot2d, reload=reload, file=file, increment=increment
;   imageArray=img, imageArr2=img2 ; outputs
;
;
;-
;   MODIFICATION HISTORY:
;     2007-12-05 srio@esrf.eu Modified for xplot2d 1.0. Documented.
PRO xplot2d_loadimage,wXplot2d, reload=reload, file=file, increment=increment,$
   imageArray=img, imageArr2=img2 ; outputs

catch, error_status
 if error_status ne 0 then begin
 message,/info,'error caught: '+!err_string
 if sdep(/w) then itmp = Dialog_Message(/Error,$
   'XPLOT2D_LOADIMAGE: error caught: '+!err_string)
  catch, /cancel
  on_error,2
  RETURN
endif

widget_control,wXplot2d,get_uvalue=info

IF Keyword_Set(reload) and Not(Keyword_Set(file)) THEN BEGIN
  ;file=info.data->Value([info.image_num],/title)
  file = info.reloadFile
  IF checkFile(file) NE 1 THEN BEGIN
    tmp = Dialog_Message(/Error, 'File not existing: '+file,$
      Dialog_Parent=wXplot2D)
    RETURN
  ENDIF
  print,'Reloading file: '+file
ENDIF

IF info.Nload EQ 0 and Not(Keyword_Set(file)) THEN BEGIN
  xplot2d_imageformat,wXplot2D,action=action
  IF action EQ 'Cancel' THEN RETURN
  widget_control,wXplot2d,get_uvalue=info
ENDIF


IF Keyword_Set(increment) THEN BEGIN
  ;file_old=info.data->Value([info.image_num],/title)
  file_old = info.reloadFile
  isep= strpos(file_old,sdep(/ds),/Reverse_Search)

 
  IF isep EQ -1 THEN BEGIN
    filepath='.'+sdep(/ds)
  ENDIF ELSE BEGIN
    filepath=strmid(file_old,0,isep+1)
    file_old=strmid(file_old,isep+1)
  ENDELSE

  IF checkFile(filepath+file_old) NE 1 THEN BEGIN
    tmp = Dialog_Message(/Error, 'File not existing: '+filepath+file_old)
    RETURN
  ENDIF
  pos = stregex(file_old, info.str.imageFormat.fileIncrementExpression, len=len)
  index_old = STRMID(file_old, pos, len)

  itmp = -1
  for ii=0L,len-1 do begin
    bline = strmid(file_old,pos+ii,1)
    if (itmp eq -1) then begin
      if Strparse(' .+-0123456789 ',string(bline(0))) eq 0 then begin
        pos=pos+1
        len=len-1
      endif else begin
        itmp=1
      endelse
    endif
  endfor
  index_old = STRMID(file_old, pos, len)
  print,'XPLOT2D: index_old, pos, len: '+index_old, pos, len

TRY2:
  index_new = Long(index_old)
  index_new = StrCompress(index_new+increment,/Rem)
  nzeroes = len-StrLen(index_new)
  tmp = ''
  IF nzeroes GE 1 THEN FOR i=0L,nzeroes-1 DO tmp=tmp+'0'
  index_new=tmp+index_new
   ;print,'*****************'+index_new
  file = StrMid(file_old,0,pos)+index_new+StrMid(file_old,pos+len)
   ;print,'*****************'+file_old
   ;print,'*****************'+file
  file=filepath+file
  IF checkFile(file) NE 1 THEN BEGIN
    tmp = Dialog_Message(/Cancel,['Incremented file not existing: '+file,$
      '','Try next one?'],Dialog_Parent=wXplot2D)
    IF tmp EQ 'Cancel' THEN RETURN
    increment=increment+1
    GoTo,TRY2
  ENDIF
ENDIF



; restore the widget value
; Widget_Control,event.id,Get_Value=wValue
fileFormat=info.str.imageFormat.format[1+Fix(info.str.imageFormat.format[0])]
filter=info.str.imageFormat.filter
convertToFloat = Fix(info.str.imageFormat.float[0])
substitute = Fix(info.str.imageFormat.substitute[0])

; if the option "apply current mask" is selected, load the image with the mask

IF Fix(info.str.imageFormat.applyMask[0]) EQ 1 THEN BEGIN
         pData=info.data->Value([info.image_num],/pointer)
         mask = *pData
ENDIF

CASE fileFormat OF
  'EDF': BEGIN
         IF N_Elements(file) EQ 0 THEN $
         file = dialog_Pickfile(/Must_Exist,filter=filter,path=filedir,get_path=filedir1, $
          /Multiple_Files,Dialog_Parent=wXplot2d)
         Widget_Control,/HourGlass
         IF file[0] EQ '' THEN RETURN
         IF N_Elements(fileDir1) NE 0 THEN  fileDir=fileDir1
         FOR i=0,N_Elements(file)-1 DO BEGIN
           img = read_edf(file[i])

           info.reLoadFile=file[i]
           Widget_Control,wXplot2d,Set_UValue=info

           IF convertToFloat EQ 1 THEN img = float(img)
           XPLOT2D_ADD_IMAGE,wXplot2d,img,title=file[i],substitute=substitute, Refresh=0
           IF Fix((info.str.imageFormat.corrections_sequence)[0]) EQ 1 THEN $
             xplot2d_calc_corrections_sequence,wXplot2d,interactive=0
           Wait,info.str.imageFormat.wait
; PROVISIONAL (MUST BE REMOVED WHEN FIXING MEMORY LEAK!) 05-10-18 srio@xanum.uam.mx
;heap_gc
           IF Fix(info.str.imageFormat.cs[0]) EQ 1 THEN xplot2d_setcolorlimits,wXplot2D,/stretch,refresh=0
           IF Fix(info.str.imageFormat.applyMask[0]) EQ 1 THEN xplot2d_apply_image_to_mask,wXplot2D,maskarray=mask,refresh=0

           IF Fix((info.str.imageFormat.recalibrate)[0]) EQ 1 THEN xplot2d_calibrate_multiring,wXplot2d,interactive=0
           XPLOT2D_REFRESH,wXplot2D,imageArray=img, imageArr2=img2
         END
         END
  'XRD': BEGIN
         IF N_Elements(file) EQ 0 THEN $
         file = dialog_Pickfile(/Must_Exist,filter=filter,path=filedir,get_path=filedir1, $
          /Multiple_Files,Dialog_Parent=wXplot2d)
         Widget_Control,/HourGlass
         IF file[0] EQ '' THEN RETURN
         IF N_Elements(fileDir1) NE 0 THEN fileDir=fileDir1
         FOR i=0,N_Elements(file)-1 DO BEGIN
           restore,file=file[i],/verbose
           info.reLoadFile=file[i]
           Widget_Control,wXplot2d,Set_UValue=info

           IF N_Elements(xplot2d_image) EQ 0 THEN BEGIN
             itmp = Dialog_Message('xplot2d_image not restored. Aborted',$
               Dialog_Parent=wXplot2D)
             Return
           ENDIF
           IF convertToFloat EQ 1 THEN xplot2d_image = float(xplot2d_image)
           XPLOT2D_ADD_IMAGE,wXplot2D,xplot2d_image,title=file[i], Refresh=0
           Wait,info.str.imageFormat.wait
; PROVISIONAL (MUST BE REMOVED WHEN FIXING MEMORY LEAK!) 05-10-18 srio@xanum.uam.mx
;heap_gc
           IF Fix(info.str.imageFormat.cs[0]) EQ 1 THEN xplot2d_setcolorlimits,wXplot2d,/stretch,refresh=0
           IF Fix(info.str.imageFormat.applyMask[0]) EQ 1 THEN xplot2d_apply_image_to_mask,wXplot2d,maskarray=mask,refresh=0
           IF Fix((info.str.imageFormat.recalibrate)[0]) EQ 1 THEN xplot2d_calibrate_multiring,wXplot2d,interactive=0
           XPLOT2D_REFRESH,wXplot2d,imageArray=img, imageArr2=img2
         ENDFOR
         END
  'MAR': BEGIN
         IF N_Elements(file) EQ 0 THEN $
         file = dialog_Pickfile(/Must_Exist,filter=filter,path=filedir,get_path=filedir1, $
          /Multiple_Files,Dialog_Parent=wXplot2d)
         Widget_Control,/HourGlass
         IF file[0] EQ '' THEN RETURN
         IF N_Elements(fileDir1) NE 0 THEN fileDir=fileDir1
         FOR i=0,N_Elements(file)-1 DO BEGIN
           im = read_mar(file[i])
           info.reLoadFile=file[i]
           Widget_Control,wXplot2d,Set_UValue=info

           print,'Extrema of the image: '+file[i],min(im), max(im)
           IF convertToFloat EQ 1 THEN im = float(im)
           XPLOT2D_ADD_IMAGE,wXplot2d,im,title=file[i],substitute=substitute, refresh=0 ; modifies info
; PROVISIONAL (MUST BE REMOVED WHEN FIXING MEMORY LEAK!) 05-10-18 srio@xanum.uam.mx
;heap_gc
           Wait,info.str.imageFormat.wait
 ; PROVISIONAL (MUST BE REMOVED WHEN FIXING MEMORY LEAK!) 05-10-18 srio@xanum.uam.mx
;heap_gc
           IF Fix(info.str.imageFormat.cs[0]) EQ 1 THEN xplot2d_setcolorlimits,wXplot2d,/stretch,refresh=0
           IF Fix(info.str.imageFormat.applyMask[0]) EQ 1 THEN xplot2d_apply_image_to_mask,wXplot2d,maskarray=mask,refresh=0
           IF Fix((info.str.imageFormat.recalibrate)[0]) EQ 1 THEN xplot2d_calibrate_multiring,wXplot2d,interactive=0
           XPLOT2D_REFRESH,wXplot2d,imageArray=img, imageArr2=img2
         ENDFOR
         END

  'RIGAKU R-AXIS': BEGIN
         IF N_Elements(file) EQ 0 THEN $
         file = dialog_Pickfile(/Must_Exist,filter=filter,path=filedir,get_path=filedir1, $
          /Multiple_Files,Dialog_Parent=wXplot2d)
         Widget_Control,/HourGlass
         IF file[0] EQ '' THEN RETURN
         IF N_Elements(fileDir1) NE 0 THEN fileDir=fileDir1
         FOR i=0,N_Elements(file)-1 DO BEGIN
           im = read_rigaku(file[i])
           info.reLoadFile=file[i]
           Widget_Control,wXplot2d,Set_UValue=info

           print,'Extrema of the image: '+file[i],min(im), max(im)
           IF convertToFloat EQ 1 THEN im = float(im)
           XPLOT2D_ADD_IMAGE,wXplot2d,im,title=file[i],substitute=substitute, refresh=0 ; modifies info
; PROVISIONAL (MUST BE REMOVED WHEN FIXING MEMORY LEAK!) 05-10-18 srio@xanum.uam.mx
;heap_gc
           Wait,info.str.imageFormat.wait
 ; PROVISIONAL (MUST BE REMOVED WHEN FIXING MEMORY LEAK!) 05-10-18 srio@xanum.uam.mx
;heap_gc
           IF Fix(info.str.imageFormat.cs[0]) EQ 1 THEN xplot2d_setcolorlimits,wXplot2d,/stretch,refresh=0
           IF Fix(info.str.imageFormat.applyMask[0]) EQ 1 THEN xplot2d_apply_image_to_mask,wXplot2d,maskarray=mask,refresh=0
           IF Fix((info.str.imageFormat.recalibrate)[0]) EQ 1 THEN xplot2d_calibrate_multiring,wXplot2d,interactive=0
           XPLOT2D_REFRESH,wXplot2d,imageArray=img, imageArr2=img2
         ENDFOR
         END

  'PRINCETON': BEGIN
         IF N_Elements(file) EQ 0 THEN $
         file = dialog_Pickfile(/Must_Exist,filter=filter,path=filedir,get_path=filedir1, $
          /Multiple_Files,Dialog_Parent=wXplot2d)
         Widget_Control,/HourGlass
         IF file[0] EQ '' THEN RETURN
         IF N_Elements(fileDir1) NE 0 THEN fileDir=fileDir1
         FOR i=0,N_Elements(file)-1 DO BEGIN
           im = read_spe(file[i])
           info.reLoadFile=file[i]
           Widget_Control,wXplot2d,Set_UValue=info

           print,'Extrema of the image: '+file[i],min(im), max(im)
           IF convertToFloat EQ 1 THEN im = float(im)
           XPLOT2D_ADD_IMAGE,wXplot2d,im,title=file[i],substitute=substitute, refresh=0 ; modifies info
; PROVISIONAL (MUST BE REMOVED WHEN FIXING MEMORY LEAK!) 05-10-18 srio@xanum.uam.mx
;heap_gc
           Wait,info.str.imageFormat.wait
 ; PROVISIONAL (MUST BE REMOVED WHEN FIXING MEMORY LEAK!) 05-10-18 srio@xanum.uam.mx
;heap_gc
           IF Fix(info.str.imageFormat.cs[0]) EQ 1 THEN xplot2d_setcolorlimits,wXplot2d,/stretch,refresh=0
           IF Fix(info.str.imageFormat.applyMask[0]) EQ 1 THEN xplot2d_apply_image_to_mask,wXplot2d,maskarray=mask,refresh=0
           IF Fix((info.str.imageFormat.recalibrate)[0]) EQ 1 THEN xplot2d_calibrate_multiring,wXplot2d,interactive=0
           XPLOT2D_REFRESH,wXplot2d,imageArray=img, imageArr2=img2
         ENDFOR
         END

  'BINARY': BEGIN
         load_bin=info.str.load_bin
;         Titles=['File (? for launching browser)','Header [bytes]', 'X size [pixels]',$
;           'Y size [pixels]','Data type']
         IF N_Elements(file) NE 0 THEN load_bin.file=file
         XScrMenu,load_bin,Action=act,Titles=info.str.load_bin_titles, /NoType,/Interp, $
         WTitle='Load binary file',dialog_Parent=wXplot2d,FieldLen=40
         IF act EQ 'DONT' THEN RETURN
         IF StrCompress(load_bin.file,/Rem) EQ '?' OR CheckFile(load_bin.file) EQ 0 THEN BEGIN
           file = dialog_Pickfile(/Must_Exist,filter=filter,path=filedir,get_path=filedir1, $
             Dialog_Parent=wXplot2d)
           IF file EQ '' THEN RETURN
           IF N_Elements(fileDir1) NE 0 THEN fileDir=fileDir1
           load_bin.file=file
         ENDIF
         Widget_Control,/HourGlass
         ;
         ; get image
         ;
         CASE Fix((load_bin.type)[0])  OF
           0: ;Undefined
           1: b=BytArr(load_bin.xsize,load_bin.ysize) ; Byte
           2: b=IntArr(load_bin.xsize,load_bin.ysize) ;Integer
           3: b=LonArr(load_bin.xsize,load_bin.ysize) ;Longword integer
           4: b=FltArr(load_bin.xsize,load_bin.ysize) ;Floating point
           5: b=DblArr(load_bin.xsize,load_bin.ysize) ;Double-precision floating
           6: b=ComplexArr(load_bin.xsize,load_bin.ysize) ;Complex floating
           7: ;String
           8: ;Structure
           9: b=DComplexBytArr(load_bin.xsize,load_bin.ysize) ;Double-precision complex
           10:    ;Pointer
           11:    ;Object reference
           12:    b=UIntArr(load_bin.xsize,load_bin.ysize) ;Unsigned Integer
           13:    b=ULonArr(load_bin.xsize,load_bin.ysize) ;Unsigned Longword Integer
           14:    b=Lon64Arr(load_bin.xsize,load_bin.ysize) ;64-bit Integer
           15:    b=ULon64Arr(load_bin.xsize,load_bin.ysize) ;Unsigned 64-bit Integer
           else: Message,'Data type not implemented.'
         ENDCASE

         openr,unit,load_bin.file,/get_lun
           a=bytarr(load_bin.header)
           readu,unit,a
           readu,unit,b
         close,unit
         info.reLoadFile=load_bin.file
         Widget_Control,wXplot2d,Set_UValue=info

         ;
         ; store image
         ;
         IF convertToFloat EQ 1 THEN b = float(b)
         info.str.load_bin=load_bin
         XPLOT2D_ADD_IMAGE,wXplot2d,b,title=file,substitute=substitute, Refresh=0
           Wait,info.str.imageFormat.wait
; PROVISIONAL (MUST BE REMOVED WHEN FIXING MEMORY LEAK!) 05-10-18 srio@xanum.uam.mx
;heap_gc
           IF Fix(info.str.imageFormat.cs[0]) EQ 1 THEN xplot2d_setcolorlimits,wXplot2d,/stretch,refresh=0
           IF Fix(info.str.imageFormat.applyMask[0]) EQ 1 THEN xplot2d_apply_image_to_mask,wXplot2d,maskarray=mask,refresh=0
           IF Fix((info.str.imageFormat.recalibrate)[0]) EQ 1 THEN xplot2d_calibrate_multiring,wXplot2d,interactive=0
           XPLOT2D_REFRESH,wXplot2d,imageArray=img, imageArr2=img2
         END  ; BINARY
  'PNG/TIFF/etc': BEGIN
         itmp = dialog_read_Image(dialog_parent=wXplot2d, Image=img, $
           Query=q, File=file)
         IF itmp EQ 0 THEN RETURN
         info.reLoadFile=file
         Widget_Control,wXplot2d,Set_UValue=info
         IF type(img) EQ 1 THEN img=Fix(img)
         IF convertToFloat EQ 1 THEN img = float(img)
         ss = Size(img)
         IF ss[0] GT 2 THEN BEGIN
           itmp = Dialog_Message(Dialog_Parent=wXplot2d,/Cancel,/Question, $
               ['Xplot2D cal only deal with images with one channel (like "grey intensity")', $
               'The current image has '+StrCompress(ss[1],/Rem)+' Channels', '',$
               'Do you want to store the different channels in different images?','',$
               'Yes: Store different channels in different images',$
               'No: Store only the first channel',$
               'Calcel: do nothing'])
           CASE itmp OF
           'Cancel': RETURN
           'No': XPLOT2D_ADD_IMAGE,wXplot2d,Reform(img[0,*,*]),title=file+' (Channel 0)',substitute=substitute, Refresh=0 
           'Yes': BEGIN
             FOR ii=0,ss[1]-1 DO BEGIN
               XPLOT2D_ADD_IMAGE,wXplot2d,Reform(img[ii,*,*]),title=file+' (Channel '+StrCompress(ii,/Rem)+')',$
                 substitute=0,Refresh=0 
             ENDFOR
             END
           ENDCASE
         ENDIF ELSE BEGIN
           XPLOT2D_ADD_IMAGE,wXplot2d,img,title=file,substitute=substitute, Refresh=0 ; modifies info
         ENDELSE
         ; XPLOT2D_ADD_IMAGE,wXplot2d,img,substitute=substitute
         pData=info.data->Value([info.image_num],/pointer)
         print,min(*pData),max(*pData),total(*pData)
         Wait,info.str.imageFormat.wait
; PROVISIONAL (MUST BE REMOVED WHEN FIXING MEMORY LEAK!) 05-10-18 srio@xanum.uam.mx
;heap_gc
         IF Fix(info.str.imageFormat.cs[0]) EQ 1 THEN xplot2d_setcolorlimits,wXplot2d,/stretch,refresh=0
         IF Fix(info.str.imageFormat.applyMask[0]) EQ 1 THEN xplot2d_apply_image_to_mask,wXplot2d,maskarray=mask,refresh=0
         IF Fix((info.str.imageFormat.recalibrate)[0]) EQ 1 THEN xplot2d_calibrate_multiring,wXplot2d,interactive=0
         XPLOT2D_REFRESH,wXplot2d,imageArray=img, imageArr2=img2

         END ; PNG/TIFF/etc

  'External Reader': BEGIN
         IF N_Elements(file) EQ 0 THEN BEGIN
           allFiles = dialog_Pickfile(/Must_Exist,filter=filter,$
                      path=filedir,get_path=filedir1, $
                      /Multiple_Files,Dialog_Parent=wXplot2d)
         ENDIF ELSE BEGIN
           allFiles=file
         ENDELSE
         Widget_Control,/HourGlass
         IF allFiles[0] EQ '' THEN RETURN
         IF N_Elements(fileDir1) NE 0 THEN fileDir=fileDir1
         FOR i=0L,N_Elements(allFiles)-1 DO BEGIN
           file = allFiles[i]
           command=info.str.imageFormat.externalReader
           itmp = execute(command)
           IF itmp NE 1 THEN BEGIN
             itmp = Dialog_Message(/Error,'Error executing: '+command,$
                Dialog_Parent=wXplot2d)
             RETURN
           ENDIF 
           im = temporary(image)
           info.reLoadFile=allFiles[i]
           Widget_Control,wXplot2d,Set_UValue=info

           print,'Extrema of the image: '+allFiles[i],min(im), max(im)
           IF convertToFloat EQ 1 THEN im = float(im)
           XPLOT2D_ADD_IMAGE,wXplot2d,im,title=allFiles[i],substitute=substitute, refresh=0 ; modifies info
; PROVISIONAL (MUST BE REMOVED WHEN FIXING MEMORY LEAK!) 05-10-18 srio@xanum.uam.mx
;heap_gc
           Wait,info.str.imageFormat.wait
 ; PROVISIONAL (MUST BE REMOVED WHEN FIXING MEMORY LEAK!) 05-10-18 srio@xanum.uam.mx
;heap_gc
           IF Fix(info.str.imageFormat.cs[0]) EQ 1 THEN xplot2d_setcolorlimits,wXplot2d,/stretch,refresh=0
           IF Fix(info.str.imageFormat.applyMask[0]) EQ 1 THEN xplot2d_apply_image_to_mask,wXplot2d,maskarray=mask,refresh=0
           IF Fix((info.str.imageFormat.recalibrate)[0]) EQ 1 THEN xplot2d_calibrate_multiring,wXplot2d,interactive=0
           XPLOT2D_REFRESH,wXplot2d,imageArray=img, imageArr2=img2
         ENDFOR
         END


  else: Message,'Case option not found: '+action
ENDCASE

info.Nload = info.Nload+1
Widget_control,wXplot2d,Set_UValue=info

END ; XPLOT2D_LOADIMAGE


;
;===============================================================================
;

;+
;
; PRO xplot2d_loadinputfile,wid,file=file,path=path,group=group
;
;-
PRO xplot2d_loadinputfile,wid,file=file,path=path,group=group

strNew = tags2struct(file=file,path=path,group=group)
IF Type(strNew) EQ Type(0) THEN RETURN
Widget_Control,wid,Get_UValue=info
str = info.str
namesNew = Tag_Names(strNew)
names = Tag_Names(str)
FOR i=0,N_Tags(strNew)-1 DO BEGIN
	  tmp = strNew.(i)
	  CASE Type(tmp) OF
	  Type({a:0}): BEGIN
	     ii = Where(names EQ namesNew[i])
	     IF ii[0] NE -1 THEN BEGIN
		tmp1=str.(ii[0])
		Copy_Structure,strNew.(i),tmp1,/onlyFirstField
		str.(ii[0])=tmp1
	     ENDIF
	     END
          ELSE:
          ENDCASE
ENDFOR
info.str=str
Widget_Control,wid,Set_UValue=info
	

END ; xplot2d_loadInputFile

;
;===============================================================================
;

;+
;
; PRO xplot2d_overplotrings,wid,dataSize=ssize,  $
;   x0=x0,y0=y0,axis1=axis1,axis2=axis2,phi=phi,$ ; output arrays (dim=number of rings)
;   dspacing=dspacing,twoTheta=twoTheta           ; output arrays (dim=number of rings)
;
;-
PRO xplot2d_overplotrings,wid,dataSize=ssize,  $
   x0=x0,y0=y0,axis1=axis1,axis2=axis2,phi=phi,$ ; output arrays (dim=number of rings)
   dspacing=dspacing,twoTheta=twoTheta           ; output arrays (dim=number of rings)

;
; dataSize kw is not needed, but it passed it saves time...
;
;
; overplots circles corresponding to peaks in a XRD profile
; (peaks are read from a file in theta-2theta)
;


on_error,2
Widget_Control,wid,Get_UValue=info

widget_control,info.wids.Graph,get_value=graphwin
wset,graphwin
p_old = !p
x_old = !x
y_old = !y
!p=info.psysvar.p
!x=info.psysvar.x
!y=info.psysvar.y

; new part: overplot rings
IF Not(Keyword_Set(ssize)) THEN BEGIN
  ;
  ; get image
  ;
  pData=info.data->Value([info.image_num],/pointer)
  ;imageTitle=info.data->Value([info.image_num],/title)
  img = *pData
  ssize=size(img)
ENDIF


; new part: overplot rings

IF info.ringsFlag EQ 1 THEN BEGIN
  linestyle=Fix((info.str.pref.ring_linestyle)[0])
  thick=Fix((info.str.pref.ring_thick)[0])

  nn= info.objRings->Info(/N_Elements)
  FOR i=0,nn-1 DO BEGIN
    tmpData = info.objRings->Value([i])
    tmpTitle = info.objRings->Value([i],/title)
    IF i EQ 0 THEN BEGIN
     allTitle=tmpTitle
     allPeak=Reform(tmpData[0,*])
     allInt=Reform(tmpData[1,*])
     allColor=Reform(tmpData[2,*])
     clr=tmpData[2,0]
    ENDIF ELSE BEGIN
     allTitle=[allTitle,tmpTitle]
     allPeak=[allPeak,Reform(tmpData[0,*])]
     allInt=[allInt,Reform(tmpData[1,*])]
     allColor=[allColor,Reform(tmpData[2,*])]
     clr=[clr,tmpData[2,0]]
    ENDELSE
  ENDFOR

  ;pA=info.objRings->Value([0],/pointer)
  ;a = *pA
  a = Make_Set(allPeak,allInt,allColor)

  ncol = N_Elements(a[*,0])
  ;
  ; normalize intensities to 100
  ;
  IF ncol GE 2 THEN a[1,*]=a[1,*]/max(a[1,*])*100
  ;
  ; take color column
  ;
  IF ncol GE 3 THEN BEGIN
    color = Reform(a[2,*])
  ENDIF ELSE BEGIN
    color=Reform(a[0,*])*0+2
  ENDELSE

  det = info.str.detector

  IF Fix( (det.useEllipses)[0])  EQ 1 THEN BEGIN
    dist = norm([det.x0-det.xS,det.y0-det.yS,det.zS])
  ENDIF ELSE BEGIN
    dist = det.zS
  ENDELSE

  detector_dist = dist*det.pixelsize*1d-4 ; cm

  ;
  ; get 2theta from dspacing
  ;
  lambda=physical_constants('hc')/det.energy
  dSpacing = reform(a[0,*])   ; d-spacing in A
  twoTheta=2.*asin(lambda/dSpacing/2) ; 2 theta in rads
  NtwoTheta = N_Elements(twoTheta)
  ; compute ellipses
  x0=FltArr(NtwoTheta)
  y0=FltArr(NtwoTheta)
  axis1=FltArr(NtwoTheta)
  axis2=FltArr(NtwoTheta)
  phi=FltArr(NtwoTheta)
  FOR i=0,NtwoTheta-1 DO BEGIN
    IF Fix( (det.useEllipses)[0])  EQ 1 THEN BEGIN
      coeffs = conic_detector2coeffs([det.x0,det.y0],[det.xS,det.yS,det.zS],$
          twoTheta[i])
      center = conic_coeffs2pars(coeffs,axes=axes,phi=phi1)
    ENDIF ELSE BEGIN
      center = [det.x0,det.y0]
      axes = Replicate(dist*tan(twoTheta[i]),2)
      phi1=0
    ENDELSE
    x0[i]=center[0]
    y0[i]=center[1]
    axis1[i]=axes[0]
    axis2[i]=axes[1]
    phi[i]=phi1
    points = conic_pars2pts(center,axes,phi1)
    Plots, points[0,*], points[1,*], linestyle=linestyle, thick=thick, color=color[i], /data
  ENDFOR

ENDIF
;
; display legend
;
;clr = color[Uniq(color)]
;IF N_Elements(clr) GT 1 THEN $
    ;Legend,linestyle=Replicate(1,N_Elements(clr)),color=clr,'phase '+StrCompress(1+IndGen(N_Elements(clr)))
    Legend1,linestyle=Replicate(linestyle,N_Elements(clr)),color=clr,allTitle

!p=p_old
!x=x_old
!y=y_old
END ; xplot2d_overplotrings


;
;===============================================================================
;

;+
;
; FUNCTION XPLOT2D_GETIMAGE,wXplot2d,index=index, pointer=ipointer
;
;-
FUNCTION XPLOT2D_GETIMAGE,wXplot2d,index=index, pointer=ipointer
;Access to the data
   widget_control,wXplot2d,get_uvalue=uvalue
   if not(keyword_set(index)) then index=uvalue.image_num
   IF Keyword_Set(iPointer) THEN $
      return,uvalue.data->Value(index,/pointer) ELSE $
      return,uvalue.data->Value(index)
END


;
;===============================================================================
;

PRO XPLOT2D_CALC_SUBSTRACT,wXplot2d,interactive=interactive
;+
;
; PRO XPLOT2D_CALC_SUBSTRACT,wXplot2d, interactive=interactive
;
;
;-

Catch, error_status
if error_status ne 0 then begin
      message,/info,'error caught: '+!error_state.msg
      if sdep(/w) then itmp = dialog_Message(/Error,$
      'XPLOT2D_CALC_SUBSTRACT: error caught: '+!error_state.msg)
      catch, /cancel
      on_error,2
endif

Widget_Control,/HourGlass
widget_control,wXplot2d,get_uvalue=info


str = info.str.calc_substract


IF Keyword_Set(interactive) THEN BEGIN
  XScrMenu,str,Action=act,Titles= info.str.calc_substract_titles, $
     Flags=info.str.calc_substract_flags, $
     Help=xplot2d_text('xplot2d_calc_substract'), $
     /NoType,/Interp, WTitle='xplot2d_calc_substract',dialog_Parent=wXplot2d
  IF act EQ 'DONT' THEN RETURN
  iResult = Fix((str.res)[0])
  info.str.calc_substract = str
  Widget_Control,wXplot2d,Set_uvalue=info
ENDIF ELSE BEGIN
  iResult = 2 ; when non-interactive, always rewrite current image
ENDELSE


pData=XPLOT2D_GETIMAGE(wXplot2d,/Pointer)
img=*pData
datasize=size(img)

; 
; get image to substract
;
CASE Fix( (str.from)[0] ) OF
  0: BEGIN  ; loaded image
       num = str.number
       pNoise=info.data->Value([num-1],/pointer)
       noise = *pNoise
     END
  1: BEGIN  ; edf file image
       file = str.file
       
       IF StrCompress(file,/rem) EQ '?' THEN BEGIN
         file = Dialog_Pickfile(Dialog_Parent=wXplot2d, Filter='*.edf', $
                Title='File with EDF file to substract')
         IF file EQ '' THEN RETURN
       ENDIF
       IF checkFile(file) NE 1 THEN BEGIN
            blah=Dialog_message( $
              ['File with EDF image to substract does not exist: '+file],/Error,$
              Dialog_Parent=wXplot2d)
            RETURN
       ENDIF
       str.file=file
       info.str.calc_substract = str
       Widget_Control,wXplot2d,Set_uvalue=info
       noise=read_edf(file)

     END
   else: RETURN
ENDCASE

noisesize=size(noise)

IF total(noisesize[0:2]) NE total(datasize[0:2]) THEN BEGIN
  blah=Dialog_message( $
    ['Data and noise images have incompatible dimensions'],/Error,$
    Dialog_Parent=wXplot2d)
  Return
ENDIF

imgNew=img-noise

IF fix( (info.str.calc_substract.shift)[0] ) EQ 1 THEN BEGIN
 imgNew = imgNew-min(imgNew)
ENDIF

CASE iResult OF
  0: xplot2d,imgNew,Wtitle='substracted'
  1: XPLOT2D_ADD_IMAGE,wXplot2d,imgNew,substitute=0,AddTitle='(substracted)'
  2: XPLOT2D_ADD_IMAGE,wXplot2d,imgNew,substitute=1,AddTitle='(substracted)'
  else:
ENDCASE

END ; xplot2d_calc_substract


;
;===============================================================================
;

PRO XPLOT2D_CALC_DIVIDE,wXplot2d,interactive=interactive
;+
;
; PRO XPLOT2D_CALC_DIVIDE,wXplot2d, interactive=interactive
;
;-

Catch, error_status
if error_status ne 0 then begin
      message,/info,'error caught: '+!error_state.msg
      if sdep(/w) then itmp = dialog_Message(/Error,$
      'XPLOT2D_CALC_DIVIDE: error caught: '+!error_state.msg)
      catch, /cancel
      on_error,2
endif

Widget_Control,/HourGlass
widget_control,wXplot2d,get_uvalue=info

str = info.str.calc_divide

IF Keyword_Set(interactive) THEN BEGIN
  XScrMenu,str,Action=act,Titles= info.str.calc_divide_titles, $
     Flags=info.str.calc_divide_flags, $
     Help=xplot2d_text('xplot2d_calc_divide'), $
     /NoType,/Interp, WTitle='xplot2d_calc_divide',dialog_Parent=wXplot2d
  IF act EQ 'DONT' THEN RETURN
  iResult = Fix((str.res)[0])
  info.str.calc_divide = str
  Widget_Control,wXplot2d,Set_uvalue=info
ENDIF ELSE BEGIN
  iResult = 2 ; when non-interactive, always rewrite current image
ENDELSE

img=XPLOT2D_GETIMAGE(wXplot2d)
datasize=size(img)

; 
; get image to divide
;
CASE Fix( (str.from)[0] ) OF
  0: BEGIN  ; loaded image
       num = str.number
       flatfield=info.data->Value([num-1])
     END
  1: BEGIN  ; edf file image
       file = str.file
       
       IF StrCompress(file,/rem) EQ '?' THEN BEGIN
         file = Dialog_Pickfile(Dialog_Parent=wXplot2d, Filter='*.edf', $
                Title='File with EDF file to divide')
         IF file EQ '' THEN RETURN
       ENDIF
       IF checkFile(file) NE 1 THEN BEGIN
            blah=Dialog_message( $
              ['File with EDF image to divide does not exist: '+file],/Error,$
              Dialog_Parent=wXplot2d)
            RETURN
       ENDIF
       str.file=file
       info.str.calc_divide = str
       Widget_Control,wXplot2d,Set_uvalue=info
       flatfield=read_edf(file)
     END
   else: RETURN
ENDCASE

flatfieldSize=size(flatField)
IF total(flatFieldSize[0:2]) NE total(datasize[0:2]) THEN BEGIN
  blah=Dialog_message( $
    ['Data and flat-field images have incompatible dimensions'],/Error,$
    Dialog_Parent=wXplot2d)
  Return
ENDIF

imgNew=(img/flatField)*str.coefficient

CASE iResult OF
  0: xplot2d,imgNew,Wtitle='divided'
  1: XPLOT2D_ADD_IMAGE,wXplot2d,imgNew,substitute=0,AddTitle='(divided)'
  2: XPLOT2D_ADD_IMAGE,wXplot2d,imgNew,substitute=1,AddTitle='(divided)'
  else:
ENDCASE

END ; xplot2d_calc_divide

;
;===============================================================================
;

PRO XPLOT2D_CALC_WARPING,wXplot2d,interactive=interactive
;+
;
; PRO XPLOT2D_CALC_WARPING,wXplot2d,interactive=interactive
;
;-

Catch, error_status
if error_status ne 0 then begin
      message,/info,'error caught: '+!error_state.msg
      if sdep(/w) then itmp = dialog_Message(/Error,$
      'XPLOT2D_CALC_WARPING: error caught: '+!error_state.msg)
      catch, /cancel
      on_error,2
endif

Widget_Control,/HourGlass
widget_control,wXplot2d,get_uvalue=info

str = info.str.calc_warping

IF Keyword_Set(interactive) THEN BEGIN
  XScrMenu,str,Action=act,Titles= info.str.calc_warping_titles, $
     Flags=info.str.calc_warping_flags, $
     Help=xplot2d_text('xplot2d_calc_warping'), $
     /NoType,/Interp, WTitle='xplot2d_calc_warping',dialog_Parent=wXplot2d
  IF act EQ 'DONT' THEN RETURN
  info.str.calc_warping = str
  Widget_Control,wXplot2d,Set_uvalue=info
  iResult = Fix((str.res)[0])
ENDIF ELSE BEGIN
  iResult = 2 ; when non-interactive, always rewrite current image
ENDELSE


pData=XPLOT2D_GETIMAGE(wXplot2d,/Pointer)
img=*pData


;
; read control points
;
file = str.file
IF StrCompress(file,/rem) EQ '?' THEN BEGIN
  file = Dialog_Pickfile(Dialog_Parent=wXplot2d, $
         Title='File with control points')
  IF file EQ '' THEN RETURN
ENDIF
IF checkFile(file) NE 1 THEN BEGIN
     itmp = Dialog_Message(Dialog_Parent=wXplot2d, $
            'Error. File not existing: '+file)
     RETURN
ENDIF

str.file=file
a=rascii(file)
info.str.calc_warping = str
Widget_Control,wXplot2d,Set_uvalue=info
;
; gauss (distorted)
;
x0=Reform(a[0,*])
y0=Reform(a[1,*])
;
; fit2d (corrected)
;
x1=Reform(a[2,*])
y1=Reform(a[3,*])

Widget_Control,/HourGlass
CASE Fix((str.method)[0]) OF
  ;
  ; warping
  ;
  ; X1, Y1 Vectors containing the locations of the control (tie) points in 
  ; the output image. 
  ; 
  ; X0, Y0 Vectors containing the location of the control (tie) points in 
  ; the input image. Xi and Yi must be the same length as Xo and Yo. 
  ;
  0: BEGIN
     warped_image = WARP_TRI(x1, y1, x0, y0, img,  /EXTRAPOL, /TPS)
     END
  1: BEGIN
     IF Fix((str.interpolation)[0]) EQ 0 THEN BEGIN
       quintic=0
       extrapol=1
     ENDIF ELSE BEGIN
       quintic=1
       extrapol=1
     ENDELSE 
     warped_image = WARP_TRI(x1, y1, x0, y0, img, QUINTIC=quintic, $
        EXTRAPOL=extrapol)
     END
  2: BEGIN
     ;
     ; polynomial warping
     ;
     ; Use POLYWARP to generate the P and Q inputs to POLY_2D:
     ; X0, Y0 Vectors of X and Y coordinates to be fit as a function of X1 and Y1.
     ; ?????
     degree = str.degree
     POLYWARP, X0, Y0, X1, Y1, degree, P, Q 
     ; Perform an image warping based on P and Q: 
     warped_image = POLY_2D(img, P, Q) 
     END
ENDCASE

;info.str.calc_warping = str
;Widget_Control,wXplot2d,Set_UValue=info

CASE iResult OF
  0: xplot2d,warped_image,Wtitle='Warped image'
  1: XPLOT2D_ADD_IMAGE,wXplot2d,warped_image,substitute=0,AddTitle='(Warped)'
  2: XPLOT2D_ADD_IMAGE,wXplot2d,warped_image,substitute=1,AddTitle='(Warped)'
  else:
ENDCASE

END ; XPLOT2D_CALC_WARPING


;
;===============================================================================
;

PRO XPLOT2D_CALC_REBIN,wXplot2d,interactive=interactive
;+
;
; PRO XPLOT2D_CALC_REBIN,wXplot2d, interactive=interactive
;
;-

Catch, error_status
if error_status ne 0 then begin
      message,/info,'error caught: '+!error_state.msg
      if sdep(/w) then itmp = dialog_Message(/Error,$
      'XPLOT2D_CALC_REBIN: error caught: '+!error_state.msg)
      catch, /cancel
      on_error,2
endif

Widget_control,wXplot2d,get_uvalue=info,/HourGlass

str = info.str.calc_rebin


IF Keyword_Set(interactive) THEN BEGIN
  XScrMenu,str,Action=act,Titles= info.str.calc_rebin_titles, $
     Flags=info.str.calc_rebin_flags, FIELDLEN=40, $
     Help=xplot2d_text('xplot2d_calc_rebin'), $
     /NoType,/Interp, WTitle='xplot2d_calc_rebin',dialog_Parent=wXplot2d
  IF act EQ 'DONT' THEN RETURN
  iResult = Fix((str.res)[0])
  info.str.calc_rebin = str
  Widget_Control,wXplot2d,Set_uvalue=info
ENDIF ELSE BEGIN
  iResult = 2 ; when non-interactive, always rewrite current image
ENDELSE

img=info.data->Value([info.image_num])
ss = size(img)

CASE Fix( (str.what)[0]) OF
  0: BEGIN  ; image size
       newSizeX = str.newSizeX
       newSizeY = str.newSizeY
     END
  1: BEGIN  ; pixel size
       newSizeX = Round( Float(ss[1]) / (str.newPixelX/str.pixelX) )
       newSizeY = Round( Float(ss[2]) / (str.newPixelY/str.pixelX) )
     END
  else:
ENDCASE

txt = 'rebinned from '+vect2string(ss[1:2])+' to '+vect2string([newSizeX,newSizeY])
print,'XPLOT2D_CALC_REBIN '+txt

CASE Fix( (str.sampling)[0]) OF
  0: newImage = congrid(img,newSizeX,str.newSizeY)
  1: newImage = congrid(img,newSizeX,str.newSizeY,/INTERP)
  2: newImage = congrid(img,newSizeX,str.newSizeY,cubic=-0.5)
ENDCASE

info.str.calc_rebin = str
Widget_Control,wXplot2d,Set_UValue=info

CASE iResult OF
  0: xplot2d,newImage,Wtitle=txt
  1: XPLOT2D_ADD_IMAGE,wXplot2d,newImage,substitute=0,AddTitle='(rebinned)'
  2: XPLOT2D_ADD_IMAGE,wXplot2d,newImage,substitute=1,AddTitle='(rebinned)'
  else:
ENDCASE
;
END ; xplot2d_calc_rebin



;
;===============================================================================
;

PRO XPLOT2D_CALC_GENERIC_OPERATION,wXplot2d,interactive=interactive
;+
;
; PRO XPLOT2D_CALC_GENERIC_OPERATION,wXplot2d, interactive=interactive
;
;-

Catch, error_status
if error_status ne 0 then begin
      message,/info,'error caught: '+!error_state.msg
      if sdep(/w) then itmp = dialog_Message(/Error,$
      'XPLOT2D_CALC_GENERIC_OPERATION: error caught: '+!error_state.msg)
      catch, /cancel
      on_error,2
endif

Widget_control,wXplot2d,get_uvalue=info,/HourGlass

str = info.str.calc_generic_operation

IF Keyword_Set(interactive) THEN BEGIN
  XScrMenu,str,Action=act,Titles= info.str.calc_generic_operation_titles, $
     Flags=info.str.calc_generic_operation_flags, FIELDLEN=40, $
     Help=xplot2d_text('xplot2d_calc_generic_operation'), $
     /NoType,/Interp, WTitle='xplot2d_calc_generic_operation',dialog_Parent=wXplot2d
  IF act EQ 'DONT' THEN RETURN
  iResult = Fix((str.res)[0])
  info.str.calc_generic_operation = str
  Widget_Control,wXplot2d,Set_uvalue=info
ENDIF ELSE BEGIN
  iResult = 2 ; when non-interactive, always rewrite current image
ENDELSE

ngood=info.data->Info(/N_GOOD_ELEMENTS)
n=info.data->Info(/N_ELEMENTS)

img=info.data->Value([info.image_num])
FOR ii=0L,n-1 DO BEGIN
   cmd = 'img'+StrCompress(ii+1,/Rem)+'=info.data->Value(['+StrCompress(ii,/Rem)+'])'
   print,cmd
   itmp = Execute(cmd)

   IF itmp NE 1 THEN BEGIN
      jtmp = dialog_Message(/Error,['Operation not understood: '+cmd, $
        ' ','Continue?'],dialog_Parent=wXplot2d,/Cancel)
      IF jtmp EQ 'Cancel' THEN RETURN
   ENDIF
ENDFOR

;
itmp = Execute(str.oper)

IF itmp NE 1 THEN BEGIN
   jtmp = dialog_Message(/Error,['Failed operation: ',str.oper],$
     dialog_Parent=wXplot2D)
   return
ENDIF

info.str.calc_generic_operation = str
Widget_Control,wXplot2d,Set_UValue=info

CASE iResult OF
  0: xplot2d,imgNew,Wtitle='generic oper: '+str.oper
  1: XPLOT2D_ADD_IMAGE,wXplot2d,imgNew,substitute=0,AddTitle='(generic oper)'
  2: XPLOT2D_ADD_IMAGE,wXplot2d,imgNew,substitute=1,AddTitle='(generic oper)'
  else:
ENDCASE
;
END ; xplot2d_calc_generic_operation


;
;===============================================================================
;

PRO XPLOT2D_CALC_CORRECTIONS_SEQUENCE,wXplot2d,interactive=interactive
;+
;
; PRO XPLOT2D_CALC_CORRECTIONS_SEQUENCE,wXplot2d, interactive=interactive
;
;-

Catch, error_status
if error_status ne 0 then begin
      message,/info,'error caught: '+!error_state.msg
      if sdep(/w) then itmp = dialog_Message(/Error,$
      'XPLOT2D_CALC_CORRECTIONS_SEQUENCE: error caught: '+!error_state.msg)
      catch, /cancel
      on_error,2
endif

Widget_control,wXplot2d,get_uvalue=info,/HourGlass


str = info.str.calc_corrections_sequence


IF Keyword_Set(interactive) THEN BEGIN
  XScrMenu,str,Action=act,Titles= info.str.calc_corrections_sequence_titles, $
     Flags=info.str.calc_corrections_sequence_flags, FIELDLEN=40, $
     Help=xplot2d_text('xplot2d_calc_corrections_sequence'), $
     /NoType,/Interp, WTitle='xplot2d_calc_corrections_sequence',dialog_Parent=wXplot2d
  IF act EQ 'DONT' THEN RETURN
  info.str.calc_corrections_sequence = str
  Widget_Control,wXplot2d,Set_UValue=info
ENDIF 

IF Fix( (str.substract)[0]) EQ 1 THEN xplot2d_calc_substract,wXplot2d,interactive=interactive
IF Fix( (str.divide)[0]) EQ 1 THEN xplot2d_calc_divide,wXplot2d,interactive=interactive
IF Fix( (str.warping)[0]) EQ 1 THEN xplot2d_calc_warping,wXplot2d,interactive=interactive
IF Fix( (str.rebin)[0]) EQ 1 THEN xplot2d_calc_rebin,wXplot2d,interactive=interactive
;
END ; xplot2d_calc_corrections_sequence

;
;===============================================================================
;

;+
;
; FUNCTION xplot2d_calc_cm,wId,image=img, roi=roi
;
;-
FUNCTION xplot2d_calc_cm,wId,image=img, roi=roi
;
; computes center of mass of current image
;

IF N_Elements(img) EQ 0 THEN BEGIN
  Widget_Control,wId,get_UValue=info
  img = info.data->Value(info.image_num)
ENDIF
Widget_Control,/HourGlass
ssize = Long(size(img))

xx = (findgen(ssize[1]))#Replicate(1,ssize[2])
yy = Replicate(1,ssize[1])#(findgen(ssize[2]))

IF Keyword_Set(roi) THEN BEGIN
      IF roi[0] NE -1 THEN BEGIN
        img = img[roi]
        xx = xx[roi]
        yy = yy[roi]
      ENDIF
ENDIF
  
minImg=min(img)
IF minImg GE 0 THEN BEGIN
      print,'XPLOT2D_CALC_CM: Masked values: None'
ENDIF ELSE BEGIN
      igood=where(img GE 0)
      IF igood[0] NE -1 THEN BEGIN
        img = img[igood]
        xx = xx[igood]
        yy = yy[igood]
        print,'XPLOT2D_CALC_CM: Good (unmasked) values: '+ $
          StrCompress(N_Elements(igood),/Rem)+' over '+$
          StrCompress(ssize[1]*ssize[2])
      ENDIF
ENDELSE

;for i=0,N_Elements(xx)-2 do oplot,[xx[i],xx[i+1]],[yy[i],yy[i+1]],psy=3,color=7

t0 = systime(1)
iTot = total(img)
xCM = Total(img*xx)/iTot
yCM = Total(img*yy)/iTot
rr = Sqrt( (xx-xCm)^2 + (yy-yCm)^2 )
rCM = Total(img*rr)/iTot

RETURN,[xCM,yCM,rCM]
END ; xplot2d_calc_cm


;
;===============================================================================
;

;+
;
; PRO xplot2d_crop,wXplot2d, $
;     outputTo=outputTo, $   ; 0=New window, 1=Substiture current image, 2=Add image
;     roi=roi                ; roi=[x1,y1,x2,y2] selected rectangle
;
;-
PRO xplot2d_crop,wXplot2d , $
    outputTo=outputTo, $   ; 0=New window, 1=Substiture current image, 2=Add image
    roi=roi

COMMON xplot2d_crop_str,str1
;
widget_control,wXplot2d,get_uvalue=info

IF N_Elements(roi) NE 0 THEN BEGIN
  corner1=[roi[0],roi[1]]
  corner2=[roi[2],roi[3]]
ENDIF ELSE BEGIN
  roi_text = info.str.roi.last
  nn = StrParse(roi_text,',',list)
  roiType=StrCompress(list[0],/Rem)
  
  IF roiType NE 'polygon' THEN BEGIN
    tmp=Dialog_Message(/Error,'Please setect a Rectangular (or polygonal) ROI',$
      Dialog_Parent=wXplot2d)
    RETURN
  ENDIF
  
  cmd = StrSubstitute(roi_text,'polygon,','points=[')
  cmd = cmd+']'
  itmp=Execute(cmd)
  IF itmp NE 1 THEN BEGIN
        tmp=Dialog_Message(/Error,['Error executing: ',cmd])
        RETURN
  ENDIF
  points=transpose(points)
  
  corner1=[min(points[0,*]),min(points[1,*])]
  corner2=[max(points[0,*]),max(points[1,*])]
ENDELSE

;axis_properties =
;{x_origin:0.0,x_pixel_unit_ratio:1.0,y_origin:0.0,y_pixel_unit_ratio:1.0}
;
; apply axis
;
ax = info.str.axis_properties
corner11=corner1*[ax.x_pixel_unit_ratio,ax.y_pixel_unit_ratio]-[ax.x_origin,ax.y_origin]
corner22=corner2*[ax.x_pixel_unit_ratio,ax.y_pixel_unit_ratio]-[ax.x_origin,ax.y_origin]

axis = [corner1[0], (corner22[0]-corner11[0])/(corner2[0]-corner1[0]), $
        corner1[1], (corner22[1]-corner11[1])/(corner2[1]-corner1[1])    ]


img = xplot2d_getimage(wXplot2d)
ss = size(img)
print,'XPLOT2D_CROP: size of the original image: '+vect2string(ss[1:2])
print,'XPLOT2D_CROP: corner1='+vect2string(corner1)+' corner2='+vect2string(corner2)

x1=(corner11[0]>0<ss[1])
y1=(corner11[1]>0<ss[2])
x2=(corner22[0]>0<ss[1])
y2=(corner22[1]>0<ss[2])
print,'XPLOT2D_CROP: cropped area from '+Vect2String([x1,y1])+' to '+$
  Vect2String([x2,y2])

IF (x1 GE x2) OR (y1 GE y2) THEN BEGIN
  itmp = Dialog_Message(/Error,['Impossible to crop',$
      'size of the original image: '+vect2string(ss[1:2]),$
      'corner1='+vect2string(corner1)+' corner2='+vect2string(corner2),$
      ''],Dialog_Parent=wXplot2d)
  RETURN
ENDIF
img = img[x1:x2,y1:y2]

IF N_Elements(outputTo) EQ 0 THEN BEGIN
  IF N_Elements(str1) EQ 0 THEN $
  str1 = {res:['0','new XPlot2d window','this XPlot2d window (overwrite current image *No Undo*)',$
     'this XPlot2d window (add image)','Another application'],$
      oper:'xsurface1,img'}
      titles=['Selected pixels'+vect2String(Long([x1,y1,x2,y2]))+' Result to:','Command']
  XScrMenu,str1,Action=act, /NoType,/Interp, flags=['1','w(0) EQ 3'],$
     titles=titles, WTitle='Crop result',dialog_Parent=wXplot2d,FieldLen=40
  IF act EQ 'DONT' THEN RETURN
  outputTo = Fix(str1.res[0])
ENDIF

CASE outputTo OF
  0: xplot2d,img,scale=1,axis=axis,cursor=2, $
         minColor=info.min, maxColor=info.max
  1: XPLOT2D_ADD_IMAGE,wXplot2d,img,substitute=1
  2: XPLOT2D_ADD_IMAGE,wXplot2d,img,substitute=0
  3: BEGIN
       itmp = Execute(str1.oper)
     END
  4: RETURN
  else:
ENDCASE


END ; xplot2d_crop

;
;===============================================================================
;

;+
;
; PRO XPLOT2D_PROFILE,wXplot2d,profile=profile
;
;-
PRO XPLOT2D_PROFILE,wXplot2d,profile=profile

on_error,2
widget_control,wXplot2d,get_uvalue=info
widget_control,info.wids.Graph,get_value=graphwin
wset,graphwin
p_old = !p
x_old = !x
y_old = !y
!p=info.psysvar.p
!x=info.psysvar.x
!y=info.psysvar.y


XPLOT2D_SET_COMMENT,wXplot2d,'Click on the two extreme points'
print,'Click on the two extreme points'
cursor,x0,y0,/data,/down   ; getting first point of segment
oplot,[x0,x0],[y0,y0],color=2,psym=1
cursor,x1,y1,/data,/down   ; getting last point of segment

	
ptrData=xplot2d_getimage(wXplot2d,/pointer)

oplot,[x1,x1],[y1,y1],color=2,psym=1
oplot,[x0,x1],[y0,y1],color=2
dd = sqrt( (x1-x0)^2 + (y1-y0)^2 )

!p=p_old
!x=x_old
!y=y_old

profile=(*ptrData)(xplot2d_get_line_points(x0,y0,x1,y1,(size(*ptrData))(1)))

xplot,indgen(n_elements(profile)),profile,/No_Block,Window_Size=[600,400], $
  title='Dist[pixels]: '+StrCompress(dd,/Rem)+', from:'+$
   vect2string([x0,y0])+' to:'+vect2string([x1,y1])

END ; xplot2d_profile


;
;===============================================================================
;


;+
;
; PRO XPLOT2D_QUIT,wXplot2d
;
;-
;   MODIFICATION HISTORY:
;     2007-12-05 srio@esrf.eu Modified for xplot2d 1.0. Documented.
PRO XPLOT2D_QUIT,wXplot2d
on_error,2
widget_control,wXplot2d,get_uvalue=info
Obj_Destroy,info.data
Obj_Destroy,info.objRings
IF Widget_Info(info.wids.detector,/Valid_Id) THEN $
   Widget_Control,info.wids.detector,/Destroy

IF Handle_Info(info.mk.mkIndices) THEN Handle_Free,info.mk.mkIndices

widget_control,wXplot2d,/destroy
END ; xplot2d_quit


;
;===============================================================================
;


;+
;
;PRO XPLOT2D_ROI_MOUSE,wXplot2d,iFlag=iFlag, $
;   center=center,axes=axes,phi=phi, coeff=coeff ; output keywords
;
;-
PRO XPLOT2D_ROI_MOUSE,wXplot2d,iFlag=iFlag, $
   center=center,axes=axes,phi=phi, coeff=coeff ; output keywords
;
; flags: 
;   2 circle: center, radius
;   3 circle: three points
;   23 ellipse: center, axis_a, axis_b
;   15 ellipse: 5 points
;   23 Cake: 3 points
;   30 Polygon: n points
;
on_error,2

widget_control,wXplot2d,get_uvalue=info
;IF N_Elements(iFlag) EQ 0 THEN iFlag=2

xplot2d_REFRESH,wXplot2d

widget_control,info.wids.Graph,get_value=graphwin
wset,graphwin
p_old = !p
x_old = !x
y_old = !y
!p=info.psysvar.p
!x=info.psysvar.x
!y=info.psysvar.y
; new part: overplot rings
; initialize vars
roi_text=''
phi=0D0
roiType=''


CASE iFlag OF
  2: BEGIN ; circle:  center and radius
       ; input points
       XPLOT2D_SET_COMMENT,wXplot2d,'Click on the centre of the circle...'
       CURSOR, Xcent, Ycent, /DOWN, /DATA
       oplot,[Xcent,Xcent],[Ycent,Ycent],psym=1

       XPLOT2D_SET_COMMENT,wXplot2d,'C='+Vect2String(Long([Xcent, Ycent]))+$
         ' Click again to determine the circle radius'
       CURSOR, Xr, Yr, /DOWN, /DATA
       oplot,[Xr,Xr],[Yr,Yr],psym=1 & wait,0.2

       ; calculate parameters
       radius = SQRT((Yr-Ycent)^2+(Xr-Xcent)^2)
       center=[Xcent,Ycent]
       axes=[radius,radius] 
       coeff=[-radius^2+Xcent^2+Ycent^2,-2*Ycent,-2*Xcent,0,1,1]
       roiType='conic'

       roi_text='conic,'+ $
                StrCompress(Xcent,/Rem)+','+$
                StrCompress(Ycent,/Rem)+','+$
                StrCompress(radius,/Rem)

     END
  3: BEGIN ; circle defined by 3 points
       XC=DblArr(3)
       YC=DblArr(3)
       FOR i=0,2 DO BEGIN
         XPLOT2D_SET_COMMENT,wXplot2d,'Click point '+StrCompress(i+1,/Rem)+$
            ' from a total of 3 points on the circle...'
         CURSOR, Xtmp, Ytmp, /DOWN, /DATA
         oplot,[xtmp,xtmp],[ytmp,ytmp],psym=1 & wait,0.2
         XC[i] = Xtmp
         YC[i] = Ytmp
       ENDFOR
       ;
       ; calculate coefficients of circunference:
       ; c[0] + c[1] y + c[2] x + c[3] x y + c[4] y^2 + c[5] x^2
       ;
       coeff = conic_pts2coeffs(XC,YC)
       ; 
       ; calculate center and radius
       ;
       center = conic_coeffs2pars(coeff,radius=radius)
       axes = [radius,radius]
       roiType='conic'
       roi_text='conic,'+ $
                StrCompress(center[0],/Rem)+','+$
                StrCompress(center[1],/Rem)+','+$
                StrCompress(radius,/Rem)
     END

  13: BEGIN ; ellipse: center, a ,b

       XPLOT2D_SET_COMMENT,wXplot2d,'Click on the centre of the ellipse...'
       CURSOR, Xcent, Ycent, /DOWN, /DATA
       oplot,[Xcent,Xcent],[Ycent,Ycent],psym=1

       XPLOT2D_SET_COMMENT,wXplot2d,'C='+Vect2String(Long([Xcent, Ycent]))+$
         ' Click again to determine the major axis'
       CURSOR, Xr1, Yr1, /DOWN, /DATA
       oplot,[Xr1,Xr1],[Yr1,Yr1],psym=1 & wait,0.2

       XPLOT2D_SET_COMMENT,wXplot2d,'C='+Vect2String(Long([Xcent, Ycent]))+$
         ' Click again to determine the minor axis'
       CURSOR, Xr2, Yr2, /DOWN, /DATA
       oplot,[Xr2,Xr2],[Yr2,Yr2],psym=1 & wait,0.2
       
       

       ; calculate pars
       center=[Xcent,Ycent]
       axis1 = SQRT((Yr1-Ycent)^2+(Xr1-Xcent)^2)
       axis2 = SQRT((Yr2-Ycent)^2+(Xr2-Xcent)^2)
       center=[Xcent,Ycent]
       axes=[axis1,axis2]
       phi=atan((Yr1-Ycent)/(Xr1-Xcent))
       coeff = conic_pars2coeffs(Xcent,Ycent,axis1,axis2,phi)
       
       roiType='conic'
       roi_text='conic,'+ $
                StrCompress(Xcent,/Rem)+','+$
                StrCompress(Ycent,/Rem)+','+$
                StrCompress(axis1,/Rem)+','+$
                StrCompress(axis2,/Rem)+','+$
                StrCompress(phi*180D0/!dpi)

     END


  14: BEGIN  ; unrotated ellipse, 4 points
       XC=DblArr(4)
       YC=DblArr(4)
       FOR i=0,3 DO BEGIN
         XPLOT2D_SET_COMMENT,wXplot2d,'Click point '+StrCompress(i+1,/Rem)+$
            ' from a total of 4 points on the ellipse...'
         CURSOR, Xtmp, Ytmp, /DOWN, /DATA
         oplot,[xtmp,xtmp],[ytmp,ytmp],psym=1 & wait,0.2
         XC[i] = Xtmp
         YC[i] = Ytmp
       ENDFOR
       ; 
       ; Compute coefficients of the conic: 
       ; c[0] + c[1] y + c[2] x + c[3] x y + c[4] y^2 + c[5] x^2
       ;
       coeff = conic_pts2coeffs(XC,YC)
       center = conic_coeffs2pars(coeff,phi=phi,axes=axes)
       coeff=coeff/coeff[0]
       roi_text='conic,'+ $
                StrCompress(center[0],/Rem)+','+$
                StrCompress(center[1],/Rem)+','+$
                StrCompress(axes[0],/Rem)+','+$
                StrCompress(axes[1],/Rem)+',0'
       roiType='conic'
       Xcent=center[0]
       Ycent=center[1]
     END



  15: BEGIN  ; ellipse, 5 points
       XC=DblArr(5)
       YC=DblArr(5)
       FOR i=0,4 DO BEGIN
         XPLOT2D_SET_COMMENT,wXplot2d,'Click point '+StrCompress(i+1,/Rem)+$
            ' from a total of 5 points on the ellipse...'
         CURSOR, Xtmp, Ytmp, /DOWN, /DATA
         oplot,[xtmp,xtmp],[ytmp,ytmp],psym=1 & wait,0.2
         XC[i] = Xtmp
         YC[i] = Ytmp
       ENDFOR
       ; 
       ; Compute coefficients of the conic: 
       ; c[0] + c[1] y + c[2] x + c[3] x y + c[4] y^2 + c[5] x^2
       ;
       coeff = conic_pts2coeffs(XC,YC)
       center = conic_coeffs2pars(coeff,phi=phi,axes=axes)
       coeff=coeff/coeff[0]
       roiType='conic'
       roi_text='conic,'+ $
                StrCompress(center[0],/Rem)+','+$
                StrCompress(center[1],/Rem)+','+$
                StrCompress(axes[0],/Rem)+','+$
                StrCompress(axes[1],/Rem)+','+$
                StrCompress(phi*180D0/!dpi)
     END


  23: BEGIN ; cake defined by 3 points
       XC=DblArr(3)
       YC=DblArr(3)
       text = ['Click on the centre of the cake...', $
        'Click again to determine the cake first lower corner', $
        'Click again to determine the cake second upper corner']
       FOR i=0,2 DO BEGIN
         XPLOT2D_SET_COMMENT,wXplot2d,text[i]
         CURSOR, Xtmp, Ytmp, /DOWN, /DATA
         oplot,[xtmp,xtmp],[ytmp,ytmp],psym=1 & wait,0.2
         XC[i] = Xtmp
         YC[i] = Ytmp
       ENDFOR
       Xcent=XC[0]
       Ycent=YC[0]
       Xl = XC[1]
       Yl = YC[1]
       Xu = XC[2]
       Yu = YC[2]

       ;calculate the cake parameters given these values
       rho1=sqrt(float(Ycent-Yl)^2+float(Xcent-Xl)^2)
       rho2=sqrt(float(Ycent-Yu)^2+float(Xcent-Xu)^2)

       if rho2 lt rho2 then begin
          blah=dialog_message(['rho2 lower than rho1!', $
         'Please make sure your third click is farther', $
         'from the centre (first click) than the second'],/ERROR)
         return
       endif

       Yl_diff=float(Yl-Ycent)
       Xl_diff=float(Xl-Xcent)
       Yu_diff=float(Yu-Ycent)
       Xu_diff=float(Xu-Xcent)
       ;since the atan function's image values are all contained in the 
       ;interval [-Pi/2;Pi/2] different cases need to be considered in 
       ;order to allow cake ROI with wider angles
       if (Yl_diff ge 0) and (Xl_diff ge 0) then phi1=Atan(Yl_diff/Xl_diff) $
       else if (Yl_diff ge 0) and (Xl_diff lt 0) then phi1=!pi-Atan(Yl_diff/(-Xl_diff)) $
       else if (Yl_diff lt 0) and (Xl_diff lt 0) then phi1=!pi+Atan((-Yl_diff)/(-Xl_diff)) $
       else if (Yl_diff lt 0) and (Xl_diff ge 0) then phi1=2*!pi+Atan(Yl_diff/Xl_diff)
       if (Yu_diff ge 0) and (Xu_diff ge 0) then phi2=Atan(Yu_diff/Xu_diff) $
       else if (Yu_diff ge 0) and (Xu_diff lt 0) then phi2=!pi-Atan(Yu_diff/(-Xu_diff)) $
       else if (Yu_diff lt 0) and (Xu_diff lt 0) then phi2=!pi+Atan((-Yu_diff)/(-Xu_diff)) $
       else if (Yu_diff lt 0) and (Xu_diff ge 0) then phi2=2*!pi+Atan(Yu_diff/Xu_diff)


       if phi1 eq phi2 then begin
          blah=dialog_message('Identical values for phi!',/ERROR)
          return
       endif
       center=[Xcent,Ycent]
       axes=[rho1,rho2]
       phi=[phi1,phi2]

       roi_text='cake,'+ $
                StrCompress(Xcent,/Rem)+','+$
                StrCompress(Ycent,/Rem)+','+$
                StrCompress(rho1,/Rem)+','+$
                StrCompress(rho2,/Rem)+','+$
                StrCompress(phi1*180/!dpi,/Rem)+','+$
                StrCompress(phi2*180/!dpi,/Rem)
      roiType='cake'
     END
     30: BEGIN ; polygon
        XPLOT2D_SET_COMMENT,wXplot2d, $
       'Left-click inside the image to insert points and RIGHT-CLICK for the last point'
        CURSOR, X1, Y1, /DATA, /DOWN  
        X = X1 & Y = Y1  
        XC=[X1]
        YC=[Y1]
        OPLOT,[X,X1], [Y,Y1], psym=1, color=3
        WHILE (!MOUSE.button NE 4) DO BEGIN  
           CURSOR, X1, Y1, /DATA, /DOWN  
           OPLOT,[X,X1], [Y,Y1], color=3
           X = X1 & Y = Y1  
           XC=[XC,X1]
           YC=[YC,Y1]
        ENDWHILE  
        ; close polygon
        OPLOT,[X1,XC[0]], [Y1,YC[0]], color=3
        XC=[XC,XC[0]]
        YC=[YC,YC[0]]
        
        roiType='polygon'
        ;roi_text='polygon,'+Vect2String(Long(XC))+','+Vect2String(Long(YC))
        roi_text='polygon,'+Vect2String(XC)+','+Vect2String(YC)
     END


     31: BEGIN ; rectangle
        XPLOT2D_SET_COMMENT,wXplot2d, $
       'Click on the first corner of the rectangle'
        CURSOR, X1, Y1, /DATA, /DOWN  
        OPLOT,[X1,X1], [Y1,Y1], psym=1, color=3
        XPLOT2D_SET_COMMENT,wXplot2d, $
       'Click on the oposite corner of the rectangle'
        CURSOR, X2, Y2, /DATA, /DOWN  
        OPLOT,[X2,X2], [Y2,Y2], psym=1, color=3

        XC=[X1,X2,X2,X1,X1]
        YC=[Y1,Y1,Y2,Y2,Y1]
        
        roiType='polygon'
        ;roi_text='polygon,'+Vect2String(Long(XC))+','+Vect2String(Long(YC))
        roi_text='polygon,'+Vect2String(XC)+','+Vect2String(YC)
     END



  else: Message,'Error: Wrong number of points'
ENDCASE


;
; set comment
;
CASE roiType OF
  'conic': BEGIN
         text = 'C='+Vect2string(Long(center)) 
         IF abs(axes[1]-axes[0]) LE 1E-8 THEN $
            text=text+',R='+StrCompress(axes[0],/Rem) ELSE $
            text=text+',Axes='+Vect2String(axes,/Rem) 

         IF abs(phi) GT 1E-8 THEN $
            text=text+',Phi='+StrCompress(phi*180/!dpi,/Rem)+' deg'
         text=[text,'Coeffs='+Vect2string(coeff)]
       END
  'cake': BEGIN
         text = ['C='+Vect2string(Long(center)) + $
                 ',Axes='+Vect2String(axes,/Rem), $
                 'Phi='+Vect2String(phi*180/!dpi,/Rem)]
       END
  'polygon': BEGIN
         text = ['X='+Vect2string(Long(XC)),$
                 'Y='+Vect2string(Long(YC))]
       END
ENDCASE
XPLOT2D_SET_COMMENT,wXplot2d, text

;
; save
;
info.str.roi.last=roi_text

; to avoid affecting the "hand tool" by the points selection
info.cursor1=[0,0,0]

widget_control,wXplot2d,set_uvalue=info
;
; plot roi
;
xplot2d_roi_plot,wXplot2d,coeff=coeff
;
; update edit window (if open)
;
IF Widget_Info(info.wids.roi,/Valid_Id) THEN BEGIN
 xplot2d_roi_edit_updatepanels,info.wids.roi 
ENDIF


!p=p_old
!x=x_old
!y=y_old


END ; xplot2d_roi_mouse


;
;===============================================================================
;

;+
;
; PRO xplot2d_roi_plot,wXplot2d, coeffs=coeff, $
;     refresh=refresh,noPlot=noPlot,noEnvelope=noEnvelope, $
;     points=points,roi_text=roi_text, $ ; output
;     center=center, axes=axes, phi=phi  ; output
;
;-
PRO xplot2d_roi_plot,wXplot2d, coeffs=coeff, $
    refresh=refresh,noPlot=noPlot,noEnvelope=noEnvelope, $
    points=points,roi_text=roi_text, $ ; output
    center=center, axes=axes, phi=phi  ; output


IF Keyword_Set(refresh) THEN xplot2d_refresh,wXplot2d

widget_control,wXplot2d,get_uvalue=info

widget_control,info.wids.Graph,get_value=graphwin
wset,graphwin
p_old = !p
x_old = !x
y_old = !y
!p=info.psysvar.p
!x=info.psysvar.x
!y=info.psysvar.y
; new part: overplot rings

roi_text = info.str.roi.last

if StrCompress(roi_text) EQ '' THEN RETURN

nn = StrParse(roi_text,',',list)
roiType=StrCompress(list[0],/Rem)
CASE roiType OF 
   'conic': BEGIN
      Xcent=Double(List[1])
      Ycent=Double(List[2])
      axis1=Double(List[3])
      IF nn GE 4 THEN axis2=Double(List[4]) ELSE axis2=axis1
      IF nn GE 5 THEN phi=Double(List[5])*!dpi/180 ELSE phi=0D0
      center=[Xcent,Ycent]
      axes=[axis1,axis2]
      IF Not(Keyword_Set(coeff)) THEN  BEGIN
        coeff=conic_pars2coeffs(Xcent,Ycent,axis1,axis2,phi)
      ENDIF
      END
   'cake': BEGIN
      Xcent=Double(List[1])
      Ycent=Double(List[2])
      axis1=Double(List[3])
      axis2=Double(List[4])
      phi1=Double(List[5])*!dpi/180 
      phi2=Double(List[6])*!dpi/180 
      END
   'polygon': 
   '': return
ENDCASE



;
; calculate points for plot
;
CASE roiType OF
  'conic': BEGIN
    ; 
    ; Compute points by evaluating the explicit equation 
    ; y=f(x)
    ;
    discriminant = coeff[3]^2-2*coeff[4]*coeff[5]
    IF discriminant GE 0 THEN BEGIN ; hyperbola or parabola
       pData=info.data->Value([info.image_num],/pointer)
       img = *pData
       datasize=size(img)
       xx=makearray1(Float(datasize[1])/10>200,0D,datasize[1])
       points = conic_coeffs2pts(coeff,xx)
    ENDIF ELSE BEGIN ; ellipse or circle
       points = conic_pars2pts(center,axes,phi)
    ENDELSE
    IF abs(axes[0]-axes[1]) GT 1E-8 THEN BEGIN
      IF Finite(axes[0]) THEN points2=conic_pars2pts(center,axes[0])
      IF Finite(axes[1]) THEN points3=conic_pars2pts(center,axes[1])
    ENDIF
    END
  'cake': BEGIN
    points=XPLOT2D_DRAW_CAKE(Xcent,Ycent,axis1,axis2,phi1,phi2)
    ; append first points to connect last point to the first
    points=transpose([transpose(points),transpose(points[*,0])])
    END
  'polygon': BEGIN
    cmd = StrSubstitute(roi_text,'polygon,','points=[')
    cmd = cmd+']'
    itmp=Execute(cmd)
    IF itmp NE 1 THEN BEGIN
      tmp=Dialog_Message(/Error,['Error executing: ',cmd])
      RETURN
    ENDIF
    points=transpose(points)
    END
ENDCASE

;
; plot 
;
IF Not(Keyword_Set(noPlot)) THEN BEGIN
  IF Not(Keyword_Set(noEnvelope)) THEN BEGIN
    IF N_Elements(points2) GT 0 THEN BEGIN
      Plots, points2[0,*], points2[1,*], color=3, linestyle=2, /data
    ENDIF
    IF N_Elements(points3) GT 0 THEN BEGIN
      Plots, points3[0,*], points3[1,*], color=3, linestyle=2, /data
    ENDIF
  ENDIF
  IF N_Elements(points) GT 0 THEN BEGIN
    Plots, points[0,*], points[1,*], color=2, /data
  ENDIF
ENDIF


!p=p_old
!x=x_old
!y=y_old

END ; xplot2d_roi_plot

;
;===============================================================================
;

;+
;
; PRO xplot2d_saveimage,wid, $,
;     format=format, $ ; Accepted values: 'EDF','XRD', 'Oth' (other).
;     file=filename
;
;-
;   MODIFICATION HISTORY:
;     2007-12-05 srio@esrf.eu Modified for xplot2d 1.0. Documented.
PRO xplot2d_saveimage,wid, $,
    format=format, $ ; Accepted values: 'EDF','XRD', 'Oth' (other).
    file=filename

catch, error_status
 if error_status ne 0 then begin
 message,/info,'error caught: '+!err_string
 if sdep(/w) then itmp = Dialog_Message(/Error,$
   'XPLOT2D_SAVEIMAGE: error caught: '+!err_string)
  catch, /cancel
  on_error,2
  RETURN
endif
widget_control,wid,get_uvalue=info,/hourglass

IF N_Elements(format) EQ 0 THEN format='EDF'

case StrMid(format,0,3) OF
         'EDF': BEGIN
         IF N_Elements(filename) EQ 0 THEN BEGIN
           filename=''
           filename=dialog_Pickfile(filter = '*.edf',file='xplot2d.edf',Dialog_Parent=wId)
           IF filename EQ '' THEN RETURN
         ENDIF
         pData=info.data->Value([info.image_num],/pointer)
         ;Print,Min(*pData),Max(*pData),Total(*pData)
         img = *pData

         tmp = { type:['3',$
;  'Byte':                  idEDF = 'UnsignedByte'
;  'Integer':               idEDF = 'SignedShort'
;  'Unsigned Integer':      idEDF = 'UnsignedShort'
;  'Longword integer':            idEDF = 'SignedInteger'
;  'Unsigned Longword Integer':   idEDF = 'UnsignedInteger'
;  '64-bit Integer':		 idEDF = 'SignedLong'
;  'Unsigned 64-bit Integer':	 idEDF = 'UnsignedLong'
;  'Floating point':              idEDF = 'FloatValue'
;  'Double-precision floating':   idEDF = 'DoubleValue'
;  else:	message,'EDF_WRITE: Data type not recognized: '+idType
             'Byte (1 byte)', $
             'Integer (2 bytes)', $
             'Longword integer (4 bytes)', $
             'Floating point (4 bytes)', $
             'Double-precision floating (8 bytes)', $
             'Unsigned Integer (2 bytes)', $
             'Unsigned Longword Integer (4 bytes)', $
             'Unsigned 64-bit Integer (8 bytes)', $
             '64-bit Integer (8 bytes)' ] }

         XscrMenu,tmp,/NOTYPE,GROUP=wid, $
                WTITLE='write EDF file',ACTION=action,FIELDLEN=20,$
                Dialog_Parent=wid , /Interpret,  $
                Titles=['Convert data to:']
         IF action EQ 'DONT' THEN RETURN

         CASE Fix( (tmp.type)[0]) OF
           0: img = bytscl(img)
           1: img = fix(img)
           2: img = long(img)
           3: img = float(img)
           4: img = double(img)
           5: img = uint(img)
           6: img = ulong(img)
           7: img = ulong64(img)
           8: img = long64(img)
         ENDCASE

         Write_edf,img,file=filename
         END
         'XRD': BEGIN
         IF N_Elements(filename) EQ 0 THEN BEGIN
           filename=''
           filename=dialog_Pickfile(filter = '*.xrd',file='xplot2d.xrd', File='xplot2d.xrd',Dialog_Parent=wId)
           IF filename EQ '' THEN RETURN
         ENDIF
         pData=info.data->Value([info.image_num],/pointer)
         ;Print,Min(*pData),Max(*pData),Total(*pData)
         xplot2d_image = *pData
         Save,xplot2d_image,file=filename
         END
         'Oth': BEGIN
         pData=info.data->Value([info.image_num],/pointer)
         ;Print,Min(*pData),Max(*pData),Total(*pData)
         itmp = dialog_Write_Image(*pData,dialog_Parent=wId,/Warn_Exist)
         END
         else: Message,'Case option not found: '+format
ENDCASE

END ; xplot2d_saveimage

;
;===============================================================================
;

;+
;
; PRO xplot2d_saveinputfile,wid
;
;-
PRO xplot2d_saveinputfile,wid

Widget_Control,wid,Get_UValue=info
	
	  str0 = info.str
	  names = Tag_Names(str0)
	  FOR i=0,N_Tags(str0)-1 DO BEGIN
             tmp = str0.(i)
	     IF Type(tmp) EQ Type({a:0}) THEN BEGIN
	        IF N_Elements(str1) EQ 0 THEN $
		  str1 = Create_Struct(names[i],tmp) ELSE $
		  str1 = Create_Struct(str1, names[i],tmp) 
	     ENDIF
	  ENDFOR
          xop_input_save,str1,File='xplot2d.xop',/tags, Group=wid,  $
            /Write, Comment='; xop/xplot2d(v'+xplot2d_version()+') input file on '+SysTime(),$
	    /NoFlagsTitles,Filter='*.xop'

END ; xplot2d_saveInputFile

;
;===============================================================================
;

;+
;
; PRO xplot2d_scale,wId,scale,max=iMax
;
;-
PRO xplot2d_scale,wId,scale,max=iMax

Widget_Control,wId,Get_UValue=info


interactive=0

IF N_Elements(scale) EQ 0 THEN BEGIN
   interactive=1
ENDIF ELSE BEGIN 
  IF scale LE 0 THEN BEGIN
    interactive=1
    iMax=1
  ENDIF
ENDELSE

IF interactive THEN BEGIN
     sValue = info.str.display.scale
     ; 
     ; compute now the max scale factor for full view of the image
     ;
     ; retrieve current image
     pData=info.data->Value([info.image_num],/pointer)
     img = *pData
     datasize=size(img)
     ; set display
     widget_control,info.wids.Graph,get_value=graphwin
     wset,graphwin
     device,get_screen_size=screen_size 

     max_acceptable_screen_size_factor=info.str.display.max_acceptable_screen_size_factor
     margin_space=info.str.display.margin_space
     fact = (max_acceptable_screen_size_factor*screen_size-2*margin_space)/datasize[1:2]
     fact=min(fact)
     fact=Long(fact*100)/100.0
     scale=fact
     IF Keyword_Set(iMax) THEN GoTo,out

     xEdit,sValue,Text='Scale factor: ', $
       Title='Scale',Action=act   , $
       dialog_Parent=wId,InfoText=['Use View->Scroll Palette if image is larger than window',$
         'Optimum for full view:'+String( fact ,Format='(G4.2)')+' ',$
         'You may set zero for optimum full view, also in Scl']
     IF act EQ 'CANCEL' THEN RETURN
     IF (sValue LE 0) THEN sValue=fact

     scale=sValue
ENDIF

out:
info.str.display.scale=scale
Widget_Control,info.wids.scale, Set_Value=scale
Widget_Control,wId, Set_UValue=info,/Hourglass ; register the changes
xplot2d_REFRESH,wId,imageInside=imageInside
END ; xplot2d_scale


;
;===============================================================================
;

;+
;
; PRO xplot2d_setcolorlimits,wid, $
;   interactive=interactive,  $
;   min=min1,max=max1, $
;   fullrange=fullrange,refresh=refresh,  $
;   stretch=stretch, changeStretch=changeStretch
;
;-
PRO xplot2d_setcolorlimits,wid, $
  interactive=interactive,  $
  min=min1,max=max1, $
  fullrange=fullrange,refresh=refresh,  $
  stretch=stretch, changeStretch=changeStretch

Widget_Control,wid,Get_Uvalue=info

pData=info.data->Value([info.image_num],/pointer)
img = *pData
tmp = Where(img LT 0)
IF tmp[0] EQ -1 THEN BEGIN
          Print,'XPLOT2D_SETCOLORLIMITS: Masked values: None'
ENDIF ELSE BEGIN
          Print,'XPLOT2D_SETCOLORLIMITS: Masked values: ',N_Elements(tmp),' over',N_Elements(img)
          img=img[where(img GE 0)]
ENDELSE

; limits of data

mind = Min(img,Max=maxd)

IF Keyword_Set(fullrange) THEN BEGIN
    min2 = MinD
    max2 = MaxD
ENDIF

IF Keyword_Set(interactive) THEN BEGIN
     out     =  { minmax, min: info.min, max: info.max}
     titles = ['Min :',$
       'Max (set Max=Min for default):']
     XSCRMENU,out,titles=titles,/notype,action=action,$
       WTITLE='Min and max intensity to be displayed', $
       Dialog_Parent=wId
     IF out.min EQ out.max THEN BEGIN
       pData=info.data->Value([info.image_num],/pointer)
       min2 = Min(img,max=max2)
     ENDIF ELSE BEGIN
       min2 = out.min
       max2 = out.max
     ENDELSE
ENDIF


IF Keyword_Set(stretch) THEN BEGIN
    data=histogram(img)
    ;xplot,lindgen(n_elements(data)),data,/No_Block,Window_Size=[600,400]
    ; set first point to zero
    data[0]=0
    data[N_Elements(data)-1]=0
    ; upper cut
    IF Keyword_Set(changeStretch) THEN BEGIN
         sValue = info.str.display.stretch
         IF info.str.display.stretch GT 0 THEN sValue = sValue*100.0 ; %
         infoText = ['Enter here the value for computing the limits of the color table when',$
                     'STRETCH is selected','',$
                    'Two options: ',$
                    '1) Enter a positive value V. In this case the the limits correspond to ',$
                    '   [m1,m2], being m1 and m2 the values of the histogram at V/100 of its',$
                    '   height. Typical values are: 5, 1, 0.1, 0.01, etc.','',$
                    '2) Enter a negative value -V. In this case the limits correspond to ',$
                    '   [m1,m2], being m1 the min(ImageData) and m2 the higher point when',$
                    '   the histogram takes the V value. Typical valies are: -2, -5, -10, etc. ','']

         xEdit,sValue,Text='New value: ',Action=act    , $
           dialog_Parent=wid,Title='Stretch edge (% of the maximum intensity value,'+$
            ' or edge level in histogram with minus sign)',infoText=infoText,XSize=40
         IF act EQ 'CANCEL' THEN RETURN
         IF sValue GE 0 THEN BEGIN
           info.str.display.stretch=sValue/100.
         ENDIF ELSE BEGIN
           info.str.display.stretch=sValue
         ENDELSE
         Widget_Control,wid, Set_UValue=info ; register the changes
    ENDIF
    stretchV = info.str.display.stretch
  IF stretchV GE 0 THEN BEGIN
    ibad = Where (data LE info.str.display.stretch*max(data,imax))
    IF ibad[0] NE -1 THEN BEGIN
      igood1 = where(ibad LT imax)
      igood2 = where(ibad GT imax)
      IF igood1[0] EQ -1 OR igood2[0] EQ -1 THEN BEGIN
        Message,/Info,'Nothing changed.'
        RETURN
      ENDIF
      ; get the extrema
      igood1=ibad[Max(igood1)]
      igood2=ibad[Min(igood2)]
      min1 = igood1 > minD
      max1 = igood2 < maxD
      min2 = min([min1,max1])
      max2 = max([min1,max1])
    ENDIF ELSE BEGIN
      Message,/Info,'Nothing changed.'
      RETURN
    ENDELSE
  ENDIF ELSE BEGIN ; NEGATIVE
    min2 = minD
    itmp = where(data GT abs(stretchV))
    IF itmp[0] NE -1 THEN max2=max(itmp) ELSE max2=maxD
  ENDELSE
ENDIF

info.min=min2
info.max=max2

widget_control,wId,set_uvalue=info
IF Keyword_Set(refresh) THEN xplot2d_refresh,wid
Xplot2d_Set_Comment,wid,'Color table covers range: '+Vect2String([min2,max2])+', data range: '+ $
    Vect2String([minD,maxD])

END ; xplot2d_setcolorlimits 

;
;===============================================================================
;

;+
;
; PRO XPLOT2D_SET_COMMENT,wXplot2d,comment
;
;-
PRO XPLOT2D_SET_COMMENT,wXplot2d,comment
;Set the comment line
   widget_control,wXplot2d,get_uvalue=uvalue
   widget_control,uvalue.wids.Comment,set_value=comment
END


;
;===============================================================================
;

;+
;
; PRO XPLOT2D_REFRESH,wXplot2d,print=print1, noMessage=noMessage, $
;    imageInside=imageInside, imageArray=img, $ ; outputs
;    imageArr2=img2, $ ; outputs
;    xrange=xrange, yrange=yrange ; outputs
;
;-
PRO XPLOT2D_REFRESH,wXplot2d,print=print1, noMessage=noMessage, $
   imageInside=imageInside, imageArray=img, $ ; outputs
   imageArr2=img2, $ ; outputs
   xrange=xrange, yrange=yrange ; outputs

on_error,2
; Displays the image in the widget_draw window,
; or creates the command to be executed for printing

widget_control,wXplot2d,get_uvalue=info

;
; first update the azimuthal integration window 
;

IF Widget_Info(info.wids.XplotAzimuth,/Valid_Id) THEN BEGIN
  Xplot2D_Azimuth_Integral,wXplot2D,out=out
ENDIF ELSE BEGIN
  out=0
ENDELSE
     

;
; retrieve current image
;
pData=info.data->Value([info.image_num],/pointer)
title=info.data->Value([info.image_num],/title)
img = *pData
datasize=size(img)

max_acceptable_screen_size_factor=info.str.display.max_acceptable_screen_size_factor
scroll_step=info.str.display.scroll_step
margin_space=info.str.display.margin_space

IF keyword_set(print1) THEN info.str.display.scale=info.str.display.scale*10

;
; display data
;
IF Not(keyword_set(noMessage)) THEN XPLOT2D_SET_COMMENT,wXplot2d,'displaying image...'

;new_?size is the total frame size including both margins
new_xsize=dataSize(1)+margin_space*2
new_ysize=dataSize(2)+margin_space*2

;
; set display
;
widget_control,info.wids.Graph,get_value=graphwin
IF Not(Keyword_Set(print1)) THEN BEGIN
  wset,graphwin
  erase
ENDIF

;
; TO CHECK...
;
IF Not(Keyword_Set(print1)) THEN $
   device,get_screen_size=screen_size ELSE $
   screen_size=200*datasize[[1,2]]   ; srio TO BE CHECKED

; compute the maximum acceptable screen size dimensions using 
; the current display settings
max_acceptable_screen_size=screen_size*max_acceptable_screen_size_factor

;
;total number of colours available for the image (32 system colours)
;
ncolors = !d.table_size - 32 

if info.str.display.ScaleType EQ 'log' then begin
        ; add one to intensity to avoid negative numbers in the result
        logImg = img+1
        ; set masked pixels to 1
        bad_indices = where(img LT 0)
        good_indices = where(img GE 0)
        IF bad_indices[0] NE -1 THEN logImg[bad_indices]=1
        ; compute log
        logImg=alog10(logImg)
        IF (info.min EQ info.max) THEN BEGIN
          min_img = min(logImg[good_indices],max=max_img)
        ENDIF ELSE BEGIN
          min_img = alog10(info.min+1)
          max_img = alog10(info.max+1)
        ENDELSE
        img2 = bytscl(logimg,min=min_img,max=max_img,top=ncolors-1-32)+byte(32)
        IF bad_indices[0] NE -1 THEN img2[bad_indices] = byte(3) ; color index for mask
endif else begin ; lin image
        IF (info.min EQ info.max) THEN BEGIN
          good_indices = where(img GE 0)
          min_img = min(img[good_indices],max=max_img)
        ENDIF ELSE BEGIN
          min_img = info.min
          max_img = info.max
        ENDELSE
        img2 = bytscl(img,min=min_img,max=max_img,top=ncolors-1-32)+byte(32)
        bad_indices = where(img LT 0)
        IF bad_indices[0] NE -1 THEN img2[bad_indices] = byte(3)
endelse

IF info.str.display.scale NE 1 THEN BEGIN
  IF info.str.display.scale LE 0 THEN BEGIN
    message,'Wrong scale'
  ENDIF
  s1 = size(img2)
  img2 = ConGrid(img2,s1[1]*info.str.display.scale,s1[2]*info.str.display.scale)
ENDIF

IF Not(Keyword_Set(print1)) THEN BEGIN
    ;
    ; DISPLAY COLOR BAR
    ;
    device, window_state=window_state
    IF window_state(info.wids.colorbar) EQ 1 THEN BEGIN ; window exists
    xplot2d_displayct,wXplot2d,MINRANGE=min_img,MAXRANGE=max_img
    ENDIF
ENDIF

;
; DISPLAY IMAGE
;
IF Not(Keyword_Set(print1)) THEN wset,graphwin

; added srio@esrf.eu 2007/12/05
p_old = !p
x_old = !x
y_old = !y

s1 = size(img2)
new_xsize=s1[1]+margin_space*2
new_ysize=s1[2]+margin_space*2


; check whether image size exceeds maximum acceptable screen size
imageInside=0
IF (new_xsize le max_acceptable_screen_size[0]) and $
   (new_ysize le max_acceptable_screen_size[1]) then BEGIN 
   ; full image in display
   imageInside=1
   widget_control,info.wids.Graph,draw_xsize=new_xsize,draw_ysize=new_ysize
   ; image can be fit in the current window so set the image position to [0,0]
   info.str.display.image_position=[0,0] 
   tv, img2, margin_space, margin_space ; , xsize=s1[1], ysize=s1[2]

   ;display the axes around the image by means of the plot function
   ;(in device units)
   margin_space=Float(margin_space)
   !p.position = [margin_space/new_xsize, margin_space/new_ysize,$
           (new_xsize-margin_space)/new_xsize,(new_ysize-margin_space)/new_ysize]
   ;xrange=[info.str.axis_properties.x_origin, $
   ;            dataSize(1)/float(info.str.axis_properties.x_pixel_unit_ratio)]
   ;yrange=[info.str.axis_properties.y_origin,$
   ;            dataSize(2)/float(info.str.axis_properties.y_pixel_unit_ratio)]
   xrange=info.str.axis_properties.x_origin+ [0, $
               dataSize(1)/float(info.str.axis_properties.x_pixel_unit_ratio)]
   yrange=info.str.axis_properties.y_origin+ [0, $
               dataSize(2)/float(info.str.axis_properties.y_pixel_unit_ratio)]
   
   !x.range=xrange
   !y.range=yrange
   !x.style=1
   !y.style=1
   plot, [0,0] ,/nodata, /data, /noerase , title=title

   info.str.display.image_position=[0,0] ; reset image origin
ENDIF ELSE BEGIN 
       ; this is the available size in pixels
       ;widget_control,info.wids.Graph,draw_xsize=xs1,draw_ysize=ys1 
       ; this is the available size for the image
       margin_space=Float(margin_space)
       xs1 = Float(!d.x_size)
       ys1 = Float(!d.y_size)
       xs2 = xs1-2*margin_space
       ys2 = ys1-2*margin_space
       ; this is the size of the (cut) image to be displayed
       xs3 = s1[1]<xs2
       ys3 = s1[2]<ys2
       
;info.str.display.image_position=[0,0]
       pos = info.str.display.image_position
       shiftX = 0
       shiftY = 0
       x4i=info.str.display.image_position[0] ; >0<(s1[1] -1)
       x4e=info.str.display.image_position[0]+xs3 ; >0<(s1[1] -1)
       y4i=info.str.display.image_position[1] ;  >0<(s1[2]-1)
       y4e=info.str.display.image_position[1]+ys3 ; >0<(s1[2]-1)

       IF x4i LT 0 THEN BEGIN
         shiftX=-x4i
         x4i=x4i>0
       ENDIF
       x4i=x4i<(s1[1]-1)

       IF x4e GE s1[1] THEN BEGIN
         x4e=x4e<(s1[1]-1)
       ENDIF
       x4e=x4e>0

       IF y4i LT 0 THEN BEGIN
         shiftY=-y4i
         y4i=y4i>0
       ENDIF
       y4i=y4i<(s1[2]-1)

       IF y4e GE s1[2] THEN BEGIN
         y4e=y4e<(s1[2]-1)
       ENDIF
       y4e=y4e>0

       tv, img2[x4i:x4e,y4i:y4e], margin_space+shiftX, margin_space+shiftY, xsize=xs3, ysize=ys3
       
       !p.position = [margin_space/xs1, margin_space/ys1,$
               1.0-margin_space/xs1,1.0-margin_space/ys1]
       ;xrange=[pos[0]+info.str.axis_properties.x_origin, $
       ;            pos[0] + (xs3/info.str.display.scale)/float(info.str.axis_properties.x_pixel_unit_ratio)]
       ;yrange=[pos[1]+info.str.axis_properties.y_origin,$
       ;            pos[1] + (ys3/info.str.display.scale)/float(info.str.axis_properties.y_pixel_unit_ratio)]
       xrange=[pos[0]/info.str.display.scale+info.str.axis_properties.x_origin, $
                   pos[0]/info.str.display.scale + $
                   (xs3/info.str.display.scale)/float(info.str.axis_properties.x_pixel_unit_ratio)]
       yrange=[pos[1]/info.str.display.scale+info.str.axis_properties.y_origin,$
                   pos[1]/info.str.display.scale + $
                   (ys3/info.str.display.scale)/float(info.str.axis_properties.y_pixel_unit_ratio)]
       
       !x.range=xrange
       !y.range=yrange
       !x.style=1
       !y.style=1
       plot, [0,0] ,/nodata, /data, /noerase , color=1 , title=title

ENDELSE ; end else image bigger than frame size section



;
; overplot marked points
;
IF info.mk.mkIndices NE 0L THEN BEGIN
     i_peaks = -1
     IF handle_info(info.mk.mkIndices) EQ 1 THEN handle_value, info.mk.mkIndices, i_peaks
     IF i_peaks[0] NE -1 THEN $
            peaks2 = array_indices(datasize[1:2],i_peaks,/dim) ELSE $
            peaks2=[-1,-1]
     oplot,peaks2[0,*],peaks2[1,*],color=info.mk.mkColor, $
           PSym=info.mk.mkPSym
ENDIF


;
; overplot azimuth integration onto the image
;
IF ((N_Elements(out) GT 1) AND $
    ( Fix((info.str.pref.overplotDiffFlag)[0]) EQ 1))  THEN BEGIN
;caca
    oo1=Reform(out[3,*]) ; +info.str.detector.x0
    oo2=Reform(out[5,*])
    ; remove background
    IF Fix((info.str.pref.overplotDiffBackground)[0]) EQ 1 THEN $
      oo2=mConvBackground(oo2,NITER=5000,DELTA=3,RET=1)
    ; normalize
    oo2=oo2/max(oo2)*N_Elements(img[0,*])
    ; shift
    oo2=oo2 ; +info.str.detector.y0
    nn1=N_elements(oo1)-1
    nn0=200
    ooX = oo1[nn0:nn1]
    ooY = oo2[nn0:nn1]
    alpha=info.str.pref.overplotDiffAngle*!pi/180
    ooXX = (ooX*cos(alpha)+ooY*sin(alpha))+info.str.detector.x0
    ooYY = (ooX*sin(alpha)*(-1)+ooY*cos(alpha))+info.str.detector.y0
    oplot,ooXX,ooYY,thick=info.str.pref.ring_Thick, $
       color=info.str.pref.overplotDiffColor
    ;
ENDIF


IF keyword_set(print1) THEN info.str.display.scale=info.str.display.scale/10  ; back to original value
widget_control,info.wids.ImageNum,set_value=info.image_num+1
info.psysvar.p=!p
info.psysvar.x=!x
info.psysvar.y=!y
widget_control,wXplot2d,set_uvalue=info
!p=p_old
!x=x_old
!y=y_old


IF Not(Keyword_Set(noMessage)) THEN XPLOT2D_SET_COMMENT,wXplot2d,'displaying image done: '+title

;
; overplot rings
;

; dataSize is passed to avoid reloading the image
IF info.ringsFlag THEN xplot2d_overplotrings,wXplot2d,dataSize=dataSize



END ; Xplot2D_Refresh


;
;===============================================================================
;

;+
;
; PRO XPLOT2D_CHANGE_IMAGE,wXplot2d,index,min=min,max=max,log=log,linear=linear
;
;-
PRO XPLOT2D_CHANGE_IMAGE,wXplot2d,index,min=min,max=max,log=log,linear=linear
;Display another image
; can also change the min/max/scale by the way
   widget_control,wXplot2d,get_uvalue=uvalue
    if(uvalue.image_num EQ index) $
       OR (index LT 0) OR (index GE uvalue.data->Info(/N_Elements))$
    then begin
       widget_control,uvalue.wids.ImageNum,set_value=uvalue.image_num+1
       return

    endif

   uvalue.image_num=index
   if(keyword_set(min)) then uvalue.min=min
   if(keyword_set(max)) then uvalue.min=max
   if(keyword_set(log)) then uvalue.str.display.ScaleType='log'
   if(keyword_set(linear)) then uvalue.str.display.ScaleType='linear'
   widget_control,wXplot2d,set_uvalue=uvalue
   xplot2d_REFRESH,wXplot2d
END

;
;===============================================================================
;

;+
;
; PRO XPLOT2D_LAST_IMAGE,wXplot2d
;
;-
PRO XPLOT2D_LAST_IMAGE,wXplot2d
;Change the image currently displayed
   widget_control,wXplot2d,get_uvalue=uvalue
   image_num=uvalue.data->INFO(/N_ELEMENTS)-1
   XPLOT2D_CHANGE_IMAGE,wXplot2d,image_num
END

;
;===============================================================================
;

;+
;
; PRO XPLOT2D_FIRST_IMAGE,wXplot2d
;
;-
PRO XPLOT2D_FIRST_IMAGE,wXplot2d
;Change the image currently displayed
   ;widget_control,wXplot2d,get_uvalue=uvalue
   XPLOT2D_CHANGE_IMAGE,wXplot2d,0
END

;
;===============================================================================
;

;+
;
; PRO XPLOT2D_NEXT_IMAGE,wXplot2d
;
;-
PRO XPLOT2D_NEXT_IMAGE,wXplot2d
;Change the image currently displayed
   widget_control,wXplot2d,get_uvalue=uvalue
   XPLOT2D_CHANGE_IMAGE,wXplot2d,uvalue.image_num+1
END

;
;===============================================================================
;

;+
;
; PRO XPLOT2D_PREVIOUS_IMAGE,wXplot2d
;
;-
PRO XPLOT2D_PREVIOUS_IMAGE,wXplot2d
;Change the image currently displayed
   widget_control,wXplot2d,get_uvalue=uvalue
   XPLOT2D_CHANGE_IMAGE,wXplot2d,uvalue.image_num-1
END

;
;===============================================================================
;

;+
; PRO XPLOT2D_ADD_IMAGE,wXplot2d, $
;    indata,title=title,substitute=substitute, $
;    refresh=refresh
;
;-
PRO XPLOT2D_ADD_IMAGE,wXplot2d, $
   indata,title=title,substitute=substitute, $
   refresh=refresh,addtitle=addtitle

on_error,2
IF N_Elements(refresh) EQ 0 THEN refresh=1
widget_control,wXplot2d,get_uvalue=uvalue
IF Keyword_Set(addTitle) THEN BEGIN
  title=uvalue.data->Value([uvalue.image_num],/Title)
  title=title+' '+addtitle
ENDIF

IF Keyword_Set(substitute) THEN BEGIN
   ;Substitute first image
   ;uvalue.data->set,0,/clean
   ;uvalue.data->set,0,value=indata,aux=indata,title=title
   ;uvalue.image_num=0

   ;uvalue.data->get,[uvalue.image_num],value=indata,aux=*pData
   imgOld=uvalue.data->Value([uvalue.image_num]) ; ,/pointer)
   uvalue.data->set,[uvalue.image_num],value=indata,aux=imgOld,title=title
ENDIF ELSE BEGIN
   ;Add an image
   uvalue.data->set,value=indata,aux=indata,title=title,/add
   uvalue.image_num=uvalue.data->INFO(/N_ELEMENTS)-1
ENDELSE
widget_control,wXplot2d,set_uvalue=uvalue
IF refresh THEN xplot2d_REFRESH,wXplot2d
END

;
;===============================================================================
;

;+
;
; PRO XPLOT2D_REPLACE_IMAGE,wXplot2d,indata, refresh=refresh
;
;-
PRO XPLOT2D_REPLACE_IMAGE,wXplot2d,indata, refresh=refresh
IF N_Elements(refresh) EQ 0 THEN refresh=1
;replace the current image by another
   widget_control,wXplot2d,get_uvalue=uvalue
   ;print,uvalue.data->Info()
   uvalue.data->set,[uvalue.image_num],value=indata,/clean
   ;print,uvalue.data->Info()
   widget_control,wXplot2d,set_uvalue=uvalue
   IF refresh THEN xplot2d_REFRESH,wXplot2d
END

;
;===============================================================================
;

;+
;
; PRO XPLOT2D_SET_SCALE,wXplot2d,min=min,max=max,log=log,linear=linear
;
;-
PRO XPLOT2D_SET_SCALE,wXplot2d,min=min,max=max,log=log,linear=linear
;Change the scaling of the display
   widget_control,wXplot2d,get_uvalue=uvalue
   if(keyword_set(min)) then uvalue.min=min
   if(keyword_set(max)) then uvalue.max=max
   if(keyword_set(log)) then uvalue.str.display.ScaleType='log'
   if(keyword_set(linear)) then uvalue.str.display.ScaleType='linear'
   widget_control,wXplot2d,set_uvalue=uvalue
   xplot2d_REFRESH,wXplot2d
END



;
;===============================================================================
;

PRO XPLOT2D_ROTATE_IMAGE,wXplot2d,direction
;+
;
; PRO XPLOT2D_ROTATE_IMAGE,wXplot2d,direction
;
; Rotates the currently displayed image, with an optionnal mirror:
;
; Depending on the value of 'direction'
; dir mirror  rotation
; 0   No       None         X0    Y0
; 1   No       90�         -Y0    X0
; 2   No       180�        -X0   -Y0
; 3   No       270�         Y0   -X0
; 4   Yes      None         Y0    X0
; 5   Yes      90�         -X0    Y0
; 6   Yes      180�        -Y0   -X0
; 7   Yes      270�         X0   -Y0
;
;-

Widget_Control,/HourGlass
if direction eq 0 then return
widget_control,wXplot2d,get_uvalue=info
pData=XPLOT2D_GETIMAGE(wXplot2d,/Pointer)
;*pData=rotate(*pData,direction)
img=rotate(*pData,direction)
info.data->set,[info.image_num],value=img,aux=*pData 
xplot2d_REFRESH,wXplot2d
END ; XPLOT2D_ROTATE_IMAGE

;
;=====================================================================
;

;+
;
; PRO XPLOT2D_APPLY_SELECTION_TO_MASK,wXplot2d,Complementary=complementary
;   Sets the current selection to act as a mask on the image 
;   (sets its pixels to -1)
;
;-
PRO XPLOT2D_APPLY_SELECTION_TO_MASK,wXplot2d,Complementary=complementary
; Sets the current selection to act as a mask on the image 
; (sets its pixels to -1)

widget_control,wXplot2d,get_uvalue=info

pData=info.data->Value([info.image_num],/pointer)
img = *pData
datasize=size(img)


xplot2d_roi_plot,wXplot2d,/noPlot,points=points

;axis_properties = {x_origin:0.0,x_pixel_unit_ratio:1.0,y_origin:0.0,y_pixel_unit_ratio:1.0}
ax = info.str.axis_properties

roi_pixels=polyfillv( ax.x_pixel_unit_ratio*points[0,*]-ax.x_origin, $
                      ax.y_pixel_unit_ratio*points[1,*]-ax.y_origin,datasize[1],datasize[2])
;roi_pixels=polyfillv( points[0,*],points[1,*],datasize[1],datasize[2])


IF Keyword_Set(complementary) THEN BEGIN
  indices=lonarr(datasize[1]*datasize[2])
  indices[roi_pixels]=1
  roi_pixels=Where(indices EQ 0)
ENDIF

if (roi_pixels)[0] gt -1 then begin 
  ; perform calculations only if the region is not empty
  img_old=img
  img[roi_pixels]=-1.0
  *pData=img
  ;write back image data to info structure
  info.data->set,[info.image_num],value=img,aux=img_old
  widget_control,wXplot2d,set_uvalue=info
  XPLOT2D_REFRESH,wXplot2d
endif else begin
  blah=dialog_message(['No selection points found!', '(Make sure the ROI is entirely within the', 'image bounds to convert it to a mask)'],/ERROR)
endelse

END


;
;===============================================================================
;

;
;+ 
;
; PRO xplot2d_setrings,wid, dspacingList, $
;         input_type=input_type         ; 0=Clear, 1=FromDabax, 2=FromFile
;                                       ; 3=set from Input (dspacingList, in)
;                                       ; 4=get list (dSpacingList, out)
;                                       ; 5=list to xplot, 6=edit list 
;-

PRO xplot2d_setrings,wid, dspacingList, $
        input_type=input_type         ; 0=Clear, 1=FromDabax, 2=FromFile
                                      ; 3=set from Input (dspacingList, in)
                                      ; 4=get list (dSpacingList, out)
                                      ; 5=list to xplot, 6=edit list 
        
        

IF N_Elements(input_type) EQ 0 THEN BEGIN
  input_type=0
  IF N_Elements(dspacingList) NE 0 THEN input_type=3
ENDIF

Widget_control,wId,get_uValue=info

;
; get/modify input data
;
;det = info.str.detector


CASE input_type OF

   0: BEGIN  ; clear
          CASE info.RingsFlag OF
                       0: Return
                       1: BEGIN
                          info.ringsFlag = 0
                          widget_control,wid ,set_uvalue=info
                          xplot2d_REFRESH,wid 
                          Return
                          END
                       else:
          ENDCASE
      END
   1: BEGIN  ; calculate diffraction lines from crystal in DABAX 
        str1 = xpowder_defaults()

        xpowder_loaducstr = str1.ucLoad
        titles=str1.ucLoad_Titles
        flags=str1.ucLoad_Flags

        XscrMenu,xpowder_loaducstr,Flags=flags,Titles=titles, $
	  Action=action,/NoType,/Interp, $
          Wtitle='Load unit cell',Dialog_Parent=wid
	IF action EQ 'DONT' THEN RETURN

        info.ringsFlag = 1
	;str1.ucLoad=xpowder_loaducstr

        CASE Fix( (xpowder_loaducstr.from)[0] ) OF
          0: BEGIN
               crystalIndex=Fix(xpowder_loaducstr.crystal[0])  ; index starts from zero
               crystalName=xpowder_loaducstr.crystal[crystalIndex+1]
               dabax_crystals,crystalIndex,str=crystalStr,cell=crystalCell
             END
          1: BEGIN
               file=StrCompress(xpowder_loaducstr.file,/Rem)
               IF file EQ '?' THEN BEGIN
                 file = Dialog_PickFile(Filter='*.ucell',Title='Select a file with unit cell')
                 IF file EQ '' THEN RETURN
                 IF CheckFile(file) NE 1 THEN BEGIN
                   itmp = Dialog_Message(/Error,'File not found: '+f)
                   Return
                 ENDIF
                 xpowder_loaducstr.file=file
               ENDIF
               crystalName=file
               crystalStr=rascii(file,Skip=1,Header=h)
               crystalCell=FltArr(6)
               openR,unit,file,/Get_Lun
               readF,unit,crystalCell
               Free_Lun,unit
               title='File: '+file
             END
        ENDCASE

        ; 
        ; confirm/edit unit cell
        ;
        cell = {a:crystalCell[0], b:crystalCell[1], c:crystalCell[2], $
                alpha:crystalCell[3], beta:crystalCell[4], gamma:crystalCell[5], $
                dSpacingMin:1.0 } 
        XscrMenu,Cell, titles=['a','b','c','alpha','beta','gamma','Min dSpacing for diffractogram'], $ 
	  Action=action,/NoType,Interp=1, NColumn=1,  $
          Wtitle='Confirm/Edit unit cell',Dialog_Parent=wid
	IF action EQ 'DONT' THEN RETURN
         crystalCell=[Cell.a, Cell.b, Cell.c, Cell.alpha, Cell.beta, Cell.gamma]
        
	; 
	; make calculations
	;

        Widget_Control,/HourGlass
        wavelength = physical_constants('hc')/info.str.detector.energy
	out2 = xpowder_calchkl(Group=wid, $
           CrystalCell=crystalCell,CrystalStr=crystalStr, $
           limitFlg='1',limitValue=Cell.dSpacingMin, $
           wavelength=wavelength)

        ;
        ; plot diffractogram
        ;
        xplot,parent=p, Double(out2),xcol=4,ycol=9,/no_block,coltitles=$
            ['h','k','l','2*Theta','d-spc','|F|','Int','M','IntAll','IntNorm'], $
            wtitle='Diffractogram for reference: '+crystalName+' at E='+$
            StrCompress(info.str.detector.energy,/Rem)
        UserSym,[0.,0.],[-1e9,0]
        xplot_controls_action,p,PSymbol=8,PSYMSign='1',Clr_Lines=2

        ; 
        ; store info
        ;
        peaks = Make_Set(Reform(out2[4,*]),Reform(out2[9,*]))
        igood = uniq(peaks[0,*],sort(peaks[0,*]))
        IF N_Elements(peaks[*,0]) EQ 2 THEN peaks=Make_Set(Reform(peaks[0,*]),$
             Reform(peaks[1,*]),Reform(peaks[0,*])*0+2)
          
        info.objRings->remove
        info.objRings->set,0,Title='Reference: '+crystalName,Flag=0, $
                   Value=peaks[*,igood],/Clean 

        widget_control,wid ,set_uvalue=info

      END

   2: BEGIN  ; file
           helpText=[$
           'A multicolumn ASCII file with one or several phases must', $
           'be loaded, with peak values of the diffractogram in ', $
           'either twotheta or d-spacing (text comments are ignored): ', $
           ' 1st column: twotheta[deg] or d-spacing [A]', $
           ' 2nd column: intensity value (for labeling rings)', $
           ' 3rd column: color index (2: red, 3:green, 4:blue, etc)','','', $
           'For using several crystalline phases, append all of them in ',$
           'the same file, and use a different color index to differentiate',$
           'them. Reflections can be removed by commenting the line and ',$
           'reloading the file.' ]


          rings = info.str.rings

          XScrMenu,rings,Action=act,Titles=info.str.rings_titles, /NoType,/Interp, $
            Flags=info.str.rings_flags, Help=helpText, $
            WTitle='Overplot rings',dialog_Parent=wid ,FieldLen=40

          IF act EQ 'DONT' THEN RETURN

          info.ringsFlag = 1

          CASE Fix( (rings.abscissas)[0] ) OF
              0: BEGIN
                 txt = 'Get file with reference peaks vs TwoTheta[deg]'
                 filter='*.peaks'
                 END
              1: BEGIN
                 txt = 'Get file with reference peaks vs d-spacing[A]'
                 filter='*.dspacing'
                 END
              else:
          ENDCASE

          IF (StrCompress(rings.file,/Rem) EQ '?') THEN BEGIN
            file = Dialog_Pickfile(Title=txt,Dialog_Parent=wid , $
                 filter=filter)
            IF file EQ '' THEN RETURN
            rings.file = file
          ENDIF ELSE BEGIN
            IF checkFile(rings.file) EQ 0 THEN BEGIN
               itmp = Dialog_Message(['File not found: '+$
                 rings.file,'Load a new one?'],/Cancel, $
                 Dialog_Parent=wid )
               IF itmp EQ 'Cancel' THEN RETURN ELSE BEGIN
                 file = Dialog_Pickfile(Title=txt,Dialog_Parent=wid )
                 IF file EQ '' THEN RETURN
                 rings.file = file[0]
               ENDELSE
            ENDIF
          ENDELSE

          peaks = rascii(rings.file)

          IF Fix( (rings.removeDuplicate)[0] ) EQ 1 THEN BEGIN
            igood = uniq(peaks[0,*],sort(peaks[0,*]))
            IF igood[0] NE -1  THEN BEGIN
              IF Fix( (rings.abscissas)[0] ) EQ 0 THEN BEGIN
                 peaks=peaks[*,igood] 
                 ; sort by phase
                 IF N_ELEMENTS(peaks[*,0]) GE 3 THEN peaks=peaks[*,sort(peaks[2,*])] 
              ENDIF ELSE BEGIN ; two theta
                 peaks=peaks[*,reverse(igood)]  ; d-spacing
                 ; sort by phase
                 IF N_ELEMENTS(peaks[*,0]) GE 3 THEN peaks=peaks[*,sort(peaks[2,*])] 
              ENDELSE
            ENDIF
	  ENDIF

          ; 
          ; compute d-spacing if input is in twoTheta
          ;
          IF Fix( (rings.abscissas)[0] ) EQ 0 THEN BEGIN 
             CASE Fix( (rings.ener_flag)[0] ) OF
               0: e0 = 8047.78
               1: e0 = 17479.34
               2: e0 = 20216.1
               3: e0 = 59318.24
               4: e0 = Float( rings.ener_val)
               else:
             ENDCASE
             lambda=physical_constants('hc')/e0 
             peaksT = Reform(peaks[0,*])*!pi/180/2 ; theta in rad
             peaks[0,*] = (lambda/2)/sin(peaksT)
          ENDIF
          IF N_Elements(peaks[*,0]) EQ 2 THEN peaks=Make_Set(Reform(peaks[0,*]),$
             Reform(peaks[1,*]),Reform(peaks[0,*])*0+2)
          
          phases = Reform(Long(peaks[2,*]))
          phases = phases[ Uniq(phases)]

          FOR i=0,N_Elements(phases)-1 DO BEGIN
          igood = where(peaks[2,*] EQ phases[i])
            IF i EQ 0 THEN BEGIN
              info.objRings->remove
              info.objRings->set,0,Title='phase with color '+StrCompress(phases[i],/Rem),Flag=0, $
                   Value=peaks[*,igood],/Clean 
            ENDIF ELSE BEGIN
               info.objRings->set,Title='phase with color '+StrCompress(phases[i],/Rem),Flag=0, $
                   Value=peaks[*,igood],/Add
            ENDELSE
          ENDFOR

          info.str.rings=rings

          widget_control,wid ,set_uvalue=info
      END
      3: BEGIN   ; set rings from input variable

          ss = size(dspacingList) 
          IF (ss[0] EQ 2) AND (ss[1] EQ 3) THEN BEGIN
           ; OK
          ENDIF ELSE BEGIN
           tmp = Dialog_Message('XPLOT2D_SETRINGS: No input list',/Error,$
             Dialog_Parent=wid)
           RETURN
          ENDELSE

          phases = Reform(Long(dspacingList[2,*]))
          phases = phases[ Uniq(phases)]

          FOR i=0,N_Elements(phases)-1 DO BEGIN
          igood = where(dspacingList[2,*] EQ phases[i])
            IF i EQ 0 THEN BEGIN
              info.objRings->remove
              info.objRings->set,0,Title='phase with color '+StrCompress(phases[i],/Rem),Flag=0, $
                   Value=dspacingList[*,igood],/Clean 
            ENDIF ELSE BEGIN
               info.objRings->set,Title='phase with color '+StrCompress(phases[i],/Rem),Flag=0, $
                   Value=dspacingList[*,igood],/Add
            ENDELSE
          ENDFOR
          info.ringsFlag=1
          widget_control,wid ,set_uvalue=info
      END
      4: BEGIN   ; get list into variable

         IF info.ringsFlag EQ 0 THEN BEGIN
           tmp = Dialog_Message('No overplotted rings',/Error,$
             Dialog_Parent=wid)
           RETURN
         ENDIF
         nn= info.objRings->Info(/N_Elements)
         FOR i=0,nn-1 DO BEGIN
           tmpData = info.objRings->Value([i])
           tmpTitle = info.objRings->Value([i],/title)
           IF i EQ 0 THEN BEGIN
            allTitle=tmpTitle
            allPeak=Reform(tmpData[0,*])
            allInt=Reform(tmpData[1,*])
            allColor=Reform(tmpData[2,*])
           ENDIF ELSE BEGIN
            allTitle=[allTitle,tmpTitle]
            allPeak=[allPeak,Reform(tmpData[0,*])]
            allInt=[allInt,Reform(tmpData[1,*])]
            allColor=[allColor,Reform(tmpData[2,*])]
           ENDELSE
         ENDFOR
         dspacingList=make_set(allPeak,allInt,allColor)
         END

      5: BEGIN   ; Export list to Xplot

         ; call itself to get data
         xplot2d_setrings,wid,input_type=4,tmp88
         IF N_Elements(tmp88) EQ 0 THEN RETURN
         ; get 2theta from dspacing
         allPeak=Reform(tmp88[0,*])
         allInt=Reform(tmp88[1,*])
         allColor=Reform(tmp88[2,*])
         lambda=physical_constants('hc')/info.str.detector.energy
         twoTheta=2.*asin(lambda/allPeak/2) ; 2 theta in rads
         twoTheta=twoTheta*180D0/!dpi

         xplot,parent=p,make_set(twoTheta,allPeak,allInt,allColor),colTitles=$
                  ['twoTheta [deg]','dSpacing [A]','Intensity','Phase Color'],$
                  /NO_Block, XCol=1, YCol=3 ,XTitle='-1',YTitle='-1'
         xplot_controls_action,p,PSYMBOL='11', PSYMSIGN='1'

      END
      6: BEGIN    ; Edit
         ; call itself to get+set data anc xtable to edit
         xplot2d_setrings,wid,input_type=4,tmp88 ; get reflections
         IF N_Elements(tmp88) EQ 0 THEN RETURN
         xtable,tmp88,xtitle=['dSpacing','Intensity','Color code'],$
            action=action,/Buttons,infoText='Edit the dspacing list...'
         IF action EQ 'Cancel' THEN RETURN
         xplot2d_setrings,wid,input_type=3,tmp88 ; set reflections
         END
      ELSE:
ENDCASE

;
; refresh
;
xplot2d_REFRESH,wid 
END  ; xplot2d_setrings


;
;=====================================================================
;
PRO XPLOT2D_EVENT,event

catch, error_status
 if error_status ne 0 then begin
 message,/info,'error caught: '+!err_string
 if sdep(/w) then itmp = Dialog_Message(/Error,$
   'XPLOT2D_EVENT: error caught: '+!err_string)
  catch, /cancel
  on_error,2
  RETURN
endif


IF Tag_Names(event,/Structure_Name) EQ 'WIDGET_KILL_REQUEST' THEN BEGIN
  xplot2d_quit,event.top
  RETURN
ENDIF

;
; Define some actions from widgets with no UValue
;
if tag_names(event,/structure_name) EQ 'WIDGET_BASE' then begin
   action = 'RESIZE'
endif else begin
   if tag_names(event,/structure_name) EQ 'WIDGET_DRAW' then begin
      action = 'CURSOR'
   endif else begin
      junk=where(tag_names(event) EQ 'VALUE',ct)
      if ct eq 0 then $
       Widget_Control, event.id, get_UValue=action $
      else action=event.value
   endelse
endelse

   ;
   ; First the actions that do not require the "info" explicitely
   ; =============================================================
   ;
   CASE String(action) OF
;     'SETDETECTOR':     BEGIN
;        xplot2d_setDetector,event.top
;        return
;        END
     'SETRINGS':     BEGIN
        Widget_Control,event.id,Get_Value=val
        CASE val OF
         'Clear':                                             input_type=0
         'Edit':                                              input_type=6
         'Display in xplot':                                  input_type=5
         'Calculate from unit cell (DABAX database/file)...': input_type=1
         'Read dSpacing or twoTheta from file...':            input_type=2
         else: 
        ENDCASE
        xplot2d_setRings,event.top,input_type=input_type
        return
        END
     'VIEW SCALE': BEGIN
        xplot2d_scale,event.top
        return
        END
     'CALC WARPING': BEGIN
        xplot2d_calc_warping,event.top,interactive=1
        return
        END
     'CALC PEAKS': BEGIN
        xplot_findpeaks,event.top,openwindow=1
        return
        END
     'CALC CORRECTIONS': BEGIN
        xplot2d_calc_corrections_sequence,event.top,interactive=1
        return
        END
     'CALC SUBSTRACT': BEGIN
        xplot2d_calc_substract,event.top,interactive=1
        return
        END
     'CALC DIVIDE': BEGIN
        xplot2d_calc_divide,event.top,interactive=1
        return
        END
     'CALC GENERIC': BEGIN
        xplot2d_calc_generic_operation,event.top,interactive=1
        return
        END
     'CALC REBIN': BEGIN
        xplot2d_calc_rebin,event.top,interactive=1
        return
        END
     'CREATE GRID IMAGE': BEGIN
        xplot2d_create_grid_image,event.top,interactive=1
        return
        END
     'CREATE TEST IMAGE': BEGIN
        xplot2d_create_test_image,event.top
        return
        END

     'EDIT PREF' : BEGIN
        Widget_Control,event.id,Get_Value=value
;caca
        CASE value OF
         'Azimuthal Integration preferences...': BEGIN
           widget_control,event.top,get_uvalue=info
           str=info.str.pref
            XScrMenu,str,Action=act,Titles=info.str.pref_titles, $
               Flags=info.str.pref_flags, $
               /NoType,/Interp, WTitle='Set preferences...',$
               dialog_Parent=event.top,FieldLen=40
            IF act EQ 'DONT' THEN RETURN
           info.str.pref=str
           Widget_Control,event.handler,Set_UValue=info
           Xplot2d_refresh,event.handler
           END
          'Set annulus width...': BEGIN
             widget_control,event.top,get_uvalue=info
             width = info.str.roi.annulus
             xedit,width,Text='Annulus width [pixels]: ',$
             Title='Annulus width', Action=action, Dialog_Parent=event.top
             IF action EQ 'CANCEL' THEN Return
             info.str.roi.annulus=width
             widget_control,event.top,set_uvalue=info
             END
          'Set load image properties...': xplot2d_imageformat,event.top
          'Define detector and beam pars...': xplot2d_setDetector,event.top
          'Set axis properties...': xplot2d_axis_properties, event.top
           else:
        ENDCASE
        return
       END
     else: ; print,'Action case not found: '+String(action)
   ENDCASE

   ;
   ; restore the info structure
   ;
   widget_control,event.top,get_uvalue=info


   ;widget_control,event.id,get_uvalue=widget
   ;print, "widget : ", widget
   ;
   ; Actions
   ;

   ;
   ;change image to be displayed
   ;
   if event.id eq info.wids.ImageNum then begin
    XPLOT2D_CHANGE_IMAGE,event.top,event.value-1
    return
   endif
   if event.id eq info.wids.scale then begin
    Widget_Control,info.wids.scale,Get_Value=val
    xplot2d_scale,event.top,val
    return
   endif

   ; debugging messages
   ;print,'Action:','**'+string(action)+'**'
   ;print, "scroll_step : ",scroll_step

   CASE action of
      'RESIZE': begin
      ; widget_control,info.wids.Graph,draw_xsize=event.x,draw_ysize=event.y
      end
      
      ; Buttons and others
      ;
      'CURSORCHANGE':begin
          tmp=info.str.cursor
          XScrMenu,tmp,Action=act,Titles=info.str.cursor_titles,$
             /NoType,/Interp, WTitle='Cursor reading', $
             dialog_Parent=event.top,FieldLen=40
          IF action EQ 'DONT' THEN RETURN
         info.str.cursor=tmp
         Widget_Control,event.top, Set_UValue=info
        end
      'CURSOR':begin


       pData=info.data->Value([info.image_num],/pointer)
       img = *pData
       datasize=size(img)

       p_old = !p
       x_old = !x
       y_old = !y
       !p=info.psysvar.p
       !x=info.psysvar.x
       !y=info.psysvar.y
       coords_d = [event.x,event.y]

       ;
       ; this fails to take the correct value after visiting another
       ; window. I do not know why. srio@esrf.eu 2007/12/10
       ;
       ;coords = convert_coord(event.x,event.y,/device,/to_data)
       coords = (coords_d-info.str.display.margin_space)/info.str.display.scale
       ;xVal=coords[0]
       ;yVal=coords[1]

       pos = Float(info.str.display.image_position)
       xVal=coords[0]+ pos[0]/info.str.display.scale
       yVal=coords[1]+ pos[1]/info.str.display.scale

       if (xVal ge 0) and (yVal ge 0) and $
         (xVal lt dataSize[1]) and $
         (yVal lt dataSize[2]) then tmp=img[xVal,yVal] else tmp=-1
       CASE Fix((info.str.cursor.type)[0]) OF
         0: BEGIN
              Widget_Control,info.wids.cursor,set_value=STring(xVal,yVal,' I=',tmp,Format='(I5,I5,A3,F8.2)')
              END
         1: BEGIN
              det = info.str.detector
              wavelength = Float(Physical_Constants('hc')/det.energy)

              IF Fix( (det.useEllipses)[0] ) EQ 1 THEN BEGIN
                rho2 = (xVal-det.xS)^2 + (yVal-det.yS)^2 + det.zS^2
                tau2 = (xVal-det.x0)^2 + (yVal-det.y0)^2 
                ; apply cosine theorem to get the angle subtended from pixel-vS with the beam
                vv2 = (det.x0-det.xS)^2 + (det.y0-det.yS)^2 + det.zS^2
                cosTwoTheta = (tau2 - rho2 - vv2)/(-2D0*Sqrt(rho2)*Sqrt(vv2))
                TwoTheta = acos(cosTwoTheta)
              ENDIF ELSE BEGIN
                tau2 = (xVal-det.x0)^2 + (yVal-det.y0)^2 
                TwoTheta=atan2(sqrt(tau2),det.zS)
              ENDELSE
              dspacing=wavelength/2/sin(TwoTheta/2)
              Widget_Control,info.wids.cursor,set_value=STring('2t:',TwoTheta*180/!pi, $
                ' d:',dspacing,' I:',tmp,Format='(A3,F5.2,A4,F5.2,A4,F8.2)')

            END
         2: BEGIN
              str = info.str.axis_properties
              xVal1 = str.x_origin+xVal/str.x_pixel_unit_ratio
              yVal1 = str.y_origin+yVal/str.y_pixel_unit_ratio
              Widget_Control,info.wids.cursor,set_value=STring(xVal1,yVal1,' I=',tmp,Format='(2F7.2,A3,F8.2)')
            END
         ELSE: Widget_Control,info.wids.cursor,set_value='ERROR!'
       ENDCASE

       ; this is for the "hand tool"
       if event.press eq 1 then begin
          info.cursor1=[xVal,yVal,1]
          widget_control,event.top,set_uvalue=info
       endif
       if event.release eq 1 and info.cursor1[2] EQ 1 then begin
          xy = (info.cursor1)[0:1]
          diff=[xVal,yVal]-xy
          IF Total(diff EQ [0,0]) NE 2 THEN BEGIN
            info.str.display.image_position=info.str.display.image_position- $
              diff*info.str.display.scale
            widget_control,event.top,set_uvalue=info
            xplot2d_refresh,event.top
          ENDIF
       endif


       IF event.press EQ 4  THEN BEGIN ; right click
         IF Widget_Info(info.wids.FindPeaks,/Valid_Id) THEN BEGIN
            xplot_findpeaks_rightclick,info.wids.FindPeaks,xVal[0],YVal[0]
	 ENDIF
       ENDIF




       !p=p_old
       !x=x_old
       !y=y_old

     END

      'FIRST IMAGE':   XPLOT2D_FIRST_IMAGE,event.top
      'LAST IMAGE':    XPLOT2D_LAST_IMAGE,event.top
      'NEXT IMAGE':    XPLOT2D_NEXT_IMAGE,event.top
      'PREVIOUS IMAGE':XPLOT2D_PREVIOUS_IMAGE,event.top
      'DELETE IMAGE': BEGIN
         numberOfImages=info.data->INFO(/N_ELEMENTS)

         IF numberOfImages EQ 1 THEN BEGIN
           itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
            'At least one image must exist. Aborted')
           Return
         ENDIF


         itmp = Dialog_Message(/Info,Dialog_Parent=event.top, $
            'Delete current image (cannot undo)? Please confirm',/Cancel)
         IF itmp EQ 'Cancel' THEN RETURN

         current_index = info.image_num
         title=(info.data)->value([current_index],/title)
         (info.data)->remove,[current_index]

         ; display new image
         Widget_Control,info.wids.imageNum,Get_Value=imgnum
             numberOfImages=info.data->INFO(/N_ELEMENTS)
             imgnum = imgnum < numberOfImages  ; number 1,2..

         info.image_num=imgnum-1           ; index 0,1,...
         Widget_Control,event.top,Set_UValue=info
         ; refresh
         Widget_Control,/HourGlass
         xplot2d_REFRESH,event.top
         XPLOT2D_SET_COMMENT,event.top,'Image removed: '+title
        END
      'DELETE ALL': BEGIN
         numberOfImages=info.data->INFO(/N_ELEMENTS)

         IF numberOfImages EQ 1 THEN BEGIN
           itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
            'At least one image must exist. Aborted')
           Return
         ENDIF


         itmp = Dialog_Message(/Info,Dialog_Parent=event.top, $
            'Delete ALL images (cannot undo)? Please confirm',/Cancel)
         IF itmp EQ 'Cancel' THEN RETURN

         current_index = info.image_num
         indices = (indgen(numberOfImages))[1:numberOfImages-1]
         (info.data)->remove,indices

         ; display new image
         numberOfImages=info.data->INFO(/N_ELEMENTS)

         info.image_num=0           ; index 0,1,...
         Widget_Control,event.top,Set_UValue=info
         ; refresh
         Widget_Control,/HourGlass
         xplot2d_REFRESH,event.top
         XPLOT2D_SET_COMMENT,event.top,'All images (except the first which must always exist) removed.'

        END

      ;
      ; File menu
      ;
      'LOAD IMAGE': XPLOT2D_LOADIMAGE, event.top
      'RELOAD IMAGE': XPLOT2D_LOADIMAGE, event.top, /reLoad
      'LOAD INCREMENTED IMAGE': XPLOT2D_LOADIMAGE, event.top, /Increment
      'LOAD DECREMENTED IMAGE': XPLOT2D_LOADIMAGE, event.top, Increment=-1
      ;'IMAGEFORMAT': xplot2d_imageformat,event.top
      'SAVE IMAGE':BEGIN
         Widget_Control,event.id,Get_Value=format
         xplot2d_saveimage,event.top,format=format
         END


      'XPLOT2D_SAVE': xplot2d_saveinputfile,event.top
      'XPLOT2D_LOAD': xplot2d_loadInputFile,event.top,group=event.top
  
      'EXIT': Xplot2d_Quit, event.top

      ;
      ; Edit menu
      ;

      'UNDO' : begin
        if (info.data->Value([info.image_num],/isauxdef)) eq 1 then begin
          img=info.data->Value([info.image_num],/Aux)
          title=info.data->Value([info.image_num],/Title)
          info.data->Set,[info.image_num],Value=img,Title=title+' (undo)'
          xplot2d_REFRESH,event.top
        endif else $
          blah=dialog_message('No modifications to be undone!',/ERROR)
        end

    'COPYPASTE':    BEGIN
       Widget_Control,event.id,Get_Value=val
       pData=info.data->Value([info.image_num],/pointer)
       CASE val OF
       'Copy...': BEGIN
         tmp = {what:['0','data array','data pointer'], $
                name:'CLIPBOARD_XOP'}
         XscrMenu,tmp,/NOTYPE,GROUP=event.top, $
                WTITLE='Copy',ACTION=action,FIELDLEN=20,$
                Dialog_Parent=event.top , /Interpret,  $
                Titles=['Copy','Main level variable name']
         if action EQ 'DONT' then return
         CASE Fix( (tmp.what)[0] ) OF
           0: (Scope_VarFetch(tmp.name, Level=1, /Enter)) = *pData
           1: (Scope_VarFetch(tmp.name, Level=1, /Enter)) = pData
         ENDCASE
         END
       'Paste': BEGIN
           tmp = Scope_VarFetch('CLIPBOARD_XOP', Level=1)
           IF Type(tmp) EQ 10 THEN tmp=*tmp ; pointer
           info.data->set,[info.image_num],value=tmp,aux=*pData,title='Image pasted (CLIPBOARD_XOP)'
           ; not needed, as we only updated a referenced object
           ;widget_control,event.top,set_uvalue=info
           XPLOT2D_REFRESH,event.top
           return
         END
       'Paste special...': BEGIN
            vars = Scope_VarName(Level=1,Count=nn)
            tmp = {from:['0','Main Level Variable','Xplot Id'], $
                what:['0','Image','Only mask'],$
                main:['0',vars], id:0L} 
            XscrMenu,tmp,/NOTYPE,GROUP=event.top, $
                WTITLE='Paste special',ACTION=action,$
                Dialog_Parent=event.top , /Interpret,  $
                Titles=['From','What to paste','Main level variables','Xplot2D Id:'], $
                Flags=['1','1','w(0) EQ 0','w(0) EQ 1']
         if action EQ 'DONT' then return
         CASE Fix((tmp.from)[0]) OF 
           0: BEGIN ; main var
              mainvar = (tmp.main)[1+Fix((tmp.main)[0])]
              img = Scope_VarFetch(mainvar, Level=1)
              IF Type(img) EQ 10 THEN img=*img ; pointer
              txt = '('+StrCompress(StrUpCase(mainvar),/Rem)+')'
              END
           1: BEGIN ; xplot id
	      IF Widget_Info(tmp.id,/Valid_Id) THEN BEGIN
                 img = xplot2d_getimage(tmp.Id)
                 txt = '(Xplot2d Id: '+StrCompress(tmp.Id,/Rem)+')'
              ENDIF ELSE BEGIN
                itmp = Dialog_Message(/Error, $
                  'Xplot2d window not found, Id: '+StrCompress(tmp.id,/Rem), $
                   Dialog_Parent=event.top)
                Return
              ENDELSE
              END
         ENDCASE
         IF Fix((tmp.what)[0]) EQ 1 THEN BEGIN ; apply only mask
           mask=temporary(img)
           ibad = where(mask LT 0) 
           IF ibad[0] EQ -1 THEN BEGIN
                itmp = Dialog_Message(/Error, $
                  'No masked points', Dialog_Parent=event.top)
                return
           ENDIF ELSE BEGIN
               img=*pData
               img[ibad]=mask[ibad]
           ENDELSE
         ENDIF
         info.data->set,[info.image_num],value=img,aux=*pData,title='Image pasted '+txt
         XPLOT2D_REFRESH,event.top


         END
       ENDCASE
       END



      'EDIT IMAGE TITLES' : BEGIN
          txt_old = info.data->value(/title)
          txt_new = txt_old
          action = ''
          XDisplayFile1,Text=txt_new,Title='Edit image labels', $
                Group=event.top,/Modal,Action=action, $
                Dialog_Parent=event.top
          IF action EQ 'DONT' THEN RETURN
          IF N_Elements(txt_new) NE N_Elements(txt_old) THEN BEGIN
                itmp = Dialog_Message(/Error, Dialog_Parent=event.top, $
                 'Number of label lines has changed. Aborted')
                RETURN
          ENDIF
          (info.data)->set,Title=txt_new
          Widget_Control,/HourGlass
          xplot2d_REFRESH,event.top
         END


      'EDIT RING TITLES' : BEGIN
          txt_old = info.objRings->value(/title)
          IF N_Elements(txt_old) EQ 1 AND StrCompress(txt_old[0],/Rem) EQ '' THEN BEGIN
                itmp = Dialog_Message(/Error, Dialog_Parent=event.top, $
                 'No rings overplotted')
                RETURN
          ENDIF
          txt_new = txt_old
          action = ''
          XDisplayFile1,Text=txt_new,Title='Edit labels of the overplotted rings', $
                Group=event.top,/Modal,Action=action, $
                Dialog_Parent=event.top
          IF action EQ 'DONT' THEN RETURN
          IF N_Elements(txt_new) NE N_Elements(txt_old) THEN BEGIN
                itmp = Dialog_Message(/Error, Dialog_Parent=event.top, $
                 'Number of label lines has changed. Aborted')
                RETURN
          ENDIF
          (info.objRings)->set,Title=txt_new
          Widget_Control,/HourGlass
          xplot2d_REFRESH,event.top
         END



      'EDITMARKED': BEGIN
       Widget_Control,event.id,Get_Value=val

       ;
       ; Action
       ;
       CASE val OF
         'Unmark all': BEGIN
	   IF Handle_Info(info.mk.mkIndices) THEN Handle_Free,info.mk.mkIndices
	   info.mk.mkIndices=0L
           widget_control, event.top, SET_UVALUE = info
	   END
         'Show list': BEGIN
	   IF Handle_Info(info.mk.mkIndices) EQ 0 THEN RETURN
	   handle_value,info.mk.mkIndices,list
           img = xplot2d_getimage(event.top)
           ss=size(img)
	   nlist=N_Elements(list)
           peaks2 = array_indices([ss[1],ss[2]],list,/dim)
           text= ['  Index    PixelX    PixelY      #', $
           String(Make_Set(list,Reform(peaks2[0,*]),Reform(peaks2[1,*]),LIndGen(nList)),Format='(I7,2G10.5,I7)')]
	   XDisplayFile1,text=text,Title='Marked points',Dialog_Parent=event.top
	   END
         'Add/Remove/Peaks...': BEGIN
           xplot_findpeaks,event.top,openwindow=1
	   END
         'Set properties...': BEGIN
	    tmp = info.mk
            XscrMenu,tmp,/NOTYPE,GROUP=event.top, $
               WTITLE='Preferences for Marked points',ACTION=action, $
               Dialog_Parent=event.top , $
               Titles=['Internal pointer to indices',$
                 'Symbol mark index (PSYM)',$
                 'Color Index (COLOR) '], /Interp
                  ; HELP=xplot_text('XX')
	       IF action EQ 'DONT' THEN RETURN
	       info.mk = tmp
               widget_control, event.top, SET_UVALUE = info
	     END
 	 else: 
       ENDCASE
       xplot2d_refresh,event.top
       END
;      'AXIS PROP' : xplot2d_axis_properties, event.top

      ;
      ; View menu
      ;
      'REFRESH': BEGIN
     Widget_Control,/HourGlass
         xplot2d_REFRESH,event.top
     END
      'VIEW INFO': BEGIN
     txt = info.data->info()
     txt2 = info.objRings->info()
     XDisplayFile1,text=['Info on images: ',txt,'','',$
             'Info on overplotted rings: ',txt2],Title='Info on stored images'
     END


      'VIEW SCROLL PALETTE': BEGIN
         IF Widget_Info(info.wids.Scroll,/Valid_ID) EQ 0 THEN BEGIN
           xplot2d_scroll_palette, event.top, no_block=info.no_block
           ;xplot2d_scroll_palette, event.top, wids.Scroll, no_block=info.no_block
           ;info.wids.Scroll=wids.Scroll
         ENDIF
         ;widget_control,event.top,set_uvalue=info
     END

      'CLR FULLRANGE':    xplot2d_setcolorlimits,event.top,/fullrange,/refresh
      'CLR STRETCH':      xplot2d_setcolorlimits,event.top,/stretch,/refresh
      'CLR STRETCHVALUE': xplot2d_setcolorlimits,event.top,/stretch,/changeStretch,/refresh
      'CLR MINMAX':       xplot2d_setcolorlimits,event.top,/interactive,/refresh


      'LOGLINEARSCALE':begin
         if info.str.display.ScaleType eq 'log' then $
            info.str.display.ScaleType='linear' $
         else info.str.display.ScaleType='log'
         widget_control,event.top,set_uvalue=info
         xplot2d_REFRESH,event.top
         XPLOT2D_SET_COMMENT,event.top,'Color table scale '+info.str.display.ScaleType
         END
      'XLOADCT': BEGIN
     xloadct, bottom=32,/Modal,Group=event.top
     Widget_Control,/HourGlass
         xplot2d_REFRESH,event.top
     END
      'SHOW COLOR TABLE': BEGIN
     xplot2d_displayct,event.top ; ,minrange=info.min,maxrange=info.max
     END
      'CALC XYPROFILE':begin

     XPLOT2D_REFRESH,event.top,xrange=xrange,yrange=yrange
     XPLOT2D_SET_COMMENT,event.top,'Left button toggles Horiz/Vert profile. Right button exits. '+$
       'Center button saves profile (in Exodus).'
     pData=info.data->Value([info.image_num],/pointer)
     info.cursor1=[0,0,0]
     Widget_Control,event.top,Set_UValue=info
     wExodus=info.wids.exodus
     margin = Replicate(info.str.display.margin_space,2)
     ;
     ; 
     ax=info.str.axis_properties
     IF (ax.x_origin NE 0) OR (ax.y_origin NE 0) OR $ 
        (ax.x_pixel_unit_ratio NE 1) OR (ax.y_pixel_unit_ratio NE 1) THEN BEGIN
        itmp = Dialog_Message(['Warning: ','XYprofiles does not work correctly',$
           'when the Xplot2d->Edit->Set axis properties is not ',$
           'the default value.','', $
           'However, you can still use the mouse central button',$
           'to export profiles (in pixel units) to EXODUS for comparison.',$
           '', 'Continue anywnay?'],/Question, $
           Dialog_Parent=event.top )
        IF itmp EQ 'No' THEN RETURN
     ENDIF

     profiles2,*pData,margin_space=margin,$
       scale=info.str.display.scale,wExodus=wExodus, $
       image_position=info.str.display.image_position, $
       xrange=xrange, yrange=yrange
     IF wExodus NE info.wids.exodus THEN BEGIN
       info.wids.exodus=wExodus
       Widget_Control,event.top,Set_UValue=info
     ENDIF
     XPLOT2D_SET_COMMENT,event.top,'Done profiles.'
     END

     'ID': begin
        itmp = Dialog_Message(/Info,Dialog_Parent=event.top, $
           'Xplot2D ID number is: '+StrCompress(event.top))
        return
        end

      ;
      ; ROI items
      ;

      'ROI':begin
      Widget_Control,event.id,Get_Value=wValue
      CASE wValue OF
          '2 points with mouse: center, radius': BEGIN
             xplot2d_roi_mouse,event.top,iFlag=2
	     END
          '3 points on the circle with mouse': BEGIN
             xplot2d_roi_mouse,event.top,iFlag=3
             END
          '3 points with mouse: center,a,b': BEGIN
             xplot2d_roi_mouse,event.top,iFlag=13
             END
          '4 points on the unrotated ellipse with mouse': BEGIN
             xplot2d_roi_mouse,event.top,iFlag=14
             END
          '5 points on the ellipse with mouse': BEGIN
             xplot2d_roi_mouse,event.top,iFlag=15
             END
          ;'3 points on the cake with mouse': BEGIN
          'Cake': BEGIN
             xplot2d_roi_mouse,event.top,iFlag=23
             END
          ;'Polygon points selection with mouse': BEGIN
          'Polygon': BEGIN
             xplot2d_roi_mouse,event.top,iFlag=30
             END
          ;'Rectangle 2-point selection with mouse': BEGIN
          'Rectangle': BEGIN
             xplot2d_roi_mouse,event.top,iFlag=31
             END
          'RePlot Last ROI': BEGIN
             xplot2d_roi_plot,event.top
             END
          'ROI/Mask Manager...': BEGIN
             wRoi = info.wids.roi
             IF Widget_Info(wRoi,/Valid_Id) THEN BEGIN
               ; bring to front
               tmp = xregistered('XPLOT2D_ROI_EDIT') ; case sensitive!
               xplot2d_roi_edit_set_tab_current,wRoi
             ENDIF ELSE BEGIN
               xplot2d_roi_edit,event.top,Parent=p
               xplot2d_roi_edit_set_tab_current,p
               info.wids.roi=p
               widget_control,event.top,set_uvalue=info
             ENDELSE
             END
      ENDCASE
      END

      'MASK' : XPLOT2D_APPLY_SELECTION_TO_MASK,event.top
      'CROP' : XPLOT2D_CROP,event.top
      'MASK2' : XPLOT2D_APPLY_SELECTION_TO_MASK,event.top,/complementary
      'MASKFROMIMAGE' : BEGIN
          XPLOT2D_APPLY_IMAGE_TO_MASK,event.top
          END
      'MASKZEROES' : BEGIN
          pData=info.data->Value([info.image_num],/pointer)
          img = *pData
          datasize=size(img)
          num=info.image_mask
          mask_pixels = Where(img EQ 0)
          if (mask_pixels)[0] gt -1 then begin
            ; perform calc only if region is not empty
            img_old=img
            img[mask_pixels]=-1
            *pData=img
            ;write back image data to info structure
            info.data->set,[info.image_num],value=img,aux=img_old
            widget_control,event.top,set_uvalue=info
            XPLOT2D_REFRESH,event.top
          endif else begin
            blah=dialog_message(['No masked pixels found!'],/ERROR,$
            Dialog_Parent=event.top)
          endelse
          END
      'MASKFROMPIXELS' : BEGIN
          pData=info.data->Value([info.image_num],/pointer)
          img = *pData
          datasize=size(img)
          num=info.image_mask

          tmp = {min:0.0,max:0.0,mask:-10}
          XScrMenu,tmp,Action=act,Titles=['Minimum pixel value to mask',$
             'Maximum pixel value to mask','mask value (negatif)'],  $
             /NoType,/Interp, WTitle='Mask pixel values', $
             dialog_Parent=event.top,FieldLen=40
             IF act EQ 'DONT' THEN RETURN
          mask_pixels = Where((img GT tmp.min) AND (img LE tmp.max))
          if (mask_pixels)[0] gt -1 then begin
            ; perform calc only if region is not empty
            img_old=img
            img[mask_pixels]=tmp.mask
            *pData=img
            ;write back image data to info structure
            info.data->set,[info.image_num],value=img,aux=img_old
            widget_control,event.top,set_uvalue=info
            XPLOT2D_REFRESH,event.top
          endif else begin
            blah=dialog_message(['No masked pixels found!'],/ERROR,$
            Dialog_Parent=event.top)
          endelse
      END

      'UNMASK' : BEGIN
          pData=info.data->Value([info.image_num],/pointer)
          img = *pData
          datasize=size(img)
          mask_pixels = Where((img LE 0))
          if (mask_pixels)[0] EQ -1 then begin
            blah=dialog_message(['No masked pixels found!'],/ERROR,$
            Dialog_Parent=event.top)
            return
          endif

          tmp = {mask:0D0}
          XScrMenu,tmp,Action=act,Titles=['Set masked pixels to value:'], $
             /NoType,/Interp, WTitle='UnMask pixels', $
             dialog_Parent=event.top,FieldLen=20
             IF act EQ 'DONT' THEN RETURN
          img_old = img
          img[mask_pixels]=tmp.mask
          *pData=img
          ;write back image data to info structure
          info.data->set,[info.image_num],value=img,aux=img_old
          widget_control,event.top,set_uvalue=info
          XPLOT2D_REFRESH,event.top
      END

      ;
      ; CALCULATIONS menu
      ;


      'CALC PARS':begin
    img = info.data->Value(info.image_num)
    tmp = where(img LT 0)
    ssize = Long(size(img))
    IF tmp[0] EQ -1 THEN BEGIN
      tmp = 'Masked values: None'
      data=img
      igood = Lindgen(ssize[1]*ssize[2])
    ENDIF ELSE BEGIN
      tmp='Masked values: '+StrCompress(N_Elements(tmp),/Rem)+' over '+$
                             StrCompress(N_Elements(img),/Rem)
      igood=where(img GE 0)
      data=img[igood]
    ENDELSE
    text=[tmp]

    tmp = min(data,max=tmpM)
    tmp2 = min(img,max=tmp2M)
    text = [text,$
           'min(masked img)='+StrCompress(tmp), $
           'max(masked img)='+StrCompress(tmpM), $
           'min(img)='+StrCompress(tmp2), $
           'max(img)='+StrCompress(tmp2M) ]
    
    XdisplayFile1,text=text, title='Image parameters'
    END




      'CALC ROT': BEGIN
    ; restore the widget value
    Widget_Control,event.id,Get_Value=wValue
    CASE wValue OF
      'Rotate 90 Clockwise':xplot2d_rotate_image,event.top,3
      'Rotate 90 CounterClockwise':xplot2d_rotate_image,event.top,1
      'Rotate 180':xplot2d_rotate_image,event.top,2
      'Mirror up/down':xplot2d_rotate_image,event.top,7
      'Mirror left/right':xplot2d_rotate_image,event.top,5
      else: Message,'CASE action not found: '+wValue
    ENDCASE
    END

      'CALC INTEGRATE01': BEGIN
          xplot2d_azimuth_integral,event.top, method = 0, window=1
          END
      'CALC INTEGRATE02': BEGIN
            xplot2d_azimuth_integral,event.top, method = 0, window=0
            if Fix((info.str.pref.overplotDiffFlag)[0]) EQ 1  then begin
            xplot2d_refresh,event.top
          endif
          END
;      'CALC INTEGRATE1': xplot2d_azimuth_integral,event.top, method = 1

      'CALC INTEGRATE_MACRO': BEGIN
        ;CASE !version.os OF
        ;   'sunos': bindir='sun'
        ;   else:  bindir=!version.os
        ;ENDCASE
        Widget_Control,/HourGlass
        path = './'

        det = info.str.detector
        dist = norm( [det.x0-det.xS,det.y0-det.yS,det.zS]) 
        dist = dist*det.pixelsize*1d-4 ; dist in cm

        title=info.data->Value(info.image_num,/title)
        txt = [$
        "; ",$
        "; xop macro for azimuthal integration --", $
        "; ",$
        "spawn,'rm output.edf'", $
        "spawn,'"+path+"saxs_angle input.edf output.edf 1 1 1 "+StrCompress(det.x0,/Rem)+" "+StrCompress(det.y0,/Rem)+" -1'", $
        "img = read_edf('output.edf')", $
        "tmp = where(img LT 0) ; masked indices", $
        " ",$
        "IF tmp[0] NE -1 THEN BEGIN ; there are masked values ",$
        "  img2 = img*0 ",$
        "  img2[ where(img GE 0) ] = 1  ",$
        "  ht = total(img2,2) ",$
        "  img[tmp] = 0 ",$
        "ENDIF ELSE BEGIN ; not masked values ", $
        "  ht=total(img*0+1,2) ", $
        "ENDELSE", $
        " ", $
        "h = total(img,2)/ht", $
        "hx = findgen(n_elements(h))", $
        "; xplot2d,img ; display polar image", $
;       "xplot,hx,h,/no_block,Window_Size=[600,400]", $
        "  ",$
        "; ", $
        "; express profile in twotheta and angle units  ", $
        "; ", $
        " ", $
        "dist = "+StrCompress(dist), $
        "pixelsize = "+StrCompress(det.pixelsize), $
        "e1 = "+StrCompress(det.energy), $
        "twotheta = 180./!pi*atan(hx*pixelsize*1e-4/dist)", $
        "dspacing = physical_constants('hc')/2.0/e1/sin(twotheta/2.*!pi/180.)", $
        "title = '"+title+"'", $
        " ", $
        "tmp = make_set(hx,twotheta ,dspacing,h)", $
        " ", $
        "xplot,tmp,/No_Block,Window_Size=[600,400],xcol=1,coltitles=['pixel','TwoTheta[deg]','dspacing','intensity'],Title=title"]

      img = info.data->Value(info.image_num)
      write_edf,img,File='input.edf'
      xop_macro,txt

      END

      'CALC XY2RPHI_AXIAL':begin
         pData=info.data->Value([info.image_num],/pointer)
         imageTitle = info.data->value([info.image_num],/title)
         img = *pData
         det  = info.str.detector
        
         
         str = info.str.xy2rphi_axial
         XScrMenu,str,Action=act,Titles=info.str.xy2rphi_axial_titles, $
            /NoType,/Interp, $
            Flags=info.str.xy2rphi_axial_flags, NCOLUMN=2,  $
            WTitle='(x,y)->(twoTheta,phi)',dialog_Parent=event.top,$
            FieldLen=40
         IF act EQ 'DONT' THEN RETURN
         info.str.xy2rphi_axial = str
         Widget_Control,event.top,Set_UValue=info
         Widget_Control,/HourGlass
         out = xy2rphi_axial(img,$
            [det.x0,det.y0], $
            [det.xS,det.yS,det.zS], $
            useEllipses=Fix((det.useEllipses)[0]),/verbose, $
            twoThetaRange=[str.twoThetaMin,str.twoThetaMax], $
            twoThetaStep=str.twoThetaStep, $
            phiRange=[str.phiMin,str.phiMax], $
            phiStep=str.phiStep, $
            axis=axis)  

         wtitle='(2Theta,phi) using v0:'+vect2String($
            [det.x0,det.y0])+' vS:'+vect2String($
            [det.xS,det.yS,det.zS])

         IF N_Elements(out) GT 1 THEN $
          xplot2d,out,scale=1.0,axis=axis,wtitle=wtitle,cursor=2,/stretch, $
          imageTitle='(x,y)->(twoTheta,phi) image from: '+imageTitle
         END

      'CALC PROFILE':begin
         xplot2d_profile,event.top
         return
         END

      'CALC HISTOGRAM':begin
         img = info.data->Value(info.image_num)
         tmp = where(img LT 0)
         IF tmp[0] EQ -1 THEN BEGIN
           Print,'Masked values: None'
           data=histogram(img)
         ENDIF ELSE BEGIN
           Print,'Masked values: ',N_Elements(tmp),' over',N_Elements(img)
           tmp=img[where(img GE 0)]
           data=histogram(tmp)
         ENDELSE
         xplot,lindgen(n_elements(data)),data,/No_Block,Window_Size=[600,400], $
            xrange=[min(data),max(where(data GT 4))]
         END


      'CALC CDF(HISTOGRAM)':begin
         img = info.data->Value(info.image_num)
         tmp = where(img LT 0)
         IF tmp[0] EQ -1 THEN BEGIN
           Print,'Masked values: None'
           data=hist_equal(img,/hist)
         ENDIF ELSE BEGIN
           Print,'Masked values: ',N_Elements(tmp),' over',N_Elements(img)
           tmp=img[where(img GE 0)]
           data=hist_equal(tmp,/hist)
         ENDELSE
         xplot,lindgen(n_elements(data)),data,/No_Block,Window_Size=[600,400]
         END

      'CALC INTV':BEGIN
         img = info.data->Value(info.image_num)
         tmp = where(img LT 0)
         IF tmp[0] EQ -1 THEN BEGIN
           Print,'Masked values: None'
         ENDIF ELSE BEGIN
           Print,'Masked values: ',N_Elements(tmp),' over',N_Elements(img)
           img[tmp]=0
         ENDELSE
         intv = total(img,2)
         xx = lindgen(n_elements(intv))
         str = info.str.axis_properties
         xx2 = str.x_origin+(xx)/str.x_pixel_unit_ratio
         ; todo: this works better, but not legal!!
         ; xx2 = str.x_origin+(xx+1)/str.x_pixel_unit_ratio
         xplot,Make_Set(xx,xx+0.5,xx2,intv,intv/N_elements(img[0,*]) ),/No_Block, $
         Window_Size=[600,400], XCOL=3,  $
          colTitles=['Pixel number','Pixel center','X axis','Intensity','Normalized Int']
         END

      'CALC INTH':BEGIN
         img = info.data->Value(info.image_num)
         tmp = where(img LT 0)
         IF tmp[0] EQ -1 THEN BEGIN
           Print,'Masked values: None'
         ENDIF ELSE BEGIN
           Print,'Masked values: ',N_Elements(tmp),' over',N_Elements(img)
           img[tmp]=0
         ENDELSE
         inth = total(img,1)
         yy = lindgen(n_elements(inth))
         str = info.str.axis_properties
         yy2 = str.y_origin+(yy)/str.y_pixel_unit_ratio
         xplot,Make_Set(yy,yy+0.5,yy2,inth,inth/N_Elements(img[*,0])),/No_Block,$
         Window_Size=[600,400], XCOL=3, $
          colTitles=['Pixel number','Pixel center','Y axis','Intensity','Normalized Int']
         END

      'CALC CENTEROFMASS':begin
          xplot2d_annulus,event.handler,roi_indices=roi,img=img, /extractROI

               IF roi[0] NE -1 THEN BEGIN
                 itmp = Dialog_Message(/Question,/Cancel,$
                   ['An annulus is defined.',$
                    'Do you want to calculate the center of mass using',$
                    'only the pixels inside the annulus? ','', $
                    'Yes=Use annulus','No=Use full image'], $
                   Dialog_Parent=event.top)
                 IF itmp EQ 'Cancel' THEN RETURN
                 IF itmp EQ 'Yes' THEN use_roi=roi
               ENDIF ELSE BEGIN
               ENDELSE
               cm = xplot2d_calc_cm(event.handler,roi=use_roi)
               itmp = Dialog_Message(/Question,['The center of mass ', $
                   'is at pixels: ',vect2string(cm[0:1]),'', $
                   'Do you want to set this values to beam [x,y] center and',$
                   '[x,y] of the sample? '], Dialog_Parent=event.top)
               IF itmp NE 'Yes' THEN RETURN
               det=info.str.detector
               det.x0=cm[0]
               det.y0=cm[1]
               det.xS=cm[0]
               det.yS=cm[1]
               info.str.detector=det
               Widget_Control,event.handler,Set_UValue=info
               Xplot2d_Refresh,event.handler
               RETURN
    END
;      'CALC GENERIC':begin
;    pData=info.data->Value([info.image_num],/pointer)
;    img = *pData
;    str = info.str.calc_generic_operation
;    XScrMenu,str,Action=act,Titles=info.str.calc_generic_operation_titles, $
;      /NoType,/Interp, $
;      WTitle='Generic image operation',dialog_Parent=event.top,FieldLen=40
;    IF act EQ 'DONT' THEN RETURN
;    itmp = Execute(str.oper)
;    IF itmp NE 1 THEN BEGIN
;      jtmp = dialog_Message(/Error,['Operation not understood: ',str.oper],$
;       dialog_Parent=event.top)
;    ENDIF
;    info.str.calc_generic_operation = str
;    Widget_Control,event.handler,Set_UValue=info
;    CASE Fix(str.res[0]) OF
;      0: xplot2d,img
;      1: XPLOT2D_ADD_IMAGE,event.top,img
;      2: BEGIN
;         ;info.data->set,[info.image_num],value=img,aux=*pData 
;         ;xplot2d_refresh,event.top
;         XPLOT2D_ADD_IMAGE,event.top,img,/Substitute
;         END
;      else:
;    ENDCASE
;    END

    'CALC XY2RPHI': BEGIN
      pData=info.data->Value([info.image_num],/pointer)
      imageTitle=info.data->Value([info.image_num],/title)
      img = *pData
      str = info.str.xy2rphi
      str.x0 = info.str.detector.x0
      str.y0 = info.str.detector.y0
      XScrMenu,str,Action=act,Titles=info.str.xy2rphi_titles, /NoType,/Interp, $
            Flags=info.str.xy2rphi_flags, NCOLUMN=3,  $
            WTitle='(x,y)->(r,phi)',dialog_Parent=event.top,FieldLen=40
      IF act EQ 'DONT' THEN RETURN
      info.str.xy2rphi = str
      widget_control,event.top,set_uvalue=info
      CASE Fix((str.pars)[0]) OF
      ;0: BEGIN
      ;   out = xy2rphi(img,/verbose, axis=axis,  $
      ;   [info.str.detector.x0,info.str.detector.y0])
      ;   wtitle='(r,phi) using center '+vect2String($
      ;   [info.str.detector.x0,info.str.detector.y0])
      ;   END
      0: BEGIN
         out = xy2rphi(img, axis=axis, /verbose,[str.x0,str.y0],$
          rrange=[str.rmin,str.rmax],nr=str.nr, $
          phirange=[str.phimin,str.phimax],nphi=str.nphi) 
         wtitle='(r,phi) using center '+vect2String([str.x0,str.y0])
         END
      1: BEGIN
         out = xy2rphi( axis=axis, /verbose)
         wtitle='(r,phi) for test image'
         END
      else:
      ENDCASE
      xplot2d,out,scale=1.0,axis=axis,wtitle=wtitle,cursor=2,/stretch, $
          imageTitle='(x,y)->(r,phi) image from: '+imageTitle
    END

      ;
      ; Simulations menu
      ;

      'SIMUL_FROMPROFILE':BEGIN
    pData=info.data->Value([info.image_num],/pointer)
    imageTitle=info.data->Value([info.image_num],/title)
    img = *pData
    ssize=size(img)

    det = info.str.detector
    dist = norm( [det.x0-det.xS,det.y0-det.yS,det.zS]) 
    dist = dist*det.pixelsize*1d-4

    str = info.str.simul_fromprofile
    str.detector_dist=dist
    str.pixelsize=det.pixelsize
    str.xsize=ssize[1]
    str.ysize=ssize[2]
    str.e1=det.energy
    str.x=det.x0
    str.y=det.y0

    titles=info.str.simul_fromprofile_titles

    XScrMenu,str,Action=act,Titles=titles, /NoType,/Interp, $
         WTitle='Simulate image',dialog_Parent=event.top,FieldLen=40, NCol=2

    IF act EQ 'DONT' THEN RETURN

    IF StrCompress(str.file_profile,/Rem) EQ '?' OR checkFile(str.file_profile) EQ 0 THEN BEGIN
      file = Dialog_Pickfile(Title='Get XRD prfile',Dialog_Parent=event.top)
      IF file EQ '' THEN RETURN
      str.file_profile=file
    ENDIF

    info.str.simul_fromprofile=str
    widget_control,event.top,set_uvalue=info

    Widget_Control,/HourGlass
        img = xplot2d_prof2image(str.file_profile,Plot=0,  $
        Dummy_Pixels=str.dummy_pixels, $
        size=[str.xsize,str.ysize], $
        center=[str.x,str.y], $
        Dist=str.detector_dist, pixelSize=str.pixelsize, $
        e0=str.e0, e1=str.e1)

    CASE Fix( (str.output)[0]) OF
     0: XPLOT2D_ADD_IMAGE,event.top,img,title='Simulated image from: '+str.file_profile
     1: XPLOT2D,img,scale=info.str.display.scale,imageTitle='Simulated image from: '+str.file_profile
    ENDCASE
    END


      'SIMUL_DSPACING':BEGIN
    pData=info.data->Value([info.image_num],/pointer)
    imageTitle=info.data->Value([info.image_num],/title)
    img = *pData
    ssize=size(img)

   simul_dspacing=info.str.simul_dspacing
   simul_dspacing.xsize=ssize[1]
   simul_dspacing.ysize=ssize[2]
   simul_dspacing.energy=info.str.detector.energy
   simul_dspacing.xc=info.str.detector.x0
   simul_dspacing.yc=info.str.detector.y0
   simul_dspacing.xs=info.str.detector.xS
   simul_dspacing.ys=info.str.detector.yS
   simul_dspacing.zs=info.str.detector.zS

    XScrMenu,simul_dspacing,Action=act,Titles=info.str.simul_dspacing_titles, /NoType,/Interp, $
         WTitle='Simulate image',dialog_Parent=event.top,FieldLen=40, NCol=2
    IF act EQ 'DONT' THEN RETURN
    Widget_Control,/HourGlass
    info.str.simul_dspacing=simul_dspacing

    size = [simul_dspacing.xsize,simul_dspacing.ysize]
    center = [simul_dspacing.xc,simul_dspacing.yc]
    sample = [simul_dspacing.xs,simul_dspacing.ys,$
              simul_dspacing.zs]
        file = simul_dspacing.file
        img = xplot2d_simul_dspacing(file,$
           size=size, center=center, sample=sample, $
           energy=simul_dspacing.energy, $
           width=simul_dspacing.width,group=event.top,error=error)

    IF error THEN RETURN
    CASE Fix( (simul_dspacing.output)[0]) OF
     0: XPLOT2D_ADD_IMAGE,event.top,img,title='Simulated image from: '+file
     1: XPLOT2D,img,scale=info.str.display.scale,imageTitle='Simulated image from: '+file
    ENDCASE
    END


;      'XRD_DATABASE':BEGIN
;        h = dabax_access('PDFrruff.dat')
;        IF type(h) NE 8 THEN BEGIN
;           itmp = Dialog_Message(/Error,dialog_parent=event.top, $
;             ['File not found in DABAX database: PDFrruff.dat', '',$
;              'You may download it from ',$
;              '    http://ftp.esrf.eu/pub/scisoft/xop2.3/DabaxFiles/',$
;              '','and install it in:',$
;                 getenv('XOP_HOME')+sdep(/ds)+'data'+sdep(/ds)+'dabax'+sdep(/ds)])
;           RETURN
;        ENDIF ELSE BEGIN
;           itmp = Dialog_Message(/Question,dialog_parent=event.top, $
;             ['This option will load the Mineral database ',$
;              'from the DABAX file PDFrruff.dat which contains',$
;              'data from http://rruff.info',$
;              ' ','The database is not yet integrated in xplot2d',$
;              'therefore, you must: ',$
;              '  1- localize the phase you need (use "Scan sel..." button in xplot) ',$
;              '  2- Save the data (dSpacing Intensity) to a file (xplot->File->Export ASCII...)',$
;              '  3- Load this file from Xplot2d->Diffraction->Overplot Rings->Read dSpacing...',$
;              ' ','Continue? '])
;            IF itmp EQ 'No' THEN RETURN
;        ENDELSE
;        widget_control,/hourglass
;        xplot,parent=p,spec=h,wtitle='PDFrruff (http://rruff.info)', $
;         xtitle='-1',ytitle='-1',title='#S',yCol=2
;
;        UserSym,[0.,0.],[-1e9,0]
;        xplot_controls_action,p,PSymbol=8,PSYMSign='1',Clr_Lines=2
;
;
;    END

      'XRD_DATABASE2':BEGIN


          event_name = tag_names(event, /structure_name)
          if event_name eq 'NEW_SCAN_EVENT' then begin
                   widget_control,/HOURGLASS
            widget_control,info.wids.pdfrruff,Get_UValue=inforruff
            handle_value,inforruff.handle,hspec
            data = spec_data(hspec,event.newscan)
            mineral=spec_name(hspec,event.newscan)
            IF N_Elements(data) GT 1 THEN BEGIN
                info.ringsFlag = 1
                info.objRings->remove
                tmp = data[0:2,*]
                tmp[2,*]=2
                info.objRings->set,0,Title=mineral,Flag=0, $
                   Value=tmp,/Clean 
                ;info.objRings->set,Title='phase with color '+StrCompress(phases[i],/Rem),Flag=0, $
                ;   Value=peaks[*,igood],/Add
                widget_control,event.top,Set_UValue=info
                xplot2d_refresh,event.top
            ENDIF
          endif else begin
            h = dabax_access('PDFrruff.dat')
            IF type(h) NE 8 THEN BEGIN
               itmp = Dialog_Message(/Error,dialog_parent=event.top, $
                 ['File not found in DABAX database: PDFrruff.dat', '',$
                  'You may download it from ',$
                  '    http://ftp.esrf.eu/pub/scisoft/xop2.3/DabaxFiles/',$
                  '','and install it in:',$
                     getenv('XOP_HOME')+sdep(/ds)+'data'+sdep(/ds)+'dabax'+sdep(/ds)])
               RETURN
            ENDIF 
            widget_control,/hourglass
            hh = handle_create(value=h)
            xspecscan, hh, caller=event.id, tlb_id=tmp, mca=0, /not_all,/no_block, $
               group_leader=event.top, title='DABAX:PDFrruff.dat (http://rruff.info)'
            info.wids.pdfrruff=tmp
            widget_control,event.top,Set_UValue=info
          endelse
    END

      'SIMUL D':BEGIN
          ;
          ; get/modify input data
          ;
      tools_profile={overplotDSpacing:[StrCompress(info.overplotDSpacing,/Rem),'No','Yes'], $
            file_dspacing:info.file_dspacing, detector_dist:info.str.detector_dist,  $
            pixelsize:info.pixelsize,  e1:info.e1, $
        x:(info.center)[0], y:(info.center)[1]}


      titles=['Overplot rings of known d', 'File (? for browser)','Distance sample-image [cm]','pixelsize [microns]', $
             'Energy of photon beam [eV]', $
             'Center in X [pixel]', 'Center in Y [pixel]']

          flags=Replicate('w(0) EQ 1',N_Elements(titles))  & flags[0]='1'

      XScrMenu,tools_profile,Action=act,Titles=titles, /NoType,/Interp, Flags=flags, $
         WTitle='Overplot rings',dialog_Parent=event.top,FieldLen=40

      IF act EQ 'DONT' THEN RETURN

          info.overplotDspacing=Fix( (tools_profile.overplotDspacing)[0])
          info.detector_dist=tools_profile.detector_dist
          info.pixelsize=tools_profile.pixelsize
          info.e1=tools_profile.e1
          (info.center)=[tools_profile.x,tools_profile.y]

      IF info.overplotDspacing EQ 1 THEN BEGIN
        IF StrCompress(tools_profile.file_Dspacing,/Rem) EQ '?' OR $
                  CheckFile(tools_profile.file_Dspacing) EQ 0 THEN BEGIN
         file = Dialog_Pickfile(Title='Get file with D-spacing Intensity')
         IF file EQ '' THEN RETURN
         tools_profile.file_dspacing=file
        ENDIF
            info.file_dspacing=tools_profile.file_dspacing
      ENDIF

          widget_control,event.top,set_uvalue=info
          ;
          ; refresh
          ;
      xplot2d_REFRESH,event.top
    END

      'SIMUL SCAN':BEGIN
         ;simul_scan = {stop:['0','No','Yes'], var:['0','detector_distance','center_x','center_y','e0','e1'], $
         ;  min:0.0,max:0.0,npoints:20L, ......}
         simul_scan = info.simul_scan
         print,'Current values ---------------------------------'
         print,'detector_distance: ',info.str.detector_dist
         print,'center_x: ',(info.center)[0]
         print,'center_y: ',(info.center)[1]
         print,'e0: ',info.e0
         print,'e1: ',info.e1
         print,'------------------------------------------------'
         XScrMenu,simul_scan,Action=act,Titles=['stop at each step','variable','min','max','number of points', $
         'Calculate azim. integ. at each step (very slow)'],  $
           /NoType,/Interp, Flags=flags, WTitle='Scan parameter', $
           dialog_Parent=event.top,FieldLen=40

         IF act EQ 'DONT' THEN RETURN

         info.simul_scan = simul_scan
         Widget_Control,event.top,Set_UValue=info

         CASE Fix((simul_scan.var)[0]) OF
           0: var_old = info.str.detector_dist
           1: var_old = (info.center)[0]
           2: var_old = (info.center)[1]
           3: var_old = info.e0
           4: var_old = info.e1
         ENDCASE
         istop = Fix( (simul_scan.stop)[0])
         FOR i=0,simul_scan.npoints-1 DO BEGIN
           var = simul_scan.min+(simul_scan.max-simul_scan.min)/(simul_scan.npoints-1)*i
           print,'Using variable: ',var

           CASE Fix((simul_scan.var)[0]) OF
             0: info.str.detector_dist = var
             1: (info.center) = [var,(info.center)[1]]
             2: (info.center) = [ (info.center)[0], var]
             3: info.e0 = var
             4: info.e1 = var
           ENDCASE
           Widget_Control,event.top,Set_UValue=info

;           IF Fix(simul_scan.azimuthIntegral[0]) EQ 1 THEN $
;             xplot2d_azimuth_integral,event.top, method = 0, window=0
           xplot2d_REFRESH,event.top
           IF istop THEN BEGIN
             itmp = Dialog_Message(/Question,'Using variable:'+String(var)+'    Continue?')
             IF itmp EQ 'No' THEN GoTo,salimos
           ENDIF
         ENDFOR

         salimos:

         itmp = Dialog_Message(/Question,'Keep last value of the scan? ('+String(var)+')')
         IF itmp EQ 'No' THEN BEGIN
           CASE Fix((simul_scan.var)[0]) OF
             0: info.str.detector_dist = var_old
             1: (info.center) = [ var_old, (info.center)[1] ]
             2: (info.center) = [ (info.center)[0], var_old]
             3: info.e0 = var_old
             4: info.e1 = var_old
           ENDCASE
           Widget_Control,event.top,Set_UValue=info
;           IF Fix(simul_scan.azimuthIntegral[0]) EQ 1 THEN $
;             xplot2d_azimuth_integral,event.top, method = 0, window=0
           xplot2d_REFRESH,event.top
         ENDIF


    END
      ;
      ; Tools menu
      ;
      'TOOLS NEW WINDOW':BEGIN
    pData=info.data->Value([info.image_num],/pointer)
    imageTitle=info.data->Value([info.image_num],/title)
    img = *pData
    xplot2d,img,scale=info.str.display.scale,minColor=info.min,maxColor=info.max,imageTitle=imageTitle
    END
      'TOOLS XPLOT':BEGIN
    Xplot,no_block=info.no_block
    END
      ;
      ; Help menu
      ;
      'Help': XHelp,'xplot2d'
      'HelpWWW': online_help,book='xplot2d.htm'
      else: Print,'Action not understood:',action
   ENDCASE
END


;
;===============================================================================
;

;+
;
; PRO XPLOT2D,indata, parent=wXplot2d, $
;     wtitle=wtitle,no_block=no_block, $
;     group=group,window_size=window_size,$
;     scale=scale,mincolor=mincolor,maxcolor=maxcolor,$
;     imagetitle=imagetitle,axis=axis, $
;     cursor=cursor1, $ ; 0:pixel, 1:twoTheta dspc, 2 XYaxes 
;     stretch=stretch   ; color stretch
;
;-
PRO XPLOT2D,indata, parent=wXplot2d, $
    wtitle=wtitle,no_block=no_block, $
    group=group,window_size=window_size,$
    scale=scale,mincolor=mincolor,maxcolor=maxcolor,$
    imagetitle=imagetitle,axis=axis, $
    cursor=cursor1, $ ; 0:pixel, 1:twoTheta dspc, 2 XYaxes 
    stretch=stretch   ; color stretch



Catch, error_status
if error_status ne 0 then begin
      message,/info,'error caught: '+!error_state.msg
      if sdep(/w) then itmp = dialog_Message(/Error,$
      'XPLOT2D: error caught: '+!error_state.msg)
      catch, /cancel
      on_error,2
endif

if keyword_set(group) then widget_control,/hourglass
;
; use non-decomposed colors to be able to change colormap (...)
;
Device,Decomposed=0,Retain=2
tek_color


;
; Input vales
;

str = xplot2d_defaults()

; by default : no_block
IF N_Elements(no_block) EQ 0 THEN no_block=1
IF N_Elements(scale) NE 0 THEN str.display.scale=scale
IF N_Elements(mincolor) EQ 0 THEN mincolor=0.0
IF N_Elements(maxcolor) EQ 0 THEN maxcolor=0.0
IF N_Elements(cursor1) NE 0 THEN BEGIN
   typ = str.cursor.type
   typ[0] = StrCompress(cursor1,/Rem)
   str.cursor.type=typ
ENDIF


IF N_Elements(axis) EQ 4 THEN BEGIN
  str.axis_properties.x_origin=axis[0]
  str.axis_properties.x_pixel_unit_ratio=axis[1]
  str.axis_properties.y_origin=axis[2]
  str.axis_properties.y_pixel_unit_ratio=axis[3]
ENDIF
;
;The ompp object which will keep all data
;
CASE type(indata) OF
     0: BEGIN   ; no data input, create junk data
         ;indata = Dist(2000)
         indata = image_illusions(0,size=2000,title=title)
         dataptr=ptr_new(indata,/no_copy)
         objData=Obj_New('ompp',dataptr,title='Test image: '+title)
       END
     7:BEGIN  ; input is a file name (string)
        indata=read_edf(indata)
        dataptr=ptr_new(indata,/no_copy)
        objData=Obj_New('ompp',dataptr,title=imagetitle)
       END
     else: BEGIN  ; input is a variable
        dataptr=ptr_new(indata) ; copy to avoid destroying input
        objData=Obj_New('ompp',dataptr,title=imagetitle)
      END
ENDCASE


; define ompp object for known phases

dataptr=ptr_new([0,0])
objRings =Obj_New('ompp',dataptr,title='')


;
; Create widgets
;

;main window
wXplot2d=Widget_base(title='',MBAR=wMenuBar,/column, $
   /TLB_KILL_REQUEST_EVENTS, Event_Pro='Xplot2D_Quit')

if n_elements(Xplot2dWinList) eq 0 then Xplot2dWinList=[wXplot2d] $
   else Xplot2dWinList=[Xplot2dWinList,wXplot2d]

;Buttons
wButtons=widget_base(wXplot2d,/Row)
   wImageNum=cw_field(wButtons,value=1,title='Image #', $
       /integer,/return_events,xsize=4)
   wtmpB = Widget_Base(wButtons,/Row)
     wtmp = Widget_Button(wTmpB,Value='|<', UValue='FIRST IMAGE',ToolTip='Go to first loaded image')
     wtmp = Widget_Button(wTmpB,Value='<',  UValue='PREVIOUS IMAGE',ToolTip='Go to previous loaded image')
     wtmp = Widget_Button(wTmpB,Value='x',  UValue='DELETE IMAGE',ToolTip='Delete current image')
     wtmp = Widget_Button(wTmpB,Value='>',  UValue='NEXT IMAGE',ToolTip='Go to next loaded image')
     wtmp = Widget_Button(wTmpB,Value='>|', UValue='LAST IMAGE',ToolTip='Go to last loaded image')
     wtmp = Widget_Button(wTmpB,Value='c f',UValue='CLR FULLRANGE',ToolTip='Color table: full range')
     wtmp = Widget_Button(wTmpB,Value='c s',UValue='CLR STRETCH',ToolTip='Color table: stretch')
     wtmp = Widget_Button(wTmpB,Value='c l',UValue='LOGLINEARSCALE',ToolTip='Color table: toggle lin/log')
     wtmp = Widget_Button(wTmpB,Value='+',  UValue='LOAD INCREMENTED IMAGE',ToolTip='Load file with incremented image')
     wtmp = Widget_Button(wTmpB,Value='-',  UValue='LOAD DECREMENTED IMAGE',ToolTip='Load file with decremented image')

   wtmp = Widget_Button(wTmpB,Value='XYZ:',  UValue='CURSORCHANGE',ToolTip='Cursor: change units')
   wCursor = Widget_Text(wButtons, Value='', xsize=30)
   wScale = CW_Field(wButtons, Title='Scl:', Value=str.display.scale, $
       xsize=4,/floating,/Return_Events)

;graph window,
dataSize=size(indata)
wGraph=widget_draw(wXplot2d,xsize=dataSize(1),ysize=dataSize(2),$
   /button_events,/motion_events, event_pro='XPLOT2D_EVENT',$
   RETAIN=2)

;Comment window, to give instructions
wComment=widget_text(wXplot2d,xsize=60,ysize=2)

;menu
wFileMenu =  WIDGET_BUTTON(wMenuBar, VALUE='File', /MENU)
   tmpIndex = 1+fix( (str.imageFormat.format)[0] )
   wLoadImage=WIDGET_BUTTON(wFileMenu, VALUE='Load Image...',UVALUE='LOAD IMAGE')
   wtmp=WIDGET_BUTTON(wFileMenu, VALUE='ReLoad Last Image',UVALUE='RELOAD IMAGE')
   wtmp=WIDGET_BUTTON(wFileMenu, VALUE='Load Incremented Image [+]',UVALUE='LOAD INCREMENTED IMAGE')
   wtmp=WIDGET_BUTTON(wFileMenu, VALUE='Load Decremented Image [-]',UVALUE='LOAD DECREMENTED IMAGE')

   wtmp=WIDGET_BUTTON(wFileMenu, VALUE='Set load image properties...',UVALUE='EDIT PREF')
   wTmpBase=WIDGET_BUTTON(wFileMenu, VALUE='Save Image',/Menu,/Sepa)
      wtmp=WIDGET_BUTTON(wTmpBase, VALUE='EDF...',UVALUE='SAVE IMAGE')
      wtmp=WIDGET_BUTTON(wTmpBase, VALUE='XRD...',UVALUE='SAVE IMAGE')
      wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Other...',UVALUE='SAVE IMAGE')

   wtmp = widget_button(wFileMenu,Value='Load xplot2d input file',UValue='XPLOT2D_LOAD',/SEPA)
   wtmp = widget_button(wFileMenu,Value='Save xplot2d input file',UValue='XPLOT2D_SAVE')


   wtmp=WIDGET_BUTTON(wFileMenu, VALUE='Quit', UVALUE='EXIT',/Sepa)





wEditMenu =  WIDGET_BUTTON(wMenuBar, VALUE='Edit', /MENU)
   wtmp=WIDGET_BUTTON(wEditMenu, VALUE='Undo last image modification',UVALUE='UNDO')
   wtmp=WIDGET_BUTTON(wEditMenu, VALUE='Copy...', UVALUE='COPYPASTE',/Sepa)
   wtmp=WIDGET_BUTTON(wEditMenu, VALUE='Paste', UVALUE='COPYPASTE')
   wtmp=WIDGET_BUTTON(wEditMenu, VALUE='Paste special...', UVALUE='COPYPASTE')
   wtmp=WIDGET_BUTTON(wEditMenu, VALUE='Delete current image [x]',UVALUE='DELETE IMAGE',/Sepa)
   wtmp=WIDGET_BUTTON(wEditMenu, VALUE='Delete ALL images',UVALUE='DELETE ALL')
   wtmp=WIDGET_BUTTON(wEditMenu, VALUE='Edit image titles',UVALUE='EDIT IMAGE TITLES',/SEPA)
   wtmp=WIDGET_BUTTON(wEditMenu, VALUE='Edit ring titles',UVALUE='EDIT RING TITLES')
;   wtmp=WIDGET_BUTTON(wEditMenu, VALUE='Set axis properties...',UVALUE='EDIT PREF')
   wMarkedPoints=WIDGET_BUTTON(wEditMenu, VALUE='Marked points', /SEPA,/MENU)
      wTmp=WIDGET_BUTTON(wMarkedPoints, VALUE='Unmark all', UVALUE='EDITMARKED')
      wTmp=WIDGET_BUTTON(wMarkedPoints, VALUE='Show list', UVALUE='EDITMARKED')
      wTmp=WIDGET_BUTTON(wMarkedPoints, VALUE='Add/Remove/Peaks...', UVALUE='EDITMARKED')
      wTmp=WIDGET_BUTTON(wMarkedPoints, VALUE='Set properties...', UVALUE='EDITMARKED')
   wtmp0=WIDGET_BUTTON(wEditMenu, VALUE='Preferences',/Menu,/Sepa)
   wtmp=WIDGET_BUTTON(wtmp0, VALUE='Set load image properties...',UVALUE='EDIT PREF')
   wtmp=WIDGET_BUTTON(wtmp0, VALUE='Set annulus width...',UVALUE='EDIT PREF')
   wtmp=WIDGET_BUTTON(wtmp0, VALUE='Set axis properties...',UVALUE='EDIT PREF')
   wtmp=WIDGET_BUTTON(wtmp0, VALUE='Azimuthal Integration preferences...',UVALUE='EDIT PREF',/Sepa)
   wtmp=WIDGET_BUTTON(wtmp0, VALUE='Define detector and beam pars...',UVALUE='EDIT PREF')

wViewMenu =  WIDGET_BUTTON(wMenuBar, VALUE='View', /MENU)
   wtmp=WIDGET_BUTTON(wViewMenu, VALUE='Refresh',UVALUE='REFRESH')
   wtmp=WIDGET_BUTTON(wViewMenu, VALUE='Scale on images',UVALUE='VIEW SCALE',/SEPA)
   wtmp=WIDGET_BUTTON(wViewMenu, VALUE='Scroll Palette',UVALUE='VIEW SCROLL PALETTE')
   wtmp=WIDGET_BUTTON(wViewMenu, VALUE='Display Color Table',UVALUE='SHOW COLOR TABLE',/SEPA)
   wtmp=WIDGET_BUTTON(wViewMenu, VALUE='Change Color Table',UVALUE='XLOADCT')
   wcolorlimits=WIDGET_BUTTON(wViewMenu, VALUE='Intensity <->Color Table',/Menu)
   wtmp=WIDGET_BUTTON(wColorLimits, VALUE='Full range [cf]',UVALUE='CLR FULLRANGE')
   wtmp=WIDGET_BUTTON(wColorLimits, VALUE='Stretch min&max intensity [cs]',UVALUE='CLR STRETCH')
   wtmp=WIDGET_BUTTON(wColorLimits, VALUE='Change stretch value...',UVALUE='CLR STRETCHVALUE')
   wtmp=WIDGET_BUTTON(wColorLimits, VALUE='Set min&max intensity...',UVALUE='CLR MINMAX',/SEPA)
   wtmp=WIDGET_BUTTON(wViewMenu, VALUE='Toggle Log/Linear Scale',UVALUE='LOGLINEARSCALE',/SEPA)
   wtmp=WIDGET_BUTTON(wViewMenu, VALUE='Info on images',UVALUE='VIEW INFO',/SEPA)
   wtmp=WIDGET_BUTTON(wViewMenu, VALUE='Image parameters',UVALUE='CALC PARS')
   wtmp=WIDGET_BUTTON(wViewMenu, VALUE='Xplot2D id number',UVALUE='ID',/SEPA)


wROIMenu =  WIDGET_BUTTON(wMenuBar, VALUE='ROI/Mask', /MENU)
   wtmp=WIDGET_BUTTON(wROIMenu, VALUE='ROI/Mask Manager...',UVALUE='ROI')
   wtmp=WIDGET_BUTTON(wROIMenu, VALUE='RePlot Last ROI',UVALUE='ROI')
   wtmpBase=WIDGET_BUTTON(wROIMenu, VALUE='Circle',/menu,/SEPA)
     wtmp=WIDGET_BUTTON(wTmpBase, VALUE='2 points with mouse: center, radius',UVALUE='ROI')
     wtmp=WIDGET_BUTTON(wTmpBase, VALUE='3 points on the circle with mouse',UVALUE='ROI')
   wtmpBase=WIDGET_BUTTON(wROIMenu, VALUE='Ellipse',/menu)
     wtmp=WIDGET_BUTTON(wTmpBase, VALUE='3 points with mouse: center,a,b',UVALUE='ROI')
     wtmp=WIDGET_BUTTON(wTmpBase, VALUE='4 points on the unrotated ellipse with mouse',UVALUE='ROI')
     wtmp=WIDGET_BUTTON(wTmpBase, VALUE='5 points on the ellipse with mouse',UVALUE='ROI')
   wtmpBase=WIDGET_BUTTON(wROIMenu, VALUE='Cake',UVALUE='ROI')
   wtmpBase=WIDGET_BUTTON(wROIMenu, VALUE='Rectangle',UVALUE='ROI')
   wtmpBase=WIDGET_BUTTON(wROIMenu, VALUE='Polygon',UVALUE='ROI')
   wtmp=WIDGET_BUTTON(wROIMenu, VALUE='Crop/Export last polygon ROI',UVALUE='CROP',/Sepa)
   wtmp=WIDGET_BUTTON(wROIMenu, VALUE='Set MASK as last ROI',UVALUE='MASK',/Sepa)
   wtmp=WIDGET_BUTTON(wROIMenu, VALUE='Set MASK as last (1-ROI)',UVALUE='MASK2')
   wtmp=WIDGET_BUTTON(wROIMenu, VALUE='Apply mask from image...',UVALUE='MASKFROMIMAGE')
     wtmp=WIDGET_BUTTON(wROIMenu, VALUE='Mask pixels with zeroes',UVALUE='MASKZEROES')
     wtmp=WIDGET_BUTTON(wROIMenu, VALUE='Mask pixels with value...',UVALUE='MASKFROMPIXELS')
     wtmp=WIDGET_BUTTON(wROIMenu, VALUE='Unmask...',UVALUE='UNMASK')


wCalcMenu =  WIDGET_BUTTON(wMenuBar, VALUE='Calculations', /MENU)

   wTmpBase=WIDGET_BUTTON(wCalcMenu, VALUE='Corrections',/Menu)
     wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Set sequence of corrections',UVALUE='CALC CORRECTIONS')
     wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Substract (dark) image',UVALUE='CALC SUBSTRACT')
     wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Divide by (flat field) image',UVALUE='CALC DIVIDE')
     wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Geometric correction (warping)',$
                      UVALUE='CALC WARPING')
     wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Change image or pixel size (rebin)',$
                        UVALUE='CALC REBIN')

   wtmp=WIDGET_BUTTON(wCalcMenu, VALUE='Generic Operation',UVALUE='CALC GENERIC',/Sepa)

   wTmpBase=WIDGET_BUTTON(wCalcMenu, VALUE='Rotate/Mirror',/Menu)
     wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Rotate 90 Clockwise',UVALUE='CALC ROT')
     wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Rotate 90 CounterClockwise',UVALUE='CALC ROT')
     wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Rotate 180',UVALUE='CALC ROT')
     wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Mirror up/down',UVALUE='CALC ROT',/SEPA)
     wtmp=WIDGET_BUTTON(wTmpBase, VALUE='Mirror left/right',UVALUE='CALC ROT')
   wtmp=WIDGET_BUTTON(wCalcMenu, VALUE='Find peaks...',UVALUE='CALC PEAKS',/Sepa)
   wtmp=WIDGET_BUTTON(wCalcMenu, VALUE='(x,y)->(r,phi)...',UVALUE='CALC XY2RPHI')
   
   wtmp=WIDGET_BUTTON(wCalcMenu, VALUE='XYProfiles',UVALUE='CALC XYPROFILE',/SEPA)
   wtmp=WIDGET_BUTTON(wCalcMenu, VALUE='Transversal Profile',UVALUE='CALC PROFILE')
   wtmp=WIDGET_BUTTON(wCalcMenu, VALUE='Histogram',UVALUE='CALC HISTOGRAM')
   wtmp=WIDGET_BUTTON(wCalcMenu, VALUE='CDF(Histogram)',UVALUE='CALC CDF(HISTOGRAM)')
   wtmp=WIDGET_BUTTON(wCalcMenu, VALUE='Integral (vertical)',UVALUE='CALC INTV')
   wtmp=WIDGET_BUTTON(wCalcMenu, VALUE='Integral (horizontal)',UVALUE='CALC INTH')

   wtmp=WIDGET_BUTTON(wCalcMenu, VALUE='Center of mass',UVALUE='CALC CENTEROFMASS',/SEPA)


wSimulMenu =  WIDGET_BUTTON(wMenuBar, VALUE='Diffraction', /MENU)
     ;wtmp=WIDGET_BUTTON(wSimulMenu, VALUE='Define detector and beam pars...',UVALUE='SETDETECTOR')
     wtmp=WIDGET_BUTTON(wSimulMenu, VALUE='Define detector and beam pars...',UVALUE='EDIT PREF')
     wtmpBase=WIDGET_BUTTON(wSimulMenu, VALUE='Overplot rings',/MENU)
        wtmp=WIDGET_BUTTON(wtmpBase, VALUE='Clear',UVALUE='SETRINGS')
        wtmp=WIDGET_BUTTON(wtmpBase, VALUE='Display in xplot',UVALUE='SETRINGS')
        wtmp=WIDGET_BUTTON(wtmpBase, VALUE='Edit',UVALUE='SETRINGS')
        wtmp=WIDGET_BUTTON(wtmpBase, VALUE='Calculate from unit cell (DABAX database/file)...',/SEPA, $
             UVALUE='SETRINGS')
        wtmp=WIDGET_BUTTON(wtmpBase, VALUE='Read dSpacing or twoTheta from file...',UVALUE='SETRINGS')
        wtmp=WIDGET_BUTTON(wtmpBase, VALUE='From Powder Diffraction database (DABAX PDFrruff.dat)...', $
             UVALUE='XRD_DATABASE2')
    wTmp=WIDGET_BUTTON(wSimulMenu, VALUE='Azimuthal Integration (isolated window)',UVALUE='CALC INTEGRATE01')
    wTmp=WIDGET_BUTTON(wSimulMenu, VALUE='Azimuthal Integration (autorefresh window)',UVALUE='CALC INTEGRATE02')
    wTmpBase=WIDGET_BUTTON(wSimulMenu, VALUE='Azimuth Integration (more)',/Menu)
       wTmp=WIDGET_BUTTON(wTmpBase, VALUE='Use Macro (saxs_angle)',UVALUE='CALC INTEGRATE_MACRO')
    wTmp=WIDGET_BUTTON(wSimulMenu, VALUE='(x,y)->(2theta,phi)',UVALUE='CALC XY2RPHI_AXIAL')

    ; todo: improve this option, commented by now...
    ;wtmp=WIDGET_BUTTON(wSimulMenu, VALUE='Simulate image from XRD profile...',UVALUE='SIMUL_FROMPROFILE',/Sepa)
    wtmp=WIDGET_BUTTON(wSimulMenu, VALUE='Simulate image from d-spacings...',UVALUE='SIMUL_DSPACING',/Sepa)
    ;wtmp=WIDGET_BUTTON(wSimulMenu, VALUE='XRD Mineral database',UVALUE='XRD_DATABASE',/Sepa)
    ;wtmp=WIDGET_BUTTON(wSimulMenu, VALUE='XRD Mineral database2',UVALUE='XRD_DATABASE2',/Sepa)
   wtmp=WIDGET_BUTTON(wSimulMenu, VALUE='Azimuthal Integration preferences...',UVALUE='EDIT PREF',/Sepa)

wToolsMenu =  WIDGET_BUTTON(wMenuBar, VALUE='Tools', /MENU)
     wtmp=WIDGET_BUTTON(wToolsMenu, VALUE='Image in new window',UVALUE='TOOLS NEW WINDOW')
     wtmp=WIDGET_BUTTON(wToolsMenu, VALUE='Plotting tool (xplot)',UVALUE='TOOLS XPLOT')
     wtmp=WIDGET_BUTTON(wToolsMenu, VALUE='Create grid...', $
          UVALUE='CREATE GRID IMAGE')
     wtmp=WIDGET_BUTTON(wToolsMenu, VALUE='Create test image (optical illusions)...', $
          UVALUE='CREATE TEST IMAGE')

wHelpMenu =  WIDGET_BUTTON(wMenuBar, VALUE='Help', /Help)
     wtmp=WIDGET_BUTTON(wHelpMenu, VALUE='xplot2d',UVALUE='Help')
     wtmp=WIDGET_BUTTON(wHelpMenu, VALUE='xplot2d manual (html)', $
          UVALUE='HelpWWW')

   IF mincolor EQ maxcolor THEN BEGIN
    mincolor=Float(min(*dataptr))
    maxcolor=Float(max(*dataptr))
   ENDIF
   if minColor LE 0 then minColor=min([.1,maxColor/100.0])



   ;
   ; define the "info" structure
   ;


   wids={roi:0L,  $            ; roi window id
         LoadImage:wLoadImage, $;entry field to choose image
         scale:wScale, $
         exodus:0L, $
         cursor:wCursor, $
         Graph:wGraph, $       ;graph window
         colorBar:0L, $ ;colorbar window ID
         Comment:wComment, $       ;comment line
         ImageNum:wImageNum, $  ;entry field to choose image
         Scroll:0L, $  ;entry field to choose image
         XplotAzimuth:0L, $    ; xplot parent for azimuth integral window
         detector:0L, $    ; xplot2d_setdetector parent 
         main:wXplot2d, $       ; parent widget id
         PDFrruff:0L, $       ; widget id for rruff database (xspecscan)
         FindPeaks:0L $       ; widget id of the xplot_findpeaks window
         }

   simul_scan = {stop:['0','No','Yes'], $
     var:['0','Photon energy','x0','y0','xS','yS','zS'], $
          min:0.0,max:0.0,npoints:20L}

   mk = xplot_defaults('mk') ; marked points


   info={      nd:2, $               ; 1=xplot,2=xplot2d   
               wids:wids, $
               data: objData,$       ;pointer to the data, as an ompp object
               objRings: objRings, $ ; pointer to the rings
               str:str, $              ; all input structurs
               psysvar:{p:!p,x:!x,y:!y}, $
               image_num:0,$         ;if multiple images, the one currently displayed. Note that this in
               Min:minColor,$             ;min value displayed
               Max:maxColor,$               ;max value displayed
               ringsFlag:0,         $    ; flag to overplot rings (from twotheta or d-spacing)
               simul_scan:simul_scan, $ ; str with info on the scanning parameters (detector parms)
               image_mask:1, $
               image_substract:1, $
               no_block:no_block, $  ;  no_block flag
               Nload:0L,          $   ; number of loaded images (history)
               cursor1:[0,0,0], $       ; to store cursor position and activate flag when click-down
               mk:mk, $                ; marked points (common code with xplot)
               reloadFile:'' $         ; file to reload
               }


   widget_control, wXplot2d, /REALIZE, set_uvalue=info, /HourGlass

   IF Keyword_Set(wTitle) THEN BEGIN
     IF strpos(wtitle,'%C') GE 0 THEN wtitle=strsubstitute(wtitle,'%',$
        'Xplot2d '+xplot2d_version()+' Id:'+StrCompress(info.wids.main,/Rem) )
   ENDIF ELSE BEGIN 
     wtitle='Xplot2d '+xplot2d_version()+$
		   ' Id:'+StrCompress(info.wids.main,/Rem)
   ENDELSE
   Widget_Control,info.wids.main,TLB_Set_Title=wTitle

    IF Keyword_Set(stretch) THEN xplot2d_setcolorlimits,wXplot2d,/stretch
   IF str.display.scale LE 0 THEN xplot2d_scale,wXplot2d,/Max
   xplot2d_REFRESH,wXplot2d

   xmanager, 'xplot2d', wXplot2d, GROUP_LEADER=group,no_block=no_block
END
