;+
;
;                         ===============================
;                         = Functional Fitting Facility =
;                         ===============================
;
; FuFiFa 
;
;
; Fufifa is an application to fit interactively a SEVERAL spectra to a FUNCTION.
;
; It uses MPFIT (from Craig Markwardt) as fitting engine: 
; http://cow.physics.wisc.edu/~craigm/idl/fitting.html
;
;
; The goal is to provide with a point-and-click application for fitting many 
; data sets (usually imported from a SPEC file, that can be prepared with 
; Xop/EWxodus) with a user-defined function. 
;
; The function can be built interactively using a set of components (straight 
; line, polynomial, peaks (gaussian, lotentzian and pseudovoigt), steps (error 
; function and arc tangent) or any user-define function). 
;
; The application allows to fit simultaneously several data sets and to
; refine individually the fits. A plot of the fitted parameters and chi square 
; versus the data set index can help to estimate the quality of the fits. 
;
; To operate with fufifa follow the following steps:
;
; 1) Start the application by either: 
;
; i) Start xop and from the Xop->Tools->FuFiFa, This will create the fufifa 
;    window.
; ii) Enter "xop fufifa" at the unix prompt
;
; 2) Import your data.
;   Press the load button in the data sector (top right of the main window).
;   You should select a file (SPEC type) which has been previously "cleaned".
;   "Cleaned SPEC file" means that:
;     i) you have removed the unwanted scans (only kept those used for fitting)
;     ii) Each scans has ONLY two or three columns
;         The first one is the abscissas X
;         The second one is the ordinates Y
;         The third (optional) one contains the errors of Y. The errors used for
;            the fit are defined in the Fit->Preferences menu item. 
;    You can use the EXODUS application (available in the XOP/XAID-XAFS toolbox)
;         for this purpose (i.e., cleaning your SPEC file).
;
;    Example: Using the file cat3.spec I
;             extracted (using EXODUS) the scans [82,83,92,93,111,112 and 113] 
;             I used column 1 for X and column 16 for Y. The result was 
;             written to file exodus.spec. This file was imported into fififa. 
;             Then I selected all scans (by pressing ctrl-click iteratively or 
;             doing multiple selection with shift-click). 
;             The result of these operations are in Fig1.
;
; 3) Define your model:
;    Using the bottom sector of your fufifa window, I defined a function 
;    consisting in a step function (errorF) and a lorentzian peak. For this 
;    I clicked "Add function" twice, one for each functional component. 
;    I did not adjust the input parameters. 
;
;    First I select my first scan (clicking on the list at the top right sector).
;    The meaning of the small buttons for each model line are the following:
;    "x" : removes the model line
;    "G" Get the guess values (using the mouse for peaks and straight lines)
;    Each parameter can be modified directly by enteing a new number, or by 
;    pressing:
;    "F" Fixes this parameter (not changed during the fit)
;    "B" Edit parameter boundaries
;
; 4) First attempt to fit.
;    If you press "**Fit**" you fit the function parameters to data. 
;    Then you can press "Copy Pars To All DataSets" that copies the fitted 
;    values for the first scan to the rest of scans. 
;
;    If you want to fit all spectra, just make a multiple selection and press 
;    the fit button. Then you can select individual scans to see better the 
;    results.
;    Perhaps some scans are not well fitted. You can select by hand the guess
;    parameters and fit again. 
;
;    Note: when using multiple selection, you can use the "Fit2" button 
;    instead of "**Fit**". In this case, the guess values for the i-th
;    selected set will be the fitted values of the (i-1)-th dataset. 
;    For single selection there is no difference between "Fit2" and "**Fit**".
;    This option gives better results (smoother) when looking to the evolution 
;    of the fitted values versus the dataset index.
;
; 5) Refinement of the fit. 
;    You can preprocess the data using the "Data" menu items to:
;      i) perform some operation (like normalization).
;      ii) Cut unwanted data and redefine your interval. Use it to remove 
;          points of the pre-edge and the incomplete last oscillation. 
;      iii) Substract a linear background (not needed in this case). 
;
;      After doing that, I added two more lorentzian peaks. I got results shown
;      in Fig4. Much better!
;
;      iv) You can ignore a part of the data for fitting purposes bu using the
;      option "Include [a,b] and/or exclude {a,b} regions"
;
; 6) Visualizing and saving results:
;     By pressing "Pars vs Data Sets" you can mahe plots of the parameters and 
;     the reduced Chi square versus the scan number. 
;     This permits you to see if there are drastic changes in some parameters.  
;
;     In order to save the full information, you can use FuFiFa->Save as... 
;     It will
;     create a SPEC-type file with all scans, error used, and models. 
;     In the scan header you can find the information on the fitted parameters 
;     and chi**2. The last scan (numbered 1000) contains the variation of 
;     the parameters versus the scan index. 
;     Of course, you can use Xplot to view this resulting file. 
;     This file can be reloaded in FuFiFa.
;
;
;
;  Non Interactive use:
;	; load ASCII Files
;	fufifa,Parent=p
;	fufifa_loadfile,p,isel=0 ; import ascii file (start browser)
;	fufifa_loadfile,p,isel=0,File='CuFoilRef.txt',append=0 ; import ascii 
;
;	; load SPEC Files
;	fufifa_loadfile,p,isel=1,File='fufifa.spec',Append=0
;
;
;
; Some functionality already implemented
; - Load components of the function from an external (customizable) file
; - Access to boundaries of parameters for the fit
; - "Expert" mode (do not show information windows)
; - Integrated in xplot and in xop 
; - Prints single or multiple plots
; - Exports fitting data to a Excel file
;
; ToDo:
; - Allow MCI inputs
; - Abort button (when fitting...)
; - Customize colors
; - Polynomial inputs from mouse
; - Interactive help
;
;  COPYRIGHT:
; 	FUFIFA belongs to XOP package and it is distributed within XOP.
; 	PLEASE REFER TO THE XOP COPYRIGHT NOTICE
; 
;  REFERENCE:
; 	Published calculations made with XOP should refer:
; 
; 	M. Sanchez del Rio and R. J. Dejus 
;         "Status of XOP: an x-ray optics software toolkit"
;         SPIE Proceedings Vol. 5536 (2004) pp.171-174
; 
;         http://dx.doi.org/10.1117/12.560903
; 
;  LAST MODIFICATION: srio@esrf.eu 2008-12-12
; 
; HISTORY:
; 
; srio@esrf.fr 7 December 2001
; Modified:
;     2003/12/09 srio@esrf.fr Fix an incompatibility problem for IDL >=5.6
;     2006/03/02 srio@esrf.fr Allows non-interactive use:  FUFIFA_LOADFILE
;     2006/10/02 srio@esrf.fr Many upgrades. Version 1.00
;     2006/10/10 srio@esrf.fr Direct export to FuFiFa
;     2006/11/07 srio@esrf.fr Debug load file with model.
;     2008/09/03 srio@esrf.fr Added "fit2" button (recurrent fit)
;     2008/12/03 srio@esrf.fr Added covariance calculations. Bug fixed when reading
;                polynomial from file. Loads/reads preferences to file. v1.03
;
;-

;
;
;========================================================================
;
FUNCTION fufifa_text,input

txt=''

CASE input OF
  'DataOperations': txt=[ $
     'This option permits to make operations in the x and y data',$
     'The user must input the new expressions for x and y:',$
     '    x=...',$
     '    y=...',$
     '   ',$
     '   Notes: ',$
     '   1) The operations are done sequentially, first for x.',$
     '      For example, if x=x-100 and y=y*x are selected, the',$
     '      y operation uses x with the substraction (x-100) already',$
     '      performed',$
     '   2) It is possible to define new intermediate variables',$
     '   3) It is possible to send several instructions separated by &',$
     '   4) An error column (a third column) may be defined. ',$
     '      -If nothing is written, no third column is added',$
     '      -enter: err=err to mantain the existing third (error) column.',$
     '      -enter a function of err,y,x,y1,x1 to create a new ',$
     '       error column, being x,y the values before operation, ',$
     '       x1,y1 the values after operation, and err the third column,',$
     '       if defined (zero otherwise)',$
     '    ',$
     '   Examples: ',$
     '   1) Shift x axis by 100: ',$
     '               x=x-100',$
     '               y=y',$
     '   2) Normalize y in [0,1]: ',$
     '               x=x',$
     '               y=y-Min(y) & y=y/Max(y) ',$
     '   3) Set the zero of x in the Max of the derivative (as in XANES):',$
     '               e0=Max(deriv(y),i) & x=x-x[i]',$
     '               y=y', $
     '   4) Normalize y to the mean value in a window: ',$
     '               Identify the window: you may use xplot to plot the',$
     '               data vs the index, by double-click on the data set.',$
     '               x=x ',$
     '               y0 = Mean(y[400:500]) & y=y/y0 ', $
     '   5) Create statistical errors:',$
     '               x=x ',$
     '               y=y ',$
     '               err=sqrt(y) ']
  'Boundaries': txt=[ $
     '         - VALUE - the starting parameter value ',$
     '',$
     '         - FIXED - a boolean value, whether the parameter is to ',$
     '                   be held fixed or not.  Fixed parameters are',$
     '                   not varied by MPFIT',$
     '',$
     '         - LIMITED - a two-element boolean array.  If the',$
     '           first/second element is set, then the parameter is',$
     '           bounded on the lower/upper side.  A parameter can be',$
     '           bounded on both sides.  Both LIMITED and LIMTIS must',$
     '           be given together.',$
     '',$
     '         - LIMITS - a two-element float or double array.  Gives',$
     '           the parameter limits on the lower and upper sides,',$
     '           respectively.  Zero, one or two of these values can',$
     '           be set, depending on the value of LIMITED.  Both ',$
     '           LIMITED and LIMITS must be given together']
  else:
ENDCASE ; fufifa_text
RETURN,txt
END
;
;========================================================================
;

FUNCTION FuFiFa_version
RETURN,'1.03'
END ; FuFiFa_version

;
;========================================================================
;

PRO fufifa_list_select,tlb,index

;
; index=-1 selects the last item in the list
;
Widget_Control,tlb,Get_UValue=state
IF index[0] LT 0 THEN BEGIN
  nn =  state.oData->info(/N_Elements)
  idx = nn-1
ENDIF ELSE idx=index
Widget_Control,state.wids.dataList,set_list_select=idx
END

;
;========================================================================
;

FUNCTION fufifa_parsedomain,sDomain1,x,Group=group,out=out,iGood=igood

;
; Inputs: sDomain1: a string with the interval definition, like
;		[2,3][5,6]{1.2,1} ; [] are " included", {} "excluded"
;         x: optional input with an array where to check the limits
; Output: Returns the number of checking intervals
; Keywords:
;	Group
;	out: an array of structures {min:0.,max:0.,inlude:0} with the limits
;	igood: if x is defined, this ae the "good" indices
;

sDomain=sDomain1
sDomain = StrCompress(sDomain,/Rem)
itmp=StrPos(sDomain,';') 
IF itmp NE -1 THEN BEGIN
   sDomain=StrMid(sDomain,0,itmp)
ENDIF
sDomain = StrCompress(sDomain,/Rem)
IF sDomain EQ '' THEN RETURN,0


;a1 = StrMatch(sDomain,'*\[*')
;a2 = StrMatch(sDomain,'*\]*')
;b1 = StrMatch(sDomain,'*{*')
;b2 = StrMatch(sDomain,'*}*')
;ierror=0
;
;IF a1 NE a2 THEN iError=1
;IF b1 NE b2 THEN iError=1

line=StrCompress(sDomain,/Rem)
interval1=''
interval2=''
iError=0

WHILE StrLen(line) GT 0 AND iError NE 1 DO BEGIN
    ; find intervals in squared brackets [x,y]
    itmp1 = StrPos(line,'[')
    itmp2 = StrPos(line,']')
    iFound=0
    IF itmp1 NE -1 AND itmp2 NE -1 THEN ifound=1
    IF iFound EQ 1 THEN BEGIN
       ilen = itmp2-itmp1+1
       interval = StrMid(line,itmp1,ilen)
       interval1=[interval1,interval]
       line1=StrMid(line,0,itmp1)+StrMid(line,itmp2+1,StrLen(line)-itmp2)
    ENDIF ELSE BEGIN
       line1=line
    ENDELSE

    ; find intervals in curly brackets {x,y}
    itmp1 = StrPos(line1,'{')
    itmp2 = StrPos(line1,'}')
    iFound=0
    IF itmp1 NE -1 AND itmp2 NE -1 THEN ifound=1
    IF iFound EQ 1 THEN BEGIN
       ilen = itmp2-itmp1+1
       interval = StrMid(line1,itmp1,ilen)
       interval2=[interval2,interval]
       line2=StrMid(line1,0,itmp1)+StrMid(line1,itmp2+1,StrLen(line1)-itmp2)
    ENDIF ELSE BEGIN
       line2=line1
    ENDELSE
    IF line2 EQ line THEN iError=1 ELSE line=line2
ENDWHILE

IF iError THEN BEGIN
    itmp = Dialog_Message(Dialog_parent=group,$
	'Expression not understood: '+line)
    RETURN,0
ENDIF

n1 = N_Elements(interval1)-1
n2 = N_Elements(interval2)-1
out=Replicate( {min:0.0, max:0.0, include:0}, n1+n2)
k=0
IF n1 GE 1 THEN BEGIN
  FOR i=0,n1-1 DO BEGIN
    itmp = Execute('tmp='+interval1[i+1])
    outi = out[k]
    outi.min=tmp[0]
    outi.max=tmp[1]
    outi.include=1
    out[k]=outi
    k=k+1
  ENDFOR
ENDIF
IF n2 GE 1 THEN BEGIN
  FOR i=0,n2-1 DO BEGIN
    stmp = StrSubstitute(interval2[i+1],'{','[')
    stmp = StrSubstitute(stmp,'}',']')
    itmp = Execute('tmp='+stmp)
    outi = out[k]
    outi.min=tmp[0]
    outi.max=tmp[1]
    outi.include=0
    out[k]=outi
    k=k+1
  ENDFOR
ENDIF
nout = N_Elements(out)

IF N_Elements(x) GT 0 THEN BEGIN
  iinclude = out.include
  itmp=Where(iinclude EQ 1)
  IF itmp[0] EQ -1 THEN flags=x*0+1 ELSE flags=x*0
  FOR i=0,nout-1 DO BEGIN
    IF iinclude[i] EQ 1 THEN BEGIN
      itmp = Where(x GE (out[i]).min AND x LE (out[i]).max)
      IF itmp[0] NE -1 THEN flags[itmp]=1
    ENDIF ELSE BEGIN
      itmp = Where(x GE (out[i]).min AND x LE (out[i]).max)
      IF itmp[0] NE -1 THEN flags[itmp]=0
    ENDELSE
  ENDFOR
  igood = Where(flags EQ 1)
ENDIF

RETURN,nout
END
;
;========================================================================
;

FUNCTION FuFiFa_GetModelParsVariation,tlb,error=error

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_GetModelParsVariation: error caught: '+!error_state.msg)
  Catch, /Cancel
  error=1
  RETURN,0
ENDIF

error=0
Widget_Control,tlb, Get_UValue=state

ndata = state.oData->info(/N_Elements)
oData = state.oData

IF ndata EQ 1 and state.oData->value(0,/IsDataDef) NE 1 THEN BEGIN
  Message,/Info,'No data'
  error=1
  RETURN,0
ENDIF

iGood = IntArr(nData)
FOR i=0,nData-1 DO IF state.oData->value(i,/IsAuxDef) EQ 1 THEN iGood[i]=1

FOR i=0,nData-1 DO BEGIN
  IF iGood[i] EQ 1 THEN BEGIN
    tmp = state.oData->value(i,/Aux)
    npars = N_Elements(tmp)
    IF N_Elements(set) LE 1 THEN set = DblArr(npars+2,nData)
    FuFiFa_chisquare,state,i,chi2=chi2
    set[0,i]=i
    set[1:npars,i]=tmp
    FuFiFa_chisquare,state,i,chi2=chi2
    set[npars+1,i]=chi2
  ENDIF 
ENDFOR
IF N_Elements(set) EQ 0 THEN BEGIN
  error=1
  RETURN,0
ENDIF

set=set[*, where(iGood EQ 1) ]

RETURN,set
END ; FuFiFa_GetModelParsVariation
;
;========================================================================
;

FUNCTION FuFiFa_GetErrorArray, data, mpfitpars, errTxt, Group=group

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_GetErrorArray: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN,0
ENDIF
errorsDefined = 0

IF StrCompress( (mpfitpars.errorsFrom)[0], /Rem) EQ '0' THEN BEGIN
  errtxt = 'Requested using errors from third column'
  ;Message,/Info,errtxt
  IF (size(data))[1] GE 3 THEN BEGIN
    errorsDefined =1 
    dataErr = data[2,*]
    errtxt = 'Requested using errors from third column: Using errors from third column'
    ;Message,/Info,errtxt
  ENDIF ELSE BEGIN
    errtxt = 'Requested using errors from third column: Undefined third column'
    Message,/Info,errtxt
  ENDELSE
ENDIF


IF errorsDefined EQ 0 THEN BEGIN
  CASE StrCompress(mpfitpars.errors[0],/Remove_All) OF
    '0': BEGIN
        errtxt = 'Using constant errors'
        ;Message,/Info,errtxt
        dataErr=Reform(data[1,*])*0+1
        END
    '1': BEGIN
        errtxt = 'Using statistical errors'
        ;Message,/Info,errtxt
        dataErr = Sqrt(Reform(data[1,*]))  ; 1*sigma = sqrt(y)
        END
    else:
  ENDCASE
ENDIF

RETURN,dataErr
END ; FuFiFa_GetErrorArray
;
;============================================================================
;
PRO FuFiFa_chisquare, state, indexLine, Group=group, Ask=ask, chi2=chi2


Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_chisquare: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

;
; get data
;
data = state.oData->value(indexLine)
dataX = Reform(data[0,*])
dataY = Reform(data[1,*])

;
; get model expression
;
oeq = state.oequation
model = oeq->getEquation()
IF model EQ '' THEN BEGIN
  itmp = Dialog_message(/Error,Dialog_parent=group, $
	'No model to fit.')
  RETURN
ENDIF
;  print,'>>> Model equation is: '+model

IF  state.oData->value(indexLine,/IsAuxDef) EQ 1 THEN BEGIN
  aux = state.oData->value(indexLine,/Aux)
  Widget_Control,state.wids.dataModel,Set_Value='Model: defined.'
  oeq->SetProperty,Param=aux   ; ,/UpdateWidgets
ENDIF ELSE BEGIN
  itmp = Dialog_message(/Error,Dialog_Parent=group,$
	['No parameters assigned to data set with index: '+$
	StrCompress(indexLine),'Apply fit to it.'])
  RETURN
ENDELSE



IF N_Elements(datax) EQ 0 THEN BEGIN
 itmp = Dialog_Message(/Error,'FuFiFa_plot: abscissas not defined')
 RETURN
ENDIF

;plot,datax,data[1,*],YRange=yRange,/NoData

oeq = state.oequation
nlines = oeq->GetProperty(/Nlines)

IF nlines EQ 0 THEN BEGIN
  itmp = Dialog_message(/Error,Dialog_Parent=group,$
    'No model defined')
  RETURN
ENDIF

dataFit=oeq->evaluate(datax)
dataErr = FuFiFa_GetErrorArray(data,state.MPFitPars)

deviates = (dataY-dataFit)/dataErr
nn = N_Elements(dataX)
chi2 = Total(deviates^2)/nn/(nn-1)

END ; FuFiFa_chisquare

;
;========================================================================
;

PRO fufifa_addFunction,wid, param=param1, noAsk=noAsk, name=name, $
  equation=equation, ntimes=ntimes

COMMON fufifa, strDataOperations, addFunctionMenu, strDataCut


Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_addFunction: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF


Widget_Control,wid, Get_UValue=state

oeq = state.oequation


IF N_Elements(addFunctionMenu) EQ 0  THEN BEGIN
	list = ['0',oeq->getProperty(/ListEqua)]
	addFunctionMenu = {ntimes:1, list:list, degree:1, center:10.0, fwhm:11.0, jump:12.0, $
		equation:'x*p[!]+p[!]',pars:'[1.0,1.0]' }
ENDIF

IF Not(Keyword_Set(noAsk)) THEN BEGIN
  flags=['1','1', 'w(1) EQ 1', $
		'w(1) GE 2 AND w(1) LE 6', $
		'w(1) GE 2 AND w(1) LE 6', $
		'w(1) GE 2 AND w(1) LE 6', $
		'w(1) EQ 7','w(1) EQ 7']
  titles=['Repeat n times','Function','Degree','Center','FWHM','Height or Jump', $
		'Function f(x,p[!],...p[!])','Guess pars']

  XScrmenu,addFunctionMenu,/Interp,/NoType,Action=action,Dialog_Parent=wid, $
          Flags=flags, Titles=titles, WTitle='Function definition', $
	  FieldLen=30
  IF action EQ 'DONT' THEN RETURN
ENDIF

;  print,oeq->info()

IF N_Elements(ntimes) EQ 0 THEN ntimes=addFunctionMenu.nTimes
FOR ii=0,nTimes-1 DO BEGIN ;<<<<<<<<<<<<<<<<<<<<
index = Fix( (addFunctionMenu.list)[0] )
IF Not(Keyword_Set(name)) THEN  BEGIN
  name = (addFunctionMenu.list)[1+Fix((addFunctionMenu.list)[0])]
ENDIF
;print,'*****************'+name+'****************'
	CASE StrCompress(StrUpCase(name),/Rem) OF 
	  'STRAIGHTLINE': BEGIN	; StraightLine
		oeq->AddLine,/straight
		isApeak = 0
		END
	  'POLYNOMIAL': BEGIN	; Polynomial
	        IF Keyword_Set(param1) THEN BEGIN
                  degree=N_Elements(param1)-1
                ENDIF ELSE BEGIN
		  degree=addFunctionMenu.degree
                ENDELSE
		oeq->AddLine,Poly=degree
		isApeak = 0
		END
	  'GAUSSIAN': BEGIN	; Gaussian
		oeq->AddLine,/Gauss,Para=[addFunctionMenu.jump,addFunctionMenu.center,addFunctionMenu.fwhm]
		isApeak = 1
		END
	  'LORENTZIAN': BEGIN	; Lorentzian
		oeq->AddLine,/Lorentzian,Para=[addFunctionMenu.jump,addFunctionMenu.center,addFunctionMenu.fwhm]
		isApeak = 1
		END
	  'PSEUDOVOIGT': BEGIN	; PseudoVoigt
		oeq->AddLine,/PseudoVoigt, $
			Para=[addFunctionMenu.jump,addFunctionMenu.center,addFunctionMenu.fwhm,0.5]
		isApeak = 1
		END
	  'STEPERRORF': BEGIN	; StepErrorF
		oeq->AddLine,/StepErrorF,Para=[addFunctionMenu.jump,addFunctionMenu.center,addFunctionMenu.fwhm]
		isApeak = 0
		END
	  'STEPATAN': BEGIN	; StepAtan
		oeq->AddLine,/StepAtan,Para=[addFunctionMenu.jump,addFunctionMenu.center,addFunctionMenu.fwhm]
		isApeak = 0
		END
	  'USERDEFINED': BEGIN	; UserDefined
                IF Not(Keyword_Set(param1)) THEN BEGIN
		  para = 0.0
		  command = 'para = '+addFunctionMenu.pars
		  itmp = Execute(command)
                ENDIF ELSE para=param1
		IF Not(Keyword_Set(equation)) THEN equation=addFunctionMenu.equation
		oeq->AddLine,equation,Para=para
		isApeak = 0
		END
	  else: Message,'Function not implemented: '+name
	ENDCASE

	IF Keyword_Set(param1) THEN BEGIN
	  oeq->SetPropertyLine,param=param1
	ENDIF
	;
	; create and store widget information
	;
	nlines=oeq->GetProperty(/NLines)

	nindices=oeq->GetProperty(/NIndices)

        istart = Fix(Total(nIndices)-nIndices[nlines-1])
        ;istart = oeq->GetPropertyLine(nLines-1,/startIndex)


	line = oeq->GetPropertyLine(nlines-1,/equation)
	values = oeq->GetPropertyLine(nlines-1,/parLine)
	name1 = oeq->GetPropertyLine(nlines-1,/name)
	flags=Fix(values*0+1)
	;wtmpbase = Widget_Base(state.wids.ModelBase)
	value={coefs:values, flags:flags }
	wtmp =CW_PField(state.wids.ModelBase,Column=0, $
		UValue=nlines-1, $	; line number info in UValue
		Value=value,Title=[name,line],PosTitle=1,$
		Frame=1,$
		pTitles='p['+StrCompress(istart+IndGen(N_Elements(values)),/Rem)+']:' )
	Widget_Control,wtmp,/Realize
	oeq->SetPropertyLine,wId=wtmp

	; 
	; set boundaries
	;
	IF Not(Keyword_Set(noAsk)) THEN BEGIN
	  IF StrUpCase(StrCompress(name,/Rem)) EQ 'PSEUDOVOIGT' THEN BEGIN
            IF ii EQ 0 THEN BEGIN
	      iAns1 = Dialog_message(/Question, Dialog_Parent=wid, $
		['Setting boundaries:',$
		'Do you want to limit the Pseudovoigt weigth',$
		'between 0 (pure gaussian) and 1 (pure lorentzian)? '])
	    ENDIF
	    IF iAns1 EQ 'Yes' THEN $
	    oeq->setPropertyItem,nlines-1,3,Limited=[1,1],Limits=[0.0,1.0]
	  ENDIF

	  IF isApeak EQ 1 THEN BEGIN
            IF ii EQ 0 THEN BEGIN
	      iAns2 = Dialog_message(/Question, Dialog_Parent=wid, $
		['Setting boundaries:',$
		'Do you want to limit the peak amplitude and width',$
		'to positive values?'])
	    ENDIF
	    IF iAns2 EQ 'Yes' THEN BEGIN
	      oeq->setPropertyItem,nlines-1,0,Limited=[1,0],Limits=[0.0,0.0]
	      oeq->setPropertyItem,nlines-1,2,Limited=[1,0],Limits=[0.0,0.0]
	    ENDIF
	  ENDIF
	ENDIF



	;
	; update data auxiliar
	;
	sel = Widget_Info(state.wids.dataList,/List_Select)
	IF sel[0] NE -1 THEN BEGIN
	nn =  state.oData->info(/N_Elements)
	IF nn EQ 0 THEN Message,'No data input.'
	; clear all stored data
	FOR i=0L,nn-1 DO BEGIN
	  state.oData->Set,i,/ClAux
	ENDFOR
	goodIndices = IndGen(nn)
	goodIndices = goodIndices(sel)
	all_pars=oeq->GetProperty()
	; set new auxiliary data
	FOR i=0L,N_Elements(goodIndices)-1 DO BEGIN
	  state.oData->Set,goodIndices[i],Aux=all_pars
	ENDFOR
	ENDIF
ENDFOR 
	; 
	; Plot
	;
	FuFiFa_Plot,wid


	END
;
;========================================================================
;
PRO FuFiFa_quit,event


Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_quit: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

Widget_Control,event.top, Get_UValue=state

IF Ptr_Valid(state.ptrUndoFit) THEN Ptr_Free,state.ptrUndoFit
Obj_Destroy,state.oData
Obj_Destroy,state.oEquation

Widget_Control,event.top,/Destroy
END ; FuFiFa_quit

;
;========================================================================
;
PRO FuFiFa_plot,tlb,index,Group=group,Print=print,yrange=yrange,$
  xrange=xrange,multiplePrint=multiplePrint


Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_plot: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

Widget_Control,tlb,Get_UValue=state

IF N_Elements(xrange) EQ 0 THEN xrange=[0,0]
IF N_Elements(yrange) EQ 0 THEN yrange=[0,0]

IF N_Elements(index) EQ 0 THEN BEGIN
index=Widget_Info(state.wids.dataList,/List_Select)
  IF index[0] EQ -1 THEN BEGIN
;    itmp = Dialog_message(/Error,Dialog_parent=group, $
;	'No data to plot. Please Load/Select data')
    RETURN
  ENDIF
ENDIF

IF NOT(Keyword_Set(print)) THEN BEGIN
  ;Widget_Control,event.top, Get_UValue=state
  widget_control,state.wids.draw, get_val = wnum
  wset,wnum
ENDIF

flag_which=state.plotDropList
oeq = state.oEquation

;
; get limits
;
nindex = N_Elements(index)
xmin1 = FltArr(nindex)
xmax1 = FltArr(nindex)
ymin1 = FltArr(nindex)
ymax1 = FltArr(nindex)
FOR i=0L,N_Elements(index)-1 DO BEGIN
  data = state.oData->value(index[i])
  datax = Reform(data[0,*])

  ; limits
  ymin1[i] = min(data[1,*],max=tmp)
  ymax1[i]=tmp
  xmin1[i] = min(datax,max=tmp)
  xmax1[i]=tmp
ENDFOR
xmin = min(xmin1)
xmax = max(xmax1)
ymin = min(ymin1)
ymax = max(ymax1)

yint = Abs(ymax-ymin)
IF (yrange[0] EQ yrange[1])  THEN yRange=[ymin-0.2*yint,ymax+0.1*yint]
IF (xrange[0] EQ xrange[1])  THEN xRange=[xmin,xmax]

;
; get data
;

FOR i=0L,N_Elements(index)-1 DO BEGIN
  data = state.oData->value(index[i])
  datax = Reform(data[0,*])

  IF Keyword_Set(multiplePrint) THEN BEGIN
    yint = Abs(ymax1[i]-ymin1[i])
    yRange=[ymin1[i]-0.2*yint,ymax1[i]+0.1*yint]
    xRange=[xmin1[i],xmax1[i]]
  ENDIF

  IF  state.oData->value(index[i],/IsAuxDef) EQ 1 THEN BEGIN
    aux = state.oData->value(index[i],/Aux)
    isModelDefined=1
    IF NOT(Keyword_Set(print)) THEN $
      Widget_Control,state.wids.dataModel,Set_Value='Model: defined.'

    ; update widgets only for single selection
    IF N_Elements(index) EQ 1 THEN oeq->SetProperty,Param=aux,/UpdateWidgets ELSE $
                                   oeq->SetProperty,Param=aux

    ; ?????   oeq->SetProperty,Param=aux,/UpdateWidgets
    ; store pars in oeq only for a single selection
    ; IF N_Elements(index) EQ 1 THEN oeq->SetProperty,Param=aux,/UpdateWidgets
    ; ?????   state.oData->set,i,Aux=aux
  ENDIF ELSE BEGIN
    isModelDefined=0
    IF NOT(Keyword_Set(print)) THEN $
      Widget_Control,state.wids.dataModel,Set_Value=' Model: undefined. '
  ENDELSE

  IF N_Elements(datax) EQ 0 THEN BEGIN
   itmp = Dialog_Message(/Error,'FuFiFa_plot: abscissas not defined')
   RETURN
  ENDIF


  IF KeyWord_Set(multiplePrint) THEN BEGIN
     titles = state.oData->value(/title)
     plot,datax,data[1,*],YRange=yRange,/NoData,xrange=xrange,$
     YStyle=1,XStyle=1,title='id:'+StrCompress(i,/rem)+' '+titles[i]
  ENDIF ELSE BEGIN
    IF i EQ 0 THEN $
         plot,datax,data[1,*],YRange=yRange,/NoData,xrange=xrange,$
         YStyle=1,XStyle=1
  ENDELSE

  ;
  ; show data
  ;
  IF flag_which[0] EQ 1 THEN BEGIN
     oplot,datax,data[1,*] ,Color=2
  ENDIF

  ;
  ; show errors
  ;
  IF flag_which[3] EQ 1 THEN BEGIN
     dataErr = fufifa_GetErrorArray(data,state.mpfitpars,Group=group)
     plotErr,datax,data[1,*]-dataErr,data[1,*]+dataErr
  ENDIF

  oeq = state.oequation
  nlines = oeq->GetProperty(/Nlines)

  IF nlines NE 0 THEN BEGIN
    IF isModelDefined THEN BEGIN
      ; 
      ; show individual models
      ; 
      IF flag_which[2] EQ 1 THEN BEGIN
        FOR j=0L,nlines-1 DO BEGIN
          oplot,datax,oeq->evaluate(datax,Line=j),Color=4
        ENDFOR
      ENDIF
      ; 
      ; show model
      ; 
      IF flag_which[1] EQ 1 THEN BEGIN
        oplot,datax,oeq->evaluate(datax),Color=3
      ENDIF
      ; 
      ; show residuals
      ; 
      IF flag_which[4] EQ 1 THEN BEGIN
        yy = data[1,*]-oeq->evaluate(datax)
        yy = yy-min(yy)+ yrange[0] ; shift to be sure it is displayed
        oplot,datax,yy,Color=5
      ENDIF
    ENDIF
  ENDIF
ENDFOR


END ; FuFiFa_plot


;
;============================================================================
;
FUNCTION fufifa_plotDropList,values


Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_plotDropList: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN,0
ENDIF

  tmp = ['Show/Hide:','Data ','Model ','Components ','Error bars ','Residuals ']

FOR i=0,N_Elements(values) -1 DO BEGIN
  IF values[i] EQ 1 THEN tmp[i+1] = tmp[i+1]+'(shown)' ELSE $
	tmp[i+1] = tmp[i+1]+'(hidden)' 
ENDFOR

RETURN,tmp
END




;
;============================================================================
;

PRO FuFiFa_MPFit_Display, fcn, x, iter, FunctArgs=fcnargs, FMT=fmt, $
         Quiet=quiet, _Extra=iterargs

Forward_Function mpfit_enorm
; this routine is called by MPFit

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_MPFit_Display: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

IF Keyword_Set(quiet) THEN RETURN
fvec = Call_Function(fcn, x, _Extra=fcnargs)
fnorm = mpfit_enorm(fvec)

XDisplayfile1_Append, [String(iter, fnorm^2, $
  Format='("Iter ",I6,"   CHI-SQUARE = ",G20.8)'), $
  '           Timing [sec]:  '+$
	  String(SysTime(1)-iterargs.t0,Format='(G6.2)')]

IF N_Elements(fmt) GT 0 THEN BEGIN
    IF Not(Keyword_Set(quiet)) THEN $
      XDisplayfile1_Append, String(p, format=fmt)
ENDIF ELSE BEGIN
    p = '  P('+StrTrim(LIndGen(N_Elements(x)),2)+') = ' $
      + StrTrim(String(x,Format='(G20.6)'),2) + '  '
    IF Not(Keyword_Set(quiet)) THEN $
      XDisplayfile1_Append, String('         '+p, Format='(A)')
ENDELSE

IF (SysTime(1)-iterargs.t0) GE iterargs.tmax THEN BEGIN
;       itmp = Widget_Message(Dialog_Parent=iterargs.parent, $
;	['Maximum time ('+StrCompress(Long(iterargs.tmax))+' sec) reached.',$
;	 'Fitting process aborted.'])
  !err=-1
ENDIF

RETURN
END ; FuFiFa_MPFit_Display



;
;============================================================================
;
PRO FuFiFa_fit, state, indexLine, indexLineFrom, Group=group, Ask=ask


Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_fit: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF
IF N_Elements(indexLineFrom) EQ 0 THEN indexLineFrom=indexLine
;
; get data
;
data = state.oData->value(indexLine)
dataX = Reform(data[0,*])
dataY = Reform(data[1,*])
;
; modify it following the domain restrictions
;
Widget_Control,state.wids.FitDomain,Get_Value=sDomain
n = fufifa_parsedomain(sDomain,datax,Group=group,out=out,iGood=igood)
IF n GT 0 THEN BEGIN
  IF igood[0] EQ -1 THEN BEGIN
    itmp = Dialog_Message(/Error,Dialog_Parent=group, $
       'No points found in the intervals: '+sDomain)
    IF StrCompress( (state.MPFitPars.log)[0] ,/Rem) EQ '1' THEN BEGIN
      XDisplayfile1_Append,['Error:',$
      'No points found in the intervals: '+sDomain]
    ENDIF
    RETURN
  ENDIF ELSE BEGIN
    IF StrCompress( (state.MPFitPars.log)[0] ,/Rem) EQ '1' THEN BEGIN
      XDisplayfile1_Append,[' ',$
        'Using [] and/or excluding {} points in the intervals: '+sDomain,$
        'Number of points reduced from: '+StrCompress(N_Elements(dataX),/Rem)+' to '+$
          StrCompress(N_Elements(iGood),/Rem)]
    ENDIF
    dataX=dataX[iGood]
    dataY=dataY[iGood]
  ENDELSE
ENDIF

;
; get model expression
;
oeq = state.oequation
model = oeq->getEquation()
IF model EQ '' THEN BEGIN
  itmp = Dialog_message(/Error,Dialog_parent=group, $
	'No model to fit.')
  RETURN
ENDIF
;  print,'>>> Model equation is: '+model

;
; get parameters structure
;
guess = oeq->getProperty(/str)

;
; store guess for UNDO purposes
;
;*(state.ptrUndoFit)=Reform(guess.value)


;
; Overwrite initial parameteres
;


IF  state.oData->value(indexLineFrom,/IsAuxDef) EQ 1 THEN BEGIN
  aux = state.oData->value(indexLineFrom,/Aux)
  Widget_Control,state.wids.dataModel,Set_Value='Model: defined.'
  oeq->SetProperty,Param=aux,/UpdateWidgets
  guess = oeq->getProperty(/str)
  ; ?????   state.oData->set,i,Aux=aux
  ;
  ; store guess for UNDO purposes
  ;
  *(state.ptrUndoFit)=aux
ENDIF ELSE BEGIN
 Widget_Control,state.wids.dataModel,Set_Value=' Model: undefined. '
 aux=guess
ENDELSE



;
;  MPFit parameters
;
quiet=Fix( (state.mpfitpars.expert)[0] )
tol = state.MPFitPars.tol
tMax=state.MPFitPars.tMax
covar = 0 
status = 0

dataErr = FuFiFa_GetErrorArray(data,state.MPFitPars,errTxt,Group=group) 

nPoints = N_Elements(dataX)
dof = nPoints-N_Elements(guess.value)+Long(Total(guess.fixed))
;
; append info to log window
;
Widget_Control,/HourGlass
IF StrCompress( (state.MPFitPars.log)[0] ,/Rem) EQ '1' THEN BEGIN
  XDisplayfile1_Append,['',$
    '*****************************'+$
    'Fitting data set # '+StrCompress(indexLine+1)+' '+$
    '*****************************',$
    'Function Expression: '+model, errtxt, $
      'Number of data points: '+StrCompress(nPoints), $
    'Initial parameters: '+Vect2String(guess.value),$
    'Fixed params (0=free,1=fix):'+Vect2String(guess.fixed),$
      'Bottom limits (0=No,1=Yes):'+Vect2String( (guess.limited[0]) ),$
    'Top limits (0=No,1=Yes):'+Vect2String( (guess.limited[1]) ),$
    'Bottom limit:'+Vect2String( (guess.limits[0]) ),$
    'Top limit:'+Vect2String( (guess.limits[1]) ),$
    'Degrees of Freedom:'+ StrCompress(dof), $
    'Tolerance value: '+StrCompress(tol), $
    'Maximum fitting time [sec]: '+StrCompress(tmax), $
    'Maximum iterations: '+StrCompress(state.MPFitPars.maxIter), $
    '','']
ENDIF

;
; Call MPFit
;
r = mpfitexpr(model, dataX, dataY, dataErr, ParInfo=guess, $
  FTol=tol, XTol=tol, GTol=tol, $
  NPrint=state.MPFitPars.nprint, MaxIter=state.MPFitPars.maxIter, $
  IterProc='fufifa_mpfit_display', $
  IterArgs={t0:systime(1), tmax:tmax, parent:group}, $
  Status=status, $	; outputs
  PError=pError, $	; output
  nIter=nIter, $	; output
  yFit=yFit, $	; output
  quiet=quiet $
   )

CASE status OF
  -1:  text='Error/Aborted'
  0:  text='improper input parameters.'
  1:  text=['both actual and predicted relative reductions',$
        'in the sum of squares are at most FTOL(=TOL).']
  2:  text=['relative error between two consecutive iterates',$
        'is at most XTOL(=TOL)']
  3:  text=['both actual and predicted relative reductions',$
        'in the sum of squares are at most FTOL(=TOL). And, ',$
        'relative error between two consecutive iterates',$
        'is at most XTOL(=TOL)']
  4:  text=['the cosine of the angle between fvec and any column',$
        'of the jacobian is at most GTOL(=TOL) in absolute value.']
  5:  text='the maximum number of iterations has been reached'
  6:  text=['FTOL(=TOL) is too small. no further reduction in the ',$
        'sum of squares is possible.']
  7:  text=['XTOL(=TOL) is too small. no further improvement in the ',$
        'approximate solution x is possible.']
  8:  text=['GTOL(=TOL) is too small. fvec is orthogonal to the ',$
        'columns of the jacobian to machine precision.']
  else: text='Unknown.'
ENDCASE


dataFit=0
p=r
x=dataX
cmd = 'dataFit = '+model
itmp = Execute(cmd)
IF itmp NE 1 THEN BEGIN
  tmp = Dialog_Message(/Error,Dialog_Parent=group, $
		'Error executing : '+cmd+'. Aborted.')
  RETURN
ENDIF

widget_control,state.wids.draw, get_val = wnum
wset,wnum
OPlot,dataX,DataFit,Color=6

; pause
deviates = (dataY-yFit)/dataErr
chi2 = Total(deviates*deviates)
	

IF StrCompress( (state.MPFitPars.log)[0] ,/Rem) EQ '1' THEN BEGIN
  XDisplayFile1_Append,['','MPFit termination status: ',text,$
	  'Number of iterations: '+StrCompress(nIter),$
	  'Chi-square: '+StrCompress(chi2),$
	  'Chi-square/N/(N-1): '+StrCompress(chi2/npoints/(npoints-1)),$
	  'Goodness of fit (=igamma(0.5*dof,chi2) ): '+ $
		StrCompress(IGamma(0.5*dof,0.5*chi2)),$
	  '','Resulting params: ',Vect2String(r),$
	  '','1-sigma errors in params: ',Vect2String(pError),$
          '***************************'+$
          'End fitting data set # '+StrCompress(indexLine+1)+' '+$
          '***************************', ' ',$
	  '']
ENDIF


;
; update results
;
IF N_Elements(ask) EQ 0 THEN ask=0
IF Keyword_Set(ask) THEN BEGIN
  itmp = Dialog_Message(Dialog_Parent=group, /Question,$
	'Update parameters with fit results?' )
  IF itmp EQ 'No' THEN RETURN
ENDIF

Widget_Control,state.wids.dataModel,Set_Value='Model: defined.'

oeq->SetProperty,Param=r,/UpdateWidgets
state.oData->set,indexLine,Aux=r

END ; FuFiFa_fit


;
;============================================================================
;
PRO FuFiFa_WriteSpecFile, tlb, Group=group, File=file, model=Model

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_WriteSpecFile: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

IF N_Elements(model) EQ 0 THEN Model=1

Widget_Control,tlb,Get_UValue=state

oData = state.oData
oeq = state.oEquation

nn =  oData->info(/N_Elements)
IF nn EQ 1 and oData->value(0,/IsDataDef) NE 1 THEN BEGIN
  Message,'No data'
  RETURN
ENDIF 


;
; get data
;

FOR i=0L,nn-1 DO BEGIN
  data = oData->value(i)
  dataX = Reform(data[0,*])
  dataY = Reform(data[1,*])

  iModel=model
  IF iModel EQ 1 THEN BEGIN
    IF  state.oData->value(i,/IsAuxDef) EQ 1 THEN BEGIN
      aux = state.oData->value(i,/Aux)
    ENDIF ELSE BEGIN
      IF NOT(Keyword_Set(print)) THEN $
        itmp = Dialog_Message(/Info,Dialog_Parent=group,$
          ['No model defined for data index: '+StrCompress(i),$
          'Writing file with no model (only data)'])
      iModel=0
    ENDELSE
  ENDIF

  IF N_Elements(datax) EQ 0 THEN BEGIN
   Message,'Abscissas not defined'
   RETURN
  ENDIF

  nlines = oeq->GetProperty(/Nlines)
  IF iModel EQ 1 THEN BEGIN
    dataErr = fufifa_GetErrorArray(data,state.mpfitpars,Group=group)
    oeq = state.oequation

    IF nlines NE 0 THEN BEGIN
      ; 
      ; model
      ; 
      oeq->SetProperty,Param=aux ; ,/UpdateWidgets
      dataModel= oeq->evaluate(datax)

      oeqStr=oeq->GetProperty(/Str) ; Load structure with limits
      ; 
      ; residuals
      ; 
      dataResiduals = dataY-oeq->evaluate(datax)
    ENDIF ELSE BEGIN
      Message,'No model available.'
    RETURN
    ENDELSE
  ENDIF

  ; 
  ; write file
  ; 
  IF i EQ 0 THEN BEGIN ; main header
     OpenW,unit,file,/Get_Lun
     PrintF,unit,'#F '+file
     PrintF,unit,'#C File created by XOP/fufifa (functional fitting facility)'
     PrintF,unit,'#D '+SysTime()
     PrintF,unit,'#UFUFIFA_VERSION '+FuFiFa_Version()
     ; 
     ; Model information (as in file Model->Save to file)
     ; 
     IF model THEN BEGIN
       PrintF,unit,'#UFUFIFA_MODEL ; '
       PrintF,unit,'#UFUFIFA_MODEL ; fufifa-model data'
       PrintF,unit,'#UFUFIFA_MODEL ; Comments start with ";"'
       PrintF,unit,'#UFUFIFA_MODEL ; fufifa version: '+fufifa_version()
       PrintF,unit,'#UFUFIFA_MODEL ; Created on: '+SysTime()
       cd,Current=pwd
       PrintF,unit,'#UFUFIFA_MODEL ; Working on: '+pwd
       PrintF,unit, $
        '#UFUFIFA_MODEL ;'+$
        ' The lines below define the model functions and their fit/guess '
       PrintF,unit, $
        '#UFUFIFA_MODEL ;'+$
        '    parameters. They can be edited keeping the format unchanged'
       PrintF,unit,'#UFUFIFA_MODEL ; '
  
       FOR ii=0L,nlines-1 DO BEGIN
         name = oeq->GetPropertyLine(ii,/Names)
              tmp = name+' | '+oeq->GetPropertyLine(ii,/Template)+' | '+ $
                vect2string(oeq->GetPropertyLine(ii,/parLine))
              printf,unit,'#UFUFIFA_MODEL '+tmp
         ; print,tmp
       ENDFOR
       Widget_Control,state.wids.fitDomain,Get_Value=tmp
       IF StrCompress(tmp[0],/Rem) NE '' THEN $
	   PrintF,unit,'#UFUFIFA_FITDOMAIN '+tmp[0]
    ENDIF ; model=1
    ; 
    ; preferences menu
    ; 
    tmps = write_str(state.MPFitPars)
    PrintF,unit,'#UFUFIFA_PREFERENCES '+tmps

    PrintF,unit,''
  ENDIF ; main header


  nPoints = N_Elements(dataX)
  ;PrintF,unit,''
  PrintF,unit,'#S '+StrCompress(i+1,/Rem)+' '+oData->value(i,/Title)

  IF imodel EQ 1 THEN BEGIN
    chi2 = Total( (dataResiduals/dataErr)^2 )
    chi2 = chi2/nPoints/(nPoints-1)
  
    PrintF,unit,'#UFUFIFA_PAR '+Vect2String(aux)
    ;PrintF,unit,'#UFUFIFA_VALUE '+Vect2String(oeqStr.value)
    PrintF,unit,'#UFUFIFA_FIXED '+Vect2String(oeqStr.fixed)
    PrintF,unit,'#UFUFIFA_LIMITED0 '+Vect2String((oeqStr.limited[0]))
    PrintF,unit,'#UFUFIFA_LIMITED1 '+Vect2String((oeqStr.limited[1]))
    PrintF,unit,'#UFUFIFA_LIMITS0 '+Vect2String((oeqStr.limits[0]))
    PrintF,unit,'#UFUFIFA_LIMITS1 '+Vect2String((oeqStr.limits[1]))
    PrintF,unit,'#UFUFIFA_CHI2 '+StrCompress(chi2)
    PrintF,unit,'#N 5'
    PrintF,unit,'#L X  Y  ErrorY  Residuals  Model'
    FOR ii=0L,nPoints-1 DO BEGIN
      PrintF,unit, $
	  dataX[ii],dataY[ii],dataErr[ii],dataResiduals[ii],dataModel[ii],$
	  Format='(5(G0.6," "))'
    ENDFOR
  ENDIF ELSE BEGIN
    PrintF,unit,'#N 2'
    PrintF,unit,'#L X  Y  '
    FOR ii=0L,nPoints-1 DO BEGIN
      PrintF,unit, $
	  dataX[ii],dataY[ii], Format='(2(G0.6," "))'
    ENDFOR
  ENDELSE
ENDFOR


IF model THEN BEGIN
  pars = FuFiFa_GetModelParsVariation(tlb,error=error)
  IF error NE 1 THEN BEGIN
    npars = N_Elements(pars[*,0])
    nn1 = N_Elements(pars[0,*])
    colTitles = '  p['+StrCompress(SIndGen(npars-2),/Rem)+']'
    colTitles=['DataSetId',colTitles,'  R_Chi**2']
    colTitles2 = ''
    FOR ii=0L,N_Elements(colTitles)-1 DO BEGIN
        colTitles2 = colTitles2+colTitles[ii]
    ENDFOR
	      
    ; this part os not longer written to avoid problems when re-read (srio@esrf.eu, 2008/12/11)
    ; PrintF,unit,''
    ; PrintF,unit,'#S 1000 Parameter variation'
    ; PrintF,unit,'#N '+StrCompress(npars)
    ; PrintF,unit,'#L '+colTitles2
    ; FOR ii=0L,nn1-1 DO BEGIN
    ;     PrintF,unit, pars[*,ii], Format='('+StrCompress(npars,/rem)+'(G0.6," "))'
    ; ENDFOR
  ENDIF
ENDIF


Free_Lun,unit


END ; FuFiFa_WriteSpecFile

;
;============================================================================
;
PRO FuFiFa_fit_event, event

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_cwfit_event: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

Widget_Control,event.top,Get_UValue=state
Widget_Control,event.id,Get_UValue=uValue


indexLine=Widget_Info(state.wids.dataList,/List_Select)
IF indexLine[0] EQ -1 THEN BEGIN
  itmp = Dialog_message(/Error,Dialog_parent=event.top, $
	'No data to fit. Please Load/Select data')
  RETURN
ENDIF


;
; Start log window
;
IF uValue EQ '**Fit**' OR uvalue EQ 'Fit2' THEN BEGIN
  Widget_Control,/HourGlass
  IF StrCompress( (state.MPFitPars.log)[0] ,/Rem) EQ '1' THEN BEGIN
    XDisplayfile1,Text=['','Running MPFit on '+SysTime(),''], $
      Group=event.top, Title='Fitting results'
  ENDIF
ENDIF

;
; Performs fitting 
;

FOR i=0L,N_Elements(indexLine)-1 DO BEGIN
  CASE uValue OF
    '**Fit**': FuFiFa_Fit,state,indexLine[i], Group=event.top, Ask=0
    'Fit2': FuFiFa_Fit,state,indexLine[i],indexLine[(i-1)>0], Group=event.top, Ask=0
    else:
  ENDCASE
ENDFOR


END ; FuFiFa_fit_event

;
;============================================================================
;
PRO FuFiFa_parField_event, event

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_parField_event: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

Widget_Control,event.top,Get_UValue=state
Widget_Control,event.id,Get_Value=value
Widget_Control,event.id,Get_UValue=linenumber


oeq = state.oequation
oData = state.oData
;  print,oeq->info()

;
; set new numerial parameters
;
IF event.type EQ '' THEN BEGIN
  ;
  ; update equation object
  ;
  oeq->setPropertyLine,linenumber,param=value.coefs

  ;
  ; update data auxiliar
  ;
  sel = Widget_Info(state.wids.dataList,/List_Select)
  IF sel[0] NE -1 THEN BEGIN
    nn =  oData->info(/N_Elements)
    IF nn EQ 1 and oData->value(0,/IsDataDef) NE 1 THEN BEGIN
      Message,/Info,'No data'
      RETURN
    ENDIF 
    goodIndices = IndGen(nn)
    goodIndices = goodIndices(sel)
    all_pars=oeq->GetProperty()
    FOR i=0L,N_Elements(goodIndices)-1 DO BEGIN
      oData->Set,goodIndices[i],Aux=all_pars
    ENDFOR
  ENDIF
  ;
  ; Plot
  ;
  FuFiFa_Plot,event.top
  ;
  ; refresh zoom window (xwindow)
  ;
  IF Widget_Info( state.wids.xwindow,/Valid_Id) THEN xwindow_refresh, state.wids.xwindow


ENDIF


;
; delete line
;
IF event.type EQ 'x' THEN BEGIN
  itmp = Dialog_Message(/Info,/Cancel,Dialog_Parent=event.top,$
    'This option deletes an equation row. Please confirm.')
  IF itmp EQ 'Cancel' THEN RETURN
  wId=oeq->getPropertyLine(lineNumber,/WId)
  oeq->deleteLine,lineNumber
tmp = oeq->info()
  FuFiFa_ModelRefresh,event.top
  print,'NLINES AFTER: ',oeq->getProperty(/nLines)
ENDIF



;
; set new fix/free flag
;
IF event.type EQ 'F' THEN BEGIN
  ; event.index EQ 0 means unsensitive
  ; fixed_flag EQ 0 means "free"
  fixed_flag = abs(value.flags[event.index]-1)
  oeq->setPropertyItem,lineNumber,event.index,Fixed=fixed_flag
ENDIF
  
;
; set boundaries
;
IF event.type EQ 'B' THEN BEGIN
  ; event.index EQ 0 means unsensitive
  ; fixed_flag EQ 0 means "free"
  str = oeq->GetPropertyLine(lineNumber,/str)
  stri = str[event.index]
  xscrmenu,stri,Titles=['Value','Fixed flag (0=No,1=Yes)', $
                        'Limited flag (0):lower, (1):upper','Limits'],$
    /NoType,Dialog_Parent=event.top,WTitle='Edit boundaries',$
    Help=fufifa_text('Boundaries'),Action=action
    IF action EQ 'DONT' THEN RETURN
  str[event.index]=stri
  oeq->SetPropertyLine,lineNumber,str=str

  ; 
  ; map widget and copy values
  ;
  wId=oeq->GetPropertyLine(lineNumber,/wId)
  Widget_Control,wId,Get_Value=tmp
  coefs = tmp.coefs
  ff = tmp.flags
  IF stri.fixed THEN ff[event.index]=0 ELSE ff[event.index]=1
  coefs[event.index]=stri.value
  tmp.flags=ff
  tmp.coefs=coefs
  Widget_Control,wId,Set_Value=tmp
  

ENDIF
  
;
; get "guess" values
;
IF event.type EQ 'G' THEN BEGIN
  ;oeq->setPropertyItem,lineNumber,event.index,Fixed=fixed_flag
  name=oeq->getPropertyLine(lineNumber,/Names)

  CASE StrUpCase(name) OF
    'STRAIGHTLINE': mytype = 'TWOPOINTS'
    'GAUSSIAN': mytype='PEAK'
    'LORENTZIAN': mytype='PEAK'
    'PSEUDOVOIGT': mytype='PEAK'
    'STEPATAN': mytype='STEP'
    'STEPERRORF': mytype='STEP'
    else: mytype='NONE'
  ENDCASE
  
  CASE mytype OF
    'NONE': BEGIN
	itmp = Dialog_message(/Info,Dialog_Parent=event.top, $
		['No mouse selection allowed.', $
		 'Please type values and <enter>'])
	END
    'TWOPOINTS': BEGIN
	fufifa_plot,event.top,Group=event.top
	IF StrCompress( (state.mpfitpars.expert)[0],/Rem) EQ '0' THEN BEGIN 
	  itmp = Dialog_message(/Info,Dialog_Parent=event.top, $
		'Enter two points with mouse')
	ENDIF
	Cursor,x1,y1,Wait=3,/Data
	OPLot,[x1,x1],[y1,y1],Color=1,Psym=2
	Wait,0.2
	Cursor,x2,y2,Wait=3,/Data
	OPLot,[x2,x2],[y2,y2],Color=1,Psym=2
	p = value.coefs
	p[1]= (y2-y1)/(x2-x1)
	p[0]= y1 - p[1]*x1
	value.coefs = p
	Widget_Control,event.id,Set_Value=value
	oeq->setPropertyLine,lineNumber,param=p
	END
    'PEAK': BEGIN
	fufifa_plot,event.top,Group=event.top
        IF StrCompress( (state.mpfitpars.expert)[0] ) EQ 0 THEN $
	  itmp = Dialog_message(/Info,Dialog_Parent=event.top, $
		['Enter three points with mouse: ', $
		 ' 1st at the left side of the peak, at half width',$
		 ' 2nd at the top of the peak',$
		 ' 3rd at the right side of the peak, at half width'])
	Cursor,xL,yL,Wait=3,/Data
;itmp = Dialog_message('Selected points: '+vect2string([xL,yL]))
	OPLot,[xL,xL],[yL,yL],Color=1,Psym=2
;itmp = Dialog_message('OPLOTED')
	Wait,0.2
;itmp = Dialog_message('WAITED')
	Cursor,xC,yC,Wait=3,/Data
;itmp = Dialog_message('Selected points: '+vect2string([xL,yL]))
	OPLot,[xC,xC],[yC,yC],Color=1,Psym=2
;itmp = Dialog_message('OPLOTED')
	Wait,0.2
;itmp = Dialog_message('WAITED')
	Cursor,xR,yR,Wait=3,/Data
;itmp = Dialog_message('Selected points: '+vect2string([xL,yL]))
	OPLot,[xR,xR],[yR,yR],Color=1,Psym=2
;itmp = Dialog_message('OPLOTED')
	Wait,0.2
;itmp = Dialog_message('WAITED')
	p = value.coefs
	p[0] = 2.0*yC-(yL+yR)
	p[1] = xC
	p[2] = abs(xR-xL)
	value.coefs = p
	Widget_Control,event.id,Set_Value=value
	oeq->setPropertyLine,lineNumber,param=p
	END
    'STEP': BEGIN
	fufifa_plot,event.top,Group=event.top
	itmp = Dialog_message(/Info,Dialog_Parent=event.top, $
		['Enter three points with mouse: ', $
		 ' 1st at the bottom-left side of the step',$
		 ' 2nd at the center (half-height) of the step',$
		 ' 3rd at the right side of the step, at top '])
	Cursor,xL,yL,Wait=3,/Data
	OPLot,[xL,xL],[yL,yL],Color=1,Psym=2
	Wait,0.2
	Cursor,xC,yC,Wait=3,/Data
	OPLot,[xC,xC],[yC,yC],Color=1,Psym=2
	Wait,0.2
	Cursor,xR,yR,Wait=3,/Data
	OPLot,[xR,xR],[yR,yR],Color=1,Psym=2
	Wait,0.2
	p = value.coefs
	p[0] = yR
	p[1] = xC
	p[2] = abs(xR-xL)
	value.coefs = p
	Widget_Control,event.id,Set_Value=value
	oeq->setPropertyLine,lineNumber,param=p
	END
  ENDCASE

  ;
  ; update data auxiliar
  ;
  sel = Widget_Info(state.wids.dataList,/List_Select)
  IF sel[0] NE -1 THEN BEGIN
    nn =  oData->info(/N_Elements)
    IF nn EQ 1 and oData->value(0,/IsDataDef) NE 1 THEN BEGIN
      Message,/Info,'No data'
      RETURN
    ENDIF 
    goodIndices = IndGen(nn)
    goodIndices = goodIndices(sel)
    all_pars=oeq->GetProperty()
    FOR i=0L,N_Elements(goodIndices)-1 DO BEGIN
      oData->Set,goodIndices[i],Aux=all_pars
    ENDFOR
  ENDIF
  ; 
  ; Plot
  ;
  FuFiFa_Plot,event.top
ENDIF
  
END



;
;============================================================================
;

PRO fufifa_modelload,wId,Commands=commands

	Widget_Control,wId, Get_UValue=state
	oeq = state.oequation

	;
	; clear equtions
	;
	nlines=oeq->GetProperty(/NLines)
	FOR i=0L,nlines-1 DO oeq->deleteLine

	; clear all stored data
	nn =  state.oData->info(/N_Elements)
	IF nn EQ 1 and state.oData->value(0,/IsDataDef) NE 1 THEN BEGIN
	  ;Message,/Info,'No data'
	  ;RETURN
	ENDIF ELSE BEGIN
	  FOR i=0L,nn-1 DO state.oData->Set,i,/ClAux
	ENDELSE

IF Not(Keyword_Set(Commands)) THEN BEGIN
	;
	; access to file
	;
	dir = state.importDir
	file = Dialog_PickFile(Title='Select a fufifa-model data file',$
            Dialog_Parent=wId,PATH=dir,GET_PATH=dir, $
	    FILTER='*.dat')
        IF file EQ '' THEN RETURN
        Widget_Control,/Hourglass
	commands = Read_Textfile(file)

	state.importDir = dir
	Widget_Control,wId,Set_UValue=state
ENDIF

FOR i=0L,N_Elements(commands)-1 DO BEGIN
  stmp = commands[i]
  IF StrMid(StrCompress(stmp,/Rem),0,1) NE ';' THEN BEGIN
    n = StrParse(stmp,'|',list)
    IF n NE 2 THEN Message,'File command not understood: '+tmp
    name = list[0]
    equation = list[1] 
    ;pars = list[2]
    pars2 = 0
    itmp = Execute('pars2='+list[2])
    IF itmp NE 1 THEN Message,'Error executing '+'pars2='+list[2]
    ;degree=N_Elements(pars2)-1
 
    fufifa_addfunction,wId,name=name,equation=equation,param=pars2, $
	noAsk=1,ntimes=1
  ENDIF
ENDFOR
END ; fufifa_modelload


;
;============================================================================
;

PRO fufifa_modelrefresh,wId
	    Widget_Control,wId, Get_UValue=state
	    oeq = state.oequation
            nLines = oeq->GetProperty(/nLines)
	    ;
	    ; deleting widgets
	    ;
	    FOR i=0,nLines-1 DO BEGIN
		wId = oeq->GetPropertyLine(i,/wId)
		Widget_Control,wId,/destroy
	    ENDFOR
	    ;
	    ; re-create widgets
	    ;
	    FOR i=0,nLines-1 DO BEGIN
	      nlines=oeq->GetProperty(/NLines)
	      nindices=oeq->GetProperty(/nindices)
              IF i EQ 0 THEN istart=0 ELSE istart=Fix(Total( nIndices[0:i-1] ))
	      line = oeq->GetPropertyLine(i,/equation)
	      name = oeq->GetPropertyLine(i,/name)
	      values = oeq->GetPropertyLine(i,/parLine)
	      flags=Fix(values*0+1)
	      value={coefs:values, flags:flags }
	      wtmp =CW_PField(state.wids.ModelBase,Column=0, $
		      UValue=i, $	; line number info in UValue
		      Value=value,Title=[name,line],PosTitle=1,$
		      Frame=1,$
		      pTitles='p['+StrCompress(istart+IndGen(N_Elements(values)),/Rem)+']:' )
	      Widget_Control,wtmp,/Realize
	      oeq->SetPropertyLine,i,wId=wtmp
	    ENDFOR

END ; FuFiFa_ModelRefresh
;
;============================================================================
;

PRO fufifa_loadfile,parent,isel=isel,Dialog_Parent=dialog_Parent, $
  append=append, $
  file=file, $              ; for isel=0,1
  data=data,title=title     ; for isel=2

;+
; PRO fufifa_loadfile,parent,isel=isel,Dialog_Parent=dialog_Parent,file=file
;
; 06-03-02 srio@esrf.fr
; 08-12-11 srio@esrf.eu import preferences from spec file
;
; isel: 0 ASCII FILE(S)
; isel: 1 SPEC CLEANED FILE
; isel: 2 DATA block (use Title keyword)
;
;-
IF N_Elements(isel) EQ 0 THEN isel=0
Widget_Control,parent,Get_UValue=state

CASE isel OF
  0: BEGIN ; ascii files
        dir = state.importDir
	IF N_Elements(file) EQ 0 THEN BEGIN
	  file = Dialog_PickFile(Title='Select ASCII file(s)',$
            Dialog_Parent=dialog_parent,PATH=dir,GET_PATH=dir, $
	    Multiple_Files=1)
	ENDIF
        IF file[0] EQ '' THEN RETURN
        Widget_Control,/Hourglass

	nfiles = N_Elements(file)
        IF nfiles EQ 1 THEN title=file[0] ELSE title='ASCII files imported'
	Widget_Control,parent,TLB_Set_Title= $
            'FuFiFa '+FuFiFa_Version()+title
	oData = state.oData
	nn =  oData->info(/N_Elements)

	IF nn EQ 1 and oData->value(0,/IsDataDef) NE 1 THEN append = 0
	IF N_Elements(append) EQ 0 THEN BEGIN
            itmp = Dialog_Message(/Question, DIALOG_PARENT=dialog_parent, $
              ['Append new data to existing one ?',$
	       '( if not, existing data will be removed )'])
	    IF itmp EQ 'Yes' THEN append=1 else append=0
	ENDIF

	FOR ifile=0,nfiles-1 DO BEGIN
	  data = RAscii(file[ifile])
	  IF append EQ 0 THEN BEGIN
	    Obj_Destroy,oData
	    oData = Obj_New('ompp',nTimes=1)
	    oData->set,0,Title=' ASCII file: '+file[ifile],Value=data
	  ENDIF ELSE BEGIN
	     oData->set,/Add,Title=' ASCII file: '+file[ifile],Value=data
	  ENDELSE
	  append=1
	ENDFOR
  
	; update list widget
	nData = oData->info(/N_Elements)
	list01 = StrArr(nData)
	FOR i=0L,nData-1 DO BEGIN
	  list01[i] = oData->value(i,/Title)
	ENDFOR
        Widget_Control,state.wids.dataList,Set_Value=list01


        state.specFile=file[nfiles-1]
        state.importDir=dir
	state.oData=oData
	Widget_Control,parent, Set_UValue=state

	widget_control,state.wids.draw, get_val = wnum
	wset,wnum
	erase
	
	END
  1: BEGIN ; spec cleaned file
        dir = state.importDir
	IF N_Elements(file) EQ 0 THEN BEGIN
	  file = Dialog_PickFile(Title='Select a SPEC file',$
            Dialog_Parent=dialog_parent,PATH=dir,GET_PATH=dir)
	ENDIF
        IF file EQ '' THEN RETURN
        Widget_Control,/Hourglass
        Widget_Control,parent,TLB_Set_Title= $
            'FuFiFa '+FuFiFa_Version()+' - Imported: '+file


        n = Spec_Access(hSpec,file)
        IF n EQ 0 THEN BEGIN
          itmp = Dialog_Message(/Error, DIALOG_PARENT=dialog_parent, $
            ['FUFIFA_EVENT: Error reading SPEC data file: ',file])
          RETURN
        ENDIF

	iLoadModel=0
	;
	; Load model (if defined)
	;
	tmp = spec_headers(hSpec,1,/Index,/All,'UFUFIFA_MODEL',/Label_Remove)
	IF N_Elements(tmp) GT 1 THEN BEGIN
	   itmp = Dialog_Message(/Question,Dialog_Parent=parent, $
		'Fitting model is defined. Do you want to load it?') 
	   IF itmp EQ 'Yes' THEN iLoadModel=1 
	ENDIF
	IF iLoadModel THEN BEGIN
		tmp = Spec_Headers(hSpec,1,/Index,/All,'UFUFIFA_MODEL',/Label_Rem)
	        fufifa_modelload,parent,commands=tmp
		strFixed=0
		tmp = Spec_Headers(hSpec,1,/Index,/All,'UFUFIFA_FIXED',/Label_Rem)
		itmp = Execute('strFixed = '+tmp)
		strLimited0=0
		tmp = Spec_Headers(hSpec,1,/Index,/All,'UFUFIFA_LIMITED0',/Label_R)
		itmp = Execute('strLimited0 = '+tmp)
		strLimited1=0
		tmp = Spec_Headers(hSpec,1,/Index,/All,'UFUFIFA_LIMITED1',/Label_R)
		itmp = Execute('strLimited1 = '+tmp)
		strLimits0=0
		tmp = Spec_Headers(hSpec,1,/Index,/All,'UFUFIFA_LIMITS0',/Label_R)
		itmp = Execute('strLimits0 = '+tmp)
		strLimits1=0
		tmp = Spec_Headers(hSpec,1,/Index,/All,'UFUFIFA_LIMITS1',/Label_R)
		itmp = Execute('strLimits1 = '+tmp)
		str = {value:0D, fixed:0, limited:[0,0], limits:[0D,0D]}
		npars = N_Elements(strFixed)
		strArr = Replicate(str,npars)
		FOR ii=0,npars-1 DO BEGIN
		   istr=strArr[ii]
		   istr.Fixed=strFixed[ii]
		   istr.Limited=[strLimited0[ii],strLimited1[ii]]
		   istr.Limits=[strLimits0[ii],strLimits1[ii]]
		   strArr[ii]=istr
		ENDFOR
	      	(state.oequation)->SetProperty,Str=strArr
		tmp = Spec_Headers(hSpec,1,/Index,/All,'UFUFIFA_FITDOMAIN',/Label_R)
		IF StrCompress(tmp,/Rem) NE '' THEN $
			Widget_Control,state.wids.fitDomain,Set_Value=tmp

                ; 
                ; map widget 
                ;
                wIds=(state.oequation)->GetProperty(/wId)
		k=0L
		FOR i=0L,N_Elements(wIds)-1 DO BEGIN
                  wId=wIds[i]
                  Widget_Control,wId,Get_Value=tmp
                  ff = tmp.flags
                  FOR j=0L,N_Elements(ff)-1 DO BEGIN
			IF strFixed[k] THEN ff[j]=0 ELSE ff[j]=1
			k=k+1
		  ENDFOR
                  tmp.flags=ff
                  Widget_Control,wId,Set_Value=tmp
		ENDFOR
  
	ENDIF


	;
	; Load preferences (if defined)
	;
	tmp = spec_headers(hSpec,1,/Index,/All,'UFUFIFA_PREFERENCES',/Label_Remove)
        iLoad=0
	IF N_Elements(tmp) GT 1 THEN BEGIN
	   itmp = Dialog_Message(/Question,Dialog_Parent=parent, $
		'FuFiFa preferences are defined in input file. Do you want to load it?') 
	   IF itmp EQ 'Yes' THEN iLoad=1 
	ENDIF
	IF iLoad THEN BEGIN
           tmp = Spec_Headers(hSpec,1,/Index,/All,'UFUFIFA_PREFERENCES',/Label_Rem)
           strNew = Make_Str(tmp)
           str = state.MPFitPars
           copy_structure,strNew,str,/OnlyFirstField
           state.MPFitPars=str
	ENDIF
	;
	; Load data
	;
	oData = state.oData

        list0 = '#S '+StrCompress(hSpec.scan.scan_n,/Rem)+' - '
        list1 = hSpec.scan.name
        ;list01= list0+list1
        list01= list1

	nData = N_Elements(list01)



	nn =  oData->info(/N_Elements)
	IF nn EQ 1 and oData->value(0,/IsDataDef) NE 1 THEN append = 0
	IF N_Elements(append) EQ 0 THEN BEGIN
            itmp = Dialog_Message(/Question, DIALOG_PARENT=dialog_parent, $
              ['Append new data to existing one ?',$
	       '( if not, existing data will be removed )'])
	    IF itmp EQ 'Yes' THEN append=1 else append=0
	ENDIF

	IF append EQ 0 THEN BEGIN
	  Obj_Destroy,oData
	  oData = Obj_New('ompp',nTimes=nData)
	  FOR i=0L,nData-1 DO BEGIN
             data = Spec_Data(hSpec,i+1,/Index)
	     aux=0
	     IF iLoadModel THEN BEGIN
		tmp = Spec_Headers(hSpec,i+1,/Index,'UFUFIFA_PAR',/Label_Remove)
		IF StrCompress(tmp,/Rem) NE '' THEN itmp = Execute('aux = '+tmp)
	     ENDIF
	     oData->set,i,Title=list01[i],Value=data,Aux=aux
	  ENDFOR
          ; Widget_Control,state.wids.dataList,Set_Value=list01
	ENDIF ELSE BEGIN
	  FOR i=0L,nData-1 DO BEGIN
             data = Spec_Data(hSpec,i+1,/Index)
	     aux=0
	     IF iLoadModel THEN BEGIN
		tmp = Spec_Headers(hSpec,i+1,/Index,'UFUFIFA_PAR',/Label_Remove)
		IF StrCompress(tmp,/Rem) NE '' THEN itmp = Execute('aux = '+tmp)
	     ENDIF
	     oData->set,/Add,Title=list01[i],Value=data,Aux=aux
	  ENDFOR
	ENDELSE

	; update list widget
	nData = oData->info(/N_Elements)
	list01 = StrArr(nData)
	FOR i=0L,nData-1 DO BEGIN
	  list01[i] = 'id:'+StrCompress(i,/Rem)+' '+oData->value(i,/Title)
	ENDFOR
        Widget_Control,state.wids.dataList,Set_Value=list01


        state.specFile=file
        state.importDir=dir
	state.oData=oData
	Widget_Control,parent, Set_UValue=state

	widget_control,state.wids.draw, get_val = wnum
	wset,wnum
	erase
	fufifa_list_select,parent,-1
	fufifa_plot,parent
	END


  2: BEGIN ; data block
	IF N_Elements(data) EQ 0 OR N_Elements(title) EQ 0 THEN BEGIN
            itmp = Dialog_Message(/Error, DIALOG_PARENT=dialog_parent, $
              'FUFIFA_LOADFILE: Define data block AND title (isel=2)')
	    Return
	ENDIF

	Widget_Control,parent,TLB_Set_Title= $
            'FuFiFa '+FuFiFa_Version()+'Data block imported'
	oData = state.oData
	nn =  oData->info(/N_Elements)

	IF nn EQ 1 and oData->value(0,/IsDataDef) NE 1 THEN append = 0
	IF N_Elements(append) EQ 0 THEN BEGIN
            itmp = Dialog_Message(/Question, DIALOG_PARENT=dialog_parent, $
              ['Append new data to existing one ?',$
	       '( if not, existing data will be removed )'])
	    IF itmp EQ 'Yes' THEN append=1 else append=0
	ENDIF

	IF append EQ 0 THEN BEGIN
	  Obj_Destroy,oData
	  oData = Obj_New('ompp',nTimes=1)
	  oData->set,0,Title=title,Value=data
	ENDIF ELSE BEGIN
	   oData->set,/Add,Title=title,Value=data
	ENDELSE
  
	; update list widget
	nData = oData->info(/N_Elements)
	list01 = StrArr(nData)
	FOR i=0L,nData-1 DO BEGIN
	  list01[i] = oData->value(i,/Title)
	ENDFOR
        Widget_Control,state.wids.dataList,Set_Value=list01


        state.specFile='<variable>'
        ;state.importDir=dir
	state.oData=oData
	Widget_Control,parent, Set_UValue=state

	widget_control,state.wids.draw, get_val = wnum
	wset,wnum
	erase
	
	END


ENDCASE

END ; fufifa_loadfile

;
;============================================================================
;
PRO FuFiFa_event, event

COMMON fufifa, strDataOperations, addFunctionMenu, strDataCut

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa_event: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

IF Tag_Names(event,/Structure_Name) EQ 'WIDGET_KILL_REQUEST' THEN BEGIN
  FuFiFa_Quit,event
  RETURN
ENDIF

Widget_Control,event.id,Get_UValue=uvalue


CASE uValue OF
  'SaveFile': BEGIN
	Widget_Control,event.top,Get_UValue=state
	IF StrCompress( (state.mpfitpars.expert)[0],/Rem) EQ '0' THEN BEGIN
	  itmp = Dialog_Message(/Info,/Cancel,Dialog_Parent=event.top,$
	    ['This option creates a file (SPEC format) containing all data ',$
	     'plus model information.'])
	  IF itmp EQ 'Cancel' THEN RETURN
	ENDIF

        dir = state.importDir
back1:
	file = Dialog_PickFile(/Write, file='fufifa.spec',$
            Dialog_Parent=event.top,PATH=dir,GET_PATH=dir)
        IF file EQ '' THEN RETURN
	IF CheckFile(file) EQ 1 THEN BEGIN
	  itmp = Dialog_Message(/Info,/Question,Dialog_Parent=event.top,$
	    ['File exists:',file,' ',$
	   'Overwrite it?'])
	  IF itmp EQ 'No' THEN GoTo,back1
	ENDIF
        Widget_Control,/Hourglass
	fufifa_WriteSpecFile,event.top,File=file,Group=event.top
        state.importDir=dir
	Widget_Control,event.top, Set_UValue=state
	END
  'SaveFileOnlyData': BEGIN
	itmp = Dialog_Message(/Info,/Cancel,Dialog_Parent=event.top,$
	  ['This option creates a file (SPEC format) containing all data ',$
	   '(no model information)'])
	IF itmp EQ 'Cancel' THEN RETURN

	Widget_Control,event.top,Get_UValue=state
        dir = state.importDir
	file = Dialog_PickFile(/Write, file='fufifa.spec',$
            Dialog_Parent=event.top,PATH=dir,GET_PATH=dir)
        IF file EQ '' THEN RETURN
        Widget_Control,/Hourglass
	fufifa_WriteSpecFile,event.top,File=file,Group=event.top,Model=0
        state.importDir=dir
	Widget_Control,event.top, Set_UValue=state
	END

  'Quit': BEGIN
	FuFiFa_Quit,event
	END

  'Refresh': BEGIN
	Widget_Control,event.top,Get_UValue=state
	IF (where(tag_names(event) EQ 'CLICKS'))[0] NE '-1' THEN BEGIN
	 IF event.clicks EQ 2 THEN BEGIN
	  nn =  state.oData->info(/N_Elements)
	  oData = state.oData
	  IF nn EQ 1 and oData->value(0,/IsDataDef) NE 1 THEN RETURN
	  sel = Widget_Info(state.wids.dataList,/List_Select)
	  goodIndices = sel(N_Elements(sel)-1)
	  oData = state.oData
	  data = oData->value(goodIndices[0],title=title)
	  title = oData->value(goodIndices[0],/title)
	  idx = FIndGen(N_Elements(data[0,*]))
	  ;xplot,Make_Set(Reform(data[0,*]),Reform(data[1,*]),idx) $
	  xplot,data,/no_block,Wtitle=title,YCol=2 ;ColTitles=['X','Y','index']
	 ENDIF
	ENDIF
	;Widget_Control,event.top,Get_UValue=state
	fufifa_plot,event.top,Group=event.top
        ;
        ; refresh zoom window (xwindow)
        ;
        IF Widget_Info( state.wids.xwindow,/Valid_Id) THEN xwindow_refresh, state.wids.xwindow

	END

  'PlotZoom': BEGIN
	printBuffer = 'fufifa_plot,'+StrCompress(event.top,/Rem)+'L'
	printBuffer = printBuffer+',/Print,'+$
	  'YRange=[parms.ymin,parms.ymax],'+$
	  'XRange=[parms.xmin,parms.xmax]'
	parms={xmin:0.0,xmax:0.0,ymin:0.0,ymax:0.0}
        xwindow,parent=p, b=printBuffer,/Edit,/ZoomFlag, $
	  parms=parms,Group=event.top,$
	  wtitle='FuFiFa ZOOM',Dialog_Parent=event.top
        ;
        ; store the xwindow widget parent to automatically update them
        ;
        Widget_Control,event.top, Get_UValue=state
        state.wids.xwindow = p
        Widget_Control,event.top, Set_UValue=state

	END
  'PlotPrint': BEGIN
	Widget_Control,event.top,Get_UValue=state
	sel = Widget_Info(state.wids.dataList,/List_Select)
	multiplePrint=0

	IF N_Elements(sel) GT 1 THEN BEGIN
	  itmp = Dialog_Message(Dialog_Parent=event.top,/Question,  $
	    /Cancel, $
	    ['Multiple data sers are selected',$
	     'Do you want a multiple page plot? ','',$
	     '   Yes: multiple page',$ 
	     '   No: single page',$ 
	     '   Cancel: Abort'])
	  CASE itmp OF
	    'Cancel': RETURN 
	    'Yes': multiplePrint=1
	    else: 
	  ENDCASE
	ENDIF

	printBuffer = 'fufifa_plot,'+StrCompress(event.top,/Rem)+'L'
	printBuffer = printBuffer+',/Print,MultiplePrint='+StrCompress(multiplePrint,/Rem)
	xprint,BUFFER=printbuffer,GROUP=event.top,/COLOR
	END

  'DataLoad': fufifa_loadfile,event.top,isel=1,Dialog_Parent=event.top

  'DataLoadASCII': fufifa_loadfile,event.top,isel=0,Dialog_Parent=event.top

  'DataOperations': BEGIN
	Widget_Control,event.top, Get_UValue=state
	IF N_Elements(strDataOperations) EQ 0 THEN BEGIN
	  strDataOperations = { $
		apply:['0','On all scans','On selected scans only'],$
		expressionX:'x=x',expressionY:'y=y/max(y)',$
                expressionErr:''}
	ENDIF
        flags=Replicate('1',N_tags(strDataOperations))
        titles=['Apply operation','Expression: New x=f(x)', $
	        'Expression: New y=f(x,y)','Error expression: err=f(x,y)']
        XScrmenu,strDataOperations,/Interp,/NoType,Action=action, $
	  group=event.top, Help=fufifa_text('DataOperations'), $
          Flags=flags, Titles=titles, WTitle='Data operations',$
	  Dialog_Parent=event.top,FieldLen=50
        IF action EQ 'DONT' THEN RETURN

	nn =  state.oData->info(/N_Elements)
	oData = state.oData
	IF nn EQ 1 and oData->value(0,/IsDataDef) NE 1 THEN BEGIN
	  Message,/Info,'No data'
	  RETURN
	ENDIF
	goodIndices = IndGen(nn)
	CASE StrCompress(strDataOperations.apply[0],/Rem) OF
	  '0':BEGIN 	; on all scans
		END
	  '1':BEGIN 	; on selected scans
		sel = Widget_Info(state.wids.dataList,/List_Select)
		IF sel[0] EQ -1 THEN BEGIN
		  itmp = Dialog_message(/Error, Dialog_Parent=event.top, $
			'Please select a data set.')
		  RETURN
		ENDIF
		goodIndices = goodIndices(sel)
		END
	ENDCASE
        Widget_Control,/HourGlass
	FOR __i=0L,N_Elements(goodIndices)-1 DO BEGIN
	   oData = state.oData
	   __data = oData->value(goodIndices[__i])
	   x = Reform(__data[0,*])
	   y = Reform(__data[1,*])
	   y0 = y
	   x0 = x
	   __itmp = Execute(strDataOperations.expressionx)
	   IF __itmp NE 1 THEN BEGIN
	     __tmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		'Error executing: '+strDataOperations.expressionx)
	     RETURN
	   ENDIF
	   __itmp = Execute(strDataOperations.expressiony)
	   IF __itmp NE 1 THEN BEGIN
	     __tmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		'Error executing: '+strDataOperations.expressiony)
	     RETURN
	   ENDIF
           IF StrCompress(strDataOperations.expressionErr,/Rem) NE '' THEN BEGIN
             y1=y
             y=y0
             x1=x
             x=x0
             IF N_Elements(__data[*,0]) GT 2 THEN err=Reform(__data[2,*]) ELSE err = y*0
	     __itmp = Execute(strDataOperations.expressionErr)
	     IF __itmp NE 1 THEN BEGIN
	       __tmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
		  'Error executing: '+strDataOperations.expressionErr)
	       RETURN
	     ENDIF
             IF N_Elements(err) EQ 1 THEN err=y*0+err
	     oData->Set,goodIndices[__i],Value=Make_Set(x1,y1,err)
             Message,/Info,'Error column added.'
           ENDIF ELSE BEGIN
	    oData->Set,goodIndices[__i],Value=Make_Set(x,y)
           ENDELSE
	ENDFOR

	FuFiFa_Plot,event.top
	
	END

  'DataCut': BEGIN
	IF N_Elements(strDataCut) EQ 0 THEN BEGIN
	  itmp = Dialog_Message(/Info,/Cancel,Dialog_Parent=event.top, $
	    ['This options permits to select an abscissas interval.',$
            'Data points external to this interval will be eliminated.',$
            'This option could be used to eliminate unwanted points before fitting.'])
	  IF itmp EQ 'Cancel' THEN RETURN
	  strDataCut = {apply:['0','On all scans','On selected scans only'],$
		selection:['0','with mouse','with keyboard'],$
		min:0.0, max:0.0 }
	ENDIF
	Widget_Control,event.top, Get_UValue=state
        flags=['1','1','w(1) EQ 1','w(1) EQ 1']
        titles=['Apply operation','Point selection', $
	   'Minumum value:','Maximum value']
        XScrmenu,strDataCut,/Interp,/NoType,Action=action,group=event.top, $
          Flags=flags, Titles=titles, WTitle='Data cut',$
	  Dialog_Parent=event.top
        IF action EQ 'DONT' THEN RETURN

	nn =  state.oData->info(/N_Elements)
	IF nn EQ 1 and state.oData->value(0,/IsDataDef) NE 1 THEN BEGIN
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
          'No data found.')
	  RETURN
	ENDIF

	goodIndices = IndGen(nn)
	CASE StrCompress(strDataCut.apply[0],/Rem) OF
	  '0':BEGIN 	; on all scans
		END
	  '1':BEGIN 	; on selected scans
		sel = Widget_Info(state.wids.dataList,/List_Select)
		IF sel[0] EQ -1 THEN BEGIN
		  itmp = Dialog_message(/Error, Dialog_Parent=event.top, $
			'Please select a data set.')
		  RETURN
		ENDIF
		goodIndices = goodIndices(sel)
		END
	ENDCASE

	IF StrCompress(strDataCut.selection[0],/Rem) EQ '0' THEN BEGIN ; mouse
	  fufifa_plot,event.top,Group=event.top
	  itmp = Dialog_message(/Info,/Cancel,Dialog_Parent=event.top, $
		'Enter two points with mouse')
	  IF itmp EQ 'Cancel' THEN RETURN
	  Cursor,x1,y1,Wait=3,/Data
	  OPLot,[x1,x1],[y1,y1],Color=1,Psym=2
	  Wait,0.2
	  Cursor,x2,y2,Wait=3,/Data
	  OPLot,[x2,x2],[y2,y2],Color=1,Psym=2
	  Wait,0.2
	  strDataCut.min=x1
	  strDataCut.max=x2
	ENDIF

	IF goodIndices[0] EQ -1 THEN BEGIN
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
          'No data selected.')
          RETURN
	ENDIF

	FOR i=0L,N_Elements(goodIndices)-1 DO BEGIN
	  oData = state.oData
	  data = oData->value(goodIndices[i])
	  xx = Reform(data[0,*])
	  igood = Where(xx GE strDataCut.min AND xx LE strDataCut.max)
	  IF igood[0] EQ -1 THEN BEGIN
	    itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	      'Error in selecting interval: no good points ')
	    RETURN
	  ENDIF
	  tmpset=data[*,igood]
	  npoints1 = N_Elements(data[0,*])
	  npoints2 = N_Elements(tmpset[0,*])
	  IF i EQ 0 THEN BEGIN
	    IF N_Elements(goodIndices) GT 1 THEN $
		add='(for the first selected data set)' ELSE add=''
	    itmp = Dialog_Message(/Question,Dialog_Parent=event.top, $
	      ['Number of abscissa points reduced from '+$
	      StrCompress(npoints1,/Rem)+' to '+ $
	      StrCompress(npoints2,/Rem),add,$
	      ' ','Accept the operation (No Undo)?'])
	    IF itmp EQ 'No' THEN RETURN
	  ENDIF
	  oData->Set,goodIndices[i],Value=tmpset
	ENDFOR
	FuFiFa_Plot,event.top
	END


  'DataBackground': BEGIN
	itmp = Dialog_Message(/Info,/Cancel,Dialog_Parent=event.top, $
	  ['This options permits to substract a (fitted) straigt line. '])
	IF itmp EQ 'Cancel' THEN RETURN

	Widget_Control,event.top, Get_UValue=state

	str = {apply:['0','On all scans','On selected scans only'],$
		selection:['0','with mouse','with keyboard'],$
		ask:['0','No','Yes'],$
		min:0.0, max:0.0 }
        flags=['1','1','1','w(1) EQ 1','w(1) EQ 1']
        titles=['Apply operation','Point selection','Confirm each fit?', $
	   'Min value for fitting interval:',$
	   'Max value for fitting interval:']
        XScrmenu,str,/Interp,/NoType,Action=action,group=event.top, $
          Flags=flags, Titles=titles, WTitle='Substract linear background',$
	  Dialog_Parent=event.top
        IF action EQ 'DONT' THEN RETURN

	nn =  state.oData->info(/N_Elements)
	IF nn EQ 1 and state.oData->value(0,/IsDataDef) NE 1 THEN BEGIN
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
          'No data found.')
	  RETURN
	ENDIF

	goodIndices = IndGen(nn)
	CASE StrCompress(str.apply[0],/Rem) OF
	  '0':BEGIN 	; on all scans
		END
	  '1':BEGIN 	; on selected scans
		sel = Widget_Info(state.wids.dataList,/List_Select)
		IF sel[0] EQ -1 THEN BEGIN
		  itmp = Dialog_message(/Error, Dialog_Parent=event.top, $
			'Please select a data set.')
		  RETURN
		ENDIF
		goodIndices = goodIndices(sel)
		END
	ENDCASE

	IF StrCompress(str.selection[0],/Rem) EQ '0' THEN BEGIN ; mouse
	  fufifa_plot,event.top,Group=event.top
	  itmp = Dialog_message(/Info,/Cancel,Dialog_Parent=event.top, $
		'Enter two points with mouse')
	  IF itmp EQ 'Cancel' THEN BEGIN
	    FuFiFa_Plot,event.top
	    RETURN
	  ENDIF
	  Cursor,x1,y1,Wait=3,/Data
	  OPLot,[x1,x1],[y1,y1],Color=1,Psym=2
	  Wait,0.2
	  Cursor,x2,y2,Wait=3,/Data
	  OPLot,[x2,x2],[y2,y2],Color=1,Psym=2
	  Wait,0.2
          str.min = x1
          str.max = x2
	ENDIF

	IF goodIndices[0] EQ -1 THEN BEGIN
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
          'No data selected.')
          RETURN
	ENDIF

	FOR i=0L,N_Elements(goodIndices)-1 DO BEGIN
	  oData = state.oData
	  data = oData->value(goodIndices[i])
	  xx = Reform(data[0,*])
	  igood = Where(xx GE str.min AND xx LE str.max)
	  IF igood[0] EQ -1 THEN BEGIN
	    itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
	      'Error in selecting interval: no good points ')
	    RETURN
	  ENDIF
	  tmpset=data[*,igood]
	  lincoeffs = LinFit(tmpset[0,*],tmpset[1,*],/Double)
	  yFit = lincoeffs[1]*xx+lincoeffs[0]
	  plot,xx,data[1,*],/NoData
	  oplot,xx,data[1,*],Color=2
	  oplot,xx,yFit,Color=10
	  IF StrCompress(str.ask[0],/Rem) EQ '1' THEN BEGIN
	    itmp = Dialog_Message(/Question,/Cancel,Dialog_Parent=event.top, $
	      ['Using data set index: '+StrCompress(goodIndices[i]), $
	      'Substract the fit from data?'])
	  ENDIF ELSE itmp = 'Yes'
	  IF itmp EQ 'Cancel' THEN BEGIN
	    FuFiFa_Plot,event.top
	    RETURN
	  ENDIF
	  IF itmp EQ 'Yes' THEN BEGIN
	    data[1,*]=data[1,*]-yFit
	    oData->Set,goodIndices[i],Value=data
	  ENDIF
	ENDFOR
	FuFiFa_Plot,event.top
	END

  'DataDeleteScan': BEGIN

	itmp = Dialog_message(/Info,/Cancel,Dialog_parent=event.top, $
	  ['This option removes permanently the selected scans',$
	  'Continue?'])
	IF itmp EQ 'Cancel' THEN RETURN
	Widget_Control,event.top,Get_UValue=state
	indices=Widget_Info(state.wids.dataList,/List_Select)
	IF indices[0] EQ -1 THEN BEGIN
	    itmp = Dialog_message(/Error,Dialog_parent=event.top, $
		'No data to plot. Please Load/Select data')
	    RETURN
	ENDIF
	;
	; remove data
	;
	oData = state.oData
        nData = oData->info(/N_Elements)
	IF nData EQ N_Elements(indices) THEN BEGIN
	  itmp = Dialog_message(/Error,Dialog_parent=event.top, $
          'I cannot remove all scans.')
           RETURN
	ENDIF
	oData->remove,indices
	;
	; update widgets
	;
	titles = oData->value(/title)
	FOR i=0,N_Elements(titles)-1 DO titles[i]='id'+StrCompress(i,/Rem)+': '+titles[i]
        Widget_Control,state.wids.dataList,Set_Value=titles

	widget_control,state.wids.draw, get_val = wnum
	wset,wnum
	erase

	END


  'DataInfo': BEGIN
	Widget_Control,event.top,Get_UValue=state
	oeq = state.oEquation
	odata = state.oData

	txt1=['','---Info on data: ',oData->Info()]
	txt2=['','---Info on model: ',oeq->Info()]
	XDisplayFile1,Text=[txt1,txt2],title='FuFiFa info',Dialog_Parent=event.top,height=48
	END

  'ModelLoad': fufifa_modelload,event.top

  'ModelSave': BEGIN
	Widget_Control,event.top,Get_UValue=state

	dir = state.importDir
	file = Dialog_PickFile(Title='Write model to a fififa-model data file',$
            Dialog_Parent=event.top,PATH=dir,GET_PATH=dir, $
	    File='fufifa_model.dat')
        IF file EQ '' THEN RETURN
        Widget_Control,/Hourglass
	IF CheckFile(file) EQ 1 THEN BEGIN
	  itmp = Dialog_Message(/Cancel,/Info,Dialog_Parent=event.top, $
		['File exists: '+file,'Overwrite it?'])
	  IF itmp EQ 'Cancel' THEN RETURN
	ENDIF
	state.importDir = dir

	oeq = state.oEquation
        nlines=oeq->GetProperty(/NLines)
	IF nlines EQ 0 THEN RETURN

	OpenW,unit,file,/Get_Lun
	PrintF,unit,'; '
	PrintF,unit,'; fufifa-model data'
	PrintF,unit,'; Comments start with ";"'
	PrintF,unit,'; fufifa version: '+fufifa_version()
	PrintF,unit,'; Created on: '+SysTime()
	cd,Current=pwd
	PrintF,unit,'; Working on: '+pwd
	PrintF,unit, $
	 '; The lines below define the model functions and their fit/guess '
	PrintF,unit, $
	 ';    parameters. They can be edited keeping the format unchanged'
	PrintF,unit,'; '

	FOR i=0L,nlines-1 DO BEGIN
	  name = oeq->GetPropertyLine(i,/Names)
          tmp = name+' | '+oeq->GetPropertyLine(i,/Template)+' | '+ $
            vect2string(oeq->GetPropertyLine(i,/parLine))
          printf,unit,tmp
	  print,tmp
	ENDFOR

	Free_Lun,unit
	Print,'File fufifa_model.dat written to disk.'

	END

  'ModelDeleteAllFunctions': BEGIN
	Widget_Control,event.top, Get_UValue=state
	oeq = state.oequation

	; clear equtions
	nlines=oeq->GetProperty(/NLines)
	FOR i=0L,nlines-1 DO oeq->deleteLine

	; clear all stored data
	nn =  state.oData->info(/N_Elements)
	IF nn EQ 1 and state.oData->value(0,/IsDataDef) NE 1 THEN BEGIN
	  Message,/Info,'No data'
	  RETURN
	ENDIF 
	FOR i=0L,nn-1 DO state.oData->Set,i,/ClAux
	END

  'ModelDeleteFunction': BEGIN
	Widget_Control,event.top, Get_UValue=state
	oeq = state.oequation
	index=0
	xEdit,index,Text='Index: ',$
		InfoText='Index of the line to delete (starting from zero)',$
		Dialog_Parent=event.top, Action=action,Title='Delete model line'
	IF action EQ 'CANCEL' THEN RETURN
	oeq->deleteLine,index


	;
	; update data auxiliar
	;
	nn =  state.oData->info(/N_Elements)
	IF nn EQ 1 and state.oData->value(0,/IsDataDef) NE 1 THEN BEGIN
	  Message,/Info,'No data'
	  RETURN
	ENDIF
	; clear all stored data
	FOR i=0L,nn-1 DO BEGIN
	  state.oData->Set,i,/ClAux
	ENDFOR
	END

  'ModelAddFunction': BEGIN
	fufifa_addFunction,event.top
	END

  'ModelCopyAll': BEGIN
	Widget_Control,event.top, Get_UValue=state
	nn=state.oEquation->GetProperty(/NPars)
	IF nn EQ 0 THEN Message,'No equation input.'
	mm = state.oData->Info(/N_Elem)
        IF mm EQ 1 THEN BEGIN
	  IF  state.oData->value(0,/IsDataDef) NE 1 THEN $
		Message,'No data input.'
        ENDIF
	;all_pars=state.oEquation->GetProperty()
	wIds = state.oEquation->GetProperty(/wId)
	FOR i=0,N_Elements(wIds)-1 DO BEGIN
	  Widget_Control,wIds[i],Get_Value=val
	  IF i EQ 0 THEN all_pars=val.coefs ELSE all_pars=[all_pars,val.coefs]
	ENDFOR
	FOR i=0L,mm-1 DO state.oData->Set,i,Aux=all_pars
	END

  'ModelCopySelected': BEGIN
	Widget_Control,event.top, Get_UValue=state
	nn=state.oEquation->GetProperty(/NPars)
	IF nn EQ 0 THEN Message,'No equation input.'
	mm = state.oData->Info(/N_Elem)
        IF mm EQ 1 THEN BEGIN
	  IF  state.oData->value(0,/IsDataDef) NE 1 THEN $
		Message,'No data input.'
        ENDIF
	;all_pars=state.oEquation->GetProperty()
	wIds = state.oEquation->GetProperty(/wId)
	FOR i=0,N_Elements(wIds)-1 DO BEGIN
	  Widget_Control,wIds[i],Get_Value=val
	  IF i EQ 0 THEN all_pars=val.coefs ELSE all_pars=[all_pars,val.coefs]
	ENDFOR
	indices=Widget_Info(state.wids.dataList,/List_Select)
	IF N_Elements(indices) LE 1 THEN RETURN
	FOR i=0L,N_Elements(indices)-1 DO BEGIN
	  state.oData->Set,indices[i],Aux=all_pars
	ENDFOR
	END

  'ModelFix': BEGIN
	Widget_Control,event.top, Get_UValue=state
	; value.flags in cw_pfield EQ 0 means unsensitive (i.e. fixed)
  	; fixed=0 in oEquation means "free"
	nlines = state.oEquation->getProperty(/NLines)
	FOR i=0L,nLines-1 DO BEGIN
	  nIndices = state.oEquation->getPropertyLine(i,/NIndices)
	  wId = state.oEquation->getPropertyLine(i,/wId)
	  Widget_Control,wId,Get_Value=value
	  value.flags = 0
	  Widget_Control,wId,Set_Value=value
	  FOR j=0L,nIndices-1 DO BEGIN
	    state.oequation->setPropertyItem,i,j,Fixed=1
	  ENDFOR
	ENDFOR
	END

  'ModelFree': BEGIN
	Widget_Control,event.top, Get_UValue=state
	; value.flags in cw_pfield EQ 0 means unsensitive (i.e. fixed)
  	; fixed=0 in oEquation means "free"
	nlines = state.oEquation->getProperty(/NLines)
	FOR i=0L,nLines-1 DO BEGIN
	  nIndices = state.oEquation->getPropertyLine(i,/NIndices)
	  wId = state.oEquation->getPropertyLine(i,/wId)
	  Widget_Control,wId,Get_Value=value
	  value.flags = 1
	  Widget_Control,wId,Set_Value=value
	  FOR j=0L,nIndices-1 DO BEGIN
	    state.oequation->setPropertyItem,i,j,Fixed=0
	  ENDFOR
	ENDFOR
	END

  'ModelUndoFit': BEGIN
	Widget_Control,event.top, Get_UValue=state
	tmp = *(state.ptrUndoFit) 
	indexLine=Widget_Info(state.wids.dataList,/List_Select)
	IF indexLine[0] EQ -1 THEN BEGIN
	  itmp = Dialog_message(/Error,Dialog_parent=event.top, $
		'No Data selected. Please Load/Select data')
	  RETURN
	ENDIF
	state.oequation->SetProperty,Param=tmp,/UpdateWidgets
	FOR i=0,N_Elements(indexLine)-1 DO BEGIN
	  state.oData->set,indexLine[i],Aux=tmp
	ENDFOR
	fufifa_plot,event.top,Group=event.top
	END


;  'ModelCovariance': BEGIN
;
;	Widget_Control,event.top, Get_UValue=state
;
;        ; build the model equation
;        nlines = state.oequation->GetProperty(/Nlines)
;        npars = state.oequation->GetProperty(/Npars)
;        myEq = ''
;        FOR i=0,nlines-1 DO BEGIN
;          ;print,state.oequation->GetPropertyLine(i,/Eq)
;          IF i NE 0 THEN myEq = myEq+'+'
;          myEq = myEq+state.oequation->GetPropertyLine(i,/Eq)
;        ENDFOR
;
;        ; get the selection lines
;	nn =  state.oData->info(/N_Elements)
;	IF nn EQ 1 and state.oData->value(0,/IsDataDef) NE 1 THEN BEGIN
;	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
;          'No data found.')
;	  RETURN
;	ENDIF
;
;	goodIndices = IndGen(nn)
;
;        nData = state.oData->info(/N_Elements)
;        list01 = StrArr(nData)
;        FOR i=0L,nData-1 DO BEGIN
;          list01[i] = 'id:'+StrCompress(i,/Rem)+' '+state.oData->value(i,/Title)
;        ENDFOR
;
;        outSigmas = fltArr(npars*2+1,N_Elements(goodIndices))
;        IF (size(outSigmas))[0] EQ 1 THEN outSigmas=Reform(outSigmas,npars*2+1,1)
;        outTitles=['DataSetId','p['+strcompress(indgen(Npars),/rem)+']',$
;                                'sigma['+strcompress(indgen(Npars),/rem)+']']
;        outTitles='col'+strcompress(IndGen(N_Elements(outTitles))+1,/rem)+': '+outTitles
;        text = ['','********** Covariance Calculations **********']
;	FOR i=0L,N_Elements(goodIndices)-1 DO BEGIN
;	  data = state.oData->value(goodIndices[i])
;          IF state.oData->value(goodIndices[i],/IsAuxDef) EQ 1 THEN BEGIN
;            aux = state.oData->value(goodIndices[i],/Aux)
;            xx=Reform(data[0,*])
;            yy=Reform(data[1,*])
;            dataErr = FuFiFa_GetErrorArray(data,state.MPFitPars,errTxt)
;            covar_fit,xx,yy,aux,sigma=dataErr,myEq,outSigma=outSigma,covar=covar,$
;            Text=txt, Verbose=0
;            ; change the info corresponding to the errors used. This is because
;            ; we always input a numerical array to covat_fit
;            txt[1]=errTxt
;            outSigmas[0,i]=goodIndices[i]
;            outSigmas[1:N_Elements(aux),i]=aux
;            outSigmas[(N_Elements(aux)+1):(2*Npars),i]=outSigma
;          ENDIF ELSE BEGIN
;            txt='Error: Model undefined.'
;          ENDELSE
;          text = [text,'',list01[goodIndices[i]],txt]
;
;	ENDFOR
;        XDisplayfile1,TITLE='Covariance Calculations', $
;            TEXT=text,GROUP=event.top
;        Xplot,outSigmas,colTitles=outTitles,Xtitle='-1',Ytitle='-1'
;
;	END

  'ModelParsVariation': BEGIN
        Widget_Control,event.top, Get_UValue=state
	pars = FuFiFa_GetModelParsVariation(event.top,error=error)
	IF error EQ 1 THEN RETURN
        ; np+2
        npars = N_Elements(pars[*,0])
        np = state.oequation->GetProperty(/Npars)


	colTitles = 'p['+StrCompress(SIndGen(np),/Rem)+']'
	colTitles=['DataSetId',colTitles,'Chi**2']

	sigTitles0 = 'sigma['+StrCompress(SIndGen(np),/Rem)+']'
	sigTitles=['DataSetId',sigTitles0]

	Widget_Control,event.top, Get_UValue=state
	list=state.oData->value('[*]',/Title)
	rowTitles=list[Reform(pars[0,*])]


        ; 
        ; get covariances
        ; 
        ; build the model equation
        nlines = state.oequation->GetProperty(/Nlines)
        npars = state.oequation->GetProperty(/Npars)
        myEq = ''
        FOR i=0,nlines-1 DO BEGIN
          ;print,state.oequation->GetPropertyLine(i,/Eq)
          IF i NE 0 THEN myEq = myEq+'+'
          myEq = myEq+state.oequation->GetPropertyLine(i,/Eq)
        ENDFOR

        ; get the selection lines
	nn =  state.oData->info(/N_Elements)
	IF nn EQ 1 and state.oData->value(0,/IsDataDef) NE 1 THEN BEGIN
	  itmp = Dialog_Message(/Error,Dialog_Parent=event.top, $
          'No data found.')
	  RETURN
	ENDIF

	goodIndices = IndGen(nn)

        nData = state.oData->info(/N_Elements)
        list01 = StrArr(nData)
        FOR i=0L,nData-1 DO BEGIN
          list01[i] = 'id:'+StrCompress(i,/Rem)+' '+state.oData->value(i,/Title)
        ENDFOR

        ;outSigmas = fltArr(npars*2+1,N_Elements(goodIndices))
        outSigmas = fltArr(np,N_Elements(goodIndices))
        ;IF (size(outSigmas))[0] EQ 1 THEN outSigmas=Reform(outSigmas,npars*2+1,1)
        IF (size(outSigmas))[0] EQ 1 THEN outSigmas=Reform(outSigmas,np,1)

;        outTitles=['DataSetId','p['+strcompress(indgen(Npars),/rem)+']',$
;                                'sigma['+strcompress(indgen(Npars),/rem)+']']
;        outTitles='col'+strcompress(IndGen(N_Elements(outTitles))+1,/rem)+': '+outTitles

        textCov = ['','********** Covariance Calculations **********']
	FOR i=0L,N_Elements(goodIndices)-1 DO BEGIN
	  data = state.oData->value(goodIndices[i])
          IF state.oData->value(goodIndices[i],/IsAuxDef) EQ 1 THEN BEGIN
            aux = state.oData->value(goodIndices[i],/Aux)
            xx=Reform(data[0,*])
            yy=Reform(data[1,*])
            dataErr = FuFiFa_GetErrorArray(data,state.MPFitPars,errTxt)
            covar_fit,xx,yy,aux,sigma=dataErr,myEq,outSigma=outSigma,covar=covar,$
            Text=txt, Verbose=0
            ; change the info corresponding to the errors used. This is because
            ; we always input a numerical array to covat_fit
            txt[1]=errTxt
            ;outSigmas[0,i]=goodIndices[i]
            ;outSigmas[1:N_Elements(aux),i]=aux
            ;outSigmas[(N_Elements(aux)+1):(2*Npars),i]=outSigma
            outSigmas[*,i]=outSigma
          ENDIF ELSE BEGIN
            txt='Error: Model undefined.'
          ENDELSE
          textCov = [textCov,'',list01[goodIndices[i]],txt]

	ENDFOR


	Widget_Control,event.id,Get_Value=val
	CASE val OF
	  'Plot': BEGIN
                    sp=size(pars)
                    so=size(outSigmas)
                    IF sp[2] NE so[2] THEN BEGIN
                       itmp = Dialog_Message( /Error,Dialog_Parent=event.top, $
                        ['Size of parameter array and sigma array do not agree',$
                         'Displaying only parameters...'])
                       colTitles='col'+strcompress(IndGen(N_Elements(colTitles))+1,/rem)+': '+colTitles
		       Xplot,pars,/No_Block,ColTitles=colTitles, YCOL=2, $
                        Wtitle='Parameter plot',Xtitle='-1',Ytitle='-1'
                    ENDIF ELSE BEGIN
                       titles=[coltitles,sigTitles0]
                       titles='col'+strcompress(IndGen(N_Elements(titles))+1,/rem)+': '+titles
                       tmp = dblarr(sp[1]+so[1],sp[2])
                       tmp[0:sp[1]-1,*]=pars
                       tmp[(sp[1]):(sp[1]+so[1]-1),*]=outSigmas
                       IF Fix( (state.MPFitPars.ErrorsFrom)[0] ) EQ 0 THEN $
                           ERRCOL='ycol+'+StrCompress(sp[1]-1,/Rem) ELSE ERRCOL=0
		       Xplot,tmp,/No_Block,YCOL=2, ERRCOL=ERRCOL,  ColTitles=titles, $
                        Wtitle='Parameter plot',Xtitle='-1',Ytitle='-1'
                    ENDELSE
		  END
	  'Text': BEGIN
			ncol = StrCompress(N_Elements(pars[*,0]),/Rem)

			txt = String('',Format='(A15)')+String(colTitles,Format='('+ncol+'A15)')
		        FOR i=0,N_Elements(pars[0,*])-1 DO BEGIN
			  line=String(rowTitles[i],Format='(A15)')+String(pars[*,i],Format='('+ncol+'G15.7)')
			  txt=[txt,line]
			ENDFOR

			ncol = StrCompress(np+1,/Rem)
			txtSig = String('',Format='(A15)')+String(sigTitles,Format='('+ncol+'A15)')
		        FOR i=0,N_Elements(pars[0,*])-1 DO BEGIN
			  line=String(rowTitles[i],Format='(A15)')+String(i,outSigmas[*,i],Format='('+ncol+'G15.7)')
			  txtSig=[txtSig,line]
			ENDFOR

                        txt = ['','****Parameters****',txt,'','****Sigmas****',txtSig,'',textCov]
			XDisplayFile1,text=txt,Title='FuFiFa: pars list'
		  END
	  'EXCEL/XML file': BEGIN
        	       dir = state.importDir
back2:
		       file = Dialog_PickFile(/Write, file='fufifa.xls',$
		         Dialog_Parent=event.top,PATH=dir,GET_PATH=dir,filter='*.xls')
		       IF file EQ '' THEN RETURN
		       IF CheckFile(file) EQ 1 THEN BEGIN
		         itmp = Dialog_Message(/Question,Dialog_Parent=event.top,$
		             ['File exists:',file,' ',$
		             'Overwrite it?'])
		         IF itmp EQ 'No' THEN GoTo,back2
		       ENDIF

		       openw,unit,file,/get_lun
		         xmlexcel_workbook,unit,/open
		           xmlexcel_worksheet,unit,/open
			     ; title
			     txt = ['',colTitles,sigTitles0]
		             xmlexcel_putline,unit,txt
		             FOR i=0,N_Elements(pars[0,*])-1 DO BEGIN
			       line1=StrCompress(pars[*,i],/Rem)
			       line2=StrCompress(outSigmas[*,i],/Rem)
			       line=[rowTitles[i],line1,line2]
		               xmlexcel_putline,unit,line
		             ENDFOR
		           xmlexcel_worksheet,unit,/close
		         xmlexcel_workbook,unit,/close
		       free_lun,unit
		       itmp = Dialog_Message(Dialog_Parent=event.top, /Question,$
			    ['File written to disk: ',file,'','Open it? '])
		       IF itmp EQ 'Yes' THEN BEGIN
                         CASE SDep() OF
                           'WINDOWS':BEGIN
                              itmp = strpos(file,'\',/reverse_search)
                              IF itmp[0] NE -1 THEN BEGIN
                                f1=StrMid(file,0,itmp)
                                f2=StrMid(file,itmp+1)
                                xop_spawn,['cd '+f1,'start '+f2],/noPath
                              ENDIF ELSE BEGIN
                                spawn,'start '+ffile
                              ENDELSE
                              END
                           else: BEGIN
                              XDisplayFile1,file,Dialog_Parent=event.top
                              END
                         ENDCASE
		       ENDIF
		  END
	  'SPEC file': BEGIN

        	       dir = state.importDir
back4:
		       file = Dialog_PickFile(/Write, file='fufifa_pars.spec',$
		         Dialog_Parent=event.top,PATH=dir,GET_PATH=dir,filter='*.spec')
		       IF file EQ '' THEN RETURN
		       IF CheckFile(file) EQ 1 THEN BEGIN
		         itmp = Dialog_Message(/Question,Dialog_Parent=event.top,$
		             ['File exists:',file,' ',$
		             'Overwrite it?'])
		         IF itmp EQ 'No' THEN GoTo,back4
		       ENDIF

		       openw,unit,file,/get_lun
                             PrintF,unit,'#F '+file
                             PrintF,unit,'#C Parameters File created by XOP/fufifa (functional fitting facility)'
                             PrintF,unit,'#D '+SysTime()
                             PrintF,unit,'#UFUFIFA_VERSION '+FuFiFa_Version()
                             PrintF,unit,'#UFUFIFA_EQUATION '+myEq
                             PrintF,unit,' '

			     ; title
			     txt = ['',colTitles,sigTitles0]
		             FOR i=0,N_Elements(outSigmas[*,0])-1 DO BEGIN
                               PrintF,unit,'#S '+StrCompress(i+1,/Rem)+' '+colTitles[i+1]
                               PrintF,unit,'#N  3'
                               PrintF,unit,'#L  DataSetId  '+colTitles[i+1]+'  '+$
                                           sigTitles0[i]
		               FOR j=0,N_Elements(outSigmas[0,*])-1 DO BEGIN
                                 PrintF,unit,j,pars[1+i,j],outSigmas[i,j]
                               ENDFOR
                               
                               PrintF,unit,''
		             ENDFOR
		       free_lun,unit
		       itmp = Dialog_Message(Dialog_Parent=event.top, /Info,$
			    ['File written to disk: ',file,'',$
                             'It can be reopened with xplot or fufifa'])

		  END
           
	ENDCASE
	END

;  'ModelRefresh': BEGIN
;	    FuFiFa_ModelRefresh,event.top
;	END


  'Preferences': BEGIN
	
	Widget_Control,event.top,Get_UValue=state
	Widget_Control,event.id,Get_Value=val
	str0=state.MPFitPars
;	CASE val OF
;	  'Edit...': BEGIN
		tmp = xop_defaults('fufifa')
        	;flags=Replicate('1',N_tags(str0))
        	;titles=['Fitting tolerance','Max of fitting iterations',$
        	;  'Use errors from third column (if exists)',$
        	;  'Data errors', 'Report printing frequency', $
		;  'Fit time limit (per scan) [sec]']
        	XScrmenu,str0,/Interp,/NoType,Action=action,group=event.top, $
          	Flags=tmp.flags, Titles=tmp.titles, WTitle='Fitting preferences',$
	  	Dialog_Parent=event.top
        	IF action EQ 'DONT' THEN RETURN
        	;Copy_Structure,str,str0
        	state.MPFitPars = str0
		Widget_Control,event.top,Set_UValue=state
;		END
;	  'Save as default': BEGIN
;                Xop_Input_Save,str0,Group=event.top, $
;                   Default='fufifa.xop',Comment='; xop/fufifa(v'+$
;                   fufifa_version()+') input file on '+SysTime()
;		END
;	  else:
;	  ENDCASE
	END

  'PlotDropList': BEGIN
	Widget_Control,event.top, Get_UValue=state
	dl = state.plotDropList
	index = event.index
	IF index EQ 0 THEN RETURN
	;flip index
	dl[index-1]=abs(dl[index-1]-1)
	tmp = fufifa_PlotDropList(dl)
	state.plotDropList = dl
	Widget_Control,state.wids.plotDropList,Set_Value=tmp
	Widget_Control,state.wids.plotDropList,Set_DropList_Select=0
	Widget_Control,event.top, Set_UValue=state

	widget_control,state.wids.draw, get_val = wnum
	wset,wnum
	erase
	fufifa_plot,event.top,Group=event.top
	
	END

  'APPLYVALUE': BEGIN
	Widget_Control,event.id,Get_Value=value
	CASE StrUpCase(value) OF
          'XOP': xop
	  'XPLOT': xplot,/no_block	
	  'XAID': xaid,/no_block	
	  'EXODUS': exodus,/no_block	
	  else: itmp = Execute(uvalue)
	ENDCASE
	END
  'Help': BEGIN
	tmp=''
        Widget_Control,event.id,Get_Value=tmp
        Xhelp,StrLowCase(tmp[0]),group=event.top
	END

  else:
ENDCASE

END  ; FuFiFa_event

;
;============================================================================
;
PRO FuFiFa, group = group, no_block=no_block, Parent=tlb

Catch, error_status
IF error_status NE 0 THEN BEGIN
  Message,/Info,'error caught: '+!error_state.msg
  itmp = Dialog_Message(/Error, $
    'FuFiFa: error caught: '+!error_state.msg)
  Catch, /Cancel
  RETURN
ENDIF

Xop_Default_CT,Group=group

IF N_Elements(no_block) EQ 0 THEN no_block=1

; 
; Widget definition
;
tlb = Widget_base(TITLE='FuFiFa (Functional Fitting Facility) '+$
	FuFiFa_version(),MBAR=wMenu, /TLB_KILL_REQUEST_EVENTS, $
	Event_Pro='FuFiFa_Quit')

;
; define menu bar
;
wtmp0 = WIDGET_BUTTON(wMenu, VALUE='FuFiFa', /MENU)
 wtmp1 = Widget_Button(wtmp0, Value='Load...',UValue='DataLoad')
 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Save as...', UVALUE='SaveFile')
 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Print Plot', UVALUE='PlotPrint',/Separator)
; wtmpB = WIDGET_BUTTON(wtmp0, VALUE='Preferences',/Menu,/Separator)
;    wtmp1 = WIDGET_BUTTON(wtmpB, VALUE='Edit...',UValue='Preferences')
;    wtmp1 = WIDGET_BUTTON(wtmpB, VALUE='Save as default',UValue='Preferences')
 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Preferences...',UValue='Preferences')
 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Quit', UVALUE='Quit',/SEPARATOR)

wtmp0 = WIDGET_BUTTON(wMenu, VALUE='Plot', /MENU)
 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Refresh',UValue='Refresh')
 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Print', UVALUE='PlotPrint')

wtmp0 = WIDGET_BUTTON(wMenu, VALUE='Data', /MENU)
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Load data (Spec cleaned file/exodus)', $
	UVALUE='DataLoad')
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Load data (ASCII)', $
	UVALUE='DataLoadASCII')
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Write data (SPEC) (no model)', $
	UVALUE='SaveFileOnlyData')
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Operations...', /Separator, $
	UVALUE='DataOperations')
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Cut data...', UVALUE='DataCut')
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Substract linear background...', $
		UVALUE='DataBackground')
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Delete selected scans', /Separator,$
	UVALUE='DataDeleteScan')

wtmp0 = WIDGET_BUTTON(wMenu, VALUE='Model', /MENU)
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Load from file...', UVALUE='ModelLoad')
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Save to file...', UVALUE='ModelSave')
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Add Function...', $
		UVALUE='ModelAddFunction',/Separator)
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Delete All Functions', $
		UVALUE='ModelDeleteAllFunctions')
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Delete Function...', $
		UVALUE='ModelDeleteFunction')
  wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Propagate parameters', $
		UVALUE='ModelPropagate')

;wtmp0 = WIDGET_BUTTON(wMenu, VALUE='Fit', /MENU)
; wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Preferences...',UValue='Preferences')

wtmp0 = WIDGET_BUTTON(wMenu, VALUE='Tools', /MENU)
 wtmpb = Widget_Button(wtmp0, VALUE='Xop', UValue='APPLYVALUE')
 wtmpb = Widget_Button(wtmp0, VALUE='Xplot', UValue='APPLYVALUE')
 wtmpb = Widget_Button(wtmp0, VALUE='XAID', UValue='APPLYVALUE')
 wtmpb = Widget_Button(wtmp0, VALUE='EXODUS', UValue='APPLYVALUE')
 wtmpb = Widget_Button(wtmp0, VALUE='Color table',/Separator, /Menu)
   wtmp1 = WIDGET_BUTTON(wtmpb, VALUE='Set Default', UVALUE='ColorTable')
   wtmp1 = WIDGET_BUTTON(wtmpb, VALUE='Change table...', UVALUE='ColorTable')
wtmp0 = WIDGET_BUTTON(wMenu, VALUE='Help', /HELP)
 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='Info on loaded data', UVALUE='DataInfo')
 wtmp1 = WIDGET_BUTTON(wtmp0, VALUE='FuFiFa', UVALUE='Help')



wBase = Widget_Base(tlb,/Column)

  ; draw + list
  wBase1 = Widget_Base(wBase,/Row)

    wBase11 = widget_base(wBase1,/Column,/Frame)
      wDraw = widget_draw(wBase11, XSIZE=500, YSIZE=350, RETAIN=2)
      wBase111 = widget_base(wBase11,/Row)
        wtmp = Widget_Button(wBase111,Value='Refresh',UValue='Refresh')
	plotDropListFlags = [1,1,1,0,1]
        tmp = FuFiFa_plotDropList(plotDropListFlags)
        wPlotDropList = Widget_DropList(wBase111,Value=tmp, $
		UValue='PlotDropList')
        wtmp = Widget_Button(wBase111,Value='Zoom',UValue='PlotZoom')

    wBase12 = widget_base(wBase1,/Column,/Frame)
      wBase121 = widget_base(wBase12,/Row)
        wtmp = Widget_Button(wBase121,Value='Load...',UValue='DataLoad')
        wtmp = Widget_Label(wBase121,Value=' Data sets  ')
      wDataModel = Widget_Label(wBase12,Value=' Model: undefined. ')
      IF float(!version.release) LE 5.5 THEN $
        wDataList = Widget_List(wBase12, Value=Replicate('',25), $
	  ySize=115,xSize=40,/Multiple,UValue='Refresh') ELSE $
        wDataList = Widget_List(wBase12, Value=Replicate('',25), $
	  ySize=22,xSize=40,/Multiple,UValue='Refresh')

  ; Controls (buttons)
  wBase2 = Widget_Base(wBase,/Row,/Frame)

    wtmp = Widget_Button(wBase2,Value='Quit',UValue='Quit')
    wtmp = Widget_Button(wBase2,Value='**Fit**',UValue='**Fit**',$
	Event_Pro='fufifa_fit_event')
    wtmp = Widget_Button(wBase2,Value='Fit2',UValue='Fit2',$
	Event_Pro='fufifa_fit_event')
    wtmp = Widget_Label(wBase2,Value='Included [a,b] and/or excluded {a,b} regions: ')
    wFitDomain = Widget_Text(wBase2,Value='',UValue='FitDomain',$
	XSize=67,/Edit,Event_Pro='fufifa_fit_event')

  ; Models
  wBase3 = Widget_Base(wBase,/Column,/Frame)
    wBase31 = Widget_Base(wBase3,/Row)
      wtmp = Widget_Label(wBase31,Value='Models ')

      wtmp = Widget_Button(wBase31,Value='Add function',UValue='ModelAddFunction')
      wtmp0 = Widget_Button(wBase31,Value='Copy pars...', /Menu)
        wtmp = Widget_Button(wtmp0,Value='to ALL data sets', $
	  UValue='ModelCopyAll')
        wtmp = Widget_Button(wtmp0,Value='to SELECTED data sets', $
	  UValue='ModelCopySelected')

      wtmp = Widget_Button(wBase31,Value='Free all pars',UValue='ModelFree')
      wtmp = Widget_Button(wBase31,Value='Fix all pars',UValue='ModelFix')
      wtmp = Widget_Button(wBase31,Value='Undo Fit',UValue='ModelUndoFit')
;      wtmp = Widget_Button(wBase31,Value='Covariances',UValue='ModelCovariance')

      wtmp0 = Widget_Button(wBase31,Value='Pars vs data sets...', /Menu)
        wtmp = Widget_Button(wtmp0,Value='Plot',UValue='ModelParsVariation')
        wtmp = Widget_Button(wtmp0,Value='Text',UValue='ModelParsVariation')
        wtmp = Widget_Button(wtmp0,Value='EXCEL/XML file',UValue='ModelParsVariation')
        wtmp = Widget_Button(wtmp0,Value='SPEC file',UValue='ModelParsVariation')

;      wtmp = Widget_Button(wBase31,Value='Refresh pars', $
;		UValue='ModelRefresh')
    ;wModelBase = Widget_Base(wBase3,/Column, $
	;Scr_XSize=600,Scr_YSize=300,$
      	;XSize    =600,YSize=    300, $
	;/Scroll, Event_Pro='fufifa_parfield_event')

    ;;wTab = Widget_Tab(wBase3,UValue='TAB')
    ;;wTab0 = Widget_Base(wTab,Title='0')

    ;;wModelBase = Widget_Base(wTab0,/Column, $
    IF SDEP() EQ 'UNIX' THEN BEGIN
      wModelBase = Widget_Base(wBase3,/Column, $
	Scr_XSize=800,Scr_YSize=300,$
     	XSize    =800,YSize=    300, $
	/Scroll, Event_Pro='fufifa_parfield_event')
    ENDIF ELSE BEGIN
      wModelBase = Widget_Base(wBase3,/Column, $
	Scr_XSize=800,Scr_YSize=200,$
	/Scroll, Event_Pro='fufifa_parfield_event')
    ENDELSE

wids={tlb:tlb, draw:wDraw, plotDropList:wPlotDropList, dataList:wDataList, $
	modelBase:wModelBase, dataModel:wDataModel, fitDomain:wFitDomain, $
        xwindow:0L}

odata = Obj_New('ompp')
oequation = Obj_New('f3equation')

;MPFitPars = {tol:1D-10, maxIter:200, $
;  errorsFrom:['1','Yes','No'], $
;  errors:['0','Constant','Statistical'],$ 
;  nPrint:10, tMax:30.0  }
tmp = Xop_Defaults('fufifa')
MPFitPars = tmp.parameters

state = {wids:wids, specFile:'', importDir:'', $
   odata:odata, oequation:oequation,  $
   plotDropList:plotDropListFlags, $	; flags show/hide for 
					;data-model-components-error
   mpFitPars: mpFitPars, $
   ptrUndoFit:ptr_New(0)  $    ; to store the pars before last fit (for Undo)
   } 

Widget_Control,tlb,Set_UValue=state

Widget_Control,tlb,/Realize

xmanager, 'fufifa', tlb, GROUP_LEADER=group,No_Block=no_block
END
