
FUNCTION avoigt1,x,a

;+
; NAME:
;	AVOIGT1 
; PURPOSE:
;	Returns the value of a pseudo-voigt function (a combination of
;	lorentzian and gaussian functions) modified with an 
;	asymmetric profile
; CATEGORY:
;	Mathematics.
; CALLING SEQUENCE:
;	Result = avoigt1(x,a)
; INPUTS:
;	x: the argument for the function (number or array)
;	a: array with the function coefficients:
;		a(0) Amplitud
;		a(1) Center
;		a(2) Full Width at Half Maximum
;		a(3) Ratio between lorentzian and Gaussisan (1=Pure
;			lorentzian, 0=Pure Gaussian, 0.5=half each)
;		a(4) Asymmetry factor (0=no asymmetry). This factor
;		     should be approx -1<a[4]<1
;
; SIDE EFFECTS:
; 	None.
; RESTRICTIONS:
;	None.
; PROCEDURE:
;	It uses voigt1 for computing the pseudovoight function. The 
;	asymmetry is modelled with a H3 (Hermite polynomial of 3rd
;	order) calculated versus (x-center)/(fwhm/2.35). Then its
;	intensity value is divided by 18 (H3(3)=18, i.e., the value
;	at 3*sigma for a Gaussian), multiplied by the asymmetry
;	factor a[4] and added 1. Note that for large values of a the
;	center of the curve is shifted with respect to the ppseudovoigt
;	center, but the returned center is the unshifted value (i.e., 
;	the center of the pseudovoigt). 
;
; MOTIVATION:
;	The procedure is motivated by asymmetry effect induced by a finite
;	slit in powder diffraction, that is modeled with a modified
;	Gaussian with a function proportional to H3. Referemce:
;	E. Prince, "The effect of finite detector slit height on peak 
;	position and shapes in powder diffraction", J. Appl. Cryst. 16 
;	(1986) pp. 508-511 (see Eq. 6)
;
; MODOFICATION HISTORY:
;	M. Sanchez del Rio. ESRF. Grenoble 23 02 2007
;
;-
y = voigt1(x,a[0:4])

IF N_Elements(a) LE 4 THEN RETURN,y

;
; compute Hermite polynomial of order 3 
; versus "normalised" x
;
tmp = 2*sqrt(2*alog(2)) ; 2.35...
xx = (x-a[1])/(a[2]/tmp)
h3 = xx^3-3*xx

; 
; Normalise it and apply to y 
;
h3b = h3/18.  ; normalize to H3 value at 3*sigma
h3c = (h3b*a[4])+1 ; multiply per asymmetry factor and add 1
yy = y *h3c

RETURN,yy 

END

