;______________________________________________________________________________
;+
; NAME:
;       SPEC_NAME()
;
; PURPOSE:
;       This function returns the name of a particular scan in a SPEC file.
;       In most of the cases the name of a scan is the full macro that was
;       used to produce obtain the data.
;
; CATEGORY:
;       Input/Output.
;
; CALLING SEQUENCE:
;       Result = SPEC_NAME(Handle, Scan_Id)
;
; INPUTS:
;       Handle   - Handle to the SPEC data file initialised by a previous call
;                  to SPEC_ACCESS().
;
;       Scan_Id  - Scan identifier. This parameter can be a numeric or string
;                  scalar value that accepts several syntax and keywords as
;                  it is explained in the `SCAN IDENTIFIERS' section at the
;                  beginning of this file). The simplest case is to use
;                  the SPEC scan number as Scan_Id.
;
; KEYWORDS:
;       INDEX:  Set this keyword to interpret a numeric value in Scan_Id
;               as the index of the scan instead of the scan number.
;
; OUTPUT: 
;       The name of the selected scan. The information is obtained from the
;       '#S' line.
;
; RESTRICTIONS:
;       This function requires a valid handle to a SPEC data file obtained by
;       a previous call to SPEC_ACCESS().
;       If Scan_Id does not represent a valid scan the function produces an
;       error.
;
; EXAMPLE:
;       To print the name of the scan 150 from a file called 'baddata', enter:
;
;       Dummy = SPEC_ACCESS(File1_handle, 'baddata')
;       PRINT, SPEC_NAME(File1_handle, 150)
;______________________________________________________________________________
;-

function spec_name, handle, scan_id, INDEX=idx
   catch, error_status
   if error_status ne 0 then begin
      catch, /cancel
      on_error, 2
      message, __cleanmsg(!err_string);, /traceback
   endif

   ; Check arguments
   if N_PARAMS() ne 2 then message, 'Incorrect number of arguments.'

   __speccheck, handle
   __specload, handle, scan_id, errmsg, INDEX=idx
   if !ERR then message, errmsg
   return, handle.scan(handle.currscan).name 
end
