Function dabax_access,input,group=group, verbose=verbose
; 
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       DABAX_ACCESS
;
; PURPOSE:
;	Access to a dabax file. Returns the spec handler.
;	
; CATEGORY:
;       DABAX (DAta Base for Atomic X-rays properties) applications.
;
; CALLING SEQUENCE:
;	handle = DABAX_ACCESS(input)
; INPUTS:
;	INPUT:  This inout can be one of the following cases:
;		1) a structure (handle for dabax file): in this
;			case the function returns the same structure.
;		2) a string with the full file name (it contains the 
;			character '/')
;		3) a string with the short file name (it does not contain
;			the character '/'). In this case the fuction gets
;			the full name using dabax_pathf().
; KEYWORDS:
;       group = The widget ID of the widget that calls the routine.
;               This is passed to widget_message in its dialog_parent
;               keyword to center the window message.
;	verbose = flag to inform about the DABAX file restored (default=0)
;
; OUTPUT:
;	A spec-handler structure
;
; PROCEDURE:
;	It uses the spec_access fuction to access the file or
;	(if the file file-'.dat'+'.HIdl' exists) restores its contents.
;
; EXAMPLE:
;	h = dabax_access( $
;	'/pgexp/applications1/XRayOptics/DABAX/data/CrystalStructures.dat')
;	 or 
;	h = dabax_access( 'CrystalStructures.dat')
;	 or 
;	hnew = dabax_access( h )
;
; MODIFICATION HISTORY:
;       96-07-10 Written by M. Sanchez del Rio (srio@esrf.fr)
;	96-08-19 MSR (srio@esrf.fr) changes handle extension 
;		.dat.IdlHandler -> .HIdl
;	96-11-25 MSR (srio@esrf.fr) fix a bug for accessing files when
;		*.Hidl does not exist. If *.Hidl creates it.
;	97-01-14 MSR (srio@esrf.fr) port to Windows.
;	97-10-16 MSR (srio@esrf.fr) adds group kw.
;	98-09-04 MSR (srio@esrf.fr) better error management
;	98-12-14 MSR (srio@esrf.fr) cosmetics for Xop2.0
;	2002-08-30 MSR (srio@esrf.fr)  checks for modification time of .dat
;		and updates .HIdl if necessary.
;	2008-07-02 srio@esrf.eu  improves error handling when missing HIdl
;-

catch, error_status
if error_status ne 0 then begin
  catch, /cancel
  Message,/Info,'error caught (returned 0): '+!err_string
  tmp = Dialog_Message(DIALOG_PARENT=group,$
    "DABAX_ACCESS: Error caught (returned 0): "+!err_string)
  Catch, /Cancel
  On_Error,2
  return,0
endif

IF N_Elements(verbose) EQ 0 THEN verbose=0
file_sep = sdep(/ds)

ntype = type(input)
case ntype of
  7: if strpos(input,file_sep) EQ -1 then begin
	file = (dabax_pathf(input,group=group)) 
	if file EQ '' then message,'Error from dabax_pathf'
     endif else file = input
  8: return,input
  else: message,'Input type not good.'
endcase

h=0
fileh = strmid(file,0,strlen(file)-4)+'.HIdl'

do_update = 0
if checkfile(fileh) then begin
  openr,lun,file,/get_lun,error=err
  IF err NE 0 THEN BEGIN
    itmp = Dialog_Message(/Error,'Cannot open file: '+file)
    RETURN,0
  ENDIF
  f1=fstat(lun)
  free_lun,lun
  openr,lun,fileh,/get_lun,error=err
  IF err NE 0 THEN BEGIN
    itmp = Dialog_Message('Cannot open file: '+fileh)
    do_update=1
  ENDIF ELSE BEGIN
    f2=fstat(lun)
    free_lun,lun

    ;srio  IF f1.mtime GT f2.mtime THEN BEGIN
    IF f1.mtime - f2.mtime GT 1.0 THEN BEGIN
      do_update = 1
    ENDIF ELSE BEGIN
      IF verbose EQ 1 THEN message,/info,'Restoring file: '+fileh
      restore,fileh
      h.specfile = file
    ENDELSE
  ENDELSE
endif else begin
  do_update = 2
endelse

;
; update/create spec handler
;
IF do_update GT 0 THEN BEGIN
 CASE do_update OF
   1: tmp = 'needs to be updated.'
   2: tmp = 'not found.'
   else: tmp = '***'
 ENDCASE
 ; *.Hidl does not exist'
 tmp = Dialog_Message(DIALOG_PARENT=group,$
    ["DABAX_ACCESS: DABAX index file "+tmp, $
     "It will be created automatically but it will",$
     "take a (long) time for reading the file and",$
     "creating the index (*.HIdl) file."]) 
 Widget_Control,/HourGlass
 if checkfile(file) then nn = spec_access(h,file) else $
   message,'File not found: '+file
 if type(h) EQ 8 then dabax_mkh,file,group=group else $
   message,'Cannot create index file for file: '+file
ENDIF


if n_elements(h) EQ 0 or type(h) NE 8 then begin
  message,/info,'Error in accessing DABAX file. Return 0'
  tmp = Dialog_Message(DIALOG_PARENT=group,$
    "DABAX_ACCESS: Error in accessing DABAX file. Return 0")
endif
return,h
end

