Function cross_calc_mix,input,descriptor,energy,col_titles,PARTIAL=PARTIAL, $
 CASEMATCH=CASEMATCH,density=density,UNIT=funit,NAME=NAME,group=group, $
 VERBOSE=verbose

; 
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       CROSS_CALC_MIX
;
; PURPOSE:
;	calculate the cross sections and attenuation coefficients
;	for compounds and mixtures.
;	
; CATEGORY:
;       X-Ray optics. DABAX data base.
;
; CALLING SEQUENCE:
;	out = cross_calc_mix(input,descriptor[,energy,col_titles])
; INPUTS:
;	input: a dabax input file for f1f2 data as accepted by dabax_access().
;	  It also accepts the empty string '' (see doc in cross_calc.pro)
;	descriptor: a string  with either the descriptor or the name of the 
;	  mixture. When using the descriptor, you must supply the density (g/cc)
;	  The descriptor sintax is the following:
;	  string with the chemical formula of the compounds with their weight 
;	  fraction between parenthesis.
;               Descriptor examples:
;               SiO2(0.807)B2O3(0.129)Na2O(0.038)Al2O3(0.022)K2O(0.004)
;               H2O(0.99)Cl2(0.01)
;               H(0.11189834)O(0.88810166)
;	  The name is one of the names defines in the DABAX file Compounds.dat
;	  When using the name input option, the keyword NAME must be set.
;		
; OPTIONAL INPUTS:
;	energy: if undefined, it uses the standard energy grid,
;		which is the grid in the data file for the tabulated
;		cases, and  dindgen(100)/99.*3.0 for the parametrized
;		data. When energy is defined, this value is used as a grid
;		for the results (it interpolates in the case of 
;		tabulated data)
;		Always in eV.
;	
; OPTIONAL OUTPUTS:
;	col_titles: A array of strings with the labels of the data
;		returned in "out"
; KEYWORDS:
;	PARTIAL= Text string that has to be matched with the column titles
;		in the DABAX data file. It is used to extract partial cross
;		sections. For example PARTIAL='Total' (default value) will
;		output the columns having this word in their labels, thus
;		the total cross section. Use for example 'Compton', or
;		'Rayleigh' if you want to extract only one of the other partial
;		cross sections or absorption coefficients. This keyword is
;		case-insensitive unless you set the CASEMATCH keyword.
;	CASEMATCH= to be used together with the PARTIAL keyword.
;		When this keyword is set to 1, the matching for the text
;		in PARTIAL is case sensitive.
;	DENSITY= the density value to be used for the calculations. If
;		not defined, and NAME is not set, then it uses 1.0 except
;		in the case when there is a single element in the
;		compund. In such a case it calls atomic_constants to get
;		the value.
;	UNIT= An integer indicating the unit of the output array
;		0 (default): barn/atom (Cross Section calculation)
;		1 : cm^2 (Cross Section calculation)
;		2 : cm^2/g (Mass Attenuation Coefficient)
;		3 : cm^-1 (Linear Attenuation Coefficient)
;	NAME: Set this keyword to a valid mixture name in the file 
;		Compounds.dat. If not set, it will interpret
;		"descriptor" as a mixture formula.
;       group = The widget ID of the widget that calls the routine.
;               This is passed to widget_message in its dialog_parent
;               keyword to center the window message.
;       verbose = when set, prints some information.
;
; OUTPUT:
;	out: an array with the values of the selected return paramaeter.
;
; PROCEDURE:
;	i) calculate attenuation mass coefficient for each elements by
;	   calling cross_calc
;	ii) Calculate the weight composition wi of each element in the 
;	   mixture by calling parse_mixture 
;	iii) The attenuation mass coefficient mu_m for the mixture is the 
;	   averaged attenuation from the elements (mu_i):
;		mu_m = Sum{ wi * mu_i } 
;	iv) Returns the value in the correspondent units:
;		UNIT=3 (cm-1) returns density*mu_m
;		UNIT=2 (cm2/g) returns mu_m
;		UNIT=1 (cm2) returns mu_m*(MW/NA) being MW the molecular
;			weight and NA the Avogadro's number. 
;		UNIT=0 (barns/at) returns mu_m*(MW/NA)*1e24
;	
;	The MW (molecular weight) is MW = Sum { fi * Ai), being Ai
;		the atomic weight of the i-th element and fi the 
;		number of atoms of the i-th type in the molecule.
;
;	IMPORTANT NOTICE: 
;	   The attenuation coefficient (UNIT = 2 and 3) is dependent on 
;	   the fractional weigths, so the result is independent if how the 
;	   mixture is defined, either 
;		i) as a function of the weight fractions 
;		ii) as a molecular mixture. 
;	   Therefore, water can be either defined as "H2O" (molecule) 
;		or "H(0.11)O(0.89)" (weight). 	
;
;	   However, for calculating the cross section of the mixture
;		(UNIT = 0 or 1) the molecular weigth enters in the 
;		calculation, and this value is different depending how the 
;		mixture is defined, because DABAX does not know the molecular 
;		weight, but calculates it from the mixture formula:
;		For example, for water:
;		   i) Molecular weight (H2O): 18.015280
;		   ii)  Molecular weight (H(0.11)O(0.89)):  14.350339
;
;		THEREFORE, FOR CROSS SECTIONS (UNIT=0,1), THE MIXTURE SHOULD BE
;		DEFINED IN ITS MOLECULAR FORM (like "H2O") AND NOT IN ITS 
;		WEIGHT FRACTION (like "H(0.11)O(0.89)"). 
;
;		PLEASE NOTE THAT THE PRE-DEFINED COMPOUNDS (IN DABAX
;		FILE Compounds.dat CONTAIN THE MIXTURE FORMULA, AND IN
;		SOME CASES (THE COMPOUNDS OBTAINED FROM NIST DATABASE)
;		THE MOLECULAR FORMULA IS NOT PRESENT (ONLY WEIGHT FORMULA)
;		THEREFORE THE CROSS SECTION VALUES (UNIT=0,1) MAY BE WRONG.
;
; REFERENCES: 
;	
;	http://physics.nist.gov/PhysRefData/XrayMassCoef/cover.html
;	http://xdb.lbl.gov/Section1/Sec_1-6.pdf
;	   
;
; EXAMPLES:
;	delvar,energy ; cleans energy variable
;	mu = cross_calc_mix('CrossSec_XCOM.dat','water',energy,col_title,$
;		unit=2,/NAME)
;	help,mu  (You get: DOUBLE    = Array(2, 80), because in the 
;	  DABAX file CrossSec_XCOM.dat thay are two colums with the 
;	  word ('Total'). 
;	plot,energy,mu(0,*),/xlog,/ylog,ytitle=col_title(0)
;
; MODIFICATION HISTORY:
;       96-11-29 Written by M. Sanchez del Rio (srio@esrf.fr)
;       97-01-10 srio@esrf.fr adds the call to atomic_constants to get
;		the density when there is a single element.
;       97-01-15 srio@esrf.fr adapts for Windows95.
;       97-10-16 srio@esrf.fr adds GROUP keyword.
;       2003-07-07 srio@esrf.fr corrects a bug in calculation:
;		all cross sectiona and attenuation factors were
;		calculated from the average value using the "fraction"
;		(not weight fraction) values: out_tot = out_tot + out*f(i)
;		This is not correct. The correct way is to calculate
;		the mass attenuation coeff (cm2/g) using the weight 
;		fraction (out_tot = out_tot + out*fw(i)) and then calculate 
;		the linear attenuation coeff. and cross sections using the
;		formulas explained before. 
;
;-
;on_error,2

IF N_Elements(verbose) EQ 0 THEN verbose=1


if keyword_set(name) then begin
  cfile = 'Compounds.dat'
  h = dabax_access(cfile,group=group)
  index = dabax_select(h,descriptor,/casematch,group=group)
  tmp = spec_data(h,index,/index)
  if not(keyword_set(density)) then begin
    rho1 = spec_headers(h,index,['#URHO'],/index)
    density = float(strmid(rho1,5,strlen(rho1)-1))
  endif
  ;not used  z = fix(tmp(0,*))
  ;not used  f = tmp(1,*)
  ;not used  s = atomic_symbols(z)
  ;
  ; get the molecular weight of the mixture
  ;
  ; this is necessary for calculating the molecular weight. However
  ; the other parameters returned are redundant with what is obtained
  ; before. It is then recommended to include the molecular weight in 
  ; the Compounds.dat file to avoid using parse_mixture here.
  ; To be done in the future. srio@esrf.fr 01/02/28
  ; 
  ; Note 2003-07-07:
  ; As the molecular weight (or the atomic weights) are not available
  ; in the file, this step is necessary
  ;
  for1 = spec_headers(h,index,['#UFOR'],/index)
  formula1 = strmid(for1,5,strlen(for1)-1)
  IF verbose THEN Message,/Info,'Using mixture with formula: '+formula1
  n = parse_mixture(formula1,s,f,z,fw,molwt=molwt,verbose=verbose)
endif else begin
  n = parse_mixture(descriptor,s,f,z,fw,molwt=molwt,verbose=verbose)
endelse

IF verbose THEN Message,/Info,'Using molecular weight: '+StrCompress(molwt)


if n_elements(s) LT 1 then begin
  message,/info,'Error in mixture descriptor/name (Returning 0): '+mixture
  return,0
endif


IF keyword_set(funit) THEN if funit EQ 3 THEN BEGIN
  if not(keyword_set(name)) and n_elements(density) eq 0 then begin
    if n_elements(s) EQ 1 then $
      density = atomic_constants(s(0),return='Density') else $ 
      density = 1.0
    message,/info,'Density value undefined, set to '+strcompress(density)
  endif
  if verbose then $
    message,/info,'Density ('+descriptor+'): '+strcompress(density)
ENDIF


;
; calculate averages mass attenuation coefficient
;
for i=0,n_elements(s)-1 do begin
  out = cross_calc(input,s(i),energy,col_titles,PARTIAL=PARTIAL, $
      CASEMATCH=CASEMATCH,density=density,UNIT=2,GROUP=group,VERBOSE=0)
  if i eq 0 then begin
     out_tot = out*fw(i) 
  endif else begin
     out_tot = out_tot + out*fw(i)
  endelse
endfor


;
; calculate average <Z/A> (only for test, not used)
;
IF verbose THEN BEGIN
  zOverA = 0.0
  zmean = 0.0
  amean = 0.0
  FOR i=0,n_elements(s)-1 DO BEGIN
     zOverA=zOverA+fw[i]*Double(z[i])/atomic_constants(z[i],return='AtomicMass')
     zMean=zMean+fw[i]*Double(z[i])
     aMean=aMean+fw[i]*atomic_constants(z[i],return='AtomicMass')
  ENDFOR
  message,/info,'<Z/A> ('+descriptor+'): '+strcompress(zOverA)
  message,/info,'<Z> ('+descriptor+'): '+strcompress(zMean)
  message,/info,'<A> ('+descriptor+'): '+strcompress(aMean)
  message,/info,'<Z>/<A> ('+descriptor+'): '+strcompress(zMean/aMean)
ENDIF


;
; change units
; 
case funit of
  0: begin		; barn
      cf = 1d+24*molWt/physical_constants('AVOGADRO')
      IF verbose THEN message,/info,'Conversion factor (cm^2/g->barn/atom): '+$
      	strcompress(cf)
      out_tot=out_tot*cf
      for ii=0,n_elements(col_titles)-1 do col_titles(ii) =$
	strsubstitute(col_titles(ii),'[cm^2/g]','[barn/atom]')
     end
  1: begin		; cm^2
      cf = molWt/physical_constants('AVOGADRO')
      IF verbose THEN message,/info,'Conversion factor (cm^2/g->cm^2): '+$
      	strcompress(cf)
      out_tot=out_tot*cf	
      for ii=0,n_elements(col_titles)-1 do col_titles(ii) =$
	strsubstitute(col_titles(ii),'[cm^2/g]','[cm^2]')
     end
  2: ; cm^2/g
  3: begin		; cm^-1
      out_tot = out_tot*density
      for ii=0,n_elements(col_titles)-1 do col_titles(ii) =$
	strsubstitute(col_titles(ii),'[cm^2/g]','[cm^-1]')
     end
endcase



return,out_tot

end
