

pro azimuth_integral, M, xx, intensity, CENTER=xy0, INCREMENT=incr, $
  EXACT=iExact, ERROR=ierror, NoMask=NoMask


;+
; NAME:
;	AZIMUTH_INTEGRAL
; PURPOSE:
;	this procedure permorms the azimuthal integration of an image
;
; CALLING SEQUENCE:
;	azimuth_integral, M, xx, intensity
; INPUTS:
;	M:  The array with 2D data matrix (may 
;           contain -1 for masked values)
; OUTPUTS:
;       xx: the abscissas array of the integral (in "pixels")
;       intensity: the intensity array of the integral
;
; KEYWORD PARAMETERS:
;       CENTER: [Xcenter,Ycenter] for the integral 
;       INCREMENT: The integration step (default=1.0)
;       EXACT: Forces exact calculation (more time consuming)
;       ERROR: a matrix with error estimation
;       NOMASK: set this keyword to 1 for considering also negative 
;           pixel values in the calculation (Default NOMASK=0, meaning
;           that negative pixel values are not used, i.e., masked)
;
;	
; MODIFICATION HISTORY:
;       M. Snchez del Ro, UAM-I, Mexico, April 2005
;       Rewrote the function by Aurelie Tallandier July 2001 
;            Grenoble to fast it up.
;	2007/11/29 srio@esrf.eu rewrites it
;	2009/12/04 srio@esrf.eu adds NoMask
;-	
;** Radial integration:
;** ******************
;**     Input:  M    =2D data matrix (may contains -1 for masked values).
;**             Keywords:
;**               CENTER=xy0  =[x0,y0] center coordinates (default [n,m]-1/2.)
;**               INCREMENT=incr =unit of ray in pixels      (default  1.0)
;**     Output: intensity =result of Radial integration.
;**             MAT         =Unrolled  polar matrix (angle,r)
;**             intensity
;**             xx        =diagram abscissa
;**             keywords:
;**               error=error     =errors
;
; rewritte srio@esrf.eu 2007/11/29
;

t00 = systime(1)

IF N_Elements(incr) EQ 0 THEN incr=1.0D0

IF N_Elements(iExact) EQ 0 THEN iExact=0


s=Long(size(M))
IF (s[0] ne 2) or (s[1] lt 2) or (s[2] lt 2) THEN BEGIN
  xx=0 & intensity=0
  RETURN
ENDIF

IF n_elements(xy0)  ne 2 then xy0=([s[1],s[2]]-1.)/2.
; IF n_elements(incr) ne 1 then incr=0.5

incr=abs(incr) <(s[1]/2) <(s[2]/2) > 0.01

dr  =incr/2.
xo  =xy0[0]  & yo=xy0[1] 


; create arrays with pixel coordinates
; shifted 0.5 units to compute center of pixels srio@esrf.eu 2007/11/29

t0 = systime(1)

;x = (findgen(s[1])+0.5) # (findgen(s[2])*0+1 )
;y = (findgen(s[1])*0+1) # (findgen(s[2])+0.5)
x = (findgen(s[1])+0.5) # Replicate(1.0,s[2])
y = Replicate(1.0,s[1]) # (findgen(s[2])+0.5)

;
; apply mask (do not considere negative pixels)
; for masked pixels, set both intensity and weight zero
;
IF keyword_Set(noMask) THEN BEGIN
  tmp = [-1]
ENDIF ELSE BEGIN
  tmp = where(m LT 0)
ENDELSE

IF tmp[0] EQ -1 THEN BEGIN
  Print,'AZIMUTH_INTEGRAL: Masked values: None'
ENDIF ELSE BEGIN
  Print,'AZIMUTH_INTEGRAL:Masked values: ', N_Elements(tmp)
  igood = LIndGen(s[1]*s[2])
  igood[tmp]=-1
  igood = igood[where(igood GE 0)]
  IF igood[0] EQ -1 THEN Message,'No good points in image.'
  m=m[igood]
  x=x[igood]
  y=y[igood]
ENDELSE

;
; computes distances from center of the pixel to center of beam
;
rr=sqrt((x-xo)^2+(y-yo)^2)
rmax=MAX(RR,min=rmin)
rmax=ceil(rmax)
rmin=floor(rmin)

print,'AZIMUTH_INTEGRAL: time for preparing arrays:',systime(1)-t0
;
; compute histogram of distances
;
t0=systime(1)
h = histogram(rr,binsize=incr,min=rmin,reverse_indices=rev, locations=xx)
; the abscissas of histogram are shifted to the center of the bin
xx=xx+dr
nh = N_Elements(xx)

print,'AZIMUTH_INTEGRAL: time for histogramming:',systime(1)-t0
;
; define output arrays
;
intensity=xx*0.0
ierror=xx*0.0


;===============================================================================
; 
; pixels contributing to main ring
;

;
; rr_rounded is a matrix of the same dimension as rr, where each value of rr has
; been substituted by the closest value in the histogram (i.e., rounded)
;
rr_rounded = rr*0
t0 = systime(1)
FOR i=0L,nh-1 DO BEGIN
  ;Set all elements of rr_rounded that are in the ith bin of rr to xx[i]
  IF REV[i] NE REV[i+1] THEN rr_rounded[Rev[Rev[I] : Rev[i+1]-1]] = xx[i]
ENDFOR
print,'AZIMUTH_INTEGRAL: time for first loop:',systime(1)-t0


; 
; Build the pixel weight for the integration
;
IF iExact NE 1 THEN BEGIN
  w=(1.-(abs(rr_rounded-rr)/(dr*2))) >0
  ; Multiply the matrix m by weight
  wm = w*m
  ; fill the output arrays
  t0 = systime(1)
  FOR i=0L,N_Elements(xx)-1 DO BEGIN
    IF REV[i] NE REV[i+1] THEN BEGIN
      num1 = total(wm[Rev[Rev[I] : Rev[i+1]-1]])
      den1 = total(w[Rev[Rev[I] : Rev[i+1]-1]])
      IF den1 NE 0 THEN intensity[i]=intensity[i]+num1/den1 ;ELSE intensity[i]=0
      ierror[i]=ierror[i]+(sqrt(num1)/den1) ; CHECK! NOT SURE
    ENDIF
  ENDFOR
  print,'AZIMUTH_INTEGRAL: time for second loop:',systime(1)-t0
;===============================================================================
ENDIF ELSE BEGIN ; exact
  ;
  ; rr_rounded is a matrix of the same dimension as rr, where each value of rr has
  ; been substituted by the closest value in the histogram (i.e., rounded)
  ;
  rr_rounded = rr*0
  rr_rounded_left = rr*0
  rr_rounded_right = rr*0
  t0 = systime(1)
  FOR i=0L,nh-1 DO BEGIN
    ;Set all elements of rr_rounded that are in the ith bin of rr to xx[i]
    IF REV[i] NE REV[i+1] THEN rr_rounded[Rev[Rev[I] : Rev[i+1]-1]] = xx[i]
    IF REV[i] NE REV[i+1] THEN rr_rounded_left[Rev[Rev[I] : Rev[i+1]-1]] = xx[i]-incr
    IF REV[i] NE REV[i+1] THEN rr_rounded_right[Rev[Rev[I] : Rev[i+1]-1]] = xx[i]+incr
  ENDFOR
  print,'AZIMUTH_INTEGRAL: time for first loop:',systime(1)-t0
  ; 
  ; Build the pixel weight for the integration
  ;
  w=(1.-(abs(rr_rounded-rr)/(dr*2))) >0
  w_left=(1.-(abs(rr_rounded_left-rr)/(dr*2))) >0
  w_right=(1.-(abs(rr_rounded_right-rr)/(dr*2))) >0
  ; Multiply the matrix m by weight
  
  wm = w*m
  wm_left = w_left*m
  wm_right = w_right*m
  
  ; fill the output arrays
  t0 = systime(1)
  FOR i=0L,N_Elements(xx)-1 DO BEGIN
    IF REV[i] NE REV[i+1] THEN BEGIN
      iInsideBin = Rev[Rev[I] : Rev[i+1]-1]
      num1 = total([wm_left[iInsideBin],wm[iInsideBin],wm_right[iInsideBin]])
      den1 = total([w_left[iInsideBin],w[iInsideBin],w_right[iInsideBin]])
        IF den1 NE 0 THEN intensity[i]=intensity[i]+num1/den1 ;ELSE intensity[i]=0
      ierror[i]=ierror[i]+(sqrt(num1)/den1) ; CHECK! NOT SURE
    ENDIF
  ENDFOR
  print,'AZIMUTH_INTEGRAL: time for second loop:',systime(1)-t0
ENDELSE
;===============================================================================
print,'AZIMUTH_INTEGRAL: Total time for azimuth_integral: ',systime(1)-t00

END
