PRO Write_EDF,Image,FILE=file,TITLE=title,bytscl=bytscl,verbose=verbose, $
   addKw=addKw



;+
; NAME:
;	WRITE_EDF
; PURPOSE:
;	this procedure writes a file in the EDF format (ESRF Data Format)
; CATEGORY:
;
; CALLING SEQUENCE:
;	write_edf,input
; INPUTS:
;	input:  a n array with an image
;
; KEYWORD PARAMETERS:
;
;	FILE: Set this keyword to the name of the output file 
;		Default: tmp.edf
;	TITLE: String with a title (stored in the EDF file header)
;	BYTSCL: If sets, the image is converted using bytscl()
;		before being written.
;	VERBOSE: If sets, prints some information (default verbose=0)
;	ADDKW: String array with keywords to be added in the header.
;              (note: the ending semicolumn should not be placed).
;	
; MODIFICATION HISTORY:
;	by M. Sanchez del Rio. ESRF. Grenoble, March 1994
;	2004-06-10 Completely rewritten. Accepts all data types. 
;	2007-12-19 srio@esrf.eu Bug fixed when writing Float header
;	2008-05-28 srio@esrf.eu added addkw kw
;
;-	
On_error,2
;
if N_elements(verbose) EQ 0  THEN verbose=0
if N_elements(bytscl) THEN bytscl=0
if not(keyword_set(file)) then file = 'tmp.edf'
if not(keyword_set(title)) then title = 'EDF file written using IDL/Write_EDF'

idType = Type(image,/verbose)

IF verbose THEN Message,/Info,'IDL type is: '+idType
CASE idType OF
  'Byte':                  idEDF = 'UnsignedByte'
  'Integer':               idEDF = 'SignedShort'
  'Unsigned Integer':      idEDF = 'UnsignedShort'
  'Longword integer':            idEDF = 'SignedInteger'
  'Unsigned Longword Integer':   idEDF = 'UnsignedInteger'
  '64-bit Integer':		 idEDF = 'SignedLong'
  'Unsigned 64-bit Integer':	 idEDF = 'UnsignedLong'
  'Floating point':              idEDF = 'FloatValue'
  'Double-precision floating':   idEDF = 'DoubleValue'
  else:	message,'EDF_WRITE: Data type not recognized: '+idType
ENDCASE
IF verbose THEN Message,/Info,'EDF type is: '+idEdf
  
ssize = size(Image)
if ssize(0) EQ 3 then nimages = n_elements(Image(0,0,*)) else nimages = 1
IF verbose THEN print,'WRITE_EDF: number of images to be written: ',nimages

CASE idType OF
  'Byte':                  nbyt = 1L
  'Integer':               nbyt = 2L
  'Longword integer':      nbyt = 4L
  'Floating point':        nbyt = 4L
  'Double-precision floating':   nbyt = 8L
  'Unsigned Integer':            nbyt = 2L
  'Unsigned Longword Integer':   nbyt = 4L
  'Unsigned 64-bit Integer':     nbyt = 8L
  '64-bit Integer':		 nbyt = 8L
  else:	Message,'data type not found: '+idType
ENDCASE

openw,Unit,file,/GET_LUN 

;
; write the header
;

for i=0,nimages-1 do begin
  if nimages EQ 1 then tmp=Image else tmp=Image(*,*,i)
  if Keyword_set(bytscl) THEN tmp=bytscl(tmp) 
  sx = n_elements(tmp(*,0))  &  sy = n_elements(tmp(0,*))
  printf,Unit,'{'
  printf,Unit,'HeaderID = EH:000001:000000:000000;'
  printf,Unit,'Image = 1 ;'
  IF nbyt GE 2 THEN BEGIN
    LITTLE_ENDIAN = (BYTE (1, 0, 1))[0]
    IF (LITTLE_ENDIAN) THEN  printf,Unit,'ByteOrder = LowByteFirst ;' ELSE $
                           printf,Unit,'ByteOrder = HighByteFirst ;'
  ENDIF
  printf,Unit,'DataType = '+idEDF+' ;'
  printf,Unit,'Dim_1 = '+strcompress(sx,/rem)+' ;'
  printf,Unit,'Dim_2 = '+strcompress(sy,/rem)+' ;'
  printf,Unit,'Size = ',strcompress(nbyt*ssize(n_elements(ssize)-1),/rem)+' ;'
  ;IF idType EQ 'Floating point' THEN printf,Unit,'ByteOrder = HighByteFirst ;'
  IF Keyword_Set(addkw) THEN BEGIN
    FOR kk=0L,N_Elements(addkw)-1 DO BEGIN
      printf,Unit,addkw[kk]+' ;'
    ENDFOR
  ENDIF
  printf,Unit,'Title = '+title+' ;'
  printf,Unit,'}'
  ;
  ; now the data block
  ;
  writeu,Unit,tmp
endfor
IF verbose THEN print,'WRITE_EDF: File '+strcompress(file,/rem)+' written to disk.'
free_lun,Unit
end
