PRO crystal_fh,str,phot_in,thetainp,text=text,out=out, forceratio=forceratio

; 
; European Synchrotron Radiation Facility (ESRF)
;+
; NAME:
;       CRYSTAL_FH
;
; PURPOSE:
;	Calculates the crystal structure factor and other parameters.
;	
; CATEGORY:
;       X-Ray optics. DABAX database.
;
; CALLING SEQUENCE:
;	crystal_fh,str,phot[,theta,text=text,out=out]
; INPUTS:
;	str: structure with inputs (as produced by bragg_calc)
;	phot: Photon energy in eV (a scalar or an array)
; OPTIONAL INPUTS:
;	theta: Scattering angle in degrees.
;		If theta is nod defined, crystal_fh considers the
;		bragg angle for the given configuration.
;	
; KEYWORDS:
;	text: a named variable where to write out a text with the
;		results of the calculations.
;	str: a named variable where to write out a structure with the
;		results of the calculations. In the case that phot is
;		an array, this is also an array of structures.
;       forceratio: set this keyword to force ratio=(sin(theta)/lambda)
;		to be value 1/2/d_spacing. If undefined, 
;               the program calculates it as ratio=sin(theta)/lambda
;
; OUTPUT:
;	The output is written in the keyword variables.
;
; PROCEDURE:
;	Adapted from crystal_fh.F (SHADOW 2.2 ray-tracing package).
;
; EXAMPLE:
;	inp = bragg_defaults()
;	out = bragg_calc(inp)
;	text = ''
;	crystal_fh,out,8000.,text=text
;
; MODIFICATION HISTORY:
;       96-07-10 Written by M. Sanchez del Rio (srio@esrf.fr)
;	2002-08-28 srio@esrf.fr adds possibility of multiple energy
;		values. Added Darwin width info.
;	2002-10-09 srio@esrf.fr modified to deal with an averaged temperature 
;		factor computed from the individual temperature factors 
;		from the different atomic sites. 
;	2006-05-18 srio@esrf.fr retrieve hc from Physical_Constants().
;		Pointed out by Eric.Le.Bigot@spectro.jussieu.fr
;       2007-05-22 srio@esrf.eu added real and imaginary parts of 
;		chi(real) and chi(imag)
;
;-
;C+++
;C	SUBROUTINE	CRYSTAL_FH
;C
;C	PURPOSE		Computes the structute factor of a crystal
;C			from data in bragg's output file.
;C
;C	ALGORITHM	
;C
;C	MODIFIED	Created by M. Sanchez del Rio  (Feb. 1996)
;C
;C---

on_error,2

if N_params() LT 1 then message,'Usage: crystal_fh,inp_str,energy,angle'

I_debug = 0
outfil = str.outfil
RN = str.rn
d_spacing = str.d_spacing
nbatom = str.nbatom
atnum = str.atnum
fract = str.fract
TEMPER = str.temper
G_0 = str.G_0
G = str.G
G_BAR = str.G_BAR
f0coeff = str.f0coeff
NPOINT = str.NPOINT
energy = str.ENERGY
fp = str.f1
fpp = str.f2

	CI	= DCOMPLEX(0.0D0,1.0D0)
;        TOANGS  =  1.239852D4 
;        TOCM    =  1.239852D-4
        TOANGS  =  physical_constants('hc')
        TOCM    =  physical_constants('hc')*1d-8

text = ''
;C
;C Computes structure factor at given wavelength and angle.
;C
	if n_elements(phot_in) EQ 0 then begin
	  phot = 0.0D0
	  read,'Input photon energy [eV]: ',phot
	endif else phot_in = double(phot_in)



;* START LOOP IN phot

FOR iphot=0,N_Elements(phot_in)-1 DO BEGIN 
	phot = phot_in[iphot]
	if n_elements(thetainp) EQ 0 then  begin
	  ;theta = 180.0D0/!dpi*asin(TOCM/PHOT/2/D_SPACING) 
	  theta = asin(TOCM/PHOT/2/D_SPACING) 
	endif else begin
	  theta = !dpi/180.0D0*thetainp
	endelse
	  ;theta = 180.0D0/!dpi*asin(TOCM/PHOT/2/D_SPACING)
;	  print,'theta Bragg [rad] is: ', theta
;	  print,'theta Bragg [deg] is: ', theta*180./!dpi
	  ;read,'Input angle [deg]: ',theta

;     message,/info,'working at energy: '+strcompress(phot,/rem)
;     message,/info,'working at angle [rads] : '+strcompress(theta,/rem)

	IF (PHOT LT ENERGY(0) OR PHOT GT ENERGY(npoint-1)) THEN BEGIN
	  message,'Incoming photon energy is out of range: '+$
		strcompress(PHOT,/REM)
	ENDIF	  
;C
;C Build the fo scattering form factor from its coefficients
;C
        IF Keyword_Set(forceratio) THEN  BEGIN
		ratio 	= 1D/2D/(D_SPACING*1D8)
	ENDIF ELSE BEGIN
		ratio 	= sin(theta)/(toangs/phot)
	ENDELSE
	if ratio GT 2 then  begin
	  message,/info, 'ratio sin(theta)/lambda > 2'
	  message,/info, 'ratio : '+strcompress(ratio,/rem)
	  message,/info, 'Paramatrizatiog for Fo may fail.'
	endif
	f0 = dblarr(nbatom)
;	message,/info, 'Ratio is : '+strcompress(ratio,/rem)

;help,f0coeff

	for j=0,nbatom-1 do begin
;print,f0coeff(j,*)
;	  if n_elements(f0coeff(0,*)) EQ 9 then begin
;	    message,/info,'Using f0_CromerMann parametrization.'
;	  endif else if n_elements(f0coeff(0,*)) EQ 11 then begin
;	    message,/info,'Using f0_WaasKirf parametrization.'
;	  endif
	  icentral = n_elements(f0coeff(0,*))
	  icentral = fix(icentral/2)
	  f0(j) = f0coeff(j,icentral)
          for i=0,icentral-1 do f0(j) = f0(j) + f0coeff(j,i) * $
		exp(-1.0d0*f0coeff(j,i+icentral+1)*ratio^2)
	  if i_debug then print,'<><>CRYSTAL_FH: F0(j) = ',F0(j)
	endfor

;C
;C Interpolate for the atomic scattering factor.
;C
	for I = 0,npoint-1 do IF (ENERGY(I) GT PHOT)	THEN GOTO, found
found:
	if i_debug then print,'Found i,energy:',i,energy(i)

	NENER	= I - 1	
	if i_debug then print,'Found nener,energy:',nener,energy(nener)
	if i_debug then print,'Found nener+1,energy:',nener+1,energy(nener+1)
	F1 = dblarr(nbatom)
	F2 = dblarr(nbatom)
	F = dcomplexarr(nbatom)
	for j=0,nbatom-1 do begin
	  F1(j)	= FP(j,NENER) + (FP(j,NENER+1) - FP(j,NENER)) * $
      		  (PHOT - ENERGY(NENER)) / (ENERGY(NENER+1) - ENERGY(NENER))
	  F2(j)	= FPP(j,NENER) + (FPP(j,NENER+1) - FPP(j,NENER)) *  $
      		  (PHOT - ENERGY(NENER)) / (ENERGY(NENER+1) - ENERGY(NENER))
	  if i_debug then print,'<><>CRYSTAL_FH: F1 = ',F1(j)
	  if i_debug then print,'<><>CRYSTAL_FH: F2 = ',F2(j)
	endfor
	R_LAM0 	= TOCM/PHOT
	if i_debug then print,'<><>CRYSTAL_FH:  R_LAM0  =', R_LAM0

	for j=0,nbatom-1 do begin
	  F(j)	= F0(j) + F1(j) + CI*F2(j)
	  if i_debug then print,'<><>CRYSTAL_FH: F = ',F(j)
	endfor
;C	
;C FH and FH_BAR are the structure factors for (h,k,l) and (-h,-k,-l).
;C
	F_0 = DCOMPLEX(0.0D0, 0.0D0)
	FH = DCOMPLEX(0.0D0, 0.0D0)
	FH_BAR = DCOMPLEX(0.0D0, 0.0D0)

; added srio@esrf.eu 2007-05-22 (requested by Claudio Ferrero)
	FHr = DCOMPLEX(0.0D0, 0.0D0)
	FHi = DCOMPLEX(0.0D0, 0.0D0)
	FH_BARr = DCOMPLEX(0.0D0, 0.0D0)
	FH_BARi = DCOMPLEX(0.0D0, 0.0D0)


        TEMPER_AVE = 1.0D0
        G_0_TOT = Total(G_0)


	for i=0,nbatom-1 do begin
;srio: DW later  	  FH 	= FH + (G(i) * F(i) * TEMPER(i) * FRACT(i)) 
	  FH 	= FH + (G(i) * F(i) * FRACT(i)) 
	  FHr 	= FHr + (G(i) * (F0(i) + F1(i)) * FRACT(i)) 
	  FHi 	= FHi + (G(i) * F2(i) * FRACT(i)) 

;C
;C The average temperature factor is made by averaging M, therefore,
;C M = (n1*M1+n2*M2)/(n1+n2) threrefore DW = DW1**(n1/(n1+n2))*DW2**(n2/(n1+n2))
;C

          TEMPER_AVE = TEMPER_AVE * (TEMPER(i))^(G_0(i)/(G_0_TOT))

;C From Brennan's private communication:
;C Changing the temperature doesn't change F0, but it does change
;C Chi0, due to changing the size of the unit cell, which changes
;C gamma.
;C MSR 96/02/14

	  F_0 	= F_0 + G_0(i) * (atnum(i) + F1(i) + CI * F2(i)) * FRACT(i) 
;srio: DW later  FH_BAR	= FH_BAR + (G_BAR(i) * F(i) * TEMPER(i) * FRACT(i))
	  FH_BAR  = FH_BAR + (G_BAR(i) * F(i) * FRACT(i))
	  FH_BARr = FH_BARr + (G_BAR(i) * (F0(i)+F1(i)) * FRACT(i))
	  FH_BARi = FH_BARi + (G_BAR(i) * F2(i) * FRACT(i))
	endfor


;C
;C multiply by the average temperature factor
;C
        FH =  FH * TEMPER_AVE
        FHr =  FHr * TEMPER_AVE
        FHi =  FHi * TEMPER_AVE
        FH_BAR = FH_BAR * TEMPER_AVE
        FH_BARr = FH_BARr * TEMPER_AVE
        FH_BARi = FH_BARi * TEMPER_AVE

	STRUCT 	= sqrt(FH * FH_BAR) 

	  if i_debug then print, '<><>CRYSTAL_FH: FH = ',FH
	  if i_debug then print, '<><>CRYSTAL_FH: FH_BAR = ',FH_BAR
	  if i_debug then print, '<><>CRYSTAL_FH: f_0 = ',f_0
	  if i_debug then print, '<><>CRYSTAL_FH: STRUCT = ',STRUCT
;C
;C   PSI_CONJ = F*( note: PSI_HBAR is PSI at -H position and is
;C   proportional to fh_bar but PSI_CONJ is complex conjugate os PSI_H) 
;C
	psi_over_f = rn*r_lam0^2/!dpi
	psi_h = rn*r_lam0^2/!dpi*fh
	 psi_hr = rn*r_lam0^2/!dpi*fhr
	 psi_hi = rn*r_lam0^2/!dpi*fhi
	psi_hbar = rn*r_lam0^2/!dpi*fh_bar
	 psi_hbarr = rn*r_lam0^2/!dpi*fh_barr
	 psi_hbari = rn*r_lam0^2/!dpi*fh_bari
	psi_0 = rn*r_lam0^2/!dpi*f_0
	psi_conj = rn*r_lam0^2/!dpi*conj(fh)
	if i_debug then print, '<><>CRYSTAL_FH: PSI_H = ',PSI_H
	if i_debug then print, '<><>CRYSTAL_FH: PSI_HBAR = ',PSI_HBAR
	if i_debug then print, '<><>CRYSTAL_FH: PSI_0 = ',PSI_0

; 
; Darwin width
;
ssvar = RN*(R_LAM0^2)*STRUCT/!DPI/SIN(2.0D0*THETA)
spvar = SSVAR*ABS(COS(2.0D0*THETA))
ssr = Double(ssvar)
spr = Double(spvar)

;C
;C computes refractive index.
;C ([3.171] of Zachariasen's book)
;C
     	REFRAC = dcomplex(1.0D0,0.0D0) - R_LAM0^2*RN*F_0/2.0D0/!dpi
	DELTA_REF  = 1.0D0 - DOUBLE(REFRAC)
	ABSORP	= 4.0d0 * !dpi *(-IMAGINARY(REFRAC)) / R_LAM0
	if i_debug then print,'<><>CRYSTAL_FH: REFRAC = ', REFRAC
	if i_debug then print,'<><>CRYSTAL_FH: DELTA_REF = ',DELTA_REF
	if i_debug then print,'<><>CRYSTAL_FH: ABSORP = ',ABSORP
	text_i =  [ $
'******************************************************',$
'Crystal data from '+str.outfil,$
'       at energy    = '+strcompress(PHOT,/rem)+' eV',$
'                    = '+strcompress(R_LAM0*1E8,/rem)+' Angstroms',$
'       and at angle = '+strcompress(THETA*180.0D0/!dpi,/rem)+' degrees',$
'                    = '+strcompress(THETA,/rem)+' rads',$
'******************************************************', $
' '	]
	tmp = ''
        for j=0,nbatom-1 do begin
          tmp = [tmp,'For atom '+strcompress(j+1,/rem)+':']
          tmp = [tmp,'       fo + fp+ i fpp = ']
          tmp = [tmp,'        '+strcompress(f0(j),/rem)+' + '+ $
		strcompress(double(f1(j)),/rem)+' + i'+ $
		strcompress(double(f2(j)),/rem)+'= ']
          tmp = [tmp,'        '+strcompress(f0(j)+f1(j)+ci*f2(j),/rem)]
          tmp = [tmp,'       Z = '+strcompress(atnum(j),/rem)]
          tmp = [tmp,'       Temperature factor = '+strcompress(temper(j),/rem)]
        endfor
	text_i = [text_i,tmp, $
'Structure factor F(0,0,0) = '+strcompress(F_0,/REM), $
'Structure factor FH = '+strcompress(FH,/REM), $
'Structure factor FH_BAR = '+strcompress(FH_BAR,/rem), $
'Structure factor F(h,k,l) = '+strcompress(STRUCT,/rem), $
' ',$
'Psi_0  = '+strcompress(psi_0,/rem), $
'Psi_H  = '+strcompress(psi_h,/rem), $
'Psi_HBar  = '+strcompress(psi_hbar,/rem), $
' ',$
'Psi_H(real) Real and Imaginary parts = '+strcompress(psi_hr,/rem), $
'Psi_H(real) Modulus  = '+strcompress(abs(psi_hr),/rem), $
'Psi_H(imag) Real and Imaginary parts = '+strcompress(psi_hi,/rem), $
'Psi_H(imag) Modulus  = '+strcompress(abs(psi_hi),/rem), $
'Psi_HBar(real) Real and Imaginary parts = '+strcompress(psi_hbarr,/rem), $
'Psi_HBar(real) Modulus  = '+strcompress(abs(psi_hbarr),/rem), $
'Psi_HBar(imag) Real and Imaginary parts = '+strcompress(psi_hbari,/rem), $
'Psi_HBar(imag) Modulus  = '+strcompress(abs(psi_hbari),/rem), $
' ',$
'Psi/F factor = '+strcompress(psi_over_f,/rem), $
' ',$
'Average Temperature factor = '+strcompress(TEMPER_AVE,/rem), $
'Refraction index = 1 - delta - i*beta :', $
'           delta = '+strcompress(DELTA_REF,/rem), $
'            beta = '+strcompress(1.0D0*IMAGINARY(REFRAC),/rem), $
'Absorption coeff = '+strcompress(ABSORP,/rem)+' cm^-1', $
' ', $
'e^2/(mc^2)/V = '+strcompress(rn,/rem)+' cm^-2', $
'd-spacing = '+strcompress(d_spacing*1.0E8,/rem)+' Angstroms',$
'SIN(theta)/Lambda = '+strcompress(Ratio,/rem), $
' ', $
'Darwin width for symmetric s-pol [microrad] = '+strcompress(2.0d6*ssr,/Rem), $
'Darwin width for symmetric p-pol [microrad] = '+strcompress(2.0d6*spr,/Rem), $
' ']

out_i = {PHOT: PHOT, THETA:THETA, F_0:F_0, FH:FH, FH_BAR:FH_BAR, $
	STRUCT:STRUCT, psi_0:psi_0, psi_h:psi_h, psi_hbar:psi_hbar, $
	DELTA_REF:DELTA_REF, REFRAC:REFRAC, ABSORP:ABSORP, RATIO:RATIO,$
	ssr:ssr, spr:spr, psi_over_f:psi_over_f}

IF iphot EQ 0 THEN text = text_i ELSE text=[text,text_i]
IF iphot EQ 0 THEN out = out_i ELSE out=[out,out_i]

ENDFOR
	
     	END
