PRO hybrid, $
    n_oe=n_oe, n_screen=n_screen, diff_plane=diff_plane, calcType=calcType, $
    focallength=focallength, distance=distance, $
    mirrorfile=mirrorfile, star=star, $
    lengthUnit=lengthUnit, npoint=npoint, showPlots=showPlots, $
    interactive=interactive, group=group, load=load1, save=save1, $
    nbins=nbins, action=action

;+
; NAME:
;     HYBRID
;
; PURPOSE:
;     Hybrid method conbining ray tracing and wave propagation based on the 
;     divergence convolution (far-field approximation)
;  
;     See Shi et al. J. Synchrotron Rad. 21 (2014)
;     http://dx.doi.org/10.1107/S160057751400650X 
;
; CATEGORY:
;	SHADOW's utilities.
;
; CALLING SEQUENCE:
;	hybrid
;
; INPUTS:
;	Input SHADOW files that must exist (extensions may change): 
;
;             screen.0102: a screen at 0 distance after the optics or an 
;                          aperture screen. Alternatively a star.01 file
;                          (set n_screen=0) or begin.dat (set n_oe=0)
;                          can be used if calcType=0
;
;             mirr.01 
;             angle.01
;             end.01
;
;             mirror.dat mirror surface profile readable by presurface
;                        (if the calcType=3)
;             star.01  (if "star" keyword is set image is convoluted and saved 
;                       for analysis (not limited by far-field approximation).
;
; OUTPUTS:
;       New SHADOW files: 
;
;            screen_hybrid.01: same as screen.0102 but with the convoluted ray 
;                         directions 
;            star_hybrid.01: The file screen_hybrid.01 ray-traced to the image 
;                         position (defined by "distance"). This is the 
;                         "far-field" image. 
;                         This file should be used for further ray-tracing.
;                            
;            star_hybrid_nf.01: if the keyword "star" is set, this file 
;                         contains the "near-field" image with the correct 
;                         ray positions obtained by convolution of the ray 
;                         tracing (star.01) and the wave-propagated field at 
;                         that  position.
;                         This file should not be used for further ray tracing.
;
; KEYWORD PARAMETERS (INPUTS IN THE PARAMETER WINDOW):
;
;       n_oe:      optical element number
;       n_screen:  screen number 
;                  Usually n_oe and n_screen are greater than zero, determining
;                  the input file screen.xxyy (xx=n_oe, yy=n_screen), but
;                  if n_oe=0 use begin.dat instead
;                  if n_screen=0 use star.xx instead
;       diff_plane: set to 'X' or 'Z' (default) to define the diffraction 
;             plane (SHADOW notation).
;       calcType: selects the type of calculation:
;                 0 = Simple aperture
;                     it computes a simple aperture by loading screen.xxyy file,
;                     propagating to a focal plane of an arbitrary lens (focal 
;                     length chosen automatically to have 50 peaks within the 
;                     aperture size), getting the angle distribution and 
;                     convolving with ray divergence, and finally saving 
;                     screen_hybrid.xxyy. The screen_hybrid.xxyy is the 
;                     output file, plus it is retraced at "distance" 
;                     into the  star_hybrid.n_oe file. These files 
;                     can be used for further ray tracing.  

;                 1 = Focusing o.e. 
;                     it loads screen.xxyy file (at zero distance after the oe),
;                     propagating to the focal plane,  getting the angle 
;                     distribution and convolving with ray divergence, 
;                     and finally saving screen_hybrid.xxyy. Also this file is
;                     ray-traced to the image position ( at "distance" 
;                     downstream) to get star_hybrid.xx output file, which 
;                     can be used for further ray tracing.  
;                 2 = Focusing o.e. with slope errors:  Same as option one, 
;                     but using the oe slope errors (must be in file 
;                     mirrrorFile)
;       focallength: focal length of the ideal lens, calculated from the mirror 
;                info 1/p+1/q=1/f) distance (propagation distance from SHADOW 
;                input q). Default: read from SIMAGE variable in end.01
;       distance: distance from screen (file screen.xxyy) to image plane
;                (star.xx).  Default: read from T_IMAGE variable in end.xx
;       star: set this flag to generate the near-field image (wave-propagated), 
;             at a star.xx position (or any other position defined by the 
;             keyword "distance").  It is saved in file star_hybrid_nf.xx
;             This is the "near field", to be compared to "far-field" which is
;             obtained by ray-tracing (no wave-propagated) the screen_new
;             to the star position.
;             Default=1
;       mirrorfile: name of file with slope errors (if calcType=2). 
;                   File format is as for SHADOW presurface. 
; 
;       lengthUnit: set to 'mm' or 'cm' (default).
;       npoint: numbet of points for FFT
;       showPlots: flag for creating output plots: 
;                  0: None, 1: Beam files (screen* files), 2: All 
;
;
;       OTHER KEYWORDS ONLY IN (COMMAND MODE)
;       save: set this keyword to create a file with the hybrid parameters 
;             that could be read again using the "load" keyword. 
;             File name is start_hybrid.xx (xx the o.e. number)
;       load: set this keyword to load a file named start_hybrid.xx 
;             (xx=n_oe) containing the inputs of the hybrid application 
;             (in format "gfile").
;       interactive: set this keyword to display the parameters window
;                    (default).
;       nbins: the number of bins for the calculation of the intensity profile 
;              of screen.xxyy.
;       action: returns 'DO' or 'DONT' if clicked in "Accept" or "Cancel"
;              in the interface.
;       group: the widget ID for the parent widget. 
;
; PROCEDURE:
;
;       See Shi et al. J. Synchrotron Rad. 21 (2014)
;       http://dx.doi.org/10.1107/S160057751400650X 
;
; EXAMPLES:
;       See example workspaces hybrid*.ws  in the example directory of
;       ShadowVUI installation. 
;
; MODIFICATION HISTORY:
;	written by Xianbo Shi, APS, February 2014
;	 2014-04-14 srio@esrf.eu Prepare for integration in shadowvui
;	 2014-06-25 srio@esrf.eu Integration in ShadowVui
;	 2014-06-27 xshi@aps.anl.gov Added the aperon flag for simple aperture
;	 2014-07-03 srio@esrf.eu merge aperon and mirroron into calcType
;	 2014-08-07 srio@esrf.eu includes bugs and items sent by Xianbo
;-

; TODO (ideas): 
;
;
; About the near field propagation for the aperture, it is a little more 
; complicated to do. Say for an aperture of size s and aperture-to-image 
; distance of d, we will have to first generate a plane field with a range 
; larger than s (how much larger depends on d). Then, applied the aperture 
; size and intensity distribution to it based on I_ray(z) obtained from 
; say screen.0101. Then propagate it in free space directly to d. Now the 
; problem is what should we convolute with at this point. I think we should 
; convolute the ideal image size at d from ray tracing, not the direct 
; beam size. To get that, we would need to use screen.0101, add a ideal 
; lens in ray tracing with parameters to focus the beam from whereever the 
; virtual source is to d.
; At this time, I think it maybe a good idea not to go into this yet.
;
;
;      - switch on-off lens:
;               -No lenses for an aperture. How to replace focallength?
;               -Automatic calculation of focallength? Could be done
;               by inserting a new screen at zero distance before the o.e.
;               and calculate the focallength from the mean deviatio in angle?
;      - include the wave calculations in an ad-hoc library.
;      - 2D method
;
;
;       For now, to trace the entire beamline, the user will have to run this 
;       code for individual OE and use the screen_hybrid file to trace the 
;       next OE, and so on.
;
;       In the future, we will add the far field calculation into the ray 
;       tracing itself. 
;       How to do it? Use flags for each OE? or a new "trace with diffraction" 
;       option?
; to keep inputs in memory for further runs
COMMON hybrid_common,str1,str1titles,str1flags

;on_error,2

forward_function histosh,readsh,getshcol,retrace

if n_elements(interactive) eq 0 then interactive = 1

;
; input structure str1 (defined here, and reused via the common block)
;

IF n_elements(str1) EQ 0 THEN BEGIN
     str1 = {  $
             n_oe:1, $
             n_screen:2, $
             diff_plane:['1','X (using cols 1,4)','Z (using cols 3,6)'], $  
             calcType:['0','Simple Aperture','Focusing o.e.','Focusing o.e. with slope errors'],$
             focallength:-1d0, $
             distance:-1d0, $
             mirrorFile:'mirror.dat', $
             star:['1','No','Yes'], $
             lengthUnit:['1','cm','mm'], $
             npoint:100000L, $
             showPlots:['1','None [expert]','Beam [default]','All [verbose]'] }
     str1titles=[ $
                 'o.e. number to apply hybrid calculation:',$
                 'screen number at zero distance from o.e.:',$
                 'Diffraction plane (SHADOW coords):',$
                 'Type of calculation:',$
                 'Focal length (if <=0 use SHADOW set):',$
                 'Distance to image (if <=0 use SHADOW set):',$
                 ;'Mirror slope errors:',$
                 'Mirror slope errors file:',$
                 'Create wave-propagated near field star_hybrid_nf.01 file?', $
                 'Length unit in use:',$
                 'Number of points for FFT:', $
                 'Create plots?']
     str1flags = replicate('1',n_elements(str1titles))
     str1flags[4] = 'w(3) GT 0'
     ;str1flags[5] = 'w(3) GT 0'
     ;str1flags[6] = 'w(3) GT 0'
     str1flags[7-1] = 'w(3) GT 1' ;  AND w(6) EQ 1'
     str1flags[8-1] = 'w(3) GT 0'
ENDIF

;
; copy keywords into str1
;
if (n_elements(n_oe) gt 0) then str1.n_oe = fix(n_oe)
if (n_elements(n_screen) gt 0) then str1.n_screen = fix(n_screen)
if keyword_set(diff_plane) then begin
   if diff_plane eq 'X' then str1.diff_plane[0] = '0'
   if diff_plane eq 'Z' then str1.diff_plane[0] = '1'
endif

if (n_elements(calcType) gt 0) then str1.calcType[0] = StrCompress(calcType,/remove_all)
if (n_elements(focallength) gt 0) then str1.focallength = focallength
if (n_elements(distance) gt 0) then str1.distance = distance
if keyword_set(mirrorfile) then str1.mirrorfile = mirrorfile
if (n_elements(star) gt 0) then str1.star[0] = strcompress(star,/remove_all) 
if keyword_set(lengthUnit) then begin
   case lengthUnit of
     'cm': str1.lengthUnit[0] = '0'
     'mm': str1.lengthUnit[0] = '1'
     else: message,'invalid lengthUnit'
   endcase
endif
if keyword_set(npoint) then str1.npoint = npoint
if (n_elements(showPlots) gt 0) then str1.showPlots[0] = strcompress(showPlots,/remove_all)
if (n_elements(interactive) gt 0) then interactive=interactive
if not(keyword_set(nbins)) then nbins=100

; 
; load input file (if wanted)
;
if keyword_set(load1) then begin
  str_n_oe = string(str1.n_oe,format="(I02)")
  file = 'start_hybrid.'+str_n_oe
  if checkfile(file) EQ 0 then begin
    print,'%HYBRID: File not found: '+file
  endif else begin
    tmp = read_gfile(file)
    copy_structure,tmp,str1
    print,'%HYBRID: Inputs loaded from file: '+file
  endelse
endif

;
; edit parameters if interactive kw is set
;
if keyword_set(interactive) then begin
  xscrmenu,str1,action=action,/interpret,titles=str1titles, $
     wtitle='Hybrid',/NoType, flags=str1flags, ncol=2, $
     help="xhelp,'hybrid'"
  if action EQ 'DONT' then return
endif
widget_control,/hourglass
;
; turn mirror off in case of aperture
;
if fix(str1.calcType[0]) eq 0 then begin
  str1.star[0] = '0'
endif

; open output file
openw,funit,'hybrid.txt',/get_lun
printf,funit,' '
printf,funit,' '
printf,funit,'  =============== hybrid ray-tracing + wave-optics ============ '
printf,funit,' '

  ;~~~~~~~~~~~~~~~~~~~~~~~~~~
  ; setup inputs
  ;~~~~~~~~~~~~~~~~~~~~~~~~~~

  if fix(str1.lengthUnit[0]) EQ  1 then begin
      angstroms2userUnit = 1d-7  ; using mm as user units
      sUnit = '[mm]'
  endif else begin
      angstroms2userUnit = 1d-8  ; using cm as user units (ShadowVUI default)
      sUnit = '[cm]'
  endelse
  printf,funit,'Using units: '+sUnit

  ; 
  ; define SHADOW files to be used: 
  ;
  str_n_oe = string(str1.n_oe,format="(I02)")
  str_n_screen = string(str1.n_screen,format="(I02)")

  ; convention:
  ; using by default screen.xxyy file, but: 
  ;     if yy=0, use star.xx
  ;     if xx=0  use begin.dat
  if str_n_oe EQ '00' then begin 
      fileShadowScreen = 'begin.dat'
      fscreen_hybrid = 'begin_hybrid.dat'
  endif else begin
      if str_n_screen EQ '00' then begin
          fileShadowScreen = 'star.'+str_n_oe
          fscreen_hybrid = 'star_hybrid.'+str_n_oe
      endif else begin
          fileShadowScreen = 'screen.'+str_n_oe+str_n_screen
          fscreen_hybrid = 'screen_hybrid.'+str_n_oe+str_n_screen
      endelse
  endelse
  
  fileShadowAngle = 'angle.'+str_n_oe
  fileShadowEnd = 'end.'+str_n_oe
  fileShadowMirr = 'mirr.'+str_n_oe

  fileShadowPresurface = str1.mirrorfile
  fileShadowStar = 'star.'+str_n_oe

  printf,funit,'  '
  printf,funit,'Using SHADOW files:  '
  printf,funit,'   screen:           '+fileShadowScreen
  if fix(str1.calcType[0]) gt 0 then begin
    printf,funit,'   o.e.:             '+fileShadowMirr
    printf,funit,'   angles:           '+fileShadowAngle
    printf,funit,'   end [parameters]: '+fileShadowEnd
  endif
  ; 
  ; check files exist
  ; 
  files = [fileShadowScreen,fileShadowEnd,fileShadowAngle,fileShadowMirr]
  
  for i=0,1 do begin
      if checkfile(files[i]) ne 1 then begin
        wtmp = dialog_message(/Error,'File not found: '+files[i],dialog_parent = group)
        printf,funit,'Error: File not found: '+files[i]
        goto,out
      endif
  endfor
  
  if fix(str1.calcType[0]) gt 0 then begin
      for i=2,n_elements(files)-1 do begin
          if checkfile(files[i]) ne 1 then begin
            wtmp = dialog_message(/Error,'File not found: '+files[i],dialog_parent = group)
            printf,funit,'Error: File not found: '+files[i]
            goto,out
          endif
      endfor
  endif
  if fix(str1.calcType[0]) eq 2 then begin
      if checkfile(fileShadowPresurface) ne 1 then begin
         wtmp = dialog_message(/Error,'File not found: '+fileShadowPresurface,dialog_parent = group)
         printf,funit,'Error: File not found: '+fileShadowPresurface
         goto,out
      endif
      printf,funit,'   o.e. errors:      '+fileShadowPresurface
  endif
  if fix(str1.star[0]) eq 1 then begin
      if checkfile(fileShadowStar) ne 1 then begin
         wtmp = dialog_message(/Error,'File not found: '+fileShadowStar,dialog_parent = group)
         printf,funit,'Error: File not found: '+fileShadowStar
         goto,out
      endif
      printf,funit,'   star (image):     '+fileShadowStar
  endif

  ; 
  ; other checks
  ; 
  gg = read_gfile(fileShadowEnd)
  if fix(str1.calcType[0]) gt 0 then begin
    if gg.F_ANGLE ne 1 then begin
      wtmp = dialog_message(/Error,'File with angles not created: '+fileShadowAngle,dialog_parent = group)
      return
    endif
    if abs(gg.SL_DIS[str1.n_screen-1]) gt 1d-8 then begin
      wtmp = dialog_message(/Error,'The aperture screen must be placed at ZERO distance from o.e. Found at distance: '+strcompress(gg.SL_DIS[1]))
      return
    endif
  endif
  if (str_n_oe NE '00') then begin
    if gg.N_SCREEN lt 1 then begin
       wtmp = dialog_message(/Error,'At least one screen must exist. Found: '+ $
              strcompress(gg.N_SCREEN))
       return
    endif
  endif

  ;
  ;parameters
  ;

  ; by default diffraction is in Z using SHADOW cols 3,6 
  ; if diffraction is in X, using cols 1,4 (i.e. those of Z minus DeltaCol) 
  if ( fix(str1.diff_plane[0]) EQ 0 ) then begin ; X
    DeltaCol = 2
    sZ = 'X'
    mirrorSagittal = 1
  endif else begin ; default, Z
    DeltaCol = 0
    sZ = 'Z'
    mirrorSagittal = 0
  endelse

  if fix(str1.calcType[0]) gt 0 then begin
    If str1.focallength le 0d0 Then BEGIN
      txt = 'Focal length not set (or <= 0), taken from focusing distance (SIMAG) in :'+fileShadowEnd
      print,'% HYBRID: '+txt
      printf,funit,''
      printf,funit,''
      printf,funit,txt
      ; srio QUESTION:
      ; > What I see not very "generic" in your code is that we define the
      ; > focusing mirror in SHADOW, and then we define again the focusing lens
      ; > (focal length) in hybrid. This is quite redundant and I am thinking how
      ; > to automatize to get a more automatic  code. From SHADOW we could obtain
      ; > in some way the change of divergence for each ray (versus Z of the
      ; > screen perhaps), and from that calculate a mean focal distance, in a
      ; > similar way that you calculate the mean photon energy.  Or even get the
      ; > focal distance from the ellipse/sphere parameters.
      ; 
      ; Xianbo:
      ; In case of mirror, the focal length of the lens should be automatically 
      ; loaded from SHADOW inputs. It should be from the ellipse parameters 
      ; (SIMAG). I make them as manual inputs just for simplicity.
      ; 
  
      ; COMMENT
      ; note that t=in the simulation we suppose a plane incident wave, so 
      ; we use SIMAG as focal distance. If a point source would be considered
      ; we'd use 1/focallength = 1/SSOUR + 1/SIMAG
      str1.focallength = gg.SIMAG
    Endif 
    if str1.focallength NE gg.SIMAG then begin
       wtmp = dialog_message(/Question,['defined focal length ('+$
          strcompress(str1.focallength)+') different from SHADOW defined ('+$
          strcompress(gg.SIMAG)+')','Copy and use SHADOW one to focal length?'] )
          
       if wtmp EQ 'Yes' then str1.focallength = gg.SIMAG
    endif
    txt = 'Focal length for lens-like focusing [mm]: '+strcompress(str1.focallength)
    print,'% HYBRID: '+txt
    printf,funit,' '
    printf,funit,txt
  endif
  
  
  If str1.distance le 0d0 THEN BEGIN
      txt = 'propagation distance not set (or <= 0), used values from q-distance (T_IMAGE) in  :'+fileShadowEnd
      print,'% HYBRID: '+txt
      printf,funit,txt
      if (str_n_oe NE '00' ) then str1.distance = gg.T_IMAGE
  Endif 
  
  if (str_n_oe NE '00' ) then begin
  if str1.distance NE gg.T_IMAGE then begin
     wtmp = dialog_message(/Question,['defined oe-plate distance ('+$
        strcompress(str1.distance)+') different from SHADOW defined ('+$
        strcompress(gg.T_IMAGE)+')','Copy and use SHADOW one to distance?'] )
        
     if wtmp EQ 'Yes' then begin
         str1.distance = gg.SIMAG
         txt = 'oe-plate distance copied from SHADOW: '+strcompress(str1.distance)
     endif
  endif
  endif
  txt = 'Propagation distance (oe-image plane) [mm]: '+strcompress(str1.distance)
  print,'% HYBRID: '+txt
  printf,funit,' '
  printf,funit,txt


  ;~~~~~~~~~~~~~~~~~~~~~~~~~~
  ;Load shadow files
  ;~~~~~~~~~~~~~~~~~~~~~~~~~~
  ;read screen file at the optics exit
  screen = ReadSh(fileShadowScreen,verbose=0)
  ;srio1
  zz_screen = reform(getshcol(screen,3-DeltaCol,nolost=1,verbose=0),/OVERWRITE)
  xp_screen = reform(getshcol(screen,4,nolost=1,verbose=0),/OVERWRITE)
  yp_screen = reform(getshcol(screen,5,nolost=1,verbose=0),/OVERWRITE)
  zp_screen = reform(getshcol(screen,6,nolost=1,verbose=0),/OVERWRITE)
  ref_screen = reform(getshcol(screen,23,nolost=1,verbose=0),/OVERWRITE)

  flg = reform(getshcol(screen,10,nolost=0,verbose=0),/OVERWRITE)
  keep = where(flg gt 0)
  z_min = min(zz_screen,max=z_max)
      
  printf,funit,''
  printf,funit,'Diffraction plane is: '+sZ
  printf,funit,''
  printf,funit,'Screen coordinate extrema '+sUnit+': ',z_min,z_max

  ;~~~~~~~~~~~~~~~~~~~~~~~~~~
  ;Process mirror
  ; -reads file with mirror height mesh
  ; -calculates the variation of the "incident angle" and the "mirror height"
  ;  versus the Z coordinate in the screen. 
  ;~~~~~~~~~~~~~~~~~~~~~~~~~~
  If fix(str1.calcType[0]) eq 2 Then begin
    printf,funit,''
    printf,funit,'Using slope errors from file: '+str1.mirrorfile
    ;
    ; get mirror data from files (mirror coord, incident angle, presurface 
    ; profile)
    ;

    ;read mirr file in the mirror coordinate
    ; srio1
    if ( mirrorSagittal eq 0 ) then begin ; default, Y
      yy_mirr = reform(getshcol(fileShadowMirr,2,nolost=1,verbose=0),/OVERWRITE)
    endif else begin ; X
      yy_mirr = reform(getshcol(fileShadowMirr,1,nolost=1,verbose=0),/OVERWRITE)
    endelse

    ;read in angle files    
    angleall = DblArr(4,screen.npoint)
    openr, lun, fileShadowAngle, /get_lun
    readf, lun, angleall
    free_lun, lun
    angle_inc = reform(angleall[1,keep])
    ; from incident angle in degree to grazing angle in mrad
    angle_inc = (90.0D - angle_inc)/180.0D*1e3*!dpi 

    ;read in mirror surface using presurface
    ; y is the coordinate for the 1d figure error
    presurface,fileShadowPresurface,surface1,/noplot
    ; wmirror stores the full mesh
    wmirror = wave_setscale(Double(surface1.z),surface1.x[0],surface1.x[1]-surface1.x[0],surface1.y[0],surface1.y[1]-surface1.y[0],/P)  ;2d surface

    if ( mirrorSagittal EQ 0 ) then begin
        ; wmirror_l stores the central profile z(y) for x=0
        ; 1d cut at x=0 of the mirror surface
        wmirror_l = wave_setscale(reform( $
          wave_findvalue(wmirror,0.0,surface1.y) $
          ),surface1.y[0],surface1.y[1]-surface1.y[0],/P) 
    endif else begin 
        ; wmirror_l stores the central profile z(x) for y=0
        ; 1d cut at y=0 of the mirror surface
        wmirror_l = wave_setscale(reform( $
          wave_findvalue(wmirror,surface1.x,0.0) $
          ),surface1.x[0],surface1.x[1]-surface1.x[0],/P) 
    endelse
    

    ;generate theta(z) curve over a continuous grid 
    npoly_angle = 3
    fitcoef_angle = Poly_fit(zz_screen,angle_inc,npoly_angle,/DOUBLE)
    wangle = wave_setscale(dindgen(1001),z_min,z_max,/I)
    wangle.w = fitcoef_angle[0]
    for i = 1, npoly_angle do begin
      wangle.w += fitcoef_angle[i]*wangle.x^i
    endfor
    ;generate l(z) curve over a continuous grid
    npoly_lz = 6
    fitcoef_lz = Poly_fit(zz_screen,yy_mirr,npoly_lz,/DOUBLE)
    wlz = wave_setscale(dindgen(1001),z_min,z_max,/I)
    wlz.w = fitcoef_lz[0]
    for i = 1, npoly_lz do begin
      wlz.w += fitcoef_lz[i]*wlz.x^i
    endfor
    if fix(str1.showPlots[0]) GE 2 then begin
        tmpX = zz_screen
        tmpAngleFit =  fitcoef_angle[0]
        tmpLzFit    =  fitcoef_lz[0]
        for i = 1, npoly_angle do begin
          tmpAngleFit += fitcoef_angle[i]*tmpX^i
        endfor
        for i = 1, npoly_lz do begin
          tmpLzFit += fitcoef_lz[i]*tmpX^i
        endfor
        xplot,parent=p, $
        make_set(zz_screen,angle_inc,tmpAngleFit,yy_mirr,tmpLzFit), $
        title='Mirror coordinates and fitted values',xtitle='-1',ytitle='-1', $
        coltitles=['Z in screen '+sUnit, $
         'Theta (grazing) [mrad] in o.e.','Theta (grazing) [mrad] in o.e. FIT',$
         'Height in o.e. '+sUnit,'Height in o.e. FIT '+sUnit]
        xplot_controls_action,p,psymsign='1',psymbol=3
    endif 
    tmps = 'angle(z) = '
    for i = 0, npoly_angle do begin
      tmps += strcompress(fitcoef_angle[i]) + '*z^'+strcompress(i,/remove_all)
    endfor
    printf,funit,''
    printf,funit,'Fit on o.e. grazing angle [mrad] vs screen z: '
    printf,funit,tmps

    tmps = 'height(z) = '
    for i = 0, npoly_lz do begin
      tmps += strcompress(fitcoef_lz[i]) + '*z^'+strcompress(i,/remove_all)
    endfor
    printf,funit,''
    printf,funit,'Fit on o.e. height vs screen z: '
    printf,funit,tmps

  Endif

  ;~~~~~~~~~~~~~~~~~~~~~~~~~~
  ;hybrid initilization block
  ;~~~~~~~~~~~~~~~~~~~~~~~~~~
  ;generate intensity profile (histogram): I_ray(z) curve 
  zz_hist = histosh(zz_screen,ref_screen,nbins=nbins,binsize=binsize) ;shifted histo values are at center of each bin
  Iray = reform(zz_hist[1,*])
  wIray = wave_setscale(Iray,zz_hist[0,0],binsize,/P)
  if fix(str1.showPlots[0]) GE 1 then begin ; 
      xplot,make_set(wIray.x,wIray.w),$
        xtitle='-1', ytitle='-1', $
        title='Intensity distribution at screen (file:'+fileShadowScreen+')', $
        coltitles=[sZ+' '+sUnit,'Intensity']
  endif

  ;average photon energy in eV
  energy = MEAN(getshcol(screen,11,nolost=1,verbose=0),/DOUBLE) 
  print,'% HYBRID: '+'Using MEAN photon energy [eV]: ',energy

  ;waLe = 12398.4172D-7/energy ;in mm, wavelength
  ;knum = 2.0D*!dpi/waLe          ;in mm^-1, wavenumber

  wale = MEAN(getshcol(screen,19,nolost=1,verbose=0),/DOUBLE) ; wavelength in A
  wale = wale * angstroms2userUnit
  print,'% HYBRID: '+'Using MEAN photon wavelength [A]: ',wale
  knum = 2.0D*!dpi/waLe        ;in [user-unit]^-1, wavenumber

  if fix(str1.calcType[0]) eq 0 then begin
    str1.focallength = (z_max-z_min)*(z_max-z_min)/wale/50.0D
    ;str1.distance = str1.focallength
  endif
  ;~~~~~~~~~~~~~~~~~~~~~~~~~~
  ;Wavepropagation block
  ;~~~~~~~~~~~~~~~~~~~~~~~~~~
  ;make plane wave
  plane = Dcomplexarr(str1.npoint)
  plane[*] = Dcomplex(1.0D,0.0D)
  wplane = wave_setscale(plane,z_min,z_max,/I)
  
  ;scale plane wave by I_ray(z)
  intscale = wave_findvalue(wIray,wplane.x)
  wplane.w[*] *= sqrt(intscale[*])

  if fix(str1.showPlots[0]) GE 2 then begin ; prepare next plot
      tmp1 = (abs(wplane.w))^2
      tmp2 = atan(imaginary(wplane.w)/real_part(wplane.w))*180d0/!dpi
      tmp0 = tmp1*0.0
      tmpplot = make_set(wplane.x, tmp1,tmp2,tmp0,tmp0,tmp0,tmp0,$
        tmp0,tmp0,tmp0,tmp0)
      coltitles=[sZ+' coord','Intensity(screen)','Phase [deg](screen)',$
         'Void','Void','Void','Void','Void','Void','Void','Void']
  endif
  
  ;apply ideal lens
  wplane.w[*] *= exp(Dcomplex(0.0D,-knum*wplane.x[*]*wplane.x[*]/str1.focallength/2.0D))
  if fix(str1.showPlots[0]) GE 2 then begin ; prepare next plot
      tmpplot[3,*] = (abs(wplane.w))^2
      tmpplot[4,*] = atan(imaginary(wplane.w)/real_part(wplane.w))*180d0/!dpi
      coltitles[3] = 'Intensity after lens'
      coltitles[4] = 'Phase [deg] after lens'
  endif
  
  ;apply mirror figure error if selected
  If fix(str1.calcType[0]) eq 2 Then begin
    ;find l values for all z values in the plane wave
    value_l = wave_findvalue(wlz,wplane.x)        
    ;find mirror height values for all z values (from l values calculated above)
    height = wave_findvalue(wmirror_l,value_l)          
    height -= (max(height)+min(height))/2.0D ;Shift the mirror height
    mirr_angle = wave_findvalue(wangle,wplane.x)*1.0D-3 ;in rad
    wplane.w[*] *= exp(Dcomplex(0.0D,-2.0D*knum*sin(mirr_angle[*])*height[*]))
    if fix(str1.showPlots[0]) GE 2 then begin ; prepare next plot
        tmpplot[5,*] = (abs(wplane.w))^2
        tmpplot[6,*] = atan(imaginary(wplane.w)/real_part(wplane.w))*180d0/!dpi
        coltitles[5] = 'Intensity after mirror'
        coltitles[6] = 'Phase [deg] after mirror'
    endif
  Endif
  
  ; srio QUESTION:  the code needs a focusing element (a focal distance).
  ;        What about if there is no focusing? just the diffraction by a slit 
  ;        of a collimated (plane wave) beam, or a point source (spherical). 
  ;        It is not complicated: just avoid applying the lens and propagate 
  ;        to an "arbitrary" long distance to get the increase in divergences...
  ; Xianbo: You are right, it could be done without a lens and propagate to 
  ;        a really large distance (ensure far field). However, the image will 
  ;        be larger than the aperture size (w). So you will need to play 
  ;        with the intial grid ranges and number of grid points. If we put a 
  ;        lens and propagate to the focal point, we can get the same far 
  ;        field result. The focal length of the lens can be any number. But 
  ;        if we choose it carefully, say focal length=w^2/lambda/n, (w is 
  ;        aperture size, n is the number of peaks in the diffraction 
  ;        pattern), the image can be a similar size of the aperture. 
  ;        In this case, the minimum number of points is needed for the FFT. 
  ;        Of course, this should be chosen automatically without user inputs. 
  ;        Also, for the aperture, this is only to provide divergence change 
  ;        after the aperture (in far field approximation). 
  ;        I don't see how to get the near field image by convolution.


  ; propagate in free space up to two different positions: 
  ; "distance" and "focallength" 

  ; 1) compute FFT of the field
  fft_delta = 1.0D/double(wplane.xsize)/wplane.xdelta
  If (str1.npoint mod 2) eq 1 then begin
    fft_offset = -fft_delta*double(str1.npoint-1)/2.0D
  endif else begin
    fft_offset = -fft_delta*double(str1.npoint)/2.0D
  endelse
  if float(!version.release) GT 8.1 then begin 
      wfou_fft = wave_setscale(fft(wplane.w,/double,/center),fft_offset,fft_delta,/P)
  endif else begin  ; CENTER kw not accepted
      wfou_fft = wave_setscale(shift(fft(wplane.w,/double),str1.npoint/2),fft_offset,fft_delta,/P)
  endelse

  ; 2) propagate in the Fourier space and back to real space 
  If fix(str1.star[0]) eq 1 then begin
    ;propagate to image (star position)
    ; (results are wstar and wInt_star)
    wfou_fft_star = wfou_fft
    wfou_fft_star.w[*] *= exp(Dcomplex(0.0D,-!dpi*waLe*str1.distance*wfou_fft_star.x[*]*wfou_fft_star.x[*])) 
    wstar = wplane
    wstar.w = fft(wfou_fft_star.w,/double,/inverse)
    if fix(str1.showPlots[0]) GE 2 then begin ; prepare next plot
        tmpplot[7,*] = (abs(wstar.w))^2
        tmpplot[8,*] = atan(imaginary(wstar.w)/real_part(wstar.w))*180d0/!dpi
        coltitles[7] = 'Intensity at '+strcompress(str1.distance)
        coltitles[8] = 'Phase [deg] at '+strcompress(str1.distance)
    endif
    ;Intensity profile
    wInt_star = wave_setscale(abs(wstar.w)*abs(wstar.w),wstar.xoffset,wstar.xdelta,/P)
  Endif

  ;for divergence, propagate to focallength 
  ; (results are wplane and wInt_angle)
  wfou_fft.w[*] *= exp(Dcomplex(0.0D,-!dpi*waLe*str1.focallength*wfou_fft.x[*]*wfou_fft.x[*])) 
  wplane.w = fft(wfou_fft.w,/double,/inverse)
  if fix(str1.showPlots[0]) GE 2 then begin ; prepare next plot
      tmpplot[9,*] = (abs(wplane.w))^2
      tmpplot[10,*] = atan(imaginary(wplane.w)/real_part(wplane.w))*180d0/!dpi
      coltitles[9] = 'Intensity at '+strcompress(str1.focallength)
      coltitles[10] = 'Phase [deg] at '+strcompress(str1.focallength)

      xplot,tmpplot,title='Wavefront intensities and phases', $
         coltitles=coltitles, $
         xtitle='-1',ytitle='-1',wtitle='Wavefront intensities and phases'
  endif

  ;angle intensity profile as a function of tan(dx) 
  ;wInt_angle = wave_setscale(abs(wplane.w)*abs(wplane.w),wplane.xoffset/str1.distance,wplane.xdelta/str1.distance,/P)
  wInt_angle = wave_setscale(abs(wplane.w)*abs(wplane.w),wplane.xoffset/str1.focallength,wplane.xdelta/str1.focallength,/P)

  ;help,/str,wInt_angle,wplane
  if fix(str1.showPlots[0]) GE 1 then begin ; prepare next plot
      xplot,make_set(wInt_angle.x,wplane.x,wInt_angle.w),$
        ;xrange=[-2,2]*1d-6, xtitle='-1', ytitle='-1', $
        xtitle='-1', ytitle='-1', $
        title='Wavefront propagation: Intensity at focallength ('+$
        strcompress(str1.focallength,/remove_all)+' '+sUnit+')', $
        coltitles=['Angle [rads]',sZ+' '+sUnit,'Intensity']
  endif

  ;~~~~~~~~~~~~~~~~~~~~~~~~~~
  ;Convolution block
  ;~~~~~~~~~~~~~~~~~~~~~~~~~~
  ;create the cumulative distribution function on intensity vs angle
  ;at focus (wInt_angle)
  wCDF = wInt_angle
  for i = 1L, wCDF.xsize-1 do begin
    wCDF.w[i]+=wCDF.w[i-1]
  endfor
  wCDF.w/=wCDF.w[wCDF.xsize-1]
  ;generate random direction shift for each good ray
  S = size(zz_screen)
  angle_shift = DblArr(S[1])
  seed = 0.2

  ; plot CDF   xplot,wCDF.x,wCDF.w

  ;~~~~~~~~~~~~~~~~~~~~~~~~~~
  ;To generate the screen_hybrid.???? for further ray tracing
  ;~~~~~~~~~~~~~~~~~~~~~~~~~~ 
  if float(!version.release) GT 8.2 then begin  
      angle_shift = atan(Interpolate(wCDF.x, FINDEX(wCDF.w,randomu(seed,S[1],/DOUBLE)), /DOUBLE, /GRID))
  endif else begin ; DOUBLE kw not accepted
      angle_shift = atan(Interpolate(wCDF.x, FINDEX(wCDF.w,randomu(seed,S[1],/DOUBLE)), /GRID))
  endelse
  ;srio1
  if ( DeltaCol EQ 0 ) then begin
      angle_ray = atan(zp_screen/yp_screen)
      angle_x = atan(xp_screen/yp_screen)
  endif else begin 
      angle_ray = atan(xp_screen/yp_screen)
      angle_x = atan(zp_screen/yp_screen)
  endelse
  angle_conv = angle_ray + angle_shift
  ;direction cosine normalization
  num = sqrt(1.0D + tan(angle_x)*tan(angle_x) + tan(angle_conv)*tan(angle_conv))  
  if ( DeltaCol EQ 0 ) then begin
      screen.ray[3,keep] = tan(angle_x)/num
      screen.ray[4,keep] = 1.0D/num
      screen.ray[5,keep] = tan(angle_conv)/num
  endif else begin 
      screen.ray[3,keep] = tan(angle_conv)/num
      screen.ray[4,keep] = 1.0D/num
      screen.ray[5,keep] = tan(angle_x)/num
  endelse
  putrays,screen,fscreen_hybrid
  txt = 'File written to disk: '+fscreen_hybrid
  print,'% HYBRID: '+txt
  printf,funit,''
  printf,funit,txt
  

  ; 
  ; now ray-trace ("far-field") the new screen file to the star position 
  ; 
  ;putrays,retrace('screen_hybrid.0102',dist=str1.distance),'star_hybrid.01'
  ;if fix(str1.calcType[0]) gt 0 then begin
  ;putrays,retrace(fscreen_hybrid,dist=str1.distance,/resetY),'star_hybrid.'+str_n_oe


  putrays,retrace(screen,dist=str1.distance,/resetY),'star_hybrid.'+str_n_oe
  txt = 'File written to disk (screen_hybrid ray-traced to star position at '+$
     strcompress(str1.distance,/remove_all)+'): star_hybrid.'+str_n_oe 
  print,'% HYBRID: '+txt
  printf,funit,''
  printf,funit,txt



  ;endif

  ;~~~~~~~~~~~~~~~~~~~~~~~~~~
  ;To generate the star_hybrid_nf.01 file for image analysis
  ;The convoluted wave propagation profile is at the image plane 
  ;The rays from SHADOW are first traced to the focal plane of the mirror
  ; and then convoluted
  ;~~~~~~~~~~~~~~~~~~~~~~~~~~  
  If fix(str1.star[0]) EQ 1 then begin
    star = ReadSh(fileShadowStar,verbose=0) 
    ; I also found another issue about the hybrid code. I think for the 
    ; near-field calculation we have to make sure the convoluted shadow image 
    ; is at the focal plane of the mirror. Otherwise we will double count 
    ; the size broadening.
    dist_tofocal = str1.focallength - gg.T_IMAGE
    star = retrace(star,dist=dist_tofocal)

    wCDF_star = wInt_star
    for i = 1L, wCDF_star.xsize-1 do begin
      wCDF_star.w[i]+=wCDF_star.w[i-1]
    endfor
    wCDF_star.w/=wCDF_star.w[wCDF_star.xsize-1]
    ;generate random position shift for each good ray
    flg_star = (star.ray)[9,*]
    ;help,flg_star
    keep_star = where(flg_star gt 0, count_star)
    zz_shift = DblArr(count_star)
    seed = 0.2
    if float(!version.release) GT 8.2 then begin
        zz_shift = Interpolate(wCDF_star.x, FINDEX(wCDF_star.w,randomu(seed,count_star,/DOUBLE)), /DOUBLE, /GRID)
    endif else begin
        zz_shift = Interpolate(wCDF_star.x, FINDEX(wCDF_star.w,randomu(seed,count_star,/DOUBLE)), /GRID)
    endelse
    ;the ray tracing part has to be at the focal plane to avoid double counting of the source size
    zz_ray = reform(star.ray[2-DeltaCol,keep_star]+(str1.focallength-str1.distance)*(star.ray[5-DeltaCol,keep_star]/star.ray[4,keep_star]))
    zz_conv = zz_ray + zz_shift
    star.ray[2-DeltaCol,keep_star] = zz_conv
    ;
    ;       srio: QUESTION: should we use the new divergences of screen?
    ;       xianbo: I don't think so. I think the near field calculation could 
    ;       only be used to get the image size at a certain beam position. 
    ;       This part is not even in the paper. It cannot be use for further 
    ;       ray tracing. So maybe we shouldn't even try to save the "star.new" 
    ;       file, but just save the intensity profile if needed.
    ; star[3,*] = screen.ray[3,*] 
    ; star[4,*] = screen.ray[4,*] 
    ; star[5,*] = screen.ray[5,*] 
    putrays,star,'star_hybrid_nf.'+str_n_oe
    txt = 'File written to disk: star_hybrid_nf.'+str_n_oe
    print,'% HYBRID: '+txt
    printf,funit,''
    printf,funit,txt
  Endif

  ;~~~~~~~~~~~~~~~~~~~~~~~~~~
  ;This block is for testing only, doesn't need for SHADOW implementation
  ;~~~~~~~~~~~~~~~~~~~~~~~~~~

if fix(str1.showPlots[0]) GE 2 AND fix(str1.calcType[0]) gt 0 then begin
  ;;;;far field ray tracing
  zz_conv_ff = zz_screen+str1.distance*tan(angle_conv) 
  zz_conv_ff_hist = histosh(zz_conv_ff,nbins=100,xrange=[-5e-4,5e-4],binsize=binsize) ;shifted histo values are at center of each bin
  I_ff_conv = reform(zz_conv_ff_hist[1,*])
  wI_ff_conv = wave_setscale(I_ff_conv,zz_conv_ff_hist[0,0],binsize,/P)

  ;;;;near field results
  If fix(str1.star[0]) eq 1 then begin
    zz_conv_nf_hist = histosh(zz_conv,nbins=100,xrange=[-5e-4,5e-4],binsize=binsize) ;shifted histo values are at center of each bin
    I_nf_conv = reform(zz_conv_nf_hist[1,*])
    wI_nf_conv = wave_setscale(I_nf_conv,zz_conv_nf_hist[0,0],binsize,/P)
    ;graph = plot(wI_nf_conv.x, wI_nf_conv.w, overplot=1, color='red', linestyle=2,thick=2)
   xplot,make_set(wI_nf_conv.x, wI_nf_conv.w,  wI_ff_conv.w), coltitles= $
     ['Vertical position [mm]','near field (star_hybrid_nf.01)', $
      'far field (star_hybrid.01 = retrace(screen_hybrid.0102))']
  EndIf else begin
   xplot,make_set(wI_ff_conv.x, wI_ff_conv.w), coltitles= $
     ['Vertical position '+sUnit,'far field (retrace(screen_hybrid.0102))']
  EndElse
endif
  ; 
  ; COMMENT: 
  ; this last testing block is equivalent to histogram the column 3 
  ; (vertical positions) of star_hybrid_nf.01 (near field, propagated in the 
  ; wave optics way) and the retraced screen_hybrid.0102 (far field, 
  ; propagated in ray-tracing way)
  ;
if fix(str1.calcType[0]) gt 0 then begin
  if fix(str1.showPlots[0]) GE 1 then begin ; prepare next plot
      xsh_histo1,'star_hybrid.01',3-DeltaCol,XRange=[-0.0005,0.0005],NoLost=1,NBins=100,title='FAR FIELD: ray-traced screen_new.0102 a distance '+sUnit+': '+strcompress(str1.distance)
      If fix(str1.star[0]) eq 1 then xsh_histo1,'star_hybrid_nf.01',3-DeltaCol,XRange=[-0.0005,0.0005],NoLost=1,NBins=100,title='NEAR FIELD: wave-propagation to distance '+sUnit+': '+strcompress(str1.distance)
  endif
endif
out: 
  free_lun,funit
  xdisplayfile1,'hybrid.txt',title='Results of hybrid run (hybrid.txt)',height=50

if keyword_set(save1) then begin
    write_gfile,str1,'start_hybrid.'+str_n_oe
endif
End






