;+
; NAME:
;	nrbInterpCoefs
;
; PURPOSE:
;	This function returns a NURBS structure containing a NURBS that 
;	is forced to pass through the grid points. 
;
;
; CATEGORY:
;	NURBS library
;
; CALLING SEQUENCE:
;	n2 = nrbInterpCoefs(n1,uu,vv)
;
; INPUTS:
;    n1:	a NURBS-structure, as created by nrbStruct(), containing
;		the grid, and other NURBS data (order, etc.)
;    uu: the parametric points, associated to the knots vector, along U direction
;    vv: the parametric points, associated to the knots vector, along V direction
;
; KEYWORD PARAMETERS:
;    WR : set this keyword to write a shadow-formatted surface file with the 
;	  interpolated grid, in the files: 
;	  COEFS.DAT = file with interpolated data grid
;	  WEIGHTS.DAT = file with weights for each point of the grid
;
; OUTPUTS:
;    n2: a NURBS structure with the new interpolated NURBS
;
; PROCEDURE:
;	It recalcualates the control points, for forcing interpolation onto the given points grid.
;	It uses the invertion properties of the matrix form of the NURBS.
;       The grid points are recalculated, but the knots (nor the parametric) 
;       values are not changed
;	It uses the internal procedure __nrbCtrlPolyMat for calculations.
;	It uses the internal procedure __nrbWriteShadowSurf for file writing.
;
; EXAMPLE:
;	
;	a=nrbconvsurf('testsurf.dat')
;	b=nrbgetknots(a,uu=uu,vv=vv)
;	n1 = nrbstruct(a,b.uknots,b.vknots)
;	n2=nrbinterpcoefs(n1,uu,vv)
;	help,/str,n2
;	nrbplot,n2,30,30
;	
; MODIFICATION HISTORY:
;	written by Davide Bianchi. ESRF. 2006
;
;-	
FUNCTION nrbinterpcoefs, nrbsurf,uu,vv, wr=wr

catch, error_status
if error_status ne 0 then begin
   message,/info,'error caught: '+!err_string
   if sdep(/w) then itmp = Dialog_Message(/Error,$
	'NURBS: nrbInterpCoefs: error caught: '+!err_string)
   catch, /cancel
   on_error,2
   RETURN,0
endif




IF n_elements(nrbsurf.dim) EQ 2 THEN BEGIN  ; SURFACE 

  Xgrid = nrbsurf.Xgrid    ;X grid coordinates
  Ygrid = nrbsurf.Ygrid    ;Y grid coordinates
  Zgrid = nrbsurf.Zgrid    ;Z grid coordinates
  Wgrid = nrbsurf.Wgrid    ;W grid values


  nx=nrbsurf.dim[0]
  ny=nrbsurf.dim[1]

  n = nx - 1       ;n is the number of the last index of X (in accordance with theory notation)
  m = ny - 1       ;m is the number of the last index of Y (in accordance with theory notation)

  ;== KNOTS RENAME==;
  ;the output is a structure, keeping the knots along U and V parametric direction
  ;UU and VV are the paraetric values generated by the centripetal algorithm
  uknots = nrbsurf.uknots
  vknots = nrbsurf.vknots         
  
  ;== CONTROL MATRIX OF BASIS FUNCTION ==;
  
  ;the propouse is to use matrix calculation for fasting the operations
  ; RU an RV ar square matrices, coming by the thoery /*(ref)*/
  ;given a grid of 3D point we have that:
  ;S(u,v)=RU##Pij##RV'   (1)
  ;where RU and RV are the control basis matrices
  ;and ## is the IDL operator for row-by-column matrix product
  ;the main idea, is that given the parametric point (UU && VV) and the knots (u_knots, && v_knots)
  ;it's easly to generate the matrix of control point.
  ;the aim is to be able to interpolate along given point, in this way it will be inverted the equation (1)
  ; Pij=RUinv##S(ur,vs)##RV'inv
  
  p=nrbsurf.p
  q=nrbsurf.q
  
  RU=__nrbctrlpolymat(uu,uknots, p, nx)  ;RU is the basis matrix (nx,nx) along the U direction
  
  RV=transpose(__nrbctrlpolymat(vv,vknots, q, ny))  ;RV is the basis matrix (ny,ny) along the V direction
  
  ; computes the denominator of the evaluation formula
  ; ## is the matrix product
  ; In the very odd IDL format: 
  ; rv is a (ny,ny) matrix  (ny columns, nx rows)
  ; wgrid are the weights is a (ny,nx) matrix
  ; ru is a (nx,nx) matrix
  ; results a [(nx,nx).(ny,nx)].(ny,ny) = (ny,nx).(ny,ny) = (ny,nx)
  det=RU##Wgrid##RV
  

  ;print, 'matrice U',ru
  ;print, 'matrisrfce U',rv

  ;for calculating the control coefficients, in the way to have the the interpolating, grid coefficients
  ;we inverte the matrix by a linear operation with the LAPACK (tool of IDL)
  ;
  RUinv=LA_INVERT(ru, /double)
  RVinv=LA_INVERT(rv, /double)



  ; General NURBS evaluation formula:  [Q] = [Ru] [W P] [Rv] / det = (x,y,z)
  ;  [Ru^-1]  [Q] det [Rv^-1]  = [Ru^-1] [Ru] [W P] [Rv] [Rv^-1] =>
  ;  ([Ru^-1]  [Q] det [Rv^-1]) = W P
  ;new grid coefficients, by inverse matrix operations
  ;These grid coefficients are affected already by the weight
  
  X_inv = (RUinv##(Xgrid*det)##RVinv) ; X*w 
  Y_inv = (RUinv##(Ygrid*det)##RVinv) ; Y*w
  Z_inv = (RUinv##(Zgrid*det)##RVinv) ; Z*w
  


  ;==OUTPUT GRID==;
  interp_grid=nrbconvsurf(xx=x_inv, yy=y_inv,zz=z_inv,weights=det)
  ;interp_grid=nrbconvsurf(xx=x_inv, yy=y_inv,zz=z_inv,weights=Wgrid)


  IF keyword_set(wr) THEN __nrbwriteshadowsurf, interp_grid
  
  intnrb = nrbstruct(interp_grid, uknots,vknots)
  
ENDIF ELSE BEGIN  ; CURVE

  ;CURVE INTERPOLATION

  Xgrid = nrbsurf.Xgrid    ;X grid coordinates
  Ygrid = nrbsurf.Ygrid    ;Y grid coordinates
  Zgrid = nrbsurf.Zgrid    ;Z grid coordinates
  Wgrid = nrbsurf.Wgrid    ;W grid values


  nx=nrbsurf.dim[0]

  n = nx - 1       ;n is the number of the last index of X (in accordance with theory notation)
  
  ;== KNOTS RENAME==;
  ;UU and VV are the paraetric values generated by the centripetal algorithm
  uknots = nrbsurf.uknots
                                

  ;== CONTROL MATRIX OF BASIS FUNCTION ==;

  ;the propouse is to use matrix calculation for fasting the operations
  ; RU an RV ar square matrices, coming by the thoery /*(ref)*/
  ;given a grid of 3D point we have that:
  ;S(u,v)=RU##Pij##RV'   (1)
  ;where RU and RV are the control basis matrices
  ;and ## is the IDL operator for row-by-column matrix product
  ;the main idea, is that given the parametric point (UU && VV) and the knots (u_knots, && v_knots)
  ;it's easly to generate the matrix of control point.
  ;the aim is to be able to interpolate along given point, in this way it will be inverted the equation (1)
  ; Pij=RUinv##S(ur,vs)##RV'inv
  
  p=nrbsurf.p
  
  RU=__nrbctrlpolymat(uu,uknots, p, nx)  ;RU is the basis matrix along the U direction
  
  
  det=RU##Wgrid
  
  
  ;print, 'matrice U',ru
  ;print, 'matrisrfce U',rv
  
  ;for calculating the control coefficients, in the way to have the the interpolating, grid coefficients
  ;we inverte the matrix by a linear operation with the LAPACK (tool of IDL)
  ;
  RUinv=LA_INVERT(ru, /double)
  
  ;new grid coefficients, by inverse matrix operations
  
  X_inv = RUinv##(Xgrid*det) ; /Wgrid
  Y_inv = RUinv##(Ygrid*det) ; /Wgrid
  Z_inv = RUinv##(Zgrid*det) ; /Wgrid
  
  ;==OUTPUT GRID==;
  interp_grid=dblarr(nx,4)
  
  interp_grid[*,0]=X_inv
  interp_grid[*,1]=Y_inv
  interp_grid[*,2]=Z_inv
  interp_grid[*,3]=det
  ;;interp_grid[*,3]=Wgrid

  ;IF keyword_set(wr) THEN __nrbwriteshadowsurf, interp_grid

  intnrb = nrbstruct(interp_grid, uknots)
  
ENDELSE

return, intnrb

END
