function __nrbrevolve,curve,pnt=pnt,vec=vec,theta=theta
;+
;
; FUNCTION nrbrevolve,curve,pnt=pnt,vec=vec,theta=theta
; Translated from MatLab code by D. Bianchi, ESRF, 2006
;
;// Function Name:
;//
;//   __nrbrevolve - Construct a NURBS surface by revolving a NURBS curve.
;//
;// Calling Sequence:
;//
;//   srf = __nrbrevolve(crv,pnt,vec[,ang])
;//
;// Parameters:
;//
;//   crv     : NURBS curve to revolve, see nrbmak.
;//
;//   pnt     : Coordinate of the point used to define the axis
;//               of rotation.
;//
;//   vec     : Vector defining the direction of the rotation axis.
;//
;//   ang     : Angle to revolve the curve, default 2*pi
;//
;// Description:
;//
;//   Construct a NURBS surface by revolving the profile NURBS curve around
;//   an axis defined by a point and vector.
;//
;// Examples:
;//
;//   Construct a sphere by rotating a semicircle around a x-axis.
;//
;//   crv = __nrbcirc(1.0,[0 0 0],0,pi);
;//   srf = nrbrevolve(crv,[0 0 0],[1 0 0]);
;//   nrbplot(srf,[20 20]);
;//
;// NOTE:
;//
;//   The algorithm:
;//
;//     1) vectrans the point to the origin (0,0,0)
;//     2) rotate the vector into alignment with the z-axis
;//
;//     for each control point along the curve
;//
;//     3) determine the radius and angle of control
;//        point to the z-axis
;//     4) construct a circular arc in the x-y plane with
;//        this radius and start angle and sweep angle theta
;//     5) combine the arc and profile, coefs and weights.
;//
;//     next control point
;//
;//     6) rotate and vectrans the surface back into position
;//        by reversing 1 and 2.
;//
;//  D.M. Spink
;//  Copyright (c) 2000.
;-

;if nargin < 3
;  error,'Not enough arguments to construct revolved surface');
;end

IF not keyword_set(theta) THEN theta = 2.0*!dpi;


;// Translate curve the center point to the origin
IF not keyword_set(pnt) THEN  pnt = dblarr(1,3)

IF n_elements(pnt) NE 3 THEN message,'All point and vector coordinates must be 3D'

IF not keyword_set(vec) THEN vec = [1,0,0]
;// Translate and rotate the curve into alignment with the z-axis
T  = vectrans(-pnt);
angx = vecangle(vec[0],vec[2])


RY = vecroty(-angx)

vectmp = RY ## transpose([vecnorm(vec,/returnNormalized),1.0])

angy = vecangle(vectmp[1],vectmp[2]);
RX = vecrotx(angy);

crv = nrbtransform(curve,RX##RY##T);


;help, /str, curve
;nrbplot, curve,100,100
;// Construct an arc
arc = __nrbcirc(1.0,oo=[0d,0d,0d],sang=0d,eang=theta);

;nrbplot, arc,100,100

;// Construct the surface
coefs = dblarr(arc.dim,4,crv.dim);

angle = vecangle(reform(crv.coefs[*,1]),reform(crv.coefs[*,0]))

radius = sqrt(total(crv.coefs[*,0:1]^2,2));

FOR i = 0,curve.dim-1 DO BEGIN
  coefs[*,*,i] = (vecrotz(angle[i]) ## vectrans([0.0, 0.0, crv.coefs[i,2]]) ##$
  vecscale([radius[i],radius[i],0])) ## arc.coefs
  coefs[*,3,i] = coefs[*,3,i]*crv.coefs[i,3];
ENDFOR

ind=where(abs(coefs) LT 1e-7)
IF ind[0] NE -1 THEN coefs[ind]=0d

surf = nrbstruct(coefs,arc.uknots,crv.uknots);

;// Rotate and vectrans the surface back into position
T = vectrans(pnt);
RX = vecrotx(-angy);
RY = vecroty(angx);

surf = nrbtransform(surf,T##RY##RX);

return, surf
END
