
;+
; NAME:
;	ROWLAND
; PURPOSE:
;	Calculates the tangential and sagittal focusing radii, and 
;	Rowland condition for gratings and asymmetric crystals.
; CATEGORY:
;	Optics
; CALLING SEQUENCE:
;	rowland
; KEYWORD PARAMETERS:
;    GROUP
;	The widget id of the parent. Also used for centered the input/output 
;	windows.
; OUTPUTS:
;	The outputs are shown in a text window.
; COMMON BLOCKS:
;	COMMON rowland str_rowland   
;	(to store the inputs for following runs)
; SIDE EFFECTS:
;	None.
; RESTRICTIONS:
;	If the IDL session support widgets, the i/o is done via widgets, 
;	otherwise it is done by terminal.
;	None.
; PROCEDURE:
;	Use well known formulas.
; MODIFICATION HISTORY:
;	Created 2001-03-08 by M. Sanchez del Rio from a c code
;	01/03/14 srio@esrf.fr force to use cm and eV
;	06/02/17 srio@esrf.fr adds mrad units and fix bug (Undefined D)
;		Removed the non-widget input/output
;	07/10/03 srio@esrf.eu added any reflection
;-
;/****************************************************************************
; *  
; *  File:                 rowland.c
; *
; *  Project:              X-ray optics utilities
; *
; *  Description:          Calculates the sagittal and tangential radii
; *                        for gratings or systems in which the incident
; *                        and reflected angles are different. It also
; *                        calculates the Rowland conditions and the step 
; *                        of the grating for a given energy.
; *
; *  Author:               Manolo Sanchez del Rio 
; *
; *  Date:                 December 1991
; *
; *****************************************************************************/

PRO rowland, Group=group

on_error,2

COMMON rowland,str_rowland

TORAD  = !dpi/180.0

;
; Inputs
;
   IF N_Elements(str_rowland) EQ 0 THEN BEGIN
   str_rowland = {i_flagMenu:['1','Incident and reflected GRAZING angle [deg]',$
			  'Incident and reflected GRAZING angle [mrad]',$
			  'Energy for asymmetrical cut crystals'], $

	  t1:5.0, t2:5.0, $

	  e:10000.0, $
          nmenu:['0','Si','Ge','Diamond'],$
	  h:1, k:1, l:1, $
	  alfa:0.0, $

	  d1:3000.0, d2:3000.0 }

   ENDIF
   flags = ['1', $
	'w(0) LT 2','w(0) LT 2',$
	'w(0) EQ 2','w(0) EQ 2','w(0) EQ 2', $
	'w(0) EQ 2','w(0) EQ 2','w(0) EQ 2', $
	'1','1']

   titles = ['Select', $
	'incident grazing angle', 'reflected grazing angle', $
	'Energy (eV)','Crystal', 'h:','k:','l:', $
        'Asymmetric cut angle [deg]', $
	'Distance source-element [cm]','Distance element-image [cm]']
          

   action=''
   XScrMenu,str_rowland,/Inter,/NoType,Flags=flags,Titles=titles,Action=action,$
	Wtitle='Focusing radii', Dialog_Parent=group,NCol=2
   IF action EQ 'DONT' THEN RETURN

   CASE StrCompress(str_rowland.i_flagMenu[0],/Rem) OF
     '0': BEGIN
		i_flag='I'
                t1 = str_rowland.t1
                t2 = str_rowland.t2
	  END
     '1': BEGIN
		i_flag='I'
                t1 = str_rowland.t1*1e-3*180/!pi
                t2 = str_rowland.t2*1e-3*180/!pi
	  END
     '2': i_flag='E'
     else:
   ENDCASE

   e = str_rowland.e
   n = Fix(str_rowland.nmenu[0])
   alfa = str_rowland.alfa
   d1 = str_rowland.d1
   d2 = str_rowland.d2
   hh = str_rowland.h
   kk = str_rowland.k
   ll = str_rowland.l
   

;
; end inputs
;
    CASE StrCompress(StrUpCase(i_flag),/Rem) OF
     'I': BEGIN
	t1 = t1*TORAD 
	t2 = t2*TORAD
	END
     'E': BEGIN
        CASE Fix(n) OF
          0: a=5.43102
          1: a=5.65735 
          2: a=3.56679
          else: Message," Urecognized input "
        ENDCASE
        d = a/sqrt(Double(hh^2+kk^2+ll^2))
        TOANGS = physical_constants('hc')
	theta=asin(TOANGS/(e*2.0*d))

	alfa = alfa*TORAD
        t1 = theta+alfa
        t2 = theta-alfa
	END
      else: Message," Urecognized input "
    ENDCASE

;    /*
;     * Calculations
;     */
    s1 = sin(t1)
    s2 = sin(t2)
    s1_2 = s1*s1
    s2_2 = s2*s2
    r = s1_2/d1 + s2_2/d2
    r = (s1+s2)/r
    rs = (s1+s2)/(1.0/d1 + 1.0/d2)

;    /*
;     * Present output.
;     */
    out = [ "******************************************************"]
    out = [ out, 'Inputs:', $
    " Distance Source-element: "+StrCompress(d1)+' cm', $
    " Distance Element-image: "+StrCompress(d2)+' cm']
    IF StrCompress(StrUpCase(i_flag),/Rem) EQ 'E' THEN BEGIN
      out=[out,$
      " energy is "+StrCompress(e,/Rem)+' eV',$
      " d-spacing is "+StrCompress(d)+ ' A', $
      " Bragg angle angle is "+StrCompress(theta/TORAD)+' degrees']
      out=[out,' ','Outputs:',$
      " Incident GRAZING angle is "+StrCompress(t1/TORAD)+' degrees = '+$
                                   StrCompress(t1*1000)+' mrad ', $
      " Reflected GRAZING angle is "+StrCompress(t2/TORAD)+' degrees = '+$
                                   StrCompress(t2*1000)+' mrad ']
    ENDIF ELSE BEGIN
      out=[out,$
      " Incident GRAZING angle is "+StrCompress(t1/TORAD)+' degrees = '+$
                                   StrCompress(t1*1000)+' mrad ', $
      " Reflected GRAZING angle is "+StrCompress(t2/TORAD)+' degrees = '+$
                                   StrCompress(t2*1000)+' mrad ', $
      " ","Outputs: "]
    ENDELSE

    out=[out,$
    " Tangential radius is "+StrCompress(r)+' cm', $
    " Sagittal radius is "+StrCompress(rs)+' cm', $
    " ", $
    " The Rowland Circle conditions are: ", $
    "     R1 = "+StrCompress(r*s1)+' cm', $
    "     R2 = "+StrCompress(r*s2)+' cm', $
    " ", $
    " For s1 = "+StrCompress(d1)+ "the Rowland Circle conditions are: ", $
    "     s2 = "+StrCompress(d1*s2/s1)+" cm ", $
    "      r = "+StrCompress(d1/s1)+" cm ", $
    "******************************************************", $
    " " ]


    XDisplayFile1,Text=out, Dialog_Parent=group, Title='Rowland' 

    FOR i=0,N_Elements(out)-1 DO Print,out[i]

END
